
{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_UpdateBar                                             }
{                                                                              }
{ **************************************************************************** }

{~~ This unit contains the component IB_UpdateBar. }
unit IB_UpdateBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,

  IB_Components;

type
{~~ These are the different button types available to this button bar.

dbUpdate puts the dataset into dsUpdating mode.
dbInsert puts the dataset into dsInserting mode.
dbDelete puts the dataset into dsDeleting mode. 
dbPost saves whatever changes have been made to the dataset.
dbCancel aborts whatever changes have been made to the dataset.}
TIB_UpdateBar_ButtonType = ( dbUpdate,
                             dbInsert,
                             dbDelete,
                             dbPost,
                             dbCancel );
{~~$<!>}
TIB_UpdateBar_ButtonSet = set of TIB_UpdateBar_ButtonType;
{~~$<!>}
TIB_UpdateBar_Button = class;

{~~ Button bar that gives easy access and control of the modifying of data.}
TIB_UpdateBar = class (TCustomPanel)
  private
    FIB_DataLink: TIB_DataLink;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_UpdateBar_ButtonSet;
    MinBtnSize: TPoint;
    FFocusedButton: TIB_UpdateBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    function GetDataSetLink: TIB_DataSetLink;
    procedure SetDataSetLink(AValue: TIB_DataSetLink);
    function GetReceiveFocus: boolean;
    procedure SetReceiveFocus( AValue: boolean );
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_UpdateBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_DataSetStateChanged( Sender: TIB_DataLink;
                                      IB_DataSetLink: TIB_DataSetLink );
    procedure IB_OutputRowStateChanged( Sender: TIB_DataLink;
                                        IB_DataSetLink: TIB_DataSetLink );
    procedure BarClick(Sender: TObject); virtual;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
{~~ Storage array for the different buttons on this button bar. }
    Buttons: array[TIB_UpdateBar_ButtonType] of TIB_UpdateBar_Button;
    constructor Create(AOwner: TComponent); override;
{~~$<!>}
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
{~~ Method used to pass the click of a single button to the owning parent.}
    procedure BtnClick(Index: TIB_UpdateBar_ButtonType);
{~~$<*>...
The currently selected button of the button set.

Use this property in the xxxxAction events to know which button was clicked.}
    property FocusedButton: TIB_UpdateBar_ButtonType read FFocusedButton;
  published
{~~$<*>...
Reference to the dataset link to perform actions upon.}
    property IB_DataSetLink: TIB_DataSetLink
        read GetDataSetLink
       write SetDataSetLink;
{~~$<*>...
This property is used in conjunction with the AnnounceFocus property of...
the IB_DataSetLink component. It is a global focusing system that allows...
the SearchBar to link itself to whereever the user's focus goes.

For example, in an MDI application the user could go from one child window...
to another. Each child window would have its own IB_DataSetLink context....
Thus, if each child window's IB_DataSetLink component's AnnounceFocus...
property is set to true the SearchBar will receive the focus and align with it.}
    property ReceiveFocus: boolean read GetReceiveFocus write SetReceiveFocus;
{~~$<*>...
Set which button are visible to the user.}
    property VisibleButtons: TIB_UpdateBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [ dbUpdate, dbInsert, dbDelete, dbPost, dbCancel ];
{~~$<*>...
Define custom behavior before the action of the button click is executed.}
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
{~~$<*>...
Define custom behavior after the action of the button click is executed.

This event will not be triggered if an exception was raised.}
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
{~~$<!>} property Align;
{~~$<!>} property DragCursor;
{~~$<!>} property DragMode;
{~~$<!>} property Enabled;
{~~$<!>} property Ctl3D;
{~~$<!>} property ParentCtl3D;
{~~$<!>} property ParentShowHint;
{~~$<!>} property PopupMenu;
{~~$<!>} property ShowHint;
{~~$<!>} property TabOrder;
{~~$<!>} property TabStop;
{~~$<!>} property Visible;
{~~$<!>} property OnClick;
{~~$<!>} property OnDblClick;
{~~$<!>} property OnDragDrop;
{~~$<!>} property OnDragOver;
{~~$<!>} property OnEndDrag;
{~~$<!>} property OnEnter;
{~~$<!>} property OnExit;
{~~$<!>} property OnResize;
  end;

  TIB_UpdateBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_UpdateBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_UpdateBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$IFDEF WIN32}
{$R IB_UpdateBar.RES}
{$ENDIF}

{------------------------------------------------------------------------------}

constructor TIB_UpdateBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_DataLink := TIB_DataLink.Create( Self );
  with FIB_DataLink do begin
    AfterAssignment := IB_DataSetStateChanged;
    OnStateChanged := IB_DataSetStateChanged;
  end;
  FVisibleButtons := [ dbUpdate, dbInsert, dbDelete, dbPost, dbCancel ];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := dbUpdate;
end;

procedure TIB_UpdateBar.IB_DataSetStateChanged( Sender: TIB_DataLink;
                                                IB_DataSetLink: TIB_DataSetLink);
begin
  ActiveChanged;
end;

procedure TIB_UpdateBar.IB_OutputRowStateChanged( Sender: TIB_DataLink;
                                                  IB_DataSetLink: TIB_DataSetLink );
begin
  ActiveChanged;
end;

function TIB_UpdateBar.GetDataSetLink: TIB_DataSetLink;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.IB_DataSetLink;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_UpdateBar.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  if FIB_DataLink <> nil then begin
    if FIB_DataLink.IB_DataSetLink <> AValue then begin
      FIB_DataLink.IB_DataSetLink := AValue;
      ActiveChanged;
    end;
  end;
end;

function TIB_UpdateBar.GetReceiveFocus: boolean;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.ReceiveFocus;
  end else begin
    Result := false;
  end;
end;

procedure TIB_UpdateBar.SetReceiveFocus( AValue: boolean );
begin
  if FIB_DataLink <> nil then begin
    FIB_DataLink.ReceiveFocus := AValue;
  end;
end;

procedure TIB_UpdateBar.InitButtons;
var
  I: TIB_UpdateBar_ButtonType;
  Btn: TIB_UpdateBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_UpdateBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      dbUpdate: begin
        Btn.Hint := 'Update Row';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'UPD_BAR_UPDATE'  );
        Btn.NumGlyphs := 2;
      end;
      dbInsert: begin
        Btn.Hint := 'Insert New Row Into DataSet';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'UPD_BAR_INSERT'  );
        Btn.NumGlyphs := 2;
      end;
      dbDelete: begin
        Btn.Hint := 'Delete Current Row From DataSet';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'UPD_BAR_DELETE'  );
        Btn.NumGlyphs := 2;
      end;
      dbPost: begin
        Btn.Hint := 'Post Row Updates';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'UPD_BAR_POST'  );
        Btn.NumGlyphs := 2;
      end;
      dbCancel: begin
        Btn.Hint := 'Cancel Row Updates';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL'  );
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_UpdateBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_UpdateBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_UpdateBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_UpdateBar.SetVisible(Value: TIB_UpdateBar_ButtonSet);
var
  I: TIB_UpdateBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;


procedure TIB_UpdateBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_UpdateBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[dbUpdate] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_UpdateBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_UpdateBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_UpdateBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_UpdateBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_UpdateBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TIB_UpdateBar.ActiveChanged;
begin
  if ( IB_DataSetLink <> nil ) and
     ( IB_DataSetLink.IB_DataSet <> nil ) and
     ( Enabled ) then with IB_DataSetLink.IB_DataSet do begin
    if not Prepared or
       ( StatementType = stSelect ) or
       ( StatementType = stSelectForUpdate ) then begin

      with Buttons[ dbPost ].Glyph do begin
        case DataSetState of
          dsUpdating:  Handle := LoadBitmap(HInstance, 'UPD_BAR_POST_UPDATE' );
          dsInserting: Handle := LoadBitmap(HInstance, 'UPD_BAR_POST_INSERT' );
          dsDeleting:  Handle := LoadBitmap(HInstance, 'UPD_BAR_POST_DELETE' );
          dsSearching: Handle := LoadBitmap(HInstance, 'UPD_BAR_POST_SEARCH' );
          else         Handle := LoadBitmap(HInstance, 'UPD_BAR_POST'        );
        end;
      end;

      with Buttons[ dbCancel ].Glyph do begin
        case DataSetState of
          dsUpdating:  Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL_UPDATE');
          dsInserting: Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL_INSERT');
          dsDeleting:  Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL_DELETE');
          dsSearching: Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL_SEARCH');
          else         Handle := LoadBitmap(HInstance, 'UPD_BAR_CANCEL'       );
        end;
      end;
      Buttons[ dbUpdate  ].Enabled := CanUpdate;
      Buttons[ dbInsert  ].Enabled := CanInsert;
      Buttons[ dbDelete  ].Enabled := CanDelete;
      Buttons[ dbPost    ].Enabled := NeedToPost;
      Buttons[ dbCancel  ].Enabled := NeedToPost;
    end else begin
      Buttons[ dbPost  ].Glyph.Handle :=LoadBitmap(HInstance, 'UPD_BAR_POST' );
      Buttons[ dbCancel].Glyph.Handle :=LoadBitmap(HInstance, 'UPD_BAR_CANCEL');
      Buttons[ dbUpdate  ].Enabled := false;
      Buttons[ dbInsert  ].Enabled := false;
      Buttons[ dbDelete  ].Enabled := false;
      Buttons[ dbPost    ].Enabled := false;
      Buttons[ dbCancel  ].Enabled := false;
    end;
  end else begin
    Buttons[ dbUpdate  ].Enabled := false;
    Buttons[ dbInsert  ].Enabled := false;
    Buttons[ dbDelete  ].Enabled := false;
    Buttons[ dbPost    ].Enabled := false;
    Buttons[ dbCancel  ].Enabled := false;
  end;
end;

procedure TIB_UpdateBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  ActiveChanged;
end;

procedure TIB_UpdateBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_UpdateBar_ButtonType;
  OldFocus: TIB_UpdateBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_UpdateBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_UpdateBar.BtnClick(Index: TIB_UpdateBar_ButtonType);
begin
  if not (csDesigning in ComponentState) and
     Assigned( FBeforeAction ) then begin
    FBeforeAction( Self );
  end;

  with FIB_DataLink do begin
    if IB_DataSet <> nil then with IB_DataSet do begin
      case Index of
        dbUpdate: Update;
        dbInsert: Insert;
        dbDelete: Delete;
        dbPost:   Post;
        dbCancel: Cancel;
      end;
    end;
  end;

  if not (csDesigning in ComponentState) and
     Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_UpdateBar.BarClick( Sender: TObject );
begin
  BtnClick (TIB_UpdateBar_Button(Sender).Index);
end;

procedure TIB_UpdateBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_UpdateBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
