
{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_TransactionBar                                        }
{                                                                              }
{ **************************************************************************** }

unit IB_TransactionBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,

  IB_Components;

{------------------------------------------------------------------------------}

type
  TIB_TransactionBar_ButtonType = ( tbStart,
                                    tbSavePoint,
                                    tbPostAll,
                                    tbCancelAll,
                                    tbCommit,
                                    tbRollback,
                                    tbClose );
                                   
  TIB_TransactionBar_ButtonSet = set of TIB_TransactionBar_ButtonType;

  TIB_TransactionBar_Button = class;
  TIB_TransactionBar = class (TCustomPanel)
  private
    FIB_TransactionLink: TIB_TransactionLink;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_TransactionBar_ButtonSet;
    MinBtnSize: TPoint;
    FFocusedButton: TIB_TransactionBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    function GetTransaction: TIB_Transaction;
    procedure SetTransaction(AValue: TIB_Transaction);
    function GetReceiveFocus: boolean;
    procedure SetReceiveFocus(AValue: boolean);
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_TransactionBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_TransactionStatusChanged( TLink: TIB_TransactionLink;
                                           Sender: TIB_Transaction);
    procedure BarClick(Sender: TObject); virtual;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    Buttons: array[TIB_TransactionBar_ButtonType] of TIB_TransactionBar_Button;
    constructor Create(AOwner: TComponent); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TIB_TransactionBar_ButtonType);
    property FocusedButton: TIB_TransactionBar_ButtonType read FFocusedButton;
  published
    property IB_Transaction: TIB_Transaction
        read GetTransaction
       write SetTransaction;
    property ReceiveFocus: boolean read GetReceiveFocus write SetReceiveFocus;
    property VisibleButtons: TIB_TransactionBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [tbStart, tbSavePoint, tbCommit, tbRollback];
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
  end;

  TIB_TransactionBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_TransactionBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_TransactionBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$R IB_TransactionBar.RES}

{------------------------------------------------------------------------------}

constructor TIB_TransactionBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_TransactionLink := TIB_TransactionLink.Create( Self );
  with FIB_TransactionLink do begin
    AfterAssignment := IB_TransactionStatusChanged;
    OnStatusChanged := IB_TransactionStatusChanged;
  end;
  FVisibleButtons := [tbStart, tbSavePoint, tbCommit, tbRollback];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := tbStart;
end;

procedure TIB_TransactionBar.IB_TransactionStatusChanged(
                                                    TLink: TIB_TransactionLink;
                                                   Sender: TIB_Transaction);
begin
  ActiveChanged;
end;

function TIB_TransactionBar.GetTransaction: TIB_Transaction;
begin
  if FIB_TransactionLink <> nil then begin
    Result :=  FIB_TransactionLink.IB_Transaction;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_TransactionBar.SetTransaction(AValue: TIB_Transaction);
begin
  if FIB_TransactionLink <> nil then begin
    if FIB_TransactionLink.IB_Transaction <> AValue then begin
      FIB_TransactionLink.IB_Transaction := AValue;
      ActiveChanged;
    end;
  end;
end;

function TIB_TransactionBar.GetReceiveFocus: boolean;
begin
  if FIB_TransactionLink <> nil then begin
    Result :=  FIB_TransactionLink.ReceiveFocus;
  end else begin
    Result := false;
  end;
end;

procedure TIB_TransactionBar.SetReceiveFocus(AValue: boolean);
begin
  if FIB_TransactionLink <> nil then begin
    FIB_TransactionLink.ReceiveFocus := AValue;
  end;
end;

procedure TIB_TransactionBar.InitButtons;
var
  I: TIB_TransactionBar_ButtonType;
  Btn: TIB_TransactionBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_TransactionBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      tbStart: begin
        Btn.Hint := 'Start Transaction';
        Btn.Caption := '';//Start';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_START'  );
        Btn.NumGlyphs := 2;
      end;
      tbSavePoint: begin
        Btn.Hint := 'Commit and Retain Current Transaction';
        Btn.Caption := '';//Savepoint';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_SAVEPOINT'  );
        Btn.NumGlyphs := 2;
      end;
      tbPostAll: begin
        Btn.Hint := 'Post All Transaction Data Sets';
        Btn.Caption := ''; //'Post All';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_POSTALL'  );
        Btn.NumGlyphs := 2;
      end;
      tbCancelAll: begin
        Btn.Hint := 'Cancel All Transaction Data Sets';
        Btn.Caption := '';//Cancel All';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_CANCELALL'  );
        Btn.NumGlyphs := 2;
      end;
      tbCommit: begin
        Btn.Hint := 'Commit and End Current Transaction';
        Btn.Caption := '';//Commit';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_COMMIT'  );
        Btn.NumGlyphs := 2;
      end;
      tbRollback: begin
        Btn.Hint := 'Rollback and End Current Transaction';
        Btn.Caption := '';//Rollback';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_ROLLBACK'  );
        Btn.NumGlyphs := 2;
      end;
      tbClose: begin
        Btn.Hint := 'Close Current Transaction';
        Btn.Caption := '';//Close';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'TRAN_BAR_CLOSE'  );
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_TransactionBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_TransactionBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_TransactionBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_TransactionBar.SetVisible(Value: TIB_TransactionBar_ButtonSet);
var
  I: TIB_TransactionBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;


procedure TIB_TransactionBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_TransactionBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[tbStart] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_TransactionBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_TransactionBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_TransactionBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_TransactionBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_TransactionBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TIB_TransactionBar.ActiveChanged;
begin
  if (IB_Transaction <> nil) and (Enabled) then begin
    with IB_Transaction do begin
      Buttons[ tbStart     ].Enabled := not Started;
      Buttons[ tbSavePoint ].Enabled :=     TranStatus in [ tsActive,
                                                            tsActivePending ];
      Buttons[ tbPostAll   ].Enabled :=     PostPendingCount > 0;
      Buttons[ tbCancelAll ].Enabled :=     PostPendingCount > 0;
      Buttons[ tbCommit    ].Enabled :=     Started;
      Buttons[ tbRollback  ].Enabled :=     Started;
      Buttons[ tbClose     ].Enabled :=     Started;
    end;
  end else begin
    Buttons[ tbStart      ].Enabled := false;
    Buttons[ tbSavePoint  ].Enabled := false;
    Buttons[ tbPostAll    ].Enabled := false;
    Buttons[ tbCancelAll  ].Enabled := false;
    Buttons[ tbCommit     ].Enabled := false;
    Buttons[ tbRollback   ].Enabled := false;
    Buttons[ tbClose      ].Enabled := false;
  end;
end;

procedure TIB_TransactionBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  ActiveChanged;
end;

procedure TIB_TransactionBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_TransactionBar_ButtonType;
  OldFocus: TIB_TransactionBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_TransactionBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_TransactionBar.BtnClick(Index: TIB_TransactionBar_ButtonType);
begin
  if not (csDesigning in ComponentState) and
     Assigned( FBeforeAction ) then begin
    FBeforeAction( Self );
  end;

  if (IB_Transaction <> nil) then
  begin
    with IB_Transaction do
    begin
      case Index of
      tbStart:     Start;
      tbSavePoint: SavePoint;
      tbPostAll:   PostAll;
      tbCancelAll: CancelAll;
      tbCommit:    Commit;
      tbRollback:  Rollback;
      tbClose:     Close;
      end;
    end;
  end;

  if not (csDesigning in ComponentState) and
     Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_TransactionBar.BarClick( Sender: TObject );
begin
  BtnClick (TIB_TransactionBar_Button(Sender).Index);
end;

procedure TIB_TransactionBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_TransactionBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
