{~~ This unit contains the IB_StatementGrid component.}
unit IB_StatementGrid;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids,

  IB_Header,
  IB_Components;

type
{~~...
This component serves as a row viewer and modifier.

It also can display extended attributes of a row's columns.}
TIB_StatementGrid = class( TStringGrid )
  private
    FIB_StatementLink: TIB_StatementLinkBase;
    FIB_RowType: TIB_RowType;
    FColumnInfo: boolean;
    FDomainInfo: boolean;
    procedure SetIB_Statement( AValue: TIB_Statement ); virtual;
    function GetIB_Statement: TIB_Statement; virtual;
    procedure SetIB_RowType( AValue: TIB_RowType ); virtual;
    procedure SetColumnInfo( AValue: boolean ); virtual;
    procedure SetDomainInfo( AValue: boolean ); virtual;
  protected
    procedure Loaded; override;
    procedure IB_StatementUpdate( Sender: TIB_StatementLinkBase;
                                  Statement: TIB_Statement );
    procedure IB_ColumnChanged( Sender: TIB_StatementLinkBase;
                                Statement: TIB_Statement;
                                Column: TIB_Column );
    procedure IB_ColumnUpdate( Sender: TIB_StatementLinkBase;
                               Statement: TIB_Statement;
                               Column: TIB_Column );
  public
    constructor Create( AOwner: TComponent ); override;
{~~$<!>}
    procedure UpdateGrid;
{~~$<!>}
    procedure CloseGrid;
{~~$<!>}
    procedure PostChanges;
  published

{~~$<*>...
If true then column attributes are included in additional columns in the grid.}
    property ColumnInfo: boolean read FColumnInfo
                                 write SetColumnInfo
                                 default false;
                                 
{~~$<*>...
If true this will include the name of the domain along with the column type.
ColumnInfo must also be true for this property to have any effect.}
    property DomainInfo: boolean read FDomainInfo
                                 write SetDomainInfo
                                 default false;
{~~$<*>...
The statement for which this grid is to display or modify row data.}
    property IB_Statement: TIB_Statement read GetIB_Statement
                                         write SetIB_Statement;
{~~$<*>...
Each statement has two rows. One for input and one for output.

This property determines which row the grid will display or modify.}
    property IB_RowType: TIB_RowType read FIB_RowType
                                     write SetIB_RowType;
  end;

implementation

{------------------------------------------------------------------------------}

constructor TIB_StatementGrid.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FIB_StatementLink := TIB_StatementLink.Create( Self );
  with FIB_StatementLink do begin
    AfterAssignment := IB_StatementUpdate;
    AfterExecute := IB_StatementUpdate;
    OnPreparedChanged := IB_StatementUpdate;
    OnInputDataChanged  := IB_ColumnChanged;
    OnInputUpdateData   := IB_ColumnUpdate;
    OnOutputDataChanged := IB_ColumnChanged;
    OnOutputUpdateData  := IB_ColumnUpdate;
  end;
  CloseGrid;
end;

procedure TIB_StatementGrid.Loaded;
begin
  inherited Loaded;
  UpdateGrid;
end;

{------------------------------------------------------------------------------}

procedure TIB_StatementGrid.SetIB_RowType( AValue: TIB_RowType );
begin
  if FIB_RowType <> AValue then begin
    FIB_RowType := AValue;
    UpdateGrid;
  end;
end;

procedure TIB_StatementGrid.SetIB_Statement( AValue: TIB_Statement);
begin
  if FIB_StatementLink <> nil then begin
    if FIB_StatementLink.IB_Statement <> AValue then begin
      FIB_StatementLink.IB_Statement := AValue;
      UpdateGrid;
    end;
  end;
end;

function TIB_StatementGrid.GetIB_Statement: TIB_Statement;
begin
  if FIB_StatementLink <> nil then begin
    Result := FIB_StatementLink.IB_Statement;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_StatementGrid.SetColumnInfo( AValue: boolean );
begin
  if ColumnInfo <> AValue then begin
    FColumnInfo := AValue;
    UpdateGrid;
  end;
end;

procedure TIB_StatementGrid.SetDomainInfo( AValue: boolean );
begin
  if DomainInfo <> AValue then begin
    FDomainInfo := AValue;
    UpdateGrid;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_StatementGrid.IB_StatementUpdate( Sender: TIB_StatementLinkBase;
                                                Statement: TIB_Statement);
begin
  UpdateGrid;
end;

procedure TIB_StatementGrid.IB_ColumnChanged( Sender: TIB_StatementLinkBase;
                                              Statement: TIB_Statement;
                                              Column: TIB_Column );
begin
  UpdateGrid;
end;

procedure TIB_StatementGrid.IB_ColumnUpdate( Sender: TIB_StatementLinkBase;
                                             Statement: TIB_Statement;
                                             Column: TIB_Column );
begin
  PostChanges;
end;

{------------------------------------------------------------------------------}

procedure TIB_StatementGrid.CloseGrid;
var
  ii: integer;
begin
  Options := Options - [goEditing];
  if ColumnInfo then begin
    ColCount := 4;
    FixedCols := 3;
  end else begin
    ColCount := 3;
    FixedCols := 2;
  end;
  for ii := 0 to RowCount - 1 do begin
    Cells[ 0, ii ] := '';
    Cells[ 1, ii ] := '';
    Cells[ 2, ii ] := '';
    if ColumnInfo then begin
      Cells[ 3, ii ] := '';
    end;
  end;
  FixedRows := 0;
  RowCount  := 1;
end;

procedure TIB_StatementGrid.UpdateGrid;
var
  ii: integer;
  tmpRow: TIB_Row;
  ss: string;
begin
  if not ( csLoading    in ComponentState ) and
     not ( csDestroying in ComponentState ) then begin
    if FIB_StatementLink.Prepared then begin
      with IB_Statement do begin
        tmpRow := nil;
        case IB_RowType of
          rtInput: begin
            tmpRow := InputRow;
            Options := Options + [goEditing];
          end;
          rtOutput: begin
            tmpRow := OutputRow;
            Options := Options - [goEditing];
          end;
        end;
        if ( tmpRow <> nil ) and ( tmpRow.ColumnCount > 0 ) then begin
          RowCount := tmpRow.ColumnCount;
          for ii := 0 to tmpRow.ColumnCount - 1 do begin
            with tmpRow[ ii ] do begin
              Cells[ 0, ii ] := AliasName;
              ss := Copy(tmpRow[ii].ClassName,12,256);
              case SQLType of
                SQL_TEXT,
                SQL_TEXT_,
                SQL_VARYING,
                SQL_VARYING_: ss := ss + ' ( ' + IntToStr( SQLLen ) + ' )';
                else
                if SQLScale <> 0 then begin
                  ss := ss + ' ( '  + IntToStr( SQLLen ) + 'b, ' +
                                      IntToStr( SQLScale ) + ' )';
                end;
              end;
              if ( DomainInfo ) then begin
                if ( DomainName <> '' ) and
                   ( Copy( DomainName, 1, 4 ) <> 'RDB$' ) then begin
                  ss := Trim( DomainName ) + ' - ' + ss;
                end;
              end;
              Cells[ 1, ii ] := ss ;
              if ColumnInfo then begin
                ss := '';
                if IsPrimary then begin
                  ss := ss + 'PK ';
                end;
                if IsForeign then begin
                  ss := ss + 'FK ';
                end;
                if IsAlternate then begin
                  ss := ss + 'AK ';
                end;
                if not IsPrimary and
                   not IsForeign and
                   not IsAlternate and
                       IsIndexed then begin
                  ss := ss + 'IN ';
                end;
                if IsNullable then begin
                  ss := ss + 'NULL ';
                end;
                if IsDefaulted then begin
                  ss := ss + 'DF ';
                end;
                Cells[ 2, ii ] := ss;
              end;
              Cells[ FixedCols, ii ] := tmpRow[ ii ].AsString;
            end;
          end;
        end else begin
          CloseGrid;
        end;
      end;
    end else begin
      CloseGrid;
    end;
  end;
end;

procedure TIB_StatementGrid.PostChanges;
var
  ii: integer;
begin
  if ( IB_Statement <> nil ) and
     ( IB_RowType = rtInput ) then begin
    FIB_StatementLink.OnInputDataChanged := nil;
    try
      with IB_Statement do begin
        for ii := 0 to InputCount - 1 do begin
          InputRow[ ii ].AsString := Cells[ FixedCols, ii ];
        end;
      end;
    finally
      FIB_StatementLink.OnInputDataChanged := IB_ColumnChanged;
    end;
  end;
end;

end.
