
{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_StatementBar                                          }
{                                                                              }
{ **************************************************************************** }

{~~ This unit contains the component IB_StatementBar. }
unit IB_StatementBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,

  IB_Components;

type
{~~ These are the different button types available to this button bar.

sbAllocate allocates a statement handle on the server.
sbDeallocate frees the statement's resources from the server.
sbPrepare prepares the current statement in the SQL property.
sbUnprepare unprepares the current statement in the SQL property.
sbExecute performs the action of the prepared statement on the server.
}
TIB_StatementBar_ButtonType = ( sbAllocate,
                                sbDeallocate,
                                sbPrepare,
                                sbUnprepare,
                                sbExecute );
{~~$<!>}
TIB_StatementBar_ButtonSet = set of TIB_StatementBar_ButtonType;
{~~$<!>}
TIB_StatementBar_Button = class;

{~~ Button bar that gives access and control of statements.}
TIB_StatementBar = class (TCustomPanel)
  private
    FIB_StatementLink: TIB_StatementLinkBase;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_StatementBar_ButtonSet;
    MinBtnSize: TPoint;
    FExecuteUnprepared: boolean;
    FFocusedButton: TIB_StatementBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    function GetStatement: TIB_Statement;
    procedure SetStatement(AValue: TIB_Statement);
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_StatementBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_StatementAllocatedChanged( Sender: TIB_StatementLinkBase;
                                            IB_Statement: TIB_Statement );
    procedure IB_StatementPreparedChanged( Sender: TIB_StatementLinkBase;
                                            IB_Statement: TIB_Statement);
    procedure BarClick(Sender: TObject); virtual;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
{~~ Storage array for the different buttons on this button bar. }
    Buttons: array[TIB_StatementBar_ButtonType] of TIB_StatementBar_Button;
    constructor Create(AOwner: TComponent); override;
{~~$<!>}
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
{~~ Method used to pass the click of a single button to the owning parent.}
    procedure BtnClick(Index: TIB_StatementBar_ButtonType);
{~~$<*>...
The currently selected button of the button set.

Use this property in the xxxxAction events to know which button was clicked.}
    property FocusedButton: TIB_StatementBar_ButtonType read FFocusedButton;
  published
{~~$<*>...
Makes the button for Executing enabled even thoug the statement is...
not prepared.}
    property ExecuteUnprepared: boolean read FExecuteUnprepared
                                        write FExecuteUnprepared
                                        default true;
{~~$<*>...
Reference to the statement to perform actions upon.}
    property IB_Statement: TIB_Statement
        read GetStatement
       write SetStatement;
{~~$<*>...
Set which button are visible to the user.}
    property VisibleButtons: TIB_StatementBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [sbPrepare, sbUnprepare, sbExecute];
{~~$<*>...
Define custom behavior before the action of the button click is executed.}
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
{~~$<*>...
Define custom behavior after the action of the button click is executed.

This event will not be triggered if an exception was raised.}
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
{~~$<!>} property Align;
{~~$<!>} property DragCursor;
{~~$<!>} property DragMode;
{~~$<!>} property Enabled;
{~~$<!>} property Ctl3D;
{~~$<!>} property ParentCtl3D;
{~~$<!>} property ParentShowHint;
{~~$<!>} property PopupMenu;
{~~$<!>} property ShowHint;
{~~$<!>} property TabOrder;
{~~$<!>} property TabStop;
{~~$<!>} property Visible;
{~~$<!>} property OnClick;
{~~$<!>} property OnDblClick;
{~~$<!>} property OnDragDrop;
{~~$<!>} property OnDragOver;
{~~$<!>} property OnEndDrag;
{~~$<!>} property OnEnter;
{~~$<!>} property OnExit;
{~~$<!>} property OnResize;
  end;

  TIB_StatementBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_StatementBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_StatementBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$R IB_StatementBar.RES}

{------------------------------------------------------------------------------}

constructor TIB_StatementBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_StatementLink := TIB_StatementLink.Create( Self );
  with FIB_StatementLink do begin
    OnAllocatedChanged := IB_StatementAllocatedChanged;
    OnPreparedChanged := IB_StatementPreparedChanged;
  end;
  FVisibleButtons := [sbPrepare, sbUnprepare, sbExecute];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := sbPrepare;
  FExecuteUnprepared := true;
end;

procedure TIB_StatementBar.IB_StatementAllocatedChanged(
                                                  Sender: TIB_StatementLinkBase;
                                                  IB_Statement: TIB_Statement);
begin
  ActiveChanged;
end;

procedure TIB_StatementBar.IB_StatementPreparedChanged(
                                                  Sender: TIB_StatementLinkBase;
                                                  IB_Statement: TIB_Statement);
begin
  ActiveChanged;
end;

function TIB_StatementBar.GetStatement: TIB_Statement;
begin
  if FIB_StatementLink <> nil then begin
    Result := FIB_StatementLink.IB_Statement;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_StatementBar.SetStatement(AValue: TIB_Statement);
begin
  if FIB_StatementLink <> nil then begin
    if FIB_StatementLink.IB_Statement <> AValue then begin
      FIB_StatementLink.IB_Statement := AValue;
      ActiveChanged;
    end;  
  end;
end;

procedure TIB_StatementBar.InitButtons;
var
  I: TIB_StatementBar_ButtonType;
  Btn: TIB_StatementBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_StatementBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      sbAllocate: begin
        Btn.Hint := 'Allocate Statement';
        Btn.Caption := 'Allocate';
      end;
      sbDeallocate: begin
        Btn.Hint := 'Deallocate Statement';
        Btn.Caption := 'Deallocate';
      end;
      sbPrepare: begin
        Btn.Hint := 'Prepare Statement';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'ST_BAR_PREPARE'  );
        Btn.NumGlyphs := 2;
      end;
      sbUnprepare: begin
        Btn.Hint := 'Unprepare Statement';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'ST_BAR_UNPREPARE'  );
        Btn.NumGlyphs := 2;
      end;
      sbExecute: begin
        Btn.Hint := 'Execute Statement';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'ST_BAR_EXECUTE'  );
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_StatementBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_StatementBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_StatementBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_StatementBar.SetVisible(Value: TIB_StatementBar_ButtonSet);
var
  I: TIB_StatementBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;


procedure TIB_StatementBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_StatementBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[sbPrepare] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_StatementBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_StatementBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_StatementBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_StatementBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_StatementBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TIB_StatementBar.ActiveChanged;
begin
  if ( IB_Statement <> nil ) and Enabled then begin
    Buttons[ sbAllocate   ].Enabled := not IB_Statement.Allocated;
    Buttons[ sbDeallocate ].Enabled :=     IB_Statement.Allocated;
    Buttons[ sbPrepare    ].Enabled := not IB_Statement.Prepared;
    Buttons[ sbUnprepare  ].Enabled :=     IB_Statement.Prepared;
    Buttons[ sbExecute    ].Enabled :=     IB_Statement.Prepared or
                                           ExecuteUnprepared;
  end else begin
    Buttons[ sbAllocate   ].Enabled := false;
    Buttons[ sbDeallocate ].Enabled := false;
    Buttons[ sbPrepare    ].Enabled := false;
    Buttons[ sbUnprepare  ].Enabled := false;
    Buttons[ sbExecute    ].Enabled := false;
  end;
end;

procedure TIB_StatementBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  ActiveChanged;
end;

procedure TIB_StatementBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_StatementBar_ButtonType;
  OldFocus: TIB_StatementBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_StatementBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_StatementBar.BtnClick(Index: TIB_StatementBar_ButtonType);
begin
  if not (csDesigning in ComponentState) and
     Assigned( FBeforeAction ) then begin
    FBeforeAction( Self );
  end;

  if (IB_Statement <> nil) then
  begin
    with IB_Statement do
    begin
      case Index of
      sbAllocate:   Allocate;
      sbDeallocate: Deallocate;
      sbPrepare:    Prepare;
      sbUnprepare:  Unprepare;
      sbExecute:    Execute;
      end;
    end;
  end;

  if not (csDesigning in ComponentState) and
     Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_StatementBar.BarClick( Sender: TObject );
begin
  BtnClick (TIB_StatementBar_Button(Sender).Index);
end;

procedure TIB_StatementBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_StatementBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
