
{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_SearchBar                                             }
{                                                                              }
{ **************************************************************************** }

{~~ This unit contains the component IB_SearchBar. }
unit IB_SearchBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,
  
  IB_Components;

type
{~~ These are the different button types available to this button bar.

sbSearch puts the dataset into dsSearching mode.
sbSave stores all of the QBE search criteria into a buffer.
sbClear clears out all QBE search criteria.
sbRecall restores from a buffer the previously saved QBE search criteria.
sbCount performs a count of all rows matching the current QBE search criteria.}
TIB_SearchBar_ButtonType = ( sbSearch,
                             sbSave,
                             sbClear,
                             sbRecall,
                             sbCount );
{~~$<!>}
TIB_SearchBar_ButtonSet = set of TIB_SearchBar_ButtonType;
{~~$<!>}
TIB_SearchBar_Button = class;
{~~ Event type used when a count has been retrieved for the current dataset.}
TRowCountEvent = procedure ( Sender: TObject; RowCount: longint ) of object;

{~~ Button bar that gives easy access and control of the searching and ...
counting capabilities of InterBase Objects datasets.}
TIB_SearchBar = class (TCustomPanel)
  private
    FIB_DataLink: TIB_DataLink;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_SearchBar_ButtonSet;
    MinBtnSize: TPoint;
    FFocusedButton: TIB_SearchBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    FRowCount: TRowCountEvent;
    function GetDataSetLink: TIB_DataSetLink;
    procedure SetDataSetLink(AValue: TIB_DataSetLink);
    function GetReceiveFocus: boolean;
    procedure SetReceiveFocus( AValue: boolean );
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_SearchBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_DataSetStateChanged( Sender: TIB_DataLink;
                                      IB_DataSetLink: TIB_DataSetLink );
    procedure BarClick(Sender: TObject); virtual;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
{~~ Storage array for the different buttons on this button bar. }
    Buttons: array[TIB_SearchBar_ButtonType] of TIB_SearchBar_Button;
    constructor Create(AOwner: TComponent); override;
{~~$<!>}
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
{~~ Method used to pass the click of a single button to the owning parent.}
    procedure BtnClick(Index: TIB_SearchBar_ButtonType);
{~~$<*>...
The currently selected button of the button set.

Use this property in the xxxxAction events to know which button was clicked.}
    property FocusedButton: TIB_SearchBar_ButtonType read FFocusedButton;
  published
{~~$<*>...
Reference to the dataset link to perform actions upon.}
    property IB_DataSetLink: TIB_DataSetLink
        read GetDataSetLink
       write SetDataSetLink;
{~~$<*>...
This property is used in conjunction with the AnnounceFocus property of...
the IB_DataSetLink component. It is a global focusing system that allows...
the SearchBar to link itself to whereever the user's focus goes.

For example, in an MDI application the user could go from one child window...
to another. Each child window would have its own IB_DataSetLink context....
Thus, if each child window's IB_DataSetLink component's AnnounceFocus...
property is set to true the SearchBar will receive the focus and align with it.}
    property ReceiveFocus: boolean read GetReceiveFocus write SetReceiveFocus;
{~~$<*>...
Set which button are visible to the user.}
    property VisibleButtons: TIB_SearchBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [ sbSearch, sbCount, sbSave, sbClear, sbRecall ];
{~~$<*>...
Define custom behavior before the action of the button click is executed.}
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
{~~$<*>...
Define custom behavior after the action of the button click is executed.

This event will not be triggered if an exception was raised.}
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
{~~$<*>...
Event that enables a custom display of the row count for the dataset.}
    property OnRowCount: TRowCountEvent read FRowCount write FRowCount;
{~~$<!>} property Align;
{~~$<!>} property DragCursor;
{~~$<!>} property DragMode;
{~~$<!>} property Enabled;
{~~$<!>} property Ctl3D;
{~~$<!>} property ParentCtl3D;
{~~$<!>} property ParentShowHint;
{~~$<!>} property PopupMenu;
{~~$<!>} property ShowHint;
{~~$<!>} property TabOrder;
{~~$<!>} property TabStop;
{~~$<!>} property Visible;
{~~$<!>} property OnClick;
{~~$<!>} property OnDblClick;
{~~$<!>} property OnDragDrop;
{~~$<!>} property OnDragOver;
{~~$<!>} property OnEndDrag;
{~~$<!>} property OnEnter;
{~~$<!>} property OnExit;
{~~$<!>} property OnResize;
  end;

{~~$<!>}
  TIB_SearchBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_SearchBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_SearchBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$R IB_SearchBar.RES}

{------------------------------------------------------------------------------}

constructor TIB_SearchBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_DataLink := TIB_DataLink.Create( Self );
  with FIB_DataLink do begin
    AfterAssignment := IB_DataSetStateChanged;
    OnStateChanged := IB_DataSetStateChanged;
  end;
  FVisibleButtons := [ sbSearch, sbCount, sbSave, sbClear, sbRecall ];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := sbSearch;
end;

procedure TIB_SearchBar.IB_DataSetStateChanged( Sender: TIB_DataLink;
                                              IB_DataSetLink: TIB_DataSetLink );
begin
  ActiveChanged;
end;

function TIB_SearchBar.GetDataSetLink: TIB_DataSetLink;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.IB_DataSetLink;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_SearchBar.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  if FIB_DataLink <> nil then begin
    FIB_DataLink.IB_DataSetLink := AValue;
  end;
end;

function TIB_SearchBar.GetReceiveFocus: boolean;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.ReceiveFocus;
  end else begin
    Result := false;
  end;
end;

procedure TIB_SearchBar.SetReceiveFocus( AValue: boolean );
begin
  if FIB_DataLink <> nil then begin
    FIB_DataLink.ReceiveFocus := AValue;
  end;
end;

procedure TIB_SearchBar.InitButtons;
var
  I: TIB_SearchBar_ButtonType;
  Btn: TIB_SearchBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_SearchBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      sbSearch: begin
        Btn.Hint := 'Search DataSet';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'SRCH_BAR_SEARCH'  );
        Btn.NumGlyphs := 2;
      end;
      sbCount: begin
        Btn.Hint := 'Count Rows';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'SRCH_BAR_COUNT'  );
        Btn.NumGlyphs := 2;
      end;
      sbSave: begin
        Btn.Hint := 'Save Criteria';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'SRCH_BAR_SAVE'  );
        Btn.NumGlyphs := 2;
      end;
      sbClear: begin
        Btn.Hint := 'Clear Criteria';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'SRCH_BAR_CLEAR'  );
        Btn.NumGlyphs := 2;
      end;
      sbRecall: begin
        Btn.Hint := 'Recall Criteria';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'SRCH_BAR_RECALL'  );
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_SearchBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_SearchBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_SearchBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_SearchBar.SetVisible(Value: TIB_SearchBar_ButtonSet);
var
  I: TIB_SearchBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TIB_SearchBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_SearchBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[ sbSearch ] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_SearchBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_SearchBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_SearchBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_SearchBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_SearchBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TIB_SearchBar.ActiveChanged;
begin
  if FIB_DataLink = nil then Exit;
  if ( FIB_DataLink.IB_DataSet <> nil ) and
     ( Enabled ) then with IB_DataSetLink.IB_DataSet do begin
    if not Prepared or
       ( StatementType = stSelect ) or
       ( StatementType = stSelectForUpdate ) then begin
      Buttons[ sbSearch ].Enabled := DataSetState <> dsSearching;
      Buttons[ sbCount  ].Enabled := Prepared;
      Buttons[ sbSave   ].Enabled := DataSetState = dsSearching;
      Buttons[ sbClear  ].Enabled := DataSetState = dsSearching;
      Buttons[ sbRecall ].Enabled := DataSetState = dsSearching;
    end else begin
      Buttons[ sbSearch ].Enabled := false;
      Buttons[ sbCount  ].Enabled := false;
      Buttons[ sbSave   ].Enabled := false;
      Buttons[ sbClear  ].Enabled := false;
      Buttons[ sbRecall ].Enabled := false;
    end;
  end else begin
    Buttons[ sbSearch ].Enabled := false;
    Buttons[ sbCount  ].Enabled := false;
    Buttons[ sbSave   ].Enabled := false;
    Buttons[ sbClear  ].Enabled := false;
    Buttons[ sbRecall ].Enabled := false;
  end;
end;

procedure TIB_SearchBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  ActiveChanged;
end;

procedure TIB_SearchBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_SearchBar_ButtonType;
  OldFocus: TIB_SearchBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_SearchBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_SearchBar.BtnClick(Index: TIB_SearchBar_ButtonType);
begin
  if not (csDesigning in ComponentState) and
     Assigned( FBeforeAction ) then begin
    FBeforeAction( Self );
  end;

  if (IB_DataSetLink <> nil) then
  begin
    with IB_DataSetLink do begin
      if IB_DataSet <> nil then with IB_DataSet do begin
        case Index of
          sbSearch: Search;
          sbCount: begin
            if Assigned( FRowCount ) then begin
              FRowCount( Self, Count );
            end else begin
              ShowMessage( 'Rows counted: ' + IntToStr( Count ));
            end;
          end;
          sbSave: begin
            case DataSetState of
              dsSearching: SaveSearch;
            end;
          end;
          sbClear: begin
            case DataSetState of
              dsSearching: ClearSearch;
            end;
          end;
          sbRecall: begin
            case DataSetState of
              dsSearching: RecallSearch;
            end;
          end;
        end;
      end;
    end;
  end;

  if not (csDesigning in ComponentState) and
     Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_SearchBar.BarClick( Sender: TObject );
begin
  BtnClick (TIB_SearchBar_Button(Sender).Index);
end;

procedure TIB_SearchBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_SearchBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
