
{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_NavigationBar                                         }
{                                                                              }
{ **************************************************************************** }

{~~ This unit contains the component IB_NavigationBar. }
unit IB_NavigationBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,

  IB_Components;

type
{~~ These are the different button types available to this button bar.

nbFirst goes to the first row in the dataset.
nbJumpBck jumps back a predefined number of rows in the dataset.
nbPrior goes to the previous row in the dataset.
nbNext goes to the next row in the dataset.
nbJumpFwd jumps forward a predefined number of rows in the dataset.
nbLast goes to the end of the dataset.
}
  TIB_NavigationBar_ButtonType = ( nbFirst,
                                   nbJumpBck,
                                   nbPrior,
                                   nbNext,
                                   nbJumpFwd,
                                   nbLast );

{~~$<!>}
  TIB_NavigationBar_ButtonSet = set of TIB_NavigationBar_ButtonType;
{~~$<!>}
  TIB_NavigationBar_Button = class;

{~~ Button bar that gives access and control of navigating through a dataset.}
TIB_NavigationBar = class (TCustomPanel)
  private
    FIB_DataLink: TIB_DataLink;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_NavigationBar_ButtonSet;
    MinBtnSize: TPoint;
    FFocusedButton: TIB_NavigationBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    function GetDataSetLink: TIB_DataSetLink;
    procedure SetDataSetLink(AValue: TIB_DataSetLink);
    function GetReceiveFocus: boolean;
    procedure SetReceiveFocus(AValue: boolean);
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_NavigationBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_DataSetRowStateChanged( Sender: TIB_DataLink;
                                         IB_DataSetLink: TIB_DataSetLink );
    procedure IB_DataSetRowDataChanged( Sender: TIB_DataLink;
                                        IB_DataSetLink: TIB_DataSetLink;
                                        IB_Column: TIB_Column );
    procedure BarClick(Sender: TObject); virtual;
    procedure SysStateChanged; virtual;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
{~~ Storage array for the different buttons on this button bar. }
    Buttons: array[TIB_NavigationBar_ButtonType] of TIB_NavigationBar_Button;
    constructor Create(AOwner: TComponent); override;
{~~$<!>}
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
{~~ Method used to pass the click of a single button to the owning parent.}
    procedure BtnClick(Index: TIB_NavigationBar_ButtonType);
{~~$<*>...
The currently selected button of the button set.

Use this property in the xxxxAction events to know which button was clicked.}
    property FocusedButton: TIB_NavigationBar_ButtonType read FFocusedButton;
  published
{~~$<*>...
Reference to the dataset link to perform actions upon.}
    property IB_DataSetLink: TIB_DataSetLink
        read GetDataSetLink
       write SetDataSetLink;
{~~$<*>...
This property is used in conjunction with the AnnounceFocus property of...
the IB_DataSetLink component. It is a global focusing system that allows...
the SearchBar to link itself to whereever the user's focus goes.

For example, in an MDI application the user could go from one child window...
to another. Each child window would have its own IB_DataSetLink context....
Thus, if each child window's IB_DataSetLink component's AnnounceFocus...
property is set to true the SearchBar will receive the focus and align with it.}
    property ReceiveFocus: boolean read GetReceiveFocus write SetReceiveFocus; 
{~~$<*>...
Set which button are visible to the user.}
    property VisibleButtons: TIB_NavigationBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [ nbFirst, nbPrior, nbNext, nbLast ];
{~~$<*>...
Define custom behavior before the action of the button click is executed.}
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
{~~$<*>...
Define custom behavior after the action of the button click is executed.

This event will not be triggered if an exception was raised.}
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
{~~$<!>} property Align;
{~~$<!>} property DragCursor;
{~~$<!>} property DragMode;
{~~$<!>} property Enabled;
{~~$<!>} property Ctl3D;
{~~$<!>} property ParentCtl3D;
{~~$<!>} property ParentShowHint;
{~~$<!>} property PopupMenu;
{~~$<!>} property ShowHint;
{~~$<!>} property TabOrder;
{~~$<!>} property TabStop;
{~~$<!>} property Visible;
{~~$<!>} property OnClick;
{~~$<!>} property OnDblClick;
{~~$<!>} property OnDragDrop;
{~~$<!>} property OnDragOver;
{~~$<!>} property OnEndDrag;
{~~$<!>} property OnEnter;
{~~$<!>} property OnExit;
{~~$<!>} property OnResize;
  end;

  TIB_NavigationBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_NavigationBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_NavigationBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$IFDEF WIN32}
{$R IB_NavigationBar.RES}
{$ENDIF}

{------------------------------------------------------------------------------}

constructor TIB_NavigationBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_DataLink := TIB_DataLink.Create( Self );
  with FIB_DataLink do begin
    AfterAssignment := IB_DataSetRowStateChanged;
    OnPreparedChanged := IB_DataSetRowStateChanged;
    OnActiveChanged := IB_DataSetRowStateChanged;
    OnStateChanged := IB_DataSetRowStateChanged;
    OnDataChanged := IB_DataSetRowDataChanged;
  end;
  FVisibleButtons := [ nbFirst,   nbPrior,  nbNext, nbLast ];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := nbFirst;
end;

procedure TIB_NavigationBar.IB_DataSetRowStateChanged( Sender: TIB_DataLink;
                                              IB_DataSetLink: TIB_DataSetLink );
begin
  SysStateChanged;
end;

procedure TIB_NavigationBar.IB_DataSetRowDataChanged( Sender: TIB_DataLink;
                                                IB_DataSetLink: TIB_DataSetLink;
                                                IB_Column: TIB_Column );
begin
  SysStateChanged;
end;

function TIB_NavigationBar.GetDataSetLink: TIB_DataSetLink;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.IB_DataSetLink;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_NavigationBar.SetDataSetLink(AValue: TIB_DataSetLink);
begin
  if FIB_DataLink <> nil then begin
    if FIB_DataLink.IB_DataSetLink <> AValue then begin
      FIB_DataLink.IB_DataSetLink := AValue;
      SysStateChanged;
    end;
  end;
end;

function TIB_NavigationBar.GetReceiveFocus: boolean;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.ReceiveFocus;
  end else begin
    Result := false;
  end;
end;

procedure TIB_NavigationBar.SetReceiveFocus(AValue: boolean);
begin
  if FIB_DataLink <> nil then begin
    FIB_DataLink.ReceiveFocus := AValue;
  end;
end;

procedure TIB_NavigationBar.InitButtons;
var
  I: TIB_NavigationBar_ButtonType;
  Btn: TIB_NavigationBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_NavigationBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      nbFirst: begin
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'NAV_BAR_FIRST'  );
        Btn.NumGlyphs := 2;
      end;
      nbJumpBck: begin
        Btn.Hint := 'Jump Backwards Through DataSet';
        Btn.Caption := 'Jump Backward';
      end;
      nbPrior: begin
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'NAV_BAR_PRIOR'  );
        Btn.NumGlyphs := 2;
      end;
      nbNext: begin
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'NAV_BAR_NEXT'  );
        Btn.NumGlyphs := 2;
      end;
      nbJumpFwd: begin
        Btn.Hint := 'Jump Forwards Through DataSet';
        Btn.Caption := 'Jump Forward';
      end;
      nbLast: begin
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'NAV_BAR_LAST'  );
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_NavigationBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_NavigationBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_NavigationBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_NavigationBar.SetVisible(Value: TIB_NavigationBar_ButtonSet);
var
  I: TIB_NavigationBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;


procedure TIB_NavigationBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_NavigationBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_NavigationBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_NavigationBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_NavigationBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_NavigationBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_NavigationBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    SysStateChanged;
end;

procedure TIB_NavigationBar.SysStateChanged;
begin
  if ( IB_DataSetLink <> nil ) and
     ( IB_DataSetLink.IB_DataSet <> nil ) and
     ( Enabled ) then with IB_DataSetLink.IB_DataSet do begin
    if not Prepared or
       ( StatementType = stSelect ) or
       ( StatementType = stSelectForUpdate ) then begin
      if Unidirectional then begin
        Buttons[ nbFirst   ].Enabled := Prepared;
        Buttons[ nbJumpBck ].Enabled := false;
        Buttons[ nbPrior   ].Enabled := false;
        Buttons[ nbNext    ].Enabled := Prepared and not EOF;
        Buttons[ nbJumpFwd ].Enabled := Prepared and not EOF;
        Buttons[ nbLast    ].Enabled := Prepared and not EOF;
      end else begin
        Buttons[ nbFirst   ].Enabled := Active             and ( RowNum <> 1 );
        Buttons[ nbJumpBck ].Enabled := Active and not BOF and ( RowNum  > 1 );
        Buttons[ nbPrior   ].Enabled := Active and not BOF and ( RowNum  > 1 );
        Buttons[ nbNext    ].Enabled := Active and not EOF;
        Buttons[ nbJumpFwd ].Enabled := Active and not EOF;
        Buttons[ nbLast    ].Enabled := Active;
      end;
    end else begin
      Buttons[ nbFirst   ].Enabled := false;
      Buttons[ nbJumpBck ].Enabled := false;
      Buttons[ nbPrior   ].Enabled := false;
      Buttons[ nbNext    ].Enabled := false;
      Buttons[ nbJumpFwd ].Enabled := false;
      Buttons[ nbLast    ].Enabled := false;
    end;
  end else begin
    Buttons[ nbFirst   ].Enabled := false;
    Buttons[ nbJumpBck ].Enabled := false;
    Buttons[ nbPrior   ].Enabled := false;
    Buttons[ nbNext    ].Enabled := false;
    Buttons[ nbJumpFwd ].Enabled := false;
    Buttons[ nbLast    ].Enabled := false;
  end;
  if ( IB_DataSetLink <> nil ) and
     ( IB_DataSetLink.IB_DataSet <> nil ) then begin
    with IB_DataSetLink.IB_DataSet do begin
      if HintFirst <> '' then begin
        Buttons[ nbFirst ].Hint := HintFirst;
      end else begin
        Buttons[ nbFirst ].Hint := 'Goto First Row In DataSet';
      end;
      if HintPrior <> '' then begin
        Buttons[ nbPrior ].Hint := HintPrior;
      end else begin
        Buttons[ nbPrior ].Hint := 'Goto Prior Row In DataSet';
      end;
      if HintNext <> '' then begin
        Buttons[ nbNext ].Hint := HintNext;
      end else begin
        Buttons[ nbNext ].Hint := 'Goto Next Row In DataSet';
      end;
      if HintLast <> '' then begin
        Buttons[ nbLast ].Hint := HintLast;
      end else begin
        Buttons[ nbLast ].Hint := 'Goto Last Row In DataSet';
      end;
    end;
  end;
end;

procedure TIB_NavigationBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  SysStateChanged;
end;

procedure TIB_NavigationBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_NavigationBar_ButtonType;
  OldFocus: TIB_NavigationBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_NavigationBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_NavigationBar.BtnClick(Index: TIB_NavigationBar_ButtonType);
begin
  if Assigned( BeforeAction ) then begin
    BeforeAction( Self );
  end;
  if (IB_DataSetLink <> nil) then
  begin
    with IB_DataSetLink do begin
      if IB_DataSet <> nil then with IB_DataSet do begin
        case Index of
          nbFirst:    First;
          nbJumpBck:  MoveBy( -5 );
          nbPrior:    Prior;
          nbNext:     Next;
          nbJumpFwd:  MoveBy(  5 );
          nbLast:     Last;
        end;
      end;
    end;
  end;
  if Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_NavigationBar.BarClick( Sender: TObject );
begin
  BtnClick (TIB_NavigationBar_Button(Sender).Index);
end;

procedure TIB_NavigationBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_NavigationBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
