{~~ This unit contains the IB_Monitor component.}
unit IB_Monitor;

interface

uses
  Classes, Forms, SysUtils, IB_Header, IB_Components;

type
{~~...
}
TIB_MonitorGroups = ( mgConnection,
                      mgTransaction,
                      mgStatement,
                      mgRow,
                      mgBlob,
                      mgArray );
{~~$<!>}
TIB_MonitorGroupSet = set of TIB_MonitorGroups;

{~~...
}
TIB_ConnectionGroup = ( cgCreate,
                        cgConnect,
                        cgDisconnect,
                        cgDrop );
{~~$<!>}
TIB_ConnectionGroupSet = set of TIB_ConnectionGroup;

{~~...
}
TIB_TransactionGroup = ( tgStart,
                         tgSavepoint,
                         tgCommit,
                         tgRollback );
{~~$<!>}
TIB_TransactionGroupSet = set of TIB_TransactionGroup;

{~~...
}
TIB_StatementGroup = ( sgAllocate,
                       sgPrepare,
                       sgDescribe,
                       sgStatementInfo,
                       sgExecute,
                       sgExecuteImmediate,
                       sgIBCursor );
{~~$<!>}
TIB_StatementGroupSet = set of TIB_StatementGroup;

TIB_RowGroup = ( rgFetch );
{~~$<!>}
TIB_RowGroupSet = set of TIB_RowGroup;                       

{~~...
Event that gives acces to each peice of information generated by the monitor.}
TIB_MonitorNewString = procedure ( Sender: TObject;
                                   NewString: string ) of object;

{~~...
}
TIB_Monitor = class ( TComponent )
  private
//    function GetIB_Session: TIB_Session; virtual;
//    procedure SetIB_Session( AValue: TIB_Session ); virtual;
  protected
    FEnabled: boolean;
//    FIB_Session: TIB_Session;
    FIB_MonitorGroups: TIB_MonitorGroupSet;
    FIB_ConnectionGroups: TIB_ConnectionGroupSet;
    FIB_TransactionGroups: TIB_TransactionGroupSet;
    FIB_StatementGroups: TIB_StatementGroupSet;
    FIB_RowGroups: TIB_RowGroupSet;
    FOnMonitorOutputItem: TIB_MonitorNewString;
    procedure SetEnabled( AValue: boolean ); virtual;
    procedure SysAddString( NewString: string ); virtual;
    procedure TakeOutMonitorHooks; virtual;
    procedure PutInMonitorHooks; virtual;
    procedure StoreOriginalHooks; virtual;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
//    property IB_Session: TIB_Session read GetIB_Session;
  published
  { Properties }
    property Enabled: boolean read FEnabled
                              write SetEnabled
                              default true;
    property IB_MonitorGroups: TIB_MonitorGroupSet read FIB_MonitorGroups
                                                   write FIB_MonitorGroups
                                                   default [ mgConnection,
                                                             mgTransaction,
                                                             mgStatement,
                                                             mgRow ];
    property IB_StatementGroups: TIB_StatementGroupSet read FIB_StatementGroups
                                                       write FIB_StatementGroups
                                                       default [ sgPrepare,
                                                                 sgExecute,
                                                           sgExecuteImmediate ];

  { Events }
    property OnMonitorOutputItem: TIB_MonitorNewString
        read FOnMonitorOutputItem
       write FOnMonitorOutputItem;
  end;

implementation

var
  Unit_MonitorList: TList;
  Unit_Enabled_Count: integer;
  Unit_Hooks_Active: boolean;

{------------------------------------------------------------------------------}

constructor TIB_Monitor.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  if Unit_MonitorList = nil then begin
    Unit_MonitorList := TList.Create;
  end;
  Unit_MonitorList.Add( Self );
  FIB_MonitorGroups := [ mgConnection, mgTransaction, mgStatement, mgRow ];
  FIB_ConnectionGroups := [ cgCreate, cgConnect, cgDisconnect, cgDrop ];
  FIB_TransactionGroups := [ tgStart, tgSavepoint, tgCommit, tgRollback ];
  FIB_StatementGroups := [ sgPrepare, sgExecute, sgExecuteImmediate ];
  FIB_RowGroups := [ rgFetch ];
  StoreOriginalHooks;
  Enabled := true; // Not the syntax: FEnabled! Must hit the setter method!
end;

destructor TIB_Monitor.Destroy;
begin
  Enabled := false;   //  Please leave this here! Must hit the setter method!
  Unit_MonitorList.Remove( Self );
  Destroying;
  if Unit_MonitorList.Count = 0 then begin
    Unit_MonitorList.Free;
    Unit_MonitorList := nil;
  end;
  inherited Destroy;
end;

{
function TIB_Monitor.GetIB_Session: TIB_Session;
begin
  if FIB_Session = nil then begin
    Result := IB_Components.DefaultIB_Session;
  end else begin
    Result := FIB_Session;
  end;
end;

procedure TIB_Monitor.SetIB_Session( AValue: TIB_Session );
begin
  if IB_Session <> AValue then begin
    if ( AValue = nil ) or
       ( AValue = IB_Components.DefaultIB_Session ) then begin
      FIB_Session := nil;
    end else begin
      FIB_Session := AValue;
    end;
  end;
end;
}

procedure TIB_Monitor.SysAddString( NewString: string );
begin
  if Assigned( FOnMonitorOutputItem ) then begin
    FOnMonitorOutputItem( Self, NewString );
  end;
end;

procedure TIB_Monitor.SetEnabled( AValue: boolean );
begin
  if Enabled <> AValue then begin
    FEnabled := AValue;
    if Enabled then begin
      Inc( Unit_Enabled_Count,  1 );
    end else begin
      Inc( Unit_Enabled_Count, -1 );
    end;
    if Unit_Enabled_Count = 0 then begin
      TakeOutMonitorHooks;
    end else if not Unit_Hooks_Active then begin
      PutInMonitorHooks;
    end;
  end;
end;

{------------------------------------------------------------------------------}

var
  isc_mon_attach_database:          Tisc_attach_database;
//isc_mon_attach_service:           Tisc_attach_service;
//isc_mon_array_gen_sdl:            Tisc_array_gen_sdl;
//isc_mon_array_get_slice:          Tisc_array_get_slice;
//isc_mon_array_lookup_bounds:      Tisc_array_lookup_bounds;
//isc_mon_array_lookup_desc:        Tisc_array_lookup_desc;
//isc_mon_array_set_desc:           Tisc_array_set_desc;
//isc_mon_array_put_slice:          Tisc_array_put_slice;
  isc_mon_blob_default_desc:        Tisc_blob_default_desc;
  isc_mon_blob_gen_bpb:             Tisc_blob_gen_bpb;
  isc_mon_blob_info:                Tisc_blob_info;
  isc_mon_blob_lookup_desc:         Tisc_blob_lookup_desc;
  isc_mon_blob_set_desc:            Tisc_blob_set_desc;
  isc_mon_cancel_blob:              Tisc_cancel_blob;
  isc_mon_cancel_events:            Tisc_cancel_events;
  isc_mon_close_blob:               Tisc_close_blob;
  isc_mon_commit_retaining:         Tisc_commit_retaining;
  isc_mon_commit_transaction:       Tisc_commit_transaction;
//isc_mon_compile_request:          Tisc_compile_request;
//isc_mon_compile_request2:         Tisc_compile_request2;
  isc_mon_create_blob:              Tisc_create_blob;
  isc_mon_create_blob2:             Tisc_create_blob2;
  isc_mon_create_database:          Tisc_create_database;
  isc_mon_database_info:            Tisc_database_info;
//isc_mon_ddl:                      Tisc_ddl;
//isc_mon_decode_date:              Tisc_decode_date;
  isc_mon_detach_database:          Tisc_detach_database;
//isc_mon_detach_service:           Tisc_detach_service;
  isc_mon_drop_database:            Tisc_drop_database;
//isc_mon_encode_date:              Tisc_encode_date;
//isc_mon_event_block:              Tisc_event_block;
//isc_mon_event_counts:             Tisc_event_counts;
//isc_mon_expand_dpb:               Tisc_expand_dpb;
//isc_mon_modify_dpb:               Tisc_modify_dpb;
//isc_mon_free:                     Tisc_free;
  isc_mon_get_segment:              Tisc_get_segment;
//isc_mon_get_slice:                Tisc_get_slice;
  isc_mon_interprete:               Tisc_interprete;
  isc_mon_open_blob:                Tisc_open_blob;
  isc_mon_open_blob2:               Tisc_open_blob2;
//isc_mon_prepare_transaction:      Tisc_prepare_transaction;
//isc_mon_prepare_transaction2:     Tisc_prepare_transaction2;
//isc_mon_print_sqlerror:           Tisc_print_sqlerror;
//isc_mon_print_status:             Tisc_print_status;
  isc_mon_put_segment:              Tisc_put_segment;
//isc_mon_put_slice:                Tisc_put_slice;
//isc_mon_que_events:               Tisc_que_events;
//isc_mon_query_service:            Tisc_query_service;
//isc_mon_receive:                  Tisc_receive;
//isc_mon_reconnect_transaction:    Tisc_reconnect_transaction;
//isc_mon_release_request:          Tisc_release_request;
//isc_mon_request_info:             Tisc_request_info;
  isc_mon_rollback_transaction:     Tisc_rollback_transaction;
//isc_mon_seek_blob:                Tisc_seek_blob;
//isc_mon_send:                     Tisc_send;
//isc_mon_start_and_send:           Tisc_start_and_send;
  isc_mon_start_multiple:           Tisc_start_multiple;
//isc_mon_start_request:            Tisc_start_request;
//isc_mon_sqlcode:                  Tisc_sqlcode;
//isc_mon_transaction_info:         Tisc_transaction_info;
//isc_mon_transact_request:         Tisc_transact_request;
//isc_mon_unwind_request:           Tisc_unwind_request;
//isc_mon_wait_for_event:           Tisc_wait_for_event;
//isc_mon_ftof:                     Tisc_ftof;
//isc_mon_print_blr:                Tisc_print_blr;
//isc_mon_set_debug:                Tisc_set_debug;
//isc_mon_qtoq:                     Tisc_qtoq;
//isc_mon_vax_integer:              Tisc_vax_integer;
//isc_mon_vtof:                     Tisc_vtof;
//isc_mon_vtov:                     Tisc_vtov;
//isc_mon_version:                  Tisc_version;
  isc_mon_dsql_execute:              Tisc_dsql_execute;
  isc_mon_dsql_execute2:             Tisc_dsql_execute2;
  isc_mon_dsql_execute_immediate:    Tisc_dsql_execute_immediate;
  isc_mon_dsql_exec_immed2:          Tisc_dsql_exec_immed2;
  isc_mon_dsql_allocate_statement:   Tisc_dsql_allocate_statement;
  isc_mon_dsql_alloc_statement2:     Tisc_dsql_alloc_statement2;
  isc_mon_dsql_free_statement:       Tisc_dsql_free_statement;
  isc_mon_dsql_prepare:              Tisc_dsql_prepare;
  isc_mon_dsql_insert:               Tisc_dsql_insert;
  isc_mon_dsql_set_cursor_name:      Tisc_dsql_set_cursor_name;
  isc_mon_dsql_describe:             Tisc_dsql_describe;
  isc_mon_dsql_describe_bind:        Tisc_dsql_describe_bind;
  isc_mon_dsql_fetch:                Tisc_dsql_fetch;
  isc_mon_dsql_fetch2:               Tisc_dsql_fetch2;
  isc_mon_dsql_sql_info:             Tisc_dsql_sql_info;
//isc_mon_event_block_asm:           Tisc_event_block_asm;

{------------------------------------------------------------------------------}

function mon_attach_database ( status:          pstatus_vector;
                               db_name_len:     short;
                               db_name:         pchar;
                               db_handle:       pisc_db_handle;
                               parm_buffer_len: short;
                               parm_buffer:     pchar
                               ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CONNECT DATABASE ' + db_name );
    end;
    end;
  end;
  Result := isc_mon_attach_database ( status,
                                      db_name_len,
                                      db_name,
                                      db_handle,
                                      parm_buffer_len,
                                      parm_buffer );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(db_handle^)));
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_create_database ( status:          pstatus_vector;
			       db_name_len:     short;
			       db_name:         pchar;
			       pdb_handle:      pisc_db_handle;
			       parm_buffer_len: short;
			       parm_buffer:     pchar;
			       f:               short
                               ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CREATE DATABASE ' + db_name );
      SysAddString( '    parm_buffer: ' + parm_buffer );
    end;
    end;
  end;
  Result := isc_mon_create_database ( status,
                                      db_name_len,
                                      db_name,
                                      pdb_handle,
                                      parm_buffer_len,
                                      parm_buffer,
                                      f );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(pdb_handle^)));
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_detach_database ( status:    pstatus_vector;
                               db_handle: pisc_db_handle
                               ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'DISCONNECT DATABASE' );
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(db_handle^)));
    end;
    end;
  end;
  Result := isc_mon_detach_database ( status, db_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_drop_database ( status:    pstatus_vector;
                             db_handle: pisc_db_handle
                             ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'DROP DATABASE' );
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(db_handle^)));
    end;
    end;
  end;
  Result := isc_mon_drop_database ( status, db_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_database_info ( status:          pstatus_vector;
				 pdb_handle:	          pisc_db_handle;
				 b:	          short;
				 c:	          pchar;
				 d:	          short;
				 e:	          pchar
                                 ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'DATABASE INFO' );
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(pdb_handle^)));
    end;
    end;
  end;
  Result := isc_mon_database_info ( status, pdb_handle, b, c, d, e );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgConnection in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function mon_commit_retaining ( status:    pstatus_vector;
                                tr_handle: pisc_tr_handle
                                ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'SAVEPOINT TRANSACTION' );
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
    end;
    end;
  end;
  Result := isc_mon_commit_retaining ( status, tr_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_commit_transaction ( status:    pstatus_vector;
                                  tr_handle: pisc_tr_handle
                                  ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'COMMIT TRANSACTION' );
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
    end;
    end;
  end;
  Result := isc_mon_commit_transaction ( status, tr_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_rollback_transaction ( status:    pstatus_vector;
                                    tr_handle: pisc_tr_handle
                                    ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'ROLLBACK TRANSACTION' );
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
    end;
    end;
  end;
  Result := isc_mon_rollback_transaction ( status, tr_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgTransaction in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_start_multiple ( status:             pstatus_vector;
                              tr_handle:          pisc_tr_handle;
                              db_handle_count:    short;
                              teb_vector_address: pointer
                              ):                  isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgStatement in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'START TRANSACTION' );
    end;
    end;
  end;
  Result := isc_mon_start_multiple ( status,
                                     tr_handle,
                                     db_handle_count,
                                     teb_vector_address );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgStatement in IB_MonitorGroups then begin
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( 'SQLCODE = '   + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

// sgAllocate,
// sgPrepare,
// sgDescribe,
// sgStatementInfo,
// sgExecute,
// sgExecuteImmediate,
// sgIBCursor

{------------------------------------------------------------------------------}

function mon_dsql_allocate_statement ( status:    pstatus_vector;
                                       pdb_handle: pisc_db_handle;
                                       pst_handle: pisc_stmt_handle
                                       ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgAllocate in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'ALLOCATE STATEMENT');
    end;
    end;
  end;
  Result := isc_mon_dsql_allocate_statement ( status,
                                              pdb_handle,
                                              pst_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgAllocate in IB_StatementGroups ) then begin
      SysAddString( 'PSTMT_HANDLE = ' + IntToStr( Integer(pst_handle)));
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(pst_handle^)));
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_alloc_statement2 ( status:       pstatus_vector;
                                     pdb_handle:   pisc_db_handle;
				     pstmt_handle: pisc_stmt_handle
                                     ):            isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgAllocate in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'ALLOCATE STATEMENT 2');
    end;
    end;
  end;
  Result := isc_mon_dsql_alloc_statement2 ( status,
                                            pdb_handle,
                                            pstmt_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgAllocate in IB_StatementGroups ) then begin
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(pstmt_handle^)));
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_free_statement ( status_vector: pstatus_vector;
                                   st_handle:     pisc_stmt_handle;
                                   option:        smallint
                                   ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       (( sgIBCursor in IB_StatementGroups ) or
        ( sgAllocate in IB_StatementGroups )) then begin
      SysAddString( '--------------------------------------------------------');
      if option = DSQL_CLOSE then begin
        if ( sgIBCursor in IB_StatementGroups ) then begin
        SysAddString( 'CLOSE CURSOR');
        end;
      end else begin
        if ( sgAllocate in IB_StatementGroups ) then begin
        SysAddString( 'DEALLOCATE STATEMENT');
        end;
      end;
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_free_statement ( status_vector,
                                          st_handle,
                                          option );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       (( sgIBCursor in IB_StatementGroups ) or
        ( sgAllocate in IB_StatementGroups )) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_prepare ( status:    pstatus_vector;
                            tr_handle: pisc_tr_handle;
                            st_handle: pisc_stmt_handle;
                            len:       smallint;
                            statement: pchar;
                            dialect:   short;
                            params:    PXSQLDA
                            ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgPrepare   in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'PREPARE STATEMENT');
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( 'PSTMT_HANDLE = ' + IntToStr( Integer(st_handle)));
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
      SysAddString( '' );
      SysAddString( statement );
      SysAddString( '' );
    end;
    end;
  end;
  Result := isc_mon_dsql_prepare ( status,
                                   tr_handle,
                                   st_handle,
                                   len,
                                   statement,
                                   dialect,
                                   params);
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgPrepare   in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_execute ( status:    pstatus_vector;
                            tr_handle: pisc_tr_handle;
                            st_handle: pisc_stmt_handle;
                            dialect:   smallint;
                            params:    PXSQLDA
                            ):         isc_status; stdcall;
var
  ii: integer;
  kk: integer;
  paramalias: string;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecute   in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'EXECUTE STATEMENT');
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_execute ( status,
                                   tr_handle,
                                   st_handle,
                                   dialect,
                                   params );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecute   in IB_StatementGroups ) then begin
      ParamAlias := '';
      if params <> nil then with params^ do begin
        for kk := 0 to sqln - 1 do begin
          if kk > 0 then begin
            ParamAlias := ParamAlias + ', ';
          end;
          ParamAlias := ParamAlias + Trim( SQLVar[ kk ].AliasName );
        end;
      end;
      SysAddString( 'INPUT PARAMETERS = [ ' +  ParamAlias + ' ]');
      SysAddString( 'OUTPUT PARAMETERS = [  ]');
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_execute2 ( status:    pstatus_vector;
                             tr_handle: pisc_tr_handle;
                             st_handle: pisc_stmt_handle;
                             a:         short;
                             inparams:  PXSQLDA;
                             outparams: PXSQLDA
                             ):         isc_status; stdcall;
var
  ii: integer;
  kk: integer;
  paramalias: string;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'EXECUTE STATEMENT 2');
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_execute2 ( status,
                                   tr_handle,
                                   st_handle,
                                   a,
                                   inparams,
                                   outparams );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      ParamAlias := '';
      if inparams <> nil then with inparams^ do begin
        for kk := 0 to sqln - 1 do begin
          if kk > 0 then begin
            ParamAlias := ParamAlias + ', ';
          end;
          ParamAlias := ParamAlias + Trim( SQLVar[ kk ].AliasName );
        end;
      end;
      SysAddString( 'INPUT PARAMETERS = [ ' +  ParamAlias + ' ]');
      ParamAlias := '';
      if outparams <> nil then with outparams^ do begin
        for kk := 0 to sqln - 1 do begin
          if kk > 0 then begin
            ParamAlias := ParamAlias + ', ';
          end;
          ParamAlias := ParamAlias + Trim( SQLVar[ kk ].AliasName );
        end;
      end;
      SysAddString( 'OUTPUT PARAMETERS = [ ' +  ParamAlias + ' ]');
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_execute_immediate ( status:    pstatus_vector;
                                      db_handle: pisc_db_handle;
                                      tr_handle: pisc_tr_handle;
                                      length:    short;
                                      statement: PChar;
                                      dialect:   short;
                                      params:    PXSQLDA
                                      ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'EXECUTE IMMEDIATE');
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(db_handle^)));
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( '' );
      SysAddString( statement );
      SysAddString( '' );
    end;
    end;
  end;
  Result := isc_mon_dsql_execute_immediate ( status,
                                             db_handle,
                                             tr_handle,
                                             length,
                                             statement,
                                             dialect,
                                             params );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_exec_immed2 ( status:    pstatus_vector;
                                db_handle: pisc_db_handle;
                                tr_handle: pisc_tr_handle;
                                length:    short;
                                statement: PChar;
                                dialect:   short;
                                params:    PXSQLDA;
                                outvar:    PXSQLDA
                                ):         isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'EXECUTE IMMEDIATE 2');
      SysAddString( 'DB_HANDLE = ' + IntToStr( Integer(db_handle^)));
      SysAddString( 'TR_HANDLE = ' + IntToStr( Integer(tr_handle^)));
      SysAddString( '' );
      SysAddString( statement );
      SysAddString( '' );
    end;
    end;
  end;
  Result := isc_mon_dsql_exec_immed2 ( status,
                                       db_handle,
                                       tr_handle,
                                       length,
                                       statement,
                                       dialect,
                                       params,
                                       outvar );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgExecuteImmediate in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_set_cursor_name ( status_vector: pstatus_vector;
                                    st_handle:     pisc_stmt_handle;
                                    cursor_name:   PChar;
                                    unusedType:    smallint
                                    ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgIBCursor  in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'OPEN CURSOR');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
      SysAddString( 'NAME = ' + cursor_name );
    end;
    end;
  end;
  Result := isc_mon_dsql_set_cursor_name ( status_vector,
                                           st_handle,
                                           cursor_name,
                                           unusedType );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgIBCursor  in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_describe ( status_vector: pstatus_vector;
                             st_handle:     pisc_stmt_handle;
                             dialect:       smallint;
                             params:        PXSQLDA
                             ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgDescribe in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'DESCRIBE STATEMENT OUTPUT');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_describe ( status_vector,
                                    st_handle,
                                    dialect,
                                    params );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgDescribe in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_describe_bind ( status:        pstatus_vector;
                                  st_handle:     pisc_stmt_handle;
                                  dialect:       smallint;
                                  params:        PXSQLDA
                                  ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgDescribe in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'DESCRIBE STATEMENT INPUT');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_describe_bind ( status,
                                         st_handle,
                                         dialect,
                                         params );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement in IB_MonitorGroups ) and
       ( sgDescribe in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_dsql_sql_info ( status:        pstatus_vector;
                             st_handle:     pisc_stmt_handle;
                             item_length:   short;
                             items:         pchar;
                             buffer_length: short;
                             result_buffer: pchar
                             ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement     in IB_MonitorGroups ) and
       ( sgStatementInfo in IB_StatementGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'GET STATEMENT INFO');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_sql_info ( status,
                                    st_handle,
                                    item_length,
                                    items,
                                    buffer_length,
                                    result_buffer );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgStatement     in IB_MonitorGroups ) and
       ( sgStatementInfo in IB_StatementGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function mon_dsql_fetch ( status_vector: pstatus_vector;
                          st_handle:     pisc_stmt_handle;
                          dialect:       smallint;
                          params:        PXSQLDA
                          ):             isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgRow in IB_MonitorGroups then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'FETCH');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
    end;
    end;
  end;
  Result := isc_mon_dsql_fetch ( status_vector,
                                 st_handle,
                                 dialect,
                                 params );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if mgRow in IB_MonitorGroups then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end else if ( mgStatement in IB_MonitorGroups ) and
                ( Result = 100 ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'FETCHED ALL');
      SysAddString( 'STMT_HANDLE = ' + IntToStr( Integer(st_handle^)));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function mon_interprete ( buffer: PChar;
                          status: ppstatus_vector
                          ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if IB_MonitorGroups <> [] then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'INTERPRETE');
      SysAddString( 'BUFFER = ' + StrPas( buffer ));
    end;
    end;
  end;
  Result := isc_mon_interprete ( buffer, status );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if IB_MonitorGroups <> [] then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure mon_blob_default_desc (  a: pISC_BLOB_DESC;
                                   b: pchar;
                                   c: pchar ); stdcall;

var
  ii: integer;
begin
  isc_mon_blob_default_desc( a, b, c );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'GET DEFAULT BLOB DESC');
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_blob_gen_bpb ( status: pstatus_vector;
			    a:	    pISC_BLOB_DESC;
			    b:	    pISC_BLOB_DESC;
			    c:	    short;
			    d:	    pchar;
			    e:	    pshort
                            ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'GENERATE BLOB PARAMETER BUFFER');
    end;
    end;
  end;
  Result := isc_mon_blob_gen_bpb( status, a, b, c, d, e );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_blob_info ( status: pstatus_vector;
			 a:	 pisc_blob_handle;
			 b:	 short;
 			 c:	 pchar;
			 d:	 short;
			 e:	 pchar
                         ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'GET BLOB INFO');
    end;
    end;
  end;
  Result := isc_mon_blob_info( status, a, b, c, d, e );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_blob_lookup_desc( status:          pstatus_vector;
				    a:	             pisc_db_handle;
				    b:	             pisc_tr_handle;
				    c:	             pchar;
				    d:	             pchar;
				    e:	             pISC_BLOB_DESC;
				    f:	             pchar
                                    ):               isc_status; stdcall ;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'LOOKUP BLOB DESCRIPTION');
    end;
    end;
  end;
  Result := isc_mon_blob_lookup_desc ( status, a, b, c, d, e, f );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;


function mon_blob_set_desc( status:          pstatus_vector;
				 a:	          pchar;
				 b:	          pchar;
				 c:	          short;
				 d:	          short;
				 e:	          short;
				 f:	          pISC_BLOB_DESC
                                 ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'SET BLOB DESCRIPTION');
    end;
    end;
  end;
  Result := isc_mon_blob_set_desc ( status, a, b, c, d, e, f );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_cancel_blob ( status:          pstatus_vector;
	      	           blob_handle:     pisc_blob_handle
                           ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CANCEL BLOB');
    end;
    end;
  end;
  Result := isc_mon_cancel_blob ( status, blob_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_close_blob ( status:          pstatus_vector;
			  blob_handle:     pisc_blob_handle
                          ):               isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CLOSE BLOB');
    end;
    end;
  end;
  Result := isc_mon_close_blob ( status, blob_handle );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_create_blob ( status:      pstatus_vector;
			   a:		pisc_db_handle;
			   b:		pisc_tr_handle;
			   c:		pisc_blob_handle;
			   d:		pISC_QUAD
                           ):           isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CREATE BLOB');
    end;
    end;
  end;
  Result := isc_mon_create_blob ( status, a, b, c, d );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_create_blob2 ( status: pstatus_vector;
			    a:	    pisc_db_handle;
			    b:	    pisc_tr_handle;
			    c:	    pisc_blob_handle;
			    d:	    pISC_QUAD;
			    e:	    short;
			    f:	    pchar
                            ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'CREATE BLOB 2');
    end;
    end;
  end;
  Result := isc_mon_create_blob2 ( status, a, b, c, d, e, f );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_get_segment ( status:  pstatus_vector;
			   a:	pisc_blob_handle;
			   b:	pshort;
			   c:	short;
			   d:	pchar
                           ):       isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'GET SEGMENT');
    end;
    end;
  end;
  Result := isc_mon_get_segment( status, a, b, c, d );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_put_segment ( status: pstatus_vector;
			   a:	   pisc_blob_handle;
			   b:	   short;
			   c:	   pchar
                           ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'PUT SEGMENT');
    end;
    end;
  end;
  Result := isc_mon_put_segment ( status, a, b, c );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_open_blob ( status:  pstatus_vector;
			 a:	  pisc_db_handle;
			 b:	  pisc_tr_handle;
			 c:	  pisc_blob_handle;
			 d:	  pISC_QUAD
                         ):       isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'OPEN BLOB');
    end;
    end;
  end;
  Result := isc_mon_open_blob ( status, a, b, c, d );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

function mon_open_blob2 ( status: pstatus_vector;
			      a:      pisc_db_handle;
			      b:      pisc_tr_handle;
			      c:      pisc_blob_handle;
			      d:      pISC_QUAD;
			      e:      short;
			      g:      pchar
                              ):      isc_status; stdcall;
var
  ii: integer;
begin
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( '--------------------------------------------------------');
      SysAddString( 'OPEN BLOB 2');
    end;
    end;
  end;
  Result := isc_mon_open_blob2 ( status, a, b, c, d, e, g );
  for ii := 0 to Unit_MonitorList.Count - 1 do begin
    with TIB_Monitor(Unit_MonitorList.Items[ ii ]) do begin
    if ( mgBlob in IB_MonitorGroups ) then begin
      SysAddString( 'SQLCODE = ' + IntToStr( Result ));
      SysAddString( '--------------------------------------------------------');
    end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function mon_cancel_events ( status:    pstatus_vector;
                             db_handle: pisc_db_handle;
                             event_id:  pisc_long
                             ):         isc_status; stdcall;
//var
//  ii: integer;
begin
  Result := isc_mon_cancel_events ( status, db_handle, event_id );
end;

(*
  Tisc_array_gen_sdl = function( status:          pstatus_vector;
				 array_desc:	  pISC_ARRAY_DESC;
				 array_1:	  pshort;
				 array_buf:	  pchar;
				 array_2:	  pshort
                                 ):               isc_status; stdcall;

  Tisc_array_get_slice = function( status:  pstatus_vector;
				   a:       pisc_db_handle;
				   b:       pisc_tr_handle;
				   c:	    PISC_QUAD;
				   d:	    PISC_ARRAY_DESC;
				   e:	    pointer;
				   f:	    PISC_LONG
                                   ):       isc_status; stdcall;

  Tisc_array_lookup_bounds = function( status:          pstatus_vector;
				       a:		pisc_db_handle;
				       b:		pisc_tr_handle;
				       c:		pchar;
				       d:		pchar;
				       e:		pISC_ARRAY_DESC
                                       ):               isc_status; stdcall;

  Tisc_array_lookup_desc = function( status:           pstatus_vector;
				     a:	               pisc_db_handle;
				     b:	               pisc_tr_handle;
				     c:	               pchar;
				     d:	               pchar;
				     e:	               pISC_ARRAY_DESC
                                     ):                isc_status; stdcall;

  Tisc_array_set_desc = function( status:           pstatus_vector;
				  a:	            pchar;
				  b:	            pchar;
				  c:	            pshort;
				  d:	            pshort;
				  e:	            pshort;
				  f:	            pISC_ARRAY_DESC
                                  ):                isc_status; stdcall;

  Tisc_array_put_slice = function( status:           pstatus_vector;
				   a:	             pisc_db_handle;
				   b:	             pisc_tr_handle;
				   c:	             pISC_QUAD;
				   d:	             pISC_ARRAY_DESC;
				   e:	             pointer;
				   f:	             pISC_LONG
                                   ):                isc_status; stdcall;

  Tisc_cancel_events = function( status:    pstatus_vector;
                                 db_handle: pisc_db_handle;
                                 event_id:  pisc_long
                                 ):         isc_status; stdcall;

  Tisc_decode_date = procedure( ib_date: Pisc_QUAD;
                                tm_date: ptm); stdcall;

  Tisc_dsql_fetch2 = function ( status_vector: pstatus_vector;
                                st_handle:     pisc_stmt_handle;
                                dialect:       smallint;
                                params:        PXSQLDA;
                                fetchtype:     smallint;
			        fetchno:       longint
                                ):             isc_status; stdcall;

  Tisc_dsql_finish = function( db_handle: pisc_db_handle
                               ):         isc_status; stdcall;

  Tisc_dsql_insert = function( status:    pstatus_vector;
			       st_handle: pisc_stmt_handle;
			       dialect:   short;
			       params:    PXSQLDA
                               ):         isc_status; stdcall;

  Tisc_encode_date = procedure( tm_date: ptm;
                                ib_date: Pisc_QUAD); stdcall;

  Tisc_event_block = function( event_buf:  pointer;
                               result_buf: pointer;
                               count:      short;
                               name1:      pchar
                               ):          longint; cdecl;
{
ISC_LONG    cdecl isc_event_block (char  *  *
					       char  *  *
					       unsigned short, ...);
}

  Tisc_event_counts = procedure( status:        pstatus_vector;
                                 buffer_length: short;
                                 event_buffer:  pchar;
                                 result_buffer: pchar ); stdcall;

{

void        cdecl isc_expand_dpb (char  *  *
					      short  *
					      ...);

int        stdcall isc_modify_dpb (char  *  *
					 short  * unsigned short,
					 char  * short );

}

  Tisc_free = function( buffer: PChar ): isc_long; stdcall;

  Tisc_get_slice = function( status:  pstatus_vector;
			     a:	      pisc_db_handle;
			     b:	      pisc_tr_handle;
 			     c:	      pISC_QUAD;
 			     d:	      short;
			     e:	      pchar;
			     f:	      short;
			     g:	      pISC_LONG;
			     h:	      ISC_LONG;
			     i:	      pointer;
			     j:	      PISC_LONG
                             ):       isc_status; stdcall;

{
  Tisc_prepare_transaction2 = function( status:          pstatus_vector;
						 isc_tr_handle  *
						 short,
						 char  *
                 ):               isc_status; stdcall;

                 void        stdcall isc_print_sqlerror (short,
					   ISC_STATUS  *

  Tisc_print_status = function( status: pstatus_vector;
                                ):      isc_status; stdcall;
}

  Tisc_put_slice = function( status:          pstatus_vector;
                             a:               pisc_db_handle;
                             b:               pisc_tr_handle;
                             c:               pISC_QUAD;
                             d:               short;
                             e:               pchar;
                             f:               short;
                             g:               pISC_LONG;
                             h:               ISC_LONG;
                             i:               pointer
                             ):               isc_status; stdcall;

  Tisc_que_events = function( status:             pstatus_vector;
                              db_handle:          pisc_db_handle;
                              event_id:           pisc_long;
                              length:             short;
                              event_buffer:       pchar;
                              event_function:     isc_callback;
                              event_function_arg: pointer
                              ):                  isc_status; stdcall;

*)

{------------------------------------------------------------------------------}

procedure TIB_Monitor.StoreOriginalHooks;
begin
  with DefaultIB_Session do begin
    if not Attached then Attach;
    isc_mon_attach_database :=          isc_attach_database;
    isc_mon_create_database :=          isc_create_database;
    isc_mon_detach_database :=          isc_detach_database;
    isc_mon_drop_database :=            isc_drop_database;
    isc_mon_database_info :=            isc_database_info;
    isc_mon_commit_retaining :=         isc_commit_retaining;
    isc_mon_commit_transaction :=       isc_commit_transaction;
    isc_mon_rollback_transaction :=     isc_rollback_transaction;
    isc_mon_start_multiple :=           isc_start_multiple;
    isc_mon_dsql_prepare :=             isc_dsql_prepare;
    isc_mon_dsql_execute :=             isc_dsql_execute;
    isc_mon_dsql_execute2 :=            isc_dsql_execute2;
    isc_mon_dsql_execute_immediate :=   isc_dsql_execute_immediate;
    isc_mon_dsql_exec_immed2 :=         isc_dsql_exec_immed2;
    isc_mon_dsql_fetch :=               isc_dsql_fetch;
    isc_mon_dsql_set_cursor_name :=     isc_dsql_set_cursor_name;
    isc_mon_dsql_free_statement :=      isc_dsql_free_statement;
    isc_mon_dsql_allocate_statement :=  isc_dsql_allocate_statement;
    isc_mon_dsql_alloc_statement2 :=    isc_dsql_alloc_statement2;
    isc_mon_dsql_describe :=            isc_dsql_describe;
    isc_mon_dsql_describe_bind :=       isc_dsql_describe_bind;
    isc_mon_dsql_sql_info :=            isc_dsql_sql_info;
    isc_mon_interprete :=               isc_interprete;
    isc_mon_blob_default_desc :=        isc_blob_default_desc;
    isc_mon_blob_gen_bpb :=             isc_blob_gen_bpb;
    isc_mon_blob_info :=                isc_blob_info;
    isc_mon_blob_lookup_desc :=         isc_blob_lookup_desc;
    isc_mon_blob_set_desc :=            isc_blob_set_desc;
    isc_mon_cancel_blob :=              isc_cancel_blob;
    isc_mon_close_blob :=               isc_close_blob;
    isc_mon_create_blob :=              isc_create_blob;
    isc_mon_create_blob2 :=             isc_create_blob2;
    isc_mon_open_blob :=                isc_open_blob;
    isc_mon_open_blob2 :=               isc_open_blob2;
    isc_mon_get_segment :=              isc_get_segment;
    isc_mon_put_segment :=              isc_put_segment;
  end;
end;

procedure TIB_Monitor.PutInMonitorHooks;
begin
  with DefaultIB_Session do begin
  isc_attach_database :=              mon_attach_database;
  isc_create_database :=              mon_create_database;
  isc_detach_database :=              mon_detach_database;
  isc_drop_database :=                mon_drop_database;
  isc_database_info :=                mon_database_info;
  isc_commit_retaining :=             mon_commit_retaining;
  isc_commit_transaction :=           mon_commit_transaction;
  isc_rollback_transaction :=         mon_rollback_transaction;
  isc_start_multiple :=               mon_start_multiple;
  isc_dsql_prepare :=                 mon_dsql_prepare;
  isc_dsql_execute :=                 mon_dsql_execute;
  isc_dsql_execute2 :=                mon_dsql_execute2;
  isc_dsql_execute_immediate :=       mon_dsql_execute_immediate;
  isc_dsql_exec_immed2 :=             mon_dsql_exec_immed2;
  isc_dsql_fetch :=                   mon_dsql_fetch;
  isc_dsql_set_cursor_name :=         mon_dsql_set_cursor_name;
  isc_dsql_free_statement :=          mon_dsql_free_statement;
  isc_dsql_allocate_statement :=      mon_dsql_allocate_statement;
  isc_dsql_alloc_statement2 :=        mon_dsql_alloc_statement2;
  isc_dsql_describe :=                mon_dsql_describe;
  isc_dsql_describe_bind :=           mon_dsql_describe_bind;
  isc_dsql_sql_info :=                mon_dsql_sql_info;
  isc_interprete :=                   mon_interprete;
  isc_blob_default_desc :=            mon_blob_default_desc;
  isc_blob_gen_bpb :=                 mon_blob_gen_bpb;
  isc_blob_info :=                    mon_blob_info;
  isc_blob_lookup_desc :=             mon_blob_lookup_desc;
  isc_blob_set_desc :=                mon_blob_set_desc;
  isc_cancel_blob :=                  mon_cancel_blob;
  isc_close_blob :=                   mon_close_blob;
  isc_create_blob :=                  mon_create_blob;
  isc_create_blob2 :=                 mon_create_blob2;
  isc_open_blob :=                    mon_open_blob;
  isc_open_blob2 :=                   mon_open_blob2;
  isc_get_segment :=                  mon_get_segment;
  isc_put_segment :=                  mon_put_segment;
  end;
  Unit_Hooks_Active := true;
end;

procedure TIB_Monitor.TakeOutMonitorHooks;
begin
  with DefaultIB_Session do begin
  isc_attach_database :=              isc_mon_attach_database;
  isc_create_database :=              isc_mon_create_database;
  isc_detach_database :=              isc_mon_detach_database;
  isc_drop_database :=                isc_mon_drop_database;
  isc_database_info :=                isc_mon_database_info;
  isc_commit_retaining :=             isc_mon_commit_retaining;
  isc_commit_transaction :=           isc_mon_commit_transaction;
  isc_rollback_transaction :=         isc_mon_rollback_transaction;
  isc_start_multiple :=               isc_mon_start_multiple;
  isc_dsql_prepare :=                 isc_mon_dsql_prepare;
  isc_dsql_execute :=                 isc_mon_dsql_execute;
  isc_dsql_execute2 :=                isc_mon_dsql_execute2;
  isc_dsql_execute_immediate :=       isc_mon_dsql_execute_immediate;
  isc_dsql_exec_immed2 :=             isc_mon_dsql_exec_immed2;
  isc_dsql_fetch :=                   isc_mon_dsql_fetch;
  isc_dsql_set_cursor_name :=         isc_mon_dsql_set_cursor_name;
  isc_dsql_free_statement :=          isc_mon_dsql_free_statement;
  isc_dsql_allocate_statement :=      isc_mon_dsql_allocate_statement;
  isc_dsql_alloc_statement2 :=        isc_mon_dsql_alloc_statement2;
  isc_dsql_describe :=                isc_mon_dsql_describe;
  isc_dsql_describe_bind :=           isc_mon_dsql_describe_bind;
  isc_dsql_sql_info :=                isc_mon_dsql_sql_info;
  isc_interprete :=                   isc_mon_interprete;
  isc_blob_default_desc :=            isc_mon_blob_default_desc;
  isc_blob_gen_bpb :=                 isc_mon_blob_gen_bpb;
  isc_blob_info :=                    isc_mon_blob_info;
  isc_blob_lookup_desc :=             isc_mon_blob_lookup_desc;
  isc_blob_set_desc :=                isc_mon_blob_set_desc;
  isc_cancel_blob :=                  isc_mon_cancel_blob;
  isc_close_blob :=                   isc_mon_close_blob;
  isc_create_blob :=                  isc_mon_create_blob;
  isc_create_blob2 :=                 isc_mon_create_blob2;
  isc_open_blob :=                    isc_mon_open_blob;
  isc_open_blob2 :=                   isc_mon_open_blob2;
  isc_get_segment :=                  isc_mon_get_segment;
  isc_put_segment :=                  isc_mon_put_segment;
  end;
  Unit_Hooks_Active := false;
end;

(*

//isc_mon_attach_service :=           isc_attach_service;
  isc_mon_array_gen_sdl :=            isc_array_gen_sdl;
  isc_mon_array_get_slice :=          isc_array_get_slice;
  isc_mon_array_lookup_bounds :=      isc_array_lookup_bounds;
  isc_mon_array_lookup_desc :=        isc_array_lookup_desc;
  isc_mon_array_set_desc :=           isc_array_set_desc;
  isc_mon_array_put_slice :=          isc_array_put_slice;

  isc_mon_blob_default_desc :=        isc_blob_default_desc;
  isc_mon_blob_gen_bpb :=             isc_blob_gen_bpb;
  isc_mon_blob_info :=                isc_blob_info;
  isc_mon_blob_lookup_desc :=         isc_blob_lookup_desc;
  isc_mon_blob_set_desc :=            isc_blob_set_desc;
  isc_mon_cancel_blob :=              isc_cancel_blob;
  isc_mon_close_blob :=               isc_close_blob;
  isc_mon_create_blob :=              isc_create_blob;
  isc_mon_create_blob2 :=             isc_create_blob2;
  isc_mon_open_blob :=                isc_open_blob;
  isc_mon_open_blob2 :=               isc_open_blob2;
  isc_mon_get_segment :=              isc_get_segment;
  isc_mon_put_segment :=              isc_put_segment;

//isc_mon_compile_request :=          isc_compile_request;
//isc_mon_compile_request2 :=         isc_compile_request2;
//isc_mon_ddl :=                      isc_ddl;
  isc_mon_decode_date :=              isc_decode_date;
//isc_mon_detach_service :=           isc_detach_service;
  isc_mon_encode_date :=              isc_encode_date;
  isc_mon_event_block :=              isc_event_block;
  isc_mon_event_counts :=             isc_event_counts;
  isc_mon_cancel_events :=            isc_cancel_events;
//isc_mon_expand_dpb :=               isc_expand_dpb;
//isc_mon_modify_dpb :=               isc_modify_dpb;
  isc_mon_free :=                     isc_free;
  isc_mon_get_slice :=                isc_get_slice;
//isc_mon_prepare_transaction :=      isc_prepare_transaction;
//isc_mon_prepare_transaction2 :=     isc_prepare_transaction2;
//isc_mon_print_sqlerror :=           isc_print_sqlerror;
//isc_mon_print_status :=             isc_print_status;
  isc_mon_put_slice :=                isc_put_slice;
  isc_mon_que_events :=               isc_que_events;
//isc_mon_query_service :=            isc_query_service;
//isc_mon_receive :=                  isc_receive;
//isc_mon_reconnect_transaction :=    isc_reconnect_transaction;
//isc_mon_release_request :=          isc_release_request;
//isc_mon_request_info :=             isc_request_info;
//isc_mon_seek_blob :=                isc_seek_blob;
//isc_mon_send :=                     isc_send;
//isc_mon_start_and_send :=           isc_start_and_send;
//isc_mon_start_request :=            isc_start_request;
  isc_mon_sqlcode :=                  isc_sqlcode;
  isc_mon_transaction_info :=         isc_transaction_info;
  isc_mon_transact_request :=         isc_transact_request;
//isc_mon_unwind_request :=           isc_unwind_request;
//isc_mon_wait_for_event :=           isc_wait_for_event;
//isc_mon_ftof :=                     isc_ftof;
//isc_mon_print_blr :=                isc_print_blr;
//isc_mon_set_debug :=                isc_set_debug;
//isc_mon_qtoq :=                     isc_qtoq;
  isc_mon_vax_integer :=              isc_vax_integer;
//isc_mon_vtof :=                     isc_vtof;
//isc_mon_vtov :=                     isc_vtov;
//isc_mon_version :=                  isc_version;
  isc_mon_dsql_insert :=              isc_dsql_insert;
  isc_mon_dsql_fetch2 :=              isc_dsql_fetch2;
//isc_mon_event_block_asm :=          isc_event_block_asm;

*)

end.

