{~~ This unit contains the TIB_CursorGrid component. }
unit IB_CursorGrid;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids,

  IB_Header,
  IB_Components;

type
{~~...
This component acts as a buffer of fetched data.}
  TIB_CursorGrid = class(TStringGrid)
  private
  { Property Storage }
    FIB_DataLink: TIB_DataLink;
    FFrozen: boolean;
    FMovingRow: boolean;
  { Property Access Methods }
    procedure SetIB_DataSetLink( AValue: TIB_DataSetLink); virtual;
    function GetIB_DataSetLink: TIB_DataSetLink; virtual;
    procedure SetFrozen( AValue: boolean ); virtual;
  { IB_CursorLink methods }
    procedure IB_DataSetStateChanged( Sender: TIB_DataLink;
                                      IB_DataSetLink: TIB_DataSetLink );
    procedure IB_CursorRowDataChanged( Sender: TIB_DataLink;
                                       IB_DataSetLink: TIB_DataSetLink;
                                       IB_Column: TIB_Column );
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
  protected
  { Inherited Methods }
    procedure Loaded; override;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
  { Utility Methods }
    procedure UpdateSQLCursorGrid;
    procedure CloseSQLCursorGrid;
    procedure ClearSQLCursorGrid;
    procedure FillSQLCursorGrid;
    procedure UpdateColumnWidths;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;

{~~...
Refresh grid when the data of the underlying dataset changes.

This happens automatically if Frozen is false.}
    procedure UpdateGrid;

{~~...
Refresh grid when the status of the underlying dataset changes.

This happens automatically if Frozen is false.}
    procedure UpdateRow;

{~~$<*#>...
This property freezes the grid so that is does not respond to changes in ...
the underlying dataset.

This can be useful if the user fetches all the rows and then wants ...
to be able to double-click on any of the fetched rows and have that row ...
re-fetched for editing without clearing the previously fetched batch of rows.

This could be accomplished by setting Frozen to true just after the last ...
row is fetched. Frozen could be set to false just prior to another ...
request to go to the last record in a selected dataset. This way the grid ...
would always represent a complete sub-set of data. Then, using this sub-set ...
the user can select and work with one record at a time by double-clicking ...
on them in the grid.}
    property Frozen: boolean read FFrozen write SetFrozen;

  published

{~~$<*>...
Reference to the DataSetLink for the grid.}
    property IB_DataSetLink: TIB_DataSetLink read GetIB_DataSetLink
                                             write SetIB_DataSetLink;
  end;

implementation

constructor TIB_CursorGrid.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FIB_DataLink := TIB_DataLink.Create( Self );
  with FIB_DataLink do begin
    OnStateChanged := IB_DataSetStateChanged;
    OnDataChanged := IB_CursorRowDataChanged;
  end;
  Options := Options - [goEditing];
end;

destructor TIB_CursorGrid.Destroy;
begin
  IB_DataSetLink := nil;
  with FIB_DataLink do begin
    OnStateChanged := nil;
    OnDataChanged  := nil;
  end;
  inherited Destroy;
end;

procedure TIB_CursorGrid.Loaded;
begin
  inherited Loaded;
  UpdateGrid;
end;

{------------------------------------------------------------------------------}

function TIB_CursorGrid.GetIB_DataSetLink: TIB_DataSetLink;
begin
  if FIB_DataLink <> nil then begin
    Result := FIB_DataLink.IB_DataSetLink;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_CursorGrid.SetIB_DataSetLink( AValue: TIB_DataSetLink);
begin
  if FIB_DataLink <> nil then begin
    if FIB_DataLink.IB_DataSetLink <> AValue then begin
      FIB_DataLink.IB_DataSetLink := AValue;
      UpdateGrid;
    end;
  end;
end;

procedure TIB_CursorGrid.SetFrozen( AValue: boolean );
begin
  if Frozen <> AValue then begin
    FFrozen := AValue;
    if Frozen = false then begin
      UpdateGrid;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_CursorGrid.IB_DataSetStateChanged( Sender: TIB_DataLink;
                                               IB_DataSetLink: TIB_DataSetLink);
begin
  UpdateGrid;
end;

procedure TIB_CursorGrid.IB_CursorRowDataChanged( Sender: TIB_DataLink;
                                                IB_DataSetLink: TIB_DataSetLink;
                                                IB_Column: TIB_Column);
begin
  try
    FMovingRow := true;
    UpdateRow;
    if IB_DataSetLink <> nil then begin
      if IB_DataSetLink.IB_DataSet <> nil then begin
        with IB_DataSetLink.IB_DataSet do begin
          if not Unidirectional and
             ( RowNum > 0 ) and
             ( RowNum < Self.RowCount ) and
             ( RowNum <> Row ) then begin
            Row := RowNum;
          end;
        end;
      end;
    end;
  finally
    FMovingRow := false;
  end;
end;

function TIB_CursorGrid.SelectCell( ACol, ARow: longint ): boolean;
begin
  if FMovingRow then begin
    Result := true;
  end else begin
    Result := inherited SelectCell( ACol, ARow );
    if Result and
      ( IB_DataSetLink <> nil ) and
      ( IB_DataSetLink.IB_DataSet <> nil ) then begin
      with IB_DataSetLink.IB_DataSet do begin
        if Active and not Unidirectional and
          ( Row > 0 ) and
          ( Row <> RowNum ) then begin
          RowNum := Row;
        end;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TIB_CursorGrid.UpdateGrid;
begin
  if Frozen or ( IB_DataSetLink = nil ) then Exit;
  if not ( csLoading    in ComponentState ) and
     not ( csDestroying in ComponentState ) then begin
    case IB_DataSetLink.DataSetState of
      dsNone: CloseSQLCursorGrid;
      dsPrepared: ClearSQLCursorGrid;
    end;
    UpdateSQLCursorGrid;
  end;
end;

procedure TIB_CursorGrid.UpdateRow;
begin
  if Frozen or ( IB_DataSetLink = nil ) then Exit;
  with IB_DataSetLink do begin
    if IB_DataSet <> nil then with IB_DataSet do begin
      if BOF then begin
        ClearSQLCursorGrid;
      end else if EOF and Unidirectional then begin
      end else begin
        FillSQLCursorGrid;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}
{   Grid handling procedures                                                   }
{------------------------------------------------------------------------------}

procedure TIB_CursorGrid.UpdateSQLCursorGrid;
var
  ii: integer;
begin
  if ( FIB_DataLink.Prepared ) then begin
    with IB_DataSetLink.IB_DataSet do begin
      ColCount := OutputCount;
      for ii := 0 to OutputCount - 1 do begin
        Cells[ ii, 0 ] := OutputRow[ ii ].DisplayName;
      end;
    end;
  end else begin
    CloseSQLCursorGrid;
  end;
end;

procedure TIB_CursorGrid.CloseSQLCursorGrid;
var
  ii: integer;
begin
  RowCount := 2;
  FixedRows := RowCount - 1;
  RowCount := FixedRows + 1;
  for ii := 0 to FixedRows do begin
    Rows[ ii ].Clear;
  end;
  ColCount := 1;
end;

procedure TIB_CursorGrid.ClearSQLCursorGrid;
var
  ii: integer;
  OldCursor: TCursor;
begin
  OldCursor := Screen.Cursor;
  try
    Screen.Cursor := crHourglass;
    with IB_DataSetLink.IB_DataSet do begin
      for ii := 1 to RowCount - 1 do begin
        Rows[ ii ].Clear;
      end;
      Self.RowCount := FixedRows + 1;
    end;
    UpdateColumnWidths;
  finally
    Screen.Cursor := OldCursor;
  end;
end;

procedure TIB_CursorGrid.FillSQLCursorGrid;
var
  Index: integer;
begin
  if ( FIB_DataLink.Active ) then begin
    with IB_DataSetLink.IB_DataSet do begin
      if Self.RowCount < RowNum + FixedRows then begin
        Self.RowCount := RowNum + FixedRows;
      end;
      for Index := 0 to OutputCount - 1 do begin
        if BOF then begin
        end else if EOF then begin
          Cells[ Index, RowNum + FixedRows - 1] := '';
        end else if OutputRow[ Index ].IsNull then begin
          Cells[ Index, RowNum + FixedRows - 1] := '<NULL>';
        end else if OutputRow[ Index ].IsBlob then begin
          Cells[ Index, RowNum + FixedRows - 1] := '(BLOB)';
        end else begin
          Cells[ Index, RowNum + FixedRows - 1] := OutputRow[ Index ].AsString;
        end;
      end;
    end;
  end;
end;

procedure TIB_CursorGrid.UpdateColumnWidths;
var
  ii: integer;
  DataWidth: integer;
  HeaderWidth: integer;
  NewWidth: integer;
begin
  if ( FIB_DataLink.Prepared ) then begin
    with IB_DataSetLink.IB_DataSet do begin
      ColCount := OutputCount;
      for ii := 0 to OutputCount - 1 do with OutputRow[ ii ] do begin
        HeaderWidth := Length( Trim( DisplayName ));
        if ( SQLType = SQL_DATE  ) or
           ( SQLType = SQL_DATE_ ) then begin
          DataWidth := 16;
        end else begin
          DataWidth := SQLLen;
        end;
        if HeaderWidth > DataWidth then begin
          NewWidth := HeaderWidth * 8;
        end else begin
          NewWidth := DataWidth * 8;
        end;
        if NewWidth < 70 then begin
          NewWidth := 70;
        end;
        if NewWidth > 200 then begin
          NewWidth := 200;
        end;
        ColWidths[ ii ] := NewWidth;
        ColWidths[ ii ] := NewWidth;
      end;
    end;
  end;
end;

procedure TIB_CursorGrid.CMEnter( var Message: TCMEnter );
begin
  FIB_DataLink.SetFocus;
  inherited;
end;

end.

