{ **************************************************************************** }
{                                                                              }
{   Delphi component TIB_ConnectionBar                                         }
{                                                                              }
{ **************************************************************************** }

unit IB_ConnectionBar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, StdCtrls,
  Forms, Graphics, ExtCtrls, Menus, Buttons, Dialogs,

  IB_Components;

{------------------------------------------------------------------------------}

type
  TIB_ConnectionBar_ButtonType = ( cbCreate,
                                   cbConnect,
                                   cbDisconnect,
                                   cbDrop );
                                   
  TIB_ConnectionBar_ButtonSet = set of TIB_ConnectionBar_ButtonType;

  TIB_ConnectionBar_Button = class;
  TIB_ConnectionBar = class (TCustomPanel)
  private
    FIB_ConnectionLink: TIB_ConnectionLink;
    ButtonWidth: Integer;
    FVisibleButtons: TIB_ConnectionBar_ButtonSet;
    MinBtnSize: TPoint;
    FFocusedButton: TIB_ConnectionBar_ButtonType;
    FBeforeAction: TNotifyEvent;
    FAfterAction: TNotifyEvent;
    function GetConnection: TIB_Connection;
    procedure SetConnection(AValue: TIB_Connection);
    function GetReceiveFocus: boolean;
    procedure SetReceiveFocus(AValue: boolean);
    procedure InitButtons;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TIB_ConnectionBar_ButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    procedure IB_ConnectionConnectedChanged( CLink: TIB_ConnectionLink;
                                             C: TIB_Connection);
    procedure BarClick(Sender: TObject); virtual;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    Buttons: array[TIB_ConnectionBar_ButtonType] of TIB_ConnectionBar_Button;
    constructor Create(AOwner: TComponent); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TIB_ConnectionBar_ButtonType);
    property FocusedButton: TIB_ConnectionBar_ButtonType read FFocusedButton;
  published
    property IB_Connection: TIB_Connection
        read GetConnection
       write SetConnection;
    property ReceiveFocus: boolean read GetReceiveFocus write SetReceiveFocus;
    property VisibleButtons: TIB_ConnectionBar_ButtonSet
        read FVisibleButtons
       write SetVisible
     default [cbConnect, cbDisconnect];
    property BeforeAction: TNotifyEvent
        read FBeforeAction
       write FBeforeAction;
    property AfterAction:  TNotifyEvent
        read FAfterAction
       write FAfterAction;
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
  end;

  TIB_ConnectionBar_Button = class(TSpeedButton)
  private
    FIndex: TIB_ConnectionBar_ButtonType;
  protected
    procedure Paint; override;
  public
    property Index: TIB_ConnectionBar_ButtonType read FIndex write FIndex;
  end;

implementation

{$R IB_ConnectionBar.RES}

{------------------------------------------------------------------------------}

constructor TIB_ConnectionBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FIB_ConnectionLink := TIB_ConnectionLink.Create( Self );
  with FIB_ConnectionLink do begin
    AfterAssignment := IB_ConnectionConnectedChanged;
    OnConnectedChanged := IB_ConnectionConnectedChanged;
  end;
  FVisibleButtons := [cbConnect, cbDisconnect];
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Ctl3D := false;
  ParentCtl3D := false;
  Width := 121;
  Height := 25;
  ButtonWidth := 0;
  FFocusedButton := cbConnect;
end;

procedure TIB_ConnectionBar.IB_ConnectionConnectedChanged(
                                                      CLink: TIB_ConnectionLink;
                                                      C: TIB_Connection);
begin
  ActiveChanged;
end;

function TIB_ConnectionBar.GetConnection: TIB_Connection;
begin
  if FIB_ConnectionLink <> nil then begin
    Result := FIB_ConnectionLink.IB_Connection;
  end else begin
    Result := nil;
  end;
end;

procedure TIB_ConnectionBar.SetConnection(AValue: TIB_Connection);
begin
  if FIB_ConnectionLink <> nil then begin
    if FIB_ConnectionLink.IB_Connection <> AValue then begin
      FIB_ConnectionLink.IB_Connection := AValue;
      ActiveChanged;
    end;
  end;
end;

function TIB_ConnectionBar.GetReceiveFocus: boolean;
begin
  if FIB_ConnectionLink <> nil then begin
    Result :=  FIB_ConnectionLink.ReceiveFocus;
  end else begin
    Result := false;
  end;
end;

procedure TIB_ConnectionBar.SetReceiveFocus(AValue: boolean);
begin
  if FIB_ConnectionLink <> nil then begin
    FIB_ConnectionLink.ReceiveFocus := AValue;
  end;
end;

procedure TIB_ConnectionBar.InitButtons;
var
  I: TIB_ConnectionBar_ButtonType;
  Btn: TIB_ConnectionBar_Button;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TIB_ConnectionBar_Button.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    case I of
      cbCreate: begin
        Btn.Hint := 'Create Database';
//        Btn.Caption := 'Create';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'CON_BAR_CREATE'  );
        Btn.NumGlyphs := 2;
      end;
      cbConnect: begin
        Btn.Hint := 'Connect Database';
//      Btn.Caption := 'Connect';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'CON_BAR_CONNECT'  );
        Btn.NumGlyphs := 2;
      end;
      cbDisconnect: begin
        Btn.Hint := 'Disconnect Database';
//      Btn.Caption := 'Disconnect';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'CON_BAR_DISCONNECT'  );
        Btn.NumGlyphs := 2;
      end;
      cbDrop: begin
        Btn.Hint := 'Drop Database';
//      Btn.Caption := 'Drop';
        Btn.Glyph.Handle := LoadBitmap(HInstance, 'CON_BAR_DROP');
        Btn.NumGlyphs := 2;
      end;
    end;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := BarClick;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
end;

procedure TIB_ConnectionBar.BtnMouseDown (Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TIB_ConnectionBar_ButtonType;
begin
  OldFocus := FocusedButton;
  FFocusedButton := TIB_ConnectionBar_Button(Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TIB_ConnectionBar.SetVisible(Value: TIB_ConnectionBar_ButtonSet);
var
  I: TIB_ConnectionBar_ButtonType;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;


procedure TIB_ConnectionBar.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TIB_ConnectionBar_ButtonType;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[cbCreate] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TIB_ConnectionBar.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TIB_ConnectionBar.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_ConnectionBar.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TIB_ConnectionBar.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TIB_ConnectionBar.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TIB_ConnectionBar.ActiveChanged;
begin
  if (IB_Connection <> nil) and (Enabled) then begin
    Buttons[ cbCreate     ].Enabled := not IB_Connection.Connected;
    Buttons[ cbConnect    ].Enabled := not IB_Connection.Connected;
    Buttons[ cbDisconnect ].Enabled :=     IB_Connection.Connected;
    Buttons[ cbDrop       ].Enabled :=     IB_Connection.Connected;
  end else begin
    Buttons[ cbCreate     ].Enabled := false;
    Buttons[ cbConnect    ].Enabled := false;
    Buttons[ cbDisconnect ].Enabled := false;
    Buttons[ cbDrop       ].Enabled := false;
  end;
end;

procedure TIB_ConnectionBar.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  ActiveChanged;
end;

procedure TIB_ConnectionBar.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TIB_ConnectionBar_ButtonType;
  OldFocus: TIB_ConnectionBar_ButtonType;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TIB_ConnectionBar.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TIB_ConnectionBar.BtnClick(Index: TIB_ConnectionBar_ButtonType);
begin
  if not (csDesigning in ComponentState) and
     Assigned( FBeforeAction ) then begin
    FBeforeAction( Self );
  end;
  if (IB_Connection <> nil) then
  begin
    with IB_Connection do
    begin
      case Index of
      cbCreate:     CreateDatabase;
      cbConnect:    Connect;
      cbDisconnect: Disconnect;
      cbDrop:       DropDatabase;
      end;
    end;
  end;
  if not (csDesigning in ComponentState) and
     Assigned( FAfterAction ) then begin
    FAfterAction( Self );
  end;
end;

procedure TIB_ConnectionBar.BarClick( Sender: TObject );
begin
  BtnClick(TIB_ConnectionBar_Button(Sender).Index);
end;

procedure TIB_ConnectionBar_Button.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TIB_ConnectionBar(Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

end.
