{ copyright:
   University Hospital Vrije Universiteit, Johan Colijn
   De Boelelaan 1117
   PO Box  7057
   1007 MB Amsterdam
   The Netherlands
   email: j.colijn@azvu.nl

   This product is provided as-is, with no express or implied warranties of
   any kind. Use this product at your own risk.

   It is free for you to use, distribute, modify, etc.... in any way
   you see fit, with the following restrictions:
   1. It may not be sold or added to a commercial product
   2. I would like to receive improvements as well as in the coding (I
      know its poor) as in English (I know its poor)

   Furthermore comments and suggestions are welcome.
}

unit IBLoad;

interface
uses SYSUtils, IBProcs;
const
  ibload_version = '1.00';

function IBLoadProc(sDBName:     string;
                    sTablename:  string;
                    sFilename:   string;
                    sUsername:   string;
                    sPassword:   string;
                    sLogfile:    string;
                    maxrec:      LongInt;
                    insert:      boolean;
                    debug:       boolean;
                    silent:      boolean;
                    log:         boolean): LongInt;
// return codes:
// positive: number of records loaded
// -1      : interbase error (see logging)
// -10     : couldn't open logfile
// -11     : inputfile not found
// -12     : couldn't open inputfile
// -20     : fields other than CHAR or VARCHAR
// -999    : error not (yet) defined

implementation

{$R-}
const
  MAXFIELDS    = 512;  // maximum aantal velden in een tabel
  MAXFIELDLEN  = 256;
  MAXRECLEN    = 2048;
  FIELDNAMELEN = 31;

  FIELDQUERY1 =
    'select rf.rdb$field_name, f.rdb$field_type, f.rdb$field_length ' +
    'from rdb$relation_fields rf, rdb$fields f where ' +
    'rf.rdb$relation_name = ';                   // deel 1 veldquery
  FIELDQUERY2 =
    ' and rf.rdb$field_source = f.rdb$field_name ' +
    'order by rf.rdb$field_position';            // deel 2 veldquery
  NFIELDSQUERY= 3;

type
  field_rec = record
    field_name : string;
    field_type: short;
    field_length: short;
  end; // storage of field properties

var
  RetCode: LongInt;
  DBHandle: isc_db_handle;
  DPB_Buffer: array [0..255] of char;
  DPB_BufPtr: integer;
  connected: boolean;
  bLog:   boolean;
  bDebug: boolean;
  bSilent: boolean;
  fields: array[0..MAXFIELDS - 1] of field_rec;
  FIn: File;
  FLog: Text;
  nfields: short;
  reclen: short;

procedure DisConnect; forward;
function Rollback(TRHandle: isc_tr_handle): boolean; forward;

procedure ProgMsg(msg: string);
begin
  if not bSilent then
    writeln(msg);

  if bLog then
    writeln(FLog, msg);
end;

function FmtInt(int: LongInt; width: smallint): string;
var
  i: smallint;
  len : smallint;
  str: string;
begin
  str := IntToStr(int);
  len := length(str);
  for i := len + 1 to width do
    str := ' ' + str;
  result :=  str;
end;

// error handling
procedure HandleIBErrors(status: pstatus_vector; TRHandle: isc_tr_handle);
var
  buffer: array[0..255] of char;
  errMsg, lastMsg: string;
  errCode: isc_status;

begin
  repeat
    errCode := isc_interprete( @buffer, @status);
    if lastMsg <> strPas( Buffer) then begin
      lastMsg := strPas( buffer);
      ProgMsg(IntToStr(errCode) + ': ' + lastMsg);
      RetCode := -1;
    end;
  until errCode = 0;
    DisConnect;
end;

// initialisation database buffer
procedure InitDPB;
begin
  FillChar(DPB_Buffer, sizeof(DPB_Buffer), #0);
  DPB_BufPtr := 0;
  DPB_Buffer[0] := char(isc_dpb_version1);
  inc(DPB_BufPtr);
end;

// build database buffer
procedure BuildDPB(item: byte; contents: string);
begin
  DPB_Buffer[DPB_BufPtr] := char(item);
  inc(DPB_BufPtr);
  DPB_Buffer[DPB_BufPtr] := char(length(contents));
  inc(DPB_BufPtr);
  StrPCopy(@DPB_Buffer[DPB_BufPtr], contents);
  inc(DPB_BufPtr, length(contents));
end;

procedure DisConnect;
var
  status: status_vector;
  errCode: isc_status;
begin
  if connected then begin
    connected := false;
    errCode := isc_detach_database(@status, @DBHandle);
    if errcode <> 0 then
      HandleIBErrors(@status, nil);
  end;
  ProgMsg(DateTimeToStr(now) + ' - Disconnect.');
end;

function Connect(sDBName: string; sUsername: string; sPassword: string): boolean;
var
  DBName: array[0..255] of char;
  status: status_vector;
  errCode: isc_status;

begin
  if connected then
  Disconnect;
  FillChar(DBName, sizeof(DBName), #0);
  StrPCopy(DBName, sDBName);
  DBHandle := nil;
  InitDPB;
  BuildDPB(isc_dpb_user_name, sUsername);
  BuildDPB(isc_dpb_password,  sPassword);
  errCode := isc_attach_database(@status, 0, @DBName, @DBHandle,
                                 DPB_BufPtr, @DPB_Buffer);
  if errCode <> 0 then
    HandleIBErrors(@status, nil)
  else begin
    connected := true;

    ProgMsg(DateTimeToStr(now) + ' - Connect user ' + sUsername +
          ' (' + sDBName + ')');
  end;
  result := connected;
end;

// begin transactie
function StartTransaction: isc_tr_handle;
var
  status: status_vector;
  errCode: isc_status;
  teb : isc_teb;
  TRHandle : isc_tr_handle;

begin
  TRHandle := nil;
  teb.db_ptr := @DBHandle;
  teb.tpb_len := 0;
  teb.tpb_ptr := nil;
  errcode := isc_start_multiple( @status, @TRHandle, 1, @teb);
  if errcode <> 0 then
    HandleIBErrors( @status, nil)
  else
    result := TRHandle;
end;

function Commit(TRHandle: isc_tr_handle): boolean;
var
  status: status_vector;
  errCode: isc_status;

begin
  errcode := isc_commit_transaction( @status, @TRHandle);
  if errcode <> 0 then begin
    HandleIBErrors(@status, nil);
    result := false;
  end
  else
    result := true;
end;

function Rollback(TRHandle: isc_tr_handle): boolean;
var
  status: status_vector;
  errCode: isc_status;

begin
  errcode := isc_rollback_transaction( @status, @TRHandle);
  if errcode <> 0 then begin
    HandleIBErrors( @status, nil);
    result := false;
  end
  else
    result := true;
end;

// get table structure
// return: number of fields
function get_fields(sTablename: string): short;
var
  status: status_vector;
  errCode: isc_status;
  fetchStatus: isc_status;
  out_da: PXSQLDA;
  out_var: XSQLVAR;
  STHandle: isc_stmt_handle;
  TRHandle: isc_tr_handle;
  i: short;
  j: short;
  field_name: array[0..FIELDNAMELEN] of char;
  field_type: short;
  field_length: short;
  flag_name: short;
  flag_type: short;
  flag_length: short;

begin
  STHandle := nil;
  // ruimte for number of columns
  getmem(out_da, SQLDA_LENGTH(NFIELDSQUERY));
  out_da^.version := SQLDA_VERSION1;
  out_da^.sqln := NFIELDSQUERY;
  TRHandle := StartTransAction;
  if TRHandle = nil then begin
    result := 0;
    exit;
  end;

  errCode := isc_dsql_allocate_statement(@status, @DBHandle, @STHandle);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    result := 0;
    exit;
  end;

  errCode := isc_dsql_prepare(@status, @TRHandle, @STHandle, 0 ,
             PChar(FIELDQUERY1 + '"' + sTablename + '"' + FIELDQUERY2), 1, nil);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    result := 0;
    exit;
  end;

  errCode := isc_dsql_execute(@status, @TRHandle, @STHandle, 1, nil);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    result := 0;
    exit;
  end;

  errCode := isc_dsql_describe(@status, @STHandle, 1, out_da);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    result := 0;
    exit;
  end;

  field_name[31] := #0; // add a null character

  i := 0;
  out_da^.sqlvar[i].sqldata := @field_name;
  out_da^.sqlvar[i].sqlind := @flag_name;
  i := 1;
  out_da^.sqlvar[i].sqldata := @field_type;
  out_da^.sqlvar[i].sqlind := @flag_type;
  i := 2;
  out_da^.sqlvar[i].sqldata := @field_length;
  out_da^.sqlvar[i].sqlind := @flag_length;

  fetchStatus := 0;
  j := 0;
  try
    fetchStatus := isc_dsql_fetch(@status, @STHandle, 1, out_da);
    while fetchStatus = 0 do begin
      fields[j].field_name := StrPas(field_name);
      fields[j].field_type := field_type;
      fields[j].field_length := field_length;
      fetchStatus := isc_dsql_fetch(@status, @STHandle, 1, out_da);
      inc(j);
    end;
    if fetchStatus <> 100 then begin
      HandleIBErrors(@status, TRHandle);
      result := 0;
      exit;
    end;
  finally;
    isc_dsql_free_statement(@status, @STHandle, DSQL_Close);
    Result := j;
    if not RollBack(TRHandle) then begin
      result := 0;
    end;
  end;
end;

// Empty import table
function DeleteTable(sTablename: string): boolean;
var
  status: status_vector;
  errCode: isc_status;
  TRHandle: isc_tr_handle;

begin
  TRHandle := StartTransAction;

  ProgMsg(DateTimeToStr(now) + ' - Start delete ' + sTablename + '...');

  errCode := isc_dsql_execute_immediate(@status, @DBHandle, @TRHandle, 0,
             PChar('delete from ' + sTablename), 1, nil);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    result := false;
    exit;
  end;

  Commit(TRHandle);
  ProgMsg(DateTimeToStr(now) + ' - OK.');
  result := true;

end;

function check_fields(nfields: short): short;
var
  i: integer;
  reclen: short;
begin
  if bDebug then begin
    ProgMsg('');
    ProgMsg(' nr  field name                      type  length');
    ProgMsg('---  ------------------------------- ----  ------');
  end;
  reclen := 0;
  for i := 0 to nfields - 1 do begin
    inc(reclen, fields[i].field_length);
    if bDebug then
      ProgMsg(FmtInt(i,3) +  ': ' + fields[i].field_name +
              FmtInt(fields[i].field_type,5) + FmtInt(fields[i].field_length,8));
    if (fields[i].field_type <> 14) and (fields[i].field_type <> 37) then begin
      ProgMsg('only type CHAR.');
      result := 0;
      RetCode := -20;
      exit;
    end;
  end;
  if bDebug then begin
    ProgMsg('                                           ------');
    ProgMsg('record length: ' + FmtInt(reclen,34) + ' bytes');
    ProgMsg('# of fields:   ' + IntToStr(nfields));
    ProgMsg('');
  end;

  Result := reclen;

end;

function OpenLogFile(sLogfile: string): boolean;
begin
  {$I-}
  AssignFile(FLog, sLogfile);
  if not FileExists(sLogfile) then
    Rewrite(FLog)
  else
    Append(FLog);
  {$I+}
  if IOResult <> 0 then begin
    ProgMsg(sLogfile +' cannot be opened.');
    RetCode := -10;
    result := false;
  end
  else
    result := true;
end;

function OpenFile(sFilename: string; reclen: short): boolean;
begin
  if not FileExists(sFilename) then begin
    ProgMsg(sFilename + ' not found.');
    RetCode := -11;
    result := false;
  end
  else begin
    {$I-}
    AssignFile(FIn, sFilename);
    FileMode := 0;
    Reset(FIn, reclen);
    {$I+}
    if IOResult <> 0 then begin
      ProgMsg(sFilename + ' cannot be opened.');
      RetCode := -12;
      result := false;
    end
    else
      Result := true;
  end;
end;

procedure ProcessFile(reclen: short; sTablename: string; maxrec: LongInt);
var
  rectel: longInt;
  bytes_read: integer;
  i: short;
  offset: short;
  len: short;
  blank: array[0..MAXFIELDLEN] of char;
  inputbuf: array [0..MAXRECLEN] of char;

  status: status_vector;
  errCode: isc_status;
  fetchStatus: isc_status;
  out_da: PXSQLDA;
  out_var: XSQLVAR;
  STHandle: isc_stmt_handle;
  TRHandle: isc_tr_handle;
  InsertSQL: string;

  nullval: integer;
  notnullval: short;

begin
  fillchar(blank,sizeof(blank),#0);
  nullval := -1;
  notnullval := 0;
  rectel := 0;
  InsertSQL := 'INSERT INTO ' + sTablename + ' (';
  for i := 0 to nfields - 1 do begin
    InsertSQL := InsertSQL + trim(fields[i].field_name);
    if i <> (nfields - 1) then InsertSQL := InsertSQL + ',';
  end;

  InsertSQL := InsertSQL + ') values (';
  for i := 0 to nfields - 1 do begin
    if i = (nfields - 1)then
      InsertSQL := InsertSQL + '?)'
    else
      InsertSQL := InsertSQL + '?,';
  end;

  if bDebug then begin
    ProgMsg('');
    ProgMsg('Insert string: ' + StrPas(PChar(InsertSQL)));
    ProgMsg('');
  end;

  STHandle := nil;
  // space for number of columns
  getmem(out_da, SQLDA_LENGTH(nfields));
  out_da^.version := SQLDA_VERSION1;
  out_da^.sqln := nfields;
  TRHandle := StartTransAction;

  errCode := isc_dsql_allocate_statement(@status, @DBHandle, @STHandle);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    exit;
  end;

  errCode := isc_dsql_prepare(@status, @TRHandle, @STHandle, 0 ,
                             PChar(InsertSQL), 1, out_da);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    exit;
  end;

  errCode := isc_dsql_describe_bind(@status, @STHandle, 1, out_da);
  if errCode <> 0 then begin
    HandleIBErrors(@status, TRHandle);
    exit;
  end;

  out_da^.sqld := out_da^.sqln;
  for i := 1 to nfields - 1 do
    out_da^.sqlvar[i].sqltype := SQL_TEXT + 1;

  ProgMsg(DateTimeToStr(now) + ' - Start dataload.');

  try
    BlockRead(FIn, inputbuf, 1, bytes_read);
    while (bytes_read <> 0) do begin
      if (maxrec <> 0) and (rectel = maxrec) then break;
      inc(RecTel);
      offset := 0;
      for i := 0 to nfields - 1 do begin
        out_da^.sqlvar[i].sqldata := @inputbuf[offset];
        len := offset + fields[i].field_length - 1;
        if StrLIComp(@inputbuf[offset], blank, fields[i].field_length) = 0 then
          out_da^.sqlvar[i].sqlind := @nullval
        else
          out_da^.sqlvar[i].sqlind := @notnullval;
        inc(offset,fields[i].field_length);
      end;
      errCode := isc_dsql_execute(@status, @TRHandle, @STHandle, 1, out_da);
      if errCode <> 0 then HandleIBErrors(@status, TRHandle);
      if bDebug and (RecTel mod 100 = 0) then
        ProgMsg(DateTimeToStr(now) + ' - record: ' + IntToStr(RecTel));
      BlockRead(FIn, inputbuf, 1, bytes_read);
    end;
  except
  on E:EInOutError do begin
    ProgMsg('');
    ProgMsg(E.Message);
    isc_dsql_free_statement(@status, @STHandle, DSQL_Close);
    RollBack(TRHandle);
    RetCode := -30;
    exit;
  end;
end;

RetCode := RecTel;
if bDebug and (RecTel mod 100 <> 0) then
  ProgMsg(DateTimeToStr(now) + ' - record: ' + IntToStr(RecTel));
ProgMsg(DateTimeToStr(now) + ' - End dataload.');
isc_dsql_free_statement(@status, @STHandle, DSQL_Close);
Commit(TRHandle);

end;


function IBLoadProc(sDBName:     string;
                    sTablename:  string;
                    sFilename:   string;
                    sUsername:   string;
                    sPassword:   string;
                    sLogfile:    string;
                    maxrec:      LongInt;
                    insert:      boolean;
                    debug:       boolean;
                    silent:      boolean;
                    log:         boolean): LongInt;
var
  ok: boolean;

begin
  retCode := -999;
  bDebug  := debug;
  bSilent := silent;
  bLog    := false;

  if (log and OpenLogFile(sLogfile)) or not log then begin
    bLog := log;
    ProgMsg(DateTimeToStr(now) + ' - Start IBLoadProcedure version ' + ibload_version);
    ProgMsg('Database:              '+ sDBName);
    ProgMsg('Table:                 '+ sTablename);
    ProgMsg('File:                  '+ sFilename);
    ProgMsg('User:                  '+ sUsername);
    if bDebug then begin
//    ProgMsg('Password:              '+ sPassword);
      ProgMsg('Logfile:               '+ slogfile);
      ProgMsg('Max.recs:              '+ IntToStr(maxrec));
      if insert then
        ProgMsg('Insert:                true')
      else
        ProgMsg('Insert:                false');
      if bDebug then
        ProgMsg('Debug:                 true')
      else
        ProgMsg('Debug:                 false');
      if bSilent then
        ProgMsg('Silent:                true')
      else
        ProgMsg('Silent:                false');
      if bLog then
        ProgMsg('Log:                   true')
      else
        ProgMsg('Log:                   false');
      ProgMsg('Maximum # of fields  : ' + IntToStr(MAXFIELDS));
      ProgMsg('Maximum record length: ' + IntToStr(MAXRECLEN));
      ProgMsg('Maximum veld length  : ' + IntToStr(MAXFIELDLEN));
    end;
    ProgMsg('');

    LoadApi;
    if Connect(sDBName, sUsername, sPassword) then begin
      nfields := get_fields(upperCase(sTablename));
      if nfields = 0 then
        writeln('Table ', sTablename, ' not found.')
      else begin
        reclen := check_fields(nfields);
        if (reclen > 0) and OpenFile(sFilename, reclen) then begin
          if insert then
            ok := DeleteTable(sTablename)
          else
            ok := true;
          if ok then
            ProcessFile(reclen, sTablename, maxrec);
          CloseFile(FIn);
        end;
      end;
      DisConnect;
    end;
    FreeApi;
  end;

  ProgMsg(DateTimeToStr(now) + ' - End IBLoadProcedure; return code: ' + IntToStr(RetCode));
  ProgMsg('-----------------');
  ProgMsg('');
  if bLog then CloseFile(FLog);
  Result := RetCode;

end;
end.
