(*
   WAV Parser Component and Utilities
   Version 01.00
   (C) 1996, Glen Why
*)

unit WavParser;

interface
uses
  Windows, MMSystem, SysUtils, Classes;

const

  FOURCC_WAVE = $45564157;
  FOURCC_FMT_ = $20746D66;
  FOURCC_DATA = $61746164;


type

  EWavParserError = class( Exception );
  EChunkNotFound = class( EWavParserError );


  TWavParser = class( TComponent )
  private
    FFileName :TFileName;
    FIOHandle :HMMIO;
    procedure SetFileName( const Value :TFileName );
    function GetChunks( ChunkName :FOURCC; ParentChunk :TMMCKINFO ):TMMCKINFO;
    function GetRiffChunk( CkType :FOURCC ) :TMMCKINFO;
  protected
    procedure FindChunk( Ck, ParentCk :PMMCKINFO; Flags :Integer );
    procedure SeekSource( P :Integer );
    procedure Rewind;
    procedure ValidateIOHandle;
  public
    destructor Destroy; override;
    function GetCkData( Ck :TMMCKINFO; Var Data ) :Integer;
    property FileName :TFileName
      read FFileName write SetFileName;
    property Chunks[ ChunkName :FOURCC; ParentChunk :TMMCKINFO ] :TMMCKINFO
      read GetChunks;
    property RiffChunk[ CkType :FOURCC ] :TMMCKINFO
      read GetRiffChunk;
  end;


  TRiffChunkDataExtractor = class( TObject )
  private
    FWavParser :TWavParser;
    FCkID :FOURCC;
    FRiffType :FOURCC;
    FRiffCk :TMMCKINFO;
    FCkInfo :TMMCKINFO;
  public
    constructor Create( const AFileName :TFileName;
     ARiffType, ACKID :FOURCC );
    destructor Destroy; override;
    procedure Extract( var Buffer );
    property DataLength :Integer
      read FCkInfo.ckSize;
    property RiffType :FOURCC
      read FRiffType;
    property CkID :FOURCC
      read FCkID;
  end;

  TWaveDataExtractor = class( TRiffChunkDataExtractor )
  public
    constructor Create( const AFileName :TFileName );
  end;

  TWaveFormatExtractor = class( TRiffChunkDataExtractor )
  public
    constructor Create( const AFileName :TFileName );
  end;

procedure Register;

implementation

uses
 Consts;

{$R WavParser.Res}
{$I WavParser.Inc}

procedure Register;
begin
 RegisterComponents( 'more...', [ TWavParser ] );
end;

{ TWavParser }

procedure TWavParser.SetFileName( const Value :TFileName );
begin

  if FIOHandle <> 0 then begin
    mmioClose( FIOHandle, 0 );
    FIOHandle := 0;
    FFileName := '';
  end;

  if Value <> '' then begin
    FIOHandle := mmioOpen( PChar( Value ), nil, MMIO_READ OR MMIO_COMPAT );
    if FIOHandle = 0 then raise EWavParserError.CreateResFmt( SFOpenError, [ Value ] );
    FFileName := Value;
  end;

end;


destructor TWavParser.Destroy;
begin
  FileName := '';
  inherited Destroy;
end;


procedure TWavParser.ValidateIOHandle;
begin
 if not LongBool(FIOHandle) then
  raise EWavParserError.CreateRes( SOutOfFile );
end;

procedure TWavParser.Rewind;
begin
  SeekSource( 0 );
end;

function TWavParser.GetRiffChunk( CkType :FOURCC ) :TMMCKINFO;
begin
  rewind;
  fillchar( result, sizeof( result ), 0 );
  with result do begin
    ckID := FOURCC_RIFF;
    fccType := CkType;
  end;
  FindChunk( @result, nil, MMIO_FINDRIFF );
end;

function TWavParser.GetCkData( Ck :TMMCKINFO; Var Data ) :Integer;
begin
  SeekSource( Ck.dwDataOffset );
  result := mmioRead( FIOHandle, @Char( Data ), Ck.ckSize );
end;

function TWavParser.GetChunks( ChunkName :FOURCC;
  ParentChunk :TMMCKINFO ):TMMCKINFO;
begin
  SeekSource( ParentChunk.dwDataOffset + 4 );
  fillchar( result, sizeof( result ), 0 );
  result.ckID := ChunkName;
  FindChunk( @result, @ParentChunk, MMIO_FINDCHUNK );
end;

procedure TWavParser.FindChunk( Ck, ParentCk :PMMCKINFO; Flags :Integer );
begin
  if mmioDescend( FIOHandle, Ck, ParentCk, Flags ) <> MMSYSERR_NOERROR
    then raise EChunkNotFound.CreateRes( SCkNotFound );
end;

procedure TWavParser.SeekSource( P :Integer );
begin
  ValidateIOHandle;
  if mmioSeek( FIOHandle, P, SEEK_SET ) = -1
    then raise EWavParserError.CreateRes( SSeekError );
end;

{ TRiffChunkDataExtractor }


constructor TRiffChunkDataExtractor.Create(
 const AFileName :TFileName; ARiffType, ACKID :FOURCC );
begin
 inherited Create;
 FRiffType := ARiffType;
 FCkID := ACkID;
 FWavParser := TWavParser.Create( nil );
 FWavParser.FileName := AFileName;
 with FWavParser do begin
   FRiffCk := RiffChunk[ FRiffType ];
   FCkInfo := Chunks[ ACkID, FRiffCk ];
 end;
end;


destructor TRiffChunkDataExtractor.Destroy;
begin
 if assigned( FWavParser ) then FWavParser.free;
 inherited Destroy;
end;


procedure TRiffChunkDataExtractor.Extract( var Buffer );
begin
  FWavParser.GetCkData( FCkInfo, Buffer );
end;

{ TWaveDataExtractor }

constructor TWaveDataExtractor.Create( const AFileName :TFileName );
begin
  inherited Create( AFileName, FOURCC_WAVE, FOURCC_DATA );
end;

{ TWaveFormatExtractor }

constructor TWaveFormatExtractor.Create( const AFileName :TFileName );
begin
  inherited Create( AFileName, FOURCC_WAVE, FOURCC_FMT_ );
end;

end.
