{
********************************************************************************
*  Mdulo:        GridBox                                                      *
*  Versin:       2.11                                                         *
*  Descripcin:   Contiene dos controles TGridBox que es un ComboBox           *
*                 multicolumna y TComboBox98 al estilo Office-97 y es FREEWARE *
*  Autor:         Favio E. Ugalde Corral                                       *
*  Creacin:       4 de agosto     de 1997                                     *
*  Modificacin:   3 de agosto     de 1998                                     *
*  E-Mail:         fugalde@geocities.com                                       *
*  URL:            http://www.geocities.com/SiliconValley/Grid/2582/           *
*  Observaciones: Requiere RxLib v2.40 (http://rx.demo.ru/), una excelente     *
*                 librera de componentes VCL freeware con cdigo fuente.      *
********************************************************************************
}

// Para utilizar pistas (Hints) utiliza la siguiente directiva de compilacin:
// {$DEFINE UseHints}
// Requiere de la excelente clase TStringAlignGrid de dominio pblico hecha por
// Andreas Hrstemeier http://www.westend.de/~hoerstemeier
unit GridBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Mask, Grids, ToolEdit, Menus
  {$IFDEF UseHints}
  ,AliGrid
  {$ENDIF}
  ;

type
  TGBStyle      = (gsDropDown, gsDropDownList, gsOwnerDraw);
  TGBSelectType = (stCell, stRow);
  TGBGridLines  = (glNone, glHorizontal, glVertical, glBoth);
  TGBDrawItemEvent  = procedure(Sender: TObject; Index, Col: Integer;
                                Rect: TRect; State: TOwnerDrawState) of object;
  TGBClickItemEvent = procedure(Sender: TObject; Index, Col: Integer;
                                var CanSelect: Boolean) of object;
  {$IFDEF UseHints}
  TGBShowItemHintEvent = procedure(Sender: TObject; Index, Col: Integer;
                                   var Hint: string) of object;
  {$ENDIF}

  TCustomGridBox = class(TCustomComboEdit)
  private
    { Private declarations }
    FCanvas: TCanvas;             // Lienzo para dibujar en el TCustomComboEdit
    FColor: TColor;               // Color del control
    FDummyPopupMenu: TPopupMenu;  // Men emergente que no hace nada para los estilos gsDropDownList, gsOwnerDraw (as se evita el de omisin de TEdit)
    FDrawingGrid: Boolean;        // Indica si se est dibujando en la cuadrcula (Grid)
    FFlat: Boolean;               // Indica si se dibuja plano o no
    FStyle: TGBStyle;             // Estilo (son los mismos que para un ComboBox)
    FSelectType: TGBSelectType;   // Tipo de seleccin de los elementos (por celda individual  por rengln completo)
    FNeedAdjustDropDown: Boolean; // Se requiere ajustar la ventana de desplegado ("Grid")
    FDropDownRows: Integer;       // Nmero de filas visibles en el "ComboBox"
    FFixedColWidths: Boolean;     // Se tienen anchos fijos (iguales) para todas las columnas
    FFixedRowHeights: Boolean;    // Se tienen altos fijos (iguales) para todas los renglones
    FItemIndex: Integer;          // Indice del elemento actual seleccionado
    FItems: TStrings;             // Elementos disponibles para seleccionar
    FItemsCount: Integer;         // Total de elementos contenidos en "FItems"
    FInternalChange: Boolean;     // Indica si se cambia la propiedad "Text" internamente
    FMouseInControl: Boolean;     // Indica si el cursor est sobre el control
    FOnDrawItem: TGBDrawItemEvent;   // Evento para dibujar un elemento
    FOnClickItem: TGBClickItemEvent; // Evento para confirmar la seleccin de un elemento
    {$IFDEF UseHints}
    FOnShowItemHint: TGBShowItemHintEvent;
    {$ENDIF}
    procedure DrawComboBox(DC: HDC);
    procedure RedrawBorder(const Clip: HRGN);
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCPaint(var Message: TMessage); message WM_NCPAINT;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    function  GetDropDownCols: Integer;
    procedure SetDropDownCols(Value: Integer);
    procedure SetDropDownRows(Value: Integer);
    function  GetGridLines: TGBGridLines;
    procedure SetGridLines(Value: TGBGridLines);
    function  GetGridLineWidth: Integer;
    procedure SetGridLineWidth(Value: Integer);
    function  GetDefaultColWidth: Integer;
    procedure SetDefaultColWidth(Value: Integer);
    function  GetDefaultRowHeight: Integer;
    procedure SetDefaultRowHeight(Value: Integer);
    function  GetColWidths(Index: Longint): Integer;
    procedure SetColWidths(Index: Longint; Value: Integer);
    function  GetRowHeights(Index: Longint): Integer;
    procedure SetRowHeights(Index: Longint; Value: Integer);
    procedure SetItemIndex(Value: Integer);
    procedure SetItems(Value: TStrings);
    function  GetItemCol(Index, Col: Integer): string;
    procedure SetItemCol(Index, Col: Integer; S: string);
    procedure SetStyle(Value: TGBStyle);
    procedure SetSelectType(Value: TGBSelectType);
    function  GetCanvas: TCanvas; // Obtiene el canvas del objeto activo
    function  GetCursor: TCursor;
    procedure SetCursor(Value: TCursor);
    procedure SetColor(Value: TColor);
    function  GetPopupMenu: TPopupMenu;
    procedure SetPopupMenu(Value: TPopupMenu);
    procedure SetFlat(Value: Boolean);
  protected
    { Protected declarations }
    procedure Loaded; override;
    procedure Change; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure ButtonClick; override;
    procedure AdjustBounds; virtual;
    procedure AcceptValue(const Value: Variant); override;
    procedure SetPopupValue(const Value: Variant); override;
    procedure DrawItem(Index, Col: Integer; Rect: TRect;
              State: TOwnerDrawState); virtual;
    function  ClickItem(Index, Col: Integer): Boolean; virtual;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
              X, Y: Integer); override;
    {$IFDEF UseHints}
    procedure ShowItemHint(Index, Col: Integer; var Hint: string); virtual;
    {$ENDIF}

    property ColWidths[Index: Longint]: Integer read GetColWidths write SetColWidths;
    property RowHeights[Index: Longint]: Integer read GetRowHeights write SetRowHeights;
    property ItemCol[Index, Col: Integer]: string read GetItemCol write SetItemCol;
    property ItemIndex: Integer read FItemIndex write SetItemIndex default -1;
    property Canvas: TCanvas read GetCanvas;
    property Cursor: TCursor read GetCursor write SetCursor default crDefault;
    property Color: TColor read FColor write SetColor default clWindow;
    property PopupMenu: TPopupMenu read GetPopupMenu write SetPopupMenu;
    property Flat: Boolean read FFlat write SetFlat default False;

    property SelectType: TGBSelectType read FSelectType write SetSelectType default stCell;
    property Style: TGBStyle read FStyle write SetStyle default gsDropDown;
    property Items: TStrings read FItems write SetItems;

    property DropDownCols: Integer read GetDropDownCols write SetDropDownCols default 5;
    property DropDownRows: Integer read FDropDownRows write SetDropDownRows default 5;
    property GridLines: TGBGridLines read GetGridLines write SetGridLines default glBoth;
    property GridLineWidth: Integer read GetGridLineWidth write SetGridLineWidth default 1;
    property DefaultColWidth: Integer read GetDefaultColWidth write SetDefaultColWidth default 64;
    property DefaultRowHeight: Integer read GetDefaultRowHeight write SetDefaultRowHeight default 24;

    property OnDrawItem: TGBDrawItemEvent read FOnDrawItem write FOnDrawItem;
    property OnClickItem: TGBClickItemEvent read FOnClickItem write FOnClickItem;
    {$IFDEF UseHints}
    property OnShowItemHint: TGBShowItemHintEvent read FOnShowItemHint write FOnShowItemHint;
    {$ENDIF}
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  published
    { Published declarations }
  end;

  TGridBox = class(TCustomGridBox)
  public
    { Public declarations }
    property ColWidths;
    property RowHeights;
    property ItemCol;
    property ItemIndex;
    property Canvas;

  published
    { Published declarations }
    property Flat;
    property SelectType;
    property Style;
    property DropDownCols;
    property DropDownRows;
    property GridLines;
    property GridLineWidth;
    property DefaultColWidth;
    property DefaultRowHeight;
    property Items;

    property AutoSelect;
    property CharCase;
    property Color;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property MaxLength;
    property OEMConvert;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;

    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;

    property OnDrawItem;
    property OnClickItem;
    {$IFDEF UseHints}
    property OnShowItemHint: TGBShowItemHintEvent read FOnShowItemHint write FOnShowItemHint;
    {$ENDIF}
  end;

  TComboBox98 = class(TCustomGridBox)
  private
    { Private declarations }
    FOnDrawItem: TDrawItemEvent;
  protected
    { Protected declarations }
    procedure AdjustBounds; override;
    procedure DrawItem(Index, Col: Integer; Rect: TRect;
              State: TOwnerDrawState); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;

    property ItemIndex;
    property Canvas;
  published
    { Published declarations }
    property Flat;
    property Style;
    property DropDownRows;
    property Items;

    property AutoSelect;
    property CharCase;
    property Color;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property MaxLength;
    property OEMConvert;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;

    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;

    property OnDrawItem: TDrawItemEvent read FOnDrawItem write FOnDrawItem;
    {$IFDEF UseHints}
    property OnShowItemHint: TGBShowItemHintEvent read FOnShowItemHint write FOnShowItemHint;
    {$ENDIF}
  end;

procedure Register;

implementation

uses Consts, ExtCtrls, MaxMin;

type
  TGridBoxStrings = class(TStrings)
  private
    FGridBox: TCustomGridBox;
  protected
    function  GetCount: Integer; override;
    function  Get(Index: Integer): string; override;
    procedure Put(Index: Integer; const S: string); override;
    function  GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    function  Add(const S: string): Integer; override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure Delete(Index: Integer); override;
    procedure Clear; override;
  end;

  {$IFDEF UseHints}
  TGrid = class(TStringAlignGrid)
  {$ELSE}
  TGrid = class(TStringGrid)
  {$ENDIF}
  private
    { Private declarations }
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    { Protected declarations }
    procedure CreateParams(var Params: TCreateParams); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
  end;

  TDropDownGrid = class(TPopupWindow)
  private
    { Private declarations }
    FStringGrid: TGrid;  // Cuadrcula "Grid"
    FClose: Boolean;     // Indica si se debe cerrar la cuadrcula

    procedure GridMouseDown(Sender: TObject; Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer);
    procedure GridMouseUp(Sender: TObject; Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer);
    procedure GridSelectCell(Sender: TObject; Col, Row: Longint;
              var CanSelect: Boolean);
    procedure GridDrawCell(Sender: TObject; Col, Row: Longint; Rect: TRect;
              State: TGridDrawState);
    {$IFDEF UseHints}
    procedure GridShowCellHint(Sender: TObject; Col, Row: Longint;
              var Hint: string);
    {$ENDIF}
  protected
    { Protected declarations }
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    function  GetValue: Variant; override;
    procedure SetValue(const Value: Variant); override;
    procedure CloseUp(Accept: Boolean); override;
    procedure AdjustDropDown(nDropDownRows: integer);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  end;

{ TGridBoxStrings }

function  TGridBoxStrings.GetCount: Integer;
begin
  Result := FGridBox.FItemsCount;
end;

function TGridBoxStrings.Get(Index: Integer): string;
begin
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    if FSelectType = stCell then
      Result := Cells[Index mod ColCount, Index div ColCount]
    else // stRow
      Result := Rows[Index].CommaText;
end;

procedure TGridBoxStrings.Put(Index: Integer; const S: string);
begin
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    if FSelectType = stCell then
      Cells[Index mod ColCount, Index div ColCount] := S
    else // stRow
      Rows[Index].CommaText := S;
  // Notifica que cambi
  SetUpdateState(False);
end;

function  TGridBoxStrings.GetObject(Index: Integer): TObject;
begin
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    if FSelectType = stCell then
      Result := Objects[Index mod ColCount, Index div ColCount]
    else // stRow
      Result := Objects[0, Index];
end;

procedure TGridBoxStrings.PutObject(Index: Integer; AObject: TObject);
begin
  //AObject := Self;
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    if FSelectType = stCell then
      Objects[Index mod ColCount, Index div ColCount] := AObject
    else // stRow
      Objects[0, Index] := AObject;
  // Notifica que cambi
  SetUpdateState(False);
end;

function  TGridBoxStrings.Add(const S: string): Integer;
begin
  try
    with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
      begin
      Result := FItemsCount;
      FItemsCount := FItemsCount + 1;
      if FSelectType = stCell then
        begin
        if (FItemsCount - 1) div ColCount >= RowCount then
          RowCount := RowCount + 1;
        Cells[Result mod ColCount, Result div ColCount] := S;
        end
      else // stRow
        begin
        RowCount := FItemsCount;
        Rows[Result].CommaText := S;
        end;
      end;
  except
    raise EStringListError.Create(SInsertLineError);
  end;
  // Notifica que cambi
  SetUpdateState(False);
end;

procedure TGridBoxStrings.Insert(Index: Integer; const S: string);
var
  nItem: Integer;
begin
  // Agrega un elemento nulo al final
  Add('');
  // Recorre los elementos y establece el nuevo
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    if FSelectType = stCell then
      begin
      for nItem := FItemsCount - 1 downto Index + 1 do
        Cells[nItem mod ColCount, nItem div ColCount] :=
          Cells[(nItem - 1) mod ColCount, (nItem - 1) div ColCount];
      Cells[Index mod ColCount, Index div ColCount] := S;
      end
    else // stRow
      begin
      for nItem := FItemsCount - 1 downto Index + 1 do
        Rows[nItem].CommaText := Rows[nItem - 1].CommaText;
      Rows[Index].CommaText := S;
      end;
  // Notifica que cambi
  SetUpdateState(False);
end;

procedure TGridBoxStrings.Delete(Index: Integer);
var
  nItem: Integer;
begin
  // Recorre los elementos y elimina el ltimo
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    begin
    if FSelectType = stCell then
      begin
      for nItem := Index to FItemsCount - 2 do
        Cells[nItem mod ColCount, nItem div ColCount] :=
          Cells[(nItem + 1) mod ColCount, (nItem + 1) div ColCount];
      FItemsCount := FItemsCount - 1;
      if FItemsCount mod ColCount = 0 then
        RowCount := RowCount - 1;
      end
    else // stRow
      begin
      for nItem := Index to FItemsCount - 2 do
        Rows[nItem].CommaText := Rows[nItem + 1].CommaText;
      FItemsCount := FItemsCount - 1;
      RowCount := FItemsCount;
      end;
    // El elemento actual debe estar en un rango vlido
    if FItemIndex >= FItemsCount then
      SetItemIndex(FItemsCount - 1);
    end;
  // Notifica que cambi
  SetUpdateState(False);
end;

procedure TGridBoxStrings.Clear;
begin
  with FGridBox, (FPopup as TDropDownGrid).FStringGrid do
    begin
    Row         := 0;
    RowCount    := 1;
    Rows[0].CommaText := '';
    FItemsCount := 0;
    // El elemento actual debe estar en un rango vlido
    SetItemIndex(-1);
    end;
  // Notifica que cambi
  SetUpdateState(False);
end;

procedure TGridBoxStrings.SetUpdateState(Updating: Boolean);
begin
  // Al terminar de actualizar se requiere ajustar la ventana de desplegado ("Grid")
  if not Updating then
    FGridBox.FNeedAdjustDropDown := True;
end;

{ TGrid }

constructor TGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  ControlStyle := [csCaptureMouse, csClickEvents, csDoubleClicks];
  Enabled := False;
  BorderStyle := bsNone;
  ScrollBars := ssVertical;
  FixedCols := 0;
  FixedRows := 0;
  RowCount  := 1;
  { *** Valores por omisin ***
  ColCount  := 5;
  GetGridLineWidth := 1;
  DefaultColWidth  := 64;
  DefaultRowHeight := 24;
    *** Valores por omisin *** }
  ParentColor := True;
  Options := [goHorzLine, goVertLine, goThumbTracking];
  TabStop := False;
end;

procedure TGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
end;

procedure TGrid.CMEnabledChanged(var Message: TMessage);
begin
  if HandleAllocated and not (csDesigning in ComponentState) then
    EnableWindow(Handle, True);
end;

procedure TGrid.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style and not (WS_BORDER or WS_TABSTOP or WS_DISABLED);
end;

{ TDropDownGrid }

constructor TDropDownGrid.Create(AOwner: TComponent);
begin
  // Llama a la rutina original
  inherited;

  FStringGrid := TGrid.Create(Self);
  with FStringGrid do
    begin
    Parent := Self;
    Align  := alClient;
    OnMouseDown  := GridMouseDown;
    OnMouseUp    := GridMouseUp;
    OnSelectCell := GridSelectCell;
    OnDrawCell   := GridDrawCell;
    {$IFDEF UseHints}
    OnShowCellHint := GridShowCellHint;
    {$ENDIF}
    Visible := True;
    end;
end;

destructor TDropDownGrid.Destroy;
begin
  Destroying;

  FStringGrid.Free;

  // Llama a la rutina original
  inherited;
end;

procedure TDropDownGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);

  FStringGrid.KeyDown(Key, Shift);
end;

procedure TDropDownGrid.GridMouseDown(Sender: TObject; Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
var
  nItem, nCol: Integer;
  Col, Row: Longint;
  CanSelect: Boolean;
begin
  // Si todava no se debe cerrar la cuadrcula y se presion el botn izquierdo
  if (not FClose) and (Button = mbLeft) and (Shift = [ssLeft]) then
    begin
    // Si se puede seleccionar la celda de la cuadrcula "Grid" por contener
    // un elemento, se calcula el nmero de elemento y columna (si es el caso)
    // y se notifica el clic sobre el elemento; si el evento "OnClickItem" as
    // lo desea se selecciona el elemento y se cierra la cuadrcula
    CanSelect := True;
    FStringGrid.MouseToCell(X, Y, Col, Row);
    GridSelectCell(Self, Col, Row, CanSelect);
    if CanSelect then
      with (Owner as TCustomGridBox) do
        begin
        // Obtiene el ndice del elemento y la columna si es el caso
        if FSelectType = stCell then
          begin
          nItem := Row * FStringGrid.ColCount + Col;
          nCol  := -1;
          end
        else // stRow
          begin
          nItem := Row;
          nCol  := Col;
          end;
        if ClickItem(nItem, nCol) then
          FClose := True;
        end;
    end;
end;

procedure TDropDownGrid.GridMouseUp(Sender: TObject; Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  // Se debe cerrar la cuadrcula?
  if FClose then
    CloseUp(True);
end;

procedure TDropDownGrid.GridSelectCell(Sender: TObject; Col, Row: Longint;
          var CanSelect: Boolean);
begin
  if (Row >= 0) and (Col >= 0) then
    with (Owner as TCustomGridBox) do
      begin
      if FSelectType = stCell then
        CanSelect := (Row * FStringGrid.ColCount + Col < FItemsCount)
      else // stRow
        CanSelect := (Row < FItemsCount);
      end
  else
    CanSelect := False;
end;

procedure TDropDownGrid.GridDrawCell(Sender: TObject; Col, Row: Longint;
          Rect: TRect; State: TGridDrawState);
var
  nItem, nCol: Integer;
  odState: TOwnerDrawState;
begin
  // Establece el estado del elemento
  if gdSelected in State then
    odState := [odSelected, odFocused]
  else
    odState := [];

  // Si no se dibuja automticamente se propone al usuario hacerlo
  if not FStringGrid.DefaultDrawing then
    begin
    // Obtiene el ndice del elemento y la columna si es el caso
    if (Owner as TCustomGridBox).FSelectType = stCell then
      begin
      nItem := Row * FStringGrid.ColCount + Col;
      nCol  := -1;
      end
    else // stRow
      begin
      nItem := Row;
      nCol  := Col;
      end;

    // Incializa valores por omisin del lienzo de la cuadrcula
    FStringGrid.Canvas.Font   := Font;
    FStringGrid.Canvas.Brush  := Brush;

    // Se utiliza el color de acuerdo al estado del control
    if odSelected in odState then
      begin
      FStringGrid.Canvas.Font.Color  := clHighlightText;
      FStringGrid.Canvas.Brush.Color := clHighlight;
      end;
    // Fondo por omisin
    FStringGrid.Canvas.FillRect(Rect);

    // Si est dentro del rango dibuja el elemento correspondiente
    with (Owner as TCustomGridBox) do
      if nItem < FItemsCount then
        begin
        FDrawingGrid := True;
        DrawItem(nItem, nCol, Rect, odState);
        FDrawingGrid := False;
        end;
    end;

  // Dado que la cuadrcula "FStringGrid" nunca recibe el foco y se hace
  // creer al usuario que s, se tiene que dibujar el foco en el elemento
  // cuando se dibuja automticamente o no (propiedad DefaultDrawing)

  // Se dibuja el foco si es necesario
  {if ((Owner as TCustomGridBox).FSelectType = stCell) and
     (odFocused in odState) then
    FStringGrid.Canvas.DrawFocusRect(Rect);}
end;

{$IFDEF UseHints}
procedure TDropDownGrid.GridShowCellHint(Sender: TObject; Col, Row: Longint;
          var Hint: string);
var
  nItem, nCol: Integer;
begin
  // Calcula el nmero de elemento
  if (Owner as TCustomGridBox).FSelectType = stCell then
    begin
    nItem := Row * FStringGrid.ColCount + Col;
    nCol  := -1;
    end
  else // stRow
    begin
    nItem := Row;
    nCol  := Col;
    end;
  // Si el elemento es vlido maneja el evento
  if nItem < (Owner as TCustomGridBox).FItemsCount then
     (Owner as TCustomGridBox).ShowItemHint(nItem, nCol, Hint);
end;
{$ENDIF}

function TDropDownGrid.GetValue: Variant;
begin
  if not (csDesigning in ComponentState) then
    with (Owner as TCustomGridBox), FStringGrid do
      if FSelectType = stCell then
        Result := Row * ColCount + Col
      else // stRow
        Result := Row;
end;

procedure TDropDownGrid.SetValue(const Value: Variant);
begin
  with (Owner as TCustomGridBox), FStringGrid do
    if Value < 0 then
      begin
      Col := 0;
      Row := 0;
      end
    else
      if FSelectType = stCell then
        begin
        Row := Value div ColCount;
        Col := Value mod ColCount;
        end
      else // stRow
        begin
        Col := 0;
        Row := Value;
        end;
end;

procedure TDropDownGrid.CloseUp(Accept: Boolean);
begin
  // Reinicia la bandera antes de cerrar la cuadrcula
  FClose := False;
  inherited CloseUp(Accept);
end;

// Ajusta el alto y ancho la ventana de desplegado ("Grid")
// Siempre se visualizan todas las columnas (como en un "ComboBox" tradicional)
// y se visualizan slo el nmero de filas especificado en "DropDownCount"
procedure TDropDownGrid.AdjustDropDown(nDropDownRows: Integer);
var
  nVScrollBarWidth,           // Ancho de la barra de desplazamiento vertical
  nWinBorder: Integer;        // Ancho/Alto del borde de la ventana de desplegado
  nIndex: Integer;            // Indice de Columnas/Renglones
  nDimension: Integer;        // Ancho/Alto de la ventana que se acumula por cada Columna/Rengln
  nGridLineWidth: Integer;    // Ancho/Alto de la lnea divisoria de la cuadrcula
  nOffset: Integer;           // Holgura de Ancho/Alto requerida cuando no existen lneas divisorias
begin
  with FStringGrid do
    begin
    Visible := False;
    { Calcula el Alto }
    // Si existen ms filas que las posibles a visualizar
    // se visualizan menos filas del total y por ende se utiliza
    // una barra de desplazamiento vertical.
    // En caso contrario se visualizan todas las filas y no es necesario
    // una barra de desplazamiento vertical.
    if RowCount > nDropDownRows then
      nVScrollBarWidth  := GetSystemMetrics(SM_CXVSCROLL)
    else
      begin
      nVScrollBarWidth  := 0;
      nDropDownRows := RowCount;
      end;
    nWinBorder := GetSystemMetrics(SM_CYBORDER);
    // Cual es el alto de la lnea divisoria horizontal
    if goHorzLine in Options then
      begin
      nOffset := 0;
      nGridLineWidth := GridLineWidth;
      end
    else
      begin
      nOffset := 1;
      nGridLineWidth := 0;
      end;
    // Si no se cambiaron los altos se toma el de omisin o
    // se sumariza los altos de cada rengln individual.
    if (Self.Owner as TCustomGridBox).FFixedRowHeights then
      Self.Height := nDropDownRows * (DefaultRowHeight + nGridLineWidth) +
                     nWinBorder + nOffset
    else
      begin
      nDimension := 0;
      for nIndex := 0 to nDropDownRows - 1 do
        nDimension := nDimension + RowHeights[nIndex];
      Self.Height := nDimension + nDropDownRows * nGridLineWidth +
                     nWinBorder + nOffset;
      end;

    { Calcula el Ancho }
    // Siempre se visualizan todas las columnas
    nWinBorder := GetSystemMetrics(SM_CXBORDER);
    // Cual es el ancho de la lnea divisoria vertical
    if goVertLine in Options then
      begin
      nOffset := 0;
      nGridLineWidth := GridLineWidth;
      end
    else
      begin
      nOffset := 1;
      nGridLineWidth := 0;
      end;
    // Si no se cambiaron los anchos se toma el de omisin o
    // se sumariza los anchos de cada columna individual.
    if (Self.Owner as TCustomGridBox).FFixedColWidths then
      Self.Width := ColCount * (DefaultColWidth + nGridLineWidth) +
                    nVScrollBarWidth + nWinBorder + nOffset
    else
      begin
      nDimension := 0;
      for nIndex := 0 to ColCount - 1 do
        nDimension := nDimension + ColWidths[nIndex];
      Self.Width := nDimension + ColCount * nGridLineWidth +
                    nVScrollBarWidth + nWinBorder + nOffset;
      end;

    Visible := True;
    end;
end;

{ TCustomGridBox }

constructor TCustomGridBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlState := ControlState + [csCreating];
  try
    FPopup := TDropDownGrid.Create(Self);
    TPopupWindow(FPopup).OnCloseUp := PopupCloseUp;
    GlyphKind := gkDropDown;
  finally
    ControlState := ControlState - [csCreating];
  end;

  FItems := TGridBoxStrings.Create;
  TGridBoxStrings(FItems).FGridBox := Self;

  FCanvas := TCanvas.Create;
  FColor  := inherited Color;

  FDummyPopupMenu := TPopupMenu.Create(Self);
  FDummyPopupMenu.AutoPopup := False;

  FItemIndex  := -1;
  FItemsCount := 0;
  FStyle      := gsDropDown;
  FSelectType := stCell;

  FDropDownRows :=  5;         // Cinco filas se muestran por omisin
  FFixedColWidths  := True;    // Se tienen anchos fijos (iguales) para todas las columnas
  FFixedRowHeights := True;    // Se tienen altos fijos (iguales) para todas los renglones

  // Por omisin se requiere ajustar la ventana de desplegado ("Grid")
  FNeedAdjustDropDown := True;

  // Por omisin no es plano
  FFlat := False;
  // Botn angosto
  Button.Width := Button.Width - 2;
end;

destructor TCustomGridBox.Destroy;
begin
  Destroying;

  FCanvas.Free;

  if FPopup <> nil then
    TPopupWindow(FPopup).OnCloseUp := nil;
  FPopup.Free;
  FPopup := nil;

  FItems.Free;

  inherited Destroy;
end;

procedure TCustomGridBox.CMMouseEnter (var Message: TMessage);
begin
  inherited;

  FMouseInControl := True;
  RedrawBorder (0);
end;

procedure TCustomGridBox.CMMouseLeave (var Message: TMessage);
begin
  inherited;

  FMouseInControl := False;
  RedrawBorder (0);
end;

procedure TCustomGridBox.CMEnabledChanged(var Message: TMessage);
begin
  inherited;

  if not(csDesigning in ComponentState) then
    if Enabled then
      inherited Color := FColor
    else
      inherited Color := clBtnFace;
end;

procedure TCustomGridBox.CMFontChanged(var Message: TMessage);
begin
  inherited;

  if not((csDesigning in ComponentState) and (csLoading in ComponentState)) then
    AdjustBounds;
end;

procedure TCustomGridBox.WMSetFocus(var Message: TWMSetFocus);
begin
  inherited;

  Invalidate;
  if not(csDesigning in ComponentState) then
    RedrawBorder (0);
  if FStyle <> gsDropDown then // gsDropDownList, gsOwnerDraw
    HideCaret(Handle);
end;

procedure TCustomGridBox.WMKillFocus(var Message: TWMKillFocus);
begin
  inherited;

  Invalidate;
  if not(csDesigning in ComponentState) then
    RedrawBorder (0);
end;

procedure TCustomGridBox.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  inherited;

  if FFlat and not(csDesigning in ComponentState) then
    InflateRect (Message.CalcSize_Params^.rgrc[0], -3, -3);
end;

procedure TCustomGridBox.WMNCPaint(var Message: TMessage);
begin
  inherited;
  RedrawBorder (Message.WParam);
end;

procedure TCustomGridBox.WMPaint(var Message: TWMPaint);
var
  DC: HDC;
  lpPaint: TPaintStruct;
begin
  if FStyle = gsDropDown then
    inherited
  else    // gsDropDownList, gsOwnerDraw
    begin
    DC := BeginPaint(Handle, lpPaint);
    DrawComboBox(DC);
    EndPaint(Handle, lpPaint);
    Message.Result := 0;
    end;
end;

procedure TCustomGridBox.WMSize(var Message: TWMSize);
begin
  inherited;

  // Si es plano se ajustan las dimensiones del botn
  if FFlat and not(csDesigning in ComponentState) then
    with Button.Parent do
      begin
      Top    := 0;
      Left   := Self.Width - Width - 6;
      Height := Self.Height - 6;
      Button.Height := Height;
      end;
end;

procedure TCustomGridBox.DrawComboBox(DC: HDC);
var
  ARect: TRect;
  State: TOwnerDrawState;
begin
  // Incializa valores por omisin
  FCanvas.Handle := DC;
  FCanvas.Font   := Font;
  FCanvas.Brush  := Brush;

  // Obtiene las dimensiones del rea del cliente sin el botn
  ARect := Rect(0, 0, ClientWidth - (Button.Width + 2), ClientHeight);

  // Se utiliza el color de acuerdo al estado del control
  if not Enabled then
    begin
    FCanvas.Font.Color  := clGrayText;
    State := [odDisabled];
    end
  else
    //if Focused or PopupVisible then
    if Focused and not PopupVisible then
      begin
      State := [odSelected, odFocused];
      FCanvas.Font.Color  := clHighlightText;
      FCanvas.Brush.Color := clHighlight;
      end;
  // Fondo por omisin
  FCanvas.FillRect(ARect);

  // Dibuja el elemento el usuario o se hace aqu mismo
  if FStyle = gsOwnerDraw then
    DrawItem(FItemIndex, -1, ARect, State)
  else     // gsDropDownList
    // Se tiene un elemento a dibujar?
    if FItemIndex >= 0 then
      FCanvas.TextRect(ARect, ARect.Left + 1, ARect.Top + 1, Text);

  // Se dibuja el foco si es necesario
  if Focused and not PopupVisible then
    FCanvas.DrawFocusRect(ARect);

  // Ya no se dibuja ms
  FCanvas.Handle := 0;
end;

procedure TCustomGridBox.RedrawBorder (const Clip: HRGN);
var
  DC: HDC;
  R: TRect;
  NewClipRgn: HRGN;
  BtnFaceBrush, WindowBrush: HBRUSH;
begin
  if FFlat then
    begin
    DC := GetWindowDC(Handle);
    try
      { Use update region }
      if (Clip <> 0) and (Clip <> 1) then
        begin
        GetWindowRect (Handle, R);
        if SelectClipRgn(DC, Clip) = ERROR then
          begin
          NewClipRgn := CreateRectRgnIndirect(R);
          SelectClipRgn (DC, NewClipRgn);
          DeleteObject (NewClipRgn);
          end;
        OffsetClipRgn (DC, -R.Left, -R.Top);
        end;

      { This works around WM_NCPAINT problem described at top of source code }
      {no!  R := Rect(0, 0, Width, Height);}
      GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
      BtnFaceBrush := GetSysColorBrush(COLOR_BTNFACE);
      WindowBrush := GetSysColorBrush(COLOR_WINDOW);
      if ((csDesigning in ComponentState) and Enabled) or
         (not(csDesigning in ComponentState) and
         (Focused or (FMouseInControl and (Screen.ActiveControl <> Self)))) then
        begin
          //(Focused or (FMouseInControl and not(Screen.ActiveControl is TCustomGridBox)))) then begin
        Button.Flat := False;
        DrawEdge (DC, R, BDR_SUNKENOUTER, BF_RECT or BF_ADJUST);
        with R do
          begin
          FillRect (DC, Rect(Left, Top, Left+1, Bottom-1), BtnFaceBrush);
          FillRect (DC, Rect(Left, Top, Right-1, Top+1), BtnFaceBrush);
          end;
        DrawEdge (DC, R, BDR_SUNKENINNER, BF_BOTTOMRIGHT);
        InflateRect (R, -1, -1);
        FrameRect (DC, R, WindowBrush);
        end
      else
        begin
        Button.Flat := True;
        FrameRect (DC, R, BtnFaceBrush);
        InflateRect (R, -1, -1);
        FrameRect (DC, R, BtnFaceBrush);
        InflateRect (R, -1, -1);
        FrameRect (DC, R, WindowBrush);
        end;
    finally
      ReleaseDC (Handle, DC);
    end;
    end;
end;

procedure TCustomGridBox.MouseDown(Button: TMouseButton; Shift: TShiftState;
          X, Y: Integer);
var
  bDoIt: Boolean;
begin
  // Evita el comportamiento de dibujo del EDIT por omisin
  // y abre la lista de desplegado, si es el caso
  bDoIt := (Shift = [ssLeft]) and not ((FStyle = gsDropDown) or PopupVisible);

  // Rutina original
  inherited MouseDown(Button, Shift, X, Y);

  // Se debe hacer lo antes descrito?
  if bDoIt then
    begin
    ReleaseCapture;  // Evitar seleccionar el texto con el ratn (TEdit emulando TComboBox)
    ButtonClick;
    end;
end;

function  TCustomGridBox.GetDropDownCols: Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.ColCount;
end;

procedure TCustomGridBox.SetDropDownCols(Value: Integer);
begin
  // Ajusta el valor al intervalo vlido
  if Value < 0 then
    Value := 1;
  with (FPopup as TDropDownGrid) do
    if FStringGrid.ColCount <> Value then
      begin
      FStringGrid.ColCount := Value;
      // Se requiere ajustar la ventana de desplegado ("Grid")
      FNeedAdjustDropDown := True;
      end;
end;

procedure TCustomGridBox.SetDropDownRows(Value: Integer);
begin
  // Ajusta el valor al intervalo vlido
  if Value < 0 then
    Value := 1;
  if FDropDownRows <> Value then
    begin
    FDropDownRows := Value;
    // Se requiere ajustar la ventana de desplegado ("Grid")
    FNeedAdjustDropDown := True;
    end;
end;

function  TCustomGridBox.GetGridLines: TGBGridLines;
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    if goHorzLine in Options then
      if goVertLine in Options then
        Result := glBoth
      else
        Result := glHorizontal
    else
      if goVertLine in Options then
        Result := glVertical
      else
        Result := glNone;
end;

procedure TCustomGridBox.SetGridLines(Value: TGBGridLines);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    case Value of
      glNone:
        Options := Options - [goHorzLine, goVertLine];
      glHorizontal:
        Options := Options + [goHorzLine] - [goVertLine];
      glVertical:
        Options := Options - [goHorzLine] + [goVertLine];
      glBoth:
        Options := Options + [goHorzLine, goVertLine];
    end;
  // Se requiere ajustar la ventana de desplegado ("Grid")
  FNeedAdjustDropDown := True;
end;

function  TCustomGridBox.GetGridLineWidth: Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.GridLineWidth;
end;

procedure TCustomGridBox.SetGridLineWidth(Value: Integer);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    if GridLineWidth <> Value then
      begin
      GridLineWidth := Value;
      // Se requiere ajustar la ventana de desplegado ("Grid")
      FNeedAdjustDropDown := True;
      end;
end;

function  TCustomGridBox.GetDefaultColWidth: Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.DefaultColWidth;
end;

procedure TCustomGridBox.SetDefaultColWidth(Value: Integer);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    begin
    DefaultColWidth := Value;
    // Se tienen anchos fijos (iguales) para todas las columnas
    FFixedColWidths  := True;
    // Se requiere ajustar la ventana de desplegado ("Grid")
    FNeedAdjustDropDown := True;
    end;
end;

function  TCustomGridBox.GetDefaultRowHeight: Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.DefaultRowHeight;
end;

procedure TCustomGridBox.SetDefaultRowHeight(Value: Integer);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    begin
    DefaultRowHeight := Value;
    // Se tienen altos fijos (iguales) para todas los renglones
    FFixedRowHeights := True;
    // Se requiere ajustar la ventana de desplegado ("Grid")
    FNeedAdjustDropDown := True;
    end;
end;

function  TCustomGridBox.GetColWidths(Index: Longint): Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.ColWidths[Index];
end;

procedure TCustomGridBox.SetColWidths(Index: Longint; Value: Integer);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    if ColWidths[Index] <> Value then
      begin
      ColWidths[Index] := Value;
      // No se tienen anchos fijos (iguales) para todas las columnas
      FFixedColWidths  := False;
      // Se requiere ajustar la ventana de desplegado ("Grid")
      FNeedAdjustDropDown := True;
      end;
end;

function  TCustomGridBox.GetRowHeights(Index: Longint): Integer;
begin
  Result := (FPopup as TDropDownGrid).FStringGrid.RowHeights[Index];
end;

procedure TCustomGridBox.SetRowHeights(Index: Longint; Value: Integer);
begin
  with (FPopup as TDropDownGrid).FStringGrid do
    if RowHeights[Index] <> Value then
      begin
      RowHeights[Index] := Value;
      // No se tienen altos fijos (iguales) para todas los renglones
      FFixedRowHeights := False;
      // Se requiere ajustar la ventana de desplegado ("Grid")
      FNeedAdjustDropDown := True;
      end;
end;

procedure TCustomGridBox.SetItemIndex(Value: Integer);
begin
  if FItemIndex <> Value then
    begin
    if Value >= FItemsCount then
      raise ERangeError.Create(SPropertyOutOfRange)
    else
      begin
      AcceptValue(Value);
      (FPopup as TDropDownGrid).SetValue(Value);
      end;
    Invalidate;
    end;
end;

procedure TCustomGridBox.SetItems(Value: TStrings);
begin
  FItems.Assign(Value);
end;

function  TCustomGridBox.GetItemCol(Index, Col: Integer): string;
begin
  if (FSelectType = stCell) or (Col < 0) then
    Result := Items[Index]
  else // stRow
    Result := (FPopup as TDropDownGrid).FStringGrid.Cells[Col, Index];
end;

procedure TCustomGridBox.SetItemCol(Index, Col: Integer; S: string);
begin
  if (FSelectType = stCell) or (Col < 0) then
    Items[Index] := S
  else // stRow
    (FPopup as TDropDownGrid).FStringGrid.Cells[Col, Index] := S;
end;

procedure TCustomGridBox.SetStyle(Value: TGBStyle);
  function GetTextHeight: Integer;
  var
    DC: HDC;
    SaveFont: HFont;
    SysMetrics, Metrics: TTextMetric;
  begin
    DC := GetDC(0);
    try
      GetTextMetrics(DC, SysMetrics);
      SaveFont := SelectObject(DC, Font.Handle);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, SaveFont);
    finally
      ReleaseDC(0, DC);
    end;
    Result := Min(SysMetrics.tmHeight, Metrics.tmHeight);
  end;
const
  CaretWidth: array[Boolean] of Byte = (1, 2);
begin
  if FStyle <> Value then
    begin
    FStyle := Value;
    // Ningn elemento seleccionado
    SetItemIndex(-1);
    if FStyle = gsDropDown then
      begin
      CreateCaret(Handle, 0, CaretWidth[fsBold in Font.Style], GetTextHeight);
      ShowCaret(Handle);
      DirectInput := True;
      end
    else     // gsDropDownList, gsOwnerDraw
      begin
      HideCaret(Handle);
      DirectInput := False;
      end;
    with (FPopup as TDropDownGrid) do
      if FStyle = gsOwnerDraw then
        FStringGrid.DefaultDrawing := False
      else     // gsDropDown, gsDropDownList
        FStringGrid.DefaultDrawing := True;
    // Ajusta el cursor y men emergente correctos para ste estilo
    SetCursor(GetCursor);
    SetPopupMenu(GetPopupMenu);
    end;
end;

procedure TCustomGridBox.SetSelectType(Value: TGBSelectType);
begin
  if FSelectType <> Value then
    begin
    // Elimina los elementos actuales para evitar anomalas
    FItems.Clear;
    FSelectType := Value;
    with (FPopup as TDropDownGrid).FStringGrid do
      begin
      if FSelectType = stCell then
        Options := Options - [goRowSelect]
      else // stRow
        Options := Options + [goRowSelect];
      end;
    end;
end;

// Obtiene el lienzo (Canvas) del objeto activo
function  TCustomGridBox.GetCanvas: TCanvas;
begin
  // Si est abierta la ventana de desplegado ("Grid") y se est dibujando
  // en el mismo se utiliza el suyo, en caso contrario,
  // se est dibujando el "ComboBox"
  if FDrawingGrid and PopupVisible then
    Result := (FPopup as TDropDownGrid).FStringGrid.Canvas
  else
    Result := FCanvas;
end;

function  TCustomGridBox.GetCursor: TCursor;
begin
  Result := Button.Cursor;
end;

procedure TCustomGridBox.SetCursor(Value: TCursor);
begin
  // Establece el mismo cursor para el botn y para la cuadrcula
  Button.Cursor := Value;
  (FPopup as TDropDownGrid).FStringGrid.Cursor := Value;

  // La flecha es el cursor por omisin para los estilos
  // gsDropDownList y gsOwnerDraw
  if FStyle = gsDropDown then
    inherited Cursor := crDefault
  else
    if Value = crDefault then
      inherited Cursor := crArrow
    else
      inherited Cursor := Value;
end;

procedure TCustomGridBox.SetColor(Value: TColor);
begin
  if FColor <> Value then
    begin
    FColor := Value;
    if Enabled then
      inherited Color := FColor;
    end;
end;

function  TCustomGridBox.GetPopupMenu: TPopupMenu;
begin
  if FDummyPopupMenu = inherited PopupMenu then
    Result := nil
  else
    Result := inherited PopupMenu;
end;

procedure TCustomGridBox.SetPopupMenu(Value: TPopupMenu);
begin
  if FStyle = gsDropDown then
    inherited PopupMenu := Value
  else   // gsDropDownList, gsOwnerDraw
    if Assigned(Value) then
      inherited PopupMenu := Value
    else
      inherited PopupMenu := FDummyPopupMenu;
end;

procedure TCustomGridBox.SetFlat(Value: Boolean);
begin
  if FFlat <> Value then
    begin
    FFlat := Value;
    if not(csDesigning in ComponentState) then
      if FFlat then
        begin
        BorderStyle := bsNone;
        Ctl3D := False;
        end
      else
        begin
        BorderStyle := bsSingle;
        Ctl3D := True;
        end;
    end;
end;

procedure TCustomGridBox.Loaded;
begin
  inherited;
  if not(csDesigning in ComponentState) then
    AdjustBounds;
end;

procedure TCustomGridBox.Change;
begin
  // Slo se consideran los cambios externos
  if not FInternalChange then
    begin
    inherited;

    // Si se puede y se digit algo se busca su respectivo ndice
    if FStyle = gsDropDown then
      if Text = '' then
        FItemIndex := -1
      else
        FItemIndex := FItems.IndexOf(Text);
    end;
end;

procedure TCustomGridBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);

  if Shift = [] then
    begin
    if PopupVisible and
       (Key in [VK_PRIOR, VK_NEXT, VK_LEFT, VK_UP, VK_RIGHT, VK_DOWN]) then
    begin
      (FPopup as TDropDownGrid).KeyDown(Key, Shift);
      Key := 0;
    end;

    // Se cambia el elemento actual con las flechas
    if FStyle = gsDropDown then
      case Key of
        VK_UP:
          if ItemIndex > 0 then
            begin
            ItemIndex := ItemIndex - 1;
            SelectAll;
            Key := 0;
            end;
        VK_DOWN:
          if ItemIndex < Items.Count - 1 then
            begin
            ItemIndex := ItemIndex + 1;
            SelectAll;
            Key := 0;
            end;
      end
    else     // gsDropDownList, gsOwnerDraw
      begin
      case Key of
        VK_PRIOR, VK_LEFT, VK_UP:
          if ItemIndex > 0 then
            ItemIndex := ItemIndex - 1;
        VK_NEXT, VK_RIGHT, VK_DOWN:
          if ItemIndex < Items.Count - 1 then
            ItemIndex := ItemIndex + 1;
      end;
      Key := 0;
      end;
    end;
end;

procedure TCustomGridBox.ButtonClick;
begin
  // Se redibuja el control, si es el caso, para dejar de resaltar
  // el elemento actual, y as simular un ComboBox.
  if FStyle <> gsDropDown then  // gsDropDownList, gsOwnerDraw
    Invalidate;

  // Si se requiere ajustar el alto y ancho la ventana de desplegado ("Grid")
  if FNeedAdjustDropDown then
    begin
    FNeedAdjustDropDown := False;
    (FPopup as TDropDownGrid).AdjustDropDown(FDropDownRows);
    end;

  // Debido a que la primera vez no se muestra todo el rengln seleccionado
  // se necesita cambiarse dos veces de columna para que Delphi reaccione
  with (FPopup as TDropDownGrid).FStringGrid do
    if (FSelectType = stRow) and
       (ColCount > 1) then
      begin
      Col := 1;
      Col := 0;
      end;

  // Llama a la rutina original
  inherited;
end;

procedure TCustomGridBox.AdjustBounds;
var
  DC: HDC;
  SaveFont: HFONT;
  Metrics: TTextMetric;
begin
  // Se ajusta el alto del control de acuerdo a la fuente para el caso
  // que se dibuje as mismo en forma automtica.
  if FStyle <> gsOwnerDraw then  // gsDropDown, gsDropDownList
    begin
    DC := GetDC(0);
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics (DC, Metrics);
    SelectObject (DC, SaveFont);
    ReleaseDC (0, DC);

    Height := Metrics.tmHeight + 6 + 2;  // Dos ms ajusta mejor
    end;
end;

procedure TCustomGridBox.AcceptValue(const Value: Variant);
begin
  if FItemIndex <> Value then
    begin
    FItemIndex := Value;
    FInternalChange := True;
    if FItemIndex >= 0 then
      Text := ItemCol[FItemIndex, 0]
    else
      Text := '';
    FInternalChange := False;
    FPopupVisible := (FPopup <> nil) and (FPopup.Visible);
    Modified := True;
    inherited Change;
    end;
end;

procedure TCustomGridBox.SetPopupValue(const Value: Variant);
begin
  // Utiliza el ndice en lugar del texto para ubicar el elemento
  (FPopup as TDropDownGrid).SetValue(FItemIndex);
end;

procedure TCustomGridBox.DrawItem(Index, Col: Integer; Rect: TRect;
          State: TOwnerDrawState);
begin
  // Si existe se utiliza el evento o se dibuja por omisin
  if Assigned(FOnDrawItem) then
    FOnDrawItem(Self, Index, Col, Rect, State)
  else
    // Se tiene un elemento a dibujar?
    if Index >= 0 then
      Canvas.TextRect(Rect, Rect.Left + 1, Rect.Top + 1, Items[Index]);
end;

function  TCustomGridBox.ClickItem(Index, Col: Integer): Boolean;
var
  CanSelect: Boolean;
begin
  // Por omisin siempre se selecciona el elemento
  CanSelect := True;

  // Si existe se utiliza el evento
  if Assigned(FOnClickItem) then
    FOnClickItem(Self, Index, Col, CanSelect);

  Result := CanSelect;
end;

{$IFDEF UseHints}
procedure TCustomGridBox.ShowItemHint(Index, Col: Integer; var Hint: string);
begin
  // Si existe se utiliza el evento
  if Assigned(FOnShowItemHint) then
    FOnShowItemHint(Self, Index, Col, Hint);
end;
{$ENDIF}


{ TComboBox98 }

constructor TComboBox98.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  // Valores por omisin para asemejar un ComboBox
  SelectType   := stRow;
  DropDownCols := 0;
  GridLines    := glNone;
end;

procedure TComboBox98.AdjustBounds;
begin
  // Se ajusta automticamente el alto del control
  // cuando no es dibujado por el usuario
  if FStyle <> gsOwnerDraw then  // gsDropDown, gsDropDownList
    inherited;

  DefaultColWidth  := Width;
  DefaultRowHeight := Height - 6;
end;

procedure TComboBox98.DrawItem(Index, Col: Integer; Rect: TRect; State: TOwnerDrawState);
begin
  // Si existe se utiliza el evento o se dibuja por omisin
  if Assigned(FOnDrawItem) then
    FOnDrawItem(Self, Index, Rect, State)
  else
    // Se tiene un elemento a dibujar?
    if Index >= 0 then
      Canvas.TextRect(Rect, Rect.Left + 1, Rect.Top + 1, Items[Index]);
end;


procedure Register;
begin
  RegisterComponents('Additional', [TGridBox, TComboBox98]);
end;

end.

