unit GifDecl;

interface

uses
  SysUtils;        { Imports Exception }

const
  { image descriptor bit masks }
  idLocalColorTable      = $80;    { set if a local color table follows }
  idInterlaced           = $40;    { set if image is interlaced }
  idSort                 = $20;    { set if color table is sorted }
  idReserved             = $0C;    { reserved - must be set to $00 }
  idColorTableSize       = $07;    { size of color table as above }
  Trailer: byte          = $3B;    { indicates the end of the GIF data stream }
  ExtensionIntroducer: byte = $21;
  ImageSeparator: Char   = ',';

  { logical screen descriptor packed field masks }
  lsdGlobalColorTable = $80;  { set if global color table follows L.S.D. }
  lsdColorResolution = $70;   { Color resolution - 3 bits }
  lsdSort = $08;              { set if global color table is sorted - 1 bit }
  lsdColorTableSize = $07;    { size of global color table - 3 bits }
                              { Actual size = 2^value+1    - value is 3 bits }

const
  CodeTableSize = 4096;

const
  MAXCODES = 4095;   { the maximum number of different codes 0 inclusive }

  CodeMask: array[0..12] of Word = (  { bit masks for use with Next code }
  0,
  $0001, $0003, $0007, $000F,
  $001F, $003F, $007F, $00FF,
  $01FF, $03FF, $07FF, $0FFF);

type
  TDecodeRecord = record
    BitsLeft         : Integer;   { bits left in byte }
    BytesLeft        : Longint;   { bytes left in block }
    NextByte         : Longint;   { the index to the next byte in the datablock array }
    CurrByte         : Longint;   { the current byte }
    CurrentY         : Integer;   { current screen locations }
    InterlacePass    : Integer;   { interlace pass number }

    LZWCodeSize      : Byte;      { minimum size of the LZW codes in bits }
    CurrCodeSize     : Integer;   { Current size of code in bits }
    ClearCode        : Integer;   { Clear code value }
    EndingCode       : Integer;   { ending code value }
    HighCode         : Word;      { highest code that does not require decoding }
  end; { TDecodeRecord }

type
  TGifHeader = packed record
    Signature: array[0..2] of char; { contains 'GIF' }
    Version: array[0..2] of char;   { '87a' or '89a' }
  end;

  {TGifType = (Gif87a, Gif89a);}

  EGifException = class(Exception)
  end;

  TLogicalScreenDescriptor = packed record
    ScreenWidth: word;              { logical screen width }
    ScreenHeight: word;             { logical screen height }
    PackedFields: byte;             { packed fields - see below }
    BackGroundColorIndex: byte;     { index to global color table }
    AspectRatio: byte;              { actual ratio = (AspectRatio + 15) / 64 }
  end; { TLogicalScreenDescriptor }

type
  TImageDescriptor = packed record
    {Separator: byte;      { fixed value of ImageSeparator }
    { I (RPS) think it's awkward to consider the separator char a
      part of the Image Descriptor, therefore commented it out }
    ImageLeftPos: word;   { Column in pixels in respect to left edge of logical screen }
    ImageTopPos: word;    { row in pixels in respect to top of logical screen }
    ImageWidth: word;     { width of image in pixels }
    ImageHeight: word;    { height of image in pixels }
    PackedFields: byte;   { see below }
  end; { TImageDescriptor }

type
  TGraphicControlExtension = packed record
    {Introducer: byte;}      { always $21 }
    {ExtensionLabel: byte;}  { always $F9 }
    BlockSize: byte;         { should be 4 }
    PackedFields: Byte;
    DelayTime: Word;
    TransparentColorIndex: Byte;
    Terminator: Byte;
  end; { TGraphicControlExtension }

  TPlainTextExtension = packed record
    {Introducer: byte;}      { always $21 }
    {ExtensionLabel: byte;}  { always $01 }
    BlockSize: byte;         { should be 12 }
    Left, Top, Width, Height: Word;
    CellWidth, CellHeight: Byte;
    TextFGColorIndex,
    TextBGColorIndex: Byte;
    { PlTxtData not yet stored; could easily be done using a
      (Delphi long) string }
  end; { TPlainTextExtension }

  TApplicationExtension = packed record
    {Introducer: byte;}      { always $21 }
    {ExtensionLabel: byte;}  { always $FF }
    BlockSize: byte;         { shpould be 11 }
    ApplicationIdentifier: array[1..8] of Byte;
    AppAuthenticationCode: array[1..3] of Byte;
    { AppData not yet stored; could easily be done using a
      (Delphi long) string }
  end; { TApplicationExtension }

type
  TCodeTable = class
    Suffix,
    Prefix: Array[1..CodeTableSize] of Word;
    CodeSize: Byte; { number of bits necessary to encode }
    TableFull: Boolean;
    FirstSlot,
    NextSlot: Word; { index where next string will be stored }
    procedure AddEntry(NewPrefix, NewSuffix: Integer);
    procedure Clear(StartingCodeSize: Byte);
    function IsInTable(PixelString: String;
                       var Index: Integer): Boolean;
  end; { TCodeTable }

type
  TEncodedBytes = class
    Value: String; { contains the encoded bytes; using a string
                     because that is a quite convenient data strucure
                     for a growing list of bytes }
    UsedBits: Byte;
    CurrentByte: Integer; { not byte, to accommodate 'overflow' }
    constructor Create;
    procedure AppendCode(CodeValue, CodeSize: Integer);
    procedure Finish(EndCode: Word; CodeSize: Byte);
  end; { TEncodedBytes }

function NextLineNo(LineNo, ImageHeight: Integer;
                    var InterlacePass: Integer): Integer;
{ Returns the next line number for an interlaced image }


implementation

(**** methods of TCodeTable *****)

procedure TCodeTable.Clear(StartingCodeSize: Byte);
var i: Integer;
begin { TCodeTable.Clear }
  for i := 1 to CodeTableSize
  do begin
    Suffix[i] := 0;
    Prefix[i] := 0;
  end;
  CodeSize := StartingCodeSize;
  FirstSlot := 1 shl (CodeSize-1) + 2;
  NextSlot := FirstSlot;
  TableFull := False;
end;  { TCodeTable.Clear }

procedure TCodeTable.AddEntry(NewPrefix, NewSuffix: Integer);
begin { TCodeTable.AddEntry }
  Prefix[NextSlot] := NewPrefix;
  Suffix[NextSlot] := NewSuffix;
  Inc(NextSlot);
  if NextSlot = 4096
  then TableFull := True
  else
    if NextSlot > (1 shl CodeSize)
    then Inc(CodeSize)
end;  { TCodeTable.AddEntry }

function TCodeTable.IsInTable(PixelString: String;
                              var Index: Integer): Boolean;
var
  Found: Boolean;
  StringIndex: Integer;
  Pixel: Byte;
  TryIndex, PrevIndex: Integer;
begin { TCodeTable.IsInTable }
  Found := True;
  StringIndex := 1;
  Pixel := Ord(PixelString[StringIndex]);
  if Length(PixelString) = 1
  then begin
    Result := True;
    Index := Pixel;
  end
  else begin
    TryIndex := FirstSlot;
    PrevIndex := Pixel;
    repeat
      Inc(StringIndex);
      Pixel := Ord(PixelString[StringIndex]);
      Found := False;
      while not Found
            and (TryIndex < NextSlot)
      do begin
        Found := (Prefix[TryIndex] = PrevIndex) and
                 (Suffix[TryIndex] = Pixel);
        if not Found
        then Inc(TryIndex)
      end;
      if Found
      then PrevIndex := TryIndex;
    until not Found or (StringIndex = Length(PixelString));
    Result := Found;
    Index := TryIndex;
  end;
end;  { TCodeTable.IsInTable }

(**** end of methods of TCodeTable *****)

constructor TEncodedBytes.Create;
begin { TEncodedBytes.Create }
  inherited Create;
  CurrentByte := 0;
  UsedBits := 0;
  Value := '';
end;  { TEncodedBytes.Create }

procedure TEncodedBytes.AppendCode(CodeValue, CodeSize: Integer);
{ Adds the compression code to the bit stream }
var NewByte: Integer;
begin { TEncodedBytes.AppendCode }
  CurrentByte := CurrentByte + (CodeValue shl UsedBits);
  UsedBits := UsedBits+CodeSize;
  while UsedBits >= 8
  do begin
    NewByte := CurrentByte shr 8;
    CurrentByte := CurrentByte and $ff;
    Value := Value + Chr(CurrentByte);
    CurrentByte := NewByte;
    UsedBits := UsedBits - 8;
  end
end;  { TEncodedBytes.AppendCode }

procedure TEncodedBytes.Finish(EndCode: Word; CodeSize: Byte);
begin { TEncodedBytes.Finish }
  AppendCode(EndCode, CodeSize);
  if UsedBits <> 0
  then Value := Value + Chr(CurrentByte);
end;  { TEncodedBytes.Finish }

(**** end of methods of TEncodedBytes *****)


function NextLineNo(LineNo, ImageHeight: Integer;
                    var InterlacePass: Integer): Integer;
begin { NextLineNo }
  { Interlace support }
  case InterlacePass of
    1: Result := LineNo + 8;
    2: Result := LineNo + 8;
    3: Result := LineNo + 4;
    4: Result := LineNo + 2;
  end;
  if Result >= ImageHeight then
  begin
    Inc(InterLacePass);
    case InterLacePass of
      2: Result := 4;
      3: Result := 2;
      4: Result := 1;
    end;
  end;
end; { NextLineNo }

end.
 