unit GifUnit;
{ Exports TGifFile and TGifSubImage classes.
Needs non-VCL units:
ColorTbl, DynArrB, GifDecl, MoreUtil, Progress, FmProgr
The latter three can be easily left out if necessary.
This version will not work well or at all under Delphi 1 because
it relies on huge strings.

Versions:
1. Started as GifUtl.pas
by Sean Wenzel (c) 1993
   Compuserve 71736,1245

2. Converted to Delphi
by Richard Dominelli
   RichardA_Dominelli@mskcc.org

3. Converted to Delphi 2 and made into an image component
by Richard Shotbolt
   Compuserve 100327,2305

4. Enhanced component as a descendant from TBitmap and register
   the GIF file format so it's ideal for include it in the xTools
   Now you can Load bitmaps from
by Stefan Bther
   www.fabula.com
   stefc@fabula.com
   CompuServe 100023,275

5.
- Added ability to encode/save GIF files as well;
- Corrected some bugs;
- Wrote it in an even more object oriented and structured style;
- Added ability to read Extension blocks (not storing them, however)
- (Also disabled the registration of the file format.
  Sorry Stefan, that was done more or less by accident,
  for my purposes I didn't need it; you might want to add it again)
by Reinier Sterkenburg
(February-March 1997; Version 1.0 on 10 March 1997)
   r.p.sterkenburg@dataweb.nl
   sterkenburg@pml.tno.nl
   www.dataweb.nl/~r.p.sterkenburg

Left Open:
- Handling of numbers of colors in TGifFile (global color map) and
  TGifSubImage (local color map) is not completely right, but it works
  in normal cases with just a global color map; ought to be completed someday
- Storage of the GIF 89a Extension blocks
  (reading has already been implemented)
- Nicer implementation (using same data structures and identifiers in
  the Decode and Encode subroutines)

Before using the GIF format in any commercial application
be sure you know the legal issues for this format!
}

interface

uses
  Classes,         { Imports TList }
  ColorTbl,        { Imports TColorMap }
  Controls,        { Imports Cursor values }
  DynArrB,         { Imports TByteArray2D }
  Forms,           { Imports Screen }
  GifDecl,         { Imports constant and type declarations }
  Graphics,        { Imports TColor }
  MoreUtil,        { Imports ShowMessage }
  Progress,        { Imports ShowProgress }
  SysUtils,        { Imports UpperCase }
  Windows;         { Imports RGB }

type
  TGifSubImage = class(TObject)
  private
    ImageDescriptor: TImageDescriptor;
    UsesLocalColorMap: Boolean;
    BitsPerPixel: Byte;
    LZWCodeSize: Byte;
    LocalColorMap: TColorTable;

    CompressedRasterdata: String;
    { Watch out under Delphi 1: length of string is then limited to 255 }
    constructor Create(NColors: Word);
    function  Copy: TGifSubImage;

    procedure DecodeStatusbyte;
    procedure ReadImageDescriptor(var infile: File);
    procedure ReadLocalColorMap(var infile: File);
    procedure ReadRasterData(var infile: File);
    procedure DecodeRasterData;
    procedure LoadFromOpenInfile(var infile: File);

    procedure EncodeStatusbyte;
    procedure WriteImageDescriptor(var outfile: File);
    procedure WriteLocalColorMap(var outfile: File);
    procedure EncodeRasterdata;
    procedure WriteRasterData(var outfile: File);
  public
    Interlaced: Boolean;
    Pixels: TByteArray2D;
  end; { TGifSubImage }

  TGifFile = class(TObject)
  private
    Header: TGifHeader;
    ScreenDescriptor: TLogicalScreenDescriptor;
    GlobalColorMapFollows: Boolean;
    GlobalColorMap: TColorTable;
    BitsPerPixel: Byte;
    SubImages: TList;
    procedure DecodeStatusByte;
    procedure ReadExtensionBlocks(var infile: File;
                                  var SeparatorChar: Char);
    procedure ReadSignature(var infile: File);
    procedure ReadScreenDescriptor(var infile: File);
    procedure ReadGlobalColorMap(var infile: File);

    procedure EncodeStatusByte;
    procedure WriteSignature(var outfile: File);
    procedure WriteScreenDescriptor(var outfile: File);
    procedure WriteGlobalColorMap(var outfile: File);
  public
    constructor Create;
    procedure AddSubImage(Colormap: TColorTable;
                          PixelMatrix: TByteArray2D);
    function  AsTBitmap: Graphics.TBitmap;
    function  GetSubImage(Index: Integer): TGifSubImage;
    procedure LoadFromFile(filename: String);
    procedure SaveToFile(filename: String);
    procedure SaveToStream(Stream: TStream);
  end; { TGifFile }


procedure BitmapToPixelmatrix(Bitmap: Graphics.TBitmap;
                              var Colormap: TColorTable;
                              var Pixels: TByteArray2D);
{ Converts the pixels of a TBitmap into a matrix of pixels (PixelArray)
and constructs the Color table in the same process. }


implementation


procedure BitmapToPixelmatrix(Bitmap: Graphics.TBitmap;
                              var Colormap: TColorTable;
                              var Pixels: TByteArray2D);
{ Converts the pixels of a TBitmap into a matrix of pixels (PixelArray)
and constructs the Color table in the same process. }
var
  i, j: Integer;
  PixelVal: TColor;
  ColorIndex: Integer;
begin { BitmapToPixelmatrix }
  Colormap.Count := 0;
  with Bitmap
  do begin
    Pixels := TByteArray2D.Create(Width, Height);
    ShowProgress(0);
    for j := 1 to Height
    do begin
      for i := 1 to Width
      do begin
        PixelVal := Canvas.Pixels[i-1, j-1];
        ColorIndex := TColorTable_GetColorIndex(ColorMap, PixelVal);
        if ColorIndex = -1
        then begin
          Colormap.Colors[Colormap.Count] := DecodeColor(PixelVal);
          ColorIndex := Colormap.Count;
          Inc(Colormap.Count); { no check on > 256 yet }
        end;
        Pixels[i, j] := ColorIndex;
      end;
      ShowProgress(j/Height)
    end;
  end; { with }
  if Colormap.Count > 2
  then if Colormap.Count <= 16
  then Colormap.Count := 16
  else if Colormap.Count < 256
  then Colormap.Count := 256;
end;  { BitmapToPixelmatrix }


procedure MakeFlat(PixelMatrix: TByteArray2D;
                   Interlaced: Boolean;
                   var PixelArray: TBigByteArray);
{ Convert a matrix of pixels into a linear array of pixels,
taking interlacing into account if necessary }
var
  InterlacePass: Integer;
  i, j, Index, LineNo: Longint;
begin { MakeFlat }
  InterlacePass := 1;
  with PixelMatrix
  do begin
    PixelArray := TBigByteArray.Create(Count1 * Count2);
    Index := 1;
    LineNo := 0;
    for j := 1 to Count2
    do begin
      for i := 1 to Count1
      do begin
        PixelArray[Index] := PixelMatrix[i, LineNo+1];
        Inc(Index);
      end;
      if not Interlaced
      then Inc(LineNo)
      else LineNo := NextLineNo(LineNo, Count2, InterlacePass);
    end;
  end; { with }
end;  { MakeFlat }


procedure ReadColor(var infile: File; var Color: TColor);
var r, g, b: Byte;
begin { ReadColor }
  BlockRead(infile, r, 1);
  BlockRead(infile, g, 1);
  BlockRead(infile, b, 1);
  Color := RGB(r, g, b)
end;  { ReadColor }

procedure WriteColor(var outfile: File; Color: TColor);
var r, g, b: Byte;
begin { WriteColor }
  r := (Color shr 4) and $FF;
  g := (Color shr 2) and $FF;
  b := Color and $FF;
  BlockWrite(outfile, r, 1);
  BlockWrite(outfile, g, 1);
  BlockWrite(outfile, b, 1);
end;  { WriteColor }

(***** TGifSubImage *****)

constructor TGifSubImage.Create(NColors: Word);
begin { TGifSubImage.Create }
  inherited Create;
  Pixels := TByteArray2D.Create(0, 0);
  ImageDescriptor.ImageLeftPos := 0;
  ImageDescriptor.ImageTopPos := 0;
  ImageDescriptor.ImageWidth := 0;
  ImageDescriptor.ImageHeight := 0;
  ImageDescriptor.PackedFields := 0;
  UsesLocalColorMap := False;
  Interlaced := False;
  case NColors of
    2: BitsPerPixel := 1;
    16: BitsPerPixel := 4;
    256: BitsPerPixel := 8;
    else raise EGifException.Create('Number of colors must be 2, 16 or 256');
  end;  { case }
  LZWCodeSize := BitsPerPixel;
  if LZWCodeSize = 1
  then Inc(LZWCodeSize);
  TColorTable_Create(LocalColorMap, NColors);
  CompressedRasterdata := '';
  EncodeStatusByte;
end;  { TGifSubImage.Create }

function TGifSubImage.Copy: TGifSubImage;
begin { TGifSubImage.Copy }
  Result := TGifSubImage.Create(LocalColormap.Count);
  Result.Pixels := Pixels.Copy;
  Result.ImageDescriptor := ImageDescriptor;
  Result.UsesLocalColorMap := UsesLocalColorMap;
  Result.Interlaced := Interlaced;
  Result.BitsPerPixel := BitsPerPixel;
  Result.LZWCodeSize := LZWCodeSize;
  Result.LocalColorMap := LocalColorMap;
  Result.CompressedRasterdata := CompressedRasterdata;
end;  { TGifSubImage.Copy }

(***** read routines *****)
procedure TGifSubImage.DecodeStatusByte;
begin { TGifSubImage.DecodeStatusByte }
  with ImageDescriptor
  do begin
    UsesLocalColorMap := (PackedFields and idLocalColorTable) = idLocalColorTable;
    Interlaced := (ImageDescriptor.PackedFields and idInterlaced) = idInterlaced;
    BitsPerPixel := 1 + ImageDescriptor.PackedFields and $07;
  end;
end;  { TGifSubImage.DecodeStatusByte }

procedure TGifSubImage.ReadImageDescriptor(var infile: File);
begin { TGifSubImage.ReadImageDescriptor }
  BlockRead(infile, ImageDescriptor, SizeOf(ImageDescriptor));
  DecodeStatusByte;
end;  { TGifSubImage.ReadImageDescriptor }

procedure TGifSubImage.ReadLocalColorMap(var infile: File);
begin { TGifSubImage.ReadLocalColorMap }
  if UsesLocalColorMap
  then
    with LocalColorMap
    do BlockRead(infile, Colors[0], Count*SizeOf(TColorItem));
end;  { TGifSubImage.ReadLocalColorMap }

procedure TGifSubImage.ReadRasterData(var infile: File);
var
  Size, RasterDataIndex: Longint;
  BlokByteCount,
  DataByte,
  ByteNo: Byte;
begin { TGifSubImage.ReadRasterData }
  BlockRead(infile, LZWCodeSize, 1);
  BlockRead(infile, BlokByteCount, 1);
  SetLength(CompressedRasterData, 0);
  Size := 1;
  RasterDataIndex := 1;
  while BlokByteCount <> 0
  do begin
    {CompressedRasterData[RasterDataIndex] := Chr(BlokByteCount);}
    Size := Size + BlokByteCount;
    SetLength(CompressedRasterData, Size);
    for ByteNo := 1 to BlokByteCount
    do begin
      BlockRead(infile, DataByte, 1);
      CompressedRasterData[RasterDataIndex] := Chr(DataByte);
      Inc(RasterDataIndex);
    end;
    BlockRead(infile, BlokByteCount, 1);
    {if BlokByteCount <> 0
    then raise EGifException.Create('Reading of multiple bloks of data not yet supported)')}
  end;
  {CompressedRasterData[RasterDataIndex] := Chr(BlokByteCount);}
end;  { TGifSubImage.ReadRasterData }


procedure InitCompressionStream(InitLZWCodeSize: Byte;
                                CompressedRasterData: String;
                                var DecodeRecord: TDecodeRecord);
begin { InitCompressionStream }
  with DecodeRecord
  do begin
    LZWCodeSize := InitLZWCodeSize;
    if not (LZWCodeSize in [2..9])    { valid code sizes 2-9 bits }
    then raise EGifException.Create('Bad code Size');
    CurrCodeSize := succ(LZWCodeSize);
    ClearCode := 1 shl LZWCodeSize;
    EndingCode := succ(ClearCode);
    HighCode := pred(ClearCode);      { highest code not needing decoding }
    NextByte := 1;
    BytesLeft := Length(CompressedRasterData);
    BitsLeft := 0;
    CurrentY := 0;
    InterlacePass := 1;
  end;
end;  { InitCompressionStream }

{------------------------------------------------------------------------------}

function NextCode(CompressedRasterData: String;
                  var DecodeRecord: TDecodeRecord): word;
{ returns a code of the proper bit size }
var LongResult: Longint;
begin { NextCode }
  with DecodeRecord
  do begin
    if BitsLeft = 0 then       { any bits left in byte ? }
    begin                      { any bytes left }
      if BytesLeft <= 0 then   { if not: error }
        raise EGifException.Create('No data while more was expected');
      CurrByte := Ord(CompressedRasterData[NextByte]);   { get a byte }
      Inc(NextByte);                          { set the next byte index }
      BitsLeft := 8;                          { set bits left in the byte }
      Dec(BytesLeft);                         { decrement the bytes left counter }
    end;
    LongResult := CurrByte shr (8 - BitsLeft);    { shift off any previously used bits}
    while CurrCodeSize > BitsLeft do          { need more bits ? }
    begin
      if BytesLeft <= 0 then                  { any bytes left in block ? }
        raise EGifException.Create('No data while more was expected');
      CurrByte := Ord(CompressedRasterData[NextByte]);   { get another byte }
      inc(NextByte);                { increment NextByte counter }
      LongResult := LongResult or (CurrByte shl BitsLeft);  { add the remaining bits to the return value }

      BitsLeft := BitsLeft + 8;               { set bit counter }
      Dec(BytesLeft);                         { decrement bytesleft counter }
    end;
    BitsLeft := BitsLeft - CurrCodeSize;      { subtract the code size from bitsleft }
    Result := LongResult and CodeMask[CurrCodeSize];{ mask off the right number of bits }
  end;
end;  { NextCode }

{------------------------------------------------------------------------------}

procedure TGifSubImage.DecodeRasterData;
{ decodes the LZW encoded raster data }
var
  SP: integer; { index to the decode stack }
  DecodeStack: array[0..MAXCODES] of byte;
               { stack for the decoded codes }
  DecodeRecord: TDecodeRecord;
  Prefix: array[0..MAXCODES] of integer; { array for code prefixes }
  Suffix: array[0..MAXCODES] of integer; { array for code suffixes }
  LineBytes: TBigByteArray;
  CurrBuf: word;  { line buffer index }

  procedure DecodeCode(var Code: word);
  { decodes a code and puts it on the decode stack }
  begin { DecodeCode }
    while Code > DecodeRecord.HighCode do
            { rip thru the prefix list placing suffixes }
    begin                    { onto the decode stack }
      DecodeStack[SP] := Suffix[Code]; { put the suffix on the decode stack }
      inc(SP);                         { increment decode stack index }
      Code := Prefix[Code];            { get the new prefix }
    end;
    DecodeStack[SP] := Code;            { put the last code onto the decode stack }
    Inc(SP);                          { increment the decode stack index }
  end;  { DecodeCode }

  procedure PopStack;
  { pops off the decode stack and puts into the line buffer }
  begin { PopStack }
    with DecodeRecord do
    while SP > 0 do
    begin
      dec(SP);
      LineBytes[CurrBuf] := DecodeStack[SP];
      inc(CurrBuf);
      if CurrBuf > ImageDescriptor.ImageWidth       { is the line full ? }
      then begin
        Application.ProcessMessages;
        Pixels.SetRow(CurrentY+1, LineBytes);
        { addition of one necessary because CurrentY is
          zero-based while ImagePixels is one-based }
        if not InterLaced
        then Inc(CurrentY)
        else CurrentY := NextLineNo(CurrentY, ImageDescriptor.ImageHeight,
                                              InterlacePass);
        CurrBuf := 1;
      end;
    end; { while SP > 0 }
  end;  { PopStack }

  procedure CheckSlotValue(var Slot, TopSlot: Word; var MaxVal: Boolean);
  begin { CheckSlotValue }
    if Slot >= TopSlot then      { have reached the top slot for bit size }
    begin                        { increment code bit size }
      if DecodeRecord.CurrCodeSize < 12 then  { new bit size not too big? }
      begin
        TopSlot := TopSlot shl 1;  { new top slot }
        inc(DecodeRecord.CurrCodeSize)       { new code size }
      end else
        MaxVal := True;       { Must check next code is a start code }
    end;
  end;  { CheckSlotValue }

var
  TempOldCode, OldCode: word;
  Code, C: word;
  MaxVal: boolean;
  Slot     : Word;     { position that the next new code is to be added }
  TopSlot  : Word;     { highest slot position for the current code size }
begin { TGifSubImage.DecodeRasterData }
  InitCompressionStream(LZWCodeSize,
                        CompressedRasterData, DecodeRecord); { Initialize decoding parameters }
  LineBytes := TBigByteArray.Create(ImageDescriptor.ImageWidth);
  OldCode := 0;
  SP := 0;
  CurrBuf := 1;
  MaxVal := False;
  C := NextCode(CompressedRasterData, DecodeRecord);  { get the initial code - should be a clear code }
  while C <> DecodeRecord.EndingCode do  { main loop until ending code is found }
  begin
    if C = DecodeRecord.ClearCode then   { code is a clear code - so clear }
    begin
      DecodeRecord.CurrCodeSize := DecodeRecord.LZWCodeSize + 1;  { reset the code size }
      Slot := DecodeRecord.EndingCode + 1;           { set slot for next new code }
      TopSlot := 1 shl DecodeRecord.CurrCodeSize;    { set max slot number }
      while C = DecodeRecord.ClearCode do
        C := NextCode(CompressedRasterData,  DecodeRecord);
          { read until all clear codes gone - shouldn't happen }
      if C = DecodeRecord.EndingCode then
        raise EGifException.Create('Bad code');     { ending code after a clear code }
      if C >= Slot then { if the code is beyond preset codes then set to zero }
        C := 0;
      OldCode := C;
      DecodeStack[SP] := C;   { output code to decoded stack }
      inc(SP);                { increment decode stack index }
    end else   { the code is not a clear code or an ending code so  }
    begin      { it must be a code code - so decode the code }
      Code := C;
      if Code < Slot then     { is the code in the table? }
      begin
        DecodeCode(Code);            { decode the code }
        if Slot <= TopSlot then
        begin                        { add the new code to the table }
          Suffix[Slot] := Code;      { make the suffix }
          Prefix[Slot] := OldCode;   { the previous code - a link to the data }
          inc(Slot);                 { increment slot number }
          CheckSlotValue(Slot, TopSlot, MaxVal);
          OldCode := C;              { set oldcode }
        end;
      end else
      begin  { the code is not in the table }
        if Code <> Slot then
          raise EGifException.Create('Bad code'); { so error out }
          { the code does not exist so make a new entry in the code table
            and then translate the new code }
        TempOldCode := OldCode;  { make a copy of the old code }
        while OldCode > DecodeRecord.HighCode { translate the old code and }
        do begin                              { place it on the decode stack }
          DecodeStack[SP] := Suffix[OldCode]; { do the suffix }
          OldCode := Prefix[OldCode];         { get next prefix }
        end;
        DecodeStack[SP] := OldCode;  { put the code onto the decode stack }
                                  { but DO NOT increment stack index }
            { the decode stack is not incremented because we are }
            { only translating the oldcode to get the first character }
        if Slot <= TopSlot then
        begin   { make new code entry }
          Suffix[Slot] := OldCode;       { first char of old code }
          Prefix[Slot] := TempOldCode;   { link to the old code prefix }
          inc(Slot);                     { increment slot }
          CheckSlotValue(Slot, TopSlot, MaxVal);
        end;
        DecodeCode(Code); { now that the table entry exists decode it }
        OldCode := C;     { set the new old code }
      end;
    end; { else (if code < slot) }
    PopStack;  { the decoded string is on the decode stack; put in linebuffer }
    C := NextCode(CompressedRasterData, DecodeRecord);  { get the next code and go at is some more }
    if (MaxVal = True) and (C <> DecodeRecord.ClearCode) then
      raise EGifException.Create('Code size overflow');
    MaxVal := False;
  end; { while C <> EndingCode }
end;  { TGifSubImage.DecodeRasterData }

procedure TGifSubImage.LoadFromOpenInfile(var infile: File);
begin { TGifSubImage.LoadFromOpenInfile }
  ReadImageDescriptor(infile);
  ReadLocalColorMap(infile);
  Pixels := TByteArray2d.Create(ImageDescriptor.ImageWidth,
                                ImageDescriptor.ImageHeight);
  ReadRasterData(infile);
  DecodeRasterData;
end;  { TGifSubImage.LoadFromOpenInfile }

(***** write routines *****)

procedure AppendPixel(var PixelString: String;
                      Pixels: TBigByteArray;
                      var NextPixelNo: Integer);
begin { AppendPixel }
  PixelString := PixelString + Chr(Pixels[NextPixelNo]);
  Inc(NextPixelNo);
end;  { AppendPixel }

procedure GoBackPixel(var PixelString: String;
                      var NextPixelNo: Integer);
begin { GoBackPixel }
  System.Delete(PixelString, Length(PixelString), 1);
  Dec(NextPixelNo);
end;  { GoBackPixel }

procedure TGifSubImage.EncodeStatusbyte;
begin { TGifSubImage.EncodeStatusbyte }
  with ImageDescriptor
  do begin
    PackedFields := 0;
    if UsesLocalColorMap
    then PackedFields := PackedFields or idLocalColorTable;
    if Interlaced
    then PackedFields := PackedFields or idInterlaced;
    if UsesLocalColorMap
    then PackedFields := PackedFields or (BitsperPixel-1);
  end;
end;  { TGifSubImage.EncodeStatusbyte }

procedure TGifSubImage.WriteImageDescriptor(var outfile: File);
var OldStatusByte: Byte;
begin { TGifSubImage.WriteImageDescriptor }
  OldStatusByte := ImageDescriptor.PackedFields;
  EncodeStatusByte;
  if ImageDescriptor.PackedFields <> OldStatusByte
  then ShowMessage('PackedFields value has been changed');
  BlockWrite(outfile, ImageDescriptor, SizeOf(ImageDescriptor));
end;  { TGifSubImage.WriteImageDescriptor }

procedure TGifSubImage.WriteLocalColorMap(var outfile: File);
begin { TGifSubImage.WriteLocalColorMap }
  if UsesLocalColorMap
  then
    with LocalColorMap
    do BlockWrite(outfile, Colors[0], Count*SizeOf(TColorItem))
end;  { TGifSubImage.WriteLocalColorMap }

procedure TGifSubImage.EncodeRasterdata;
var
  PixelArray: TBigByteArray;
  CodeTable: TCodeTable;
  ClearCode: Word;
  EndCode: Word;
  FirstPixel: Byte;
  OldCode, Code: Integer;
  PixelString: String;
  NextPixelNo: Integer;
  Found: Boolean;
  TableIndex: Integer;
  EncodedBytes: TEncodedBytes;
begin { TGifSubImage.EncodeRasterdata }
  MakeFlat(Pixels, Interlaced, PixelArray);
  CodeTable := TCodeTable.Create;
  CodeTable.Clear(LZWCodeSize+1);
  ClearCode := 1 shl LZWCodeSize;
  EndCode := ClearCode + 1;
  EncodedBytes := TEncodedBytes.Create;
  EncodedBytes.AppendCode(ClearCode, CodeTable.CodeSize);
  NextPixelNo := 1;
  FirstPixel := PixelArray[NextPixelNo];
  EncodedBytes.AppendCode(FirstPixel, CodeTable.CodeSize);
  OldCode := FirstPixel;
  Inc(NextPixelNo);
  ShowProgress(0);
  repeat
    PixelString := '';
    AppendPixel(PixelString, PixelArray, NextPixelNo);
    CodeTable.AddEntry(OldCode, Ord(PixelString[1]));
    Found := True;
    while Found and (NextPixelNo <= PixelArray.Count)
    do begin
      AppendPixel(PixelString, PixelArray, NextPixelNo);
      Found := CodeTable.IsInTable(PixelString, TableIndex)
    end;
    if not Found
    then GoBackPixel(PixelString, NextPixelNo);
    if not CodeTable.IsInTable(PixelString, Code)
    then raise EGifException.Create('Pixelstring not found in codetable');
    EncodedBytes.AppendCode(Code, CodeTable.CodeSize);
    if CodeTable.TableFull and (NextPixelNo <= PixelArray.Count)
    then begin
      EncodedBytes.AppendCode(ClearCode, CodeTable.CodeSize);
      CodeTable.Clear(LZWCodeSize+1);
      FirstPixel := PixelArray[NextPixelNo];
      EncodedBytes.AppendCode(FirstPixel, CodeTable.CodeSize);
      OldCode := FirstPixel;
      Inc(NextPixelNo);
      ShowProgress(NextPixelNo/PixelArray.Count);
      Application.ProcessMessages;
    end
    else OldCode := Code;
  until (NextPixelNo > PixelArray.Count);
  EncodedBytes.Finish(EndCode, CodeTable.CodeSize);
  CompressedRasterData := EncodedBytes.Value;
  CodeTable.Free;
  EncodedBytes.Free;
  PixelArray.Free;
  ShowProgress(1);
end;  { TGifSubImage.EncodeRasterdata }

procedure TGifSubImage.WriteRasterData(var outfile: File);
var
  Block: String;
  BlokByteCount: Byte;
begin { TGifSubImage.WriteRasterData }
  BlockWrite(outfile, LZWCodeSize, 1);
  while Length(CompressedRasterdata) > 255
  do begin
    Block := System.Copy(CompressedRasterdata, 1, 255);
    BlokByteCount := 255;
    BlockWrite(outfile, BlokByteCount, 1);
    BlockWrite(outfile, Block[1], 255);
    System.Delete(CompressedRasterData, 1, 255)
  end;
  BlokByteCount := Length(CompressedRasterData);
  if BlokByteCount <> 0
  then begin
    BlockWrite(outfile, BlokByteCount, 1);
    BlockWrite(outfile, CompressedRasterData[1], BlokByteCount);
  end;
  BlokByteCount := 0;
  BlockWrite(outfile, BlokByteCount, 1);
end;  { TGifSubImage.WriteRasterData }

(***** end of TGifSubImage *****)

(***** TGifFile *****)

constructor TGifFile.Create;
begin { TGifFile.Create }
  inherited Create;
  Header.Signature := 'GIF';
  Header.Version := '87a';
  ScreenDescriptor.ScreenWidth := 0;
  ScreenDescriptor.ScreenHeight := 0;
  ScreenDescriptor.PackedFields := 0;
  ScreenDescriptor.BackGroundcolorIndex := 0;
  ScreenDescriptor.AspectRatio := 0;
  GlobalColorMapFollows := True;
  BitsPerPixel := 8;  { arbitrary; other choices would be 1 or 4 }
  {GlobalColorMap := TColorTable.Create;}
  TColorTable_Create(GlobalColormap, 256);
  SubImages := TList.Create;
end;  { TGifFile.Create }

procedure TGifFile.AddSubImage(Colormap: TColorTable;
                               PixelMatrix: TByteArray2D);
var NewSubImage: TGifSubImage;
begin { TGifFile.AddSubImage }
  NewSubImage := TGifSubImage.Create(GlobalColormap.Count);
  if SubImages.Count = 0
  then GlobalColormap := Colormap
  else begin
    NewSubImage.UsesLocalColormap := True;
    NewSubImage.LocalColormap := Colormap;
  end;
  NewSubImage.Pixels := PixelMatrix.Copy;
  NewSubImage.ImageDescriptor.ImageWidth := PixelMatrix.Count1;
  NewSubImage.ImageDescriptor.ImageHeight := PixelMatrix.Count2;
  SubImages.Add(NewSubImage);
  if ScreenDescriptor.ScreenWidth < PixelMatrix.Count1
  then ScreenDescriptor.ScreenWidth := PixelMatrix.Count1;
  if ScreenDescriptor.ScreenHeight < PixelMatrix.Count2
  then ScreenDescriptor.ScreenHeight := PixelMatrix.Count2;
end;  { TGifFile.AddSubImage }

function TGifFile.AsTBitmap: Graphics.TBitmap;
var Stream: TMemoryStream;
begin { TGifFile.AsTBitmap }
  Stream := TMemoryStream.Create;
  try
    Self.SaveToStream(Stream);
    Result := Graphics.TBitmap.Create;
    Result.LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;  { TGifFile.AsTBitmap }

function TGifFile.GetSubImage(Index: Integer): TGifSubImage;
begin
  Result := SubImages[Index-1]
end;  { TGifFile.GetSubImage }

(***** Read routines *****)

procedure TGifFile.ReadSignature(var infile: File);
begin { TGifFile.ReadSignature }
  BlockRead(infile, Header, SizeOf(TGifHeader));
  if (Header.Version <> '87a') and (Header.Version <> '89a') and
     (Header.Version <> '87A') and (Header.Version <> '89A') 
  then raise EGifException.Create('Gif Version must be 87a or 89a');
end;  { TGifFile.ReadSignature }

procedure TGifFile.DecodeStatusByte;
var
  ColorResolutionBits: Byte;
begin { TGifFile.DecodeStatusByte }
  GlobalColorMapFollows := (ScreenDescriptor.PackedFields and lsdGlobalColorTable) = lsdGlobalColorTable;  { M=1 }
  {if not GlobalColorMapFollows
  then raise EGifException.Create('No Global ColorMap');}
  ColorResolutionbits := 1 + (ScreenDescriptor.PackedFields and lsdColorResolution) shr 4;
  GlobalColorMap.Count := 1 shl ColorResolutionbits;
  BitsPerPixel := 1 + ScreenDescriptor.PackedFields and $07;
end;  { TGifFile.DecodeStatusByte }

procedure TGifFile.ReadScreenDescriptor(var infile: File);
{var ZeroByte: Byte;}{turns out to be used for aspect ratio }
begin { TGifFile.ReadScreenDescriptor }
  BlockRead(infile, ScreenDescriptor, SizeOf(ScreenDescriptor));
  DecodeStatusByte;
end;  { TGifFile.ReadScreenDescriptor }

procedure TGifFile.ReadGlobalColorMap(var infile: File);
begin { TGifFile.ReadGlobalColorMap }
  if GlobalColorMapFollows
  then
    with GlobalColorMap
    do BlockRead(infile, Colors[0], Count*SizeOf(TColorItem));
end;  { TGifFile.ReadGlobalColorMap }

procedure TGifFile.ReadExtensionBlocks(var infile: File;
                                       var SeparatorChar: Char);
{ The '!' has alrady been read before calling }

  procedure ReadDataBlocks(var infile: File);
  { data not yet stored }
  var
    BlockSize: Byte;
    Buffer: array[0..255] of Byte;
  begin { ReadDataBlocks }
    repeat
      BlockRead(infile, BlockSize, 1);
      if BlockSize <> 0
      then BlockRead(infile, Buffer, BlockSize);
    until BlockSize = 0;
  end;  { ReadDataBlocks }

var
  ExtensionLabel: Byte;
  Terminator: Byte;
  GCE: TGraphicControlExtension;
  PTE: TPlainTextExtension;
  APPE: TApplicationExtension;
     { local; these are not (yet) used; this is just to
       not crash on gif files that have those blocks }
begin { TGifFile.ReadExtensionBlocks }
  while SeparatorChar = '!'
  do begin
    BlockRead(infile, ExtensionLabel, 1);
    case ExtensionLabel of
      $F9: BlockRead(infile, GCE, SizeOf(GCE));
                                   { graphic control extension }
      $FE: ReadDataBlocks(infile); { comment extension }
      $01: begin                   { plain text extension }
             BlockRead(infile, PTE, SizeOf(PTE));
             ReadDataBlocks(infile);
           end;
      $FF: begin                   { application extension }
             BlockRead(infile, APPE, SizeOf(APPE));
             ReadDataBlocks(infile);
           end;
      else raise EGifException.Create('Unrecognized extension block.'+
                 #13+#10 + 'Code = $' + IntToHex(ExtensionLabel, 2));
    end; { case }
    BlockRead(infile, SeparatorChar, 1);
  end;
end;  { TGifFile.ReadExtensionBlocks }

procedure TGifFile.LoadFromFile(filename: String);
var
  infile: File;
  SeparatorChar: Char;
  NewSubImage: TGifSubimage;
begin { TGifFile.LoadFromFile }
  Screen.Cursor := crHourGlass;
  SubImages := TList.Create;
  AssignFile(infile, filename);
  Reset(infile, 1);
  ReadSignature(infile);
  ReadScreenDescriptor(infile);
  ReadGlobalColorMap(infile);
  BlockRead(infile, SeparatorChar, 1);
  while SeparatorChar <> ';'
  do begin
    ReadExtensionBlocks(infile, SeparatorChar);
    if SeparatorChar = ','
    then begin
      NewSubImage := TGifSubImage.Create(GlobalColormap.Count);
      NewSubImage.LoadFromOpenInfile(infile);
      SubImages.Add(NewSubImage);
      BlockRead(infile, SeparatorChar, 1);
    end;
  end;
  CloseFile(infile);
  Screen.Cursor := crDefault;
end;  { TGifFile.LoadFromFile }

(***** write routines *****)

procedure TGifFile.EncodeStatusByte;
var
  ColorResolutionBits: Byte;
begin { TGifFile.EncodeStatusByte }
  with ScreenDescriptor
  do begin
    PackedFields := 0;
    if GlobalColorMapFollows
    then PackedFields := PackedFields + lsdGlobalColorTable;
    case GlobalColorMap.Count of
      2: ColorResolutionBits := 1;
      16: ColorResolutionBits := 4;
      256: ColorResolutionBits := 8;
      else raise EGifException.Create('unexpected number of colors')
    end;
    PackedFields := PackedFields + (ColorResolutionBits-1) shl 4;
    PackedFields := PackedFields + (BitsPerPixel-1);
  end;
end;  { TGifFile.EncodeStatusByte }

procedure TGifFile.WriteSignature(var outfile: File);
begin { TGifFile.WriteSignature }
  BlockWrite(outfile, Header, SizeOf(TGifHeader));
end;  { TGifFile.WriteSignature }

procedure TGifFile.WriteScreenDescriptor(var outfile: File);
begin { TGifFile.WriteScreenDescriptor }
  EncodeStatusByte;
  BlockWrite(outfile, ScreenDescriptor, SizeOf(ScreenDescriptor));
end;  { TGifFile.WriteScreenDescriptor }

procedure TGifFile.WriteGlobalColorMap(var outfile: File);
begin { TGifFile.WriteGlobalColorMap }
  if GlobalColorMapFollows
  then
    with GlobalColorMap
    do BlockWrite(outfile, Colors[0], Count*SizeOf(TColorItem))
end;  { TGifFile.WriteGlobalColorMap }

procedure TGifFile.SaveToFile(filename: String);
var
  outfile: File;
  ImageSeparator: Char;
  ImageNo: Integer;
  SubImage: TGifSubimage;
begin { TGifFile.SaveToFile }
  Screen.Cursor := crHourGlass;
  AssignFile(outfile, filename);
  Rewrite(outfile, 1);
  WriteSignature(outfile);
  WriteScreenDescriptor(outfile);
  WriteGlobalColorMap(outfile);
  ImageSeparator := ',';
  for ImageNo := 1 to SubImages.Count
  do begin
    BlockWrite(outfile, ImageSeparator, 1);
    SubImage := SubImages[ImageNo-1];
    SubImage.EncodeRasterdata;
    SubImage.WriteImageDescriptor(outfile);
    SubImage.WriteLocalColorMap(outfile);
    SubImage.WriteRasterData(outfile);
  end;
  ImageSeparator := ';';
  BlockWrite(outfile, ImageSeparator, 1);
  CloseFile(outfile);
  Screen.Cursor := crDefault;
end;  { TGifFile.SaveToFile }


procedure CreateBitHeader(GifFile: TGifFile;
                          var bmHeader: TBitmapInfoHeader);
{ This routine takes the values from the GIF image
  descriptor and fills in the appropriate values in the
  bit map header struct. }
begin { CreateBitHeader }
  with BmHeader do
  begin
    biSize           := Sizeof(TBitmapInfoHeader);
    biWidth          := GifFile.GetSubImage(1).ImageDescriptor.ImageWidth;
    biHeight         := GifFile.GetSubImage(1).ImageDescriptor.ImageHeight;
    biPlanes         := 1;            {Arcane and rarely used}
    biBitCount       := 8;            {Hmmm Should this be hardcoded ?}
    biCompression    := BI_RGB;       {Sorry Did not implement compression in this version}
    biSizeImage      := 0;            {Valid since we are not compressing the image}
    biXPelsPerMeter  :=143;           {Rarely used very arcane field}
    biYPelsPerMeter  :=143;           {Ditto}
    biClrUsed        := 0;            {all colors are used}
    biClrImportant   := 0;            {all colors are important}
  end;
end;  { CreateBitHeader }

procedure TGifFile.SaveToStream(Stream: TStream);
{ Saves it as a .bmp! }
var
  BitFile: TBitmapFileHeader;
  BmHeader : TBitmapInfoHeader; {File Header for bitmap file}
  i: integer;
  Line: integer;
  ch: char;
  x: integer;
  LineBytes: TBigByteArray;
begin { TGifFile.SaveToStream }
  with BitFile do begin
    with GetSubImage(1).ImageDescriptor do
    bfSize := (3*255) + Sizeof(TBitmapFileHeader) +
              Sizeof(TBitmapInfoHeader) + (ImageHeight*ImageWidth);
    bfReserved1 := 0; {not currently used}
    bfReserved2 := 0; {not currently used}
    bfOffBits := (4*256)+ Sizeof(TBitmapFileHeader)+
                          Sizeof(TBitmapInfoHeader);
  end;
  CreateBitHeader(Self, bmHeader);
  {Write the file header}
  with Stream do begin
    Position:=0;
    ch:='B';
    Write(ch,1);
    ch:='M';
    Write(ch,1);
    Write(BitFile.bfSize,sizeof(BitFile.bfSize));
    Write(BitFile.bfReserved1,sizeof(BitFile.bfReserved1));
    Write(BitFile.bfReserved2,sizeof(BitFile.bfReserved2));
    Write(BitFile.bfOffBits,sizeof(BitFile.bfOffBits));
    {Write the bitmap image header info}
    Write(BmHeader,sizeof(BmHeader));
    {Write the BGR palete inforamtion to this file}
    with GetSubImage(1) do
    if UsesLocalColormap then {Use the local color table}
    begin
      for i:= 0 to 255 do
      begin
        Write(LocalColormap.Colors[i].Blue,1);
        Write(LocalColormap.Colors[i].Green,1);
        Write(LocalColormap.Colors[i].Red,1);
        Write(ch,1); {Bogus palette entry required by windows}
      end;
    end else {Use the global table}
    begin
      for i := 0 to 255 do
      begin
        Write(GlobalColormap.Colors[i].Blue,1);
        Write(GlobalColormap.Colors[i].Green,1);
        Write(GlobalColormap.Colors[i].Red,1);
        Write(ch,1); {Bogus palette entry required by windows}
      end;
    end;

    {Init the Line Counter}
    {Write out File lines in reverse order}
    with GetSubImage(1) do
    for Line := ImageDescriptor.ImageHeight downto 1
    do begin
 {Go through the line list in reverse order looking for the
  current Line. Use reverse order since non interlaced gifs are
  stored top to bottom.  Bmp file need to be written bottom to top}
      LineBytes := Pixels.CopyRow(Line);
      x := ImageDescriptor.ImageWidth;
      Write(LineBytes.Address^, x);
      ch := chr(0);
      while (x and 3) <> 0 do { Pad up to 4-byte boundary with zeroes }
      begin
        Inc(x);
        Write(ch, 1);
      end;
    end;
    Position := 0; { reset memory stream}
  end;
end;  { TGifFile.SaveToStream }

(***** end of methods of TGifFile *****)



end.
