{
 Program    : Find and Replace
 Version    : 1.1 Freeware
 Author     : Tom van Breukelen

 Module     : PgmStart.Pas
 Description: Startup form for the main window
}

unit PgmStart;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, ShellApi, PgmQuery, PgmStatus, PgmFilter,
  ComCtrls, Registry, DragDrop, MinMax, appexec;

type
  TFStart = class(TForm)
    MainMenu1  : TMainMenu;
    File1      : TMenuItem;
    Close1     : TMenuItem;
    Help1      : TMenuItem;
    About1     : TMenuItem;
    Panel1     : TPanel;
    GroupBox1  : TGroupBox;
    GroupBox2  : TGroupBox;
    OpenDialog1: TOpenDialog;
    ListBox1   : TListBox;
    Button1    : TButton;
    Button2    : TButton;
    Button3    : TButton;
    Button4    : TButton;
    Image1     : TImage;
    StatusBar1 : TStatusBar;
    PopupMenu1 : TPopupMenu;
    Associate1 : TMenuItem;
    Notepad1   : TMenuItem;
    N1         : TMenuItem;
    Remove1    : TMenuItem;
    DragDrop1  : TDragDrop;
    Options1   : TMenuItem;
    FileFilter1: TMenuItem;
    MinMax1    : TMinMax;
    AppExec1: TAppExec;
    procedure FormShow(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FileFilter1Click(Sender: TObject);
    procedure DragDrop1FilesDropped(NumItems: Integer);
    procedure About1Click(Sender: TObject);
    procedure BrowseClick(Sender: TObject);
    procedure ScanDir(FileName: string);
    procedure FindReplaceClick(Sender: TObject);
    procedure ListBox1Click(Sender: TObject);
    procedure ListBox1KeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure ListBox1Delete(Sender: TObject);
    procedure Associate1Click(Sender: TObject);
    procedure Notepad1Click(Sender: TObject);
    procedure ClearClick(Sender: TObject);
    procedure ExitClick(Sender: TObject);
  private
    procedure DisplayHint(Sender: TObject);
    function  ScanFile(FileName: string): boolean;
    procedure UpdateFStatus;
    procedure SetLock(All: boolean);
  public
  end;

  ERegistryError = class(Exception);
  TFindRepl = class(TComponent)
  private
    FRegistry : TRegistry;
    FWindow   : array[1..2] of integer;
    FFileExt  : string;
    FSubDir   : boolean;
  protected
    function  GetWindow(Index: integer): integer;
    procedure SetWindow(Index: integer; Value: integer);
    procedure SetFileExt(S: string);
    procedure SetSubDir(B: boolean);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Height : integer index 1 read GetWindow write SetWindow;
    property Width  : integer index 2 read GetWindow write SetWindow;
    property FileExt: string  read FFileExt write SetFileExt;
    property SubDir : boolean read FSubDir  write SetSubDir;
  end;

const
  C_RegPath = 'Software\TvBSoftware\FindRepl';
  C_Window  : array[1..2] of string = ('Window Height','Window Width');
  C_FileExt = 'File Extensions';
  C_SubDir  = 'Include Subdirectories';

var
  FStart   : TFStart;
  FFindRepl: TFindRepl;
  OldHeight: integer;

implementation

{$R *.DFM}

var
  IgnoreCase : boolean;
  WholeWords : boolean;
  Prompt     : boolean;
  Backup     : boolean;
  FindOnly   : boolean;
  FindText   : string;
  ReplaceText: string;
  TotalBytes, BytesRead, BytesWritten, TotalFiles, TotalOccur: integer;

{
 Main form activation and destruction
}

procedure TFStart.FormShow(Sender: TObject);
begin
  OldHeight    := 0;
  Application.OnHint := DisplayHint;
  FFindRepl    := TFindRepl.Create(Self);
  FStart.Height:= FFindRepl.Height;
  FStart.Width := FFindRepl.Width;
  SetLock(False);
end;

procedure TFStart.FormResize(Sender: TObject);
begin
  if OldHeight <> FStart.Height then
  begin
    OldHeight  := FStart.Height;
    Image1.Top := OldHeight - 120;
  end;
end;

procedure TFStart.FormDestroy(Sender: TObject);
begin
  FFindRepl.Height:= FStart.Height;
  FFindRepl.Width := FStart.Width;
  FFindRepl.Free;
end;

procedure TFStart.DisplayHint(Sender: TObject);
begin
  StatusBar1.Panels[2].Text := Application.Hint;
end;

{
 Menu item click events
}

procedure TFStart.ExitClick(Sender: TObject);
begin
  close;
end;

procedure TFStart.FileFilter1Click(Sender: TObject);
var
  result : integer;
begin
  FFilter := TFFilter.Create(Application);
  FFilter.Edit1.Text := FFindRepl.FileExt;
  FFilter.CheckBox1.Checked := FFindRepl.SubDir;
  result := FFilter.ShowModal;
  if result = mrOk then
  begin
    FFindRepl.FileExt := Trim(LowerCase(FFilter.Edit1.Text));
    FFindRepl.SubDir  := FFilter.CheckBox1.Checked;
  end;
  FFilter.Free;
end;

procedure TFStart.About1Click(Sender: TObject);
begin
  MessageDlg('Find and Replace/version 1.1' + #10 +
  'Freeware edition' + #10#10 +
  'Copyright(c) 1998 Tom van Breukelen' + #10 +
  'For the latest version of the program contact:' + #10 +
  'van_breukelen@swissonline.ch', mtInformation, [mbok],0);
end;

{
 Buttons click events
}

procedure TFStart.BrowseClick(Sender: TObject);
begin
  ListBox1.Clear;
  SetLock(False);
  if OpenDialog1.Execute then ListBox1.Items  := OpenDialog1.Files;
  SetLock(False);
end;

procedure TFStart.FindReplaceClick(Sender: TObject);
var
  T: TDateTime;
  I: integer;
  SearchRec: TSearchRec;
begin
  FQuery := TFQuery.Create(Application);
  FQuery.FindOnly := False;
  if TButton(Sender).Tag = 0 then FQuery.FindOnly := True;
  SetLock(True);
  I           := FQuery.ShowModal;
  IgnoreCase  := FQuery.IgnoreCase;
  WholeWords  := FQuery.WholeWords;
  Prompt      := FQuery.Prompt;
  Backup      := FQuery.Backup;
  FindOnly    := FQuery.FindOnly;
  FindText    := FQuery.FindText;
  ReplaceText := FQuery.ReplaceText;
  FQuery.Free;

  if I = mrOk then
  begin
    TotalBytes := 0;
    for I := 0 to ListBox1.Items.Count -1 do
    begin
      FindFirst(ListBox1.Items[I], faAnyFile, SearchRec);
      TotalBytes := TotalBytes + SearchRec.Size;
      FindClose(SearchRec);
    end;

    FStatus := TFStatus.Create(Application);
    if FindOnly then FStatus.Caption := 'Finding text in progress...'
    else FStatus.Caption := 'Replacing text in progress....';
    FStatus.Show;
    BytesRead    := 0;
    BytesWritten := 0;
    TotalFiles   := 0;
    TotalOccur   := 0;
    I            := 0;
    UpdateFStatus;
    T := Now + EncodeTime(0,0,2,500);         {wait max. 2,5 seconds}

    repeat
      if FStatus.ModalResult = mrAbort then
      begin
        MessageDlg('File scanning operation' + #10 + 'has been stopped',
        mtInformation,[mbOk],0);
        Break;
      end;
      if ScanFile(ListBox1.Items[I]) then inc(I)
      else ListBox1.Items.Delete(I);
      StatusBar1.Panels[1].Text := InttoStr(ListBox1.Items.Count) + '  ';
      Application.ProcessMessages;
    until I = ListBox1.Items.Count;

    while T > Now do Application.ProcessMessages;
    FStatus.Free;
  end;
  SetLock(True);
end;

procedure TFStart.ClearClick(Sender: TObject);
begin
  ListBox1.Clear;
  SetLock(False);
end;

{
 ListBox click event (activate Popup menu items in case a record is selected)
}

procedure TFStart.ListBox1Click(Sender: TObject);
begin
  if ListBox1.ItemIndex > -1 then
  begin
    Associate1.Enabled := True;
    Notepad1.Enabled   := True;
    Remove1.Enabled    := True;
  end;
end;

{
 ListBox keydown events (delete key only)
}

procedure TFStart.ListBox1KeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (ListBox1.ItemIndex > -1) and (Key = VK_DELETE) then
  ListBox1Delete(Self);
end;

procedure TFStart.ListBox1Delete;
var
  I: integer;
begin
  I := 0;
  SetLock(True);
  repeat
    if ListBox1.Selected[I] then ListBox1.Items.Delete(I)
    else inc(I);
  until I = ListBox1.Items.Count;
  SetLock(True);
end;

{
 ListBox1's popup menu item click events
}

procedure TFStart.Associate1Click(Sender: TObject);
begin
  try
    AppExec1.ExeName := ListBox1.Items[ListBox1.ItemIndex];
    AppExec1.ExeParm := '';
    AppExec1.Execute;
  except
    MessageDlg(AppExec1.GetErrorString, mtWarning, [mbCancel], 0);
  end;
end;

procedure TFStart.Notepad1Click(Sender: TObject);
begin
  try
    AppExec1.ExeName := 'Notepad.exe';
    AppExec1.ExeParm := ListBox1.Items[ListBox1.ItemIndex];
    AppExec1.Execute;
  except
    MessageDlg(AppExec1.GetErrorString, mtWarning, [mbCancel], 0);
  end;
end;

{
 Windows Explorer File Drag and drop events
}
procedure TFStart.DragDrop1FilesDropped(NumItems: Integer);
var
  I : integer;
  S : string;
begin
  ListBox1.Clear;
  S := FStart.Caption;
  SetLock(True);
  for I := 0 to NumItems -1 do ScanDir(DragDrop1.DroppedFiles[I]);
  SetLock(True);
  FStart.Caption := S;
end;

procedure TFStart.ScanDir(FileName: string);
var
  SearchRec : TSearchRec;
  LFile,LDir: TStringList;
  Result    : integer;
  S         : string;
  I         : integer;
begin
  LFile  := TStringList.Create;
  LDir   := TStringList.Create;
  Result := Pos(#0, FileName);
  if Result > 0 then FileName := Copy(FileName,1,Result -1);
  Result := FindFirst(FileName, faAnyFile, SearchRec);
  FindClose(SearchRec);
  S      := FileName;

  I := SearchRec.Attr and faDirectory;
  if I > 0 then
    begin
      Result := FindFirst(FileName + '\*.*', faAnyFile, SearchRec);
      while Result = 0 do
      begin
         S := FileName + '\' + SearchRec.Name;
         I := SearchRec.Attr and faDirectory;
         if I > 0 then
           begin
             if FFindRepl.SubDir and (SearchRec.Name[1] <> '.') then
             begin
               LDir.Add(S);
             end;
           end
         else
           if (SearchRec.Attr and faReadOnly) + (SearchRec.Attr and faHidden) +
           (SearchRec.Attr and faSysFile) = 0 then
           begin
             if (pos(FFindRepl.FileExt,'*.*') > 0) or
             (pos(LowerCase(ExtractFileExt(S)) + ';',FFindRepl.FileExt + ';') > 0) then
             LFile.Add(S);
           end;
         Result := FindNext(SearchRec);
         Application.ProcessMessages;
      end;
      FindClose(SearchRec);
    end
  else
    if (SearchRec.Attr and faReadOnly) + (SearchRec.Attr and faHidden) +
    (SearchRec.Attr and faSysFile) = 0 then
    begin
      if (pos(FFindRepl.FileExt,'*.*') > 0) or
      (pos(LowerCase(ExtractFileExt(S)) + ';',FFindRepl.FileExt + ';') > 0) then
       LFile.Add(S);
    end;

  FStart.Caption := S;
  LFile.Sorted := True;
  LDir.Sorted := True;
  for I := 0 to LFile.Count - 1 do ListBox1.Items.Add(LFile.Strings[I]);
  LFile.Free;
  for I := 0 to LDir.Count - 1 do ScanDir(LDir.Strings[I]);
  LDir.Free;
end;

{
 Common procedures
}

procedure TFStart.SetLock(All: boolean);
begin
  Button2.Enabled      := ListBox1.Items.Count > 0 ;
  Button3.Enabled      := Button2.Enabled;
  Button4.Enabled      := Button2.Enabled;
  Associate1.Enabled   := False;
  Notepad1.Enabled     := False;
  Remove1.Enabled      := False;
  if All then
  begin
    File1.Enabled      := not File1.Enabled;
    Options1.Enabled   := File1.Enabled;
    Help1.Enabled      := File1.Enabled;
    GroupBox1.Enabled  := File1.Enabled;
    GroupBox2.Enabled  := File1.Enabled;
    if File1.Enabled then Screen.Cursor := crDefault
    else Screen.Cursor := crHourglass;
  end;
  StatusBar1.Panels[1].Text := InttoStr(ListBox1.Items.Count) + '  ';
end;

procedure TFStart.UpdateFStatus;
begin
  FStatus.Edit1.Text := IntToStr(TotalFiles);
  FStatus.Edit2.Text := IntToStr(TotalOccur);
  FStatus.Edit3.Text := IntToStr(BytesRead);
  FStatus.Edit4.Text := IntToStr(BytesWritten);
  if TotalBytes > 0 then
  FStatus.ProgressBar1.Position := (BytesRead * 100) div TotalBytes;
  Application.ProcessMessages;
end;

{
 Here is what we were all waiting for...
}

function TFStart.ScanFile(FileName: string): boolean;
var
  FromFle, BakFle: file;                      {required for block read/writes}
  TmpFle: TextFile;                           {required for writeln}
  Rec, I, J, NumRead: Integer;
  Buf: array[1..8192] of Char;                {buffer for block reads}
  BakStr,TmpStr,FText,S1,S2,S3: string;
  BeginChar, EndChar: Char;                   {'Whole Words' test requirement}
  Update,NoUpdate,PromptIt: Boolean;          {Update 'flags'}
  Click: Word;
begin
  TmpStr := ExtractFilePath(FileName) + 'Tmpfile.txt';
  AssignFile(FromFle,FileName);

  {$I-}
  Reset(FromFle,1);
  {$I+}
  if IOResult <> 0 then
  begin
    MessageDlg('Error accessing file ' + FileName, mtWarning, [mbOk], 0);
    result := false;
    exit;
  end;

  AssignFile(TmpFle,TmpStr);
  {$I-}
  Rewrite(TmpFle);
  {$I+}
  if IOResult <> 0 then
  begin
    MessageDlg('Error creating file ' + TmpStr, mtWarning, [mbOk], 0);
    CloseFile(FromFle);
    result := false;
    exit;
  end;

  NoUpdate := True;
  PromptIt := Prompt;
  inc(TotalFiles);
  BakStr   := FileName + '.bak';
  AssignFile(BakFle,BakStr);
  FText    := FindText;

  if IgnoreCase then FText := UpperCase(FText);
  repeat
    BlockRead(FromFle,Buf,SizeOf(Buf),NumRead);
    if NumRead > 0 then
    begin
      Rec := FilePos(FromFle);
      if Eof(FromFle) then
      begin
        if Buf[NumRead] <> #10 then           {File doesn't end with a line feed}
        begin
          Buf[NumRead + 1] := #13;
          Buf[NumRead + 2] := #10;
          inc(NumRead,2);
        end;
      end;

      for I := NumRead downto 1 do            {Get the position of the last}
      begin                                   {carriage return in the buffer}
        if Buf[I] = #13 then break            {sothat we won't end up in the}
        else dec(Rec);                        {middle of a physical record}
      end;
      S1 := Copy(Buf,1,I - 1);
      S2 := S1;                               {required for the find & replace}
      if IgnoreCase then S2 := UpperCase(S2);
      BytesRead := BytesRead + Length(S1) + 2;

      if pos(FText,S2) > 0 then
      begin
        BeginChar := ' ';
        EndChar   := ' ';
        repeat
          I := pos(FText,S2);
          if I > 0 then
          begin
            Update := True;
            if WholeWords then
            begin
              if I > 1 then BeginChar := S2[I - 1];
              if I + Length(FText) >= Length(S2) then
              EndChar := S2[I + Length(FText)];
              case Ord(BeginChar) of          {see your ASCII character codes table}
                48..57, 65..90, 95, 97..122, 127..255 : Update := False
              else
                begin
                  case Ord(EndChar) of
                    48..57, 65..90, 95, 97..122, 127..255 : Update := False
                  end;
                end;
              end;
            end;

            if Update then
            begin
              if PromptIt and (FindOnly = False) then
              begin
                for J := I downto 1 do if S1[J] = #10 then break;
                if J > 1 then inc(J,2);
                S3 := Copy(S1,J,200);
                J := pos(#10,S3);
                if J > 0 then S3 := Copy(S3,1,J - 1);
                S3 := 'Replace "' + FindText + '" with "' + ReplaceText +
                '" in file' + #10 + FileName + #10 + 'For the following line:' +
                #10#10 + TrimLeft(S3);

                Click := MessageDlg(S3, mtconfirmation,[mbYes, mbNo, mbCancel, mbAll], 0);
                case Click of
                  mrNo     : Update   := False;
                  mrCancel :
                    begin
                      Update := False;
                      FText  := #254#255
                    end;
                  mrAll    : PromptIt := False;
                end;
              end;

              if Update then
              begin
                NoUpdate := False;
                inc(TotalOccur);
                delete(S1,I,length(FText));
                delete(S2,I,length(FText));
                if ReplaceText <> '' then
                begin
                  insert(ReplaceText,S1,I);
                  insert(ReplaceText,S2,I);
                end;
              end;
            end;
            S2[I] := Chr(Ord(S2[I]) + 1);     {to avoid an endless repeating loop}
          end;
        until I = 0;
      end;

      if not FindOnly then
      begin
        if S1 <> '' then WriteLn(TmpFle, S1);
        BytesWritten := BytesWritten + length(S1) + 2;
      end;

      UpdateFStatus;
      inc(Rec);
      Seek(FromFle,Rec);
    end;
  until NumRead = 0;
  CloseFile(FromFle);
  CloseFile(TmpFle);

  result := not NoUpdate;
  if NoUpdate or FindOnly then
    Erase(TmpFle)
  else begin
    if Backup then
      begin
        if FileExists(BakStr) then Erase(BakFle);
        Rename(FromFle,BakStr)
      end
    else Erase(FromFle);
    Rename(TmpFle,FileName);
  end;
end;

{
 And finally some "real" programming
}

constructor TFindRepl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FRegistry := TRegistry.Create;
  if not FRegistry.OpenKey(C_RegPath, true) then
  raise ERegistryError.Create('Registry access violation, cannot open key ' + C_Regpath);

  with FRegistry do
  begin
    if ValueExists(C_Window[1]) then FWindow[1] := ReadInteger(C_Window[1])
    else SetWindow(1,143);
    if ValueExists(C_Window[2]) then FWindow[2] := ReadInteger(C_Window[2])
    else SetWindow(2,515);
    if ValueExists(C_FileExt) then FFileExt := ReadString(C_FileExt)
    else SetFileExt('.bas;.bat;.c;.fac;.htm;.html;.ini;.pas;.txt');
    if ValueExists(C_SubDir) then FSubDir := ReadBool(C_SubDir)
    else SetSubDir(True);
  end;
end;

destructor TFindRepl.Destroy;
begin
  FRegistry.CloseKey;
  FRegistry.Free;
  inherited Destroy;
end;

function TFindRepl.GetWindow(Index: integer): integer;
begin
  Result := FWindow[Index];
end;

procedure TFindRepl.SetWindow(Index: integer; Value: integer);
begin
  FRegistry.WriteInteger(C_Window[Index], Value);
  FWindow[Index] := Value;
end;

procedure TFindRepl.SetFileExt(S: string);
begin
  FRegistry.WriteString(C_FileExt, S);
  FFileExt := S;
end;

procedure TFindRepl.SetSubDir(B: boolean);
begin
  FRegistry.WriteBool(C_SubDir, B);
  FSubDir := B;
end;

end.
