{
TAppExec is a non-visual freeware component which you place onto your form to
enable easily initiated execution of other Windows or DOS applications from
within your own original Delphi application.

The basis is a combination of the original TAppExec and TExecFile freeware
components with some additions of my own. Main changes, I've made (apart from
merging the components together), is the ability to execute associated files.
Instead of using the FindExecutable API, which I found to be rather unreliable,
the new AppExec retrieves the file associations directly from the Registry.
}

unit appexec;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes,
  Forms;

type
  EAppExec = class(Exception);
  EAppExecChDir = class(EAppExec);
  EAppExecWinExec = class(EAppExec);

  TWaitStyle = (wRegular, wSuspend);
  TAppExec = class(TComponent)
  private
    FErrNo      : integer;
    FMode       : integer;
    FMsg        : TMsg;
    FExeName    : string;
    FExeCmd     : string;
    FExePath    : string;
    FExeParm    : string;
    FWindowState: TWindowState;
    FChangeDir  : boolean;
    FWait       : boolean;
    FWaitStyle  : TWaitStyle;
    FIsWaiting  : boolean;
  protected
    procedure SetExeName(AExeName: string);
    procedure SetExeParm(AExeParm: string);
    procedure SetExePath(AExePath: string);
    procedure SetWindowState(AWindowState: TWindowState);
  public
    constructor Create(AOwner: TComponent); override;
    function  GetErrorString: string;
    procedure Execute;
    procedure Clear;
  published
    property ChangeDir: boolean read FChangeDir write FChangeDir;
    property ErrNo: integer read FErrNo default -1;
    property ExeName: string read FExeName write SetExeName;
    property ExeParm: string read FExeParm write SetExeParm;
    property ExeCmd:  string read FExeCmd;
    property ExePath: string read FExePath write SetExePath;
    property Wait:boolean read FWait write FWait;
    property WaitStyle: TWaitStyle read FWaitStyle write FWaitStyle default wRegular;
    property WindowState: TWindowState read FWindowState write SetWindowState;
    property IsWaiting: boolean read FIsWaiting;
  end;

procedure Register;

implementation


procedure Register;
begin
  RegisterComponents('FreeWare', [TAppExec]);
end;

constructor TAppExec.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FMode := SW_SHOWNORMAL;
  FErrNo := -1;
end;

procedure TAppExec.Execute;
var
  InstanceID : THandle;
  ApplHandle : THandle;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
begin
  if FChangeDir and (FExePath <> '') then
  begin
    try
      ChDir(FExePath);
    except
      on E:Exception do raise EAppExecChDir.Create(E.Message);
    end;
  end;
  FillChar(StartupInfo,Sizeof(StartupInfo),#0);
  StartupInfo.cb := Sizeof(StartupInfo);
  StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
  StartupInfo.wShowWindow := FMode;
  FErrNo := 0;
  if not CreateProcess(nil,
    PChar(FExeCmd),                { pointer to command line string }
    nil,                           { pointer to process security attributes }
    nil,                           { pointer to thread security attributes }
    false,                         { handle inheritance flag }
    CREATE_NEW_CONSOLE or          { creation flags }
    NORMAL_PRIORITY_CLASS,
    nil,                           { pointer to new environment block }
    nil,                           { pointer to current directory name }
    StartupInfo,                   { pointer to STARTUPINFO }
    ProcessInfo) then              { pointer to PROCESS_INF }
    begin
      FErrNo := GetLastError();
      raise EAppExecWinExec.Create(GetErrorString);
    end
  else begin
    if FWait then
    begin
      if FWaitStyle = wRegular then begin
        FIsWaiting := True;
        repeat
          while PeekMessage(FMsg,0,0,0,PM_REMOVE) do
          begin
            if FMsg.Message = WM_QUIT then halt(FMsg.wParam);
            TranslateMessage(FMsg);
            DispatchMessage(FMsg);
          end;
          if WaitforSingleObject(ProcessInfo.hProcess,0) <> WAIT_TIMEOUT then
          begin
            FIsWaiting := False;
            Application.ProcessMessages;
          end;
        until not FIsWaiting;
        end
      else begin
        WaitForSingleObject(ProcessInfo.hProcess,INFINITE);
      end;
      FIsWaiting := False;
    end;
    FErrNo := 0;
  end;
end;

procedure TAppExec.SetExeName(AExeName: string);
const
  BufferSize = 540;
var
  Buffer : PChar;
  I: Longint;
  aValue: string;
begin
  FExeCmd := AExeName;
  FErrNo  := -1;
  Buffer  := StrAlloc(BufferSize);
  aValue  := '';
  if pos('.',AExeName) = 0 then StrPCopy(Buffer,AExeName)
  else StrPCopy(Buffer,ExtractFileExt(AExeName));
  I := BufferSize;
  if RegQueryValue(hKey_Classes_Root,Buffer,Buffer,I) = ERROR_SUCCESS then
  begin
    StrPCopy(Buffer,StrPas(Buffer) + '\Shell\Open\Command');
    I := BufferSize;
    if RegQueryValue(hKey_Classes_Root,Buffer,Buffer,I)<> ERROR_SUCCESS then
    begin
      StrPCopy(Buffer,AExeName + '\Shell\Open\Command');
      I := BufferSize;
      if RegQueryValue(hKey_Classes_Root,Buffer,Buffer,I)<> ERROR_SUCCESS then
      raise EAppExecWinExec.Create(GetErrorString);
    end;
    aValue := StrPas(Buffer);
    I := Pos('%',aValue);
    if I > 0 then begin
      Delete(aValue,I,2);
      Insert(FExeCmd,aValue,I);
      FExeCmd := aValue;
      I := Pos('%',FExeCmd);
      if I > 0 then Delete(FExeCmd,I,2);
      if FExeParm > '' then FExeCmd := FExeCmd + FExeParm;
      end
    else if Length(aValue) > 0 then
      FExeCmd := aValue + ' ' + FExeCmd;
  end;
  StrDispose(Buffer);
  if aValue = '' then raise EAppExecWinExec.Create(GetErrorString);
  FErrNo  := 2;
  FExeName := AExeName;
end;

procedure TAppExec.SetExeParm(AExeParm: string);
begin
  if FExeParm <> AExeParm then
  begin
    FExeParm := AExeParm;
    if FExeName <> '' then SetExeName(FExeName);
  end;
end;

procedure TAppExec.SetExePath(AExePath: string);
begin
  if FExePath <> AExePath then
  begin
    FExePath := AExePath;
    if ((FExePath[Length(FExePath)] = '\') and
      (FExePath <> '\') and
      (not ((Length(FExePath) = 3) and (FExePath[2] = ':') and (FExePath[3] = '\')))
      ) then
      FExePath := Copy(FExePath, 1, Length(FExePath) - 1);
  end;
end;

procedure TAppExec.SetWindowState(AWindowState: TWindowState);
const
  Mode: array[wsNormal..wsMaximized] of Word =
    (SW_SHOWNORMAL, SW_SHOWMINIMIZED, SW_SHOWMAXIMIZED);
begin
  if FWindowState <> AWindowState then
  begin
    FMode := Mode[AWindowState];
    FWindowState := AWindowState;
  end;
end;

procedure TAppExec.Clear;
begin
  FErrNo   := -1;
  FExeName := '';
  FExeCmd  := '';
  FExePath := '';
  FExeParm := '';
end;

function TAppExec.GetErrorString: string;
begin
  case FErrNo of
    -1  : result := 'The specified file is not associated with a program';
     0  : result := 'System was out of memory, executable file was corrupt, or relocations were invalid';
     2  : result := 'File was not found';
     3  : result := 'Path was not found';
     5  : result := 'Attempt was made to dynamically link to a task, or there was a sharing or network-protection error';
     6  : result := 'Library required separate data segments for each task';
     8  : result := 'There was insufficient memory to start the application';
     10 : result := 'Windows version was incorrect';
     11 : result := 'Executable file was invalid. Either it was not a Windows application or there was an error in the .EXE image';
     12 : result := 'Application was designed for a different operating system';
     13 : result := 'Application was designed for MS-DOS 4.0';
     14 : result := 'Type of executable file was unknown';
     15 : result := 'Attempt was made to load a real-mode application (developed for an earlier version of Windows)';
     16 : result := 'Attempt to load second instance of an executable containing multiple data segments not marked read-only';
     19 : result := 'Attempt was made to load a compressed executable file. The file must be decompressed before it can be loaded';
     20 : result := 'Dynamic-link library (DLL) file was invalid. One of the DLLs required to run this application was corrupt';
     21 : result := 'Application requires 32-bit extensions';
     26 : result := 'A sharing violation occurred';
     27 : result := 'The filename association is incomplete or invalid';
     28 : result := 'The DDE transaction could not be completed because the request timed out';
     29 : result := 'The DDE transaction failed';
     30 : result := 'The DDE transaction could not be completed because other DDE transactions were being processed';
     32 : result := 'The specified dynamic-link library was not found';
  else
    result := 'The specified file could not be executed';
  end;
end;

end.
