unit FileList;

{
Description:

This unit produces a list of files, which can be sorted in various ways by
clicking on the top of the column, like Explorer.  It extends the TListView
component adding an AddFile method to add a file to the end of the list.  The
details for the file are extracted from the directory using FindFirst and the
ShellFileOperations.  Files that don't exist will not be added to the list.

Notes:

This unit requires the ShortDateFormat and LongTimeFormat to be
defined so for the time being, these have been defined in the unit.

Revision history:

1997 Apr 27  V1.0.0  First version
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls;

type
  TFileListView = class(TListView)
  private
    { Private declarations }
    FSortColumn: integer;
    FSortForward: boolean;
    procedure FCompare(Sender: TObject; Item1,
      Item2: TListItem; Data: Integer; var Compare: Integer);
    procedure FColumnClick(Sender: TObject;
      Column: TListColumn);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent);  override;
    procedure CreateWnd;  override;
    function AddFile (filename: string;  var filesize: integer): boolean;
  published
    { Published declarations }
    property AllocBy default 32;
    property HideSelection default False;
    property MultiSelect default True;
    property ReadOnly default True;
    property ShowColumnHeaders default True;
    property ViewStyle default vsReport;
  end;

procedure Register;


implementation

uses ShellAPI;

procedure Register;
begin
  RegisterComponents('Davids', [TFileListView]);
end;


constructor TFileListView.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);
  // changed default properties
  AllocBy := 32;
  HideSelection := False;
  MultiSelect := True;
  ReadOnly := True;
  ShowColumnHeaders := True;
  ViewStyle := vsReport;
  // point to the sort method function
  OnCompare := FCompare;
  // point to the column click function
  OnColumnClick := FColumnClick;
  FSortForward := True;
  FSortColumn := 1;    // default is sort on size
  ShortDateFormat := 'yyyy/mm/dd';
  LongTimeFormat := 'hh:nn';
end;


procedure TFileListView.CreateWnd;
begin
  Inherited;
  if Columns.Count = 0 then
    begin
    with Columns.Add do
      begin
      Caption := 'Name';
      Width := 240;
      end;
    with Columns.Add do
      begin
      Caption := 'Size';
      Width := 70;
      Alignment := taRightJustify;
      end;
    with Columns.Add do
      begin
      Caption := 'Type';
      Width := 70;
      end;
    with Columns.Add do
      begin
      Caption := 'Modified';
      Width := 110;
      end;
    with Columns.Add do
      begin
      Caption := 'Attributes';
      Alignment := taRightJustify;
      end;
//      Width := ColumnTextWidth;
    if csDesigning in ComponentState then {};
    end;
end;


procedure TFileListView.FColumnClick(Sender: TObject;
  Column: TListColumn);
var
  required_column: integer;
begin
  required_column := Column.Index;
  if required_column = FSortColumn
  then FSortForward := not FSortForward
  else
    begin
    FSortColumn := required_column;
    FSortForward := True;
    end;
  SortType := stData;
  SortType := stNone;
end;


procedure TFileListView.FCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
// this procedure returns a variable Compare based on the file size
// given in the list box
var
  size1, size2: integer;
  date1, date2: TDateTime;
  result: integer;
begin
  case FSortColumn of
    0: begin     // sort on path name
       result := CompareText (Item1.Caption, Item2.Caption);
       end;
    1: begin     // sort on file size
       size1 := StrToInt (Item1.SubItems.Strings [0]);
       size2 := StrToInt (Item2.SubItems.Strings [0]);
       result := (size1 - size2);
       end;
    3: begin     // sort on file modified date
       date1 := StrToDateTime (Item1.SubItems.Strings [2]);
       date2 := StrToDateTime (Item2.SubItems.Strings [2]);
       result := Round (1E5*(date1 - date2));
       end;
    else
      begin
      result := CompareText (Item1.SubItems.Strings [FSortColumn - 1],
                             Item2.SubItems.Strings [FSortColumn - 1]);
      end;
  end;
  if FSortForward
  then Compare := result
  else Compare := -result;
end;


function TFileListView.AddFile (filename: string;  var filesize: integer): boolean;
// Format a single file entry for the matches ListView, recovering
// the size, attributes and modification time from the FindFirst,
// and the type name from a SHGetFileInfo call.
// Returns the file size of the named file
var
  s: TSearchRec;            // for the attributes, size and date-time
  ShInfo: TSHFileInfo;      // for the type name and icon info
  attributes: string;       // where we build "RHSA"
begin
  Result := False;
  if (SHGetFileInfo (PChar (filename),
                    0,
                    ShInfo, SizeOf (ShInfo),
                    SHGFI_TYPENAME) <> 0)
    and
      (FindFirst (filename, faAnyFile, s) = 0) then
    with s do
      begin
      attributes := '';       // build the attributes string
      if (faReadonly and Attr) <> 0 then attributes := 'R';
      if (faHidden and Attr) <> 0 then attributes := attributes + 'H';
      if (faSysfile and Attr) <> 0 then attributes := attributes + 'S';
      if (faArchive and Attr) <> 0 then attributes := attributes + 'A';
      with Items.Add do       // returns a TListItem for the "with" clause
        begin
        Caption := LowerCase (filename);      // set the main item caption
        SubItems.Add (Trim (IntToStr (s.Size)));     // add the size
        SubItems.Add (Trim (ShInfo.szTypeName));     // add the TypeName
        SubItems.Add (DateTimeToStr (FileDateToDateTime (Time)));
        SubItems.Add (attributes);                   // add the attributes
        end;
      filesize := s.Size;
      Result := True;
      end;
  FindClose (s);
end;


end.
