(*
   
  TFileIterator Component
  Version 1.00.00
  (C) 1996, Glen Why. No rights reserved

*)
unit FileIterator;

interface
uses
 Windows, SysUtils, Classes;

type

  EFileIteratorError = class( Exception );

  TFileIteratorOption = ( fiPath, fiDetails, fiRecurseFolders );
  TFileIteratorOptions = Set of TFileIteratorOption;

  TFileIteratorStatus = ( fsIdle, fsActive, fsTerminated );

  PFileInfo = ^TFileInfo;
  TFileInfo = record
    Size :Integer;
    Time :Integer;
    Attr :Integer;
  end;

  TAddFileEvent = procedure( Sender :TObject; const FileName :string;
   const FileInfo :TFileInfo ) of object;

  TAddFileQueryEvent = procedure( Sender :TObject; const FileName :string;
   const FileInfo :TFileInfo; var CanAdd :Boolean ) of object;

  TFileIterator = class( TComponent )
  private
    FOptions :TFileIteratorOptions;
    FStatus :TFileIteratorStatus;
    FRootFolder :String;
    FOnAddFile :TAddFileEvent;
    FOnAddFileQuery :TAddFileQueryEvent;
    FOnActivate :TNotifyEvent;
    FOnTerminate :TNotifyEvent;
    procedure SetOptions( Value :TFileIteratorOptions );
    procedure SetRootFolder( const Value :String );
    function GetIsIdle :Boolean;
    procedure ScanFolder( Folder :String );
  protected
    procedure AddFile( const FileName :string;
     const FileInfo :TFileInfo ); virtual;
    procedure AddFileQuery( const FileName :string;
     const FileInfo :TFileInfo; Var CanAdd :Boolean ); virtual;
    procedure Activate; virtual;
    procedure Terminate; virtual;
  public
    constructor Create( anOwner :TComponent ); override;
    destructor Destroy; override;
    procedure Iterate;
    procedure Cancel;
    property Status :TFileIteratorStatus
      read FStatus;
    property IsIdle :Boolean
      read GetIsIdle;
  published
    property Options :TFileIteratorOptions
      read FOptions write SetOptions;
    property RootFolder :string
      read FRootFolder write SetRootFolder;
    property OnAddFile :TAddFileEvent
      read FOnAddFile write FOnAddFile;
    property OnAddFileQuery :TAddFileQueryEvent
      read FOnAddFileQuery write FOnAddFIleQuery;
    property OnActivate :TNotifyEvent
      read FOnActivate write FOnActivate;
    property OnTerminate :TNotifyEvent
      read FOnTerminate write FOnTerminate;
  end;



  TFileIteratorThread = Class( TThread )
  private
    FIterator :TFileIterator;
  protected
    procedure Execute; override;
  public
    constructor Create( Iterator :TFileIterator;
     CreateSuspended :Boolean );
    procedure Terminate;
  end;



implementation

{$Resource FileIterator.res}
{$Include FileIterator.inc}

procedure IteratorError( MsgID :Integer );
begin
 raise EFileIteratorError.CreateRes( MsgID );
end;


{ TFileIteratorThread }

procedure TFileIteratorThread.Execute;
begin
  if assigned( FIterator ) then FIterator.Iterate;
end;


procedure TFileIteratorThread.Terminate;
begin
  if assigned( FIterator ) then
   if FIterator.Status = fsActive then FIterator.Cancel;
  inherited Terminate;
end;


constructor TFileIteratorThread.Create(
 Iterator :TFileIterator; CreateSuspended :Boolean );
begin
  inherited Create( CreateSuspended );
  FIterator := Iterator;
  FreeOnTerminate := true;
end;

{ TFileIterator }

function TFileIterator.GetIsIdle :Boolean;
begin
 result := ( Status = fsIdle );
end;

procedure TFileIterator.SetOptions( Value :TFileIteratorOptions );
begin
 if not ( IsIdle  or ( csLoading in COmponentState ))
  then IteratorError( SOptionsChangeError );
 FOptions := Value;
end;

procedure TFileIterator.SetRootFolder( const Value :String );
begin
 if not ( IsIdle  or ( csLoading in COmponentState ))
  then IteratorError( SOptionsChangeError );
 FRootFolder := Value;
end;

procedure TFileIterator.AddFile( const FileName :string;
 const FileInfo :TFileInfo );
begin
  if assigned( FOnAddFile ) then FOnAddFIle( Self, FileName, FileInfo );
end;

procedure TFileIterator.AddFileQuery( const FileName :string;
  const FileInfo :TFileInfo; Var CanAdd :Boolean );
begin
  if assigned( FOnAddFileQuery )
   then FOnAddFileQuery( Self, FileName, FileInfo, CanAdd );
end;

procedure TFileIterator.Activate;
begin
 if assigned( FOnActivate ) then FOnActivate( Self );
end;

procedure TFileIterator.Terminate;
begin
 if assigned( FOnTerminate ) then FOnTerminate( Self );
end;

constructor TFileIterator.Create( anOwner :TComponent );
begin
 inherited Create( anOwner );
 FStatus := fsIdle;
 FOptions := [ fiPath, fiDetails, fiRecurseFolders ];
 FRootFolder := 'C:';
end;

destructor TFileIterator.Destroy;
begin
 inherited;
end;

procedure TFileIterator.Cancel;
begin
  if Status = fsActive then FStatus := fsTerminated;
end;

procedure TFileIterator.Iterate;
begin
 if not IsIdle then IteratorError( SActive );
 if not ( Assigned( FOnAddFile ) or
  Assigned( FOnAddFileQuery ) ) then exit;
 FStatus := fsActive;
 try
   Activate;
   ScanFolder( RootFOlder );
 finally
   Terminate;
   FStatus := fsIdle;
 end;
end;

procedure TFileIterator.ScanFolder( Folder :String );
var
  R :Integer;
  D :TFileInfo;
  S :TSearchRec;
  B :Boolean;
begin
  FillChar( S, SizeOf( S ), 0 );
  if FOlder[ Length( Folder ) ] = '\'
   then SetLength( FOlder, Length( FOlder ) - 1 );
  R := FindFIrst( Folder + '\*.*', faAnyFile, S );
  try
   while ( R = 0 ) and ( Status <> fsTerminated ) do
    begin
      if ( S.Attr and faDirectory ) <> 0
       then begin
         if ( S.Name[ 1 ] <> '.' ) and ( fiRecurseFolders in Options )
          then ScanFolder( Folder + '\' + S.Name );
       end
       else
        if ( ( S.Attr and faVolumeID ) = 0 )
         then begin
           if ( fiDetails in Options ) then
             with D do begin
               Attr := S.Attr;
               Size := S.Size;
               Time := S.Time;
             end;
           B := true;
           if ( fiPath in Options )
             then begin
               AddFileQuery( Folder + '\' + S.Name, D, B );
               if b then AddFile( Folder + '\' + S.Name, D );
             end
             else begin
               AddFileQuery( S.Name , D, B );
               if ( B ) then AddFile( S.Name, D );
             end;
         end;
      R := FindNext( S );
    end;
  finally
    FindClose( S );
  end;
end;


end.
