(*******************************************)
(* Founder Lite Library                    *)
(* Version 1.0                             *)
(* Common tools and utils                  *)
(*                                         *)
(*******************************************)
(* (c) 1996-98 Evgesoft Company.           *)
(* All right reserved.                     *)
(*                                         *)
(*******************************************)
(* Internet: dev@evgesoft.com              *)
(* WWW:                                    *)
(*   http://www.evgesoft.com/dev/          *)
(*******************************************)

unit fdr_toolbar;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, CommCtrl, Menus, Buttons, fdr_Common;

const

  CapSize: integer = 20;

const

  NumPaletteEntries = 20;

  crHand    = 10;

  CM_ABDEACTIVE   = CM_BASE + 1200;

type

{ TitaGlyphs }

  TCellRange = 1..MaxInt;

  TitaGlyphs = class(TComponent)
  private
    FPicture: TPicture;
    FRows: TCellRange;
    FCols: TCellRange;
    FBitmap: TBitmap;
    FMasked: Boolean;
    FMaskColor: TColor;
    FOnChange: TNotifyEvent;
    function GetCell(Index: Integer): TBitmap;
    function GetDefaultMaskColor: TColor;
    function GetHeight: Integer;
    function GetWidth: Integer;
    function IsMaskStored: Boolean;
    procedure PictureChanged(Sender: TObject);
    procedure SetHeight(Value: Integer);
    procedure SetPicture(Value: TPicture);
    procedure SetWidth(Value: Integer);
  protected
    procedure Changed; dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Draw(Canvas: TCanvas; X, Y, Index: Integer);
    procedure DrawCenter(Canvas: TCanvas; Rect: TRect; Index: Integer);
    procedure LoadBitmapRes(Instance: THandle; ResID: PChar);
    property GraphicCell[Index: Integer]: TBitmap read GetCell;
  published
    property Cols: TCellRange read FCols write FCols default 1;
    property Height: Integer read GetHeight write SetHeight stored False;
    property Masked: Boolean read FMasked write FMasked default True;
    property Rows: TCellRange read FRows write FRows default 1;
    property Picture: TPicture read FPicture write SetPicture;
    property MaskColor: TColor read FMaskColor write FMaskColor stored IsMaskStored;
    property Width: Integer read GetWidth write SetWidth stored False;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

  TButtonLayout = (blGlyphLeft, blGlyphRight, blGlyphTop, blGlyphBottom);
  TButtonState = (bsUp, bsDisabled, bsDown, bsExclusive);
  TButtonStyle = (bsAutoDetect, bsWin31, bsNew);
  TNumGlyphs = 1..4;

  TitaSpeedButton = class(TGraphicControl)
  private
    FEnableFlat: boolean;
    FGroupIndex: Integer;
    FGlyph: Integer;
    FGlyphs: TitaGlyphs;
    FImages: TImageList;
    FDown: Boolean;
    FDragging: Boolean;
    FAllowAllUp: Boolean;
    FLayout: TButtonLayout;
    FFlat: Boolean;
    FMouseInControl: Boolean;
    FOnMouseEnter: TNotifyEvent;
    FOnMouseLeave: TNotifyEvent;
    procedure UpdateExclusive;
    function GetGlyph: Integer;
    procedure SetGlyph(Value: Integer);
    procedure SetDown(Value: Boolean);
    procedure SetFlat(Value: Boolean);
    procedure SetAllowAllUp(Value: Boolean);
    procedure SetGroupIndex(Value: Integer);
    procedure SetLayout(Value: TButtonLayout);
    procedure UpdateTracking;
    procedure WMLButtonDblClk(var Message: TWMLButtonDown); message WM_LBUTTONDBLCLK;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMButtonPressed(var Message: TMessage); message CM_BUTTONPRESSED;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;

    procedure SetGlyphs(Value: TitaGlyphs);
    procedure SetImages(Value: TImageList);
  protected
    FState: TButtonState;
    function GetPalette: HPALETTE; override;
    procedure Loaded; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Click; override;
  published
    property AllowAllUp: Boolean read FAllowAllUp write SetAllowAllUp default False;
    property GroupIndex: Integer read FGroupIndex write SetGroupIndex default 0;
    property Down: Boolean read FDown write SetDown default False;
    property Caption;
    property Enabled;
    property Flat: Boolean read FFlat write SetFlat default False;
    property Font;
    property GlyphIndex: Integer read GetGlyph write SetGlyph;
    property Glyphs: TitaGlyphs read FGlyphs write SetGlyphs;
    property Images: TImageList read FImages write SetImages;
    property Layout: TButtonLayout read FLayout write SetLayout default blGlyphLeft;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property Visible;
    property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TitaPopupButton = class(TitaSpeedButton)
  private
    FMenu: TPopupMenu;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
  protected
  public
    procedure Click; override;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  published
    property Menu: TPopupMenu read FMenu write FMenu;
  end;

{ Tool bar button components }

  TitaTBSeparator = class(TGraphicControl)
  private
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure SetParent(AParent: TWinControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Paint; override;
  published
    property Align;
    property Width;
    property Height;
  end;

  TitaTBDragBox = class(TGraphicControl)
  private
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure SetParent(AParent: TWinControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Paint; override;
  published
    property Align;
    property Width;
    property Height;
  end;

  TitaTBButton = class(TitaSpeedButton)
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
  end;

  TitaTBPopup = class(TitaPopupButton)
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
  end;

  TitaOfficePopupMenu = class;

{ Tool bar component }

  TitaDock = class(TitaCustomPanel)
  private
  protected
    { Protected declarations }
    procedure Loaded; override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property BevelOuter;
    property BevelInner;
    property BevelSides;
    property BorderColor;
    property VerticalAlignment;
    property Align;
    property Caption;
    property PopupMenu;
  end;

  TToolBarStyle = (psStatic, psFloat);
  
  TitaToolBar = class(TitaCustomPanel)
  private
    OldHeight: integer;
    Step: TPoint;
    Ctrls: array [0..100] of TControl;
    FStyle: TToolBarStyle;
    FAutoSave: boolean;
    FCloseButton: boolean;
    FAllowResize: boolean;
    FDragBox: TitaTBDragBox;
    FOnShow: TNotifyEvent;
    FOnHide: TNotifyEvent;
    FOnFloat: TNotifyEvent;
    FOnStatic: TNotifyEvent;
    function FindDock(const Pos: TPoint): TitaDock;
    function FindToolBar(const Pos: TPoint): TitaToolBar;
    procedure SwapToolBar(ToolBar: TitaToolBar);
    procedure DeleteFromDock;
    function isInsert(TB: TitaToolBar): integer;

    procedure DoStatic;
    procedure DoFloat;
    procedure WMSize(var Msg: TWMSize);  message WM_SIZE;
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;

    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Paint; override;
    function GetClientRect: TRect; override;

    procedure InsertToDock;
    procedure Aligning;
    procedure SortControls;
    procedure CorrectAlign;

    procedure PublicMouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
    procedure PublicMouseMove(Shift: TShiftState; X, Y: Integer);
    procedure PublicMouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
    procedure HidePanel;
    procedure ShowPanel;
    function isFloat(var X,Y: Integer): boolean;
    procedure FloatToPos(X,Y: Integer);
  published
    property Caption;
    property PopupMenu;
    property Visible;
    property Font;
    property DragMode;
    property DragCursor;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag; 
    property AutoSave: boolean read FAutoSave write FAutoSave;
    property CloseButton: boolean read FAutoSave write FAutoSave;
    property AllowResize: boolean read FAllowResize write FAllowResize;
    property OnShow: TNotifyEvent read FOnShow write FOnShow;
    property OnHide: TNotifyEvent read FOnHide write FOnHide;
    property OnFloat: TNotifyEvent read FOnFloat write FOnFloat;
    property OnStatic: TNotifyEvent read FOnStatic write FOnStatic;
  end;

{ Menu items }

  TitaMenuButton = class(TitaSpeedButton)
  private
    FPrevWndProc: Pointer;
    FObjectInst: Pointer;
    FMenu: TPopupMenu;
    FMenuGroupIndex: integer;
    procedure MyWndProc(var Msg: TMessage);
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Loaded; override;
    procedure Paint; override;
    procedure Click; override;
  published
    property Align;
    property Menu: TPopupMenu read FMenu write FMenu;
    property MenuGroupIndex: integer read FMenuGroupIndex
      write FMenuGroupIndex default 0;
  end;

  TitaMenuBar = class(TitaToolBar)
  private
  protected
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;
  published
  end;

  TMenuBannerPosition = (mbpLeft, mbpRight);
  TPicturePosition = (ppTop, ppBottom, ppCenter);
  
  TitaMenuBanner = class(TPersistent)
  private
    FBackColor: TColor;
    FPicture: TPicture;
    FPosition: TMenuBannerPosition;
    FPicturePosition: TPicturePosition;
    procedure SetPicture(Value: TPicture);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property BackColor: TColor read FBackColor
      write FBackColor;
    property Picture: TPicture read FPicture
      write SetPicture;
    property Position: TMenuBannerPosition read FPosition
      write FPosition;
    property PicturePosition: TPicturePosition read FPicturePosition
      write FPicturePosition;
  end;
  
  TitaOfficePopupMenu = class(TitaCustomPopupMenu)
  private
    FGlyphsEdit: integer;
    FGlyphs: TitaGlyphs;
    FImages: TImageList;
    FBanner: TitaMenuBanner;
    FHeight: integer;
    procedure DoDrawItem(MenuItem: TMenuItem; Canvas: TCanvas;
      ItemState: Cardinal; ItemRect: TRect); override;
    procedure DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas); override;

    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;

    procedure SetGlyphs(Value: TitaGlyphs);
    procedure SetImages(Value: TImageList);

    function  ProcessAccel(Key: Word): TMenuItem;
  protected
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;

    procedure Popup(X, Y: integer); override;
  published
    property Banner: TitaMenuBanner read FBanner
      write FBanner;
    property Glyphs: TitaGlyphs read FGlyphs write SetGlyphs;
    property Images: TImageList read FImages write SetImages;
    property GlyphsEdit: Integer read FGlyphsEdit write FGlyphsEdit;
  end;

procedure DrawBitmapTransparent(Dest: TCanvas; XOrigin, YOrigin: Integer;
  Bitmap: TBitmap; TransparentColor: TColor);

implementation {==============================================================}

uses FLPfrm97, ShellAPI;

{$R fdr_toolbar.res }

const

  Creator: boolean = false;

var
  bmpArrow: TBitmap;
  bmpCheck: TBitmap;

{ DrawButtonFace - returns the remaining usable area inside the Client rect.}
function DrawButtonFace(Canvas: TCanvas; const Client: TRect;
  BevelWidth: Integer; Style: TButtonStyle; IsRounded, IsDown,
  IsFocused: Boolean): TRect;
var
  NewStyle: Boolean;
  R: TRect;
  DC: THandle;
begin
  NewStyle := ((Style = bsAutoDetect) and NewStyleControls) or (Style = bsNew);

  R := Client;
  with Canvas do
  begin
    if NewStyle then
    begin
      Brush.Color := clBtnFace;
      Brush.Style := bsSolid;
      DC := Canvas.Handle;    { Reduce calls to GetHandle }

      if IsDown then
      begin    { DrawEdge is faster than Polyline }
        DrawEdge(DC, R, BDR_SUNKENINNER, BF_TOPLEFT);              { black     }
        DrawEdge(DC, R, BDR_SUNKENOUTER, BF_BOTTOMRIGHT);          { btnhilite }
        Dec(R.Bottom);
        Dec(R.Right);
        Inc(R.Top);
        Inc(R.Left);
        DrawEdge(DC, R, BDR_SUNKENOUTER, BF_TOPLEFT or BF_MIDDLE); { btnshadow }
      end
      else
      begin
        DrawEdge(DC, R, BDR_RAISEDOUTER, BF_BOTTOMRIGHT);          { black }
        Dec(R.Bottom);
        Dec(R.Right);
        DrawEdge(DC, R, BDR_RAISEDINNER, BF_TOPLEFT);              { btnhilite }
        Inc(R.Top);
        Inc(R.Left);
        DrawEdge(DC, R, BDR_RAISEDINNER, BF_BOTTOMRIGHT or BF_MIDDLE); { btnshadow }
      end;
    end
    else
    begin
      Pen.Color := clWindowFrame;
      Brush.Color := clBtnFace;
      Brush.Style := bsSolid;
      Rectangle(R.Left, R.Top, R.Right, R.Bottom);

      { round the corners - only applies to Win 3.1 style buttons }
      if IsRounded then
      begin
        Pixels[R.Left, R.Top] := clBtnFace;
        Pixels[R.Left, R.Bottom - 1] := clBtnFace;
        Pixels[R.Right - 1, R.Top] := clBtnFace;
        Pixels[R.Right - 1, R.Bottom - 1] := clBtnFace;
      end;

      if IsFocused then
      begin
        InflateRect(R, -1, -1);
        Brush.Style := bsClear;
        Rectangle(R.Left, R.Top, R.Right, R.Bottom);
      end;

      InflateRect(R, -1, -1);
      if not IsDown then
        Frame3D(Canvas, R, clBtnHighlight, clBtnShadow, BevelWidth)
      else
      begin
        Pen.Color := clBtnShadow;
        PolyLine([Point(R.Left, R.Bottom - 1), Point(R.Left, R.Top),
          Point(R.Right, R.Top)]);
      end;
    end;
  end;

  Result := Rect(Client.Left + 1, Client.Top + 1,
    Client.Right - 2, Client.Bottom - 2);
  if IsDown then OffsetRect(Result, 1, 1);
end;

{ TitaGlyphs }

constructor TitaGlyphs.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  FBitmap := TBitmap.Create;
  FRows := 1;
  FCols := 1;
  FMaskColor := GetDefaultMaskColor;
  FMasked := True;
end;

destructor TitaGlyphs.Destroy;
begin
  FPicture.OnChange := nil;
  FBitmap.Free;
  FPicture.Free;
  inherited Destroy;
end;

procedure TitaGlyphs.Changed;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

{**************************************************************************}

{ Transparent bitmap }

function Max(A, B: Longint): Longint;
begin
  if A > B then Result := A
  else Result := B;
end;

function Min(A, B: Longint): Longint;
begin
  if A < B then Result := A
  else Result := B;
end;

const
{ TBitmap.GetTransparentColor from GRAPHICS.PAS use this value }
  TransparentMask = $02000000;

procedure DrawTransparentBitmapRect(DC: HDC; Bitmap: HBitmap; xStart, yStart,
  Width, Height: Integer; Rect: TRect; TransparentColor: TColorRef);
var
{$IFDEF WIN32}
  BM: Windows.TBitmap;
{$ELSE}
  BM: WinTypes.TBitmap;
{$ENDIF}
  cColor: TColorRef;
  bmAndBack, bmAndObject, bmAndMem, bmSave: HBitmap;
  bmBackOld, bmObjectOld, bmMemOld, bmSaveOld: HBitmap;
  hdcMem, hdcBack, hdcObject, hdcTemp, hdcSave: HDC;
  ptSize, ptRealSize, ptBitSize, ptOrigin: TPoint;
begin
  hdcTemp := CreateCompatibleDC(DC);
  SelectObject(hdcTemp, Bitmap);      { Select the bitmap    }
  GetObject(Bitmap, SizeOf(BM), @BM);
  ptRealSize.x := Min(Rect.Right - Rect.Left, BM.bmWidth - Rect.Left);
  ptRealSize.y := Min(Rect.Bottom - Rect.Top, BM.bmHeight - Rect.Top);
  DPtoLP(hdcTemp, ptRealSize, 1);
  ptOrigin.x := Rect.Left;
  ptOrigin.y := Rect.Top;
  DPtoLP(hdcTemp, ptOrigin, 1);       { Convert from device  }
                                      { to logical points    }
  ptBitSize.x := BM.bmWidth;          { Get width of bitmap  }
  ptBitSize.y := BM.bmHeight;         { Get height of bitmap }
  DPtoLP(hdcTemp, ptBitSize, 1);
  if (ptRealSize.x = 0) or (ptRealSize.y = 0) then begin
    ptSize := ptBitSize;
    ptRealSize := ptSize;
  end
  else ptSize := ptRealSize;
  if (Width = 0) or (Height = 0) then begin
    Width := ptSize.x;
    Height := ptSize.y;
  end;

  { Create some DCs to hold temporary data }
  hdcBack   := CreateCompatibleDC(DC);
  hdcObject := CreateCompatibleDC(DC);
  hdcMem    := CreateCompatibleDC(DC);
  hdcSave   := CreateCompatibleDC(DC);
  { Create a bitmap for each DC. DCs are required for a number of }
  { GDI functions                                                 }
  { Monochrome DC }
  bmAndBack   := CreateBitmap(ptSize.x, ptSize.y, 1, 1, nil);
  bmAndObject := CreateBitmap(ptSize.x, ptSize.y, 1, 1, nil);
  bmAndMem    := CreateCompatibleBitmap(DC, Max(ptSize.x, Width), Max(ptSize.y, Height));
  bmSave      := CreateCompatibleBitmap(DC, ptBitSize.x, ptBitSize.y);
  { Each DC must select a bitmap object to store pixel data }
  bmBackOld   := SelectObject(hdcBack, bmAndBack);
  bmObjectOld := SelectObject(hdcObject, bmAndObject);
  bmMemOld    := SelectObject(hdcMem, bmAndMem);
  bmSaveOld   := SelectObject(hdcSave, bmSave);
  { Set proper mapping mode }
  SetMapMode(hdcTemp, GetMapMode(DC));

  { Save the bitmap sent here, because it will be overwritten }
  BitBlt(hdcSave, 0, 0, ptBitSize.x, ptBitSize.y, hdcTemp, 0, 0, SRCCOPY);
  { Set the background color of the source DC to the color,         }
  { contained in the parts of the bitmap that should be transparent }
  cColor := SetBkColor(hdcTemp, TransparentColor);
  { Create the object mask for the bitmap by performing a BitBlt()  }
  { from the source bitmap to a monochrome bitmap                   }
  BitBlt(hdcObject, 0, 0, ptSize.x, ptSize.y, hdcTemp, ptOrigin.x, ptOrigin.y,
    SRCCOPY);
  { Set the background color of the source DC back to the original  }
  { color                                                           }
  SetBkColor(hdcTemp, cColor);
  { Create the inverse of the object mask }
  BitBlt(hdcBack, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0,
    NOTSRCCOPY);
  { Copy the background of the main DC to the destination }
  BitBlt(hdcMem, 0, 0, Width, Height, DC, xStart, yStart,
    SRCCOPY);
  { Mask out the places where the bitmap will be placed }
  StretchBlt(hdcMem, 0, 0, Width, Height, hdcObject, 0, 0,
    ptSize.x, ptSize.y, SRCAND);
  {BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0, SRCAND);}
  { Mask out the transparent colored pixels on the bitmap }
  BitBlt(hdcTemp, ptOrigin.x, ptOrigin.y, ptSize.x, ptSize.y, hdcBack, 0, 0,
    SRCAND);
  { XOR the bitmap with the background on the destination DC }
  StretchBlt(hdcMem, 0, 0, Width, Height, hdcTemp, ptOrigin.x, ptOrigin.y,
    ptSize.x, ptSize.y, SRCPAINT);
  {BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcTemp, ptOrigin.x, ptOrigin.y,
    SRCPAINT);}
  { Copy the destination to the screen }
  BitBlt(DC, xStart, yStart, Max(ptRealSize.x, Width), Max(ptRealSize.y, Height),
    hdcMem, 0, 0, SRCCOPY);
  { Place the original bitmap back into the bitmap sent here }
  BitBlt(hdcTemp, 0, 0, ptBitSize.x, ptBitSize.y, hdcSave, 0, 0, SRCCOPY);

  { Delete the memory bitmaps }
  DeleteObject(SelectObject(hdcBack, bmBackOld));
  DeleteObject(SelectObject(hdcObject, bmObjectOld));
  DeleteObject(SelectObject(hdcMem, bmMemOld));
  DeleteObject(SelectObject(hdcSave, bmSaveOld));
  { Delete the memory DCs }
  DeleteDC(hdcMem);
  DeleteDC(hdcBack);
  DeleteDC(hdcObject);
  DeleteDC(hdcSave);
  DeleteDC(hdcTemp);
end;

function WidthOf(R: TRect): Integer;
begin
  Result := R.Right - R.Left;
end;

function HeightOf(R: TRect): Integer;
begin
  Result := R.Bottom - R.Top;
end;

procedure InternalDrawTransBmpRect(Dest: TCanvas; X, Y, W, H: Integer;
  Rect: TRect; Bitmap: TBitmap; TransparentColor: TColor);
var
  MemImage: TBitmap;
  R: TRect;
begin
  MemImage := TBitmap.Create;
  try
    R := Bounds(0, 0, Bitmap.Width, Bitmap.Height);
    if TransparentColor = clNone then begin
      if (WidthOf(Rect) <> 0) and (HeightOf(Rect) <> 0) then R := Rect;
      MemImage.Width := WidthOf(R);
      MemImage.Height := HeightOf(R);
      MemImage.Canvas.CopyRect(Bounds(0, 0, MemImage.Width, MemImage.Height),
        Bitmap.Canvas, R);
      if (W = 0) or (H = 0) then Dest.Draw(X, Y, MemImage)
      else Dest.StretchDraw(Bounds(X, Y, W, H), MemImage);
    end
    else  begin
      MemImage.Width := WidthOf(R);
      MemImage.Height := HeightOf(R);
      MemImage.Canvas.CopyRect(R, Bitmap.Canvas, R);
//      TransparentColor := MemImage.Canvas.Pixels[0, MemImage.Height - 1];
      DrawTransparentBitmapRect(Dest.Handle, MemImage.Handle, X, Y, W, H,
        Rect, ColorToRGB(TransparentColor and not TransparentMask));
      { TBitmap.TransparentColor property return TColor value equal   }
      { to (Bitmap.Canvas.Pixels[0, Height - 1] or TransparentMask).  }
    end;
  finally
    MemImage.Free;
  end;
end;

procedure DrawBitmapRectTransparent(Dest: TCanvas; XOrigin, YOrigin: Integer;
  Rect: TRect; Bitmap: TBitmap; TransparentColor: TColor);
begin
  InternalDrawTransBmpRect(Dest, XOrigin, YOrigin, 0, 0, Rect, Bitmap,
    TransparentColor);
end;

procedure DrawBitmapTransparent(Dest: TCanvas; XOrigin, YOrigin: Integer;
  Bitmap: TBitmap; TransparentColor: TColor);
begin
  InternalDrawTransBmpRect(Dest, XOrigin, YOrigin, 0, 0, Rect(0, 0, 0, 0),
    Bitmap, TransparentColor);
end;

procedure TitaGlyphs.Draw(Canvas: TCanvas; X, Y, Index: Integer);
var
  Image: TGraphic;
begin
  if Index < 0 then Image := Picture.Graphic
  else Image := GraphicCell[Index];
  if Image <> nil then begin
    if FMasked and (FMaskColor <> clNone) and
      (Picture.Graphic is TBitmap) then
      DrawBitmapTransparent(Canvas, X, Y, TBitmap(Image), FMaskColor)
    else Canvas.Draw(X, Y, Image);
  end;
end;

procedure TitaGlyphs.DrawCenter(Canvas: TCanvas; Rect: TRect; Index: Integer);
var
  X, Y: Integer;
begin
  X := (Rect.Left + Rect.Right - Width) div 2;
  Y := (Rect.Bottom + Rect.Top - Height) div 2;
  Draw(Canvas, X, Y, Index);
end;

procedure TitaGlyphs.LoadBitmapRes(Instance: THandle; ResID: PChar);
var
  Bmp: TBitmap;
begin
  Bmp := TBitmap.Create;
  try
    Bmp.Handle := LoadBitmap(Instance, ResID);
    Picture.Assign(Bmp);
  finally
    Bmp.Free;
  end;
end;

procedure AssignBitmapCell(Source: TGraphic; Dest: TBitmap; Cols, Rows,
  Index: Integer);
var
  Temp: TBitmap;
  BWidth, BHeight: Integer;
  SrcR, DestR: TRect;
begin
  if (Source <> nil) and (Dest <> nil) then begin
    if Source is TBitmap then Temp := Source as TBitmap
    else Temp := TBitmap.Create;
    try
      if Cols <= 0 then Cols := 1;
      if Rows <= 0 then Rows := 1;
      if Index < 0 then Index := 0;
      if not (Source is TBitmap) then begin
        Temp.Width := Source.Width;
        Temp.Height := Source.Height;
        Temp.Canvas.Draw(0, 0, Source);
      end;
      BWidth := Temp.Width div Cols;
      BHeight := Temp.Height div Rows;
      DestR := Bounds(0, 0, BWidth, BHeight);
      SrcR := Bounds((Index mod Cols) * BWidth,
        (Index div Cols) * BHeight, BWidth, BHeight);
      with Dest do begin
        Width := BWidth;
        Height := BHeight;
        Canvas.CopyRect(DestR, Temp.Canvas, SrcR);
      end;
    finally
      if not (Source is TBitmap) then Temp.Free;
    end;
  end;
end;

function TitaGlyphs.GetCell(Index: Integer): TBitmap;
begin
  AssignBitmapCell(Picture.Graphic, FBitmap, Cols, Rows, Index);
  Result := FBitmap;
end;

function TitaGlyphs.GetDefaultMaskColor: TColor;
begin
  Result := clOlive;
  if (Picture.Graphic <> nil) and (Picture.Graphic is TBitmap) then
    Result := TBitmap(Picture.Graphic).TransparentColor and
      not TransparentMask;
end;

function TitaGlyphs.GetHeight: Integer;
begin
  Result := Picture.Height div FRows;
end;

function TitaGlyphs.GetWidth: Integer;
begin
  Result := Picture.Width div FCols;
end;

function TitaGlyphs.IsMaskStored: Boolean;
begin
  Result := MaskColor <> GetDefaultMaskColor;
end;

procedure TitaGlyphs.PictureChanged(Sender: TObject);
begin
  FMaskColor := GetDefaultMaskColor;
  if not (csReading in ComponentState) then Changed;
end;

procedure TitaGlyphs.SetHeight(Value: Integer);
begin
  if (Value > 0) and (Picture.Height div Value > 0) then
    Rows := Picture.Height div Value;
end;

procedure TitaGlyphs.SetWidth(Value: Integer);
begin
  if (Value > 0) and (Picture.Width div Value > 0) then
    Cols := Picture.Width div Value;
end;

procedure TitaGlyphs.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

{ TitaDock }

constructor TitaDock.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Height := 48;
  Align := alTop;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Caption := '';
end;

procedure TitaDock.Loaded;
begin
  inherited Loaded;
  If (csDesigning in ComponentState) and (ControlCount = 0) Then
  begin
    Width := 10;
    Height := 10;
  end;
  If not (csDesigning in ComponentState) and (ControlCount = 0) Then
  begin
    Width := 0;
    Height := 0;
  end;
end;

procedure TitaDock.Paint;
begin
  inherited Paint; 
  If csDesigning in ComponentState Then
    with Canvas do
    begin
      Pen.Color := clBtnShadow;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width-1, Height-1);
    end;
end;

{ TitaToolBar }

constructor TitaToolBar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  Left := 0; Top := 0;
  Height := 26;
  Caption := ClassName + IntToStr(AOwner.ComponentCount);
  BevelInner := bvRaised;
  BevelOuter := bvNone;
  Visible := true;
  Creator := true;
  FAllowResize := false;
  FAutoSave := true;
  FCloseButton := true;

  FDragBox := TitaTBDragBox.Create(Self);
  FDragBox.Left := 0;
  FDragBox.Align := alLeft;
  FDragBox.Parent := Self;
  FDragBox.Visible := true;
end;

procedure TitaToolBar.Loaded;
begin
  inherited Loaded;
  Creator := true;
  SortControls;
  If Parent.Align in [alTop, alBottom] Then
    OldHeight := Height
  else
    OldHeight := Width;
  Creator := false;
end;

procedure TitaToolBar.CorrectAlign;
var
  t: integer;
begin
  // Correcing align
  case Parent.Align of
    alTop, alBottom, alNone:
    begin
      for t := 0 to ControlCount-1 do
        If Ctrls[t] <> nil Then
          If not (Ctrls[t].Align in [alLeft,alRight]) then
            If (Ctrls[t].Align = alBottom) then
              Ctrls[t].Align := alRight
            else
            begin
              Ctrls[t].Left := Ctrls[t].Top;
              Ctrls[t].Align := alLeft;
            end;
    end;
    alLeft, alRight:
    begin
      for t := 0 to ControlCount-1 do
        If Ctrls[t] <> nil Then
          If not (Ctrls[t].Align in [alTop,alBottom]) Then
            If (Ctrls[t].Align = alRight) then
              Ctrls[t].Align := alBottom
            else
            begin
              Ctrls[t].Top := Ctrls[t].Left;
              Ctrls[t].Align := alTop;
            end;
    end;
  end;
end;
 
procedure TitaToolBar.SortControls;
var
  i, j: integer;
  TempCtrl: TControl;
begin
  // Sorting controls by position
  for i := 0 to ControlCount-1 do
    Ctrls[i] := Controls[i];
  for i := 0 to ControlCount-1 do
    for j := 0 to ControlCount-2 do
      If Parent.Align in [altop, alBottom, alNone] Then
      begin
        If Ctrls[j].Left > Ctrls[j+1].Left Then
        begin
          TempCtrl := Ctrls[j+1];
          Ctrls[j+1] := Ctrls[j];
          Ctrls[j] := TempCtrl;
        end;
      end
      else
      begin
        If (Ctrls[j].Align in [alTop, alBottom]) and
           (Ctrls[j+1].Align in [alTop, alBottom]) and
           (Ctrls[j].Top > Ctrls[j+1].Top) Then
        begin
          TempCtrl := Ctrls[j];
          Ctrls[j] := Ctrls[j+1];
          Ctrls[j+1] := TempCtrl;
        end;
      end;
end;

procedure TitaToolBar.Paint;
begin
  inherited Paint;
end;

procedure TitaToolBar.PublicMouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  MouseDown(Button, Shift, X,Y);
end;

procedure TitaToolBar.PublicMouseMove(Shift: TShiftState; X, Y: Integer);
begin
  MouseMove(Shift, X, Y);
end;

procedure TitaToolBar.PublicMouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  MouseUp(Button, Shift, X, Y);
end;

function TitaToolBar.GetClientRect: TRect;
begin
  Result := inherited GetClientRect;
  InflateRect(Result, -1, -1);
end;

procedure TitaToolBar.Aligning;
begin
  If Left < 0 Then Left := 0;
  If Left + Width > Parent.Width Then Left := Parent.Width - Width;
  If Top < 0 Then Top := 0;
  If Top + Height > Parent.Height Then Top := Parent.Height - Height;
  FDragBox.Top := 0;
  FDragBox.Left := 0;
end;

procedure TitaToolBar.DeleteFromDock;
var
  t: integer;
  Multi: boolean;
begin
  If not Creator Then Hide;
  BevelInner := bvNone;
  FDragBox.Hide;
  case Parent.Align of
    alTop, alBottom: begin
      // Change height of dock
      Multi := false;
      for t := 0 to Parent.ControlCount-1 do
        If (Parent.Controls[t].Top = Top) and
           (Parent.Controls[t].Height = Height) and
           (Parent.Controls[t] <> Self) Then
          Multi := true;
      If not Multi Then
      begin
        Parent.Height := Parent.Height - Height;
        // Change position other toolbars
        for t := 0 to Parent.ControlCount-1 do
          If Parent.Controls[t].Top >= Top + Height Then
            Parent.Controls[t].Top := Parent.Controls[t].Top - Height;
      end;
    end;
    alLeft, alRight: begin
      // Change height of dock
      Multi := false;
      for t := 0 to Parent.ControlCount-1 do
        If (Parent.Controls[t].Left = Left) and
           (Parent.Controls[t].Width = Width) and
           (Parent.Controls[t] <> Self) Then Multi := true;
      If not Multi Then
      begin
        Parent.Width := Parent.Width - Width;
        // Change position other toolbars
        for t := 0 to Parent.ControlCount-1 do
          If Parent.Controls[t].Left >= Left + Width Then
            Parent.Controls[t].Left := Parent.Controls[t].Left - Width;
      end;
      // Restore position
      If Height > Width Then
      begin
        Step := Point(Step.Y, Step.X);
        T := Width;
        Width := Height;
        Height := T;
        for t := ControlCount-1 DownTo 0 do
          If Ctrls[t] <> nil Then
            If Ctrls[t].Align = alBottom then
              Ctrls[t].Align := alRight
            else
              Ctrls[t].Align := alLeft;
      end;
    end;
  end;
  If not Creator Then Show;
end;

procedure TitaToolBar.InsertToDock;
var
  T: Integer;
  Multi: boolean;
  Min: TitaToolBar;
begin
  Hide;
  If not Creator Then DoStatic;
  BevelInner := bvRaised;
  FDragBox.Show;
  case Parent.Align of
    alTop, alBottom: begin
      If Height > Width Then
      begin
        Step := Point(Step.Y, Step.X);
        T := Width;
        Width := Height;
        Height := T;
        // Change child Ctrls align
        for t := 0 to ControlCount-1 do
          If Ctrls[t] <> nil Then
            If Ctrls[t].Align = alBottom then
              Ctrls[t].Align := alRight
            else
            begin
              Ctrls[t].Left := Ctrls[t].Top;
              Ctrls[t].Align := alLeft;
            end;
      end;
      Top := Parent.Height;
      Multi := false;
      Min := Self;
      for t := 0 to Parent.ControlCount-1 do
        If (Parent.Controls[t].Top = Top) and
           (Parent.Controls[t].Height = Height) and
           (Parent.Controls[t] <> Self) Then
        begin
          Multi := true;
          If Parent.Controls[t].Left < Min.Left Then
          Min := Parent.Controls[t] as TitaToolBar;
        end;
      If not Multi Then
        Parent.Height := Parent.Height + Height;
      If Multi and (Min = Self) Then
        Parent.Height := Parent.Height + Height;
    end;
    alLeft, alRight: begin
      If Width > Height Then
      begin
        Step := Point(Step.Y, Step.X);
        T := Width;
        Width := Height;
        Height := T;
        // Change child Ctrls align
        for t := 0 to ControlCount-1 do
          If Ctrls[t] <> nil Then
            If Ctrls[t].Align = alRight then
              Ctrls[t].Align := alBottom
            else
            begin
              Ctrls[t].Top := Ctrls[t].Left;
              Ctrls[t].Align := alTop;
            end;
      end;
      Left := Parent.Width;
      Parent.Width := Parent.Width + Width;
    end;
  end;
  Show;
  If not Creator Then MouseCapture := true;
end;

function TitaToolBar.isInsert(TB: TitaToolBar): integer;
var
  i: integer;
  Min, Max: integer;
  isLine: boolean; 
begin
  // in one height ?
  isInsert := -1;
  case Parent.Align of
    alTop, alBottom: If Height <> TB.Height Then Exit;
    alLeft, alRight: If Width <> TB.Width Then Exit;
  end;
  // Verifing
  Max := 0;
  case Parent.Align of
    alTop, alBottom: Min := Parent.Width;
    alLeft, alRight: Min := Parent.Height;
  end;
  isLine := false;
  with Parent do
    for i := 0 to ControlCount-1 do
      If (Controls[i] is TitaToolBar)  then
        case Align of
          alTop, alBottom:
            If (Controls[i] as TitaToolBar).Top = Self.Top Then
              with (Controls[i] as TitaToolBar) do
              begin
                isLine := true;
                If Left < Min then Min := Left;
                If Left + Width > Max then Max := Left + Width;
              end;
          alLeft, alRight:
            If (Controls[i] as TitaToolBar).Left = Self.Left Then
              with (Controls[i] as TitaToolBar) do
              begin
                isLine := true;
                If Top < Min then Min := Top;
                If Top + Height > Max then Max := Top + Height;
              end;
        end;
  If not isLine Then Exit;
  // Return Result
  case Parent.Align of
    alTop, alBottom:
      If (Min < TB.Width) then
        If (Parent.Width-Max < TB.Width) then exit
        else
          isInsert := Max
      else
        isInsert := 0;
    alLeft, alRight:
      If (Min < TB.Height) then
        If (Parent.Height-Max < TB.Height) then exit
        else
          isInsert := Max
      else
        isInsert := 0;
  end;
end;

procedure TitaToolBar.SwapToolBar(ToolBar: TitaToolBar);
var
  i, T: integer;
  T1, T2: TitaToolBar;
begin
  If (ToolBar = nil) or (ToolBar = Self) Then Exit;
  case Parent.Align of
    alTop, alBottom:
      begin
        If ToolBar.Top > Top Then
        begin
          T1 := Self;
          T2 := ToolBar;
        end
        else
        begin
          T1 := ToolBar;
          T2 := Self;
        end;
        If T1.Top = T2.Top Then
        begin
          // in one line
          If (T2.Left < T1.Left) and (T2.Left+T2.Width > T1.Left) Then
            T2.Left := T1.Left-T2.Width;
          If (T2.Left+T2.Width > T1.Left) and (T2.Left < T1.Left+T1.Width) then
            T1.Left := T1.Left+T1.Width;
        end
        else
        begin
          // Shift tool bar > T2
          Hide;
          ToolBar.Hide;
          with Parent do
            If ToolBar.isInsert(Self) <> -1 Then
            begin
              // Multi tool bar
              Self.Left := ToolBar.isInsert(Self);
              Self.Top := ToolBar.Top;
              Height := Height - ToolBar.Height;
              for i := 0 to ControlCount-1 do
                If (Controls[i] is TitaToolBar) and
                   (Controls[i].Top > T2.Top) Then
                    Controls[i].Top := Controls[i].Top - ToolBar.Height;
              Self.Show;
              ToolBar.Show;
              Exit;
            end
            else
              for i := 0 to ControlCount-1 do
                If (Controls[i] is TitaToolBar) and
                   (Controls[i].Top > T2.Top) Then
                    Controls[i].Top := Controls[i].Top - (T2.Height - T1.Height);
           // Shift T1
          T := T1.Top;
          T1.Top := T2.Top;
          // Shift tool bar > T1
          with Parent do
            for i := 0 to ControlCount-1 do
              If (Controls[i] is TitaToolBar) and
                 (Controls[i].Top > T) Then
                   Controls[i].Top := Controls[i].Top + (T2.Height - T1.Height);
          // Shift T2
          T2.Top := T;
          // Find other TB in line
          with Parent do
            for i := 0 to ControlCount-1 do
              If (Controls[i] is TitaToolBar) and
                 (Controls[i].Top = ToolBar.Top) and
                 (Controls[i] <> ToolBar) Then
                    (Controls[i] as TitaToolBar).Top := Self.Top;
          Show;
          ToolBar.Show;
        end;
      end;
    alLeft, alRight:
      begin
        If ToolBar.Left > Left Then
        begin
          T1 := Self;
          T2 := ToolBar;
        end
        else
        begin
          T1 := ToolBar;
          T2 := Self;
        end;
        If T1.Left = T2.Left Then
        begin
          // in one line
          If (T2.Top < T1.Top) and (T2.Top+T2.Height > T1.Top) Then
            T2.Top := T1.Top-T2.Height;
          If (T2.Top+T2.Height > T1.Top) and (T2.Top < T1.Top+T1.Height) then
            T1.Top := T1.Top+T1.Height;
        end
        else
        begin
          // Shift tool bar > T2
          Hide;
          ToolBar.Hide;
          with Parent do
            If ToolBar.isInsert(Self) <> -1 Then
            begin
              // Multi tool bar
              Self.Top := ToolBar.isInsert(Self);
              Self.Left := ToolBar.Left;
              Width := Width - ToolBar.Width;
              for i := 0 to ControlCount-1 do
                If (Controls[i] is TitaToolBar) and
                   (Controls[i].Left > T2.Left) Then
                    Controls[i].Left := Controls[i].LEft - ToolBar.Width;
              Self.Show;
              ToolBar.Show;
              Exit;
            end
            else
              for i := 0 to ControlCount-1 do
                If (Controls[i] is TitaToolBar) and
                   (Controls[i].Left > T2.LEft) Then
                    Controls[i].Left := Controls[i].Left - (T2.Width - T1.Width);
           // Shift T1
          T := T1.Left;
          T1.Left := T2.Left;
          // Shift tool bar > T1
          with Parent do
            for i := 0 to ControlCount-1 do
              If (Controls[i] is TitaToolBar) and
                 (Controls[i].Left > T) Then
                   Controls[i].Left := Controls[i].Left + (T2.Width - T1.Width);
          // Shift T2
          T2.Left := T;
          // Find other TB in line
          with Parent do
            for i := 0 to ControlCount-1 do
              If (Controls[i] is TitaToolBar) and
                 (Controls[i].Left = Self.Left) and
                 (Controls[i] <> Self) Then
                    (Controls[i] as TitaToolBar).Left := ToolBar.Left;
          Show;
          ToolBar.Show;
        end;
      end;
  end;
  Application.ProcessMessages;
end;

procedure TitaToolBar.WMSize(var Msg: TWMSize);
begin
  inherited;
  with Msg do
    If Parent.Align in [alTop, alBottom] Then
      FDragBox.Align := alLeft
    else
      FDragBox.Align := alTop;
  FDragBox.Top := 0;
  FDragBox.Left := 0;
end;

{ Tempolary variables }

procedure TitaToolBar.MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  inherited;
  If Button = mbLeft Then begin
    If FStyle = psStatic Then
    // Static
      FStyle := psStatic
    else
    // Float
      FStyle := psFloat;
    Step := Point(X, Y);
  end;
end;

const
  Bisy: boolean = false;

procedure TitaToolBar.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  D: TitaDock;
  FP: TFPform97;
  P: Tpoint;
  T: integer;
begin
  If Bisy Then Exit;
  Bisy := true;
  inherited;
  If ssLeft in Shift Then begin
    D := FindDock(Point(X, Y));
    If D <> nil Then
    begin
    // Make static
      If Parent = D Then
      begin
        SwapToolBar(FindToolBar(Point(X, Y)));
        If Parent.Align in [alTop, alBottom] Then
        begin
          T := Left;
          T := T + X - Step.X;
          If (T < 0) Then T := 0;
          If (T + Width > Parent.Width) Then T := Parent.Width - Width;
          MoveWindow(Handle, T, Top, Width, Height, true);
        end;
        If Parent.Align in [alLeft, alRight] Then
        begin
          T := Top;
          T := T + Y - Step.Y;
          If (T + Height > Parent.Height) Then T := Parent.Height - Height;
          If (T < 0) Then T := 0;
          MoveWindow(Handle, Left, T, Width, Height, true);
        end;
      end
      else
      begin
        If Parent is TFPform97 Then
        begin
          FStyle := psStatic;
          FP := (Parent as TFPform97);
          SetParent(D);
          InsertToDock;
          FP.Free;
        end
        else
        begin
          DeleteFromDock;
          SetParent(D);
          InsertToDock;
        end;
      end;
    end
    else
    begin
    // Make float
      If FStyle = psStatic Then
      begin
        D := Parent as TitaDock;
        DeleteFromDock;
        case D.Align of
          alLeft, alRight:
          begin
            Step.X := Y;
            Step.Y := CapSize;
          end;
          alTop, alBottom:
          begin
            If Y < 0 Then
            begin
              Step.X := X;
              Step.Y := Height + Y;
            end
            else
            begin
              Step.X := X;
              Step.Y := CapSize;
            end;
          end;
        end;
      end;
      P := ClientToScreen(Point(X - Step.X, Y - Step.Y));
      FloatToPos(P.X, P.Y);
      If not Creator Then MouseCapture := true;
    end;
  end;
  Bisy := false;
end;

procedure TitaToolBar.MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
begin
  inherited
end;

procedure TitaToolBar.DoFloat;
begin
  If Assigned(FOnFloat) Then FOnFloat(Self);
end;

procedure TitaToolBar.DoStatic;
begin
  If Assigned(FOnStatic) Then FOnStatic(Self);
end;

function TitaToolBar.isFloat(var X,Y: Integer): boolean;
begin
  Result := false;
  X := Left;
  Y := Top;
  If (FStyle = psFloat) Then begin
    X := (Parent as TForm).Left + 1;
    Y := (Parent as TForm).Top + CapSize + 1;
    Result := true;
  end;
end;

procedure TitaToolBar.FloatToPos(X,Y: Integer);
var
  FP: TFPform97;
  V: Boolean;
begin
  If FStyle = psFloat Then
  begin
  // Allready float
    Parent.Left := X;
    Parent.Top := Y - CapSize;
    Exit;
  end;
  If not Creator Then DoFloat;
  FStyle := psFloat;
  V := Visible;
  FP := TFPform97.Create(nil);
  If FAllowResize Then FP.BorderStyle := bsSizeToolWin;
  If FCloseButton Then FP.BorderIcons := [biSystemMenu]
  else FP.BorderIcons := [];
  FP.Left := X;
  FP.Top := Y - CapSize;
  FP.ClientWidth := Width;
  FP.ClientHeight := Height;
  FP.Caption := Caption;
  FP.Panel := self;
  If V Then begin
    FP.Show;
    Hide;
  end;
  Left := 0; Top := 0;
  SetParent(FP);
  If V Then Show;
end;

procedure TitaToolBar.HidePanel;
begin
  If not Visible Then Exit;
  If Assigned(FOnHide) Then FOnHide(self);
  If FStyle = psFloat Then begin
    Parent.Hide;
    Hide;
    Exit;
  end;
  DeleteFromDock;
  Hide;
end;

procedure TitaToolBar.ShowPanel;
begin
  If Visible Then Exit;
  If Assigned(FOnShow) Then FOnShow(self);
  If FStyle = psFloat Then begin
    Show;
    Parent.Show;
    Exit;
  end;
  Show;
  InsertToDock;
end;

function TitaToolBar.FindToolBar(const Pos: TPoint): TitaToolBar;
var
  P: TPoint;
  R: TRect;
  i: integer;
begin
  Result := nil;
  P := Point(Pos.X + Left, Pos.Y + Top);
  for i := 0 to Parent.ControlCount - 1 do
  begin
    R := Parent.Controls[i].BoundsRect;
    InflateRect(R, -1, -1);
    If (Parent.Controls[i] is TitaToolBar) and
       (Parent.Controls[i] <> Self) Then
    begin
      If (Parent.Align in [alTop, alBottom]) and
         (((Top < R.Top) and (P.Y > (R.Top + R.Bottom) div 2)) or
          ((Top > R.Top) and (P.Y < (R.Top + R.Bottom) div 2)))
      Then
      begin
        Result := Parent.Controls[i] as TitaToolBar;
        Exit;
      end;
      If (Parent.Align in [alLeft, alRight]) and
         (((Left < R.Left) and (P.X > (R.Left + R.Right) div 2)) or
          ((Left > R.Left) and (P.X < (R.Left + R.Right) div 2)))
      Then
      begin
        Result := Parent.Controls[i] as TitaToolBar;
        Exit;
      end;
    end;
  end;
end;

function TitaToolBar.FindDock(const Pos: TPoint): TitaDock;
var
  i: Integer;
  R, R1, R2: TRect;
begin
  for i := 0 To Owner.ComponentCount-1 do
    If (Owner.Components[i] is TitaDock) then begin
      Result := Owner.Components[i] as TitaDock;
      // Self Rect
      If FStyle = psFloat Then
      begin
        IntersectRect(R1, BoundsRect, Parent.ClientRect);
        InflateRect(R1, -1, -1);
        R1.TopLeft := Parent.ClientToScreen(R1.TopLeft);
        R1.BottomRight := Parent.ClientToScreen(R1.BottomRight);
        If Result.Align in [alLeft, alRight] Then
        begin
          R1 := Rect(Pos.X-5, Pos.Y-5, Pos.X+5, Pos.Y+5);
          R1.TopLeft := ClientToScreen(R1.TopLeft);
          R1.BottomRight := ClientToScreen(R1.BottomRight);
        end;
      end
      else
      begin
        R1 := Rect(Pos.X-5, Pos.Y-5, Pos.X+5, Pos.Y+5);
        R1.TopLeft := ClientToScreen(R1.TopLeft);
        R1.BottomRight := ClientToScreen(R1.BottomRight);
      end;
      // Dock rect
      R2.TopLeft := Result.ClientToScreen(Point(0, 0));
      R2.BottomRight := Result.ClientToScreen(Point(Result.Width, Result.Height));
      If R2.Left = R2.Right Then
        InflateRect(R2, 1, -1);
      If R2.Top = R2.Bottom Then
        InflateRect(R2, -1, 1);
      case Result.Align of
        alTop, alBottom:
          If IntersectRect(R, R1, R2) Then Exit;
        alLeft, alRight:
          If IntersectRect(R, R1, R2) Then Exit;
      end;
    end;
  Result := nil;
end;

{ TSpeedButton }

var
  Pattern: TBitmap = nil;
  ButtonCount: Integer = 0;

procedure CreateBrushPattern;
var
  X, Y: Integer;
begin
  Pattern := TBitmap.Create;
  Pattern.Width := 8;
  Pattern.Height := 8;
  with Pattern.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := clBtnFace;
    FillRect(Rect(0, 0, Pattern.Width, Pattern.Height));
    for Y := 0 to 7 do
      for X := 0 to 7 do
        if (Y mod 2) = (X mod 2) then  { toggles between even/odd pixles }
          Pixels[X, Y] := clBtnHighlight;     { on even/odd rows }
  end;
end;

constructor TitaSpeedButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetBounds(0, 0, 22, 22);
  ControlStyle := [csCaptureMouse, csOpaque, csDoubleClicks];
  GlyphIndex := 0;
  ParentFont := True;
  FLayout := blGlyphLeft;
  Inc(ButtonCount);
  Flat := true;
  FEnableFlat := true;
end;

destructor TitaSpeedButton.Destroy;
begin
  Dec(ButtonCount);
  if ButtonCount = 0 then
  begin
    Pattern.Free;
    Pattern := nil;
  end;
  inherited Destroy;
end;

procedure TitaSpeedButton.SetGlyphs(Value: TitaGlyphs);
begin
  FImages := nil;
  FGlyphs := Value;
end;

procedure TitaSpeedButton.SetImages(Value: TImageList);
begin
  FGlyphs := nil;
  FImages := Value;
end;

procedure TitaSpeedButton.Notification(AComponent: TComponent;
      Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FGlyphs) then
    FGlyphs := nil;
  if (Operation = opRemove) and (AComponent = FImages) then
    FImages := nil;
end;

procedure TitaSpeedButton.Paint;
var
  PaintRect, R, R1: TRect;
  DrawFlags: Integer;
  B: TBitmap;
  Sz: String;
  i, j: integer;
  State: Integer;
begin
  if not Enabled and not (csDesigning in ComponentState) then
  begin
    FState := bsDisabled;
    FDragging := False;
  end
  else if FState = bsDisabled then
    if FDown and (GroupIndex <> 0) then
      FState := bsExclusive
    else
      FState := bsUp;

  Canvas.Font := Self.Font;

  PaintRect := Rect(0, 0, Width, Height);

  if not FFlat then
  begin
    DrawFlags := DFCS_BUTTONPUSH or DFCS_ADJUSTRECT;
    if FState in [bsDown, bsExclusive] then
      DrawFlags := DrawFlags or DFCS_PUSHED;
    DrawFrameControl(Canvas.Handle, PaintRect, DFC_BUTTON, DrawFlags);
  end
  else
  begin
    Canvas.Brush.Color := clBtnFace;
    if FMouseInControl then
    begin
      if FState in [bsDown, bsExclusive] then Canvas.Pen.Color := clBtnShadow
      else Canvas.Pen.Color := clBtnHighlight;
    end
    else Canvas.Pen.Color := clBtnFace;
    Canvas.Rectangle(PaintRect.Left, PaintRect.Top, PaintRect.Right,
      PaintRect.Bottom);
    if FMouseInControl then
    begin
      if FState in [bsDown, bsExclusive] then Canvas.Pen.Color := clBtnHighlight
      else Canvas.Pen.Color := clBtnShadow;
      Canvas.PolyLine([
        Point(PaintRect.Left, PaintRect.Bottom - 1),
        Point(PaintRect.Right - 1, PaintRect.Bottom - 1),
        Point(PaintRect.Right - 1, PaintRect.Top - 1)
        ]);
    end;
  end;

  if FState in [bsDown, bsExclusive] then
    OffsetRect(PaintRect, 1, 1);

  if (FState = bsExclusive) and (not FFlat or not FMouseInControl) then
  begin
    if Pattern = nil then CreateBrushPattern;
    Canvas.Brush.Bitmap := Pattern;
    Canvas.FillRect(PaintRect);
  end;

  B := nil;
  Sz := Caption;
  State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
  If (FGlyphs <> nil) or (FImages <> nil) Then
  begin
    If FGlyphs <> nil Then
      B := FGlyphs.GraphicCell[FGlyph];
    If FImages <> nil Then
    begin
      B := TBitmap.Create;
      FImages.GetBitmap(FGlyph, B);
    end;
    If B <> nil Then
    begin
      case FLayout of
        blGlyphLeft:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, -(R.Right-R.Left-B.Width) div 2, 0);
            If Caption <> '' Then
            begin
              InflateRect(R, Canvas.TextWidth(Caption) div 2, 0);
              R1 := R; R1.Left := R1.Left + B.Width + 4;
              State := DT_VCENTER or DT_LEFT or DT_SINGLELINE;
            end;
            R.Top := R.Top+(R.Bottom-R.Top-B.Height) div 2;
          end;
        blGlyphRight:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, -(R.Right-R.Left-B.Width) div 2, 0);
            If Caption <> '' Then
            begin
              InflateRect(R, Canvas.TextWidth(Caption) div 2, 0);
              R1 := R; R1.Right := R1.Right - B.Width - 4;
              State := DT_VCENTER or DT_Right or DT_SINGLELINE;
            end;
            R.Left := R.Right-B.Width;
            R.Top := R.Top+(R.Bottom-R.Top-B.Height) div 2;
          end;
        blGlyphTop:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, 0, -(R.Bottom-R.Top-B.Height) div 2);
            If Caption <> '' Then
            begin
              InflateRect(R, 0, Canvas.TextHeight(Caption) div 2);
              R1 := R; R1.Top := R1.Top + B.Height + 4;
              R1.Bottom := R1.Top + Canvas.TextHeight(Caption);
              State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
            end;
            R.Left := R.Left+(R.Right-R.Left-B.Width) div 2;
          end;
        blGlyphBottom:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, 0, -(R.Bottom-R.Top-B.Height) div 2);
            If Caption <> '' Then
            begin
              InflateRect(R, 0, Canvas.TextHeight(Caption) div 2);
              R1 := R; R1.Bottom := R1.Bottom - B.Height - 4;
              R1.Top := R1.Bottom - Canvas.TextHeight(Caption);
              State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
            end;
            R.Left := R.Left+(R.Right-R.Left-B.Width) div 2;
            R.Top := R.Bottom - B.Height;
          end;
      end;
      If Enabled Then
      begin
        If FGlyphs <> nil Then
          FGlyphs.Draw(Canvas, R.Left, R.Top, FGlyph);
        If FImages <> nil Then
          DrawBitmapTransparent(Canvas, R.Left, R.Top, B,
            B.Canvas.Pixels[0, B.Height-1]);
      end;
      If (not Enabled) Then
      begin
        for i := 0 to B.Width do
          for j := 0 to B.Height do
            If B.Canvas.Pixels[i, j] = clBlack Then
            begin
              Canvas.Pixels[R.Left+i+1, R.Top+j+1] := clBtnHighlight;
              Canvas.Pixels[R.Left+i, R.Top+j] := clBtnShadow;
            end;
      end;
      If Enabled Then
      begin
        Canvas.Brush.Style := bsClear;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State)
      end
      else
      begin
        Canvas.Brush.Style := bsClear;
        OffsetRect(R1, 1, 1);
        Canvas.Font.Color := clBtnHighlight;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State);
        OffsetRect(R1, -1, -1);
        Canvas.Font.Color := clBtnShadow;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State);
      end;
    end;
  end;

  If B = nil Then
    If Enabled Then
    begin
      Canvas.Brush.Style := bsClear;
      DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
        DT_VCENTER or DT_CENTER or DT_SINGLELINE)
    end
    else
    begin
      Canvas.Brush.Style := bsClear;
      OffsetRect(PaintRect, 1, 1);
      Canvas.Font.Color := clBtnHighlight;
      DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
        DT_VCENTER or DT_CENTER or DT_SINGLELINE);
      OffsetRect(PaintRect, -1, -1);
      Canvas.Font.Color := clBtnShadow;
      DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
        DT_VCENTER or DT_CENTER or DT_SINGLELINE);
    end;

  if FFlat and (FState in [bsDown, bsExclusive]) then
  begin
    Canvas.Pen.Color := clBtnShadow;
    Canvas.PolyLine([Point(0, Height), Point(0, 0), Point(Width, 0)]);
    Canvas.Pen.Color := clBtnHighlight;
    Canvas.PolyLine([Point(0, Height - 1), Point(Width - 1, Height - 1),
      Point(Width - 1, -1)]);
  end;

  if FFlat and (csDesigning in ComponentState) then
  begin
    Canvas.Pen.Color := clBtnShadow;
    Canvas.Pen.Style := psDot;
    Canvas.Brush.Style := bsClear;
    Canvas.Rectangle(ClientRect.Left, ClientRect.Top, ClientRect.Right,
      ClientRect.Bottom);
  end;

  If (B <> nil) and (FImages <> nil) Then B.Free;
end;

procedure TitaSpeedButton.UpdateTracking;
var
  P: TPoint;
begin
  if FFlat then
  begin
    GetCursorPos(P);
    FMouseInControl := Enabled and (FindDragTarget(P, True) = Self);
  end;
end;

procedure TitaSpeedButton.Loaded;
begin
  inherited Loaded;
end;

procedure TitaSpeedButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) and Enabled then
  begin
    if not FDown then
    begin
      FState := bsDown;
      Repaint;
    end;
    FDragging := True;
  end;
end;

procedure TitaSpeedButton.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  NewState: TButtonState;
begin
  inherited MouseMove(Shift, X, Y);
  if FDragging then
  begin
    if not FDown then NewState := bsUp
    else NewState := bsExclusive;
    if (X >= 0) and (X < ClientWidth) and (Y >= 0) and (Y <= ClientHeight) then
      if FDown then NewState := bsExclusive else NewState := bsDown;
    if NewState <> FState then
    begin
      FState := NewState;
      Repaint;
    end;
  end;
end;

procedure TitaSpeedButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  DoClick: Boolean;
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FDragging then
  begin
    FDragging := False;
    DoClick := (X >= 0) and (X < ClientWidth) and (Y >= 0) and (Y <= ClientHeight);
    if FGroupIndex = 0 then
    begin
      { Redraw face in-case mouse is captured }
      FState := bsUp;
      FMouseInControl := False;
      if not (FState in [bsExclusive, bsDown]) then Repaint;
    end
    else
      if DoClick then SetDown(not FDown)
      else
      begin
        if FDown then FState := bsExclusive;
        Repaint;
      end;
    if DoClick then Click;
    UpdateTracking;
    Invalidate;
  end;
end;

procedure TitaSpeedButton.Click;
begin
  inherited Click;
end;

function TitaSpeedButton.GetPalette: HPALETTE;
begin
  If (FGlyphs <> nil) and (FGlyph > 0) Then
    Result := FGlyphs.GraphicCell[FGlyph].Palette
  else
    Result := 0;
end;

function TitaSpeedButton.GetGlyph: Integer;
begin
  Result := FGlyph;
end;

procedure TitaSpeedButton.SetGlyph(Value: Integer);
begin
  FGlyph := Value;
  Invalidate;
end;

procedure TitaSpeedButton.UpdateExclusive;
var
  Msg: TMessage;
begin
  if (FGroupIndex <> 0) and (Parent <> nil) then
  begin
    Msg.Msg := CM_BUTTONPRESSED;
    Msg.WParam := FGroupIndex;
    Msg.LParam := Longint(Self);
    Msg.Result := 0;
    Parent.Broadcast(Msg);
  end;
end;

procedure TitaSpeedButton.SetDown(Value: Boolean);
begin
  if FGroupIndex = 0 then Value := False;
  if Value <> FDown then
  begin
    if FDown and (not FAllowAllUp) then Exit;
    FDown := Value;
    if Value then
    begin
      if FState = bsUp then Invalidate;
      FState := bsExclusive
    end
    else
    begin
      FState := bsUp;
      Invalidate;
    end;
    if Value then UpdateExclusive;
  end;
end;

procedure TitaSpeedButton.SetFlat(Value: Boolean);
begin
  if Value <> FFlat then
  begin
    FFlat := Value;
    Invalidate;
  end;
end;

procedure TitaSpeedButton.SetGroupIndex(Value: Integer);
begin
  if FGroupIndex <> Value then
  begin
    FGroupIndex := Value;
    UpdateExclusive;
  end;
end;

procedure TitaSpeedButton.SetLayout(Value: TButtonLayout);
begin
  if FLayout <> Value then
  begin
    FLayout := Value;
    Invalidate;
  end;
end;

procedure TitaSpeedButton.SetAllowAllUp(Value: Boolean);
begin
  if FAllowAllUp <> Value then
  begin
    FAllowAllUp := Value;
    UpdateExclusive;
  end;
end;

procedure TitaSpeedButton.WMLButtonDblClk(var Message: TWMLButtonDown);
begin
  inherited;
  if FDown then DblClick;
end;

procedure TitaSpeedButton.CMEnabledChanged(var Message: TMessage);
const
  NewState: array[Boolean] of TButtonState = (bsDisabled, bsUp);
begin
  UpdateTracking;
  Invalidate;
end;

procedure TitaSpeedButton.CMButtonPressed(var Message: TMessage);
var
  Sender: TitaSpeedButton;
begin
  if Message.WParam = FGroupIndex then
  begin
    Sender := TitaSpeedButton(Message.LParam);
    if Sender <> Self then
    begin
      if Sender.Down and FDown then
      begin
        FDown := False;
        FState := bsUp;
        Invalidate;
      end;
      FAllowAllUp := Sender.AllowAllUp;
    end;
  end;
end;

procedure TitaSpeedButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and Enabled then
    begin
      Click;
      Result := 1;
    end else
      inherited;
end;

procedure TitaSpeedButton.CMFontChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TitaSpeedButton.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TitaSpeedButton.CMSysColorChange(var Message: TMessage);
begin
  Invalidate;
end;

procedure TitaSpeedButton.CMMouseEnter(var Message: TMessage);
begin
  inherited;
//  if not ForegroundTask then Exit;
  if FFlat and (not FMouseInControl) and Enabled and FEnableFlat then
  begin
    If Assigned(FOnMouseEnter) Then FOnMouseEnter(Self);
    FMouseInControl := True;
    Invalidate;
  end;
end;

procedure TitaSpeedButton.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if FFlat and FMouseInControl and Enabled and FEnableFlat then
  begin
    If Assigned(FOnMouseLeave) Then FOnMouseLeave(Self);
    FMouseInControl := False;
    Invalidate;
  end;
end;

{ TitaTBSeparator }

constructor TitaTBSeparator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 8;
  Height := 8;
  Cursor := crHand;
end;

procedure TitaTBSeparator.SetParent(AParent: TWinControl);
begin
  inherited SetParent(AParent);
end;

procedure TitaTBSeparator.Paint;
begin
  with Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := clBtnFace;
    Pen.Style := psClear;
    Rectangle(0, 0, Width, Height);
    Pen.Style := psSolid;
    case Align of
      alLeft, alRight:
      begin
        Pen.Color := clBtnShadow;
        MoveTo(Width div 2-1, 0);
        LineTo(Width div 2-1, Height);
        Pen.Color := clBtnHighlight;
        MoveTo(Width div 2, 0);
        LineTo(Width div 2, Height);
      end;
      alTop, alBottom:
      begin
        Pen.Color := clBtnShadow;
        MoveTo(0, Height div 2-1);
        LineTo(Width, Height div 2-1);
        Pen.Color := clBtnHighlight;
        MoveTo(0, Height div 2);
        LineTo(Width, Height div 2);
      end;
    end;
    If csDesigning in ComponentState Then
    begin
      Brush.Style := bsClear;
      Pen.Style := psDot;
      Pen.Color := clBtnShadow;
      Rectangle(0, 0, Width-1, Height-1);
    end;
  end;
end;

procedure TitaTBSeparator.MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseDown(Button, Shift, P.X, P.Y);
  end;
end;

procedure TitaTBSeparator.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseMove(Shift, P.X, P.Y);
  end;
end;

procedure TitaTBSeparator.MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseUp(Button, Shift, P.X, P.Y);
  end;
end;

{ TitaTBDragBox }

constructor TitaTBDragBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 12;
  Height := 12;
  Cursor := crHand;
end;

procedure TitaTBDragBox.SetParent(AParent: TWinControl);
begin
  inherited SetParent(AParent);
end;

procedure TitaTBDragBox.Paint;
const
  Step = 4;
begin
  with Canvas do
  begin
    Brush.Color := clBtnFace;
    Pen.Style := psClear;
    Rectangle(0, 0, Width, Height);
    Pen.Style := psSolid;
    case Align of
      alLeft, alRight:
      begin
        // First
        Pen.Color := clBtnHighlight;
        MoveTo(Step - 1, 0);
        LineTo(Step - 2, 0);
        LineTo(Step - 2, Height);
        Pen.Color := clBtnShadow;
        MoveTo(Step - 1, Height - 1);
        LineTo(Step, Height - 1);
        LineTo(Step, -1);
        // Second
        Pen.Color := clBtnHighlight;
        MoveTo(Step + 2, 0);
        LineTo(Step + 1, 0);
        LineTo(Step + 1, Height);
        Pen.Color := clBtnShadow;
        MoveTo(Step + 2, Height - 1);
        LineTo(Step + 3, Height - 1);
        LineTo(Step + 3, -1);
      end;
      alTop, alBottom:
      begin
        // First
        Pen.Color := clBtnHighlight;
        MoveTo(0, Step - 1);
        LineTo(0, Step - 2);
        LineTo(Width, Step - 2);
        Pen.Color := clBtnShadow;
        MoveTo(Width - 1, Step - 1);
        LineTo(Width - 1, Step);
        LineTo(-1, Step);
        // Second
        Pen.Color := clBtnHighlight;
        MoveTo(0, Step + 2);
        LineTo(0, Step + 1);
        LineTo(Width, Step + 1);
        Pen.Color := clBtnShadow;
        MoveTo(Width - 1, Step + 2);
        LineTo(Width - 1, Step + 3);
        LineTo(-1, Step + 3);
      end;
    end;
  end;
end;

procedure TitaTBDragBox.MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseDown(Button, Shift, P.X, P.Y);
  end;
end;

procedure TitaTBDragBox.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseMove(Shift, P.X, P.Y);
  end;
end;

procedure TitaTBDragBox.MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer);
var
  P: TPoint;
begin
  If Parent is TitaToolBar Then
  begin
    P := ClientToScreen(Point(X, Y));
    P := Parent.ScreenToCLient(P);
    (Parent as TitaToolBar).PublicMouseUp(Button, Shift, P.X, P.Y);
  end;
end;

{ TitaPopupButton }

constructor TitaPopupButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Align := alNone;
  Randomize;
  GroupIndex := Random(10000);
  AllowAllUp := true;
end;

destructor TitaPopupButton.Destroy;
begin
  inherited Destroy;
end;

procedure TitaPopupButton.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FMenu) then
    FMenu := nil;
end;

procedure TitaPopupButton.Click;
var
  P: TPoint;
begin
  inherited Click;
  If Assigned(FMenu) Then Begin
    P := ClientToScreen(Point(0, Height));

    FMenu.Popup(P.X, P.Y);
  end;
end;

procedure TitaPopupButton.CMMouseEnter(var Message: TMessage);
begin
  inherited;
end;

procedure TitaPopupButton.CMMouseLeave(var Message: TMessage);
begin
  inherited;
end;

{ Tool Button classes }

{ TitaTBButton }

constructor TitaTBButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Align := alLeft;
end;

{ TitaTBPopup }

constructor TitaTBPopup.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Align := alLeft;
end;

{ Menu routings =====================================================}

const
  PopupButtonHookActive: Integer = 0;
  ActiveForm: TForm = nil;
  isCan: boolean = false;
  Pushed: boolean = false;
  ActivePushed: TitaMenuButton = nil;
  CurrentPushed: TitaMenuButton = nil; // Menu button
  OldActivePushed: TitaMenuButton = nil;

var
  MouseHook: HHOOK;
  KeyboardHook: HHOOK;
  Buzy: boolean = false;
  KeyBuzy: boolean = false;


function MouseHookProc(nCode: Integer; Msg: Longint; var D: TMouseHookStruct): Longint; stdcall;
var
  i: integer;
  R: TRect;
  P: TPoint;
  PB: TitaMenuButton;
begin
  If Buzy Then
  begin
    Result := CallNextHookEx(0, nCode, Msg, Longint(@D));
    Exit;
  end;
  Buzy := true;
  Result := CallNextHookEx(MouseHook, nCode, Msg, Longint(@D));
  If ActiveForm = nil Then
  begin
    Buzy := false;
    Exit;
  end;
  with ActiveForm do
  begin
    If (Pushed) and (Msg = WM_MOUSEMOVE) Then
    begin
      for i := 0 to ComponentCount-1 do
        If Components[i] is TitaMenuButton Then
        begin
          PB := (Components[i] as TitaMenuButton);
          If PB = ActivePushed Then Continue;
          If not PB.Visible Then Continue;
          R.TopLeft := PB.ClientToScreen(PB.ClientRect.TopLeft);
          R.BottomRight := PB.ClientToScreen(PB.ClientRect.BottomRight);
          P := D.pt;
          If PointInRect(P, R) Then
          begin
            ActivePushed.FMouseInControl := false;
            ActivePushed.Down := false;
            Buzy := false;
            isCan := false;
            SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
            Exit;
          end;
        end;
      // MDI Merging Menus
      If ActiveForm.FormStyle = fsMDIForm Then
        If ActiveForm.ActiveMDIChild <> nil Then
          with ActiveForm.ActiveMDIChild do
            for i := 0 to ComponentCount-1 do
              If Components[i] is TitaMenuButton Then
              begin
                PB := (Components[i] as TitaMenuButton);
                If PB = ActivePushed Then Continue;
                If not PB.Visible Then Continue;
                R.TopLeft := PB.ClientToScreen(PB.ClientRect.TopLeft);
                R.BottomRight := PB.ClientToScreen(PB.ClientRect.BottomRight);
                P := D.pt;
                If PointInRect(P, R) Then
                begin
                  ActivePushed.FMouseInControl := false;
                  ActivePushed.Down := false;
                  Buzy := false;
                  isCan := false;
                  SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
                  Exit;
                end;
              end;
    end;
  end;
  Buzy := false;
end;

function KeyboardHookProc(Code, wParam, lParam: Integer): Integer; export; stdcall;
const
  kbd_Transition = 1 shl 31; // transition code = 0 if keydown, = 1 if key up
  kbd_AltDown = 1 shl 29;    // alt pressed
  // directions for GetNextMenuButton98
  drLeft  = -1;
  drRight = 1;

  function GetNextMenuButton(const Control: TControl; const Direction: Integer): TitaMenuButton;
  var
    i: integer;
    P: TControl;
  begin
    with Control.Parent as TitaToolBar do
    begin
      i := 0;
      while Ctrls[i] <> Control do
        Inc(i);
      P := nil;
      while P = P do
      begin
        If Direction = drLeft Then
          If i > 0 Then Dec(i) else i := ControlCount;
        If Direction = drRight Then
          If i < ControlCount Then Inc(i) else i := 0;
        P := Ctrls[i];
        If (P is TitaMenuButton) Then Break;
      end;
      Result := P as TitaMenuButton;
    end;
  end;

  function GetFirstMenuButton: TitaMenuButton;
  var
    i: integer;
    P: TControl;
  begin
    with CurrentPushed.Parent as TitaToolBar do
    begin
      P := nil;
      i := 0;
      while P = P do
      begin
        If i < ControlCount Then Inc(i) else i := 0;
        P := Ctrls[i];
        If (P is TitaMenuButton) Then Break;
      end;
      Result := P as TitaMenuButton;
    end;
  end;

  function SearchShortcut: Integer;
  begin
    Result:= 0;
  end;

var
  ShiftState: TShiftState;
  i: integer;
  M: TMenuItem;
begin
  Result := CallNextHookEx(KeyboardHook, Code, wParam, lParam);
  if (Code >= 0) then
  begin
    // only OnKeyDown (not OnKeyUp)
    If Screen.ActiveForm <> CurrentPushed.Owner as TForm Then
      Exit;
    If not Pushed and not KeyBuzy Then
      with CurrentPushed.Owner do
      begin
        ShiftState := KeyDataToShiftState(lParam);
        for i := 0 to ComponentCount-1 do
          If Components[i] is TitaOfficePopupMenu Then
            with Components[i] as TitaOfficePopupMenu do
            begin
              M := FindItem(ShortCut(wParam, ShiftState), fkShortCut);
              If M <> nil then
              begin
                KeyBuzy := true;
                Result := 0;
                M.Click;
                KeyBuzy := false;
                Exit;
              end;
            end;
      end;

    if (lParam and kbd_Transition <> 0) then
    begin
      KeyBuzy := false;
      Exit;
    end;

    If (ActivePushed <> nil) and (Pushed) and (ActivePushed.FMenu <> nil) Then
    begin
      // Hot Key in Menu
      M := (ActivePushed.FMenu as TitaOfficePopupMenu).ProcessAccel(wParam);
      If M <> nil Then
      begin
        ActivePushed.FMouseInControl := false;
        ActivePushed.Down := false;
        Pushed := false;
        isCan := false;
        SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
        M.Click;
        ActivePushed := nil;
        Result := 0;
        Exit;
      end;
    end;
    case wParam of
      vk_Escape:
        If ActivePushed <> nil Then
        begin
          // DeActive Menu
          ActivePushed.FMouseInControl := false;
          ActivePushed.Down := false;
          ActivePushed.Invalidate;
          isCan := false;
          SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
          ActivePushed := nil;
          Pushed := false;
          Result := 1;
        end;
      vk_Return:
        begin
        end;
      vk_Menu, vk_F10:
        If ActivePushed <> nil Then
        begin
          // DeActive Menu
          ActivePushed.FMouseInControl := false;
          ActivePushed.Down := false;
          ActivePushed.Invalidate;
          isCan := false;
          SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
          Result := 1;
          ActivePushed := nil;
          Pushed := false;
        end
        else
        begin
          ActivePushed := GetFirstMenuButton;
          ActivePushed.FMouseInControl := true;
          ActivePushed.Invalidate;
          Result := 1;
        end;
      vk_Up:
        begin
        end;
      vk_Down:
        If (ActivePushed <> nil) and (ActivePushed.Down = false) Then
        begin
          // Click Menu
          ActivePushed.Click;
          Result := 1;
        end;
      vk_Left:
        If (not KeyBuzy) and (ActivePushed <> nil) Then
        begin
          CurrentPushed := GetNextMenuButton(ActivePushed, drLeft);
          If (ActivePushed <> nil) and (Pushed) Then
          begin
            ActivePushed.FMouseInControl := false;
            ActivePushed.Down := false;
            isCan := false;
            SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
          end;
          KeyBuzy := true;
          If Pushed Then
          begin
            ActivePushed := CurrentPushed;
            ActivePushed.Click
          end
          else
          begin
            ActivePushed.FMouseInControl := false;
            ActivePushed.Invalidate;
            ActivePushed := CurrentPushed;
            ActivePushed.FMouseInControl := true;
            ActivePushed.Invalidate;
          end;
          Result := 1;
          KeyBuzy := false;
        end;
      vk_Right:
        If (not KeyBuzy) and (ActivePushed <> nil) Then
        begin
          CurrentPushed := GetNextMenuButton(ActivePushed, drRight);
          If (ActivePushed <> nil) and (Pushed) Then
          begin
            ActivePushed.FMouseInControl := false;
            ActivePushed.Down := false;
            isCan := false;
            SendMessage(HWND_BROADCAST, WM_CANCELMODE, 0, 0);
          end;
          KeyBuzy := true;
          If Pushed Then
          begin
            ActivePushed := CurrentPushed;
            ActivePushed.Click
          end
          else
          begin
            ActivePushed.FMouseInControl := false;
            ActivePushed.Invalidate;
            ActivePushed := CurrentPushed;
            ActivePushed.FMouseInControl := true;
            ActivePushed.Invalidate;
          end;
          Result := 1;
          KeyBuzy := false;
        end;
    end;
  end
  else begin
    // support for system menu
    if (wParam = vk_Space)
    and (lParam and kbd_AltDown <> 0)
    then
    begin
      PostMessage(GetActiveWindow, wm_NCLButtonDown, htSysMenu, 0);
      Result:= 1;
      Exit;
    end;
    // no one of menus is popupped
  end;
end;

{ TitaMenuButton }

const
  WndProcCount: Integer = 0;

constructor TitaMenuButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  If not(csDesigning in ComponentState) and (PopupButtonHookActive = 0) Then
  begin
    MouseHook := SetWindowsHookEx(WH_MOUSE, @MouseHookProc, HInstance, 0);
    KeyboardHook := SetWindowsHookEx(WH_KEYBOARD, @KeyboardHookProc, HInstance, 0);
  end;
  Inc(PopupButtonHookActive);
  CurrentPushed := Self;
  AllowAllUp := true;
  GroupIndex := random(10000);
end;

procedure TitaMenuButton.Loaded;
begin
  inherited Loaded;
  If (FMenu = nil) or (FMenu.WindowHandle = 0) Then Exit;
  If not(csDesigning in ComponentState) and (WndProcCount = 0) Then
  begin
    // Override WNd Proc
    FObjectInst := MakeObjectInstance(MyWndProc);
    FPrevWndProc := Pointer(GetWindowLong(FMenu.WindowHandle,
      GWL_WNDPROC));
    SetWindowLong(FMenu.WindowHandle, GWL_WNDPROC,
      LongInt(FObjectInst));
  end;
  Inc(WndProcCount);
end;

procedure TitaMenuButton.Notification(AComponent: TComponent;
      Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FMenu) then
    FMenu := nil;
end;

destructor TitaMenuButton.Destroy;
begin
  Dec(PopupbuttonHookActive);
  If not(csDesigning in ComponentState) and (PopupButtonHookActive = 0) Then
  begin
    UnhookWindowsHookEx(KeyboardHook);
    UnhookWindowsHookEx(MouseHook);
  end;

  Dec(WndProcCount);
  If not(csDesigning in ComponentState) and (WndProcCount = 0) Then
  begin
    // Destroy
    SetWindowLong(FMenu.WindowHandle, GWL_WNDPROC,
      LongInt(FPrevWndProc));
    FreeObjectInstance(FObjectInst);
    // ...
  end;
  inherited Destroy;
end;

procedure TitaMenuButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and Enabled and Pushed then
    begin
      Click;
      Result := 1;
    end else
      If not Pushed Then inherited;
end;

procedure TitaMenuButton.Click;
var
  P: TPoint;
begin
  If Assigned(FMenu) Then
    If OldActivePushed = Self Then
    begin
      OldActivePushed := nil;
      ActivePushed := nil;
      ActiveForm := nil;
      FMouseInControl := false;
      Down := false;
      Pushed := false;
      isCan := false;
    end
    else
    Begin
      If Align in [alLeft, alRight] Then
        P := ClientToScreen(Point(0, Height))
      else
        P := ClientToScreen(Point(Width, 0));

      If Owner is TForm Then
        If ((Owner as TForm).FormStyle = fsMDIChild) Then
          // MDI Application
          ActiveForm := (Owner.Owner as TForm)
        else
          ActiveForm := (Owner as TForm)
      else
        ActiveForm := Screen.ActiveForm;

      FMouseInControl := true;
      Down := true;
      Pushed := true;
      ActivePushed := Self;
      OldActivePushed := ActivePushed;

      FMenu.Popup(P.X, P.Y);

      FMouseInControl := false;
      Down := false;
    end;
end;

procedure TitaMenuButton.CMMouseEnter(var Message: TMessage);
var
  P: TPoint;
begin
  inherited CMMouseEnter(Message);
  If (Pushed) and Assigned(FMenu) and (ActivePushed <> Self) and
     isCan Then
  begin
    isCan := false;
    If Align in [alLeft, alRight] Then
      P := ClientToScreen(Point(0, Height))
    else
      P := ClientToScreen(Point(Width, 0));


    If Owner is TForm Then
      If ((Owner as TForm).FormStyle = fsMDIChild) Then
        // MDI Application
        ActiveForm := (Owner.Owner as TForm)
      else
        ActiveForm := (Owner as TForm)
    else
      ActiveForm := Screen.ActiveForm;

    FMouseInControl := true;
    Down := true;
    Pushed := true;
    ActivePushed := Self;
    OldActivePushed := ActivePushed;

    FMenu.Popup(P.X, P.Y);
  end;
end;

procedure TitaMenuButton.CMMouseLeave(var Message: TMessage);
begin
  inherited CMMouseLeave(Message);
end;

procedure TitaMenuButton.MyWndProc(var Msg: TMessage);
begin
  with Msg do
  begin
    case Msg of
      WM_COMMAND:
      begin
        If ActivePushed <> nil Then
        begin
          ActivePushed.FMouseInControl := false;
          ActivePushed.Down := false;
        end;
        ActiveForm := nil;
        ActivePushed := nil;
        Pushed := false;
      end;
      WM_EXITMENULOOP:
      If not isCan Then
      begin
        If ActivePushed <> nil Then
        begin
          ActivePushed.FMouseInControl := false;
          ActivePushed.Down := false;
        end;
        ActiveForm := nil;
        ActivePushed := nil;
        Pushed := false;
      end;
      WM_CANCELMODE:
      begin
        isCan := true;
        OldActivePushed := nil;
      end;
    end;
    Result := CallWindowProc(FPrevWndProc, FMenu.WindowHandle,
      Msg, WParam, LParam);
  end;
end;

procedure TitaMenuButton.Paint;
var
  PaintRect, R, R1: TRect;
  DrawFlags: Integer;
  B: TBitmap;
  Sz: String;
  i, j: integer;
  State: Integer;
begin
  if not Enabled and not (csDesigning in ComponentState) then
  begin
    FState := bsDisabled;
    FDragging := False;
  end
  else if FState = bsDisabled then
    if FDown and (GroupIndex <> 0) then
      FState := bsExclusive
    else
      FState := bsUp;

  Canvas.Font := Self.Font;

  PaintRect := Rect(0, 0, Width, Height);

  if not FFlat then
  begin
    DrawFlags := DFCS_BUTTONPUSH or DFCS_ADJUSTRECT;
    if FState in [bsDown, bsExclusive] then
      DrawFlags := DrawFlags or DFCS_PUSHED;
    DrawFrameControl(Canvas.Handle, PaintRect, DFC_BUTTON, DrawFlags);
  end
  else
  begin
    Canvas.Brush.Color := clBtnFace;
    if FMouseInControl then
    begin
      if FState in [bsDown, bsExclusive] then Canvas.Pen.Color := clBtnShadow
      else Canvas.Pen.Color := clBtnHighlight;
    end
    else Canvas.Pen.Color := clBtnFace;
    Canvas.Rectangle(PaintRect.Left, PaintRect.Top, PaintRect.Right,
      PaintRect.Bottom);
    if FMouseInControl then
    begin
      if FState in [bsDown, bsExclusive] then Canvas.Pen.Color := clBtnHighlight
      else Canvas.Pen.Color := clBtnShadow;
      Canvas.PolyLine([
        Point(PaintRect.Left, PaintRect.Bottom - 1),
        Point(PaintRect.Right - 1, PaintRect.Bottom - 1),
        Point(PaintRect.Right - 1, PaintRect.Top - 1)
        ]);
    end;
  end;

  if FState in [bsDown, bsExclusive] then
    OffsetRect(PaintRect, 1, 1);

  if (FState = bsExclusive) and (not FFlat or not FMouseInControl) then
  begin
    if Pattern = nil then CreateBrushPattern;
    Canvas.Brush.Bitmap := Pattern;
    Canvas.FillRect(PaintRect);
  end;

  B := nil;
  Sz := Caption;
  State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
  If FGlyphs <> nil Then
  begin
    B := FGlyphs.GraphicCell[FGlyph];
    If B <> nil Then
    begin
      case FLayout of
        blGlyphLeft:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, -(R.Right-R.Left-B.Width) div 2, 0);
            If Caption <> '' Then
            begin
              InflateRect(R, Canvas.TextWidth(Caption) div 2, 0);
              R1 := R; R1.Left := R1.Left + B.Width + 4;
              State := DT_VCENTER or DT_LEFT or DT_SINGLELINE;
            end;
            R.Top := R.Top+(R.Bottom-R.Top-B.Height) div 2;
          end;
        blGlyphRight:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, -(R.Right-R.Left-B.Width) div 2, 0);
            If Caption <> '' Then
            begin
              InflateRect(R, Canvas.TextWidth(Caption) div 2, 0);
              R1 := R; R1.Right := R1.Right - B.Width - 4;
              State := DT_VCENTER or DT_Right or DT_SINGLELINE;
            end;
            R.Left := R.Right-B.Width;
            R.Top := R.Top+(R.Bottom-R.Top-B.Height) div 2;
          end;
        blGlyphTop:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, 0, -(R.Bottom-R.Top-B.Height) div 2);
            If Caption <> '' Then
            begin
              InflateRect(R, 0, Canvas.TextHeight(Caption) div 2);
              R1 := R; R1.Top := R1.Top + B.Height + 4;
              R1.Bottom := R1.Top + Canvas.TextHeight(Caption);
              State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
            end;
            R.Left := R.Left+(R.Right-R.Left-B.Width) div 2;
          end;
        blGlyphBottom:
          begin
            R := PaintRect;
            InflateRect(R, -2, -2);
            InflateRect(R, 0, -(R.Bottom-R.Top-B.Height) div 2);
            If Caption <> '' Then
            begin
              InflateRect(R, 0, Canvas.TextHeight(Caption) div 2);
              R1 := R; R1.Bottom := R1.Bottom - B.Height - 4;
              R1.Top := R1.Bottom - Canvas.TextHeight(Caption);
              State := DT_VCENTER or DT_CENTER or DT_SINGLELINE;
            end;
            R.Left := R.Left+(R.Right-R.Left-B.Width) div 2;
            R.Top := R.Bottom - B.Height;
          end;
      end;
      If Enabled Then
        FGlyphs.Draw(Canvas, R.Left,
          R.Top, FGlyph);
      If (not Enabled) Then
      begin
        for i := 0 to B.Width do
          for j := 0 to B.Height do
            If B.Canvas.Pixels[i, j] = clBlack Then
            begin
              Canvas.Pixels[R.Left+i+1, R.Top+j+1] := clBtnHighlight;
              Canvas.Pixels[R.Left+i, R.Top+j] := clBtnShadow;
            end;
      end;
      If Enabled Then
      begin
        Canvas.Brush.Style := bsClear;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State)
      end
      else
      begin
        Canvas.Brush.Style := bsClear;
        OffsetRect(R1, 1, 1);
        Canvas.Font.Color := clBtnHighlight;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State);
        OffsetRect(R1, -1, -1);
        Canvas.Font.Color := clBtnShadow;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R1, State);
      end;
    end;
  end;

  If B = nil Then
    If Enabled Then
    begin
      If Align in [alTop, alBottom] Then
      begin
        B := TBitmap.Create;
        B.Width := Height;
        B.Height := Width;
        with B.Canvas do
        begin
          Brush.Color := clBtnFace;
          Brush.Style := bsSolid;
          Rectangle(-1, -1, Height+1, Width+1);
          Brush.Style := bsClear;
          Font.Color := clBlack;
          R1 := Rect(0, 0, Height, Width);
          DrawText(Handle, PChar(Sz), Length(Sz), R1,
            DT_VCENTER or DT_CENTER or DT_SINGLELINE);
          for i := 1 to B.Width do
            for j := 1 to B.Height do
              If Pixels[i, j] = clBlack Then
                Self.Canvas.Pixels[j, Height-i] := clBlack;
        end;
      end
      else
      begin
        Canvas.Brush.Style := bsClear;
        DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
          DT_VCENTER or DT_CENTER or DT_SINGLELINE)
      end;
    end
    else
    begin
      Canvas.Brush.Style := bsClear;
      OffsetRect(PaintRect, 1, 1);
      Canvas.Font.Color := clBtnHighlight;
      DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
        DT_VCENTER or DT_CENTER or DT_SINGLELINE);
      OffsetRect(PaintRect, -1, -1);
      Canvas.Font.Color := clBtnShadow;
      DrawText(Canvas.Handle, PChar(Sz), Length(Sz), PaintRect,
        DT_VCENTER or DT_CENTER or DT_SINGLELINE);
    end;

  if FFlat and (FState in [bsDown, bsExclusive]) then
  begin
    Canvas.Pen.Color := clBtnShadow;
    Canvas.PolyLine([Point(0, Height), Point(0, 0), Point(Width, 0)]);
    Canvas.Pen.Color := clBtnHighlight;
    Canvas.PolyLine([Point(0, Height - 1), Point(Width - 1, Height - 1),
      Point(Width - 1, -1)]);
  end;

  if FFlat and (csDesigning in ComponentState) then
  begin
    Canvas.Pen.Color := clBtnShadow;
    Canvas.Pen.Style := psDot;
    Canvas.Brush.Style := bsClear;
    Canvas.Rectangle(ClientRect.Left, ClientRect.Top, ClientRect.Right,
      ClientRect.Bottom);
  end;
end;

{ TitaOfficePopupMenu }

{ TitaMenuBanner }

constructor TitaMenuBanner.Create;
begin
  FBackColor := 0;
  FPicture := TPicture.Create;
end;

destructor TitaMenuBanner.Destroy;
begin
  FPicture.Free;
end;

procedure TitaMenuBanner.Assign(Source: TPersistent);
begin
  if Source is TitaMenuBanner then
  begin
    FBackColor := (Source as TitaMenuBanner).BackColor;
    FPosition := (Source as TitaMenuBanner).Position;
    FPicturePosition := (Source as TitaMenuBanner).PicturePosition;
    FPicture.Assign((Source as TitaMenuBanner).Picture);
  end;
  inherited Assign(Source);
end;

procedure TitaMenuBanner.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

{ TitaOfficePopupMenu }

constructor TitaOfficePopupMenu.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Style := msOwnerDraw;
  FBanner := TitaMenuBanner.Create;
end;

procedure TitaOfficePopupMenu.Loaded;
begin
  inherited Loaded;
end;

destructor TitaOfficePopupMenu.Destroy;
begin
  FBanner.Free;
  inherited Destroy;
end;

procedure TitaOfficePopupMenu.CMDialogChar(var Message: TCMDialogChar);
var
  i: integer;
begin
  with Message do
    for i := 0 to Items.Count-1 do
      if IsAccel(CharCode, Items[i].Caption) then
      begin
        Items[i].Click;
        Result := 1;
      end;
  inherited;
end;

function TitaOfficePopupMenu.ProcessAccel(Key: Word): TMenuItem;

 function FindAccelerator(MenuItems: TMenuItem): TMenuItem;
  var i: Integer;

  begin
    Result:= nil;
    for i:= 0 to MenuItems.Count - 1 do begin
      if ( Result = nil ) and (MenuItems[i].Enabled) then begin
        if (MenuItems[i].Count > 0) then
          Result:= FindAccelerator(MenuItems[i])
        else
          if ( IsAccel(Key, MenuItems[i].Caption) ) then
            Result:= MenuItems[i];
      end;
    end;
  end;
begin
  Result:= nil;
  if ( csDesigning in ComponentState ) then Exit;
  Result := FindAccelerator(Items);
end;

procedure TitaOfficePopupMenu.SetGlyphs(Value: TitaGlyphs);
begin
  FImages := nil;
  FGlyphs := Value;
end;

procedure TitaOfficePopupMenu.SetImages(Value: TImageList);
begin
  FGlyphs := nil;
  FImages := Value;
end;

procedure TitaOfficePopupMenu.Popup(X, Y: integer);
begin
  FHeight := 0;
  inherited Popup(X, Y);
end;

procedure TitaOfficePopupMenu.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FGlyphs) then
    FGlyphs := nil;
  if (Operation = opRemove) and (AComponent = FImages) then
    FImages := nil;
end;

const
  ImageSize: integer = 20;

procedure TitaOfficePopupMenu.DoDrawItem(MenuItem: TMenuItem; Canvas: TCanvas;
      ItemState: Cardinal; ItemRect: TRect);
var
  Sz: String;
  R: TRect;
  B: Tbitmap;
  Pos, i, j: integer;
begin
  // Draw Banner {!!!}
  If FBanner.Picture.Width > 0 Then
  begin
    Canvas.Brush.Color := FBanner.BackColor;
    case FBanner.PicturePosition of
      ppTop: Pos := 0;
      ppBottom: Pos := FHeight - FBanner.Picture.Height;
    else
      Pos := (FHeight - FBanner.Picture.Height) div 2;
    end;
    case FBanner.Position of
      mbpLeft:
      begin
        Canvas.Rectangle(ItemRect.Left, ItemRect.Top,
          ItemRect.Left+FBanner.Picture.Width+1, ItemRect.Bottom+1);
        Canvas.Draw(ItemRect.Left, Pos, FBanner.Picture.Graphic);
        Inc(ItemRect.Left, FBanner.Picture.Width);
      end;
      mbpRight:
      begin
        Canvas.Rectangle(ItemRect.Right-FBanner.Picture.Width,
          ItemRect.Top, ItemRect.Right+1, ItemRect.Bottom+1);
        Canvas.Draw(ItemRect.Right-FBanner.Picture.Width,
          Pos, FBanner.Picture.Graphic);
        Dec(ItemRect.Right, FBanner.Picture.Width);
      end;
    end;
  end;
  // Draw separate
  If MenuItem.Caption[1] = '-' Then
  begin
    With Canvas,ItemRect do
    begin
      Pen.Style := psSolid;
      Pen.Color := clBtnShadow;
      MoveTo(Left+1, Top+(Bottom-Top) div 2);
      LineTo(Right-1, Top+(Bottom-Top) div 2);
      Pen.Color := clBtnHighlight;
      MoveTo(Left+1, Top+(Bottom-Top) div 2+1);
      LineTo(Right-1, Top+(Bottom-Top) div 2+1);
      If Length(MenuItem.Caption) > 1 then
      begin
        // Text in Separator
        R := ItemRect;
        Brush.Color := clBtnFace;
        Font.Color := clMenuText;
        Sz := Copy(MenuItem.Caption, 2, Length(MenuItem.Caption));
        TextOut(Left+(Right-Left-TextWidth(Sz)) div 2,
          Top+(Bottom-Top-TextHeight(Sz)) div 2, Sz);
      end;
    end;
    Exit;
  end;
  // Draw item rect
  If ItemState and ODS_SELECTED <> 0 Then
    If ItemState and ODS_Grayed = 0 Then
      Canvas.Brush.Color := clHighlight
    else
      Canvas.Brush.Color := clbtnFace
  else
    Canvas.Brush.Color := clBtnFace;
  Canvas.Rectangle(ItemRect.Left + ImageSize + 2, ItemRect.Top,
      ItemRect.Right+1, ItemRect.bottom+1);
  // Draw images
  If (ItemState and ODS_SELECTED <> 0) and
     (ItemState and ODS_CHECKED = 0) and
     (ItemState and ODS_GRAYED = 0) and
     ((FGlyphs <> nil) or (FImages <> nil)) and (MenuItem.Tag > 0) Then
  begin
    Canvas.Pen.Style := psSolid;
    Canvas.Pen.Color := clBtnHighlight;
    Canvas.MoveTo(ItemRect.Left, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left, ItemRect.Top);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top);
    Canvas.Pen.Color := clBtnShadow;
    Canvas.MoveTo(ItemRect.Left, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top);
  end
  else
    Canvas.Rectangle(ItemRect.Left, ItemRect.Top,
      ItemRect.Left+ImageSize+3, ItemRect.bottom+1);
  B := nil;
  // Draw checked
  If ItemState and ODS_CHECKED <> 0 Then
  begin
    If ItemState and ODS_SELECTED = 0 Then
    begin
      If Pattern = nil Then CreateBrushPattern;
      Canvas.Brush.Bitmap := Pattern;
    end
    else
    begin
      Canvas.Brush.Style := bsSolid;
      Canvas.Brush.Color := clMenu;
    end;
    R := ItemRect;
    R.Right := R.Left + ImageSize + 1;
    InflateRect(R, -1, -1);
    Canvas.FillRect(R);
    Canvas.Pen.Style := psSolid;
    Canvas.Pen.Color := clBtnShadow;
    Canvas.MoveTo(ItemRect.Left, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left, ItemRect.Top);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top);
    Canvas.Pen.Color := clBtnHighlight;
    Canvas.MoveTo(ItemRect.Left, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top+ImageSize);
    Canvas.LineTo(ItemRect.Left+ImageSize, ItemRect.Top);
    If ItemState and ODS_SELECTED <> 0 Then
      Canvas.Brush.Color := clHighlight
    else
      Canvas.Brush.Color := clMenu;
    If not MenuItem.RadioItem Then
      for i := 0 to bmpcheck.width do
        for j := 0 to bmpcheck.Height do
          If (bmpcheck.Canvas.Pixels[i, j] = clBlack) or
             (bmpcheck.Canvas.Pixels[i, j] = clWhite) Then
            Canvas.Pixels[itemRect.Left+(ImageSize-bmpCheck.Width) div 2+1+i,
              itemRect.Top+(ImageSize-bmpCheck.Height) div 2+2+j]
              := bmpcheck.Canvas.Pixels[i, j];
  end
  else
    If ((FGlyphs <> nil) or (FImages <> nil)) and (MenuItem.Tag > 0) Then
    begin
      If FGlyphs <> nil Then
        B := FGlyphs.GraphicCell[MenuItem.Tag-1];
      If FImages <> nil Then
      begin
        B := TBitmap.Create;
        FImages.GetBitmap(MenuItem.Tag-1, B);
      end;
      If (ItemState and ODS_GRAYED = 0) Then
      begin
        If FGlyphs <> nil Then
          FGlyphs.DrawCenter(Canvas, Rect(ItemRect.Left+1,
            ItemRect.Top+1, ItemRect.Left+ImageSize,
            ItemRect.Top+ImageSize), MenuItem.Tag-1);
        If FImages <> nil Then
          DrawBitmapTransparent(Canvas, ItemRect.Left+1, ItemRect.Top+1,
            B, B.Canvas.Pixels[0, B.Height-1]);
      end;
      If (ItemState and ODS_GRAYED <> 0) Then
      begin
        InflateRect(ItemRect, -(ImageSize - FGlyphs.Width) div 2,
          -(ImageSize - FGlyphs.Height) div 2);
        for i := 0 to B.Width do
          for j := 0 to B.Height do
            If B.Canvas.Pixels[i, j] = clBlack Then
            begin
              Canvas.Pixels[ItemRect.Left+i, itemRect.Top+j] := clBtnHighlight;
              Canvas.Pixels[ItemRect.Left-1+i, itemRect.Top-1+j] := clBtnShadow;
            end;
        InflateRect(ItemRect, (ImageSize - FGlyphs.Width) div 2,
          (ImageSize - FGlyphs.Height) div 2);
      end;
    end;
  If MenuItem.RadioItem Then
  begin
    If (FGlyphs <> nil) and (MenuItem.Tag > 0) Then
    begin
      If (ItemState and ODS_GRAYED = 0) Then
        FGlyphs.DrawCenter(Canvas, Rect(ItemRect.Left+1,
          ItemRect.Top+1, ItemRect.Left+ImageSize,
          ItemRect.Top+ImageSize), MenuItem.Tag-1);
    end;
    If (FImages <> nil) and (MenuItem.Tag > 0) Then
    begin
      If (ItemState and ODS_GRAYED = 0) Then
        DrawBitmapTransparent(Canvas, ItemRect.Left+1, ItemRect.Top+1,
          B, B.Canvas.Pixels[0, B.Height-1]);
    end;
  end;
  If (B <> nil) and (FImages <> nil) Then B.Free;
  // Draw item text
  Canvas.Pen.Style := psSolid;
  If ItemState and ODS_SELECTED <> 0 Then
    Canvas.Font.Color := clHighlightText
  else
  Canvas.Font.Color := clMenuText;
  Sz := MenuItem.Caption;
  R := ItemRect;
  InflateRect(R, 0, -(R.Bottom - R.Top - Canvas.TextHeight(Sz)) div 2);
  OffsetRect(R, ImageSize+3, 0);
  If ItemState and ODS_GRAYED = 0 Then
    DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R, DT_VCENTER or DT_LEFT)
  else
  begin
    Canvas.Brush.Style := bsClear;
    Canvas.Font.Color := clBtnHighlight;
    OffsetRect(R, 1, 1);
    DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R, DT_VCENTER or DT_LEFT);
    Canvas.Font.Color := clBtnShadow;
    OffsetRect(R, -1, -1);
    DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R, DT_VCENTER or DT_LEFT);
  end;
  If MenuItem.ShortCut <> 0 Then
  begin
    Sz := ShortCutToText(MenuItem.ShortCut);
    R := ItemRect;
    InflateRect(R, 0, -(R.Bottom - R.Top - Canvas.TextHeight(Sz)) div 2);
    OffsetRect(R, -ImageSize, 0);
    DrawText(Canvas.Handle, PChar(Sz), Length(Sz), R, DT_VCENTER or DT_RIGHT);
  end;
end;

procedure TitaOfficePopupMenu.DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas);
begin
  // Counting item text width and height
  Width := ImageSize*2 + Canvas.TextWidth(MenuItem.Caption) + 2;
  If MenuItem.ShortCut <> 0 Then
    Inc(Width, Canvas.TextWidth(ShortCutToText(MenuItem.ShortCut)));
  If FBanner.Picture.Width > 0 Then
    Inc(Width, FBanner.Picture.Width+2);
  Height := ImageSize + 1;
  Inc(FHeight, Height)
end;

{ TMenuBar 97 }

constructor TitaMenuBar.Create(AOwner: TComponent);
var
  B: TBitmap;
  M: TMenuItem;
begin
  inherited Create(AOwner);
  FCloseButton := false;
  Height := 22;
end;

procedure TitaMenuBar.Loaded;
begin
  inherited Loaded;
  SortControls;
end;

destructor TitaMenuBar.Destroy;
begin
  inherited Destroy;
end;

procedure TitaMenuBar.Notification(AComponent: TComponent;
      Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
end;

procedure TitaMenuBar.Paint;
begin
  case Parent.Align of
    alTop, alBottom: If Width <> Parent.Width Then
      begin
        Left := 0;
        Width := Parent.Width;
        Exit;
      end;
    alLeft, alRight: If Height <> Parent.Height Then
      begin
        Top := 0;
        Height := Parent.Height;
        Exit;
      end;
  end;
  inherited Paint;
end;

initialization
  bmpArrow := TBitmap.Create;
  bmpArrow.Handle := LoadBitmap(HInstance, 'ARROW');
  bmpCheck := TBitmap.Create;
  bmpCheck.Handle := LoadBitmap(HInstance, 'CHECK');
  Screen.Cursors[crHand] := LoadCursor(HInstance, 'HAND');
  randomize;
finalization
  bmpArrow.Free;
  bmpCheck.Free;
end.

