(*******************************************)
(* Founder Lite Library                    *)
(* Version 1.0                             *)
(* Common tools and utils                  *)
(*                                         *)
(*******************************************)
(* (c) 1996-98 Evgesoft Company.           *)
(* All right reserved.                     *)
(*                                         *)
(*******************************************)
(* Internet: dev@evgesoft.com              *)
(* WWW:                                    *)
(*   http://www.evgesoft.com/dev/          *)
(*******************************************)

unit fdr_common;

interface

uses
    Windows, SysUtils, WinTypes, WinProcs, Classes, Graphics, Controls,
    StdCtrls, ExtCtrls, Menus, Messages, Forms, buttons;

const

// MsgBox constants

  msAbort         = IDABORT;
  msCancel        = IDCANCEL;
  msIgnore        = IDIGNORE;
  msNo            = IDNO;
  msOk            = IDOK;
  msRetry         = IDRETRY;
  msYes           = IDYES;

// Menu state                  WM_DrawItem

  msSelected            = ODS_SELECTED;
  msChecked             = ODS_CHECKED;
  msGrayed              = ODS_GRAYED;
  msDefault             = ODS_DEFAULT;

// Components

type

  TPanelBevel = (bvNone, bvLowered, bvRaised);

const
    ULBevelColor: array[ TPanelBevel ] of TColor =
      ( clWindow, clBtnShadow, clBtnHighlight );

    LRBevelColor: array[ TPanelBevel ] of TColor =
      ( clWindow, clBtnHighlight, clBtnShadow );
    TextAlignments: array[ TAlignment ] of Word =
      ( dt_Left, dt_Right, dt_Center );
type

    TSide = ( sdLeft, sdTop, sdRight, sdBottom );
    TSides = set of TSide;
    TVerticalAlignment = ( vaTop, vaCenter, vaBottom );

    TitaCustomPanel = class( TCustomPanel )
    private
      FBevelInner: TPanelBevel;
      FBevelOuter: TPanelBevel;
      FBevelWidth: TBevelWidth;
      FBorderWidth: TBorderWidth;
      FBorderStyle: TBorderStyle;
      FFullRepaint: Boolean;
      FLocked: Boolean;
      FOnResize: TNotifyEvent;
      FAlignment: TAlignment;
      FTrans: Boolean;
      FVerticalAlignment: TVerticalAlignment;
      FShowCaption: Boolean;
      procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
      procedure CMCtl3DChanged(var Message: TMessage); message CM_CTL3DCHANGED;
      procedure CMIsToolControl(var Message: TMessage); message CM_ISTOOLCONTROL;
      procedure WMWindowPosChanged(var Message: TWMWindowPosChanged); message WM_WINDOWPOSCHANGED;
      procedure SetAlignment(Value: TAlignment);
      procedure SetBevelInner(Value: TPanelBevel);
      procedure SetBevelOuter(Value: TPanelBevel);
      procedure SetBevelWidth(Value: TBevelWidth);
      procedure SetBorderWidth(Value: TBorderWidth);
      procedure SetBorderStyle(Value: TBorderStyle);
      procedure SetTrans(Value: boolean);
      procedure SetShowCaption(Value: boolean);
      procedure SetBevelSides( Value: TSides );
      procedure SetBorderColor( Value: TColor );
      procedure SetBorderSides( Value: TSides );
      procedure SetVerticalAlignment( Value: TVerticalAlignment );
    protected
      FBevelSides: TSides;
      FBorderColor: TColor;
      FBorderSides: TSides;
      procedure DrawSides( Rect: TRect; ULColor, LRColor: TColor;
                          Sides: TSides );
      procedure DrawBevel( var Rect: TRect; ULColor, LRColor: TColor;
                          Width: Integer; Sides: TSides );
      function GetClientRect: TRect; override;
      procedure Paint; override;
      procedure CreateParams(var Params: TCreateParams); override;
      procedure AlignControls(AControl: TControl; var Rect: TRect); override;
      procedure Resize; dynamic;
      property Alignment: TAlignment read FAlignment write SetAlignment default taCenter;
      property BevelInner: TPanelBevel read FBevelInner write SetBevelInner default bvNone;
      property BevelOuter: TPanelBevel read FBevelOuter write SetBevelOuter default bvRaised;
      property BevelWidth: TBevelWidth read FBevelWidth write SetBevelWidth default 1;
      property BorderWidth: TBorderWidth read FBorderWidth write SetBorderWidth default 0;
      property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsNone;
      property Color default clBtnFace;
      property FullRepaint: Boolean read FFullRepaint write FFullRepaint default True;
      property Locked: Boolean read FLocked write FLocked default False;
      property ParentColor default False;
      property OnResize: TNotifyEvent read FOnResize write FOnResize;
    public
      constructor Create( AOwner: TComponent ); override;
      property BevelSides: TSides read FBevelSides write SetBevelSides
        default [sdLeft, sdTop, sdRight, sdBottom];
      property BorderColor: TColor read FBorderColor write SetBorderColor
        default clBtnFace;
      property BorderSides: TSides read FBorderSides write SetBorderSides;
      property VerticalAlignment: TVerticalAlignment read FVerticalAlignment
        write SetVerticalAlignment default vaCenter;
      property Transparent: boolean read FTrans write SetTrans;
      property ShowCaption: boolean read FShowCaption write SetShowCaption;
    published
    end;

    TitaPanel = class(TitaCustomPanel)
    published
      property BevelSides;
      property BorderColor;
      property BorderSides;
      property VerticalAlignment;
      property Transparent;
      property ShowCaption;
      property Align;
      property Alignment;
      property BevelInner;
      property BevelOuter;
      property BevelWidth;
      property BorderStyle;
      property BorderWidth;
      property DragCursor;
      property Enabled;
      property Caption;
      property Color;
      property Ctl3D;
      property Font;
      property Locked;
      property ParentColor;
      property ParentCtl3D;
      property ParentFont;
      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property TabStop;
      property Visible;
      property OnClick;
      property OnDblClick;
      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnEnter;
      property OnExit;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
      property OnResize;
    end;

  TTextStyle = ( tsNone, tsRaised, tsRecessed );

  TitaLabel = class( TLabel )
  private
    FTextStyle : TTextStyle;
    FOnMouseEnter: TNotifyEvent;
    FOnMouseLeave: TNotifyEvent;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure DoDrawText(var Rect: TRect; Flags: Word);
  protected
    procedure Paint; override;
    procedure SetTextStyle( Value : TTextStyle );
  public
    constructor Create( AOwner : TComponent ); override;
  published
    property TextStyle : TTextStyle read FTextStyle write SetTextStyle
                                      default tsRecessed;
    property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseLeave: TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
  end;

{ TitaCustomMainMenu }

  TMenuStyle = (msStandard, msOwnerDraw);

  TMeasureMenuItem = procedure (MenuItem: TMenuItem;
    var Width, Height: UINT; Canvas: TCanvas) of object;
  TDrawMenuItem = procedure (MenuItem: TMenuItem; Canvas: TCanvas;
    ItemState: Integer; ItemRect: TRect) of object;

  TitaCustomMenu = class(TMainMenu)
  private
    FMeasureItem: TMeasureMenuItem;
    FDrawItem: TDrawMenuItem;
    FStyle: TMenuStyle;

    FPrevWndProc: Pointer;
    FObjectInst: Pointer;

    procedure WndProc(var Msg: TMessage);

    procedure SetStyle(Value: TMenuStyle);

    procedure DrawItem(var Msg: TWMDrawItem); message WM_DRAWITEM;
    procedure MeasureItem(var Msg: TWMMEASUREITEM); message WM_MEASUREITEM;

    procedure ModifyItems(M: TMenuItem; Std: boolean);
    procedure SetMenuItem(Value: TMenuItem; Std: boolean);
  protected
    procedure DoDrawItem(MenuItem: TMenuItem; Canvas: TCanvas;
      ItemState: Cardinal; ItemRect: TRect); virtual;
    procedure DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas); virtual;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
    procedure Loaded; override;
    property Style: TMenuStyle read FStyle write SetStyle;
    property OnDrawItem: TDrawMenuItem read FDrawItem write FDrawItem;
    property OnMeasureItem: TMeasureMenuItem read FMeasureItem
      write FMeasureItem;
  published
  end;

  TitaMenu = class(TitaCustomMenu)
  published
    property Style;
    property OnDrawItem;
    property OnMeasureItem;
  end;

{ TitaCustomPopupMenu }

  TitaCustomPopupMenu = class(TPopupMenu)
  private
    FMeasureItem: TMeasureMenuItem;
    FDrawItem: TDrawMenuItem;
    FStyle: TMenuStyle;

    FPrevWndProc: Pointer;
    FObjectInst: Pointer;

    procedure WndProc(var Mssg: TMessage);

    procedure SetStyle(Value: TMenuStyle);

    procedure DrawItem(var Msg: TWMDrawItem); message WM_DRAWITEM;
    procedure MeasureItem(var Msg: TWMMEASUREITEM); message WM_MEASUREITEM;

    procedure ModifyItems(M: TMenuItem; Std: boolean);
    procedure SetMenuItem(Value: TMenuItem; Std: boolean);

    function  ProcessAccel(Message: TWMMenuChar ): Integer;
  protected
    procedure DoDrawItem(MenuItem: TMenuItem; Canvas: TCanvas;
      ItemState: Cardinal; ItemRect: TRect); virtual;
    procedure DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas); virtual;
    procedure Popup(X, Y: integer); override;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
    procedure Loaded; override;
    property Style: TMenuStyle read FStyle write SetStyle;
    property OnDrawItem: TDrawMenuItem read FDrawItem write FDrawItem;
    property OnMeasureItem: TMeasureMenuItem read FMeasureItem
      write FMeasureItem;
  published
  end;

  TitaPopupMenu = class(TitaCustomPopupMenu)
  published
    property Style;
    property OnDrawItem;
    property OnMeasureItem;
  end;

// Message boxes

procedure MsgText(AText: String);
procedure MsgError(AText: String);
procedure MsgStop(AText: String);
procedure MsgWarning(AText: String);
function MsgOkCancel(AText: String): Integer;
function MsgYesNo(AText: String): Integer;
function MsgYesNoCancel(AText: String): Integer;

// Additional function

function PointInRect(P: TPoint; R: TRect): boolean;

implementation {==============================================================}

{ TitaCustomPanel }

constructor TitaCustomPanel.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  ControlStyle := [csAcceptsControls, csCaptureMouse, csClickEvents,
    csSetCaption, csOpaque, csDoubleClicks, csReplicatable];
  Width := 185;
  Height := 41;
  FAlignment := taCenter;
  BevelOuter := bvRaised;
  BevelWidth := 1;
  FBorderStyle := bsNone;
  Color := clBtnFace;
  FFullRepaint := True;
  FBevelSides := [ sdLeft, sdTop, sdRight, sdBottom ];
  FBorderColor := clBtnFace;
  FVerticalAlignment := vaCenter;
  FShowCaption := false;
  FTrans := false;
end;

procedure TitaCustomPanel.SetShowCaption(Value: boolean);
begin
  if FShowCaption <> Value then
  begin
    FShowCaption := Value;
    Invalidate;
  end;
end;

procedure TitaCustomPanel.CreateParams(var Params: TCreateParams);
const
  BorderStyles: array[TBorderStyle] of Longint = (0, WS_BORDER);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    Style := Style or BorderStyles[FBorderStyle];
    if NewStyleControls and Ctl3D and (FBorderStyle = bsSingle) then
    begin
      Style := Style and not WS_BORDER;
      ExStyle := ExStyle or WS_EX_CLIENTEDGE;
    end;
    WindowClass.style := WindowClass.style and not (CS_HREDRAW or CS_VREDRAW);
  end;
end;

procedure TitaCustomPanel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TitaCustomPanel.CMCtl3DChanged(var Message: TMessage);
begin
  if NewStyleControls and (FBorderStyle = bsSingle) then RecreateWnd;
  inherited;
end;

procedure TitaCustomPanel.CMIsToolControl(var Message: TMessage);
begin
  if not FLocked then Message.Result := 1;
end;

procedure TitaCustomPanel.Resize;
begin
  if Assigned(FOnResize) then FOnResize(Self);
end;

procedure TitaCustomPanel.WMWindowPosChanged(var Message: TWMWindowPosChanged);
var
  BevelPixels: Integer;
  Rect: TRect;
begin
  if FullRepaint or (Caption <> '') then
    Invalidate
  else
  begin
    BevelPixels := BorderWidth;
    if BevelInner <> bvNone then Inc(BevelPixels, BevelWidth);
    if BevelOuter <> bvNone then Inc(BevelPixels, BevelWidth);
    if BevelPixels > 0 then
    begin
      Rect.Right := Width;
      Rect.Bottom := Height;
      if Message.WindowPos^.cx <> Rect.Right then
      begin
        Rect.Top := 0;
        Rect.Left := Rect.Right - BevelPixels - 1;
        InvalidateRect(Handle, @Rect, True);
      end;
      if Message.WindowPos^.cy <> Rect.Bottom then
      begin
        Rect.Left := 0;
        Rect.Top := Rect.Bottom - BevelPixels - 1;
        InvalidateRect(Handle, @Rect, True);
      end;
    end;
  end;
  inherited;
  if not (csLoading in ComponentState) then Resize;
end;

procedure TitaCustomPanel.AlignControls(AControl: TControl; var Rect: TRect);
var
  BevelSize: Integer;
begin
  BevelSize := BorderWidth;
  if BevelOuter <> bvNone then Inc(BevelSize, BevelWidth);
  if BevelInner <> bvNone then Inc(BevelSize, BevelWidth);
  InflateRect(Rect, -BevelSize, -BevelSize);
  inherited AlignControls(AControl, Rect);
end;

procedure TitaCustomPanel.SetAlignment(Value: TAlignment);
begin
  FAlignment := Value;
  Invalidate;
end;

procedure TitaCustomPanel.SetBevelInner(Value: TPanelBevel);
begin
  FBevelInner := Value;
  Realign;
  Invalidate;
end;

procedure TitaCustomPanel.SetBevelOuter(Value: TPanelBevel);
begin
  FBevelOuter := Value;
  Realign;
  Invalidate;
end;

procedure TitaCustomPanel.SetBevelWidth(Value: TBevelWidth);
begin
  FBevelWidth := Value;
  Realign;
  Invalidate;
end;

procedure TitaCustomPanel.SetBorderWidth(Value: TBorderWidth);
begin
  FBorderWidth := Value;
  Realign;
  Invalidate;
end;

procedure TitaCustomPanel.SetBorderStyle(Value: TBorderStyle);
begin
  if FBorderStyle <> Value then
  begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;

procedure TitaCustomPanel.SetTrans(Value: Boolean);
begin
  if FTrans <> Value then
  begin
    FTrans := Value;
    Invalidate;
  end;
end;

procedure TitaCustomPanel.SetBevelSides( Value: TSides );
begin
 if FBevelSides <> Value then
  begin
   FBevelSides := Value;
   Invalidate;
  end;
end;

procedure TitaCustomPanel.SetBorderColor( Value: TColor );
begin
  if FBorderColor <> Value then
  begin
   FBorderColor := Value;
   Invalidate;
  end;
end;

procedure TitaCustomPanel.SetBorderSides( Value: TSides );
begin
  if FBorderSides <> Value then
  begin
   FBorderSides := Value;
   Invalidate;
  end;
end;

procedure TitaCustomPanel.SetVerticalAlignment( Value: TVerticalAlignment );
begin
 if FVerticalAlignment <> Value then
  begin
   FVerticalAlignment := Value;
   Invalidate;
  end;
end;

procedure TitaCustomPanel.DrawSides( Rect: TRect; ULColor, LRColor: TColor;
                               Sides: TSides );
begin
  with Canvas, Rect do
  begin
    Pen.Color := ULColor;
    if sdLeft in Sides then
    begin
      MoveTo( Left, Top );
      LineTo( Left, Bottom );
    end;
    if sdTop in Sides then
    begin
      MoveTo( Left, Top );
      LineTo( Right, Top );
    end;
    Pen.Color := LRColor;
    if sdRight in Sides then
    begin
      MoveTo( Right - 1, Top );
      LineTo( Right - 1, Bottom );
    end;
    if sdBottom in Sides then
    begin
      MoveTo( Left, Bottom - 1 );
      LineTo( Right, Bottom - 1 );
    end;
  end;
end;

procedure TitaCustomPanel.DrawBevel( var Rect: TRect; ULColor, LRColor: TColor;
                               Width: Integer; Sides: TSides);
var
   i: Integer;
begin
 Canvas.Pen.Width := 1;
 for i := 1 to Width do
 begin
   DrawSides( Rect, UlColor, LRColor, Sides);
   InflateRect( Rect, -1, -1);
 end;
end;

function TitaCustomPanel.GetClientRect: TRect;
begin
 Result := inherited GetClientRect;
 if sdLeft in FBorderSides then inc( Result.Left );
 if sdTop in FBorderSides then inc( Result.Top );
 if sdRight in FBorderSides then dec( Result.Right );
 if sdBottom in FBorderSides then dec( Result.Bottom );
end;

procedure TitaCustomPanel.Paint;
var
   DrawRct, TextRct: TRect;
   TempStz: array[ 0..255 ] of Char;
   H: Integer;
begin
 DrawRct := Rect( 0, 0, Width, Height );
 with Canvas, DrawRct do
  begin
   DrawSides( DrawRct, clBlack, clBlack, FBorderSides );
   if sdLeft in FBorderSides then inc( Left );
   if sdTop in FBorderSides then inc( Top );
   if sdRight in FBorderSides then dec( Right );
   if sdBottom in FBorderSides then dec( Bottom );
   if TPanelBevel(BevelOuter) <> bvNone then
    DrawBevel( DrawRct, ULBevelColor[ TPanelBevel(BevelOuter) ], LRBevelColor[ TPanelBevel(BevelOuter) ],
               BevelWidth, FBevelSides );
   if BorderWidth > 0 then
    DrawBevel( DrawRct, FBorderColor, FBorderColor, BorderWidth, FBevelSides);
   if TPanelBevel(BevelInner) <> bvNone then
    DrawBevel( DrawRct, ULBevelColor[ TPanelBevel(BevelInner) ], LRBevelColor[ TPanelBevel(BevelInner) ],
               BevelWidth, FBevelSides );
   Brush.Color := Color;
   If not FTrans Then
     FillRect(DrawRct);
   Brush.Style := bsClear;
   Font := Self.Font;
   StrPCopy( TempStz, Caption );
   TextRct := DrawRct;
   If FShowCaption Then
     H := DrawText( Handle, TempStz, -1, TextRct,
                   dt_CalcRect or dt_WordBreak or dt_ExpandTabs or dt_VCenter or
                    TextAlignments[ Alignment ] )
   else
     H := 0;
   if FVerticalAlignment = vaCenter then
    begin
     Top := ( ( Bottom + Top ) - H ) shr 1;
     Bottom := Top + H;
    end
   else
    if FVerticalAlignment = vaBottom then
     Top := Bottom - H - 1;
   If FShowCaption Then
   DrawText( Handle, TempStz, -1, DrawRct, dt_WordBreak or dt_ExpandTabs
             or dt_VCenter or TextAlignments[ Alignment ] );

  end;
end;

{ TitaLabel }

constructor TitaLabel.Create( AOwner : TComponent );
begin
  inherited Create( AOwner );
  FTextStyle := tsRecessed;
end;

procedure TitaLabel.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  If Assigned(FOnMouseEnter) Then FOnMouseEnter(Self);
end;

procedure TitaLabel.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  If Assigned(FOnMouseLeave) Then FOnMouseLeave(Self);
end;

procedure TitaLabel.SetTextStyle( Value : TTextStyle );
begin
  if Value <> FTextStyle then
  begin
    FTextStyle := Value;
    Invalidate;
  end;
end;

procedure TitaLabel.DoDrawText( var Rect : TRect; Flags : Word );
var
  Text: array[ 0..255 ] of Char;
  TmpRect: TRect;
  UpperColor: TColor;
  LowerColor: TColor;
begin
  GetTextBuf(Text, SizeOf(Text));
  if (Flags and DT_CALCRECT <> 0) and
     ((Text[0] = #0) or ShowAccelChar and (Text[0] = '&') and
      (Text[1] = #0)) then
      StrCopy(Text, ' ');
  if not ShowAccelChar then
    Flags := Flags or DT_NOPREFIX;
  Canvas.Font := Font;

  UpperColor := clBtnHighlight;
  LowerColor := clBtnShadow;
  if FTextStyle = tsRecessed then
  begin
    UpperColor := clBtnShadow;
    LowerColor := clBtnHighlight;
  end;
  if FTextStyle in [ tsRecessed, tsRaised ] then
  begin
    TmpRect := Rect;
    OffsetRect( TmpRect, 1, 1 );
    Canvas.Font.Color := LowerColor;
    DrawText(Canvas.Handle, Text, StrLen(Text), TmpRect, Flags);

    TmpRect := Rect;
    OffsetRect( TmpRect, -1, -1 );
    Canvas.Font.Color := UpperColor;
    DrawText(Canvas.Handle, Text, StrLen(Text), TmpRect, Flags);
  end;

  Canvas.Font.Color := Font.Color;
  if not Enabled then
    Canvas.Font.Color := clGrayText;
  DrawText(Canvas.Handle, Text, StrLen(Text), Rect, Flags);
end;

procedure TitaLabel.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var
  Rect: TRect;
begin
  with Canvas do
  begin
    if not Transparent then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    DoDrawText( Rect, ( DT_EXPANDTABS or DT_WORDBREAK ) or
                Alignments[ Alignment ] );
  end;
end;


// Messages

var
  Sz: array [0..200] of Char;

procedure MsgText(AText: String);
begin
  strPCopy(Sz, AText);
  Application.MessageBox(Sz, 'Message', MB_ICONINFORMATION or MB_OK);
end;

procedure MsgError(AText: String);
begin
  strPCopy(Sz, AText);
  Application.MessageBox(Sz, 'Error', MB_ICONERROR or MB_OK);
end;

procedure MsgStop(AText: String);
begin
  strPCopy(Sz, AText);
  Application.MessageBox(Sz, 'Stop', MB_ICONSTOP or MB_OK);
end;

procedure MsgWarning(AText: String);
begin
  strPCopy(Sz, AText);
  Application.MessageBox(Sz, 'Warning', MB_ICONWARNING or MB_OK);
end;

function MsgYesNo(AText: String): Integer;
begin
  strPCopy(Sz, AText);
  Result := Application.MessageBox(Sz, 'Question', MB_ICONQUESTION or MB_YESNO);
end;

function MsgOkCancel(AText: String): Integer;
begin
  strPCopy(Sz, AText);
  Result := Application.MessageBox(Sz, 'Question', MB_ICONQUESTION or MB_OKCANCEL);
end;

function MsgYesNoCancel(AText: String): Integer;
begin
  strPCopy(Sz, AText);
  Result := Application.MessageBox(Sz, 'Question', MB_ICONQUESTION or MB_YESNOCANCEL);
end;

function PointInRect(P: TPoint; R: TRect): boolean;
begin
  Result := (P.X >= R.Left) and (P.X <= R.right) and
     (P.Y >= R.Top) and (P.Y <= R.bottom);
end;

{ TitaCustomMenu }

const

  ActiveMenu: TitaCustomMenu = nil;

constructor TitaCustomMenu.Create( AOwner: TComponent );
begin
  inherited Create(AOwner);

  ActiveMenu := Self;

  If not(csDesigning in ComponentState) Then begin
    // Override WNd Proc
    FObjectInst := MakeObjectInstance(WndProc);
    FPrevWndProc := Pointer(GetWindowLong((Owner as TForm).Handle,
      GWL_WNDPROC));
    SetWindowLong((Owner as TForm).Handle, GWL_WNDPROC,
      LongInt(FObjectInst));
  end;
end;

destructor TitaCustomMenu.Destroy;
begin
{  If not (csDesigning in ComponentState) Then begin
  end;}
  inherited Destroy;
end;

procedure TitaCustomMenu.WndProc(var Msg: TMessage);
var
  M: TWMMEASUREITEM;
  D: TWMDrawItem;
begin
  with Msg do
  begin
    case Msg of
      WM_DESTROY:
      begin
        // Destroy
        SetWindowLong((Owner as TForm).Handle, GWL_WNDPROC,
          LongInt(FPrevWndProc));
        FreeObjectInstance(FObjectInst);
      end;
      WM_DRAWITEM:
      If (ActiveMenu <> nil) and (ActiveMenu.FStyle <> msStandard) Then
      begin
        D.Msg := Msg;
        D.Ctl := wParam;
        D.DrawItemStruct := Pointer(lParam);
        D.Result := (Owner as TForm).Handle;
        DrawItem(D);
        Result := D.Result;
        Exit;
      end;
      WM_MEASUREITEM:
      If (ActiveMenu <> nil) and (ActiveMenu.FStyle <> msStandard) Then
      begin
        M.Msg := Msg;
        M.IDCtl := wParam;
        M.MeasureItemStruct := Pointer(lParam);
        M.Result := (Owner as TForm).Handle;
        MeasureItem(M);
        Result := M.Result;
      end;
    end;
    Result := CallWindowProc(FPrevWndProc, (Owner as TForm).Handle,
      Msg, WParam, LParam);
  end;
end;

procedure TitaCustomMenu.ModifyItems(M: TMenuItem; Std: boolean);
var
  i: integer;
begin
  with M do
    for i := 0 to M.Count-1 do
    begin
      If Items[i].Parent.Name <> '' Then
        SetMenuItem(Items[i], Std);
      If Items[i].Count > 0 Then
        ModifyItems(Items[i], Std);
    end;
end;

procedure TitaCustomMenu.Loaded;
begin
  inherited Loaded;
  ModifyItems(Items, FStyle = msStandard);
end;

procedure TitaCustomMenu.SetStyle(Value: TMenuStyle);
begin
  FStyle := Value;
  ModifyItems(Items, FStyle = msStandard);
end;

procedure TitaCustomMenu.SetMenuItem;
const
  IBreaks: array[TMenuBreak] of Longint = (MFT_STRING, MFT_MENUBREAK, MFT_MENUBARBREAK);
var
  FMenuItem: TMenuItem;
begin
  FMenuItem := Value;
  If not (csDesigning in ComponentState) Then
    If not Std Then
      ModifyMenu(FMenuItem.Parent.Handle, FMenuItem.Command,
        MF_BYCOMMAND or MF_OWNERDRAW or IBreaks[FMenuItem.Break],
        FMenuItem.Command, PChar(FMenuItem.Command))
    else
      ModifyMenu(FMenuItem.Parent.Handle, FMenuItem.Command,
        MF_BYCOMMAND or IBreaks[FMenuItem.Break],
        FMenuItem.Command, PChar(FMenuItem.Command))
end;

procedure TitaCustomMenu.DoDrawItem;
begin
  If Assigned(FDrawItem) Then
    FDrawItem(MenuItem, Canvas, ItemState, ItemRect);
end;

procedure TitaCustomMenu.DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas);
begin
  If Assigned(FMeasureItem) Then
    FMeasureItem(MenuItem, Width, Height, Canvas);
end;

procedure TitaCustomMenu.MeasureItem(var Msg: TWMMEASUREITEM);
var
  M: TMenuItem;
  Can: TCanvas;
begin
  with Msg.MeasureItemStruct^ do
  begin
    CTLtype := ODT_MENU;
    ItemID := 0;
    // Counting item text width and height
    M := FindItem(itemData, fkCommand);
    If M <> nil Then
    begin
      Can := TCanvas.Create;
      If Screen.ActiveForm <> nil Then
      begin
        Can.Handle := Screen.ActiveForm.Canvas.Handle;
        DoMeasureItem(M, ItemWidth, ItemHeight, Can);
      end;
      Can.Free;
    end;
  end;
end;

procedure TitaCustomMenu.DrawItem(var Msg: TWMDrawItem);
var
  Can: TCanvas;
  M: TMenuItem;
begin
  with Msg.DrawItemStruct^ do
  begin
    Can := TCanvas.Create;
    Can.Handle := hDC;
    Can.Brush.Style := bsSolid;
    Can.Pen.Color := clBlack;
    Can.Pen.Style := psClear;
    M := FindItem(itemData, fkCommand);
    // Draw item rect
    If M <> nil Then
      DoDrawItem(M, Can, ItemState, rcItem);
    // Free canvas
    Can.Free;
  end;
end;

{ TitaCustomPopupMenu }

const
  ActivePopup: TitaCustomPopupMenu = nil;
  OldActivePopup: TitaCustomPopupMenu = nil;
  WndProcCount: integer = 0;

constructor TitaCustomPopupMenu.Create( AOwner: TComponent );
begin
  inherited Create(AOwner);
  // Override WNd Proc
  If not(csDesigning in ComponentState) and (WndProcCount = 0) Then begin
    FObjectInst := MakeObjectInstance(WndProc);
    FPrevWndProc := Pointer(GetWindowLong(WindowHandle,
      GWL_WNDPROC));
    SetWindowLong(WindowHandle, GWL_WNDPROC,
      LongInt(FObjectInst));
  end;
  Inc(WndProcCount);
end;

destructor TitaCustomPopupMenu.Destroy;
begin
  Dec(WndProcCount);
  If not(csDesigning in ComponentState) and (WndProcCount = 0) Then
  begin
    SetWindowLong(WindowHandle, GWL_WNDPROC,
      LongInt(FPrevWndProc));
    FreeObjectInstance(FObjectInst);
  end;
  inherited Destroy;
end;

var
  isExitMenuLoop: boolean = false;
  
procedure TitaCustomPopupMenu.WndProc(var Mssg: TMessage);
var
  M: TWMMEASUREITEM;
  D: TWMDrawItem;
begin
  with Mssg do
  begin
    case Msg of
      WM_COMMAND:
      If ((ActivePopup <> nil) and (ActivePopup.FStyle <> msStandard))
         or (isExitMenuLoop) Then
      begin
        ActivePopup := OldActivePopup;
        If ActivePopup.FindItem(wParam, fkCommand) <> nil Then
        begin
          isExitMenuLoop := false;
          ActivePopup.DispatchCommand(wParam);
          ActivePopup := nil;
        end;
      end;
      WM_EXITMENULOOP:
      begin
        OldActivePopup := ActivePopup;
        ActivePopup := nil;
        isExitMenuLoop := true;
      end;
      WM_DRAWITEM:
      If (ActivePopup <> nil) and (ActivePopup.FStyle <> msStandard) Then
      begin
        D.Msg := Msg;
        D.Ctl := wParam;
        D.DrawItemStruct := Pointer(lParam);
        D.Result := WindowHandle;
        ActivePopup.DrawItem(D);
        Result := D.Result;
      end;
      WM_MEASUREITEM:
      If (ActivePopup <> nil) and (ActivePopup.FStyle <> msStandard) Then
      begin
        M.Msg := Msg;
        M.IDCtl := wParam;
        M.MeasureItemStruct := Pointer(lParam);
        M.Result := WindowHandle;
        ActivePopup.MeasureItem(M);
        Result := M.Result;
      end;
    end;
    Result := CallWindowProc(FPrevWndProc, WindowHandle,
      Msg, WParam, LParam);
  end;
end;

procedure TitaCustomPopupMenu.ModifyItems(M: TMenuItem; Std: boolean);
var
  i: integer;
begin
  with M do
    for i := 0 to Count-1 do
    begin
      SetMenuItem(Items[i], Std);
      If Items[i].Count > 0 Then
        ModifyItems(Items[i], Std);
    end;
end;

procedure TitaCustomPopupMenu.Loaded;
begin
  inherited Loaded;
end;

procedure TitaCustomPopupMenu.SetStyle(Value: TMenuStyle);
begin
  FStyle := Value;
  ModifyItems(Items, FStyle = msStandard);
end;

procedure TitaCustomPopupMenu.SetMenuItem;
const
  IBreaks: array[TMenuBreak] of Longint = (MFT_STRING, MFT_MENUBREAK, MFT_MENUBARBREAK);
var
  FMenuItem: TMenuItem;
begin
  FMenuItem := Value;
  If not (csDesigning in ComponentState) Then
    If not Std Then
      ModifyMenu(FMenuItem.Parent.Handle, FMenuItem.Command,
        MF_BYCOMMAND or MF_OWNERDRAW or IBreaks[FMenuItem.Break],
        FMenuItem.Command, PChar(FMenuItem.Command))
    else
      ModifyMenu(FMenuItem.Parent.Handle, FMenuItem.Command,
        MF_BYCOMMAND or IBreaks[FMenuItem.Break],
        FMenuItem.Command, PChar(FMenuItem.Command));
end;

procedure TitaCustomPopupMenu.DoDrawItem;
begin
  If Assigned(FDrawItem) Then
    FDrawItem(MenuItem, Canvas, ItemState, ItemRect);
end;

procedure TitaCustomPopupMenu.DoMeasureItem(MenuItem: TMenuItem;
      var Width, Height: UINT; Canvas: TCanvas);
begin
  If Assigned(FMeasureItem) Then
    FMeasureItem(MenuItem, Width, Height, Canvas);
end;

procedure TitaCustomPopupMenu.MeasureItem(var Msg: TWMMEASUREITEM);
var
  M: TMenuItem;
  Can: TCanvas;
begin
  If ActivePopup = nil Then Exit;
  with Msg.MeasureItemStruct^ do
  begin
    CTLtype := ODT_MENU;
    // Counting item text width and height
    M := FindItem(itemData, fkCommand);
    If M <> nil Then
    begin
      ItemID := M.Command;
      ItemData := M.Command;
      Can := TCanvas.Create;
      Can.Handle := CreateDC('DISPLAY', nil, nil, nil);
      DoMeasureItem(M, ItemWidth, ItemHeight, Can);
      Can.Free;
    end
    else
      ItemID := 0;
  end;
end;

procedure TitaCustomPopupMenu.DrawItem(var Msg: TWMDrawItem);
var
  Can: TCanvas;
  M: TMenuItem;
begin
  If ActivePopup = nil Then Exit;
  with Msg.DrawItemStruct^ do
  begin
    Can := TCanvas.Create;
    Can.Handle := hDC;
    Can.Brush.Style := bsSolid;
    Can.Pen.Color := clBlack;
    Can.Pen.Style := psClear;
    M := FindItem(itemData, fkCommand);
    If M = nil Then Exit;
    If not M.Enabled Then
      ItemState := ItemState or msGrayed;
    If M.Checked Then
      ItemState := ItemState or msChecked;
    // Draw item rect
    DoDrawItem(M, Can, ItemState, rcItem);
    // Free canvas
    Can.Free;
  end;
end;

procedure TitaCustomPopupMenu.Popup(X, Y: integer);
const
  Flags: array[TPopupAlignment] of Word = (TPM_LEFTALIGN, TPM_RIGHTALIGN,
    TPM_CENTERALIGN);
begin
  If ActivePopup = Self Then Exit;
  ActivePopup := Self;
  ModifyItems(Items, FStyle = msStandard);
//  inherited Popup(X, Y);

  TrackPopupMenu(Items.Handle, Flags[Alignment] or TPM_RIGHTBUTTON,
    X, Y, 0 , WindowHandle, nil);
end;

function TitaCustomPopupMenu.ProcessAccel(Message: TWMMenuChar): Integer;

 function FindAccelerator(MenuItems: TMenuItem): Integer;
 var
   i: Integer;
 begin
   Result:= -1;
   for i:= 0 to MenuItems.Count - 1 do
   begin
     if (Result = -1) and (MenuItems[i].Enabled) then
     begin
       if (MenuItems.Handle <> Message.Menu) then
         Result:= FindAccelerator(MenuItems[i]);
       if (MenuItems.Handle = Message.Menu) then
         if (IsAccel(Word(Message.User), MenuItems[i].Caption)) then
         begin
           Result:= MenuItems[i].MenuIndex
         end;
     end;
   end;
 end;

begin
  Result := -1;
  if (csDesigning in ComponentState) then Exit;
//  if (InternalMenu <> nil) then
  Result := FindAccelerator(Items);
end;

initialization
  RegisterClass(TitaMenu);
finalization
end.

