unit PcxMap;

interface

uses Windows, SysUtils, Classes, Graphics, Dialogs;

const

  { Colori }
  RGB_RED           =  0;
  RGB_GREEN         =  1;
  RGB_BLUE          =  2;
  WRGB_RED          =  2;
  WRGB_GREEN        =  1;
  WRGB_BLUE         =  0;
  RGB_SIZE          =  3;

  { Maschere }
  MaskTable: array[0..7] of Byte = ($0080, $0040, $0020, $0010, $0008, $0004, $0002, $0001);
  BitTable:  array[0..7] of Byte = ($0001, $0002, $0004, $0008, $0010, $0020, $0040, $0080);

  { error constants }
  geNoError         =  0;  { no errors found }
  geNoFile          =  1;  { tif file not found }
  geNotPcx          =  2;  { file is not a gif file }
  geFileTooLong     =  3;  { no Global Color table found }
  geBadData         =  4;  { image descriptor preceeded by other unknown data }
  geFormatUnsup     =  5;  { Block has no data }
  geUnExpectedEOF   =  6;  { unexpected EOF }
  geBadCodeSize     =  7;  { bad code size }
  geBadCode         =  8;  { Bad code was found }
  geBitSizeOverflow =  9;  { bit size went beyond 12 bits }
  geNoBMP           = 10;  { Could not make BMP file }

  ErrName: Array[1..10] of string = (
	'PCX file not found',
	'Not a PCX file',
	'File Too big',
	'Bad data',
	'Unsupported Format',
 	'Unexpected EOF',
	'Bad code size',
	'Bad code',
	'Bad bit size',
  'Bad bitmap');


type
  EPcxException = class(Exception)
  end;

type
  GraphicLine      = array [0..3072] of byte;
  PBmLine         = ^TBmpLinesStruct;
  TBmpLinesStruct = record
    LineData  : GraphicLine;
    LineNo    : Integer;
  end;

Type
  TColorItem = record			{ one item a a color table }
    Red: Byte;
    Green: Byte;
    Blue: Byte;
  end;

Type
  T16ColTable = Array[0..15] of TColorItem;	  { the 16 color table }

Type
  T256ColTable = Array[0..255] of TColorItem;	{ the 256 color table }

type
  TPCXHeader = record
    Manufacturer: Byte;
    Version: Byte;
    Encoding: Byte;
    Bits: Byte;
    Xmin, Ymin: Word;
    Xmax, Ymax: Word;
    Hres: Word;
    Vres: Word;
    Palette: T16ColTable;
    Reserved: Byte;
    ColourPlanes: Byte;
    BytesPerLine: Word;
    PaletteType: Word;
    Filler: Array [1..58] Of Byte;
  End;

Type
  TPcxmap = class(TBitmap)
  private
    FStream: TMemoryStream;
    PCXH: TPCXHeader;
    Width: Integer;
    Depth: Integer;
    Bits: Integer;
    Palette: T256ColTable;
    BmHeader: TBitmapInfoHeader;              { File Header for bitmap file}
    LineBuffer: GraphicLine;                  { array for buffer line output }
    ExtraBuffer: GraphicLine;                  { array for buffer line output }
    ImageLines: TList;                        { Image data}
    CurrentX, CurrentY  : Integer;            { current screen locations }
    procedure Error(ErrCode: integer);
    Procedure ReadPcx;
    Function ReadLine(Bytes: Integer): Boolean;
    procedure DefinePalette;
    procedure CreateLine;
    procedure CreateBitHeader;                {Takes Tif info. and converts to BMP}
    procedure SavePcxToStream(Stream: TStream);
  public
    constructor Create; override;
    destructor Destroy; override;
    procedure LoadFromStream(Stream: TStream); override;
  end;


implementation

uses Math;

constructor TPcxMap.Create;
begin
  inherited Create;
  FStream := nil;
  ImageLines := TList.Create;
end;

destructor TPcxMap.Destroy;
begin
  ImageLines.Free;
  inherited Destroy;
end;

procedure TPcxMap.Error(ErrCode: integer);
begin
  raise EPcxException.Create(ErrName[ErrCode]);
end;

procedure TPcxMap.LoadFromStream(Stream: TStream);
begin
  try
    FStream := TMemoryStream.Create;
    FStream.CopyFrom(Stream, Stream.Size);
    FStream.Position := 0;
    CurrentX := 0;
    CurrentY := 0;
    ReadPcx;
    CreateBitHeader;
    try
      SavePcxToStream(FStream);
      inherited LoadFromStream(FStream);
    Finally
      FStream.Free;
    end;
  Except
  end;
end;

Procedure TPcxMap.ReadPcx;
Var
  A, I, J, K, M, N, X, Bytes: Integer;
  ILbuf, IEbuf: Integer;
Begin
  FStream.Position := 0;
  If FStream.Read(PCXH, SizeOf(PCXH)) <> SizeOf(PCXH) Then
  Begin
    Error(geBadData);
    Exit;
  End;
  Width := PCXH.Xmax - PCXH.Xmin + 1;
  Depth := PCXH.Ymax - PCXH.Ymin + 1;
  If (PCXH.Bits = 8) And (PCXH.ColourPlanes = 3) Then Bits := 24
  Else If (PCXH.Bits = 1) Then Bits := PCXH.ColourPlanes
  Else Bits := PCXH.Bits;
  ImageLines.Clear;
  ImageLines.Capacity := Depth;
  If PCXH.Manufacturer <> 10 then
  Begin
    Error(geNotPcx);
    Exit;
  End;
  If (Bits > 1) And (Bits <= 4) Then
  Begin
    If (PCXH.Bits = 4) And (PCXH.ColourPlanes = 1) Then
      Bytes := PCXH.BytesPerLine
    Else Bytes := PCXH.BytesPerLine * Bits;
  End
  Else If Bits = 24 Then Bytes := PCXH.BytesPerLine * RGB_SIZE
  Else Bytes := PCXH.BytesPerLine;
  N := (Width + 7) Div 8;
  If Bytes > 3072 Then
  Begin
    Error(geFileTooLong);
    Exit;
  End;
  For I := 0 To Depth - 1 Do
  Begin
    If Not ReadLine(Bytes) Then
    Begin
      Error(geBadData);
      Exit;
    End;
    If (Bits > 1) And (Bits <= 4) Then
    Begin
      If (PCXH.Bits <> 4) Or (PCXH.ColourPlanes <> 1) Then
      Begin
        For M := 0 To 2047 Do ExtraBuffer[M] := 0;
        ILbuf := 0;
        IEbuf := 0;
        For J := 0 To Bits - 1 Do
        Begin
          For M := 0 To PCXH.BytesPerLine - 1 Do ExtraBuffer[M + IEbuf] := LineBuffer[M + ILbuf];
          ILbuf := ILbuf + PCXH.BytesPerLine;
          IEbuf := IEbuf + N;
        End;
        For M := 0 To 2047 Do LineBuffer[M] := 0;
        X := 0;
        J := 0;
        Repeat
          A := 0;
          IEbuf := 0;
          For K := 0 To Bits - 1 Do
          Begin
            If (ExtraBuffer[IEBuf + (J Shr 3)] And MaskTable[J And 7]) > 0 Then
              A := A Or BitTable[K];
            IEBuf := IEBuf + N;
          End;
          LineBuffer[X] := (A And 15) Shl 4;
          Inc(J);
          If J < Width Then
          Begin
            A := 0;
            IEbuf := 0;
            For K := 0 To Bits - 1 Do
            Begin
              If (ExtraBuffer[IEBuf + (J Shr 3)] And MaskTable[J And 7]) > 0 Then
                A := A Or BitTable[K];
              IEBuf := IEBuf + N;
            End;
            LineBuffer[X] := LineBuffer[X] Or (A And 15);
          End;
          Inc(J);
          Inc(X);
        Until J = Width;
      End;
    End
    Else If Bits = 24 Then
    Begin
      For M := 0 To 2047 Do ExtraBuffer[M] := LineBuffer[M];
      For J := 0 To Width - 1 Do
      Begin
        LineBuffer[J * RGB_SIZE + WRGB_RED] := ExtraBuffer[J];
        LineBuffer[J * RGB_SIZE + WRGB_GREEN] :=
          ExtraBuffer[RGB_GREEN * PCXH.BytesPerLine + J];
        LineBuffer[J * RGB_SIZE + WRGB_BLUE] :=
          ExtraBuffer[RGB_BLUE * PCXH.BytesPerLine + J];
      End;
    End;
    CreateLine;
  End;
  DefinePalette;
End;

Function TPcxMap.ReadLine(Bytes: Integer): Boolean;
Var
  N:  Integer;
  C, I: Byte;
  Ret: Boolean;
Begin
  Ret := False;
  N := 0;
  Repeat
    If FStream.Read(C, 1) <> 1 Then Break;
    If (C And $00C0) = $00C0 Then
    Begin
      I := C And $003F;
      If FStream.Read(C, 1) <> 1 Then Break;
      While I > 0 Do
      Begin
        LineBuffer[N] := C;
        Inc(N);
        Dec(I);
      End;
    End
    Else
    Begin
      LineBuffer[N] := C;
      Inc(N);
    End;
  Until N >= Bytes;
  If N = Bytes Then Ret := True;
  Result := Ret;
End;

procedure TPcxMap.DefinePalette;
Var
  I: Integer;
  C: Byte;
Begin
  C := 0;
  For I := 0 To 15 do
  Begin
    Palette[I].Red := PCXH.Palette[I].Red;
    Palette[I].Green := PCXH.Palette[I].Green;
    Palette[I].Blue := PCXH.Palette[I].Blue;
  End;
  If (Bits = 8) And (PCXH.Version >= 5) Then
  Begin
    FStream.Seek(-769, soFromEnd);
    FStream.Read(C, 1);
    If C = 12 Then
    Begin
      If FStream.Read(Palette, 768) <> 768 Then Error(geBadData);
    End
    Else Error(geBadData);
  End
End;

procedure TPcxMap.CreateBitHeader;
{ This routine takes the values from the TIF image
  descriptor and fills in the appropriate values in the
  bit map header struct. }
begin
  with BmHeader do
  begin
    biSize           := Sizeof(TBitmapInfoHeader);
    biWidth          := Width;
    biHeight         := Depth;
    biPlanes         := 1;            {Arcane and rarely used}
    biBitCount       := Bits;         {Number of bit per pixel}
    biCompression    := BI_RGB;       {Sorry Did not implement compression in this version}
    biSizeImage      := 0;            {Valid since we are not compressing the image}
    biXPelsPerMeter  :=143;           {Rarely used very arcane field}
    biYPelsPerMeter  :=143;           {Ditto}
    biClrUsed        := 0;            {all colors are used}
    biClrImportant   := 0;            {all colors are important}
  end;
end;

procedure TPcxMap.CreateLine;
var
  p: PBmLine;
begin
  New(p);                       {Fill in the data}
  p^.LineData := LineBuffer;
  p^.LineNo := CurrentY;        {Add it to the list of lines}
  ImageLines.Add(p);            {Prepare for the next line}
  Inc(CurrentY);
end;

procedure TPcxMap.SavePcxToStream(Stream: TStream);
var
  BitFile: TBitmapFileHeader;
  i: integer;
  ch: char;
  p: PBmLine;
  PalSize, LineWidth, x: integer;
begin
  If Bits = 1 Then LineWidth := (Width + 7) Div 8
  Else If Bits = 4 Then LineWidth := ((Width + 7) Div 8) Shl 2
  Else If Bits = 8 Then LineWidth := Width
  Else LineWidth := Width * RGB_SIZE;
  PalSize := (4 * Round(Power(2, Bits)));
  If Bits > 8 Then PalSize := 0;
  with BitFile do
  begin
    bfSize := PalSize +
              Sizeof(TBitmapFileHeader) +  {Color map info}
	            Sizeof(TBitmapInfoHeader) +
              (((Depth * LineWidth) * Bits) Div 8);
    bfReserved1 := 0; {not currently used}
    bfReserved2 := 0; {not currently used}
    bfOffBits := PalSize +
                 Sizeof(TBitmapFileHeader)+
                 Sizeof(TBitmapInfoHeader);
  end;
  with Stream do
  begin                                     {Write the file header}
    Position := 0;
    ch := 'B';
    Write(ch, 1);
    ch := 'M';
    Write(ch, 1);
    Write(BitFile.bfSize,sizeof(BitFile.bfSize));
    Write(BitFile.bfReserved1,sizeof(BitFile.bfReserved1));
    Write(BitFile.bfReserved2,sizeof(BitFile.bfReserved2));
    Write(BitFile.bfOffBits,sizeof(BitFile.bfOffBits));
    Write(BmHeader,sizeof(BmHeader));       {Write the bitmap image header info}
    If Bits = 1 Then
      Write(#0#0#0#0#255#255#255#0, 8)
    Else If Bits = 4 Then
      for i := 0 to 15 do                     {Write the BGR palete inforamtion to this file}
      begin
        Write(Palette[i].Blue,1);
        Write(Palette[i].Green,1);
        Write(Palette[i].Red,1);
        Write(ch,1); {Bogus palete entry required by windows}
      end
    Else If Bits = 8 Then
      for i := 0 to 255 do                     {Write the BGR palete inforamtion to this file}
      begin
        Write(Palette[i].Blue,1);
        Write(Palette[i].Green,1);
        Write(Palette[i].Red,1);
        Write(ch,1); {Bogus palete entry required by windows}
      end;
    ch := chr(0);
    for i := (ImageLines.Count - 1) downto 0  do
    begin
      p := ImageLines.Items[i];
      x := LineWidth;
      Write(p^.LineData, x);
      while (x and 3) <> 0 do { Pad up to 4-byte boundary with zeroes }
      begin
        Inc(x);
        Write(ch, 1);
      end;
    end;
    Position := 0; { reset mewmory stream}
    for i := (ImageLines.Count - 1) downto 0  do
    begin
      p := ImageLines.Items[i];
      Dispose(P);
    End;
  end;
end;

end.
