unit FileVDlg;

interface

{                                 Programmed by

                                  Andrea Molino
                                easytarg@mbox.vol.it

  TFileListViewDlg
  TFileListPopupMenu
  TDirectoryComboDlg
  TFileNameEditDlg

  In this file you can found of some components to implement an "Enhanced Open Dialog".
  You can try to understand as they work and excuse me: I have not enough time to
  document my work.

  Fell Free of use this code as you want, but if you gain some money with it, remember of me!

  Please send me any suggestion and bug report.
}

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, Menus,
  ComCtrls, CommCtrl, StdCtrls, ShlObj, ShellAPI, Ole2;

Const
  DefaultMask = '*.*';
  FullFileTypes = FILE_ATTRIBUTE_READONLY Or FILE_ATTRIBUTE_HIDDEN Or FILE_ATTRIBUTE_SYSTEM;
  Slashes: array [False..True] of PChar = ('','\');
  SysFoldid: Array [0..2] Of Integer = (CSIDL_DRIVES, CSIDL_NETWORK, CSIDL_BITBUCKET);
  SysFoldNm: Array [0..2] Of String = ('Drives', 'NetWork', 'Recycled');

Type

  TFileListPopupMenu = Class;
  TDirectoryComboDlg = Class;
  TFileNameEditDlg = Class;

  TFileListViewAttr = (flvReadOnly, flvHidden, flvSystem);
  TFileListViewType = Set Of TFileListViewAttr;

  TSortedBy = (sbName, sbSize, sbType, sbDate, sbAttr, sbNameInv, sbSizeInv, sbTypeInv, sbDateInv, sbAttrInv);

  TFileListViewViewChangeEvent = Procedure(Sender: TObject; Style: TViewStyle) of object;
  TFileListViewDirChangeEvent = Procedure(Sender: TObject; OldDir, Directory: String; CanPrev: Boolean) of object;
  TFileListViewFileSelectEvent = Procedure(Sender: TObject; FileName: String) of object;
  TFileListViewDropEvent = Procedure(Files: TStrings; X, Y: Integer) of object;

  TFileListViewDlg = Class(TCustomListView)
  Private
    FPopupMenu: TFileListPopupMenu;
    FDirCombo: TDirectoryComboDlg;
    FFileEdit: TFileNameEditDlg;
    FDirectory: String;
    FMask: String;
    FFileViewType: TFileListViewType;
    FSortedBy: TSortedBy;
    FReadEnabled: Boolean;
    FNetWorkEnabled: Boolean;
    FResolveLink: Boolean;
    FExecuteOpen: Boolean;
    FileImageList: TImageList;
    FileLargeImageList: TImageList;
    FLastColumnSorted: Integer;
    FDirection: Boolean;
    OldName, OldCapt: String;
    FileDroppedList: TStrings;
    FOnViewChange: TFileListViewViewChangeEvent;
    FOnDirectoryChange: TFileListViewDirChangeEvent;
    FOnFileSelect: TFileListViewFileSelectEvent;
    FOnFileDrop: TFileListViewDropEvent;
    Function  GetDrive: char;
    Function  GetFileName: string;
    Function  GetFilePath: string; virtual;
    Function  GetViewIconStyle: TViewStyle;
    Procedure SetDrive(Value: char);
    Procedure SetDirCombo(Value: TDirectoryComboDlg);
    Procedure SetFileEdit(Value: TFileNameEditDlg);
    Procedure SetFileName(NewFile: string);
    Procedure SetFileViewType(Value: TFileListViewType);
    Procedure SetDirectory(NewDirectory: string);
    Procedure SetMask(NewMask: string);
    Procedure SetSortedBy(Value: TSortedBy);
    Procedure SetViewIconStyle(Value: TViewStyle);
    Procedure WMDROPFILES(var Message: TWMDropFiles); message WM_DROPFILES;
  Protected
    Procedure CreateWnd; Override;
    Procedure Loaded; override;
    Procedure Change(Item: TListItem; Change: Integer); Override;
    Procedure ColClick(Column: TListColumn); Override;
    Procedure DblClick; Override;
    Procedure KeyDown(var Key: Word; Shift: TShiftState); Override;
    Function  CanEdit(Item: TListItem): Boolean; Override;
    Procedure Edit(const Item: TLVItem); Override;
    Procedure Notification(AComponent: TComponent; Operation: TOperation); Override;
    Procedure ReadSpecialFolder(idCont, idRoot: PItemIDList; Fold: Integer; UseItem, UseDisp: Boolean);
    Procedure ReadDeskTopFolder;
    Procedure ReadSystemResource;
    Procedure ReadNetworkResource(idRoot: PItemIDList);
    Procedure ReadRecycledBin;
    Function  ReadDirectoryFiles(FileMask: string; Attributes: DWORD): Boolean;
    Procedure ReadFileNames; virtual;
    Procedure SortBy(SType: TSortedBy);
    Function  IsDirReadOny: Boolean;
    Function  IsSysFold: Boolean;
    Function  IsDeskLink: Boolean;
    Function  IsDir: Boolean;
    Function  IsUnit: Boolean;
    Function  IsNetCont: Boolean;
    Function  GetDriveTypeStr(Root: String): String;
    Procedure ShellOpenEx(Com, Par: String);
    Procedure SendToPath(Path: String);
  public
    Constructor Create(AOwner: TComponent); Override;
    Destructor Destroy; Override;
    Procedure ReRead;
    Function  PrevDirectory: Boolean;
    Procedure Rename;
    Procedure DeleteFiles;
    Procedure NewFolder;
    Procedure SendTo(Path: String);
    Procedure ShellOpen;
    Function GetSelectedFiles(FList: TStrings): Integer;
    property Drive: char read GetDrive write SetDrive;
    property Directory: string read FDirectory write SetDirectory;
    property FileName: string read GetFilePath write SetFileName;
  published
    Property DirCombo: TDirectoryComboDlg Read FDirCombo Write SetDirCombo;
    Property FileEdit: TFileNameEditDlg Read FFileEdit Write SetFileEdit;
    Property FileViewType: TFileListViewType Read FFileViewType Write SetFileViewType
                                              Default [flvReadOnly, flvHidden, flvSystem];
    Property Mask: string Read FMask Write SetMask;
    Property SortedBy: TSortedBy read FSortedBy Write SetSortedBy Default sbName;
    Property ViewIconStyle: TViewStyle Read GetViewIconStyle Write SetViewIconStyle;
    Property NetWorkEnabled: Boolean Read FNetWorkEnabled Write FNetWorkEnabled Default False;
    Property ReadEnabled: Boolean Read FReadEnabled Write FReadEnabled Default False;
    Property ResolveLink: Boolean Read FResolveLink Write FResolveLink Default False;
    Property ExecuteOpen: Boolean Read FExecuteOpen Write FExecuteOpen Default False;
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragMode;
    property ReadOnly;
    property Font;
    property HideSelection;
    property ParentShowHint;
    property ShowHint;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property MultiSelect;
    property DragCursor;
    property OnViewChange: TFileListViewViewChangeEvent Read FOnViewChange Write FOnViewChange;
    property OnFileSelect: TFileListViewFileSelectEvent Read FOnFileSelect Write FOnFileSelect;
    property OnDirectoryChange: TFileListViewDirChangeEvent Read FOnDirectoryChange Write FOnDirectoryChange;
    property OnFileDrop: TFileListViewDropEvent read FOnFileDrop write FOnFileDrop;
    property OnClick;
    property OnDblClick;
    property OnChange;
    property OnChanging;
    property OnColumnClick;
    property OnCompare;
    property OnDeletion;
    property OnEdited;
    property OnEditing;
    property OnEnter;
    property OnExit;
    property OnInsert;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  End;

  TFileListPopupMenu = Class(TPopupMenu)
  Private
    FFileListView: TFileListViewDlg;
    SendToList: TStrings;
    Procedure SetFileListView(Value: TFileListViewDlg);
  Protected
    Function ExtNewItem(const ACaption: string; AShortCut: TShortCut;
                         AChecked, AEnabled, ARadio: Boolean;
                          AGroup: Integer; AOnClick: TNotifyEvent;
                           hCtx: Word; const AName: string): TMenuItem;
    Procedure GetSendTo;
    Procedure GeneralItemClick(Sender: TObject);
  Public
    Constructor Create(AOwner: TComponent); Override;
    Destructor Destroy; Override;
    Procedure Popup(X, Y: Integer); Override;
    Property FileListView: TFileListViewDlg Read FFileListView Write SetFileListView;
  End;

  TDirectoryComboDlg = Class(TCustomComboBox)
  Private
    FDirectory: String;
    DirLst: TStringList;
    FFileListView: TFileListViewDlg;
    ImageList: TImageList;
    DriveItem: Integer;
    FOnDirectoryChange: TFileListViewDirChangeEvent;
    Function  GetDrive: char;
    Procedure SetDrive(Value: char);
    procedure SetDirectory(NewDirectory: string);
    Procedure SetFileListView(Value: TFileListViewDlg);
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CNDrawItem(var Message: TWMDrawItem); message CN_DRAWITEM;
    procedure ResetItemHeight;
  protected
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure Click; override;
    procedure BuildList; virtual;
    Function  FindDirItem(Dir: String): Integer;
    procedure DisplayDir; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); Override;
    Destructor Destroy; Override;
    Property Drive: Char Read GetDrive Write SetDrive;
    Property Directory: String Read FDirectory Write SetDirectory;
  published
    property Color;
    property Ctl3D;
    Property FileListView: TFileListViewDlg Read FFileListView Write SetFileListView;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnDirectoryChange: TFileListViewDirChangeEvent Read FOnDirectoryChange Write FOnDirectoryChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

  TFileNameEditDlg = Class(TEdit)
  Private
    FFileListView: TFileListViewDlg;
    Procedure SetFileListView(Value: TFileListViewDlg);
  Protected
    Procedure KeyPress(var Key: Char); Override;
    Procedure Notification(AComponent: TComponent; Operation: TOperation); Override;
  Published
    Property FileListView: TFileListViewDlg Read FFileListView Write SetFileListView;
  End;

Function GetSHSpecialFolderInfo(Handle: THandle; Var Info: TSHFileInfo; Root: Integer): Boolean;
Function GetSHLinkedFileFromLink(Handle: THandle; LinkFileName: String): String;
Function GetSHRealPath(Handle: THandle; var Dest: string; Root: Integer): Boolean;
{Function GetSHRealPath(Handle: THandle; idRoot: PItemIDList): String;}
Function DoSHFileOp(Handle: THandle; Operation: UINT;
                      Items: TStrings; TargetDir: String; Var Aborted: Boolean): Boolean;
Function FileTimeToFileDateStr(FileTime: TFileTime): String;
Function SlashSep(Path, S: String): String;
Function DirectoryExists(Path: String): Boolean;

Function GetItemHeight(Font: TFont): Integer;
Function CustomSortProc(Item1, Item2: TListItem; ParamSort: integer): integer; stdcall;

Procedure Register;

implementation

uses Consts, ComStrs;


procedure Register;
Begin
  RegisterComponents('MyGold', [TFileListViewDlg]);
  RegisterComponents('MyGold', [TFileNameEditDlg]);
  RegisterComponents('MyGold', [TDirectoryComboDlg]);
End;

{ ******************* GENERAL FUNCTION **************************** }

Function DoSHFileOp(Handle: THandle; Operation: UINT;
                      Items: TStrings; TargetDir: String; Var Aborted: Boolean): Boolean;
Var
  I, Ret: Integer;
  FileOpDesc: TSHFileOpStruct;
  FileNames: String;
  {Source: PChar;}
  Target: Array [0..MAX_PATH] Of Char;
Begin
  Ret := 1;
  Aborted := False;
  FileNames := '';
  For I := 0 To Items.Count-1 Do
    FileNames := FileNames + Items[I] + #0;{ '|';}
  StrPCopy(Target, TargetDir + #0);
  {Source := StrAlloc(Length(FileNames) + 1);
  Try
    StrCopy(Source, PChar(FileNames + #0));
    For I := 0 To Length(FileNames) Do
      If Source[I] = '|' Then Source[I] := #0;}
  With FileOpDesc Do
  Begin
    Wnd := Handle;
    wFunc := Operation;
    pFrom := {Source;} PChar(FileNames + #0);
    pTo := Target;
    fFlags := FOF_ALLOWUNDO;
    hNameMappings := Nil;
    lpszProgressTitle := '';
  End;
  Try
    Ret := SHFileOperation(FileOpDesc);
    Aborted := FileOpDesc.fAnyOperationsAborted;
  Except
    Ret := 1;
  End;
  {Finally
    StrDispose(Source);
  End;}
  Result := (Ret = 0);
End;

Function GetSHSpecialFolderInfo(Handle: THandle; Var Info: TSHFileInfo; Root: Integer): Boolean;
Var
  idRoot: PItemIDList;
Begin
  Result := False;
  Try
    If SHGetSpecialFolderLocation(Handle, Root, idRoot) = NOERROR Then
    Begin
      SHGetFileInfo(PChar(idRoot), 0, Info, SizeOf(Info), SHGFI_PIDL Or
                     SHGFI_SYSICONINDEX Or SHGFI_SMALLICON Or
                     SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);
      Result := True;
    End;
  Except
  End;
End;

Function GetSHLinkedFileFromLink(Handle: THandle; LinkFileName: String): String;
Var
  Buffer: Array [0..2048] Of Char;
  FStream: TFileStream;
  P: Pchar;
  L: Integer;
Begin                  // This is a Shit Procedure but..... (Look Down)
  Result := '';
  FStream := TFileStream.Create(LinkFileName, fmOpenRead);
  L := FStream.Read(Buffer, SizeOf(Buffer));
  FStream.Free;
  P := Buffer;
  L := Integer(P) + L;
  While (Integer(P) < L) And (Result = '') Do
  Begin
    P := P + Length(StrPas(P)) + 1;
    If Pos(':\', StrPas(P)) = 3 Then Result := StrPas(P+1);
  End;
  If Result <> '' Then
    While (Integer(P) < L) Do
    Begin
      P := P + Length(StrPas(P)) + 1;
      If Pos('\\', StrPas(P)) > 0 Then Break
      Else If Pos(':\', StrPas(P)) = 2 Then
      Begin
        Result := StrPas(P);
        Break;
      End;
    End
  Else
  Begin
    P := Buffer;
    While (Integer(P) < L) Do
    Begin
      P := P + Length(StrPas(P)) + 1;
      If Pos('\\', StrPas(P)) = 1 Then Result := StrPas(P);
    End;
  End;
{Var                   // ....this won't work!! (If You can tell me Why!!)
  hres: HRESULT;
  psl: IShellLink;
  szGotPath : array[0..MAX_PATH-1] of char;
  szDescription : array[0..MAX_PATH-1] of char;
  wfd: TWin32FindData;
  ppf: IPersistFile;
  wsz: array[0..MAX_PATH-1] of WideChar;
Begin
 Result := '';
 If CoCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, psl) = S_OK Then
 Begin
   If psl.QueryInterface(IID_IPersistFile, ppf) = S_OK Then
   Begin
     MultiByteToWideChar(CP_ACP, 0,@LinkFileName[1],-1,wsz,MAX_PATH);
     If ppf.Load(wsz, STGM_READ) = S_OK Then
     Begin
       If psl.Resolve(Handle, SLR_ANY_MATCH) = S_OK Then
       Begin
         If psl.GetPath(szGotPath, MAX_PATH, wfd, SLGP_SHORTPATH) <> S_OK Then Exit;
         If psl.GetDescription(szDescription, MAX_PATH) <> S_OK Then Exit;
         // Result := StrPas(szGotPath) + '|' + StrPas(szDescription);
         Result := StrPas(szGotPath);
       End;
     End;
     ppf.Release;
   End;
   psl.Release;
 End;}
End;

Function GetSHRealPath(Handle: THandle; Var Dest: String; Root: Integer): Boolean;
var
  ShellMalloc: IMALLOC;
  shBuff: PChar;
  idRoot: PItemIDList;
Begin
  Result := False;
  SetLength(Dest, MAX_PATH);  // Make sure their will be enough room in dest.
  if SHGetMalloc(ShellMalloc) = NOERROR then
  Begin
    try
      shBuff := PChar(ShellMalloc.Alloc(MAX_PATH)); // Shell allocate buffer.
      if assigned(shBuff) then
      Begin
        try
          SHGetSpecialFolderLocation(Handle, Root, idRoot);
          If SHGetPathFromIDList(idRoot, shBuff) Then
          Begin
            Dest := shBuff;
            Result := True;
          End;
        Except
          ShellMalloc.Free(idRoot); // Clean up after ourselves
        End;
      End;
    finally
      ShellMalloc.Release; // Clean-up.
    End;
  End;
End;

{Function GetSHRealPath(Handle: THandle; idRoot: PItemIDList): String;
var
  ShellMalloc: IMALLOC;
  shBuff: PChar;
  Dest: String;
Begin
  Dest := '';
  SetLength(Dest, MAX_PATH);  // Make sure their will be enough room in dest.
  If SHGetMalloc(ShellMalloc) = NOERROR then
  try
    shBuff := PChar(ShellMalloc.Alloc(MAX_PATH)); // Shell allocate buffer.
    if assigned(shBuff) then
    try
      If SHGetPathFromIDList(idRoot, shBuff) Then Dest := shBuff;
    Except
      ShellMalloc.Free(idRoot); // Clean up after ourselves
    End;
  finally
    ShellMalloc.Release; // Clean-up.
  End;
  Result := Dest;
End;}

Function FileTimeToFileDateStr(FileTime: TFileTime): String;
Var
  LocFTime: TFileTime;
  SysFTime: TSystemTime;
  DateStr: String;
Begin
  FileTimeToLocalFileTime(FileTime, LocFTime);
  FileTimeToSystemTime(LocFTime, SysFTime);
  With SysFTime Do
    DateStr := Format('%.2d/%.2d/%4d   %.2d.%.2d.%.2d', [wDay, wMonth, wYear, wHour, wMinute, wSecond]);
  Result := DateTimeToStr(StrToDateTime(DateStr));
End;

Function SlashSep(Path, S: String): String;
Begin
  If Path = '' Then Result := ''
  Else Result := Format('%s%s%s',[Path, Slashes[Path[Length(Path)] <> '\'], S]);
End;

Function DirectoryExists(Path: String): Boolean;
Var
  FileInfo: TWin32FindData;
  FindFileH: THandle;
  Aux: DWORD;
Begin
  If Pos('\\', Path) = 1 Then Result := True
  Else
  Begin
    Result := False;
    FindFileH := FindFirstFile(PChar(Path), FileInfo);
    If FindFileH <> INVALID_HANDLE_VALUE then
    Begin
      Windows.FindClose(FindFileH);
      If (FileInfo.dwFileAttributes And FILE_ATTRIBUTE_DIRECTORY) > 0 Then
        Result := True;
    End;
    Result := Result Or GetDiskFreeSpace(PChar(Path), Aux, Aux, Aux, Aux);
  End;
End;

Function GetItemHeight(Font: TFont): Integer;
var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  Result := Metrics.tmHeight;
end;

Function CustomSortProc(Item1, Item2: TListItem; ParamSort: integer): integer; stdcall;

  Function NumbToIntDef(SNumb: String; DefVal: LongInt): LongInt;
  Var
    I: Integer;
    Aux: String;
  Begin
    Aux := '';
    For I := 1 To Length(SNumb) Do
    Begin
      If SNumb[I] = ' ' Then Break;
      If SNumb[I] <> ThousandSeparator Then Aux := Aux + SNumb[I];
    End;
    Result := StrToIntDef(Aux, DefVal);
  End;

Begin
  Result := 0;
  If (TListItem(Item1).SubItems[3][1] = 'X') And (TListItem(Item2).SubItems[3][1] <> 'X') Then Result := -1
  Else If (TListItem(Item2).SubItems[3][1] = 'X') And (TListItem(Item1).SubItems[3][1] <> 'X') Then Result := 1
  Else If (TListItem(Item1).SubItems[3][1] = 'd') And (TListItem(Item2).SubItems[3][1] <> 'd') Then Result := -1
  Else If (TListItem(Item2).SubItems[3][1] = 'd') And (TListItem(Item1).SubItems[3][1] <> 'd') Then Result := 1
  Else
    Case ParamSort Of
      1, 6:  Result := lstrcmp(PChar(TListItem(Item1).Caption),
                                PChar(TListItem(Item2).Caption));
      2, 7:  Result := NumbToIntDef(TListItem(Item1).SubItems[0], -1) -
                        NumbToIntDef(TListItem(Item2).SubItems[0], -1);
      3, 8:  Result := lstrcmp(PChar(TListItem(Item1).SubItems[1]),
                                PChar(TListItem(Item2).SubItems[1]));
      4, 9:   If StrToDateTime(TListItem(Item1).SubItems[2]) <
                  StrToDateTime(TListItem(Item2).SubItems[2]) Then Result := 1
              Else Result := -1;
      5, 10: Result := lstrcmp(PChar(TListItem(Item1).SubItems[3]),
                                PChar(TListItem(Item2).SubItems[3]));
    End;
  If ParamSort > 5 Then Result := Result * (-1);
End;


{ ************************** CREATE / DESTROY ***************************** }

Constructor TFileListViewDlg.Create(AOwner: TComponent);
Var
  Sfi: TShfileInfo;
Begin
  inherited Create(AOwner);
  FileImageList := TImageList.Create(Self);
  FileLargeImageList := TImageList.Create(Self);
  Try
    FileImageList.Handle :=
      SHGetFileInfo('', 0, Sfi, SizeOf(TShfileInfo),
        SHGFI_SYSICONINDEX Or SHGFI_SMALLICON);
    FileImageList.ShareImages := True;
    SmallImages := FileImageList;

    FileLargeImageList.Handle :=
      SHGetFileInfo('', 0, Sfi, SizeOf(TShfileInfo),
        SHGFI_SYSICONINDEX Or SHGFI_LARGEICON);
    FileLargeImageList.ShareImages := True;
    LargeImages := FileLargeImageList;
  except;
  End;
  FileDroppedList := TStringList.Create;
  FPopupMenu := TFileListPopupMenu.Create(Self);
  FPopupMenu.AutoPopup := True;
  FPopupMenu.FileListView := Self;
  PopupMenu := FPopupMenu;
  GetDir(0, FDirectory); { initially use current dir on default drive }
  FMask := DefaultMask;  { default file mask is all }
  MultiSelect := False;
  ViewStyle := vsList;
  FSortedBy := sbName;
  FReadEnabled := False;
  FNetWorkEnabled := False;
  FResolveLink := False;
  FExecuteOpen := False;
  FLastColumnSorted := -1;
  FDirection := False;
  FOnViewChange := Nil;
  FOnDirectoryChange := Nil;
  FOnFileSelect := Nil;
  FOnFileDrop := Nil;
  FFileViewType := [flvReadOnly, flvHidden, flvSystem];
End;

Destructor TFileListViewDlg.Destroy;
Begin
  FPopupMenu.Free;
  FileImageList.Free;
  FileLargeImageList.Free;
  FileDroppedList.Free;
  Inherited Destroy;
End;


{ **************************** PRIVATE ********************************** }

Function TFileListViewDlg.GetDrive: char;
Begin
  If FDirectory = '' Then Result := ' '
  Else Result := UpCase(FDirectory[1]);
End;

function TFileListViewDlg.GetFileName: string;
Begin
  Result := '';
  If (Selected <> Nil) And (Not (IsSysFold Or IsUnit Or IsDir Or IsNetCont)) Then
  Begin
    If IsDeskLink And FResolveLink Then
      Result := ExtractFileName(Selected.SubItems[5])
    Else Result := ExtractFileName(Selected.SubItems[4]);
  End;
End;

Function TFileListViewDlg.GetFilePath: string;
Begin
  Result := '';
  If IsDeskLink And FResolveLink Then
    Result := Selected.SubItems[5]
  Else If Selected <> Nil Then
    Result := SlashSep(FDirectory, Selected.SubItems[4]);
End;

procedure TFileListViewDlg.SetDrive(Value: char);
Var
  Aux: DWORD;
Begin
  if (UpCase(Value) <> Drive) Then
  Begin
    If GetDiskFreeSpace(PChar(Value + ':\'), Aux, Aux, Aux, Aux) Then
    Begin
      FDirectory := Format('%s:\', [Value]);
      ReRead;
    End;
  End;
End;

procedure TFileListViewDlg.SetDirectory(NewDirectory: string);
Var
  OldDir: String;
Begin
  If AnsiCompareText(NewDirectory, FDirectory) <> 0 then
  Begin
    OldDir := FDirectory;
    If (NewDirectory = '\\') Then NewDirectory := 'NETWORK';
    If (UpperCase(NewDirectory) = 'DESKTOP')
      Or (UpperCase(NewDirectory) = 'DRIVES')
        Or (UpperCase(NewDirectory) = 'NETWORK')
         Or (UpperCase(NewDirectory) = 'RECYCLED') Then
    Begin
      FDirectory := NewDirectory;
      ReRead;
    End
    Else If Length(NewDirectory) > 2 Then
      Begin
        If DirectoryExists(NewDirectory) Then
        Begin
          FDirectory := NewDirectory;
          ReRead;
        End;
      End
    Else If Length(NewDirectory) > 0 Then
      SetDrive(NewDirectory[1]);
    If Assigned(FOnDirectoryChange) Then
      FOnDirectoryChange(Self, OldDir, FDirectory, (UpperCase(FDirectory) <> 'DESKTOP'));
    If FDirCombo <> Nil Then
      If FdirCombo.Directory <> FDirectory Then FdirCombo.Directory := FDirectory;
  End;
End;

procedure TFileListViewDlg.SetFileName(NewFile: string);
Var
  FPath: String;
Begin
  If (AnsiCompareText(NewFile, GetFileName) <> 0) And FileExists(NewFile) Then
  Begin
    FPath := ExtractFileDir(NewFile);
    If FPath <> '' Then SetDirectory(FPath);
    Application.ProcessMessages;
    Selected := FindCaption(0, NewFile, True, True, True);
  End;
End;

Procedure TFileListViewDlg.SetDirCombo(Value: TDirectoryComboDlg);
Begin
  FDirCombo := Value;
  If FDirCombo <> nil then
  Begin
    FDirCombo.FreeNotification(Self);
    FDirCombo.Directory := FDirectory;
  End;
End;

procedure TFileListViewDlg.SetFileEdit(Value: TFileNameEditDlg);
Begin
  FFileEdit := Value;
  If FFileEdit <> nil then
  Begin
    FFileEdit.FreeNotification(Self);
    FFileEdit.Text := GetFileName;
  End;
End;

procedure TFileListViewDlg.SetFileViewType(Value: TFileListViewType);
Begin
  if Value <> FFileViewType then
  Begin
    FFileViewType := Value;
    ReRead;
  End;
End;

procedure TFileListViewDlg.SetMask(NewMask: string);
Begin
  if FMask <> NewMask then
  Begin
    FMask := NewMask;
    ReRead;
  End;
End;

procedure TFileListViewDlg.SetSortedBy(Value: TSortedBy);
Begin
  If Value <> FSortedBy Then
  Begin
    FSortedBy := Value;
    SortBy(Value);
  End;
End;

procedure TFileListViewDlg.SetViewIconStyle(Value: TViewStyle);
Begin
  If Value <> ViewStyle Then
  Begin
    ViewStyle := Value;
    SortBy(FSortedBy);
    If Assigned(FOnViewChange) Then FOnViewChange(Self, ViewStyle);
  End;
End;

Function TFileListViewDlg.GetViewIconStyle: TViewStyle;
Begin
  Result := ViewStyle
End;

Procedure TFileListViewDlg.WMDROPFILES(var Message: TWMDropFiles);
Var
  I, DropCount, BufSize: integer;
  FileName: PChar;
  Point: TPoint;
  Abort: Boolean;
Begin
  BufSize := 0;
  FileDroppedList.Clear;
  DropCount := DragQueryFile(Message.Drop, $FFFFFFFF, Nil, BufSize);
  Try
    For I := 0 to DropCount - 1 Do
    Begin
      BufSize := DragQueryFile(Message.Drop, I, Nil, BufSize) + 1;
      FileName := StrAlloc(BufSize + 1);
      Try
        DragQueryFile(Message.Drop, I, FileName, BufSize);
        FileDroppedList.Add(FileName);
        DragQueryPoint(Message.Drop, Point);
      Finally
        StrDispose(FileName);
      End;
    End;
    DragFinish(Message.Drop);
    If DropCount > 0 Then
    Begin
      If Not IsDirReadOny Then
      Begin
        DoSHFileOp(Parent.Handle, FO_COPY, FileDroppedList, FDirectory, Abort);
        ReRead;
      End;
      If Assigned(FOnFileDrop) Then FOnFileDrop(FileDroppedList, Point.X, Point.Y);
    End;
  Finally
  End;
End;


{ ********************** PROTECTED OVERRIDE ******************************* }

procedure TFileListViewDlg.CreateWnd;
Var
  I: Integer;
Begin
  Inherited CreateWnd;
  Columns.Clear;
  For I := 0 To 4 Do Columns.Add;
  Columns[0].Caption := 'Name';
  Columns[1].Caption := 'Size';
  Columns[2].Caption := 'Type';
  Columns[3].Caption := 'Last Modify';
  Columns[4].Caption := 'Attr';
  Columns[0].Width := 135;
  Columns[1].Width := 60;
  Columns[2].Width := 125;
  Columns[3].Width := 120;
  Columns[4].Width := 50;
  Columns[1].Alignment := taRightJustify;
  Columns[2].Alignment := taLeftJustify;
  Columns[3].Alignment := taLeftJustify;
  Columns[4].Alignment := taRightJustify;

  ColumnClick := True;
  ShowColumnHeaders := True;
  IconOptions.Arrangement := iaTop;
  IconOptions.AutoArrange := True;
  If Assigned(FOnDirectoryChange) Then
    FOnDirectoryChange(Self, '', FDirectory, (UpperCase(FDirectory) <> 'DESKTOP'));
  If FDirCombo <> Nil Then
    If FdirCombo.Directory <> FDirectory Then FdirCombo.Directory := FDirectory;
  ReRead;
End;

Procedure TFileListViewDlg.Loaded;
Begin
  Inherited Loaded;
  If Not (csDesigning In ComponentState) Then
    DragAcceptFiles(Handle, True);
End;

procedure TFileListViewDlg.Change(Item: TListItem; Change: Integer);
Begin
  Inherited Change(Item, Change);
  If (FFileEdit <> Nil) And (GetFileName <> '') Then
    FileEdit.Text := GetFileName;
End;

procedure TFileListViewDlg.ColClick(Column: TListColumn);
Var
  Value: TSortedBy;
  I: Integer;
  SaveCursor: TCursor;
Begin
  Inherited ColClick(Column);
  If Column.Index = FLastColumnSorted Then
    FDirection := Not FDirection
  Else
  Begin
    FLastColumnSorted := Column.Index;
    FDirection := False;
  End;
  If FDirection Then
    I := Column.Index + 6
  Else I := Column.Index + 1;
  FSortedBy := TSortedBy(I-1);

  SaveCursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  CustomSort(@CustomSortProc, I);
  Screen.Cursor := SaveCursor;
End;

procedure TFileListViewDlg.DblClick;
Var
  Drv: Char;
Begin
  Inherited DblClick;
  If IsSysFold Then
    Directory := Selected.SubItems[4]
  Else If IsDeskLink Then
  Begin
    Directory := Selected.SubItems[5];
    If Assigned(FOnFileSelect) Then FOnFileSelect(Self, GetFilePath);
  End
  Else If IsDir Then
  Begin
    If UpperCase(FDirectory) <> 'NETWORK' Then
      Directory := SlashSep(FDirectory, Selected.SubItems[4])
    Else Directory := Selected.SubItems[4];
  End
  Else If IsUnit Then
    Directory := Selected.SubItems[4]
  Else If IsNetCont Then
    ReadNetworkResource(Selected.Data)
  Else If FExecuteOpen Then
    ShellOpen
  Else If Assigned(FOnFileSelect) Then
    FOnFileSelect(Self, GetFilePath);
End;

procedure TFileListViewDlg.KeyDown(var Key: Word; Shift: TShiftState);
Begin
  Case Key of
    VK_BACK   : If not IsEditing Then PrevDirectory;
    VK_RETURN : If not IsEditing Then DblClick;
    VK_DELETE : If not IsEditing Then DeleteFiles;
    VK_F5     : If not IsEditing Then ReRead;
  End;
  Inherited KeyDown(Key, Shift);
End;

Function TFileListViewDlg.CanEdit(Item: TListItem): Boolean;
Begin
  Result := False;
  If Not IsDirReadOny Then
  Begin
    OldName := SlashSep(FDirectory, Item.SubItems[4]);
    OldCapt := Item.Caption;
    Result := Inherited CanEdit(Item);
  End;
End;

Procedure TFileListViewDlg.Edit(const Item: TLVItem);
Var
  NewName, Ext: String;
  Abort: Boolean;
Begin
  Inherited Edit(Item);
  If Selected <> Nil Then
  Begin
    Ext := ExtractFileExt(Selected.SubItems[4]);
    NewName := SlashSep(FDirectory, Selected.Caption);
    If OldCapt+Ext = Selected.SubItems[4] Then NewName := NewName + Ext;
    If Selected.Caption = '' Then
      Selected.Caption := OldCapt
    Else If UpperCase(NewName) <> UpperCase(OldName) Then
    Begin
      FileDroppedList.Clear;
      FileDroppedList.Add(OldName);
      If Not DoSHFileOp(Parent.Handle, FO_RENAME, FileDroppedList, NewName, Abort) Then
        Selected.Caption := OldCapt
      Else
      Begin
        Items.Delete(Selected.Index);
        If Not ReadDirectoryFiles(NewName, FullFileTypes) Then
          ReadDirectoryFiles(NewName, FullFileTypes Or FILE_ATTRIBUTE_DIRECTORY);
        SortBy(FSortedBy);
      End;
    End;
  End;
End;

{ *************************** PROTECTED ********************************** }

Procedure TFileListViewDlg.Notification(AComponent: TComponent; Operation: TOperation);
Begin
  Inherited Notification(AComponent, Operation);
  if (Operation = opRemove) Then
  Begin
    If (AComponent = FFileEdit) Then FFileEdit := Nil;
    If (AComponent = FDirCombo) Then FDirCombo := Nil;
  End;
End;

Procedure TFileListViewDlg.SortBy(SType: TSortedBy);
Var
  I: Integer;
  SaveCursor: TCursor;
Begin
  I := Ord(SType) + 1;
  SaveCursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  CustomSort(@CustomSortProc, I);
  Screen.Cursor := SaveCursor;
End;

Function TFileListViewDlg.IsDirReadOny: Boolean;
Begin
  Result := (UpperCase(FDirectory) = 'DESKTOP') Or
             (UpperCase(FDirectory) = 'DRIVES') Or
              (UpperCase(FDirectory) = 'NETWORK') Or
               (UpperCase(FDirectory) = 'RECYCLED');
End;

Function TFileListViewDlg.IsSysFold: Boolean;
Begin
  If Selected <> Nil Then
    Result := (Selected.SubItems[3][1] = 'X')
  Else Result := False;
End;

Function TFileListViewDlg.IsDeskLink: Boolean;
Begin
  If Selected <> Nil Then
    Result := (Selected.SubItems[3][1] = 'L')
  Else Result := False;
End;

Function TFileListViewDlg.IsDir: Boolean;
Begin
  If Selected <> Nil Then
    Result := Selected.SubItems[3][1] = 'd'
  Else Result := False;
End;

Function TFileListViewDlg.IsUnit: Boolean;
Begin
  If Selected <> Nil Then
    Result := Selected.SubItems[3][1] = 'U'
  Else Result := False;
End;

Function TFileListViewDlg.IsNetCont: Boolean;
begin
  If Selected <> Nil Then
    Result := Selected.SubItems[3][1] = 'N'
  Else Result := False;
end;

Function TFileListViewDlg.GetDriveTypeStr(Root: String): String;
Var
  DrvType: Integer;
Begin
  DrvType := GetDriveType(PChar(Root));
  Case DrvType Of
    0              : Result := 'Unknown';
    1              : Result := 'Not exist';
    DRIVE_REMOVABLE: Result := 'Removable Unit';
    DRIVE_FIXED	   : Result := 'Fixed Unit';
    DRIVE_REMOTE	 : Result := 'Network Unit';
    DRIVE_CDROM	   : Result := 'CD-ROM Unit';
    DRIVE_RAMDISK	 : Result := 'RAM Disk Unit';
  End;
End;

Procedure TFileListViewDlg.ShellOpenEx(Com, Par: String);
Var
  Comnd, Param: Array [0..MAX_PATH] Of Char;
Begin
  If Selected <> Nil Then
  Begin
    StrPCopy(Comnd, Com);
    StrPCopy(Param, Par);
    ShellExecute(Parent.Handle, PChar('Open'), Comnd, Param, PChar(''), SW_SHOW);
  End;
End;

Procedure TFileListViewDlg.SendToPath(Path: String);
Var
  Abort: Boolean;
  I: Integer;
Begin
  ChDir(FDirectory);
  FileDroppedList.Clear;
  For I := 0 To Items.Count-1 Do
    If Items[I].Selected Then
      FileDroppedList.Add(Items[I].SubItems[4]);
  If FileDroppedList.Count > 0 Then
    DoSHFileOp(Parent.Handle, FO_COPY, FileDroppedList, Path, Abort);
End;

Procedure TFileListViewDlg.ReadSpecialFolder(idCont, idRoot: PItemIDList; Fold: Integer; UseItem, UseDisp: Boolean);
Var
  ShFold: ISHELLFOLDER;
  EnumList: IEnumIDList;
  ShPoint: Pointer;
  idItem: PItemIDList;
  Fetched: ULONG;
  Attrib, Aux: UINT;
  sPath, sDisp: TStrRet;
  Info: TSHFileInfo;
  NewItem: TListItem;
  FileInfo: TWin32FindData;
  FindFileH: THandle;
  FSize, FDate: String;
Begin
  Fetched := 0;
  Try
    If SHGetDesktopFolder(ShFold) = NOERROR Then
    Begin
      If idCont <> Nil Then
      Begin
        If ShFold.BindToObject(idCont, Nil, IID_IShellFolder, ShPoint) = NOERROR Then
          ShFold := ShPoint;
      End;
      If ShFold.BindToObject(idRoot, Nil, IID_IShellFolder, ShPoint) = NOERROR Then
        ShFold := ShPoint;
      If ShFold.EnumObjects(Parent.Handle, Fold Or SHCONTF_NONFOLDERS Or SHCONTF_INCLUDEHIDDEN, EnumList) = NOERROR Then
      Begin
        EnumList.Reset;
        While EnumList.Next(1, idItem, Fetched) = NOERROR Do
        Begin
          ShFold.GetDisplayNameOf(idItem, SHGDN_FORPARSING, sPath);
          ShFold.GetDisplayNameOf(idItem, {*SHGDN_FORPARSING}{SHGDN_INFOLDER}SHGDN_NORMAL, sDisp);
          ShFold.GetAttributesOf(1, idItem, Attrib);

          NewItem := Items.Add;
          If UseItem Then
            SHGetFileInfo(PChar(idItem), 0, Info, SizeOf(Info), SHGFI_PIDL Or
                           SHGFI_SYSICONINDEX Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME)
          Else
            SHGetFileInfo({Pchar('\\ ')}sPath.cStr, 0, Info, SizeOf(Info),
                           SHGFI_SYSICONINDEX Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);

          FindFileH := FindFirstFile(sPath.cStr, FileInfo);
          If FindFileH <> INVALID_HANDLE_VALUE Then
          Begin
            FDate := FileTimeToFileDateStr(FileInfo.ftLastWriteTime);
            FSize := FormatFloat('###,###,###', (FileInfo.nFileSizeHigh * MAXDWORD) + FileInfo.nFileSizeLow);
            Windows.FindClose(FindFileH);
          End;

          If SmallImages <> Nil Then
            NewItem.ImageIndex := Info.Iicon;
          If UseDisp Then NewItem.Caption := StrPas(Info.szDisplayName)
          Else NewItem.Caption := StrPas(sDisp.cStr);
          NewItem.Data := idItem;
          NewItem.SubItems.Add(FSize);
          NewItem.SubItems.Add(StrPas(Info.szTypeName));
          NewItem.SubItems.Add(FDate);
          NewItem.SubItems.Add(IntToStr(Attrib));
          NewItem.SubItems.Add(StrPas(sPath.cStr));
        End;
      End;
    End;
  Except
  End;
End;

Procedure TFileListViewDlg.ReadDeskTopFolder;
Var
  Info: TSHFileInfo;
  NewItem: TListItem;
  idItem: PItemIDList;
  I: Integer;
  Link, Ext: String;
  Oldcur: Tcursor;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  For I := 0 to 2 do
    If GetSHSpecialFolderInfo(Parent.Handle, Info, SysFoldid[I]) Then
    Begin
      NewItem := Items.Add;
      If SmallImages <> Nil Then
        NewItem.ImageIndex := Info.Iicon;
      NewItem.Caption := StrPas(Info.szDisplayName);
      NewItem.SubItems.Add('');
      NewItem.SubItems.Add(StrPas(Info.szTypeName));
      NewItem.SubItems.Add('');
      NewItem.SubItems.Add('X');
      NewItem.SubItems.Add(SysFoldNm[I]);
    End;
  If SHGetSpecialFolderLocation(Parent.Handle, CSIDL_DESKTOP, idItem) = NOERROR Then
  Begin
    ReadSpecialFolder(Nil, idItem, 0, True, True);
    For I := Items.Count-1 DownTo 0 Do
      If FileExists(Items[I].SubItems[4]) Then
      Begin
        Ext := UpperCase(ExtractFileExt(Items[I].SubItems[4]));
        If Ext = '.LNK' Then Link := GetSHLinkedFileFromLink(Handle, Items[I].SubItems[4])
        Else Link := '';
        If Link <> '' Then
        Begin
          Items[I].SubItems[3] := 'L';
          Items[I].SubItems.Add(Link);
        End
        Else If (Ext <> '.LNK') Then
        Begin
          Items[I].SubItems[3] := 'L';
          Items[I].SubItems.Add(Items[I].SubItems[4]);
        End
        Else Items.Delete(I);
      End
      Else If Items[I].SubItems[3] <> 'X' Then Items.Delete(I);
  End;
  Items.Endupdate;
  Screen.Cursor := Oldcur;
End;

Procedure TFileListViewDlg.ReadSystemResource;
Var
  MoreInfo: TSHFileInfo;
  Oldcur: Tcursor;
  NewItem: TListItem;
  Bits: Set Of 0..25;
  Ct: Integer;
  DSize: Integer;
  Drv: Char;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  Application.ProcessMessages;
  Integer(Bits) := GetLogicalDrives;
  For Ct := 0 To 25 Do
    If (Ct In Bits) Then
    Begin
      Drv := Char(Ct + Ord('A'));
      NewItem := Items.Add;
      SHGetFileInfo(PChar(Drv + ':\'), 0, MoreInfo, SizeOf(MoreInfo),
                      SHGFI_SYSICONINDEX Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);
      If SmallImages <> Nil Then
        NewItem.ImageIndex := MoreInfo.Iicon;
      NewItem.Caption := StrPas(MoreInfo.szDisplayName);
      DSize := DiskSize(Ct+1) Div 1024;
      If DSize > 10000 Then
        NewItem.SubItems.Add(FormatFloat('###,###,### Mb', DSize Div 1000))
      Else If DSize > 0 Then
        NewItem.SubItems.Add(FormatFloat('###,###,### Kb', DSize))
      Else NewItem.SubItems.Add('0 Kb');
      NewItem.SubItems.Add(GetDriveTypeStr(Drv + ':\'));
      NewItem.SubItems.Add('');
      NewItem.SubItems.Add('U');
      NewItem.SubItems.Add(Drv + ':\');
    End;
  Items.Endupdate;
  Screen.Cursor := Oldcur;
End;

Procedure TFileListViewDlg.ReadNetworkResource(idRoot: PItemIDList);
Var
  idItem: PItemIDList;
  Ret, I: Integer;
  Oldcur: Tcursor;
  NewItem : TListItem;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  If FNetWorkEnabled Then
  Begin
    Ret := SHGetSpecialFolderLocation(Parent.Handle, CSIDL_NETWORK, idItem);
    If (idRoot = Nil) And (Ret = NOERROR) Then
    Begin
      ReadSpecialFolder(Nil, idItem, SHCONTF_FOLDERS, False, False);
      For I := 0 To Items.Count-1 Do Items[I].SubItems[3] := 'N';
      Items[0].Caption := 'All The Network';
      Items[0].ImageIndex := 13;
    End
    Else If Ret = NOERROR Then
    Begin
      ReadSpecialFolder(idItem, idRoot, SHCONTF_FOLDERS, False, False);
      For I := 0 To Items.Count-1 Do
        If Pos('\\', Items[I].SubItems[4]) = 1 Then Items[I].SubItems[3] := 'd'
        Else Items[I].SubItems[3] := 'N';
    End;
  End
  Else
  Begin
    NewItem := Items.Add;
    NewItem.Caption := 'All The Network';
    NewItem.ImageIndex := 13;
    For I := 0 To 4 Do NewItem.SubItems.Add('');
    NewItem.SubItems[3] := 'N';
  End;
  Items.Endupdate;
  Screen.Cursor := Oldcur;
{var
  NetCont: TNetResource;
  PNetCnt: PNetResource;
  NetRes: Array [0..2] Of TNetResource;
  ResEnumH: THandle;
  Ret, ResCount, BufSize: DWORD;
  MoreInfo: TSHFileInfo;
  Oldcur: Tcursor;
  NewItem: TListItem;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  Application.ProcessMessages;
  BufSize := SizeOf(NetRes);
  If FDirectory = '\\' Then
    PNetCnt := Nil
  Else
  Begin
    With NetCont Do
    Begin
      dwScope := RESOURCE_CONNECTED;
      dwType := RESOURCETYPE_DISK;
      dwDisplayType := 0;
      dwUsage := 0;
      lpLocalName := '';
      lpRemoteName := PChar(FDirectory);
      lpComment := '';
      lpProvider := '';
    End;
    PNetCnt := @NetCont;
  End;
  If WNetOpenEnum(RESOURCE_GLOBALNET, RESOURCETYPE_DISK, 0, PNetCnt, ResEnumH) = NO_ERROR Then
  Begin
    ResCount := 1;
    While WNetEnumResource(ResEnumH, ResCount, @NetRes, BufSize) = NO_ERROR Do
    Begin
      ResCount := 1;
      With NetRes[0] Do
      Begin
        If lpRemoteName <> Nil Then
        Begin
          NewItem := Items.Add;
          SHGetFileInfo(PChar(lpRemoteName), 0, MoreInfo, SizeOf(MoreInfo),
                          shgfi_sysiconindex Or shgfi_smallicon Or
                          shgfi_displayname Or SHGFI_TYPENAME);
          If SmallImages <> Nil Then
            NewItem.ImageIndex := MoreInfo.Iicon;
          NewItem.Caption := StrPas(lpRemoteName);
          NewItem.SubItems.Add('0');
          NewItem.SubItems.Add(StrPas(MoreInfo.szTypeName));
          NewItem.SubItems.Add('');
          If dwUsage = RESOURCEUSAGE_CONTAINER Then
            NewItem.SubItems.Add('N')
          Else NewItem.SubItems.Add('d');
        End;
      End;
    End;
    WNetCloseEnum(ResEnumH);
  End;

  Items.Endupdate;
  Screen.Cursor := Oldcur;}
End;

Procedure TFileListViewDlg.ReadRecycledBin;
Var
  idItem: PItemIDList;
  I: Integer;
  Oldcur: Tcursor;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  If SHGetSpecialFolderLocation(Parent.Handle, CSIDL_BITBUCKET, idItem) = NOERROR Then
  Begin
    ReadSpecialFolder(Nil, idItem, 0, True, False);
    For I := 0 To Items.Count-1 Do
      Items[I].SubItems[3] := 'Deleted';
  End;
  Items.Endupdate;
  Screen.Cursor := Oldcur;
End;

Function TFileListViewDlg.ReadDirectoryFiles(FileMask: string; Attributes: DWORD): Boolean;
var
  NewItem : TListItem;
  Attrib: String;
  FileInfo: TWin32FindData;
  FindFileH: THandle;
  FullName: String;
  MoreInfo: TSHFileInfo;
  FName, FDate: String;
  FSize: Integer;
  VStyle: TViewStyle;
Begin
  Result := False;
  ChDir(FDirectory);
  Result := False;
  FindFileH := FindFirstFile(PChar(FileMask), FileInfo);
  If FindFileH <> INVALID_HANDLE_VALUE then
  Begin
    Repeat
      With FileInfo Do
        If ((dwFileAttributes And FILE_ATTRIBUTE_DIRECTORY) =
           (Attributes And FILE_ATTRIBUTE_DIRECTORY)) And
           ((dwFileAttributes And FILE_ATTRIBUTE_READONLY) <=
           (Attributes And FILE_ATTRIBUTE_READONLY)) And
           ((dwFileAttributes And FILE_ATTRIBUTE_HIDDEN) <=
           (Attributes And FILE_ATTRIBUTE_HIDDEN)) And
           ((dwFileAttributes And FILE_ATTRIBUTE_SYSTEM) <=
           (Attributes And FILE_ATTRIBUTE_SYSTEM)) Then
        Begin
          Result := True;
          FName := StrPas(cFileName);
          If (FName <> '.') And (FName <> '..') Then
          Begin
            FDate := FileTimeToFileDateStr(ftLastWriteTime);
            NewItem := Items.Add;
            FullName := SlashSep(FDirectory, FName);
            SHGetFileInfo(PChar(FullName), 0, MoreInfo, SizeOf(MoreInfo),
                            SHGFI_SYSICONINDEX Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);
            If SmallImages <> Nil Then
              NewItem.ImageIndex := MoreInfo.Iicon;

            If StrPas(MoreInfo.szDisplayName) = '' Then
            Begin
              NewItem.Caption := FName;
              NewItem.ImageIndex := 0;
            End
            Else NewItem.Caption := StrPas(MoreInfo.szDisplayName);

            FSize :=  (nFileSizeHigh * MAXDWORD) + nFileSizeLow;
            If FSize = 0 Then NewItem.SubItems.Add(' ')
            Else NewItem.SubItems.Add(FormatFloat('###,###,###', FSize));
            NewItem.SubItems.Add(StrPas(MoreInfo.szTypeName));
            NewItem.SubItems.Add(FDate);
            Attrib := '';
            If (dwFileAttributes And FILE_ATTRIBUTE_DIRECTORY) <> 0 Then Attrib := Attrib + 'd';
            If (dwFileAttributes And FILE_ATTRIBUTE_READONLY)  <> 0 Then Attrib := Attrib + 'r';
            If (dwFileAttributes And FILE_ATTRIBUTE_HIDDEN)    <> 0 Then Attrib := Attrib + 'h';
            If (dwFileAttributes And FILE_ATTRIBUTE_SYSTEM)    <> 0 Then Attrib := Attrib + 's';
            If (dwFileAttributes And FILE_ATTRIBUTE_ARCHIVE)   <> 0 Then Attrib := Attrib + 'a';
            Attrib := Attrib + ' ';
            NewItem.SubItems.Add(Attrib);
            NewItem.SubItems.Add(FName);

            If UpperCase(ExtractFileExt(FName)) = '.LNK' Then
              FName := GetSHLinkedFileFromLink(Handle, FullName)
            Else FName := '';
            If FName <> '' Then
            Begin
              NewItem.SubItems[3] := 'L';
              NewItem.SubItems.Add(FName);
            End
          End;
        End;
    Until Not FindNextFile(FindFileH, FileInfo);
    Windows.FindClose(FindFileH);
  End;
End;

Procedure TFileListViewDlg.ReadFileNames;
Var
  SaveCursor: TCursor;
  VStyle: TViewStyle;
  DirAtr, FilAtr: DWORD;
  MaskPtr: PChar;
  Ptr: PChar;
Begin
  {Perform( WM_SETREDRAW, 0, 0 );}
  VStyle := ViewStyle;
  If VStyle <> vsReport Then ViewStyle := vsList;
  Items.BeginUpDate;
  Items.Clear;
  SaveCursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  Try
    FilAtr := 0;
    If flvReadOnly In FileViewType Then FilAtr := FilAtr Or FILE_ATTRIBUTE_READONLY;
    If flvHidden   In FileViewType Then FilAtr := FilAtr Or FILE_ATTRIBUTE_HIDDEN;
    If flvSystem   In FileViewType Then FilAtr := FilAtr Or FILE_ATTRIBUTE_SYSTEM;
    DirAtr := FilAtr Or FILE_ATTRIBUTE_DIRECTORY;
    ReadDirectoryFiles(SlashSep(FDirectory, '*.*'), DirAtr);

    MaskPtr := PChar(FMask);
    While MaskPtr <> Nil Do
    Begin
      Ptr := StrScan(MaskPtr, ';');
      If Ptr <> Nil Then Ptr^ := #0;
      ReadDirectoryFiles(SlashSep(FDirectory, StrPas(MaskPtr)), FilAtr);
      If Ptr <> Nil Then
      Begin
        Ptr^ := ';';
        Inc (Ptr);
      End;
      MaskPtr := Ptr;
    End;
  Finally
    Screen.Cursor := SaveCursor;
  End;
  Items.EndUpDate;
  If VStyle <> vsReport Then ViewStyle := VStyle;
  SortBy(FSortedBy);
  {Perform( WM_SETREDRAW, 1, 0 );}
End;


{ *************************** PUBLIC ********************************** }

Procedure TFileListViewDlg.ReRead;
Begin
  If FReadEnabled Then
  Begin
    If UpperCase(FDirectory) = 'DESKTOP' Then ReadDeskTopFolder
    Else If UpperCase(FDirectory) = 'DRIVES' Then ReadSystemResource
    Else If UpperCase(FDirectory) = 'NETWORK' Then ReadNetworkResource(Nil)
    Else If UpperCase(FDirectory) = 'RECYCLED' Then ReadRecycledBin
    Else ReadFileNames;
  End;
End;

Function TFileListViewDlg.PrevDirectory: Boolean;
Var
  NewDir: String;
Begin
  If (UpperCase(FDirectory) = 'DRIVES') Or
      (UpperCase(FDirectory) = 'NETWORK') Or
       (UpperCase(FDirectory) = 'RECYCLED') Then NewDir := 'DeskTop'
  Else If UpperCase(FDirectory) <> 'DESKTOP' Then
  Begin
    NewDir := ExtractFileDir(FDirectory);
    If NewDir = FDirectory Then NewDir := 'Drives'
    Else If NewDir = '\\' Then NewDir := 'NetWork';
  End
  Else NewDir := FDirectory;

  If NewDir <> FDirectory Then Directory := NewDir;
  Application.ProcessMessages;
  Result := (UpperCase(FDirectory) <> 'DESKTOP');
End;

Procedure TFileListViewDlg.Rename;
Begin
  If Selected <> Nil Then
    Selected.EditCaption;
End;

Procedure TFileListViewDlg.DeleteFiles;
Var
  Abort: Boolean;
  I: Integer;
Begin
  If Not IsDirReadOny Then
  Begin
    FileDroppedList.Clear;
    For I := 0 To Items.Count-1 Do
      If Items[I].Selected Then
        FileDroppedList.Add(Items[I].SubItems[4]);
    DoSHFileOp(Parent.Handle, FO_DELETE, FileDroppedList, '', Abort);
    ReRead;
  End;
End;

Procedure TFileListViewDlg.NewFolder;
Var
  NewDir: String;
  I: Integer;
Begin
  If Not IsDirReadOny Then
  Begin
    I := 2;
    NewDir := SlashSep(FDirectory, 'New Folder');
    While DirectoryExists(NewDir) Do
    Begin
      NewDir := SlashSep(FDirectory, 'New Folder ' + IntToStr(I));
      Inc(I);
    End;
    CreateDir(NewDir);
    ReadDirectoryFiles(NewDir, FILE_ATTRIBUTE_DIRECTORY);
    Selected := Items[Items.Count-1];
    SetFocus;
    Application.ProcessMessages;
    Items[Items.Count-1].EditCaption;
  End;
End;
Procedure TFileListViewDlg.SendTo(Path: String);
Var
  FName: String;
Begin
  If Selected <> Nil then
  Begin
    If UpperCase(ExtractFileExt(Path)) = '.LNK' Then
    Begin
      FName := GetSHLinkedFileFromLink(Handle, Path);
      If FName <> '' Then
      Begin
        If DirectoryExists(FName) Then SendToPath(FName)
        Else ShellOpenEx(FName, GetFilePath);
      End;
    End
    Else If DirectoryExists(Path) Then SendToPath(Path)
  End;
End;

Procedure TFileListViewDlg.ShellOpen;
Begin
  If Selected <> Nil Then ShellOpenEx(GetFilePath, '');
End;

Function TFileListViewDlg.GetSelectedFiles(FList: TStrings): Integer;
Var
  I: Integer;
Begin
  FList.Clear;
  Result := 0;
  For I := 0 To Items.Count-1 Do
    With Items[I] Do
      If (Items[I].Selected)     And
         (SubItems[3][1] <> 'd') And
         (SubItems[3][1] <> 'U') And
         (SubItems[3][1] <> 'N') And
         (SubItems[3][1] <> 'X') Then
      Begin
        FList.Add(SlashSep(FDirectory, Caption));
        Result := Result + 1;
      End
End;


{ *************************** TFileNameEditDlg ****************************** }

Constructor TFileListPopupMenu.Create(AOwner: TComponent);
Begin
  Inherited Create(AOwner);
  SendToList := TStringList.Create;
  Items.Add(ExtNewItem('&View Style', 0, False, True, False, 0, Nil, 0, 'SubMenuView'));
    Items.Items[0].Add(ExtNewItem('&Large icons', 0, False, True, True, 1, GeneralItemClick, 0, 'ItemViewIcon'));
    Items.Items[0].Add(ExtNewItem('&Small Icons', 0, False, True, True, 1, GeneralItemClick, 0, 'ItemViewSmal'));
    Items.Items[0].Add(ExtNewItem('&List', 0, False, True, True, 1, GeneralItemClick, 0, 'ItemViewList'));
    Items.Items[0].Add(ExtNewItem('&Report', 0, False, True, True, 1, GeneralItemClick, 0, 'ItemViewRepr'));
  Items.Add(NewLine);
  Items.Add(ExtNewItem('&Sort Icons', 0, False, True, False, 0, Nil, 0, 'SubMenuSort'));
    Items.Items[2].Add(ExtNewItem('by &Name', 0, False, True, True, 2, GeneralItemClick, 0, 'ItemSortName'));
    Items.Items[2].Add(ExtNewItem('by &Size', 0, False, True, True, 2, GeneralItemClick, 0, 'ItemSortSize'));
    Items.Items[2].Add(ExtNewItem('by &Type', 0, False, True, True, 2, GeneralItemClick, 0, 'ItemSortType'));
    Items.Items[2].Add(ExtNewItem('by &Last Modify', 0, False, True, True, 2, GeneralItemClick, 0, 'ItemSortDate'));
    Items.Items[2].Add(ExtNewItem('by &Attributes', 0, False, True, True, 2, GeneralItemClick, 0, 'ItemSortAttr'));
  Items.Add(NewLine);
  Items.Add(ExtNewItem('Send &To', 0, False, True, False, 0, Nil, 0, 'SubMenuSend'));
  GetSendTo;
  Items.Add(NewLine);
  Items.Add(ExtNewItem('&Rename', 0, False, False, False, 0, GeneralItemClick, 0, 'ItemOpRename'));
  Items.Add(ExtNewItem('&Delete', 0, False, False, False, 0, GeneralItemClick, 0, 'ItemOpDelete'));
  Items.Add(ExtNewItem('&New Folder', 0, False, True, False, 0, GeneralItemClick, 0, 'ItemOpFolder'));
End;

Destructor TFileListPopupMenu.Destroy;
Begin
  SendToList.Free;
  Inherited Destroy;
End;

Procedure TFileListPopupMenu.SetFileListView(Value: TFileListViewDlg);
Begin
  FFileListView := Value;
End;

Function TFileListPopupMenu.ExtNewItem(const ACaption: string; AShortCut: TShortCut;
                                        AChecked, AEnabled, ARadio: Boolean;
                                         AGroup: Integer; AOnClick: TNotifyEvent;
                                          hCtx: Word; const AName: string): TMenuItem;
Begin
  Result := NewItem(ACaption, AShortCut, AChecked, AEnabled, AOnClick, hCtx, AName);
  With Result Do
  Begin
    RadioItem := ARadio;
    GroupIndex := AGroup;
  End;

End;

Procedure TFileListPopupMenu.GetSendTo;
Var
  SendToDir, FName, FullName: String;
  FileInfo: TWin32FindData;
  FindFileH: THandle;
  MoreInfo: TSHFileInfo;
  I: Integer;
  NewItem: TMenuItem;
Begin
  I := 0;
  SendToList.Clear;
  If GetSHRealPath(Handle, SendToDir, CSIDL_SENDTO) Then
  Begin
    FindFileH := FindFirstFile(PChar(SlashSep(SendToDir,'*.LNK')), FileInfo);
    If FindFileH <> INVALID_HANDLE_VALUE then
    Begin
      Repeat
        With FileInfo Do
        Begin
          FName := StrPas(cFileName);
          If (FName <> '.') And (FName <> '..') Then
          Begin
            FullName := SlashSep(SendToDir, FName);
            SHGetFileInfo(PChar(FullName), 0, MoreInfo, SizeOf(MoreInfo), SHGFI_DISPLAYNAME);
            NewItem := ExtNewItem(StrPas(MoreInfo.szDisplayName), 0,
                                False, True, True, 2, GeneralItemClick, 0, 'SendTo'+IntToStr(I));
            Items.Items[4].Add(NewItem);
            SendToList.Add(FullName);
            Inc(I);
          End;
        End;
      Until Not FindNextFile(FindFileH, FileInfo);
      Windows.FindClose(FindFileH);
    End;
  End;
End;

Procedure TFileListPopupMenu.GeneralItemClick(Sender: TObject);
Var
  Idx: Integer;
Begin
  If      TMenuItem(Sender).Name = 'ItemViewList' Then FFileListView.ViewIconStyle := vsList
  Else If TMenuItem(Sender).Name = 'ItemViewRepr' Then FFileListView.ViewIconStyle := vsReport
  Else If TMenuItem(Sender).Name = 'ItemViewSmal' Then FFileListView.ViewIconStyle := vsSmallIcon
  Else If TMenuItem(Sender).Name = 'ItemViewIcon' Then FFileListView.ViewIconStyle := vsIcon
  Else If TMenuItem(Sender).Name = 'ItemSortName' Then FFileListView.SortedBy := sbName
  Else If TMenuItem(Sender).Name = 'ItemSortSize' Then FFileListView.SortedBy := sbSize
  Else If TMenuItem(Sender).Name = 'ItemSortType' Then FFileListView.SortedBy := sbType
  Else If TMenuItem(Sender).Name = 'ItemSortDate' Then FFileListView.SortedBy := sbDate
  Else If TMenuItem(Sender).Name = 'ItemSortAttr' Then FFileListView.SortedBy := sbAttr
  Else If TMenuItem(Sender).Name = 'ItemOpRename' Then FFileListView.Rename
  Else If TMenuItem(Sender).Name = 'ItemOpDelete' Then FFileListView.DeleteFiles
  Else If TMenuItem(Sender).Name = 'ItemOpFolder' Then FFileListView.NewFolder
  Else If Pos('SendTo', TMenuItem(Sender).Name) = 1 Then
  Begin
    Idx := StrToIntDef(Copy(TMenuItem(Sender).Name, 7, 5), -1);
    If Idx > -1 Then FFileListView.SendTo(SendToList[Idx]);
  End;
End;

Procedure TFileListPopupMenu.Popup(X, Y: Integer);
Var
  SortType: Integer;
Begin
  SortType := Ord(FFileListView.SortedBy);
  If SortType > 4 Then SortType := SortType - 5;
  Items[6].Enabled := (FFileListView.Selected <> Nil);
  Items[7].Enabled := (FFileListView.Selected <> Nil);
  Items[0].Items[Ord(FFileListView.ViewIconStyle)].Checked := True;
  Items[2].Items[SortType].Checked := True;
  Inherited Popup(X, Y);
End;

{ *************************** TFileNameEditDlg ****************************** }

Procedure TFileNameEditDlg.SetFileListView(Value: TFileListViewDlg);
Begin
  FFileListView := Value;
  If FFileListView <> nil then
    FFileListView.FreeNotification(Self);
End;

Procedure TFileNameEditDlg.KeyPress(var Key: Char);
Begin
  Inherited KeyPress(Key);
  If Key = Char(VK_RETURN) Then
  Begin
    FFileListView.Mask := Text;
    Key := #0;
  End;
End;

Procedure TFileNameEditDlg.Notification(AComponent: TComponent; Operation: TOperation);
Begin
  Inherited Notification(AComponent, Operation);
  If (Operation = opRemove) And (AComponent = FFileListView) Then FFileListView := nil;
End;



{ *************************** TFileNameEditDlg ****************************** }

Constructor TDirectoryComboDlg.Create(AOwner: TComponent);
Var
  Sfi: TShfileInfo;
Begin
  Inherited Create(AOwner);
  DirLst := TStringList.Create;
  ImageList := TImageList.Create(Self);
  Try
    ImageList.Handle :=
      SHGetFileInfo('', 0, Sfi, SizeOf(TShfileInfo),
        SHGFI_SYSICONINDEX Or SHGFI_SMALLICON);
    ImageList.ShareImages := True;
    ImageList.BlendColor := clHighlight;
  except;
  End;
  GetDir(0, FDirectory);
  Style := csOwnerDrawFixed;
  DriveItem := -1;
  ResetItemHeight;
  FOnDirectoryChange := Nil;
End;

Destructor TDirectoryComboDlg.Destroy;
Begin
  DirLst.Free;
  ImageList.Free;
  Inherited Destroy;
End;

procedure TDirectoryComboDlg.CreateWnd;
Begin
  inherited CreateWnd;
  BuildList;
End;

procedure TDirectoryComboDlg.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TDirectoryComboDlg.CNDrawItem(var Message: TWMDrawItem);
var
  State: TOwnerDrawState;
begin
  with Message.DrawItemStruct^ do
  begin
    State := TOwnerDrawState(WordRec(LongRec(itemState).Lo).Lo);
    Canvas.Handle := hDC;
    Canvas.Font := Font;
    Canvas.Brush := Brush;
    If (Integer(itemID) >= 0) and (odSelected in State) then
    begin
      Canvas.Brush.Color := clHighlight;
      Canvas.Font.Color := clHighlightText
    end;
    If Integer(itemID) >= 0 then
      DrawItem(itemID, rcItem, State) else
      Canvas.FillRect(rcItem);
    Canvas.Handle := 0;
  end;
end;

Procedure TDirectoryComboDlg.ResetItemHeight;
Var
  nuHeight: Integer;
Begin
  nuHeight :=  GetItemHeight(Font);
  if nuHeight < 16 then nuHeight := 16;
  ItemHeight := nuHeight;
End;

procedure TDirectoryComboDlg.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
Var
  ImageIdx, Indent: Integer;
  DispStr: String;
  SavCol: TColor;
Begin
  ImageIdx := StrToIntDef(Copy(Items[Index], 1, 4), 0);
  Indent := StrToIntDef(Copy(Items[Index], 5, 3), 0);
  DispStr := Copy(Items[Index], Pos(';', Items[Index])+1, 256);
  If (Rect.Top < ItemHeight) And (Rect.Top > 0) Then Indent := 0;
  SavCol := Canvas.Brush.Color;
  Canvas.Brush.Color := Color;
  Canvas.FillRect(Rect);
  Canvas.Brush.Color := SavCol;
  If odFocused In State Then ImageList.DrawingStyle := dsSelected
  Else ImageList.DrawingStyle := dsNormal;
  ImageList.Draw(Canvas, Rect.Left + Indent + 2,
                           (Rect.Top + Rect.Bottom - 16) Div 2, ImageIdx);
  Rect.Left := Rect.Left + Indent + 22;
  DrawText(Canvas.Handle, PChar(DispStr), -1, Rect,
             DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
End;

procedure TDirectoryComboDlg.Click;
Begin
  Inherited Click;
  Directory := Copy(Items[ItemIndex], 8, Pos(';', Items[ItemIndex])-8);
End;

procedure TDirectoryComboDlg.BuildList;
var
  MoreInfo: TSHFileInfo;
  Oldcur: Tcursor;
  Bits: Set Of 0..25;
  Ct: Integer;
  Drv: Char;
Begin
  Oldcur := Screen.Cursor;
  Screen.Cursor := crHourglass;
  Items.BeginUpdate;
  Items.Clear;
  If GetSHSpecialFolderInfo(Parent.Handle, MoreInfo, CSIDL_DESKTOP) Then
      Items.Add(Format('%.4d',[MoreInfo.Iicon]) +
                '000' +
                'DeskTop;' +
                StrPas(MoreInfo.szDisplayName))
  Else Items.Add('00000DeskTop;DeskTop');
  If GetSHSpecialFolderInfo(Parent.Handle, MoreInfo, CSIDL_DRIVES) Then
      Items.Add(Format('%.4d',[MoreInfo.Iicon]) +
                '010' +
                'Drives;' +
                StrPas(MoreInfo.szDisplayName))
  Else Items.Add('00010Drives;Drives');

  Application.ProcessMessages;
  Integer(Bits) := GetLogicalDrives;
  For Ct := 0 To 25 Do
    If (Ct In Bits) Then
    Begin
      Drv := Char(Ct + Ord('A'));
      SHGetFileInfo(PChar(Drv + ':\'), 0, MoreInfo, SizeOf(MoreInfo),
                      SHGFI_SYSICONINDEX Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);
      Items.Add(Format('%.4d',[MoreInfo.Iicon]) +
                '020' +
                Drv + ':\;' +
                StrPas(MoreInfo.szDisplayName));
    End;
  If GetSHSpecialFolderInfo(Parent.Handle, MoreInfo, CSIDL_NETWORK) Then
      Items.Add(Format('%.4d',[MoreInfo.Iicon]) +
                '010' +
                'NetWork;' +
                StrPas(MoreInfo.szDisplayName))
  Else Items.Add('00010NetWork;NetWork');
  If GetSHSpecialFolderInfo(Parent.Handle, MoreInfo, CSIDL_BITBUCKET) Then
      Items.Add(Format('%.4d',[MoreInfo.Iicon]) +
                '010' +
                'Recycled;' +
                StrPas(MoreInfo.szDisplayName))
  Else Items.Add('00010Recycled;Recycled');
  DisplayDir;
  Items.Endupdate;
  Screen.Cursor := Oldcur;
End;

Function TDirectoryComboDlg.FindDirItem(Dir: String): Integer;
Var
  I: Integer;
Begin
  Result := -1;
  For I := 0 To Items.Count-1 Do
    If UpperCase(Copy(Items[I], 8, Pos(';', Items[I])-8)) = UpperCase(Dir) Then
    Begin
      Result := I;
      Break;
    End;
End;

procedure TDirectoryComboDlg.DisplayDir;
Var
  MoreInfo: TSHFileInfo;
  I, Indent, OpenIcon: Integer;
  Root, ParDir: String;
Begin
  If DriveItem > -1 Then
  Begin
    For I := Items.Count-1 DownTo DriveItem + 1 Do
      If StrToInt(Copy(Items[I], 5, 3)) > 20 Then
        Items.Delete(I);
    DriveItem := -1
  End;
  If (UpperCase(FDirectory) = 'DESKTOP') Or
      (UpperCase(FDirectory) = 'DRIVES') Or
       (UpperCase(FDirectory) = 'NETWORK') Or
        (UpperCase(FDirectory) = 'RECYCLED') Then Root := FDirectory
  Else If FDirectory[1] = '\' Then Root := 'NetWork'
  Else Root := FDirectory[1] + ':\';
  DriveItem := FindDirItem(Root);
  ParDir := FDirectory;
  DirLst.Clear;
  While Length(ParDir) > Length(Root) Do
  Begin
    DirLst.Insert(0, ParDir);
    ParDir := ExtractFileDir(ParDir);
  End;
  Indent := 30;
  OpenIcon := 0;
  For I := 0 To DirLst.Count-1 Do
  Begin
    If I = DirLst.Count-1 Then OpenIcon := SHGFI_OPENICON;
    SHGetFileInfo(PChar(DirLst[I]), 0, MoreInfo, SizeOf(MoreInfo),
                    SHGFI_SYSICONINDEX Or OpenIcon Or SHGFI_DISPLAYNAME Or SHGFI_TYPENAME);
    Items.Insert(DriveItem+1+I, Format('%.4d',[MoreInfo.Iicon]) +
                                Format('%.3d',[Indent]) +
                                DirLst[I] + ';' +
                                StrPas(MoreInfo.szDisplayName));
    Indent := Indent + 10;
  End;
  ItemIndex := FindDirItem(FDirectory);
End;

Function TDirectoryComboDlg.GetDrive: char;
Begin
  If FDirectory = '' Then Result := ' '
  Else Result := UpCase(FDirectory[1]);
End;

Procedure TDirectoryComboDlg.SetDrive(Value: char);
Var
  MoreInfo: TSHFileInfo;
  Ret: Integer;
  Aux: DWORD;
Begin
  if (UpCase(Value) <> Drive) Then
  Begin
    If GetDiskFreeSpace(PChar(Value + ':\'), Aux, Aux, Aux, Aux) Then
    Begin
      FDirectory := Format('%s:\', [Value]);
      DisplayDir;
    End;
  End;
End;

Procedure TDirectoryComboDlg.SetDirectory(NewDirectory: string);
Var
  OldDir: String;
Begin
  OldDir := FDirectory;
  If AnsiCompareText(NewDirectory, FDirectory) <> 0 then
  Begin
    If (NewDirectory = '\\') Then NewDirectory := 'NETWORK';
    If (UpperCase(NewDirectory) = 'DESKTOP')
      Or (UpperCase(NewDirectory) = 'DRIVES')
        Or (UpperCase(NewDirectory) = 'NETWORK')
          Or (UpperCase(NewDirectory) = 'RECYCLED') Then
    Begin
      FDirectory := NewDirectory;
      Buildlist;
    End
    Else If Length(NewDirectory) > 2 Then
      Begin
        If DirectoryExists(NewDirectory) Then
        Begin
          FDirectory := NewDirectory;
          DisplayDir;
        End;
      End
    Else If Length(NewDirectory) > 0 Then
      SetDrive(NewDirectory[1]);

    If Assigned(FOnDirectoryChange) Then
      FOnDirectoryChange(Self, OldDir, FDirectory, (UpperCase(FDirectory) <> 'DESKTOP'));
    If FFileListView <> Nil Then
      If FFileListView.Directory <> FDirectory Then FFileListView.Directory := FDirectory;
  End;
  ItemIndex := FindDirItem(FDirectory);
End;

Procedure TDirectoryComboDlg.SetFileListView(Value: TFileListViewDlg);
Begin
  FFileListView := Value;
  If FFileListView <> nil then
    FFileListView.FreeNotification(Self);
End;

Procedure TDirectoryComboDlg.Notification(AComponent: TComponent; Operation: TOperation);
Begin
  Inherited Notification(AComponent, Operation);
  If (Operation = opRemove) And (AComponent = FFileListView) Then FFileListView := nil;
End;

End.
