/* Modified Version for Delphi project, Eric W. Engler, Mar 5, 1997 */
/*---------------------------------------------------------------------------
  extract.c
  This file contains the high-level routines ("driver routines") for extrac-
  ting and testing zipfile members.  It calls the low-level routines in files
  explode.c, inflate.c, unreduce.c and unshrink.c.
  Contains:  extract_or_test_files()
             store_info()
             extract_or_test_member()
             TestExtraField()
             memextract()
             memflush()
             fnfilter()
  ---------------------------------------------------------------------------*/
#define UNZIP_INTERNAL
#include "unzip.h"
#include "crypt.h"

static int store_info OF((__GPRO));
static int extract_or_test_member OF((__GPRO));
static int TestExtraField OF((__GPRO__ uch *ef, unsigned ef_len));

#ifdef CRYPT
   static char Far SkipCantGetPasswd[] =
     "   skipping: %-22s  unable to get password\n";
   static char Far SkipIncorrectPasswd[] =
     "   skipping: %-22s  incorrect password\n";
   static char Far FilesSkipBadPasswd[] =
     "%d file%s skipped because of incorrect password.\n";
   static char Far MaybeBadPasswd[] =
     "    (may instead be incorrect password)\n";
#endif

/**************************************/
/*  Function extract_or_test_files()  */
/**************************************/
int extract_or_test_files(__G)    /* return PK-type error code */
     __GDEF
{
    uch *cd_inptr;
    int i, j, cd_incnt, renamed, query, filnum=(-1), blknum=0;
    int error, error_in_archive=PK_COOL, *fn_matched=NULL, *xn_matched=NULL;
    ush members_remaining, num_skipped=0, num_bad_pwd=0;
    long cd_bufstart, bufstart, inbuf_offset, request;
    LONGINT old_extra_bytes=0L;
    long fsize;

/*---------------------------------------------------------------------------
    Since the central directory lies at the end of the zipfile, and the 
    member files lie at the beginning or middle or wherever, it is not very
    efficient to read a central directory entry, jump to the member file
    and extract it, and then jump back to the central directory. 

    Instead, we read from the central directory the pertinent information 
    for a block of files, then go extract/test the whole block.  Thus, this 
    routine contains two small(er) loops within a very large outer loop:
    the first of the small ones reads a block of files from the central 
    directory; the second extracts or tests each file; and the outer one 
    loops over blocks. 

    Because of this jumping around, we can afford to be lenient if an 
    error occurs in one of the member files:  we should still be able to
    go find the other members, since we know the offset of each from the 
    beginning of the zipfile.
  ---------------------------------------------------------------------------*/
    diag("in extract_or_test_files of extract.c");
    G.pInfo = G.info;
    // We're going to check each member file inside the archive
    members_remaining = G.ecrec.total_entries_central_dir;
    Trace((stderr, "members_remaining = %d, unmatched fspecs = %d",
       members_remaining, G.filespecs));

#if (defined(CRYPT))
    G.newzip = TRUE;
    G.reported_backslash = FALSE;
#endif

    /* malloc space for check on unmatched filespecs (OK if one or both NULL) */
    if (G.filespecs > 0  
     && (fn_matched=(int *)malloc(G.filespecs*sizeof(int))) != (int *)NULL)
        for (i = 0;  i < G.filespecs;  ++i)
            fn_matched[i] = FALSE;

    diag("starting main loop - extract.c");
/*---------------------------------------------------------------------------
    Begin main loop over blocks of member files.  
  ---------------------------------------------------------------------------*/
    while (members_remaining) {
        j = 0;

        /* Loop through files in central directory, storing offsets, file
         * attributes, case-conversion and text-conversion flags until block
         * size is reached.
         */
        /*==========================================================*/
        while (members_remaining && (j < DIR_BLKSIZ)) {
            --members_remaining;
            G.pInfo = &G.info[j];

            Trace((stderr,"New iter: members_remaining after this pass = %d",
                members_remaining));
            if (readbuf(__G__ G.sig, 4) == 0) {
                error_in_archive = PK_EOF;
                fprintf(stderr,"premature eof in archive");
                members_remaining = 0;  /* ...so no more left to do */
                break;
            }
            // validate the central hdr signature:
            if (strncmp(G.sig, G.central_hdr_sig, 4)) {  /* just to make sure */
                unzerr(ZE_FORM);
                error_in_archive = PK_BADERR;
                members_remaining = 0;  /* ...so no more left to do */
                break;
            }

            /* process_cdir_file_hdr() in process.c, sets pInfo->hostnum
               and pInfo->lcflag */
            if ((error = process_cdir_file_hdr(__G)) != PK_COOL) {
                unzerr(ZE_FORM);
                diag("a 'not cool' error in archive");
                members_remaining = 0;  /* ...so no more left to do */
                break;
            }
            diag("back in extract_or_test_files of extract.c");
            if ((error = do_string(__G__ G.crec.filename_length, DS_FN)) !=
                 PK_COOL)
            {
                if (error > error_in_archive)
                    error_in_archive = error;
                if (error > PK_WARN) {  /* fatal:  no more left to do */
                    unzerr(ZE_FORM);
                    members_remaining = 0;
                    break;
                }
                else
                   fprintf(stderr,"warning error: %d on file: %s",error,G.filename);
            }
            if ((error = do_string(__G__ G.crec.extra_field_length,
                EXTRA_FIELD)) != 0)
            {
                if (error > error_in_archive)
                    error_in_archive = error;
                if (error > PK_WARN) {  /* fatal */
                    unzerr(ZE_FORM);
                    members_remaining = 0;
                    break;
                }
                else
                   fprintf(stderr,"warning error: %d on file: %s",error,G.filename);
            }
            if ((error = do_string(__G__ G.crec.file_comment_length, SKIP))
                != PK_COOL)
            {
                if (error > error_in_archive)
                    error_in_archive = error;
                if (error > PK_WARN) {  /* fatal */
                    unzerr(ZE_FORM);
                    members_remaining = 0;
                    break;
                }
                else
                   fprintf(stderr,"warning error: %d on file: %s",error,G.filename);
            }
            if (G.process_all_files) {
                if (store_info(__G))
                   {
                   Trace((stderr,"all_files, Matched: %s", G.filename));
                   ++j;  /* file is OK; info[] stored; continue with next */
                   }
                else
                   {
                   Trace((stderr,"Skipped: %s", G.filename));
                   ++num_skipped;
                   }
            } else {
                // We're not processing all files
                int   do_this_file = FALSE;
                char  **pfn = G.pfnames-1;

                while (*++pfn)
                    if (match(G.filename, *pfn, G.C_flag)) 
                       {
                       do_this_file = TRUE;   /* ^-- ignore case or not? */
                       if (fn_matched)
                          {
                          Trace((stderr,"Matched %s", G.filename));
                          fn_matched[(int)(pfn-G.pfnames)] = TRUE;
                          }
                       break;       /* found match, so stop looping */
                       }
                if (do_this_file)
                    if (store_info(__G))
                        {
                        Trace((stderr,"file is OK: %s", G.filename));
                        ++j;       /* file is OK */
                        }
                    else
                        {
                        Trace((stderr,"file skipped: %s", G.filename));
                        ++num_skipped;  /* unsupp. compression or encryption */
                        }
            } /* end if (process_all_files) */

        } /* end while-loop (adding files to current block) */

        /* save position in central directory so can come back later */
        cd_bufstart = G.cur_zipfile_bufstart;
        cd_inptr = G.inptr;
        cd_incnt = G.incnt;

        if ((G.vflag > 0) || (global_trace_opt))
           fprintf(stdout,"UNZDLL expects to extract %d files",j);
    /*-----------------------------------------------------------------------
        Second loop:  process files in current block, extracting or testing
        each one.
      -----------------------------------------------------------------------*/
        diag("starting second loop - THIS TIME we take action. extract.c");

        for (i = 0; i < j; ++i) {
            filnum = i + blknum*DIR_BLKSIZ;
            G.pInfo = &G.info[i];

//          NOVELL_BUG_FAILSAFE:
            G.dne = FALSE;  /* assume file exists until stat() says otherwise */
  
           /* if the target position is not within the current input buffer
             * (either haven't yet read far enough, or (maybe) skipping back-
             * ward), skip to the target position and reset readbuf(). */
            request = G.pInfo->offset + G.extra_bytes;
            inbuf_offset = request % INBUFSIZ;
            bufstart = request - inbuf_offset;

            Trace((stderr, "loc A: request = %ld, inbuf_offset = %ld\n",
                   request, inbuf_offset));
            if (request < 0) {
                unzerr(ZE_FORM);
                error_in_archive = PK_ERR;
                if (filnum == 0 && G.extra_bytes != 0L) {
                    fprintf(stderr,"attempting to recompensate");
                    old_extra_bytes = G.extra_bytes;
                    G.extra_bytes = 0L;
                    request = G.pInfo->offset; /* could also check if this != 0 */
                    inbuf_offset = request % INBUFSIZ;
                    bufstart = request - inbuf_offset;
                    Trace((stderr, "loc B: request = %ld, inbuf_offset = %ld\n",
                      request, inbuf_offset));
                } else {
                    error_in_archive = PK_BADERR;
                    diag("loc A: hosed - try next file");
                    continue;  /* this one hosed; try next */
                }
            }

            /* try again */
            if (request < 0) {
                Trace((stderr, "the recompensated request is still < 0\n"));
                unzerr(ZE_FORM);
                error_in_archive = PK_BADERR;
                continue;
            } else if (bufstart != G.cur_zipfile_bufstart) {
                Trace((stderr, "bufstart != cur_zipfile_bufstart\n"));
#ifdef USE_STRM_INPUT
                fseek((FILE *)G.zipfd,(LONGINT)bufstart,SEEK_SET);
                G.cur_zipfile_bufstart = ftell((FILE *)G.zipfd);
#else /* !USE_STRM_INPUT */
                G.cur_zipfile_bufstart =
                     lseek(G.zipfd,(LONGINT)bufstart,SEEK_SET);
#endif /* ?USE_STRM_INPUT */
                if ((G.incnt = read(G.zipfd,(char *)G.inbuf,INBUFSIZ)) <= 0)
                {
                    unzerr(ZE_FORM);
                    error_in_archive = PK_BADERR;
                    diag("B. hosed - try next file");
                    continue;   /* can still do next file */
                }
                G.inptr = G.inbuf + (int)inbuf_offset;
                G.incnt -= (int)inbuf_offset;
            } else {
                G.incnt += (int)(G.inptr-G.inbuf) - (int)inbuf_offset;
                G.inptr = G.inbuf + (int)inbuf_offset;
            }
            /* should be in proper position now, so check for sig */
            if (readbuf(__G__ G.sig, 4) == 0) {  /* bad offset */
                unzerr(ZE_FORM);
                error_in_archive = PK_BADERR;
                continue;   /* but can still try next one */
            }
            if (strncmp(G.sig, G.local_hdr_sig, 4)) {
                unzerr(ZE_FORM);
                error_in_archive = PK_ERR;
                if ((filnum == 0 && G.extra_bytes != 0L) ||
                    (G.extra_bytes == 0L && old_extra_bytes != 0L)) {
                    fprintf(stderr, "Attempting to Recompensate");
                    if (G.extra_bytes) {
                        old_extra_bytes = G.extra_bytes;
                        G.extra_bytes = 0L;
                    } else
                        G.extra_bytes = old_extra_bytes;  /* third attempt */
                    ZLSEEK(G.pInfo->offset)
                    if (readbuf(__G__ G.sig, 4) == 0) {  /* bad offset */
                        unzerr(ZE_FORM);
                        error_in_archive = PK_BADERR;
                        continue;   /* but can still try next one */
                    }
                    if (strncmp(G.sig, G.local_hdr_sig, 4)) {
                        unzerr(ZE_FORM);
                        error_in_archive = PK_BADERR;
                        continue;
                    }
                } else
                    {
                    diag("C: hosed - try next file");
                    continue;  /* this one hosed; try next */
                    }
            }
            diag("about to process local file hdr");
            if ((error = process_local_file_hdr(__G)) != PK_COOL) {
                unzerr(ZE_FORM);
                error_in_archive = error;   /* only PK_EOF defined */
                diag("D. hosed - try next file");
                continue;   /* can still try next one */
            }
            if ((error = do_string(__G__ G.lrec.filename_length, DS_FN)) !=
                 PK_COOL)
            {
                if (error > error_in_archive)
                    error_in_archive = error;
                if (error > PK_WARN) {
                    unzerr(ZE_FORM);
                    diag("E. hosed - try next file");
                    continue;   /* go on to next one */
                }
            }
            Trace((stderr, "Good entry for: %s", G.filename));

            if (G.extra_field != (uch *)NULL) {
                free(G.extra_field);
                G.extra_field = (uch *)NULL;
            }
            if ((error =
                 do_string(__G__ G.lrec.extra_field_length,EXTRA_FIELD)) != 0)
            {
                if (error > error_in_archive)
                    error_in_archive = error;
                if (error > PK_WARN) {
                    unzerr(ZE_FORM);
                    diag("F. hosed - try next file");
                    continue;   /* go on */
                }
            }
            /* Just about to extract file:  if extracting to disk, check if
             * already exists, and if so, take appropriate action according to
             * fflag/uflag/overwrite_all/etc. (we couldn't do this in upper
             * loop because we don't store the possibly renamed filename[] in
             * info[])
             */
            if (!G.tflag)
            {
                renamed = FALSE;   /* user hasn't renamed output file yet */
//startover:
                query = FALSE;
                /* for files from DOS FAT, check for use of backslash instead
                 *  of slash as directory separator (bug in some zipper(s); so
                 *  far, not a problem in HPFS, NTFS or VFAT systems)
                 */
                if (G.pInfo->hostnum == FS_FAT_ && !strchr(G.filename, '/')) {
                    char *p=G.filename-1;

                    diag("parsing a FAT file");
                    while (*++p) {
                        if (*p == '\\') {
                            if (!G.reported_backslash) {
                                unzerr(ZE_FORM);
                                G.reported_backslash = TRUE;
                                if (!error_in_archive)
                                    error_in_archive = PK_WARN;
                            }
                            *p = '/';
                        }
                    }
                }
                /* Mapname can create dirs if not freshening or if renamed */
                if ((error = mapname(__G__ renamed)) > PK_WARN) {
                    if (error == IZ_CREATED_DIR) {
                        /* GRR:  add code to set times/attribs on dirs--
                         * save to list, sort when done (a la zip), set
                         * times/attributes on deepest dirs first */
                    } else if (error == IZ_VOL_LABEL) {
                        unzerr(ZE_FORM);
                    } else if (error > PK_ERR  &&  error_in_archive < PK_ERR)
                        {
                        diag("G. hosed - try next file");
                        error_in_archive = PK_ERR;
                        }
                    Trace((stderr, "mapname(%s) returns error = %d\n",
                      G.filename, error));
                    continue;   /* go on to next file */
                }

                diag("starting switch near Novell failsafe in extract.c");
                switch (check_for_newer(__G__ G.filename)) {
                    case DOES_NOT_EXIST:
                        diag("does not exist");
//                      NOVELL_BUG_FAILSAFE:
                        G.dne = TRUE;   /* stat() says file DOES NOT EXIST */
                        /* if freshening, don't skip if just renamed */
                        if (G.fflag && !renamed)
                            continue;   /* freshen (no new files):  skip */
                        break;

                // EWE NOTE: for some reason, the "newer and older"
                // terminology in the following cases is bogus
                    case EXISTS_AND_OLDER:
                        if (G.overwrite_none) {
                            fprintf(stderr,"File exists: %s", G.filename);
                            continue;   /* never overwrite:  skip file */
                        }
                        if (!G.overwrite_all)
                            query = TRUE;
                        break;
                    case EXISTS_AND_NEWER:             /* (or equal) */
                        if (G.overwrite_none || (G.uflag && !renamed)) {
                            fprintf(stderr,"File exists: %s", G.filename);
                            continue;  /* skip if update/freshen & orig name */
                        }
                        if (!G.overwrite_all)
                            query = TRUE;
                        break;
                } /* end switch */
            } /* end if (extracting to disk) */

#ifdef CRYPT
            if (G.pInfo->encrypted && (error = decrypt(__G)) != PK_COOL) {
                if (error == PK_MEM2) {
                    if (error > error_in_archive)
                        error_in_archive = error;
                    // SkipCantGetPasswd
                } else {  /* (error == PK_WARN) */
                    if (!((G.tflag && G.qflag) || (!G.tflag && !QCOND2)))
                    // SkipIncorrectPasswd
                    ++num_bad_pwd;
                }
                continue;   /* go on to next file */
            }
#endif /* CRYPT */
            G.disk_full = 0;
 
            if (global_abort_sw)
               {
               unzerr(ZE_ABORT);
               break;
               }
            diag("about to call 'extract_or_test_member'");
            fsize = G.lrec.ucsize;
            user_callback(1, 0, fsize, G.filename); // initial progress call

            // ====================================================
            // Function:  extract_or_test_member() does the unzip
            // ====================================================
            if ((error = extract_or_test_member(__G)) != PK_COOL) {
                diag("error occured while extracting or testing");
                if (error > error_in_archive)
                    error_in_archive = error;       /* ...and keep going */
                if (G.disk_full > 1) {
#ifdef DYNALLOC_CRCTAB
                    nearfree((zvoid near *)CRC_32_TAB);
#endif /* DYNALLOC_CRCTAB */
                    if (fn_matched)
                        free((zvoid *)fn_matched);
                    if (xn_matched)
                        free((zvoid *)xn_matched);
                    return error_in_archive;        /* (unless disk full) */
                }
            } else {
               fprintf(stdout,"Unzipped file %s of size %ld", G.filename, fsize);
               files_acted_on++;
               user_callback(2,0,0,NULL);  // final progress call
           }
        } /* end for-loop (i:  files in current block) */

        /* Jump back to where we were in the central directory, then go and do
         * the next batch of files.
         */
        diag("jump back in central dir to where we were");
#ifdef USE_STRM_INPUT
        fseek((FILE *)G.zipfd, (LONGINT)cd_bufstart, SEEK_SET);
        G.cur_zipfile_bufstart = ftell((FILE *)G.zipfd);
#else /* !USE_STRM_INPUT */
        G.cur_zipfile_bufstart = lseek(G.zipfd,(LONGINT)cd_bufstart,SEEK_SET);
#endif /* ?USE_STRM_INPUT */
        read(G.zipfd, (char *)G.inbuf, INBUFSIZ);  /* been here before... */
        G.inptr = cd_inptr;
        G.incnt = cd_incnt;
        ++blknum;
    } /* end while-loop (blocks of files in central directory) */

    diag("done with big outer block in extract.c");
    user_callback(3,0,0,NULL);  // done with a batch of files
/*---------------------------------------------------------------------------
    Check for unmatched filespecs on command line and print warning if any
    found.  Free allocated memory.
  ---------------------------------------------------------------------------*/
    if (fn_matched) {
        diag("fn_matched was true");
        for (i = 0;  i < G.filespecs;  ++i)
            if (!fn_matched[i]) {
                fprintf(stderr,"Filespec Not Matched: %s",G.pfnames[i]);
                if (error_in_archive <= PK_WARN)
                    error_in_archive = PK_FIND;   /* some files not found */
            }
        free((zvoid *)fn_matched);
    }

/*---------------------------------------------------------------------------
    Double-check that we're back at the end-of-central-directory record, and
    print quick summary of results, if we were just testing the archive. 
  ---------------------------------------------------------------------------*/
    if (readbuf(__G__ G.sig, 4) == 0)
        {
        diag("bad signature at end of archive, or premature EOF");
        error_in_archive = PK_EOF;
        }
    if (strncmp(G.sig, G.end_central_sig, 4)) {    /* just to make sure */
        fprintf(stderr,"Bad Ending Signature for Central dir");
        if (!error_in_archive)       /* don't overwrite stronger error */
            error_in_archive = PK_WARN;
    }

    ++filnum;  /* initialized to -1, so now zero if no files found */
    Trace((stderr, "filnum = %d", filnum));

    if (G.tflag) {
        // testing archive
        int num=filnum - num_bad_pwd;

        if (G.qflag < 2) {         /* GRR 930710:  was (G.qflag == 1) */
            if (error_in_archive)
                fprintf(stderr,"Error In Archive %s %s",
                  (error_in_archive == 1) ? "warning-" : "", G.zipfn);
            else if (num == 0)
                fprintf(stderr, "Zero Files Tested %s", G.zipfn);
            else if (G.process_all_files && (num_skipped+num_bad_pwd == 0))
                fprintf(stderr,"no error in %s", G.zipfn);
            else
                fprintf(stderr,"No Error In %d Tested Files of %s", num, G.zipfn);
            if (num_skipped > 0)
                fprintf(stderr,"Skipped %d Files of %s", num_skipped, G.zipfn);
#ifdef CRYPT
            if (num_bad_pwd > 0)
                fprintf(stderr,"Files with bad pwd: %d", num_bad_pwd);
#endif /* CRYPT */
        } else if ((G.qflag == 0) && !error_in_archive && (num == 0))
            fprintf(stderr, "Zero Files Tested %s", G.zipfn);
    }

    /* give warning if files not tested or extracted (first condition can still
     * happen if zipfile is empty and no files specified on command line) */
    if ((filnum == 0) && error_in_archive <= PK_WARN)
        {
        fprintf(stderr,"no files found");
        error_in_archive = PK_FIND;   /* no files found at all */
        }
    else if ((num_skipped > 0) && !error_in_archive)
        {
        fprintf(stderr,"some files skipped");
        error_in_archive = PK_WARN;
        }
#ifdef CRYPT
    else if ((num_bad_pwd > 0) && !error_in_archive)
        error_in_archive = PK_WARN;
#endif /* CRYPT */

    return error_in_archive;
} /* end function extract_or_test_files() */

/***************************/
/*  Function store_info()  */
/***************************/
static int store_info(__G)   /* return 0 if skipping, 1 if OK */
     __GDEF
{
#ifdef SFX
#  define UNKN_COMPR \
   (G.crec.compression_method!=STORED && G.crec.compression_method!=DEFLATED)
#else
#  ifdef COPYRIGHT_CLEAN  /* no reduced or tokenized files */
#    define UNKN_COMPR  (G.crec.compression_method>SHRUNK && \
     G.crec.compression_method!=IMPLODED && G.crec.compression_method!=DEFLATED)
#  else /* !COPYRIGHT_CLEAN */
#    define UNKN_COMPR \
     (G.crec.compression_method>IMPLODED && G.crec.compression_method!=DEFLATED)
#  endif /* ?COPYRIGHT_CLEAN */
#endif

/*---------------------------------------------------------------------------
    Check central directory info for version/compatibility requirements.
  ---------------------------------------------------------------------------*/
    G.pInfo->encrypted = G.crec.general_purpose_bit_flag & 1;       /* bit field */
    G.pInfo->ExtLocHdr = (G.crec.general_purpose_bit_flag & 8) == 8;/* bit field */
    G.pInfo->textfile = G.crec.internal_file_attributes & 1;        /* bit field */
    G.pInfo->crc = G.crec.crc32;
    G.pInfo->compr_size = G.crec.csize;

    switch (G.aflag) {
        case 0:
            G.pInfo->textmode = FALSE;   /* bit field */
            break;
        case 1:
            G.pInfo->textmode = G.pInfo->textfile;   /* auto-convert mode */
            break;
        default:  /* case 2: */
            G.pInfo->textmode = TRUE;
            break;
    }

    // EWE note: all platforms define VMS_UNZIP_VERSION (currently 42)
    if (G.crec.version_needed_to_extract[1] == VMS_) {
        if (G.crec.version_needed_to_extract[0] > VMS_UNZIP_VERSION) {
            if (!((G.tflag && G.qflag) || (!G.tflag && !QCOND2)))
                fprintf(stderr,"Unsupported zip version or hosttype");
            return 0;
        }
#ifndef VMS   /* won't be able to use extra field, but still have data */
        else if (!G.tflag && !G.overwrite_all) {   /* if -o, extract regardless */
            fprintf(stderr, "Warning - file's format may be incorrect: %s",
               G.filename);
            return 0;
        }
#endif /* !VMS */
    /* usual file type:  don't need VMS to extract */
    } else if (G.crec.version_needed_to_extract[0] > UNZIP_VERSION) {
        if (!((G.tflag && G.qflag) || (!G.tflag && !QCOND2)))
            fprintf(stderr,"Unsupported zip version or hosttype");
        return 0;
    }
    if UNKN_COMPR {
        if (!((G.tflag && G.qflag) || (!G.tflag && !QCOND2)))
            fprintf(stderr,"Unsupported compression type");
        return 0;
    }
#ifndef CRYPT
    if (G.pInfo->encrypted) {
        if (!((G.tflag && G.qflag) || (!G.tflag && !QCOND2)))
            fprintf(stderr,"Skipping encrypted file: %s", G.filename);
        return 0;
    }
#endif /* !CRYPT */

    /* map whatever file attributes we have into the local format */
    mapattr(__G);   /* GRR:  worry about return value later */
    G.pInfo->offset = (long) G.crec.relative_offset_local_header;
    return 1;
} /* end function store_info() */

/***************************************/
/*  Function extract_or_test_member()  */
/***************************************/
static int extract_or_test_member(__G)    /* return PK-type error code */
     __GDEF
{
    char *nul="[empty] ", *txt="[text]  ", *bin="[binary]";
#ifdef CMS_MVS
    char *ebc="[ebcdic]";
#endif
    register int b;
    int r, error;
    ulg wsize;

    error=PK_COOL;  // init 
    Trace((stderr,"start extract_or_test_member: %s",G.filename));
/*---------------------------------------------------------------------------
    Initialize variables, buffers, etc.
  ---------------------------------------------------------------------------*/
    G.bits_left = 0;
    G.bitbuf = 0L;       /* unreduce and unshrink only */
    G.zipeof = 0;
    G.newfile = TRUE;
    G.crc32val = CRCVAL_INITIAL;
    Trace((stderr,"initializing G.crc32val to %08X",G.crc32val));

    if (G.tflag) {  // if test desired
        if (!G.qflag)
            fprintf(stderr,"Testing %s", G.filename);
    } else {
        if (open_outfile(__G))
            return PK_DISK;
    }

/*---------------------------------------------------------------------------
    Unpack the file.
  ---------------------------------------------------------------------------*/
    diag("unpack the file");
    defer_leftover_input(__G);    /* so NEXTBYTE bounds check will work */

    switch (G.lrec.compression_method) {
        case STORED:
            wsize = WSIZE;
            G.outptr = slide;
            G.outcnt = 0L;
            while ((b = NEXTBYTE) != EOF && !G.disk_full) {
                *G.outptr++ = (uch)b;
                if (++G.outcnt == wsize) { // EWE: wsize = 32K
                    user_callback(2, 0, 0, NULL); // bump up progress bar 
                    flush( __G__ slide, G.outcnt, 0);
                    G.outptr = slide;
                    G.outcnt = 0L;
                }
            }
            if (G.outcnt)          /* flush final (partial) buffer */
                flush(__G__ slide, G.outcnt, 0);
            break;

#ifndef SFX
        case SHRUNK:
            if ((r = unshrink(__G)) != PK_COOL) {
                if ((G.tflag && G.qflag) || (!G.tflag && !QCOND2))
                    fprintf(stderr,"Error unzipping file %s", G.filename);
                else
                    fprintf(stderr,"Error unzipping files");
                error = r;
            }
            break;

#ifndef COPYRIGHT_CLEAN
        case REDUCED1:
        case REDUCED2:
        case REDUCED3:
        case REDUCED4:
            unreduce(__G);
            break;
#endif /* !COPYRIGHT_CLEAN */

        case IMPLODED:
            if (((r = explode(__G)) != 0) && (r != 5)) {   /* treat 5 specially */
                if ((G.tflag && G.qflag) || (!G.tflag && !QCOND2))
                    fprintf(stderr,"Error unzipping file %s", G.filename);
                else
                    fprintf(stderr,"Error unzipping files");
                error = (r == 3)? PK_MEM3 : PK_ERR;
            }
            if (r == 5) {
                int warning = ((ulg)G.used_csize <= G.lrec.csize);

                if ((G.tflag && G.qflag) || (!G.tflag && !QCOND2))
                    fprintf(stderr,"Error unzipping file %s", G.filename);
                else
                    fprintf(stderr,"Error unzipping files");
                error = warning? PK_WARN : PK_ERR;
            }
            break;
#endif /* !SFX */

        case DEFLATED:
#ifndef USE_ZLIB  /* zlib's function is called inflate(), too */
#  define UZinflate inflate
#endif
            if ((r = UZinflate(__G)) != 0) {
                if ((G.tflag && G.qflag) || (!G.tflag && !QCOND2))
                    fprintf(stderr,"Error unzipping file %s", G.filename);
                else
                    fprintf(stderr,"Error unzipping files");
                error = (r == 3)? PK_MEM3 : PK_ERR;
            }
            break;

        default:   /* should never get to this point */
            diag("should NEVER get here");
            fprintf(stderr,"Error unzipping files");
            /* close and delete file before return? */
            undefer_input(__G);
            return PK_WARN;
    } /* end switch (compression method) */

    if (G.disk_full) {            /* set by flush() */
        if (G.disk_full > 1) {
            undefer_input(__G);
            return PK_DISK;
        }
        error = PK_WARN;
    } 

    if (error != PK_COOL)
       Trace((stderr,"had an error of %d before closing file", error));
/*---------------------------------------------------------------------------
    Close the file and set its date and time (not necessarily in that order),
    and make sure the CRC checked out OK.  Logical-AND the CRC for 64-bit
    machines (redundant on 32-bit machines).
  ---------------------------------------------------------------------------*/
#ifdef VMS                 /* VMS:  required even for stdout! (final flush) */
    if (!G.tflag)          /* don't close NULL file */
        close_outfile(__G);
#else
    if (!G.tflag && !G.cflag)   /* don't close NULL file or stdout */
        close_outfile(__G);
#endif /* VMS */

    /* GRR todo: CONVERT close_outfile() TO NON-VOID:  CHECK FOR ERRORS! */

    if (error > PK_WARN) {/* don't print redundant CRC error if error already */
        undefer_input(__G);
        return error;
    }

    Trace((stderr,"After extraction, G.crc32val = %08X",G.crc32val));
    Trace((stderr,"File's CRC in local hdr = %08X",G.lrec.crc32));
    if (G.crc32val != G.lrec.crc32) {
        /* if quiet enough, we haven't output the filename yet:  do it */
#ifdef CRYPT
        if (G.pInfo->encrypted)
            fprintf(stderr,"May be Bad Passwd for file: %s", G.filename);
#endif
        fprintf(stdout,"After extraction, file %s had a CRC error", G.filename);
        error = PK_ERR;
    } else if (G.tflag) {
        if (G.extra_field) {
            if ((r = TestExtraField(__G__ G.extra_field,
                               G.lrec.extra_field_length)) > error)
                error = r;
        } else if (!G.qflag)
            fprintf(stderr," OK");
    } else {
        if (QCOND2 && !error)   /* GRR:  is stdout reset to text mode yet? */
            fprintf(stderr, "unexpected error in extract.c");
    }
    undefer_input(__G);

    if (error != PK_COOL)
       Trace((stderr,"extract_or_test_member returning error: %d", error));
    return error;
} /* end function extract_or_test_member() */

/*******************************/
/*  Function TestExtraField()  */
/*******************************/
static int TestExtraField(__G__ ef, ef_len)
    __GDEF
    uch *ef;
    unsigned ef_len;
{
    ush ebID;
    unsigned ebLen;

    /* we know the regular compressed file data tested out OK, or else we
     * wouldn't be here ==> print filename if any extra-field errors found
     */
    while (ef_len >= EB_HEADSIZE) {
        ebID = makeword(ef);
        ebLen = (unsigned)makeword(ef+EB_LEN);

        if (ebLen > (ef_len - EB_HEADSIZE)) {
           /* Discovered some extra field inconsistency! */
            fprintf(stderr,"Bad length on file %s", G.filename);
            return PK_ERR;
        }
        ef_len -= (ebLen + EB_HEADSIZE);
        ef += (ebLen + EB_HEADSIZE);
    }
    if (!G.qflag)
        fprintf(stderr,"OK");
    return PK_COOL;
} /* end function TestExtraField() */

/***************************/
/*  Function memextract()  */
/***************************/
int memextract(__G__ tgt, tgtsize, src, srcsize)  /* extract compressed */
    __GDEF                                        /*  extra field block; */
    uch *tgt, *src;                               /*  return PK-type error */
    ulg tgtsize, srcsize;                         /*  level */
{
    uch *old_inptr=G.inptr;
    int  old_incnt=G.incnt, error=PK_OK;
    ush  method;
    ulg  extra_field_crc;
    int r;

    method = makeword(src);
    extra_field_crc = makelong(src+2);

    /* compressed extra field exists completely in memory at this location: */
    G.inptr = src + 2 + 4;      /* method and extra_field_crc */
    G.incnt = (int)(G.csize = (long)(srcsize - (2 + 4)));
    G.mem_mode = TRUE;
    G.outbufptr = tgt;
    G.outsize = tgtsize;

    switch (method) {
        case STORED:
            memcpy((char *)tgt, (char *)G.inptr, (extent)G.incnt);
            G.outcnt = G.csize;   /* for CRC calculation */
            break;
        case DEFLATED:
            G.outcnt = 0L;
            if ((r = UZinflate(__G)) != 0) {
                if (!G.tflag)
                    fprintf(stderr,"Error unzipping files");
                error = (r == 3)? PK_MEM3 : PK_ERR;
            }
            if (G.outcnt == 0L)   /* inflate's final FLUSH sets outcnt */
                break;
            break;
        default:
            if (G.tflag)
                error = PK_ERR | ((int)method << 8);
            else {
                fprintf(stderr, "unsupported extra field");
                error = PK_ERR;  /* GRR:  should be passed on up via SetEAs() */
            }
            break;
    }
    G.inptr = old_inptr;
    G.incnt = old_incnt;
    G.mem_mode = FALSE;
    if (!error) {
        register ulg crcval = crc32(CRCVAL_INITIAL, tgt, (extent)G.outcnt);
        if (crcval != extra_field_crc) {
            if (G.tflag)
                error = PK_ERR | (DEFLATED << 8);  /* kludge for now */
            else {
                fprintf(stderr,"bad extra field CRC");
                error = PK_ERR;
            }
        }
    }
    return error;
} /* end function memextract() */

/*************************/
/*  Function memflush()  */
/*************************/
int memflush(__G__ rawbuf, size)
    __GDEF
    uch *rawbuf;
    ulg size;
{
    if (size > G.outsize)
        return 50;   /* more data than output buffer can hold */

    memcpy((char *)G.outbufptr, (char *)rawbuf, (extent)size);
    G.outbufptr += (unsigned int)size;
    G.outsize -= size;
    G.outcnt += size;
    return 0;
} /* end function memflush() */

/*************************/
/*  Function fnfilter()  */      /* here instead of in list.c for SFX */
/*************************/
char *fnfilter(raw, space)       /* convert name to safely printable form */
    char *raw;
    uch *space;
{
#ifndef NATIVE   /* ASCII:  filter ANSI escape codes, etc. */
    register uch *r=(uch *)raw, *s=space;
    while (*r)
        if (*r < 32)
            *s++ = '^', *s++ = (uch)(64 + *r++);
        else
            *s++ = *r++;
    *s = 0;
    return (char *)space;
#else
    return raw;
#endif
} /* end function fnfilter() */
