unit Zoomer;

interface
              
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, ExtCtrls, Buttons;

type
  params = record
    xscale,yscale   : double;
    xmin,xmax       : double;
    ymin,ymax       : double;
    xmid,ymid       : double;
  end;

type
  Point2D   = record
    x,y : Double;
  end;

type
  Zoom_panel = class(TCustomPanel)
  private
    { Private declarations }
    FOnPaint     : TNotifyEvent;
    FOnMouseDown : TMouseEvent;
    FOnMouseMove : TMouseMoveEvent;
    FOnZoomin    : TNotifyEvent;
    FOnZoomout   : TNotifyEvent;
    FOnZoomreset : TNotifyEvent;
    // visible things
    scrollpanel_ud  : TPanel;
    scrollpanel_lr  : TPanel;
    ScrollBar_ud    : TScrollBar;
    ScrollBar_lr    : TScrollBar;
    zoom_in_button  : TSpeedButton;
    zoom_out_button : TSpeedButton;
    zoomresetbutton : TSpeedButton;
    Coords          : TLabel;
    // bookkeeping
    original_params : params;
    current_params  : params;
    Zooming_in      : boolean;
    Zooming_out     : boolean;
    zoomtimer       : TTimer;
  protected
    { Protected declarations }
  public
    ClientArea      : TPaintBox;
    AspectRatio     : double;
    initialized     : boolean;

    constructor Create(AOwner: TComponent);     override;
    procedure   Resize;                         override;
    procedure   WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure   Mouse_Down(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
    procedure   Mouse_Move(Sender: TObject; Shift: TShiftState; X, Y: Integer);

    procedure   ScrollBars_Scroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
    procedure   ReCentre(mx,my:integer);
    procedure   Zoom(factor:double);
    procedure   Zoom_in_out(Sender:TObject);
    procedure   Zoom_mousedown(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
    procedure   Zoom_mouseup(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
    procedure   Zoom_timer_event(Sender:TObject);

    procedure   set_parameters(xmn,xmx,ymn,ymx:double; xm,ym:integer; aspect:double);
    function    real_to_screen(x,y:double) : TPoint;
    function    screen_to_real(P1:TPoint) : Point2D;
    { Public declarations }
  published
    { Published declarations }
    property Align;
    property Alignment;
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Color;
    property Ctl3D;
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown : TMouseEvent     read FOnMouseDown write FOnMouseDown;
    property OnMouseMove : TMouseMoveEvent read FOnMouseMove write FOnMouseMove;
    property OnMouseUp;
    property OnPaint     : TNotifyEvent    read FOnPaint     write FOnPaint;
    property OnResize;
    property OnStartDrag;
    property OnZoomin    : TNotifyEvent    read FOnZoomin    write FOnZoomin;
    property OnZoomout   : TNotifyEvent    read FOnZoomout   write FOnZoomout;
    property OnZoomreset : TNotifyEvent    read FOnZoomreset write FOnZoomreset;
  end;

procedure Register;

implementation

{$R images.res}

procedure Register;
begin
  RegisterComponents('Custom', [Zoom_panel]);
end;
///////////////////////////////////////////////////////////////////////////////
// Create all the visible items
// Set the Owner and Parent properties so Delphi deletes them all for us.
///////////////////////////////////////////////////////////////////////////////
constructor Zoom_panel.Create(AOwner: TComponent);
var TempBitmap : TBitmap;
begin
  inherited;
  initialized := false;
  ControlStyle := ControlStyle - [csSetCaption];

  scrollpanel_ud            := TPanel.Create(Self);
  scrollpanel_ud.Visible    := True;
  scrollpanel_ud.Parent     := Self;
  scrollpanel_ud.Align      := AlRight;

  scrollpanel_lr            := TPanel.Create(Self);
  scrollpanel_lr.Visible    := True;
  scrollpanel_lr.Parent     := Self;
  scrollpanel_lr.Align      := AlBottom;

  ScrollBar_ud              := TScrollBar.Create(scrollpanel_ud);
  ScrollBar_ud.Visible      := True;
  ScrollBar_ud.Parent       := scrollpanel_ud;
  ScrollBar_ud.Kind         := sbVertical;
  ScrollBar_ud.Position     := 50;
  ScrollBar_ud.LargeChange  := 10;
  ScrollBar_ud.OnScroll     := ScrollBars_Scroll;

  ScrollBar_lr              := TScrollBar.Create(scrollpanel_lr);
  ScrollBar_lr.Visible      := True;
  ScrollBar_lr.Parent       := scrollpanel_lr;
  ScrollBar_lr.Kind         := sbHorizontal;
  ScrollBar_lr.Position     := 50;
  ScrollBar_lr.LargeChange  := 10;
  ScrollBar_lr.OnScroll     := ScrollBars_Scroll;

  TempBitmap                := TBitmap.Create;
  TempBitmap.LoadFromResourceName(HInstance,'Z_RESET');
  zoomresetbutton           := TSpeedButton.Create(scrollpanel_lr);
  zoomresetbutton.Parent    := scrollpanel_lr;
  zoomresetbutton.Glyph.Assign(TempBitmap);
  zoomresetbutton.Onclick   := Zoom_in_out;

  TempBitmap.LoadFromResourceName(HInstance,'Z_PLUS');
  zoom_in_button             := TSpeedButton.Create(scrollpanel_lr);
  zoom_in_button.Parent      := scrollpanel_lr;
  zoom_in_button.Glyph.Assign(TempBitmap);
  zoom_in_button.OnMouseDown := Zoom_mousedown;
  zoom_in_button.OnMouseUP   := Zoom_mouseup;

  TempBitmap.LoadFromResourceName(HInstance,'Z_MINUS');
  zoom_out_button             := TSpeedButton.Create(scrollpanel_lr);
  zoom_out_button.Parent      := scrollpanel_lr;
  zoom_out_button.Glyph.Assign(TempBitmap);
  zoom_out_button.OnMouseDown := Zoom_mousedown;
  zoom_out_button.OnMouseUP   := Zoom_mouseup;
  TempBitmap.Free;

  Zooming_in          := false;
  Zooming_out         := false;
  zoomtimer           := TTimer.Create(scrollpanel_lr);
  zoomtimer.OnTimer   := Zoom_timer_event;
  zoomtimer.Enabled   := false;
  zoomtimer.Interval  := 500;

  ClientArea                := TPaintbox.Create(Self);
  ClientArea.Visible        := True;
  ClientArea.Parent         := Self;
  ClientArea.Align          := AlClient;
  ClientArea.OnMouseMove    := Mouse_Move;
  ClientArea.OnMouseDown    := Mouse_Down;

  Coords                    := TLabel.Create(scrollpanel_lr);
  Coords.Font.Name          := 'FF_ROMAN';
  Coords.Parent             := scrollpanel_lr;
  Coords.Caption            := '(0,0)';
  resize;
end;
///////////////////////////////////////////////////////////////////////////////
// Adjust everything after a resize
///////////////////////////////////////////////////////////////////////////////
procedure Zoom_panel.Resize;
begin
  inherited;
  scrollpanel_lr.height     := GetSystemMetrics(SM_CYHSCROLL)*2;
  scrollpanel_ud.width      := GetSystemMetrics(SM_CXVSCROLL);
  scrollbar_lr.left         := 0;
  scrollbar_lr.top          := 0;
  scrollbar_lr.height       := scrollpanel_lr.height div 2;
  scrollbar_lr.width        := scrollpanel_lr.width - scrollpanel_ud.width*6;

  scrollpanel_lr.BevelInner := bvNone;
  scrollpanel_lr.BevelOuter := bvNone;

  Coords.Font.Height        := scrollbar_lr.Height-2;
  Coords.Left               := 8;//3*(scrollbar_lr.width div 4);
  Coords.Top                := scrollbar_lr.Height+1;

  scrollbar_ud.left         := 0;
  scrollbar_ud.top          := 0;
  scrollbar_ud.width        := scrollpanel_ud.width;
  scrollbar_ud.height       := scrollpanel_ud.height;

  zoom_out_button.left      := scrollbar_lr.width;
  zoom_out_button.Width     := scrollbar_ud.width*2;
  zoom_out_button.height    := scrollbar_lr.height*2;

  zoom_in_button.left       := scrollbar_lr.width + zoom_out_button.width;
  zoom_in_button.Width      := scrollbar_ud.width*2;
  zoom_in_button.height     := scrollbar_lr.height*2;

  zoomresetbutton.left      := scrollbar_lr.width + zoom_out_button.width*2;
  zoomresetbutton.width     := scrollbar_ud.width*2;
  zoomresetbutton.height    := scrollbar_lr.height*2;

  zoomresetbutton.Top       := 0;
  zoom_in_button.Top        := 0;
  zoom_out_button.Top       := 0;
end;
///////////////////////////////////////////////////////////////////////////////
// Events we pass on to user
///////////////////////////////////////////////////////////////////////////////
procedure Zoom_panel.WMPaint(var Message: TWMPaint);
begin
  inherited;
  if Assigned(FOnPaint) then FOnPaint(Self);
end;

procedure Zoom_panel.Mouse_Down(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
begin
  inherited;
  if Button = mbRight then ReCentre(x,y);
  if Assigned(FOnMouseDown) then FOnMouseDown(Sender,Button,Shift,X,Y);
end;

procedure Zoom_panel.Mouse_Move(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var s1,s2 : string;
    P1    : Point2D;
begin
  inherited;
  if Assigned(FOnMouseMove) then FOnMouseMove(Sender,Shift,X,Y);
  if not initialized then exit;
  P1 := screen_to_real(Point(x,y));
  s1 := FloatTostrF(P1.x,ffFixed,9,2);
  s2 := FloatTostrF(P1.y,ffFixed,9,2);
  Coords.Caption := '('+s1+', '+s2+')';
end;
///////////////////////////////////////////////////////////////////////////////
// Scrolling and recentring
///////////////////////////////////////////////////////////////////////////////
procedure Zoom_panel.ScrollBars_Scroll(Sender:TObject; ScrollCode:TScrollCode; var ScrollPos:Integer);
var acx,acy,cx,cy : double;
    moved         : integer;
begin
  if scrollpos<>(Sender as TScrollbar).position then with current_params do begin
    if sender=scrollbar_lr then begin
      cx   := xmid;
      moved := ScrollPos - scrollbar_lr.position;
      xmid := xmid + moved*(xmax - xmin)/50;
      xmin := xmin - (cx-xmid);
      xmax := xmax - (cx-xmid);
    end;
    if sender=scrollbar_ud then begin
      cy   := ymid;
      moved := ScrollPos - scrollbar_ud.position;
      ymid := ymid - moved*(ymax - ymin)/50;
      ymin := ymin - (cy-ymid);
      ymax := ymax - (cy-ymid);
    end;
    Refresh;
  end;
end;

procedure Zoom_panel.ReCentre(mx,my:integer); // screen coords (mouse)
var mid   : Point2D;
    xt,yt : double;
begin
  if not initialized then exit;
  mid := screen_to_real(Point(mx,my));
  with current_params do begin
    xmid   := mid.x;
    ymid   := mid.y;
    xt     := (xmax-xmin)*xscale;
    yt     := (ymax-ymin)*yscale;
    xmax   := xmid+(xt/xscale)/2;
    xmin   := xmid-(xt/xscale)/2;
    ymax   := ymid+(yt/yscale)/2;
    ymin   := ymid-(yt/yscale)/2;
  end;
  ClientArea.Refresh;
  ScrollBar_lr.Position     := 50;
  ScrollBar_ud.Position     := 50;
end;
///////////////////////////////////////////////////////////////////////////////
// Section dealing with Zooming in/out
///////////////////////////////////////////////////////////////////////////////
procedure Zoom_panel.Zoom(factor:double);
var xmid,ymid,xt,yt : double;
begin
  if not initialized then exit;
  with current_params do begin
    xmid   := (xmin+xmax)/2;
    ymid   := (ymin+ymax)/2;
    xt     := (xmax-xmin)*xscale;
    yt     := (ymax-ymin)*yscale;
    xscale := xscale/factor;
    yscale := yscale/factor;
    xmax   := xmid+(xt/xscale)/2;
    xmin   := xmid-(xt/xscale)/2;
    ymax   := ymid+(yt/yscale)/2;
    ymin   := ymid-(yt/yscale)/2;
  end;
  ClientArea.Refresh;
  ScrollBar_lr.Position     := 50;
  ScrollBar_ud.Position     := 50;
end;

procedure Zoom_panel.Zoom_in_out(Sender:TObject);
begin
  if not initialized then exit;
  if Sender=zoom_out_button then begin
    zoom(1/0.8);
    If Assigned(FOnZoomout) then FOnZoomout(self);
  end
  else if Sender=zoom_in_button then begin
    zoom(0.8);
    If Assigned(FOnZoomin) then FOnZoomin(self);
  end
  else if Sender=zoomresetbutton then begin
    current_params := original_params;
    Zoom(1);
    If Assigned(FOnZoomreset) then FOnZoomreset(self);
  end;
end;

procedure Zoom_panel.Zoom_mousedown(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
begin
  if Sender=zoom_in_button then begin
    zooming_in := true; zooming_out := false;
    zoomtimer.enabled := true;
    zoomtimer.Interval := 500;   // initial pause of 0.5 seconds
    Zoom_in_out(zoom_in_button);
  end
  else if Sender=zoom_out_button then begin
    zooming_out := true; zooming_in := false;
    zoomtimer.enabled := true;
    zoomtimer.Interval := 500;   // initial pause of 0.5 seconds
    Zoom_in_out(zoom_out_button);
  end;
end;

procedure Zoom_panel.Zoom_mouseup(Sender:TObject; Button:TMouseButton; Shift:TShiftState; X,Y:Integer);
begin
  zooming_in        := false;
  zooming_out       := false;
  zoomtimer.enabled := false;
end;

procedure Zoom_panel.Zoom_timer_event(Sender:TObject);
begin
  zoomtimer.Interval := 25;   // now go for speedy zooming
  if zooming_in then Zoom_in_out(zoom_in_button)
  else if zooming_out then Zoom_in_out(zoom_out_button);
end;
///////////////////////////////////////////////////////////////////////////////
// Initialization of window scaling etc.
///////////////////////////////////////////////////////////////////////////////
procedure Zoom_panel.set_parameters(xmn,xmx,ymn,ymx:double; xm,ym:integer; aspect:double);
var tempx,tempy : integer;
begin
  with original_params do begin
    if aspect<>0 then AspectRatio := aspect
    else AspectRatio := 1;
    tempx := ClientArea.Width  -2*xm;
    tempy := ClientArea.Height -2*ym;
    if tempx<(tempy*AspectRatio) then begin
      if (xmx-xmn)<>0 then xscale := tempx/(xmx-xmn)
      else xscale := 1;
      if (ymx-ymn)<>0 then yscale := tempx/((ymx-ymn)*AspectRatio)
      else yscale := 1;
    end else begin
      if (xmx-xmn)<>0 then xscale := tempy*AspectRatio/(xmx-xmn)
      else xscale := 1;
      if (ymx-ymn)<>0 then yscale := tempy/(ymx-ymn)
      else yscale := 1;
    end;
    if AspectRatio=1 then begin
      if xscale<yscale then yscale:=xscale else xscale:=yscale;
    end;
    xmid := (xmx+xmn)/2;
    ymid := (ymx+ymn)/2;
    xmin := xmid - (ClientArea.Width/2)/xscale;
    xmax := xmid + (ClientArea.Width/2)/xscale;
    ymin := ymid - (ClientArea.Height/2)/yscale;
    ymax := ymid + (ClientArea.Height/2)/yscale;
  end;
  current_params := original_params;
  initialized    := true;
end;
///////////////////////////////////////////////////////////////////////////////
// Coordinate transformations
///////////////////////////////////////////////////////////////////////////////
function Zoom_panel.real_to_screen(x,y:double) : TPoint;
begin
  with current_params do begin
    result.x := round((x-xmin)*xscale);
    result.y := round((ymax-y)*yscale);
  end;
end;

function Zoom_panel.screen_to_real(P1:TPoint) : Point2D;
begin
  with current_params do begin
    result.x := P1.x/xscale + xmin;
    result.y := ymax -P1.y/yscale;
  end;
end;

end.
