// DXF File reader object/code
// John Biddiscombe J.Biddiscombe@rl.ac.uk
//
// Thanks very much to John F Herbster for the original DXF class that got this
// started --- extract from his header follows...
//
// Pgm. 07/14/95 by John F Herbster, CIS:72714,3445, Houston, TX.
// for Rick Rogers (CIS:74323,3573).

unit DXF_read;

interface

uses
  { Borland }
  Windows,SysUtils,StdCtrls,ComCtrls,Dialogs,Classes,Graphics,
  { Mine }
  DXF_structs,DXF_Utils,Thinkbox;

///////////////////////////////////////////////////////////////////////////////
// DXF_Reader class definition
///////////////////////////////////////////////////////////////////////////////
Const
  MaxSizeOfBuf = 4096;

type
  tCharArray = array [0..MaxSizeOfBuf-1] of char;

type
  abstract_entity = class;

  DXF_Reader = class
  private
    IO_chan    : file;
    SizeOfBuf  : integer;
    num_in_buf : integer;
    ii         : integer;
    EC,fCode   : integer;
    pBuf       : ^tCharArray;
    Line_num   : longint;
    fLine      : shortstring;

    progress   : TProgressBar;
    // special additions to make parsing easier...
    backflag  : boolean;
    procedure   go_back_to_last(code:integer; str:shortstring);
    // end of special additions
    function    NextGroupCode: integer;
    function    ValStr: shortstring;
    function    ValDbl: double;
    function    ValInt: integer;
    function    code_and_string(var group:integer; var s:string) : boolean;
    function    code_and_double(var group:integer; var d:double) : boolean;
    function    read_2Dpoint(var p1:Point3D)                     : boolean;
    function    skip_upto_section(name:string)                   : boolean;
    function    read_entity_data(ent:abstract_entity)            : boolean;
    function    read_generic(var layer:integer)                  : abstract_entity;
  public
    // Extents in (x,y) of the dataset
    min_extents    : Point3D;
    max_extents    : Point3D;
    // We will read the Entities in the layers into this list
    DXF_Layers     : TList;
    colour_BYLAYER : boolean;
    Aspect         : double;
    skipped        : TStrings;
    // Constructors and destructors
    Constructor Create (const aName: shortstring);
    Destructor  Destroy;                           override;
    // Header section
    function    move_to_header_section : boolean;
    function    read_header            : boolean;
    function    get_min_extent         : Point3D;
    function    get_max_extent         : Point3D;
    // Blocks section
    function    move_to_blocks_section : boolean;
    function    read_blocks            : boolean;
    // Tables section
    function    move_to_tables_section : boolean;
    function    read_tables : boolean;
    function    read_layer_information : boolean;
    function    read_vport_information : boolean;
    function    layer_num(layername:string) : integer;
    // Entities section
    function    move_to_entity_section : boolean;
    function    read_entities          : boolean;
    // entities that we can handle
    function    read_point    : boolean;
    function    read_insert   : boolean;
    function    read_text     : boolean;
//    function    read_attrib   : boolean;
    function    read_line     : boolean;
    function    read_polyline : boolean;
    function    read_circle   : boolean;
    function    read_arc      : boolean;
    // These are the main routines to use
    function    read_file                 : boolean;
    function    remove_empty_layers       : boolean;
    function    release_control_of_layers : TList;
    procedure   set_skipped_list(s:TStrings);
  end;

///////////////////////////////////////////////////////////////////////////////
// This is a simple class used only during file reads, it should not be used
// as a base for any objects. It is to simplify problems like...
// some files have 2D coords, some 3D, circles & arcs etc have extra group
// codes etc. etc. etc.
// Add extra group codes if you need to recognize them
///////////////////////////////////////////////////////////////////////////////
  abstract_entity = class
    p1,p2    : Point3D;
    radius_h : double;
    angle1   : double;
    angle2   : double;
    colour   : integer;
    flag_70  : integer;
    attflag  : integer;
    s        : string;
    tagstr   : string;
    layer    : string;
    procedure setx1(d:double);
    procedure sety1(d:double);
    procedure setz1(d:double);
    procedure setx2(d:double);
    procedure sety2(d:double);
    procedure setz2(d:double);
    procedure setradius_or_height(d:double);
    procedure setang1(d:double);
    procedure setang2(d:double);
    procedure setcolour(c:integer);
    procedure setattflag(i:integer);
    procedure setflag_70(i:integer);
    procedure setstring(s_:string);
    procedure settag(s_:string);
    procedure setlayer(s_:string);
  end;

///////////////////////////////////////////////////////////////////////////////
// DXF File read exceptions will be this type
///////////////////////////////////////////////////////////////////////////////
type
  DXF_read_exception = class(Exception)
    line_number : integer;
    constructor create(err_msg:string; line:integer);
  end;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// implementation
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
implementation

{ --------------------------------------------------------------------------- }
{ -------------------             DXFReader           ----------------------- }
{ --------------------------------------------------------------------------- }
Constructor DXF_Reader.Create (const aName: shortstring);
begin
  Inherited Create;
  AssignFile(IO_chan,aName);
  Reset(IO_chan,1);
  SizeOfBuf:=MaxSizeOfBuf;
  GetMem(pBuf,SizeOfBuf);
  DXF_Layers        := TList.Create;
  colour_BYLAYER    := false;
  Line_num          := 0;
  Aspect            := 1;
  backflag          := false;
  progress          := Thinking_box.bar;
  progress.position := 0;
  progress.max      := FileSize(IO_chan) div MaxSizeOfBuf;
  min_extents       := aPoint3D(0,0,0);
  max_extents       := aPoint3D(0,0,0);
end;

destructor DXF_Reader.Destroy;
var lp1 : integer;
begin
  if (DXF_Layers<>nil) then
    for lp1 := 0 to DXF_Layers.count-1 do DXF_Layer(DXF_Layers[lp1]).Free;
  DXF_Layers.Free;
  CloseFile(IO_chan);
  FreeMem(pBuf,SizeOfBuf);
  Inherited Destroy;
end;
{ --------------------------------------------------------------------------- }
{ Routines for fetching codes and values
{ --------------------------------------------------------------------------- }
procedure DXF_Reader.go_back_to_last(code:integer; str:shortstring);
begin
  fCode    := code;
  fLine    := str;
  backflag := true;
end;

function DXF_Reader.NextGroupCode: integer;
  function GotMore: boolean;
  begin
    BlockRead(IO_chan,pBuf^,SizeOfBuf,num_in_buf); ec:=IoResult; ii:=0;
    If (ec=0) and (num_in_buf=0) then ec:=-1; GotMore:=(ec=0);
    progress.position := progress.position+1;
  end{GotMore};

  function GotLine: boolean;
  const CR=#13; LF=#10;
  var c: char;
  begin
    byte(fLine[0]):=0;
    While (ii<num_in_buf) or GotMore do begin
      c:=pBuf^[ii]; inc(ii);
      If (c<>CR) and (length(fLine)<255) then begin
        inc(fLine[0]); fLine[length(fLine)]:=c
      end
      else if c=CR then
        if (ii<num_in_buf) or GotMore then begin
          if pBuf^[ii]=LF then begin inc(ii); break; end;
        end;
    end;
    GotLine:=(ec=0) and (c=CR);
    inc(Line_num);
  end;

begin {NextGroupCode}
  if backflag then begin
    result   := fCode;
    backflag := false;
  end
  else begin
    If GotLine then Val(fLine,fCode,ec);
    If ec<>0 then fCode:=-2
    else if not GotLine then fCode:=-2;
    Result:=fCode;
  end;
end {NextGroupCode};

function DXF_Reader.ValStr: shortstring;
begin Result:=fLine end;

function DXF_Reader.ValDbl: double;
begin
  Val(fLine,Result,ec);
  If ec<>0 then raise DXF_read_exception.Create('Invalid Floating point conversion',line_num);
end;

function DXF_Reader.ValInt: integer;
begin
  Val(fLine,Result,ec);
  If ec<>0 then raise DXF_read_exception.Create('Invalid Integer conversion',line_num);
end;

function DXF_Reader.code_and_string(var group:integer; var s:string) : boolean;
var astr : string;
begin
  result := true;
  group  := NextGroupCode;
  if group>=0 then s := ValStr
  else result := false;
  // useful in debugging
  //  if (group=0) then begin astr := IntToStr(group)+' '+s; alb.Items.Add(astr); end;
end;

function DXF_Reader.code_and_double(var group:integer; var d:double) : boolean;
begin
  result := true;
  group  := NextGroupCode;
  if group>=0 then d := Valdbl
  else result := false;
end;

// This routine is just for the $EXT and should be used with care....
function DXF_Reader.read_2Dpoint(var p1:Point3D) : boolean;
var Groupcode : integer;
begin
  repeat Groupcode:=NextGroupCode;
  until (Groupcode=10) or (Groupcode<0);
  if Groupcode<0 then begin result:=false; exit; end;
  p1.x := Valdbl;
  result := code_and_double(Groupcode,p1.y);  { y next              }
end;

function DXF_Reader.skip_upto_section(name:string) : boolean;
var Group   : integer;
    s       : string;
begin
  result := false;
  repeat
    if not code_and_string(Group,s) then break;
    if (Group=0) then begin
      if (s='SECTION') then begin
        if not code_and_string(Group,s) then break;
        if (Group=2) and (s=name) then result := true
        else if skipped<>nil then Skipped.Add(s);
      end else if skipped<>nil then Skipped.Add(s);
    end;
  until (result);
end;
{ --------------------------------------------------------------------------- }
{ Header section
{ --------------------------------------------------------------------------- }
function DXF_Reader.move_to_header_section : boolean;
begin
  result := skip_upto_section('HEADER');
end;

function DXF_Reader.read_header : boolean;
var Group : integer;
    s     : string;
begin
  result := false;
  repeat
    if not code_and_string(Group,s) then break;
    if (group=9) and (s='$EXTMAX') then begin
      if not read_2Dpoint(max_extents) then break;
    end;
    if (group=9) and (s='$EXTMIN') then begin
      if not read_2Dpoint(min_extents) then break;
    end;
    if (group=9) and (s='$CECOLOR') then begin
      if (NextGroupCode=62) and (ValInt=256) then colour_BYLAYER := true;
    end;
    result := (Group=0) and (s='ENDSEC');
  until result;
end;

function DXF_Reader.get_min_extent : Point3D;
begin
  result := min_extents;
end;

function DXF_Reader.get_max_extent : Point3D;
begin
  result := max_extents;
end;
{ --------------------------------------------------------------------------- }
{ Blocks section
{ --------------------------------------------------------------------------- }
function DXF_Reader.move_to_blocks_section : boolean;
begin
  result := skip_upto_section('BLOCKS');
end;

function DXF_Reader.read_blocks : boolean;
begin
  // I don't need any block info yet...
  result := true;
end;
{ --------------------------------------------------------------------------- }
{ Tables (Layers - VPort) section
{ --------------------------------------------------------------------------- }
function DXF_Reader.move_to_tables_section : boolean;
begin
  result := skip_upto_section('TABLES');
end;

function DXF_Reader.read_tables : boolean;
var Group : integer;
    s     : string;
begin
  result := false;
  repeat
    if not code_and_string(Group,s) then break;
    if (Group=0) and (s='TABLE') then begin
      if not code_and_string(Group,s) then break;
      if (Group=2) then begin
        if (s='LAYER') then read_layer_information
        else if (s='VPORT') then read_vport_information
        else if skipped<>nil then Skipped.Add(s);
      end;
    end;
    result := (Group=0) and (s='ENDSEC');
  until result;
end;

function DXF_Reader.read_layer_information : boolean;
var Group,Lay_num : integer;
    s             : string;
begin
  lay_num := -1;
  result  := false;
  repeat
    if not code_and_string(Group,s) then break;
    if (Group=0) then begin
      if (s='LAYER') then begin
        if not code_and_string(Group,s) then break;
        if (Group=2) then lay_num := DXF_Layers.Add(DXF_Layer.create(s));
      end
      else if (s='ENDTAB') then result := true
      else if skipped<>nil then Skipped.Add(s);
    end
    else if (Group=62) and (lay_num<>-1) then
      DXF_Layer(DXF_Layers[lay_num]).Colour := ValInt;
  until result;
end;

function DXF_Reader.read_vport_information : boolean;
var Group : integer;
    s     : string;
begin
  result := false;
  repeat
    if not code_and_string(Group,s) then break;
    if (Group=0) then begin
      if (s='VPORT') then begin
        if not code_and_string(Group,s) then break;
        if (Group=2) then begin
          if (s='*ACTIVE') then repeat
            if not code_and_string(Group,s) then break;
            if (Group=41) then Aspect := ValDbl;
            result := (Group=0) and (s='ENDTAB');
          until (result)
          else if skipped<>nil then Skipped.Add(s);
        end;
      end
      else if skipped<>nil then Skipped.Add(s);
    end
  until (result);
end;

function DXF_Reader.layer_num(layername:string) : integer;
var lp1 : integer;
begin
  result := -1;
  for lp1:=0 to DXF_Layers.count-1 do begin
    if DXF_Layer(DXF_Layers[lp1]).name=layername then begin
      result := lp1;
      exit;
    end;
  end;
end;
{ --------------------------------------------------------------------------- }
{ Entities section
{ --------------------------------------------------------------------------- }
function DXF_Reader.move_to_entity_section : boolean;
begin
  result := skip_upto_section('ENTITIES');
end;

function DXF_Reader.read_entities : boolean;
var Groupcode : integer;
    s         : string;
begin
  result := false;
  repeat
    try
      if not code_and_string(Groupcode,s) then break;
      if (Groupcode=0) then begin
        if (s='POINT') then begin if not read_point then
          raise DXF_read_exception.Create('Error reading Insert entity',line_num); end
        else if (s='INSERT') then begin if not read_insert then
          raise DXF_read_exception.Create('Error reading Insert entity',line_num); end
        else if (s='TEXT') then begin if not read_text then
          raise DXF_read_exception.Create('Error reading Text entity',line_num); end
{ Attribs are tagged to Inserts
        else if (s='ATTRIB') then begin if not read_attrib then
          raise DXF_read_exception.Create('Error reading Attrib entity',line_num); end
}
        else if (s='LINE') then begin if not read_line then
          raise DXF_read_exception.Create('Error reading Line entity',line_num); end
        else if (s='POLYLINE') then begin if not read_polyline then
          raise DXF_read_exception.Create('Error reading Polyline entity',line_num); end
        else if (s='CIRCLE') then begin if not read_circle then
          raise DXF_read_exception.Create('Error reading Circle entity',line_num); end
        else if (s='ARC') then begin if not read_arc then
          raise DXF_read_exception.Create('Error reading Arc entity',line_num); end
        else if (s='ENDSEC') then result := true
        else if skipped<>nil then Skipped.Add(s);
      end;
    except
      on E:DXF_read_exception do begin
        stopped_thinking;
        if MessageBox(0,@E.message[1],'DXF read error warning',MB_OKCANCEL)=IDCANCEL then
          raise DXF_read_exception.Create('User aborted',-1);
        thinking_bar(nil,'Reading DXF file...');
      end;
      on E:Exception do Showmessage(E.Message);
    end;
  until result;
end;
{ --------------------------------------------------------------------------- }
{ Entity reading code
{ --------------------------------------------------------------------------- }
function DXF_Reader.read_entity_data(ent:abstract_entity) : boolean;
var Groupcode : integer;
begin
  repeat
    Groupcode := NextGroupCode;
    case Groupcode of
      8  : ent.setlayer(ValStr);
      10 : ent.setx1(Valdbl);
      20 : ent.sety1(Valdbl);
      30 : ent.setz1(Valdbl);
      40 : ent.setradius_or_height(Valdbl);
      50 : ent.setang1(Valdbl);
      51 : ent.setang2(Valdbl);
      11 : ent.setx2(Valdbl);
      21 : ent.sety2(Valdbl);
      31 : ent.setz2(Valdbl);
      70 : ent.setflag_70(ValInt);
      62 : ent.setcolour(ValInt);
      66 : ent.setattflag(ValInt);
      1  : ent.setstring(ValStr);
      2  : ent.settag(ValStr);
    end;
  until (Groupcode<=0); // end or fault;
  if Groupcode<0 then begin result:=false; exit; end;
  // we need to put the code=0, and valstr back, so the next entity starts
  // with the zero when neccessary
  go_back_to_last(Groupcode,fline);
  result := true;
end;

function DXF_Reader.read_generic(var layer:integer) : abstract_entity;
var ent : abstract_entity;
    s   : string;
begin
  result := nil;
  ent := abstract_entity.create;
  if read_entity_data(ent) then begin
    layer := layer_num(ent.layer);
    if layer>=0 then begin
      result := ent;
      exit;
    end else begin
      s := ent.layer;
      ent.Free;
      raise DXF_read_exception.Create('Invalid Layer ('+s+')',line_num);
    end;
  end;
end;

function DXF_Reader.read_point : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Point_.create(ent.p1,ent.colour));
    ent.Free;
    result := true;
  end;
end;

function DXF_Reader.read_text : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Text_.create(ent.p1,ent.s,ent.radius_h,ent.colour));
    ent.Free;
    result := true;
  end;
end;

function DXF_Reader.read_insert : boolean;
var ent,ent2 : abstract_entity;
    code,num : integer;
    layer    : integer;
    atts     : array[0..255] of Attrib_;
begin
  result := true;
  num := 0;
  ent := read_generic(layer);
  if ent<>nil then begin
    if ent.attflag=1 then begin
      repeat
        result := (Nextgroupcode=0);
        if result and (ValStr='ATTRIB') then begin
          ent2 := read_generic(layer);
          if ent2<>nil then begin
            atts[num] := Attrib_.create(ent2.p1,ent2.s,ent2.tagstr,ent2.radius_h,ent2.colour);
            ent2.Free;
            inc(num);
          end else result := false;
        end;
      until (not result) or (ValStr='SEQEND');
      if result then Nextgroupcode; // remove the SEQEND put back 
    end;
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Insert_.create(ent.p1,ent.colour,num,@atts[0]));
    ent.Free;
  end
  else result := false;
end;
{
function DXF_Reader.read_attrib : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Attrib_.create(ent.p1,ent.s,ent.tagstr,ent.radius_h,ent.colour));
    ent.Free;
    result := true;
  end;
end;
}
function DXF_Reader.read_line : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Line_.create(ent.p1,ent.p2,ent.colour));
    ent.Free;
    result := true;
  end;
end;

function DXF_Reader.read_polyline : boolean;
var ent1,ent2    : abstract_entity;
    vertices,lp1 : integer;
    exceeded     : boolean;
    v1,v2        : integer;
    tempvert     : array[0..1023] of Point3D;
    layercode    : integer;
    closed_poly  : boolean;
begin
  result := false; exceeded := false;
  closed_poly := false;
  ent1   := abstract_entity.create;
  // read initial polyline data
  if not read_entity_data(ent1) then begin ent1.Free; exit; end;
  layercode := layer_num(ent1.layer);
  if layercode=-1 then exit;
  vertices  := 0;
  // read each vertex
  ent2   := abstract_entity.create;
  repeat
    if (NextGroupCode=0) and (ValStr = 'VERTEX') then begin
      if read_entity_data(ent2) then begin
        tempvert[vertices] := ent2.p1;
        inc(vertices);
      end else begin ent1.Free; ent2.Free; exit; end;
    end;
  until fLine='SEQEND';
  // this should set result to true, because 0 SEQEND is next
  result := NextGroupCode=0;
  if ((ent1.Flag_70) and 1)=1 then closed_poly := true;
  if vertices<=max_vertices_per_polyline then begin
    DXF_Layer(DXF_Layers[layercode]).add_entity_to_layer(
      Polyline_.create(vertices,@tempvert[0],ent1.colour,closed_poly));
  end else begin
    closed_poly := false;
    exceeded   := true;
    v1         := 0;
    v2         := vertices;
    while v2>max_vertices_per_polyline do begin
      DXF_Layer(DXF_Layers[layercode]).add_entity_to_layer(
        Polyline_.create(max_vertices_per_polyline,@tempvert[v1],ent1.colour,closed_poly));
      v2 := v2 - max_vertices_per_polyline;
      v1 := v1 + max_vertices_per_polyline;
    end;
    DXF_Layer(DXF_Layers[layercode]).add_entity_to_layer(
      Polyline_.create(v2,@tempvert[v1],ent1.colour,closed_poly));
  end;
  ent1.Free; ent2.Free;
  if exceeded then
    raise DXF_read_exception.Create('Polyline contained '+IntToStr(vertices)+
      ' points : Broken into multiple segments',line_num);
end;

function DXF_Reader.read_circle : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Circle_.create(ent.p1,ent.radius_h,ent.colour));
    ent.Free;
    result := true;
  end;
end;

function DXF_Reader.read_arc : boolean;
var ent   : abstract_entity;
    layer : integer;
begin
  ent := read_generic(layer);
  if ent<>nil then begin
    DXF_Layer(DXF_Layers[layer]).add_entity_to_layer(
      Arc_.create(ent.p1,ent.radius_h,ent.angle1,ent.angle2,ent.colour));
    ent.Free;
    result := true;
  end;
end;
///////////////////////////////////////////////////////////////////////////////
// Main routines to use
///////////////////////////////////////////////////////////////////////////////
function DXF_Reader.read_file : boolean;
var lp1 : integer;
begin
  result := true;
  thinking_bar(nil,'Reading DXF file...');
  try
    if not (move_to_header_section and read_header) then
      raise DXF_read_exception.Create('No Header or invalid Header section in DXF file',-1);
    if not (move_to_tables_section and read_tables) then
      raise DXF_read_exception.Create('No Layers or invalid Tables section in DXF file',-1);
    if not (move_to_entity_section and read_entities) then
      raise DXF_read_exception.Create('No Entities or invalid Entities section in DXF file',-1);
  except
    on E:DXF_read_exception do begin
      stopped_thinking;
      MessageBox(0,@E.message[1],'DXF Read Error',MB_ICONWARNING);
    end;
    on E:EAccessViolation do begin
      stopped_thinking;
      MessageDlg(E.message, mtWarning, [mbOK], 0);
    end;
  end;
  if p1_eq_p2_3D(min_extents,aPoint3D(0,0,0)) or p1_eq_p2_3D(max_extents,aPoint3D(0,0,0)) then begin
    thinking(nil,'File contained no Max/Min extents. Scanning...');
    sleep(750); // just a delay to let the message be visible
    for lp1:=0 to DXF_layers.count-1 do
      DXF_Layer(DXF_Layers[lp1]).max_min_extents(max_extents,min_extents);
  end;
  stopped_thinking;
end;

function DXF_Reader.remove_empty_layers : boolean;
var lp1   : integer;
    layer : DXF_layer;
begin
   for lp1 := DXF_Layers.count-1 downto 0 do begin
     layer :=  DXF_Layers[lp1];
     if layer.num_lists=0 then begin
       DXF_Layers.Remove(layer);
       layer.Free;
     end;
  end;
  result := (DXF_Layers.count>0);
end;

// Hand over ownership of the layers, the owner of the entity lists
// is now responsible for their destruction
function DXF_Reader.release_control_of_layers : TList;
begin
  result     := DXF_Layers;
  DXF_Layers := nil;
end;

// Since we're not reading all groupcodes, we offer the chance
// to dump the main titles into a list so we can see what
// we've missed
procedure DXF_Reader.set_skipped_list(s:TStrings);
begin
  skipped := s;
end;
///////////////////////////////////////////////////////////////////////////////
// abstract_entity class member functions
///////////////////////////////////////////////////////////////////////////////
procedure abstract_entity.setx1(d:double);
begin p1.x := d; end;
procedure abstract_entity.sety1(d:double);
begin p1.y := d; end;
procedure abstract_entity.setz1(d:double);
begin p1.z := d; end;
procedure abstract_entity.setx2(d:double);
begin p2.x := d; end;
procedure abstract_entity.sety2(d:double);
begin p2.y := d; end;
procedure abstract_entity.setz2(d:double);
begin p2.z := d; end;
procedure abstract_entity.setradius_or_height(d:double);
begin radius_h := d; end;
procedure abstract_entity.setang1(d:double);
begin angle1 := d; end;
procedure abstract_entity.setang2(d:double);
begin angle2 := d; end;
procedure abstract_entity.setcolour(c:integer);
begin colour := c; end;
procedure abstract_entity.setattflag(i:integer);
begin attflag := i; end;
procedure abstract_entity.setflag_70(i:integer);
begin flag_70 := i; end;
procedure abstract_entity.setstring(s_:string);
begin s := s_; end;
procedure abstract_entity.settag(s_:string);
begin tagstr := s_; end;
procedure abstract_entity.setlayer(s_:string);
begin layer := s_; end;
///////////////////////////////////////////////////////////////////////////////
// DXF File exception
///////////////////////////////////////////////////////////////////////////////
constructor DXF_read_exception.create(err_msg:string; line:integer);
begin
  if line>-1 then
    message := err_msg + #13#10 + 'Error occured at or near line number ' + IntToStr(line)
  else message := err_msg;
end;

initialization
end.


