unit DXF_Utils;

interface

uses
  Math, SysUtils, DXF_Structs;

///////////////////////////////////////////////////////////////////////////////
// General point 3D stuff
///////////////////////////////////////////////////////////////////////////////
function aPoint3D(a,b,c:double)       : Point3D;
function Point3DToStr(p:Point3D)      : string;
function p1_eq_p2_3D(p1,p2:Point3D)   : boolean;
function p1_eq_p2_2D(p1,p2:Point3D)   : boolean;
function p1_minus_p2(p1,p2:Point3D)   : Point3D;
function p1_plus_p2 (p1,p2:Point3D)   : Point3D;
function mag(p1:Point3D)              : double;
function dist3D(p1,p2:Point3D)        : double;
function sq_dist3D(p1,p2:Point3D)     : double;
function sq_dist2D(p1,p2:Point3D)     : double;
///////////////////////////////////////////////////////////////////////////////
// Vector 3D stuff
///////////////////////////////////////////////////////////////////////////////
function dot_product(p1,p2:Point3D)   : double;
function cross_product(p1,p2:Point3D) : Point3D;
function angle(p1,p2,p3:Point3D; do_3D:boolean) : double;
///////////////////////////////////////////////////////////////////////////////
// Bounds
///////////////////////////////////////////////////////////////////////////////
procedure max_bound(var bounds:Point3D; point:Point3D);
procedure min_bound(var bounds:Point3D; point:Point3D);
function  dmin(a,b:double) : double;
function  dmax(a,b:double) : double;
///////////////////////////////////////////////////////////////////////////////
// Memory
///////////////////////////////////////////////////////////////////////////////
function  allocate_points(n:integer) : ppointlist;
procedure deallocate_points(pts:ppointlist; n:integer);

implementation

function aPoint3D(a,b,c:double) : Point3D;
begin
  result.x := a; result.y := b; result.z := c;
end;

function Point3DToStr(p:Point3D) : string;
begin
  result := '(' + FloatToStrF(p.x,ffFixed,7,2) + ','#9 +
                  FloatToStrF(p.y,ffFixed,7,2) + ','#9 +
                  FloatToStrF(p.z,ffFixed,7,2) + ')';
end;

function p1_eq_p2_3D(p1,p2:Point3D) : boolean;
begin
  result := (p1.x=p2.x) and (p1.y=p2.y) and (p1.z=p2.z);
end;

function p1_eq_p2_2D(p1,p2:Point3D) : boolean;
begin
  result := (p1.x=p2.x) and (p1.y=p2.y);
end;

function p1_minus_p2(p1,p2:Point3D) : Point3D;
begin
  result.x := p1.x-p2.x;
  result.y := p1.y-p2.y;
  result.z := p1.z-p2.z;
end;

function p1_plus_p2(p1,p2:Point3D) : Point3D;
begin
  result.x := p1.x+p2.x;
  result.y := p1.y+p2.y;
  result.z := p1.z+p2.z;
end;

function mag(p1:Point3D) : double;
begin
  with p1 do result := Sqrt( sqr(x) + sqr(y) + sqr(z) );
end;

function dist3D(p1,p2:Point3D) : double;
begin
  with p1_minus_p2(p2,p1) do result := Sqrt( sqr(x) + sqr(y) + sqr(z) );
end;

function sq_dist3D(p1,p2:Point3D) : double;
begin
  with p1_minus_p2(p2,p1) do result := sqr(x) + sqr(y) + sqr(z);
end;

function sq_dist2D(p1,p2:Point3D) : double;
begin
  with p1_minus_p2(p2,p1) do result := sqr(x) + sqr(y);
end;
///////////////////////////////////////////////////////////////////////////////
// Vector 3D stuff
///////////////////////////////////////////////////////////////////////////////
function dot_product(p1,p2:Point3D) : double;
begin
  result := p1.x*p2.x + p1.y*p2.y + p1.z*p2.z;
end;

function cross_product(p1,p2:Point3D) : Point3D;
begin
  result.x := p1.y*p2.z - p1.z*p2.y;
  result.y := p1.z*p2.x - p1.x*p2.z;
  result.z := p1.x*p2.y - p1.y*p2.x;
end;

function angle(p1,p2,p3:Point3D; do_3D:boolean) : double;
var v1,v2 : Point3D;
    d1,d2 : double;
begin
  v1 := p1_minus_p2(p2,p1);
  v2 := p1_minus_p2(p3,p2);
  if not do_3D then begin v1.z := 0; v2.z := 0; end;
  d1 := Mag(v1);
  d2 := Mag(v2);
  if ((d1=0) or (d2=0)) then result := 0
  else begin
    d1 := dot_product(v1,v2)/(d1*d2);
    if abs(d1)<=1 then result := ArcCos(d1)
    else result := 0;
  end;
end;
///////////////////////////////////////////////////////////////////////////////
// Bounds
///////////////////////////////////////////////////////////////////////////////
procedure max_bound(var bounds:Point3D; point:Point3D);
begin
  if point.x>bounds.x then bounds.x := point.x;
  if point.y>bounds.y then bounds.y := point.y;
  if point.z>bounds.z then bounds.z := point.z;
end;

procedure min_bound(var bounds:Point3D; point:Point3D);
begin
  if point.x<bounds.x then bounds.x := point.x;
  if point.y<bounds.y then bounds.y := point.y;
  if point.z<bounds.z then bounds.z := point.z;
  if bounds.x<0 then begin
    bounds.x := bounds.x;
  end;
end;

function dmin(a,b:double) : double;
begin
  if a<b then result := a else result := b;
end;

function dmax(a,b:double) : double;
begin
  if a>b then result := a else result := b;
end;
///////////////////////////////////////////////////////////////////////////////
// Memory
///////////////////////////////////////////////////////////////////////////////
function allocate_points(n:integer) : ppointlist;
begin
  Getmem(result,n*SizeOf(Point3D));
end;

procedure deallocate_points(pts:ppointlist; n:integer);
begin
  Freemem(pts,n*SizeOf(Point3D));
end;

end.
