unit DXF_structs;

interface

uses
  { Borland }
  Windows,Classes,Graphics,SysUtils,Dialogs,Math;

///////////////////////////////////////////////////////////////////////////////
// Useful definitions
///////////////////////////////////////////////////////////////////////////////
const
  max_vertices_per_polyline = 1024;

type
  coord_convert = function(x,y:double) : TPoint of Object;

type
  file_type = (off,geo,pslg);

type
  PPoint3D  = ^Point3D;
  Point3D   = record
    x,y,z   : double;
  end;

  ppointlist = ^pointlist;
  pointlist  = array[0..0] of Point3D;

  pintlist = ^intlist;
  intlist  = array[0..0] of integer;

  pattrlist = ^attrlist;
  attrlist  = array[0..0] of double;

///////////////////////////////////////////////////////////////////////////////
// DXF_Entity - abstract base class - override where neccessary
// All DXF objects will become sub classes of this
///////////////////////////////////////////////////////////////////////////////
type
  DXF_Entity = class
    colour : TColor;
    colinx : integer;
    constructor create;
    destructor  destroy; override;
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); virtual; abstract;
    procedure   DrawVertices(acanvas:TCanvas; map_fn:coord_convert); virtual; abstract;
    procedure   setcolour(col:integer);
    function    count_points   : integer; virtual;
    function    count_lines    : integer; virtual;
    function    count_polys_open   : integer; virtual;
    function    count_polys_closed : integer; virtual;
    function    proper_name    : string;  virtual;
    function    details : string; virtual; abstract;
    procedure   write_to_DXF(var IO:textfile; layer:string);    virtual;
    procedure   max_min_extents(var emax,emin:Point3D);     virtual; abstract;
    function    closest_vertex_square_distance_2D(p:Point3D) : double;  virtual; abstract;
    function    closest_vertex(p:Point3D) : Point3D; virtual; abstract;
  end;
///////////////////////////////////////////////////////////////////////////////
// Point Definition
///////////////////////////////////////////////////////////////////////////////
type
  Point_ = class(DXF_Entity)
    p1 : Point3D;
    constructor create(p:Point3D; col:integer);
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    procedure   DrawVertices(acanvas:TCanvas; map_fn:coord_convert); override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
    function    closest_vertex_square_distance_2D(p:Point3D) : double; override;
    function    closest_vertex(p:Point3D) : Point3D; override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Text Definition
///////////////////////////////////////////////////////////////////////////////
type
  Text_ = class(Point_)
    h  : double;
    s  : string;
    constructor create(p:Point3D; ss:string; height:double; col:integer);
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Attrib Definition
///////////////////////////////////////////////////////////////////////////////
type
  Attrib_ = class(Text_)
    tagstr : string;
    constructor create(p:Point3D; ss,tag:string; height:double; col:integer);
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
  end;
type
  patt_array = ^att_array;
  att_array  = array[0..0] of Attrib_;
///////////////////////////////////////////////////////////////////////////////
// Insert Definition (optionally contains attribs)
///////////////////////////////////////////////////////////////////////////////
type
  Insert_ = class(Point_)
    num_attribs : integer;
    attribs     : array[0..3] of Attrib_;
    constructor create(p:Point3D; col:integer; numatts:integer; atts:patt_array);
    destructor  destroy; override;
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Line Definition
///////////////////////////////////////////////////////////////////////////////
type
  Line_ = class(Point_)
    p2 : Point3D;
    constructor create(p_1,p_2:Point3D; col:integer);
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    procedure   DrawVertices(acanvas:TCanvas; map_fn:coord_convert); override;
    function    count_points   : integer; override;
    function    count_lines    : integer; override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
    function    closest_vertex_square_distance_2D(p:Point3D) : double; override;
    function    closest_vertex(p:Point3D) : Point3D; override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Circle Definition
///////////////////////////////////////////////////////////////////////////////
type
  Circle_ = class(Point_)
    radius : double;
    constructor create(p_1:Point3D; radius_:double; col:integer);
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Arc Definition
///////////////////////////////////////////////////////////////////////////////
type
  Arc_ = class(Circle_)
    angle1,angle2 : double;
    constructor create(p_1:Point3D; radius_,sa,ea:double; col:integer);
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Polyline Definition
///////////////////////////////////////////////////////////////////////////////
type
  Polyline_ = class(DXF_Entity)
    closed      : boolean;
    numvertices : integer;
    polypoints  : ppointlist;
    constructor create(numpoints:integer; points:ppointlist; col:integer; closed_:boolean);
    destructor  destroy; override;
    procedure   Draw(acanvas:TCanvas; map_fn:coord_convert); override;
    procedure   DrawVertices(acanvas:TCanvas; map_fn:coord_convert); override;
    function    count_points   : integer; override;
    function    count_lines    : integer; override;
    function    count_polys_open   : integer; override;
    function    count_polys_closed : integer; override;
    function    details : string; override;
    procedure   write_to_DXF(var IO:textfile; layer:string);     override;
    procedure   max_min_extents(var emax,emin:Point3D);      override;
    function    closest_vertex_square_distance_2D(p:Point3D) : double; override;
    function    closest_vertex(p:Point3D) : Point3D; override;
  end;
///////////////////////////////////////////////////////////////////////////////
// Entity_List class definition
// An entity list is a collection of entities (in this case all the same type)
// I wanted to keep polylines & lines etc in seperate lists, so the DXF_Layer
// will automatically handle this.
///////////////////////////////////////////////////////////////////////////////
type
  DXF_Layer  = class;

  Entity_List = class
  private
    function    add_at_end(entity:DXF_Entity) : boolean;
    function    insert(entity:DXF_Entity) : boolean;
  public
    list_name      : string;
    parent_layer   : DXF_Layer;
    Kludge_layer   : DXF_Layer; // see selection.save...
    entities       : TList;
    sorted         : boolean;
    constructor create(l_name:string);
    destructor  destroy; override;
    property    name : string read list_name write list_name;
    function    add_entity_to_list(entity:DXF_Entity) : boolean;
    procedure   draw_primitives(acanvas:TCanvas; map_fn:coord_convert);
    procedure   draw_vertices(acanvas:TCanvas; map_fn:coord_convert);
    function    num_entities : integer;
    function    count_points   : integer;
    function    count_lines    : integer;
    function    count_polys_open   : integer;
    function    count_polys_closed : integer;
    procedure   max_min_extents(var emax,emin:Point3D);
    function    closest_vertex_square_distance_2D(p:Point3D; var cl:DXF_Entity) : double;
end;
///////////////////////////////////////////////////////////////////////////////
// DXF_layer class definition
// A collection of entity lists. One for each type.
///////////////////////////////////////////////////////////////////////////////
  DXF_Layer  = class
    layer_name   : string;
    layer_colinx : integer;
    entity_names : TStringList;
    entity_lists : TList;
    constructor create(l_name:string);
    destructor  destroy; override;
    procedure   delete(aname:string; releasemem:boolean);
    property    Colour : integer read layer_colinx write layer_colinx;
    property    name : string read layer_name write layer_name;
    function    add_entity_to_layer(entity:DXF_Entity) : boolean;
    // Add a pre filled list (save selected to file... see selected lists)
    procedure   add_entity_list(elist:Entity_List);
    // utilities
    function    num_lists : integer;
    procedure   max_min_extents(var emax,emin:Point3D);
    function    create_or_find_list_type(aname:string) : Entity_List;
  end;
///////////////////////////////////////////////////////////////////////////////
// DXF_Object class definition
// A Collection of DXF_Layers - eg a whole DXF file.
///////////////////////////////////////////////////////////////////////////////
type
  DXF_Object  = class
    DXF_name     : string;
    layer_lists  : TList;
    emax         : Point3D;
    emin         : Point3D;
    AspectRatio  : double;
    // Create an empty object
    constructor create(aname:string);
    // Create an object and load from file
    constructor create_from_file(aname:string; skipped:Tstrings);
    destructor  destroy; override;
    procedure   save_to_file(aname:string);
    property    name : string read DXF_name write DXF_name;
    function    num_layers : integer;
    // add an empty layer
    function    new_layer(aname:string; DUPs_OK:boolean) : DXF_Layer;
    // add a pre-filled layer
    function    add_layer(layer:DXF_Layer) : boolean;
    // return the layer with a given name
    function    layer(aname:string) : DXF_Layer;
    // add an entity to a named layer
    function    add_entity_to_layer(entity:DXF_Entity; aname:string) : boolean;
    // Useful ones
    procedure   remove_empty_layers_and_lists;
    procedure   copy_to_strings(ts:TStrings);
    function    get_min_extent         : Point3D;
    function    get_max_extent         : Point3D;
    // update the extents (not really needed)
    procedure   max_min_extents(var emax,emin:Point3D);
  end;
///////////////////////////////////////////////////////////////////////////////
// Selection_lists class definition
// A collection of entity lists. Used by mouse selection routines
///////////////////////////////////////////////////////////////////////////////
type
  selection_lists = class
    entity_lists : TList;
    constructor create;
    destructor  destroy; override;
    procedure   save_to_DXF_file(aname:string);
    function    find_closest_2D_point(p:Point3D; var ent:DXF_Entity) : Point3D;
  end;
///////////////////////////////////////////////////////////////////////////////
// DXF exceptions will be this type
///////////////////////////////////////////////////////////////////////////////
type
  DXF_exception = class(Exception);
///////////////////////////////////////////////////////////////////////////////
// Default AutoCad layer colours (1..7) - (8..user defined)
///////////////////////////////////////////////////////////////////////////////
const
  BYLAYER = 256;
const
  def_cols = 12;
  DXF_Layer_Colours : array[0..def_cols] of TColor = (clBlack, // zero - not used
    clRed,    clYellow, clLime,   clAqua,   clBlue,   clPurple, {clWhite}clBlack,
    clOlive,  clFuchsia,clTeal,   clGray,   clDkGray);
///////////////////////////////////////////////////////////////////////////////
// Memory check variables
///////////////////////////////////////////////////////////////////////////////
var
  entities_in_existence  : integer;
  Ent_lists_in_existence : integer;
  layers_in_existence    : integer;
  DXF_Obj_in_existence   : integer;

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// implementation
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
implementation

uses
  DXF_read, DXF_write, DXF_Utils;

const
  EOL = #13#10;

function outstr(f:double) : string;
begin
  result := FloatToStrF(f,ffFixed,7,3);
  //result := FloatToStr(f);
end;

procedure draw_cross(acanvas:TCanvas; p1:TPoint);
var pa,pb : TPoint;
begin
  pa.x := p1.x-2; pa.y := p1.y-2;
  pb.x := p1.x+3; pb.y := p1.y+3;
  acanvas.Moveto(pa.x,pa.y);
  acanvas.Lineto(pb.x,pb.y);
  pa.x := p1.x-2; pa.y := p1.y+2;
  pb.x := p1.x+3; pb.y := p1.y-3;
  acanvas.Moveto(pa.x,pa.y);
  acanvas.Lineto(pb.x,pb.y);
end;
///////////////////////////////////////////////////////////////////////////////
// DXF_Entity - abstract base class - override where neccessary
///////////////////////////////////////////////////////////////////////////////
constructor DXF_Entity.create;
begin
  inc(entities_in_existence);
end;

destructor DXF_Entity.destroy;
begin
  dec(entities_in_existence);
  inherited destroy;
end;

procedure DXF_Entity.setcolour(col:integer);
begin
  colinx := col;
  colour := DXF_Layer_Colours[col mod (def_cols+1)];
end;

function DXF_Entity.count_points   : integer;
begin result := 1; end;

function DXF_Entity.count_lines    : integer;
begin result := 0; end;

function DXF_Entity.count_polys_open    : integer;
begin result := 0; end;

function DXF_Entity.count_polys_closed  : integer;
begin result := 0; end;

function DXF_Entity.proper_name : string;
var temp : string;
begin
  temp := UpperCase(ClassName);
  result := Copy(temp,1,Length(temp)-1);
end;

procedure DXF_Entity.write_to_DXF(var IO:textfile; layer:string);
begin
  writeln(IO,0 ,EOL,proper_name);
  writeln(IO,8 ,EOL,layer);
  writeln(IO,62,EOL,colinx);
end;

///////////////////////////////////////////////////////////////////////////////
// Point
///////////////////////////////////////////////////////////////////////////////
constructor Point_.create(p:Point3D; col:integer);
begin
  inherited create;
  p1 := p;
  setcolour(col);
end;

procedure Point_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var po : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  po := map_fn(p1.x,p1.y);
  draw_cross(acanvas,po);
end;

procedure Point_.DrawVertices(acanvas:TCanvas; map_fn:coord_convert);
var po : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  po := map_fn(p1.x,p1.y);
  draw_cross(acanvas,po);
end;

function Point_.details : string;
begin
  result := ClassName + EOL + Point3DToStr(p1);
end;

procedure Point_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,10,EOL,outstr(p1.x));
  writeln(IO,20,EOL,outstr(p1.y));
  writeln(IO,30,EOL,outstr(p1.z));
end;

procedure Point_.max_min_extents(var emax,emin:Point3D);
begin
  max_bound(emax,p1); min_bound(emin,p1);
end;

function Point_.closest_vertex_square_distance_2D(p:Point3D) : double;
begin
  result := sq_dist2D(p1,p);
end;

function Point_.closest_vertex(p:Point3D) : Point3D;
begin
  result := p1;
end;
///////////////////////////////////////////////////////////////////////////////
// Text
///////////////////////////////////////////////////////////////////////////////
constructor Text_.create(p:Point3D; ss:string; height:double; col:integer);
begin
  inherited create(p,col);
  h      := height;
  if ss<>'' then s := ss;
//  else s := '('+FloatToStr(p1.x)+','+FloatToStr(p1.y)+')';
end;

procedure Text_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var pa,dummy1,dummy2 : TPoint;
    Fheight          : integer;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  // kludgy method for scaling text heights
  dummy1  := map_fn(0,0);
  dummy2  := map_fn(0,h);
  Fheight := 2+(dummy1.y-dummy2.y);
  if FHeight=2 then exit;
  with acanvas.Font do begin
    if Height<>Fheight then Height := Fheight;
    if color<>colour then color := colour;
  end;
  pa := map_fn(p1.x,p1.y);
  acanvas.TextOut(pa.x,pa.y,s);
end;

function Text_.details : string;
begin
  result := inherited details + EOL +
            s + EOL + 'Height = ' + outstr(h);
end;

procedure Text_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,40,EOL,outstr(h));
  writeln(IO,1 ,EOL,s);
end;

procedure Text_.max_min_extents(var emax,emin:Point3D);
begin
  max_bound(emax,p1); min_bound(emin,p1);
end;
///////////////////////////////////////////////////////////////////////////////
// Attrib
///////////////////////////////////////////////////////////////////////////////
constructor Attrib_.create(p:Point3D; ss,tag:string; height:double; col:integer);
begin
  inherited create(p,ss,height,col);
  tagstr := tag;
end;

function Attrib_.details : string;
begin
  result := inherited details + EOL + TagStr;
end;

procedure Attrib_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,2 ,EOL,tagstr);
end;
///////////////////////////////////////////////////////////////////////////////
// Insert
///////////////////////////////////////////////////////////////////////////////
constructor Insert_.create(p:Point3D; col:integer; numatts:integer; atts:patt_array);
var lp1 : integer;
begin
  inherited create(p,col);
  num_attribs := numatts;
  if num_attribs>4 then raise Exception.Create('This version only handles 3 ATTRIBs');
  for lp1:=0 to num_attribs-1 do attribs[lp1] := atts^[lp1];
end;

destructor Insert_.destroy;
var lp1 : integer;
begin
  for lp1:=0 to num_attribs-1 do attribs[lp1].Free;
  inherited destroy;
end;

procedure Insert_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var lp1 : integer;
begin
  inherited;
  for lp1:=0 to num_attribs-1 do attribs[lp1].Draw(acanvas, map_fn);
end;

function Insert_.details : string;
var lp1 : integer;
begin
  result := inherited details + EOL + 'Attribs : ' + IntToStr(num_attribs) + EOL;
  for lp1:=0 to num_attribs-1 do
    result := result + 'Attrib ' + IntToStr(lp1+1) + EOL + attribs[lp1].details + EOL;
end;

procedure Insert_.write_to_DXF(var IO:textfile; layer:string);
var lp1 : integer;
begin
  inherited;
  if num_attribs>0 then begin
    writeln(IO,66,EOL,1);
    for lp1:=0 to num_attribs-1 do attribs[lp1].write_to_DXF(IO,layer);
    writeln(IO,0,EOL,'SEQEND');
  end
  else writeln(IO,66,EOL,0);
end;

procedure Insert_.max_min_extents(var emax,emin:Point3D);
begin
  inherited;
end;
///////////////////////////////////////////////////////////////////////////////
// Line
///////////////////////////////////////////////////////////////////////////////
constructor Line_.create(p_1,p_2:Point3D; col:integer);
begin
  inherited create(p_1,col);
  p2 := p_2;
end;

procedure Line_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var pa,pb : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  pa := map_fn(p1.x,p1.y);
  pb := map_fn(p2.x,p2.y);
  acanvas.Moveto(pa.x,pa.y);
  acanvas.Lineto(pb.x,pb.y);
end;

procedure Line_.DrawVertices(acanvas:TCanvas; map_fn:coord_convert); 
var po : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  po := map_fn(p1.x,p1.y);
  draw_cross(acanvas,po);
  po := map_fn(p2.x,p2.y);
  draw_cross(acanvas,po);
end;

function Line_.count_points : integer;
begin result := 2; end;

function Line_.count_lines : integer;
begin result := 1; end;

function Line_.details : string;
begin
  result := inherited details + EOL + Point3DToStr(p2);
end;

procedure Line_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,11,EOL,outstr(p2.x));
  writeln(IO,21,EOL,outstr(p2.y));
  writeln(IO,31,EOL,outstr(p2.z));
end;

procedure Line_.max_min_extents(var emax,emin:Point3D);
begin
  max_bound(emax,p1); min_bound(emin,p1);
  max_bound(emax,p2); min_bound(emin,p2);
end;

function Line_.closest_vertex_square_distance_2D(p:Point3D) : double;
begin
  result := dmin(sq_dist2D(p1,p),sq_dist2D(p2,p));
end;

function Line_.closest_vertex(p:Point3D) : Point3D;
begin
  if sq_dist2D(p1,p)<sq_dist2D(p2,p) then result := p1 else result := p2;
end;
///////////////////////////////////////////////////////////////////////////////
// Circle
///////////////////////////////////////////////////////////////////////////////
constructor Circle_.create(p_1:Point3D; radius_:double; col:integer);
begin
  inherited create(p_1,col);
  radius := radius_;
end;

procedure Circle_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var pa,pb : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  pa := map_fn(p1.x-radius,p1.y-radius);
  pb := map_fn(p1.x+radius,p1.y+radius);
  // bug in Ellipse routine causes crash if extents are too small
  if (pb.x>pa.x+1) and (pa.y>pb.y+1) then
    acanvas.Ellipse(pa.x,pa.y,pb.x,pb.y)
  else acanvas.pixels[pa.x,pa.y] := acanvas.Pen.Color;
end;

function Circle_.details : string;
begin
  result := inherited details + EOL +
            'Radius = ' + outstr(radius);
end;

procedure Circle_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,40,EOL,outstr(radius));
end;

procedure Circle_.max_min_extents(var emax,emin:Point3D);
var lp1 : integer;
begin
  max_bound(emax, p1_plus_p2 (p1, aPoint3D(radius,radius,0)));
  min_bound(emin, p1_minus_p2(p1, aPoint3D(radius,radius,0)));
end;
///////////////////////////////////////////////////////////////////////////////
// Arc
///////////////////////////////////////////////////////////////////////////////
constructor Arc_.create(p_1:Point3D; radius_,sa,ea:double; col:integer);
begin
  inherited create(p_1,radius_,col);
  angle1 := DegToRad(sa);
  angle2 := DegToRad(ea);
end;

procedure Arc_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var pu,pv,pw,px : TPoint;
var po,pa,pb : TPoint;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  pu := map_fn(p1.x-radius,p1.y-radius);
  pv := map_fn(p1.x+radius,p1.y+radius);
  pw := map_fn(p1.x + cos(angle1)*radius,p1.y + sin(angle1)*radius);
  px := map_fn(p1.x + cos(angle2)*radius,p1.y + sin(angle2)*radius);
  if (pv.x>pu.x+1) and (pu.y>pv.y+1) then
    acanvas.Arc(pu.x,pu.y,pv.x,pv.y,pw.x,pw.y,px.x,px.y)
  else
  acanvas.pixels[pu.x,pu.y] := acanvas.Pen.Color;
end;

function Arc_.details : string;
begin
  result := inherited details + EOL +
            'Angle 1 = ' + outstr(angle1) + EOL +
            'Angle 2 = ' + outstr(angle2);
end;

procedure Arc_.write_to_DXF(var IO:textfile; layer:string);
begin
  inherited;
  writeln(IO,50,EOL,outstr(RadToDeg(angle1)));
  writeln(IO,51,EOL,outstr(RadToDeg(angle2)));
end;

procedure Arc_.max_min_extents(var emax,emin:Point3D);
var lp1          : integer;
    ax,ay,bx,by  : double;
    thisboundary : integer;
    lastboundary : integer;
begin
  // the end points of the arc
  ax := p1.x + radius*cos(angle1);
  ay := p1.y + radius*sin(angle1);
  bx := p1.x + radius*cos(angle2);
  by := p1.y + radius*sin(angle2);
  max_bound(emax, aPoint3D(ax,ay,0));
  min_bound(emin, aPoint3D(ax,ay,0));
  max_bound(emax, aPoint3D(bx,by,0));
  min_bound(emin, aPoint3D(bx,by,0));
  // long arcs may extend along the axes (quadrants) (eg 1 to 359 ->90,180,270)
  lastboundary := 90*((trunc(RadToDeg(angle2))+89) div 90);
  if lastboundary=360 then lastboundary := 0;
  thisboundary := 90*((trunc(RadToDeg(angle1))+90) div 90);
  if thisboundary=360 then thisboundary := 0;
  while thisboundary<>lastboundary do begin
    ax := p1.x + radius*cos(DegToRad(thisboundary));
    ay := p1.y + radius*sin(DegToRad(thisboundary));
    max_bound(emax, aPoint3D(ax,ay,0));
    min_bound(emin, aPoint3D(ax,ay,0));
    thisboundary := thisboundary+90;
    if thisboundary=360 then thisboundary := 0;
  end;
end;
///////////////////////////////////////////////////////////////////////////////
// Polyline
///////////////////////////////////////////////////////////////////////////////
constructor Polyline_.create(numpoints:integer; points:ppointlist; col:integer; closed_:boolean);
var lp1 : integer;
begin
  inherited create;
  numvertices := numpoints;
  if closed_ then closed := true
  else if p1_eq_p2_3D(points[0],points[numvertices-1]) then begin
    closed := true;
    dec(numvertices);
  end
  else closed := false;
  polypoints := allocate_points(numvertices);
  for lp1:=0 to numvertices-1 do polypoints^[lp1] := points^[lp1];
  setcolour(col);
end;

destructor Polyline_.destroy;
begin
  deallocate_points(polypoints,numvertices);
  inherited destroy;
end;

procedure Polyline_.Draw(acanvas:TCanvas; map_fn:coord_convert);
var PointArray : array[0..max_vertices_per_polyline-1] of TPoint;
    lp1,tn     : integer;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  for lp1:=0 to numvertices-1 do
    PointArray[lp1] := map_fn(polypoints^[lp1].x,polypoints^[lp1].y);
  if not closed then acanvas.Polyline(Slice(PointArray,numvertices))
  else acanvas.Polygon(Slice(PointArray,numvertices));
end;

procedure Polyline_.DrawVertices(acanvas:TCanvas; map_fn:coord_convert);
var po  : TPoint;
    lp1 : integer;
begin
  with acanvas.Pen do if Color<>colour then Color:=colour;
  for lp1:=0 to numvertices-1 do begin
    po := map_fn(polypoints^[lp1].x,polypoints^[lp1].y);
    draw_cross(acanvas,po);
  end;
end;

function Polyline_.count_points   : integer;
begin result := numvertices; end;

function Polyline_.count_lines : integer;
begin result := numvertices; end;

function Polyline_.count_polys_open : integer;
begin if not closed then result := 1 else result := 0;end;

function Polyline_.count_polys_closed : integer;
begin if closed then result := 1 else result := 0;end;

function Polyline_.details : string;
var lp1 : integer;
begin
  result := ClassName;
  for lp1:=0 to numvertices-1 do result := result + EOL + Point3DToStr(polypoints^[lp1]);
end;

procedure Polyline_.write_to_DXF(var IO:textfile; layer:string);
var lp1 : integer;
begin
  inherited;
  if closed then writeln(IO,70 ,EOL,1+8) // 1+8 = closed+3D
  else writeln(IO,70 ,EOL,8);
  for lp1:=0 to numvertices-1 do begin
    writeln(IO,0 ,EOL,'VERTEX');
    writeln(IO,10,EOL,outstr(polypoints^[lp1].x));
    writeln(IO,20,EOL,outstr(polypoints^[lp1].y));
    writeln(IO,30,EOL,outstr(polypoints^[lp1].z));
  end;
  writeln(IO,0 ,EOL,'SEQEND');
end;

procedure Polyline_.max_min_extents(var emax,emin:Point3D);
var lp1 : integer;
begin
  for lp1:=0 to numvertices-1 do begin
    max_bound(emax,polypoints^[lp1]); min_bound(emin,polypoints^[lp1]);
  end;
end;

function Polyline_.closest_vertex_square_distance_2D(p:Point3D) : double;
var lp1 : integer;
begin
  result := 1E10;
  for lp1:=0 to numvertices-1 do
    result := dmin(result,sq_dist2D(polypoints^[lp1],p));
end;

function Polyline_.closest_vertex(p:Point3D) : Point3D;
var lp1,c : integer;
    d1,d2 : double;
begin
  d1 := 1E10;
  for lp1:=0 to numvertices-1 do begin
    d2 := sq_dist2D(polypoints^[lp1],p);
    if d2<d1 then begin
      result := polypoints^[lp1];
      d1 := d2;
    end;
  end;
end;

///////////////////////////////////////////////////////////////////////////////
// Entity_List class implementation
///////////////////////////////////////////////////////////////////////////////
constructor Entity_List.create(l_name:string);
begin
  list_name      := l_name;
  entities       := TList.Create;
  inc(Ent_lists_in_existence);
end;

destructor Entity_List.destroy;
var lp1 : integer;
begin
  for lp1:=0 to (entities.Count-1) do DXF_Entity(entities[lp1]).Free;
  entities.Free;
  dec(Ent_lists_in_existence);
  inherited destroy;
end;

function Entity_List.add_entity_to_list(entity:DXF_Entity) : boolean;
begin
  if sorted then result := insert(entity)
  else           result := add_at_end(entity);
end;

function Entity_List.add_at_end(entity:DXF_Entity) : boolean;
begin
  entities.Add(entity);
end;

function Entity_List.insert(entity:DXF_Entity) : boolean;
begin
  entities.Add(entity);
end;

procedure Entity_List.draw_primitives(acanvas:TCanvas; map_fn:coord_convert);
var lp1 : integer;
begin
  for lp1:=0 to (entities.Count-1) do
    DXF_Entity(entities[lp1]).Draw(acanvas, map_fn);
end;

procedure Entity_List.draw_vertices(acanvas:TCanvas; map_fn:coord_convert);
var lp1 : integer;
begin
  for lp1:=0 to (entities.Count-1) do
    DXF_Entity(entities[lp1]).DrawVertices(acanvas, map_fn);
end;

function Entity_List.num_entities : integer;
begin
  result := entities.Count;
end;

function Entity_List.count_points : integer;
var lp1 : integer;
begin
  result := 0;
  for lp1:=0 to (entities.Count-1) do
    result := result + DXF_Entity(entities[lp1]).count_points;
end;

function Entity_List.count_lines : integer;
var lp1 : integer;
begin
  result := 0;
  for lp1:=0 to (entities.Count-1) do
    result := result + DXF_Entity(entities[lp1]).count_lines;
end;

function Entity_List.count_polys_open : integer;
var lp1 : integer;
begin
  result := 0;
  for lp1:=0 to (entities.Count-1) do
    result := result + DXF_Entity(entities[lp1]).count_polys_open;
end;

function Entity_List.count_polys_closed : integer;
var lp1 : integer;
begin
  result := 0;
  for lp1:=0 to (entities.Count-1) do
    result := result + DXF_Entity(entities[lp1]).count_polys_closed;
end;

procedure Entity_List.max_min_extents(var emax,emin:Point3D);
var lp1 : integer;
begin
  for lp1:=0 to (entities.Count-1) do
    DXF_Entity(entities[lp1]).max_min_extents(emax,emin);
end;

function Entity_List.closest_vertex_square_distance_2D(p:Point3D; var cl:DXF_Entity) : double;
var lp1 : integer;
    cl_ : DXF_Entity;
    t   : double;
begin
  result := 1E10;
  for lp1:=0 to (entities.Count-1) do begin
    cl_ := DXF_Entity(entities[lp1]);
    t   := cl_.closest_vertex_square_distance_2D(p);
    if t<result then begin
      cl := cl_;
      result := t;
    end;
  end;
end;
///////////////////////////////////////////////////////////////////////////////
// DXF_layer class implementation
///////////////////////////////////////////////////////////////////////////////
constructor DXF_Layer.create(l_name:string);
begin
  layer_name   := l_name;
  entity_names := TStringList.Create;
  entity_lists := TList.Create;
  inc(layers_in_existence);
end;

destructor DXF_Layer.destroy;
var lp1 : integer;
    el : Entity_List;
begin
  if num_lists>0 then for lp1:=num_lists-1 downto 0 do begin
    el := Entity_List(entity_lists[lp1]);
    el.Free;
  end;
  entity_names.Free;
  entity_lists.Free;
  dec(layers_in_existence);
  inherited destroy;
end;

procedure DXF_Layer.delete(aname:string; releasemem:boolean);
var lp1 : integer;
    el  : Entity_List;
begin
  for lp1:=num_lists-1 downto 0 do begin
    el := Entity_List(entity_lists[lp1]);
    if el.name=aname then begin
      entity_lists.remove(el);
      if releasemem then el.Free;
      entity_names.delete(lp1);
    end;
  end;
end;

function DXF_Layer.add_entity_to_layer(entity:DXF_Entity) : boolean;
var i  : integer;
    el : Entity_List;
begin
  i := entity_names.IndexOf(entity.ClassName);
  if i=-1 then begin
    el := Entity_List.create(entity.ClassName);
    el.parent_layer := self;
    i  := entity_lists.Add(el);
    if i<>entity_names.Add(entity.ClassName) then
      raise Exception.Create('Entity list ID mismatch');
    // This has never been raised yet, but might as well be sure.
  end;
  Entity_List(entity_lists[i]).add_entity_to_list(entity);
  if ((entity.Colour=0) or (entity.Colour=BYLAYER)) then
    entity.setColour(layer_colinx);
  result := true;
end;

procedure DXF_Layer.add_entity_list(elist:Entity_List);
var i : integer;
begin
  i := entity_names.IndexOf(elist.name);
  if i<>-1 then raise Exception.create('Attempted to add two lists with same name');
  elist.parent_layer := self;
  i  := entity_lists.Add(elist);
  if i<>entity_names.Add(elist.Name) then
    raise Exception.Create('Entity list ID mismatch');
end;

function DXF_Layer.num_lists : integer;
begin
  result := entity_names.Count;
end;

procedure DXF_Layer.max_min_extents(var emax,emin:Point3D);
var lp1 : integer;
begin
  for lp1:=0 to num_lists-1 do Entity_List(entity_lists[lp1]).max_min_extents(emax,emin);
end;

function DXF_Layer.create_or_find_list_type(aname:string) : Entity_List;
var inx : integer;
begin
  inx := entity_names.IndexOf(aname);
  if inx=-1 then begin
    result := Entity_List.create(aname);
    result.parent_layer := self;
    inx    := entity_lists.Add(result);
    if inx<>entity_names.Add(aname) then
      raise Exception.Create('Entity list ID mismatch');
  end
  else result := Entity_List(entity_lists[inx]);
end;
///////////////////////////////////////////////////////////////////////////////
// DXF_Object class implementation
///////////////////////////////////////////////////////////////////////////////
constructor DXF_Object.create(aname:string);
begin
  layer_lists := TList.create;
  if aname<>'' then DXF_name := aname
  else DXF_name := 'Untitled';
  emax        := aPoint3D(0,0,0);
  emin        := aPoint3D(0,0,0);
  AspectRatio := 1;
  inc(DXF_Obj_in_existence);
end;

constructor DXF_Object.create_from_file(aname:string; skipped:Tstrings);
var reader : DXF_Reader;
begin
  Reader:=DXF_Reader.Create(aname);
  Reader.set_skipped_list(skipped);
  With Reader do if (read_file) then begin
    name := ExtractFileName(aname);
    emax := get_max_extent;
    emin := get_min_extent;
    AspectRatio  := Aspect;
    layer_lists  := release_control_of_layers;
  end
  else begin
    layer_lists := TList.create;
    DXF_name    := aname;
    emax        := aPoint3D(0,0,0);
    emin        := aPoint3D(0,0,0);
    AspectRatio := 1;
  end;
  Reader.Free;
  inc(DXF_Obj_in_existence);
end;

destructor DXF_Object.destroy;
var lp1 : integer;
begin
  for lp1:=0 to layer_lists.Count-1 do
    DXF_Layer(layer_lists.Items[lp1]).Free;
  layer_lists.Free;
  dec(DXF_Obj_in_existence);
  inherited destroy;
end;

procedure DXF_Object.save_to_file(aname:string);
var Writer : DXF_Writer;
begin
  writer := DXF_writer.create(aname,layer_lists);
  writer.write_file;
  writer.free;
end;

function DXF_Object.num_layers : integer;
begin
  result := layer_lists.Count
end;

function DXF_Object.new_layer(aname:string; DUPs_OK:boolean) : DXF_Layer;
var lp1 : integer;
begin
  for lp1:=0 to layer_lists.Count-1 do begin
    if DXF_Layer(layer_lists[lp1]).name=aname then begin
      if not DUPs_OK then raise DXF_Exception.Create('Attempted to create layer with existing name');
      result := DXF_Layer(layer_lists[lp1]);
      exit;
    end;
  end;
  result := DXF_Layer.Create(aname);
  layer_lists.Add(result);
end;

function DXF_Object.add_layer(layer:DXF_Layer) : boolean;
var lp1  : integer;
begin
  for lp1:=0 to layer_lists.Count-1 do
    if DXF_Layer(layer_lists[lp1]).name=layer.name then
      raise DXF_Exception.Create('Attempted to add layer with existing name');
  layer_lists.Add(layer);
end;

function DXF_Object.layer(aname:string) : DXF_Layer;
var lp1 : integer;
begin
  result := nil;
  for lp1:=0 to layer_lists.Count-1 do
    if DXF_Layer(layer_lists[lp1]).name=aname then begin
      result := DXF_Layer(layer_lists[lp1]);
      exit;
    end;
end;

// Avoid using this if possible because we have to search for layer name every time
function DXF_Object.add_entity_to_layer(entity:DXF_Entity; aname:string) : boolean;
var lp1 : integer;
begin
  for lp1:=0 to layer_lists.Count-1 do
    if DXF_Layer(layer_lists[lp1]).name=aname then begin
      DXF_Layer(layer_lists[lp1]).add_entity_to_layer(entity);
      result := true;
      exit;
    end;
  raise DXF_Exception.Create('Attempted to add to unnamed layer');
end;

procedure DXF_Object.remove_empty_layers_and_lists;
var lp1,lp2 : integer;
    layer   : DXF_Layer;
    el      : Entity_List;
begin
  for lp1:=layer_lists.Count-1 downto 0 do begin
    layer := DXF_Layer(layer_lists[lp1]);
    for lp2:=layer.num_lists-1 downto 0 do begin
      el := Entity_List(layer.entity_lists[lp2]);
      if el.num_entities=0 then begin
        layer.entity_lists.remove(el);
        layer.entity_names.delete(lp2);
        el.Free;
        if layer.entity_lists.count<>layer.entity_names.count then
          showmessage('Internal error : Layer lists and names mismatch'); 
      end;
    end;
    if layer.num_lists=0 then begin
      layer_lists.remove(layer);
      layer.Free;
    end;
  end;
end;

procedure DXF_Object.copy_to_strings(ts:TStrings);
var lp1,lp2,pos : integer;
    layer       : DXF_Layer;
begin
  ts.Add(DXF_name);
  for lp1:=0 to layer_lists.count-1 do begin
    layer := layer_lists[lp1];
    pos := ts.Add('  '+layer.name);
    ts.Objects[pos] := layer;
    for lp2:=0 to layer.num_lists-1 do begin
      pos := ts.Add('    '+Entity_List(layer.entity_lists[lp2]).name);
      ts.Objects[pos] := layer.entity_lists[lp2];
    end;
  end;
end;

function DXF_Object.get_min_extent : Point3D;
begin
  result := emin;
end;

function DXF_Object.get_max_extent : Point3D;
begin
  result := emax;
end;

procedure DXF_Object.max_min_extents(var emax,emin:Point3D);
var lp1 : integer;
begin
  for lp1:=0 to layer_lists.Count-1 do
    DXF_Layer(layer_lists[lp1]).max_min_extents(emax,emin);
end;
///////////////////////////////////////////////////////////////////////////////
// Selection_lists class implementation
///////////////////////////////////////////////////////////////////////////////
constructor selection_lists.create;
begin
  entity_lists := TList.Create;;
end;

destructor selection_lists.destroy;
begin
  entity_lists.Free;
  inherited destroy;
end;

procedure selection_lists.save_to_DXF_file(aname:string);
var lp1,lp2 : integer;
    DXF     : DXF_Object;
    layer   : DXF_layer;
    el      : Entity_List;
begin
  DXF := DXF_Object.create('');
  for lp1:=0 to entity_lists.count-1 do begin
    el := Entity_List(entity_lists[lp1]);
    el.Kludge_layer := el.parent_layer; // we need to keep track of where they came from
    layer := DXF.new_layer(el.parent_layer.name,true);
    layer.add_entity_list(el);
  end;
  DXF.save_to_file(aname);
  // now get the lists back from the temporary DXF object (without it deleting them)
  for lp1:=DXF.layer_lists.count-1 downto 0 do begin
    layer := DXF_Layer(DXF.layer_lists[lp1]);
    for lp2:=layer.entity_lists.count-1 downto 0 do
      layer.delete(Entity_List(layer.entity_lists[lp2]).name,FALSE);
  end;
  DXF.Free;
  // reset the parent layer of the entity_lists
  for lp1:=0 to entity_lists.count-1 do begin
    el := Entity_List(entity_lists[lp1]);
    el.parent_layer := el.Kludge_layer; // we stored them temporarily
  end;
end;

function selection_lists.find_closest_2D_point(p:Point3D; var ent:DXF_Entity) : Point3D;
var lp1       : integer;
    dist,mind : double;
    entx      : DXF_Entity;
begin
  mind := 1E10;
  for lp1:=0 to entity_lists.count-1 do begin
    dist := Entity_List(entity_lists[lp1]).closest_vertex_square_distance_2D(p,entx);
    if dist<mind then begin
      result := entx.closest_vertex(p);
      ent    := entx;
      mind   := dist;
    end;
  end;
end;
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// initialization
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
initialization
  entities_in_existence  := 0;
  Ent_lists_in_existence := 0;
  layers_in_existence    := 0;
  DXF_Obj_in_existence   := 0;
end.
