library sysuser;
{
    This is a port of the SYSUSER example program included
    in the WSAPI developer's kit (as shipped with WebSite Pro).
    Port by Paul Gallagher, 1997 <paulpg@ozemail.com.au>
}
// TITLE:		SYSUSER.C
//
// FACILITY:	Authentication extension for WebSite web server
//
// ABSTRACT:	Maintains a new realm ("Windows NT Users").
//              The system realm should not contain any user or
//              group information. On subsequent thread attachments,
//              given that the specified URL registers as a URL
//              governed by the system realm, the code tries to
//              verify that the user and password given matches
//              an entry in the system security database. Return
//              TRUE on success, otherwise return passed status
//              from the server's normal authentication check.
//
// ENVIRONMENT:	WebSite 1.1 or later, WSAPI 0.15
//				Microsoft Windows Windows NT 3.51 or later only
//									(not compliant with Windows 95)
//				Developed under Microsoft Visual C++ Version 2.0
//
// AUTHOR:		Arsenio Santos   <arsenio@ora.com>
//
// Edit Log:
//
// When			Who		What
//----------	---		--------------------------------------------------
// 02-Mar-96	ars		Initial edit
// 26-Jan-97    paulg           Port to Delphi
//========================================================================

{ Important note about DLL memory management: ShareMem must be the
  first unit in your library's USES clause AND your project's (select
  View-Project Source) USES clause if your DLL exports any procedures or
  functions that pass strings as parameters or function results. This
  applies to all strings passed to and from your DLL--even those that
  are nested in records and classes. ShareMem is the interface unit to
  the DELPHIMM.DLL shared memory manager, which must be deployed along
  with your DLL. To avoid using DELPHIMM.DLL, pass string information
  using PChar or ShortString parameters. }

uses
  SysUtils,
  Classes,
  Windows,
  Registry,
  WSAPI in '..\..\lib\WSAPI.pas';

Const
  REALM_NAME   = 'Windows NT System';
  REALM_STRING = 'SOFTWARE\Denny\WebServer\CurrentVersion\Authentication\Basic\' + REALM_NAME;

Type
  PTokenUser = ^TTokenUser;
  TTokenUser = record
    user : TSIDAndAttributes;
  end;

var
  szDomain : string;

procedure Initialize_Module; forward;
function Determine_Domain_Controller : string; forward;

//========================================================================
//
//	Process() - Try to verify the username and password passed from
//  the client.  If they match against the system security database,
//  Process returns TRUE, indicating to the server that no further
//  authentication is necessary.  Otherwise, it returns FALSE and
//	cues the server into performing business-as-usual.
//========================================================================
function Process(tp : PTCTX) : integer; export; cdecl;
var
  bResult : boolean;
  hBogus : THandle;
begin
  //
  //  First, if the currently accessed realm doesn't match
  //  the realm maintained by this DLL, then this DLL should
  //  have nothing to do with said realm's authentication.
  //
  if (strcomp(tp^.auth_name, REALM_NAME)<>0) then begin
    Result:=AUTH_INTERNAL;
    exit;
  end;
  //
  //  One problem with this system of new realms and pseudo-
  //  auth-users is that someone can try to bypass authentication
  //  by using the target auth-user name.  This prevents that.
  //
  if (strcomp(tp^.user, 'SysUser')=0) then begin
    Result:=AUTH_REJECT;
    exit;
  end;
  //
  //  Try to verify the username and password against the
  //  system security.  Note that the code switches per OS.
  //
  bResult := LogonUser(tp^.user, PChar(szDomain),
                       tp^.pass, LOGON32_LOGON_INTERACTIVE,
                       LOGON32_PROVIDER_DEFAULT, hBogus);
  //
  //  Since we're certain this is the system realm, this
  //  conditional is fairly tight -- if the system verification
  //  succeeds, we set the server up to succeed; otherwise,
  //  send back a 401 message to the client.
  //
  if (bResult) then begin
    CloseHandle(hBogus);
    strpcopy(tp^.user_to_check, 'SysUser');
    Result:=AUTH_ACCEPT;
  end
  else
    Result:=AUTH_REJECT;
end;

//========================================================================
//
//	Initialize_Module() - Figure out which operating system we're running
//  on, and add the appropriate group key in the registry.
//========================================================================
procedure Initialize_Module;
var
  registry : TRegistry;
begin
  //
  szDomain:=Determine_Domain_Controller();
  //
  //  Now, open the registry key for realm information and either
  //  open an existing realm or create a new one.  If the open/create
  //  call fails, bomb out of the process.
  //
  try
    //  get registry object
    //
    registry := TRegistry.create;
    //
    //  attempt to verify/build expected keys/values
    //
    try
      registry.RootKey := HKEY_LOCAL_MACHINE;
      if not registry.KeyExists(REALM_STRING) then begin
        if registry.CreateKey(REALM_STRING) then begin
          //
          //  If this key is being created for the first time,
	  //  we need to set it up with one user in the User group;
	  //  this user is what WebSite will wind up using for its
	  //  in-house authentication.
	  //
          //  do Users sub-key
          if registry.OpenKey(REALM_STRING + '\Users',true) then begin
            registry.WriteString('SysUser','');
            registry.CloseKey;
          end;
          //
          //  do Groups subkey
          registry.CreateKey(REALM_STRING + '\Groups');
          if registry.OpenKey(REALM_STRING + '\Groups\Users',true) then begin
            registry.WriteString('SysUser','');
            registry.CloseKey;
          end;
        end;
      end;
    finally
      registry.free;
    end;
  except
    //  hide any registry errors
  end;
end;

//========================================================================
//
//	Determine_Domain_Controller() - Query the NT-native RPC
//  calls which specify the machine name of the current
//  primary domain controller.
//========================================================================
function Determine_Domain_Controller : string;
var
  szDomain, szAccountName : array[0..255] of char;
  dwInfoBufferSize, dwAccountSize, dwDomainSize : DWORD;
  hAccessToken : THandle;
  InfoBuffer : array[0..1000] of char;
  pInfoBuffer : PTokenUser;
  snu : SID_NAME_USE;
begin
  szAccountName := '';
  szDomain := '';
  dwAccountSize := sizeof(szAccountName);
  dwDomainSize := sizeof(szDomain);
  pInfoBuffer := PTokenUser(@InfoBuffer);
  try
    //
    //  This code looks at the current process' token info,
    //  and from it derives the name of the primary domain
    //  controller.
    //
    OpenProcessToken(GetCurrentProcess(), TOKEN_READ, @hAccessToken);
    GetTokenInformation(hAccessToken,
                      TokenUser,
                      @InfoBuffer,
                      sizeof(InfoBuffer), dwInfoBufferSize);
    LookupAccountSid(nil, pInfoBuffer^.User.Sid,
                     szAccountName, dwAccountSize,
                     szDomain, dwDomainSize, snu);
    Result := szDomain;
  except
    //  hide any errors getting domain
    Result := '';
  end;
end;

exports
  Process;

begin
  // module initialisation
  if bind_wsapi(MAJOR_VERSION, MINOR_VERSION, FALSE) then
    Initialize_Module()
  else ExitCode:=1; // cause DLL load to fail
end.
