unit TreeNT;

// TTreeNT  - An enhanced TreeView control with significant speed improvement
//            and almost all additional features implemented, introduced since IE 3.0
//            (COMCTL32.DLL version 4.70: checkboxes, custom draw,...).
// Version     - 1.0.1
// Last Change - 19. November 1997
//
// things left out:
//   - TreeView_GetLastVisible
//   - TVIS_EXPANDPARTIAL 
//   - TVE_COLLAPSERESET (to reset property ExpandedOnce)
//   - TVM_SETTOOLTIPS, TVM_GETTOOLTIPS
//     (for tooltip control handling, node hints are implemented nonetheless)
//   - TreeView_SetBorder, TreeView_GetBorder (no info available yet)
//   - TVS_NOEVENHEIGHT (to disable that an item's height must be even)
//   - insert mark color (insert marks are implemented nonetheless)
//   - TVM_SETUNICODEFORMAT, TVM_GETUNICODEFORMAT
//
// written by Dipl. Ing. Mike Lischke
//            Lischke@hotmail.com

{$R-}

interface

uses Windows, Classes, CommCtrl, Controls, DsgnIntf, ExtCtrls, Forms,
     Graphics, Messages, SysUtils;

type

  TCustomTreeNT = class;
  TTreeNTNodes  = class;

  TNodeState      = (nsChecked,nsCut,nsDisabled,nsDropHilited,nsExpanded,
                     nsExpandedOnce,nsFocused,nsGrayed,nsHot,nsIndeterminate,
                     nsMarked,nsSelected);
  TNodeStates     = set of TNodeState;

  TNodeAttachMode = (naAdd,naAddFirst,naAddChild,naAddChildFirst,naInsert);
  TAddMode        = (taAddFirst,taAdd,taInsert);

  PNodeInfo = ^TNodeInfo;
  TNodeInfo = packed record
                ImageIndex    : Integer;
                SelectedIndex : Integer;
                StateIndex    : Integer;
                OverlayIndex  : Integer;
                Checked       : Boolean;
                Data          : Pointer;
                Count         : Integer;
                Color         : TColor;
                ParentColor   : Boolean;
                ParentFont    : Boolean;
                FontData      : TFontData;
                FontColor     : TColor;
                Text          : String[255];
              end;

  TTreeNTNode = class(TPersistent)
  private
    FOwner         : TTreeNTNodes;
    FText          : String;
    FData          : Pointer;
    FItemId        : HTreeItem;
    FFont          : TFont;
    FParentColor,
    FParentFont    : Boolean;
    FImageIndex,
    FSelectedIndex,
    FOverlayIndex,
    FStateIndex    : Integer;
    FDeleting      : Boolean;
    FColor         : TColor;
    FItemized      : Boolean;
    FUpdateCount   : Integer;
    FProgramExpand,
    FProgramCollapse: SmallInt;

    procedure ExpandItem(Expand: Boolean; Recurse: Boolean);
    function GetAbsoluteIndex: Integer;
    function GetExpanded: Boolean;
    function GetExpandedOnce: Boolean;
    function GetLevel: Integer;
    function GetParent: TTreeNTNode;
    function GetChildren: Boolean;
    function GetCut: Boolean;
    function GetChecked: Boolean;
    function GetDropTarget: Boolean;
    function GetFocused: Boolean;
    function GetFont: TFont;
    function GetIndex: Integer;
    function GetItem(Index: Integer): TTreeNTNode;
    function GetSelected: Boolean;
    function GetState(NodeState: TNodeState): Boolean;
    function GetCount: Integer;
    function GetTreeNT: TCustomTreeNT;
    function HasVisibleParent: Boolean;
    procedure InternalMove(ParentNode,Node: TTreeNTNode; HItem: HTreeItem; AddMode: TAddMode);
    function IsEqual(Node: TTreeNTNode): Boolean;
    function IsNodeVisible: Boolean;
    procedure ReadData(Stream: TStream; Info: PNodeInfo);
    procedure SetChecked(Value: Boolean);
    procedure SetChildren(Value: Boolean);
    procedure SetColor(AValue: TColor);
    procedure SetCut(Value: Boolean);
    procedure SetData(Value: Pointer);
    procedure SetDropTarget(Value: Boolean);
    procedure SetFont(AFont: TFont);
    procedure SetItem(Index: Integer; Value: TTreeNTNode);
    procedure SetExpanded(Value: Boolean);
    procedure SetFocused(Value: Boolean);
    procedure SetImageIndex(Value: Integer);
    procedure SetOverlayIndex(Value: Integer);
    procedure SetParentColor(AValue: Boolean);
    procedure SetParentFont(AValue: Boolean);
    procedure SetSelectedIndex(Value: Integer);
    procedure SetSelected(Value: Boolean);
    procedure SetStateIndex(Value: Integer);
    procedure SetText(const S: String);
    procedure WriteData(Stream: TStream; Info: PNodeInfo);
  public
    constructor Create(AOwner: TTreeNTNodes);
    destructor Destroy; override;
    function AlphaSort: Boolean;
    procedure Assign(Source: TPersistent); override;
    procedure Collapse(Recurse: Boolean);
    function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    procedure Delete;
    procedure DeleteChildren;
    function DisplayRect(TextOnly: Boolean): TRect;
    function EditText: Boolean;
    procedure EndEdit(Cancel: Boolean);
    procedure Expand(Recurse: Boolean);
    function GetFirstChild: TTreeNTNode;
    function GetHandle: HWND;
    function GetLastChild: TTreeNTNode;
    function GetNext: TTreeNTNode;
    function GetNextChild(Value: TTreeNTNode): TTreeNTNode;
    function GetNextSibling: TTreeNTNode;
    function GetNextVisible: TTreeNTNode;
    function GetPrev: TTreeNTNode;
    function GetPrevChild(Value: TTreeNTNode): TTreeNTNode;
    function GetPrevSibling: TTreeNTNode;
    function GetPrevVisible: TTreeNTNode;
    function HasAsParent(Value: TTreeNTNode): Boolean;
    function IndexOf(Value: TTreeNTNode): Integer;
    function IsUpdating: Boolean;
    procedure MakeVisible;
    procedure MoveTo(Destination: TTreeNTNode; Mode: TNodeAttachMode);
    property AbsoluteIndex: Integer read GetAbsoluteIndex;
    property Count: Integer read GetCount;
    function CountEquals(N: integer): Boolean;
    procedure BeginUpdate;
    procedure EndUpdate;

    property Checked: Boolean read GetChecked write SetChecked;
    property Color: TColor read FColor write SetColor;
    property Cut: Boolean read GetCut write SetCut;
    property Data: Pointer read FData write SetData;
    property Deleting: Boolean read FDeleting;
    property DropTarget: Boolean read GetDropTarget write SetDropTarget;
    property Focused: Boolean read GetFocused write SetFocused;
    property Expanded: Boolean read GetExpanded write SetExpanded;
    property ExpandedOnce: Boolean read GetExpandedOnce;
    property Font: TFont read GetFont write SetFont;
    property Handle: HWND read GetHandle;
    property HasChildren: Boolean read GetChildren write SetChildren;
    property ImageIndex: Integer read FImageIndex write SetImageIndex;
    property Index: Integer read GetIndex;
    property IsVisible: Boolean read IsNodeVisible;
    property Item[Index: Integer]: TTreeNTNode read GetItem write SetItem; default;
    property ItemId: HTreeItem read FItemId;
    property Level: Integer read GetLevel;
    property OverlayIndex: Integer read FOverlayIndex write SetOverlayIndex;
    property Owner: TTreeNTNodes read FOwner;
    property Parent: TTreeNTNode read GetParent;
    property ParentColor: Boolean read FParentColor write SetParentColor default True;
    property ParentFont: Boolean read FParentFont write SetParentFont default True;
    property Selected: Boolean read GetSelected write SetSelected;
    property SelectedIndex: Integer read FSelectedIndex write SetSelectedIndex;
    property StateIndex: Integer read FStateIndex write SetStateIndex;
    property Text: String read FText write SetText;
    property TreeView: TCustomTreeNT read GetTreeNT;
  end;

  TTreeNTNodes = class(TPersistent)
  private
    FOwner : TCustomTreeNT;
    FUpdateCount : Integer;
    FDeleting: Boolean;
    procedure AddedNode(ParentNode: TTreeNTNode);
    function GetHandle: HWND;
    procedure ReadData(Stream: TStream);
    procedure Repaint(Node: TTreeNTNode);
    procedure WriteData(Stream: TStream);
  protected
    function AddItem(Parent, Target: HTreeItem; const Item: TTVItem; AddMode: TAddMode): HTreeItem;
    function InternalAddObject(Node: TTreeNTNode; const S: String; Ptr: Pointer; AddMode: TAddMode): TTreeNTNode;
    procedure DefineProperties(Filer: TFiler); override;
    function CreateItem(Node: TTreeNTNode): TTVItem;
    function GetCount: Integer;
    procedure SetItem(Index: Integer; Value: TTreeNTNode);
    procedure SetUpdateState(Updating: Boolean);
  public
    constructor Create(AOwner: TCustomTreeNT);
    destructor Destroy; override;
    function AddChildFirst(Node: TTreeNTNode; const S: String): TTreeNTNode;
    function AddChild(Node: TTreeNTNode; const S: String): TTreeNTNode;
    function AddChildObjectFirst(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;
    function AddChildObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;
    function AddFirst(Node: TTreeNTNode; const S: String): TTreeNTNode;
    function Add(Node: TTreeNTNode; const S: String): TTreeNTNode;
    function AddObjectFirst(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;
    function AddObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;
    procedure Assign(Source: TPersistent); override;
    procedure BeginUpdate;
    procedure Clear;
    procedure Delete(Node: TTreeNTNode);
    procedure EndUpdate;
    function GetFirstNode: TTreeNTNode;
    function GetNode(ItemId: HTreeItem): TTreeNTNode;
    function GetNodeFromIndex(Index: Integer): TTreeNTNode;
    function Insert(Node: TTreeNTNode; const S: String): TTreeNTNode;
    function InsertObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;
    function IsDeleting: Boolean;
    function IsUpdating: Boolean;
    property Count: Integer read GetCount;
    property Handle: HWND read GetHandle;
    property Item[Index: Integer]: TTreeNTNode read GetNodeFromIndex; default;
    property Owner: TCustomTreeNT read FOwner;
  end;


  ETreeNTError = class(Exception);

  THitTest     = (htAbove, htBelow, htNowhere, htOnItem, htOnButton, htOnIcon,
                  htOnIndent, htOnLabel, htOnRight, htOnStateIcon, htToLeft, htToRight);
  THitTests    = set of THitTest;

  TTVChangingEvent   = procedure(Sender: TObject; Node: TTreeNTNode; var AllowChange: Boolean) of object;
  TTVChangedEvent    = procedure(Sender: TObject; Node: TTreeNTNode) of object;
  TTVDesignClickEvent = procedure(Sender: TObject; Node: TTreeNTNode; var AllowSelect: Boolean) of object;
  TTVEditingEvent    = procedure(Sender: TObject; Node: TTreeNTNode; var AllowEdit: Boolean) of object;
  TTVEditedEvent     = procedure(Sender: TObject; Node: TTreeNTNode; var S: String) of object;
  TTVExpandingEvent  = procedure(Sender: TObject; Node: TTreeNTNode; var AllowExpansion: Boolean) of object;
  TTVCollapsingEvent = procedure(Sender: TObject; Node: TTreeNTNode; var AllowCollapse: Boolean) of object;
  TTVExpandedEvent   = procedure(Sender: TObject; Node: TTreeNTNode) of object;
  TTVHintEvent       = function(Sender: TObject; Node: TTreeNTNode): String of object;
  TTVCompareEvent    = procedure(Sender: TObject; Node1,Node2: TTreeNTNode; Data: Integer; var Compare: Integer) of object;
  TTVPaintEvent      = procedure(Sender: TObject; Canvas: TCanvas) of object;
  TTVItemPaintEvent  = procedure(Sender: TObject; Node: TTreeNTNode; Canvas: TCanvas; ItemRect: TRect; NodeStates: TNodeStates) of object;
  TTVSingleExpandingEvent = procedure(Sender: TObject; Node: TTreeNTNode; var AutoCollapse: Boolean) of object;

  TSortType = (stNone, stData, stText, stBoth);
  TScrollDirection = set of (sdLeft,sdUp,sdRight,sdDown);
  TTreeOptions = set of (toCheckBoxes,toFullRowSelect,toHideSelection,toHotTrack,
                         toInfoTip,toNoScroll,toReadOnly,toToolTips,toShowButtons,
                         toShowLines,toShowRoot,toSingleExpand,toWantReturn);

  TCustomTreeNT = class(TWinControl)
  private
    FAutoScroll: Boolean;
    FDesignerMode: Boolean;
    FBorderStyle: TBorderStyle;
    FImages: TImageList;
    FStateImages: TImageList;
    FImageChangeLink: TChangeLink;
    FStateChangeLink: TChangeLink;
    FScrollTimer: TTimer;
    FScrollDirection: TScrollDirection;
    FScrollCount: Integer;
    FDragImage: TImageList;
    FTreeNTNodes: TTreeNTNodes;
    FSortType: TSortType;
    FSaveItems: TStringList;
    FSaveTopIndex: Integer;
    FSaveIndex: Integer;
    FSaveIndent: Integer;
    FSaveItemHeight: Integer;
    FScrollTime: Integer;
    FMemStream: TMemoryStream;
    FEditInstance: Pointer;
    FDefEditProc: Pointer;
    FEditHandle: HWND;
    FDragged: Boolean;
    FRClicked: Boolean;
    FOptions: TTreeOptions;
    FLastDropTarget: TTreeNTNode;
    FDragNode: TTreeNTNode;
    FDragObject: TDragObject;
    FTreeCanvas: TCanvas;
    FDesigner: TFormDesigner;

    FAfterPaint: TTVPaintEvent;
    FBeforePaint: TTVPaintEvent;
    FAfterItemPaint: TTVItemPaintEvent;
    FBeforeItemPaint: TTVItemPaintEvent;
    FOnDesignClick: TTVDesignClickEvent;
    FOnEditing: TTVEditingEvent;
    FOnEdited: TTVEditedEvent;
    FOnExpanded: TTVExpandedEvent;
    FOnExpanding: TTVExpandingEvent;
    FOnCollapsed: TTVExpandedEvent;
    FOnCollapsing: TTVCollapsingEvent;
    FOnChanging: TTVChangingEvent;
    FOnChange: TTVChangedEvent;
    FOnCompare: TTVCompareEvent;
    FOnDeletion: TTVExpandedEvent;
    FOnGetImageIndex: TTVExpandedEvent;
    FOnGetSelectedIndex: TTVExpandedEvent;
    FOnHint: TTVHintEvent;
    FOnSingleExpanded: TTVSingleExpandingEvent;
    procedure ActivateScrollTimer;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMCtl3DChanged(var Message: TMessage); message CM_CTL3DCHANGED;
    procedure CMDrag(var Message: TCMDrag); message CM_DRAG;
    procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
    procedure DoScroll(Sender: TObject);
    procedure DoDragOver(Source: TDragObject; X, Y: Integer);
    procedure EditWndProc(var Message: TMessage);
    procedure FontChanged(Sender: TObject);
    procedure GetImageIndex(Node: TTreeNTNode);
    procedure GetSelectedIndex(Node: TTreeNTNode);
    function GetDropTarget: TTreeNTNode;
    function GetIndent: Integer;
    function GetItemHeight: ShortInt;
    function GetNodeFromItem(const Item: TTVItem): TTreeNTNode;
    function GetScrollTime: Integer;
    function GetSearchString: String;
    function GetSelection: TTreeNTNode;
    function GetTopItem: TTreeNTNode;
    procedure ImageListChange(Sender: TObject);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetDropTarget(Value: TTreeNTNode);
    procedure SetImageList(Value: HImageList; Flags: Integer);
    procedure SetIndent(Value: Integer);
    procedure SetImages(Value: TImageList);
    procedure SetItemHeight(Value: ShortInt);
    procedure SetOptions(Values: TTreeOptions);
    procedure SetScrollTime(Value: Integer);
    procedure SetSelection(Value: TTreeNTNode);
    procedure SetSortType(Value: TSortType);
    procedure SetStateImages(Value: TImageList);
    procedure SetStyle(Value: Integer; UseStyle: Boolean);
    procedure SetTreeNTNodes(Value: TTreeNTNodes);
    procedure SetTopItem(Value: TTreeNTNode);
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHitTest;
  protected
    function CanEdit(Node: TTreeNTNode): Boolean; dynamic;
    function CanChange(Node: TTreeNTNode): Boolean; dynamic;
    function CanCollapse(Node: TTreeNTNode): Boolean; dynamic;
    function CanExpand(Node: TTreeNTNode): Boolean; dynamic;
    procedure Change(Node: TTreeNTNode); dynamic;
    procedure Collapse(Node: TTreeNTNode); dynamic;
    function CreateNode: TTreeNTNode; virtual;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    procedure DoAutoScroll(X,Y: Integer); virtual;
    procedure DoEndDrag(Target: TObject; X, Y: Integer); dynamic;
    procedure DoStartDrag(var DragObject: TDragObject); override;
    procedure Edit(const Item: TTVItem); dynamic;
    procedure Expand(Node: TTreeNTNode); dynamic;
    function GetDragImages: TCustomImageList; override;
    procedure Loaded; override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SetDragMode(Value: TDragMode); override;
    procedure WndProc(var Message: TMessage); override;

    property AutoScroll: Boolean read FAutoScroll write FAutoScroll default True;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property DesignerMode: Boolean read FDesignerMode write FDesignerMode;
    property Images: TImageList read FImages write SetImages;
    property Indent: Integer read GetIndent write SetIndent;
    property ItemHeight: ShortInt read GetItemHeight write SetItemHeight;
    property Items: TTreeNTNodes read FTreeNTNodes write SetTreeNTNodes;
    property Options: TTreeOptions read FOptions write SetOptions default [toShowLines,toShowRoot];
    property ScrollTime: Integer read GetScrollTime write SetScrollTime;
    property SearchString: String read GetSearchString;
    property SortType: TSortType read FSortType write SetSortType default stNone;
    property StateImages: TImageList read FStateImages write SetStateImages;

    property AfterPaint: TTVPaintEvent read FAfterPaint write FAfterPaint;
    property BeforePaint: TTVPaintEvent read FBeforePaint write FBeforePaint;
    property AfterItemPaint: TTVItemPaintEvent read FAfterItemPaint write FAfterItemPaint;
    property BeforeItemPaint: TTVItemPaintEvent read FBeforeItemPaint write FBeforeItemPaint;
    property OnCollapsing: TTVCollapsingEvent read FOnCollapsing write FOnCollapsing;
    property OnCollapsed: TTVExpandedEvent read FOnCollapsed write FOnCollapsed;
    property OnChanging: TTVChangingEvent read FOnChanging write FOnChanging;
    property OnChange: TTVChangedEvent read FOnChange write FOnChange;
    property OnCompare: TTVCompareEvent read FOnCompare write FOnCompare;
    property OnDeletion: TTVExpandedEvent read FOnDeletion write FOnDeletion;
    property OnDesignClick: TTVDesignClickEvent read FOnDesignClick write FOnDesignClick;
    property OnEditing: TTVEditingEvent read FOnEditing write FOnEditing;
    property OnEdited: TTVEditedEvent read FOnEdited write FOnEdited;
    property OnExpanding: TTVExpandingEvent read FOnExpanding write FOnExpanding;
    property OnExpanded: TTVExpandedEvent read FOnExpanded write FOnExpanded;
    property OnGetImageIndex: TTVExpandedEvent read FOnGetImageIndex write FOnGetImageIndex;
    property OnGetSelectedIndex: TTVExpandedEvent read FOnGetSelectedIndex write FOnGetSelectedIndex;
    property OnHint: TTVHintEvent read FOnHint write FOnHint;
    property OnSingleExpanded: TTVSingleExpandingEvent read FOnSingleExpanded write FOnSingleExpanded;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AlphaSort: Boolean;
    function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    procedure FullCollapse;
    procedure FullExpand;
    function GetHitTestInfoAt(X, Y: Integer): THitTests;
    function GetNodeAt(X, Y: Integer): TTreeNTNode;
    function IsEditing: Boolean;
    procedure LoadFromFile(const FileName: String);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const FileName: String);
    procedure SaveToStream(Stream: TStream);
    procedure ShowInsertMark(Node: TTreeNTNode; After: Boolean);

    property Designer: TFormDesigner read FDesigner;
    property DropTarget: TTreeNTNode read GetDropTarget write SetDropTarget;
    property Selected: TTreeNTNode read GetSelection write SetSelection;
    property TopItem: TTreeNTNode read GetTopItem write SetTopItem;
  end;

  TTreeNT = class(TCustomTreeNT)
  published
    property Align;
    property AutoScroll;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property Indent;
    property Images;
    property Items;
    property ItemHeight;
    property Options;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ScrollTime;
    property SearchString;
    property SortType;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property Visible;

    property AfterPaint;
    property BeforePaint;
    property AfterItemPaint;
    property BeforeItemPaint;
    property OnClick;
    property OnEnter;
    property OnExit;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnEditing;
    property OnEdited;
    property OnExpanding;
    property OnExpanded;
    property OnCollapsing;
    property OnCompare;
    property OnCollapsed;
    property OnChanging;
    property OnChange;
    property OnDeletion;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property OnHint;
    property OnSingleExpanded;
  end;

procedure Register;

//------------------------------------------------------------------------------

implementation

uses Consts, ComStrs, CommCtrlEx, Dialogs, TNTEditor;

//------------------------------------------------------------------------------

{$IFDEF VER100}

  procedure TreeNTError(Msg: String);

  begin
    raise ETreeNTError.Create(Msg);
  end;

{$ELSE}

  procedure TreeNTError(MsgID: Integer);

  begin
    raise ETreeNTError.CreateRes(MsgID);
  end;

{$ENDIF}

//----------------- TTreeNTNode ------------------------------------------------

function DefaultTreeNTSort(Node1,Node2: TTreeNTNode; lParam: Integer): Integer; stdcall;

begin
  with Node1 do
    if assigned(TreeView.OnCompare)
      then TreeView.OnCompare(TreeView,Node1,Node2,lParam,Result)
      else Result:=lstrcmp(PChar(Node1.Text),PChar(Node2.Text));
end;

//------------------------------------------------------------------------------

constructor TTreeNTNode.Create(AOwner: TTreeNTNodes);

begin
  inherited Create;
  FOverlayIndex:=-1;
  FStateIndex:=-1;
  FImageIndex:=-1;
  FSelectedIndex:=-1;
  FOwner:=AOwner;
  FDeleting:=False;
  FParentColor:=True;
  FParentFont:=True;

  FItemized:=False;
  FUpdateCount:=0;
  FProgramExpand:=0;
  FProgramCollapse:=0;
end;

//------------------------------------------------------------------------------

destructor TTreeNTNode.Destroy;

var AParent    : TTreeNTNode;
    CheckValue : Integer;

begin
  FDeleting:=True;
  If assigned(FOwner) and not FOwner.IsDeleting then
  begin
    AParent:=Parent;
    if (AParent <> nil) and (not AParent.IsUpdating) then
    begin
      if AParent.IndexOf(Self) > -1 then CheckValue:=1
                                    else CheckValue:=0;
      if AParent.CountEquals(CheckValue) then
      begin
        AParent.Expanded:=False;
        AParent.HasChildren:=False;
      end;
    end;
  end;

  if ItemId <> nil then TreeView_DeleteItem(Handle,ItemID);
  Data:=nil;
  FFont.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetHandle: HWND;

begin
  Result:=TreeView.Handle;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetTreeNT: TCustomTreeNT;

begin
  Result:=Owner.Owner;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.HasAsParent(Value: TTreeNTNode): Boolean;

begin
  if Self = Value then Result:=True
                  else
    if Parent <> nil then Result:=Parent.HasAsParent(Value)
                     else Result:=False;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetText(const S: String);

var Item: TTVItem;

begin
  FText:=S;
  with Item do
  begin
    Mask:=TVIF_TEXT or TVIF_HANDLE;
    hItem:=ItemId;
    pszText:=LPSTR_TEXTCALLBACK;
  end;
  TreeView_SetItem(Handle,Item);
  if (TreeView.SortType in [stText,stBoth]) and FItemized then
    if Parent <> nil then Parent.AlphaSort
                     else TreeView.AlphaSort; // no parent means top-level node
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetData(Value: Pointer);

begin
  FData:=Value;
  If FItemized and not Deleting then
  begin
    if (TreeView.SortType in [stData, stBoth]) and
       assigned(TreeView.OnCompare)            then
    begin
      if Parent <> nil then Parent.AlphaSort
                       else TreeView.AlphaSort; // no parent means top-level node
    end;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetState(NodeState: TNodeState): Boolean;

var Item: TTVItem;

begin
  Result:=False;
  with Item do
  begin
    Mask:=TVIF_STATE or TVIF_HANDLE;
    hItem:=ItemId;
    if TreeView_GetItem(Handle,Item) then
      case NodeState of
        nsCut: Result:=(state and TVIS_CUT) <> 0;
        nsFocused: Result:=(state and TVIS_FOCUSED) <> 0;
        nsSelected: Result:=(state and TVIS_SELECTED) <> 0;
        nsExpanded: Result:=(state and TVIS_EXPANDED) <> 0;
        nsDropHilited: Result:=(state and TVIS_DROPHILITED) <> 0;
        nsChecked: Result:=(State and TVIS_CHECKED) <> 0;
        nsExpandedOnce: Result:=(state and TVIS_EXPANDEDONCE) <> 0;
      end
      else Result:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetImageIndex(Value: Integer);

var Item: TTVItem;

begin
  FImageIndex:=Value;
  with Item do
  begin
    Mask:=TVIF_IMAGE or TVIF_HANDLE;
    hItem:=ItemId;
    iImage:=I_IMAGECALLBACK;
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetParentColor(AValue: Boolean);

begin
  if FParentColor <> AValue then
  begin
    FParentColor:=AValue;
    if assigned(TreeView) and not TreeView.Items.IsUpdating then FOwner.Repaint(Self);
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetParentFont(AValue: Boolean);

begin
  if FParentFont <> AValue then
  begin
    FParentFont:=AValue;
    if FParentFont then
    begin
      FFont.Free;
      FFont:=nil;
    end
    else FFont:=TFont.Create;
    if assigned(TreeView) and not TreeView.Items.IsUpdating then FOwner.Repaint(Self);
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetSelectedIndex(Value: Integer);

var Item: TTVItem;

begin
  FSelectedIndex:=Value;
  with Item do
  begin
    Mask:=TVIF_SELECTEDIMAGE or TVIF_HANDLE;
    hItem:=ItemId;
    iSelectedImage:=I_IMAGECALLBACK;
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetOverlayIndex(Value: Integer);

var Item: TTVItem;

begin
  FOverlayIndex:=Value;
  with Item do
  begin
    Mask:=TVIF_STATE or TVIF_HANDLE;
    stateMask:=TVIS_OVERLAYMask;
    hItem:=ItemId;
    state:=IndexToOverlayMask(OverlayIndex + 1);
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetStateIndex(Value: Integer);

var Item: TTVItem;

begin
  FStateIndex:=Value;
  if Value >= 0 then Dec(Value);
  with Item do
  begin
    Mask:=TVIF_STATE;
    stateMask:=TVIS_STATEIMAGEMASK;
    hItem:=ItemId;
    state:=IndexToStateImageMask(Value + 1);
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.ExpandItem(Expand: Boolean; Recurse: Boolean);

var
  Flag : Integer;
  Node : TTreeNTNode;
  TV   : TCustomTreeNT;

begin
  if Recurse then
  begin
    Node:=Self;
    repeat
      Node.ExpandItem(Expand,False);
      Node:=Node.GetNext;
    until (Node = nil) or not Node.HasAsParent(Self);
  end
  else
  begin
    TV:=TreeView;
    If TV = nil then Exit;
    if Expand then // try to expand
    begin
      Inc(FProgramExpand);
      try
        if not Expanded and TV.CanExpand(Self) then
        begin
          Flag:=TVE_EXPAND;
          TreeView_Expand(Handle,ItemID,Flag);
          TV.Expand(Self);
        end;
      finally
        Dec(FProgramExpand);
      end;
    end
    else
    begin
      Inc(FProgramCollapse);
      try
        If Expanded then
        begin  
          if TV.CanCollapse(Self) then
          begin
            Flag:=TVE_COLLAPSE;
            TreeView_Expand(Handle,ItemID,Flag);
            TV.Collapse(Self);
          end;
        end;
      finally
        Dec(FProgramCollapse);
      end;
    end; 
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.Expand(Recurse: Boolean);

begin
  ExpandItem(True,Recurse);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.Collapse(Recurse: Boolean);

begin
  ExpandItem(False, Recurse);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetExpanded: Boolean;

begin
  Result:=GetState(nsExpanded);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetExpandedOnce: Boolean;

begin
  Result:=GetState(nsExpandedOnce);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetExpanded(Value: Boolean);

begin
  if Value then Expand(False)
           else Collapse(False);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetSelected: Boolean;

begin
  Result:=GetState(nsSelected);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetSelected(Value: Boolean);


begin
  if Value then TreeView_SelectItem(Handle,ItemID)
           else
    if Selected then TreeView_SelectItem(Handle,nil);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetCut: Boolean;

begin
  Result:=GetState(nsCut);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetColor(AValue: TColor);

begin
  if FColor <> AValue then
  begin
     FColor:=AValue;
     FParentColor:=False;
     if assigned(TreeView) and not TreeView.Items.IsUpdating then FOwner.Repaint(Self);
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetCut(Value: Boolean);

var Item     : TTVItem;
    Template : Integer;
    
begin
  if Value then Template:=-1
           else Template:=0;
  with Item do
  begin
    Mask:=TVIF_STATE or TVIF_HANDLE;
    hItem:=ItemId;
    stateMask:=TVIS_CUT;
    state:=stateMask and Template;
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetChecked: Boolean;

begin
  Result:=GetState(nsChecked);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetDropTarget: Boolean;

begin
  Result:=GetState(nsDropHilited);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetDropTarget(Value: Boolean);

begin
  if Value then TreeView_SelectDropTarget(Handle,ItemId)
           else
    if DropTarget then TreeView_SelectDropTarget(Handle,nil);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetChildren: Boolean;

var Item: TTVItem;

begin
  Item.Mask:=TVIF_CHILDREN or TVIF_HANDLE;
  Item.hItem:=ItemId;
  if TreeView_GetItem(Handle,Item) then Result:=Item.cChildren > 0
                                    else Result:=False;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetFocused(Value: Boolean);

var Item     : TTVItem;
    Template : Integer;

begin
  if Value then Template:=-1
           else Template:=0;
  with Item do
  begin
    Mask:=TVIF_STATE or TVIF_HANDLE;
    hItem:=ItemId;
    stateMask:=TVIS_FOCUSED;
    state:=stateMask and Template;
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetFocused: Boolean;

begin
  Result:=GetState(nsFocused);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetParent: TTreeNTNode;

begin
  with FOwner do Result:=GetNode(TreeView_GetParent(Handle,ItemID));
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetNextSibling: TTreeNTNode;

begin
  with FOwner do Result:=GetNode(TreeView_GetNextSibling(Handle,ItemID));
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetPrevSibling: TTreeNTNode;

begin
  with FOwner do Result:=GetNode(TreeView_GetPrevSibling(Handle,ItemID));
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetNextVisible: TTreeNTNode;

begin
  if IsVisible then with FOwner do Result:=GetNode(TreeView_GetNextVisible(Handle,ItemId))
               else Result:=nil;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetPrevVisible: TTreeNTNode;

begin
  with FOwner do Result:=GetNode(TreeView_GetPrevVisible(Handle,ItemID));
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetNextChild(Value: TTreeNTNode): TTreeNTNode;

begin
  if Value <> nil then Result:=Value.GetNextSibling
                  else Result:=nil;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetPrevChild(Value: TTreeNTNode): TTreeNTNode;

begin
  if Value <> nil then Result:=Value.GetPrevSibling
                  else Result:=nil;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetFirstChild: TTreeNTNode;

begin
  with FOwner do Result:=GetNode(TreeView_GetChild(Handle,ItemID));
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetLastChild: TTreeNTNode;

var Node: TTreeNTNode;

begin
  Result:=GetFirstChild;
  if Result <> nil then
  begin
    Node:=Result;
    repeat
      Result:=Node;
      Node:=Result.GetNextSibling;
    until Node = nil;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetNext: TTreeNTNode;

var NodeID,
    ParentID : HTreeItem;
    Handle   : HWND;

begin
  Handle:=FOwner.Handle;
  NodeID:=TreeView_GetChild(Handle,ItemID);
  if NodeID = nil then NodeID:=TreeView_GetNextSibling(Handle,ItemID);
  ParentID:=ItemId;
  while (NodeID = nil) and (ParentID <> nil) do
  begin
    ParentID:=TreeView_GetParent(Handle, ParentID);
    NodeID:=TreeView_GetNextSibling(Handle, ParentID);
  end;
  Result:=FOwner.GetNode(NodeID);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetPrev: TTreeNTNode;

var Node: TTreeNTNode;

begin
  Result:=GetPrevSibling;
  if Result <> nil then
  begin
    Node:=Result;
    repeat
      Result:=Node;
      Node:=Result.GetLastChild;
    until Node = nil;
  end else
    Result:=Parent;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetAbsoluteIndex: Integer;

var Node: TTreeNTNode;

begin
  Result:=-1;
  Node:=Self;
  while Node <> nil do
  begin
    Inc(Result);
    Node:=Node.GetPrev;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetFont: TFont;

begin
  if FFont = nil then FFont:=TFont.Create;
  FParentFont:=False;
  Result:=FFont;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetIndex: Integer;

var  Node : TTreeNTNode;

begin
  Result:=-1;
  Node:=Self;
  while Node <> nil do
  begin
    Inc(Result);
    Node:=Node.GetPrevSibling;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetItem(Index: Integer): TTreeNTNode;

begin
  Result:=GetFirstChild;
  while (Result <> nil) and (Index > 0) do
  begin
    Result:=GetNextChild(Result);
    Dec(Index);
  end;
  if Result = nil then TreeNTError(SListIndexError);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetChecked(Value: Boolean);

var Item: TTVItem;

begin
  FillChar(Item,SizeOf(Item),0);
  with Item do
  begin
    Mask:=TVIF_STATE or TVIF_HANDLE;
    StateMask:=TVIS_STATEIMAGEMASK;
    if Value then State:=TVIS_CHECKED
             else State:=TVIS_CHECKED shr 1;
    hItem:=ItemId;
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetChildren(Value: Boolean);

var Item : TTVItem;

begin
  with Item do
  begin
    Mask:=TVIF_CHILDREN or TVIF_HANDLE;
    hItem:=ItemId;
    cChildren:=Ord(Value);
  end;
  TreeView_SetItem(Handle,Item);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetFont(AFont: TFont);

begin
  if FFont = nil then FFont:=TFont.Create;
  FFont.Assign(AFont);
  FParentFont:=False;
  if assigned(TreeView) and not TreeView.Items.IsUpdating then FOwner.Repaint(Self);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.SetItem(Index: Integer; Value: TTreeNTNode);

begin
  Item[Index].Assign(Value);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.IndexOf(Value: TTreeNTNode): Integer;

var Node: TTreeNTNode;

begin
  Result:=-1;
  Node:=GetFirstChild;
  while (Node <> nil) do
  begin
    Inc(Result);
    if Node = Value then Break;
    Node:=GetNextChild(Node);
  end;
  if Node = nil then Result:=-1;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.IsUpdating: Boolean;

begin
  Result:=(FUpdateCount > 0) or FDeleting;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetCount: Integer;

var Node: TTreeNTNode;

begin
  Result:=0;
  Node:=GetFirstChild;
  while Node <> nil do
  begin
    Inc(Result);
    Node:=Node.GetNextChild(Node);
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.CountEquals(N: Integer): Boolean;

var Node : TTreeNTNode;
    Cnt  : integer;

begin
  Cnt:=0;
  Result:=False;
  Node:=GetFirstChild;
  while Node <> nil do
  begin
    Inc(Cnt);
    Node:=Node.GetNextChild(Node);
    If Cnt > N then Exit;
  end;
  If Cnt = N then Result:=True;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.BeginUpdate;

begin
  Inc(FUpdateCount);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.EndUpdate;

begin
  if FUpdateCount > 0 then Dec(FUpdateCount);
  if (FUpdateCount = 0) and assigned(FOwner) then
  begin
    FOwner.Repaint(Self);
    if FOwner.Owner.SortType in [stText,stBoth] then AlphaSort;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.EndEdit(Cancel: Boolean);

begin
  TreeView_EndEditLabelNow(Handle,Cancel);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.InternalMove(ParentNode,Node: TTreeNTNode; HItem: HTreeItem; AddMode: TAddMode);

var I          : Integer;
    NodeId     : HTreeItem;
    TreeNTItem : TTVItem;
    Children,
    IsSelected : Boolean;

begin
  if (AddMode = taInsert) and (Node <> nil)
    then NodeId:=Node.ItemId
    else NodeId:=nil;
  Children:=HasChildren;
  IsSelected:=Selected;
  if (Parent <> nil) and (Parent.Count = 1) then
  begin
    Parent.Expanded:=False;
    Parent.HasChildren:=False;
  end;
  with TreeNTItem do
  begin
    Mask:=TVIF_PARAM or TVIF_HANDLE;
    hItem:=ItemId;
    lParam:=0;
  end;
  TreeView_SetItem(Handle,TreeNTItem);
  with Owner do HItem:=AddItem(HItem,NodeId,CreateItem(Self),AddMode);
  if HItem = nil then
  {$IFDEF VER100}
    raise EOutOfResources.Create(sInsertError);
  {$ELSE}
    raise EOutOfResources.CreateRes(sInsertError);
  {$ENDIF}
  for I:=Count - 1 downto 0 do Item[I].InternalMove(Self,nil,HItem,taAddFirst);
  TreeView_DeleteItem(Handle,ItemID);
  FItemId:=HItem;
  Assign(Self);
  HasChildren:=Children;
  Selected:=IsSelected;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.MoveTo(Destination: TTreeNTNode; Mode: TNodeAttachMode);

var AddMode       : TAddMode;
    Node          : TTreeNTNode;
    HItem         : HTreeItem;
    OldOnChanging : TTVChangingEvent;
    OldOnChange   : TTVChangedEvent;

begin
  OldOnChanging:=TreeView.OnChanging;
  OldOnChange:=TreeView.OnChange;
  TreeView.OnChanging:=nil;
  TreeView.OnChange:=nil;
  try
    if (Destination = nil) or not Destination.HasAsParent(Self) then
    begin
      AddMode:=taAdd;
      if (Destination <> nil) and not (Mode in [naAddChild, naAddChildFirst]) then
        Node:=Destination.Parent else
        Node:=Destination;
      case Mode of
        naAdd,
        naAddChild: AddMode:=taAdd;
        naAddFirst,
        naAddChildFirst: AddMode:=taAddFirst;
        naInsert:
          begin
            Destination:=Destination.GetPrevSibling;
            if Destination = nil then AddMode:=taAddFirst
            else AddMode:=taInsert;
          end;
      end;
      if Node <> nil then
        HItem:=Node.ItemId else
        HItem:=nil;
      InternalMove(Node, Destination, HItem, AddMode);
      Node:=Parent;
      if Node <> nil then
      begin
        Node.HasChildren:=True;
        Node.Expanded:=True;
      end;                         
    end;
  finally
    TreeView.OnChanging:=OldOnChanging;
    TreeView.OnChange:=OldOnChange;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.MakeVisible;

begin
  TreeView_EnsureVisible(Handle,ItemID);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.GetLevel: Integer;

var Node : TTreeNTNode;

begin
  Result:=0;
  Node:=Parent;
  while Node <> nil do
  begin
    Inc(Result);
    Node:=Node.Parent;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.IsNodeVisible: Boolean;

var Rect: TRect;

begin
  Result:=TreeView_GetItemRect(Handle,ItemID,Rect,True);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.HasVisibleParent: Boolean;

begin
  Result:=assigned(Parent) and Parent.Expanded;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.EditText: Boolean;

begin
  Result:=TreeView_EditLabel(Handle,ItemID) <> 0;
end;

//------------------------------------------------------------------------------

function TTreeNTNode.DisplayRect(TextOnly: Boolean): TRect;

begin
  FillChar(Result,SizeOf(Result),0);
  TreeView_GetItemRect(Handle,ItemID,Result,TextOnly);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.AlphaSort: Boolean;

begin
  Result:=CustomSort(nil,0);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;

var SortCB : TTVSortCB;

begin
  with SortCB do
  begin
    if not assigned(SortProc) then lpfnCompare:=@DefaultTreeNTSort
                              else lpfnCompare:=SortProc;
    hParent:=ItemId;
    lParam:=Data;
  end;
  Result:=TreeView_SortChildrenCB(Handle, SortCB, 0);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.Delete;

begin
  if not Deleting then Free;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.DeleteChildren;

var Node: TTreeNTNode;

begin
  If FDeleting then Exit; 
  FDeleting:=True;
  try
    Node:=GetFirstChild;
    While Node <> nil do
    begin
      Node.Delete;
      Node:=GetFirstChild;
      while assigned(Node) and Node.Deleting do Node:=GetNextChild(Node);
    end;
  finally
    Expanded:=False;
    HasChildren:=False;
    FDeleting:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.Assign(Source: TPersistent);

var Node : TTreeNTNode;

begin
  if Source is TTreeNTNode then
  begin
    Node:=TTreeNTNode(Source);
    Text:=Node.Text;
    Data:=Node.Data;
    ImageIndex:=Node.ImageIndex;
    SelectedIndex:=Node.SelectedIndex;
    // workaround for checkbox style
    if assigned(TreeView) and
       assigned(TreeView.StateImages) then StateIndex:=Node.StateIndex;
    OverlayIndex:=Node.OverlayIndex;
    Focused:=Node.Focused;
    DropTarget:=Node.DropTarget;
    Cut:=Node.Cut;
    HasChildren:=Node.HasChildren;
    FParentFont:=Node.FParentFont;
    FParentColor:=Node.FParentColor;
    if not FParentFont then
    begin
      if FFont = nil then FFont:=TFont.Create;
      FFont.Assign(Node.FFont);
    end
    else
    begin
      FFont.Free;
      FFont:=nil;
    end;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

function TTreeNTNode.IsEqual(Node: TTreeNTNode): Boolean;

begin
  Result:=(Text = Node.Text) and (Data = Node.Data);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.ReadData(Stream: TStream; Info: PNodeInfo);

var I,Size,ItemCount : Integer;

begin
  Stream.ReadBuffer(Size, SizeOf(Size));
  Stream.ReadBuffer(Info^, Size);
  Text:=Info.Text;
  ImageIndex:=Info.ImageIndex;
  SelectedIndex:=Info.SelectedIndex;
  // workaround for checkbox style
  if assigned(TreeView) and
     assigned(TreeView.StateImages) then StateIndex:=Info.StateIndex;
  OverlayIndex:=Info.OverlayIndex;
  Checked:=Info.Checked;
  Data:=Info.Data;
  Color:=Info.Color;
  ParentColor:=Info.ParentColor;
  ParentFont:=Info.ParentFont;
  if not ParentFont then
  begin
    FFont:=TFont.Create;
    FFont.Height:=Info.FontData.Height;
    FFont.Name:=Info.FontData.Name;
    FFont.Pitch:=Info.FontData.Pitch;
    FFont.Style:=Info.FontData.Style;
    FFont.Color:=Info.FontColor;
  end;
  ItemCount:=Info.Count;
  for I:=0 to ItemCount-1 do
    with Owner.AddChild(Self,'') do ReadData(Stream,Info);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNode.WriteData(Stream: TStream; Info: PNodeInfo);

var I, Size, L,ItemCount: Integer;

begin
  L:=Length(Text);
  if L > 255 then L:=255;
  Size:=SizeOf(TNodeInfo)+L-255;
  Info.Text:=Text;
  Info.ImageIndex:=ImageIndex;
  Info.SelectedIndex:=SelectedIndex;
  Info.OverlayIndex:=OverlayIndex;
  Info.StateIndex:=StateIndex;
  Info.Checked:=Checked;
  Info.Data:=Data;
  Info.Color:=FColor;
  Info.ParentFont:=FParentFont;
  Info.ParentColor:=FParentColor;
  if not FParentFont then
  begin
    Info.FontData.Height:=FFont.Height;
    Info.FontData.Name:=FFont.Name;
    Info.FontData.Pitch:=FFont.Pitch;
    Info.FontData.Style:=FFont.Style;
    Info.FontColor:=FFont.Color;
  end;
  ItemCount:=Count;
  Info.Count:=ItemCount;
  Stream.WriteBuffer(Size,SizeOf(Size));
  Stream.WriteBuffer(Info^,Size);
  for I:=0 to ItemCount-1 do Item[I].WriteData(Stream,Info);
end;

//----------------- TTreeNTNodes -----------------------------------------------

constructor TTreeNTNodes.Create(AOwner: TCustomTreeNT);

begin
  inherited Create;
  FOwner:=AOwner;
end;

//------------------------------------------------------------------------------

destructor TTreeNTNodes.Destroy;

begin
  Clear;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.GetCount: Integer;

begin
  if Owner.HandleAllocated then Result:=TreeView_GetCount(Handle)
                           else Result:=0;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.GetHandle: HWND;

begin
  Result:=Owner.Handle;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.Delete(Node: TTreeNTNode);

begin
  if (Node.ItemId = nil) and assigned(Owner.FOnDeletion) then
    Owner.FOnDeletion(Self,Node);
  Node.Delete;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.Clear;

begin
  if Owner.HandleAllocated then
  try
    FDeleting:=True;
    BeginUpdate;
    TreeView_DeleteAllItems(Handle);
    EndUpdate;
  finally
    FDeleting:=False;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddChildFirst(Node: TTreeNTNode; const S: String): TTreeNTNode;

begin
  Result:=AddChildObjectFirst(Node,S,nil);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddChildObjectFirst(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;

begin
  Result:=InternalAddObject(Node,S,Ptr,taAddFirst);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddChild(Node: TTreeNTNode; const S: String): TTreeNTNode;

begin
  Result:=AddChildObject(Node,S,nil);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddChildObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;

begin
  Result:=InternalAddObject(Node,S,Ptr,taAdd);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddFirst(Node: TTreeNTNode; const S: String): TTreeNTNode;

begin
  Result:=AddObjectFirst(Node,S,nil);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddObjectFirst(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;

begin
  if Node <> nil then Node:=Node.Parent;
  Result:=InternalAddObject(Node,S,Ptr,taAddFirst);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.Add(Node: TTreeNTNode; const S: String): TTreeNTNode;

begin
  Result:=AddObject(Node,S,nil);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.Repaint(Node: TTreeNTNode);

var R : TRect;

begin
  while (Node <> nil) and not Node.IsVisible do Node:=Node.Parent;
  if Node <> nil then
  begin
    R:=Node.DisplayRect(False);
    InvalidateRect(Owner.Handle,@R,True);
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;

begin 
  if Node <> nil then Node:=Node.Parent;
  Result:=InternalAddObject(Node,S,Ptr,taAdd);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.Insert(Node: TTreeNTNode; const S: String): TTreeNTNode;

begin
  Result:=InsertObject(Node,S,nil);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.AddedNode(ParentNode: TTreeNTNode);

begin
  if assigned(ParentNode) then
  begin
    ParentNode.HasChildren:=True;
    If not ParentNode.IsUpdating then Repaint(ParentNode);
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.InsertObject(Node: TTreeNTNode; const S: String; Ptr: Pointer): TTreeNTNode;

var Item,
    ItemID  : HTreeItem;
    Parent  : TTreeNTNode;
    AddMode : TAddMode;

begin
  Result:=Owner.CreateNode;
  try
    Parent:=nil;
    Item:=nil;
    ItemId:=nil;
    AddMode:=taInsert;
    if Node <> nil then
    begin
      Parent:=Node.Parent;
      if Parent <> nil then Item:=Parent.ItemId;
      Node:=Node.GetPrevSibling;
      if Node <> nil then ItemId:=Node.ItemId
                     else AddMode:=taAddFirst;
    end;
    Result.Data:=Ptr;
    Result.Text:=S;
    Item:=AddItem(Item,ItemID,CreateItem(Result),AddMode);
    if Item = nil then 
    {$IFDEF VER100}
      raise EOutOfResources.Create(sInsertError);
    {$ELSE}
      raise EOutOfResources.CreateRes(sInsertError);
    {$ENDIF}
    Result.FItemId:=Item;
    AddedNode(Parent);
  except
    Result.Free;
    raise;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.IsDeleting: Boolean;

begin
  Result:=FDeleting;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.IsUpdating: Boolean;

begin
  Result:=FUpdateCount > 0;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.InternalAddObject(Node: TTreeNTNode; const S: String; Ptr: Pointer; AddMode: TAddMode): TTreeNTNode;

var Item : HTreeItem;

begin
  Result:=Owner.CreateNode;
  try
    if Node <> nil then Item:=Node.ItemId
                   else Item:=nil;
    Result.Data:=Ptr;
    Result.Text:=S;
    Item:=AddItem(Item,nil,CreateItem(Result),AddMode);
    if Item = nil then 
    {$IFDEF VER100}
      raise EOutOfResources.Create(sInsertError);
    {$ELSE}
      raise EOutOfResources.CreateRes(sInsertError);
    {$ENDIF}
    Result.FItemId:=Item;
    AddedNode(Node);
    if Owner.SortType in [stText,stBoth] then
    begin
      If Node = nil then
        if not IsUpdating then Owner.AlphaSort else
      else
        if not Node.IsUpdating then Node.AlphaSort;
    end;
  except
    Result.Free;
    raise;
  end;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.CreateItem(Node: TTreeNTNode): TTVItem;

begin
  with Result do
  begin
    Mask:=TVIF_TEXT or TVIF_PARAM or TVIF_IMAGE or
          TVIF_SELECTEDIMAGE or TVIF_CHILDREN;
    lParam:=Longint(Node);
    pszText:=LPSTR_TEXTCALLBACK;
    iImage:=I_IMAGECALLBACK;
    iSelectedImage:=I_IMAGECALLBACK;
    cChildren:=0;
  end;
  Node.FItemized:=True;
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.AddItem(Parent, Target: HTreeItem; const Item: TTVItem; AddMode: TAddMode): HTreeItem;

var InsertStruct : TTVInsertStruct;

begin
  with InsertStruct do
  begin
    hParent:=Parent;
    case AddMode of
      taAddFirst : hInsertAfter:=TVI_FIRST;
      taAdd      : hInsertAfter:=TVI_LAST;
      taInsert   : hInsertAfter:=Target;
    end;
  end;
  InsertStruct.Item:=Item;
  Result:=TreeView_InsertItem(Handle,InsertStruct);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.GetFirstNode: TTreeNTNode;

begin
  Result:=GetNode(TreeView_GetRoot(Handle));
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.GetNodeFromIndex(Index: Integer): TTreeNTNode;

begin
  Result:=GetFirstNode;
  while (Index <> 0) and (Result <> nil) do
  begin
    Result:=Result.GetNext;
    Dec(Index);
  end;
  if Result = nil then TreeNTError(sInvalidIndex);
end;

//------------------------------------------------------------------------------

function TTreeNTNodes.GetNode(ItemId: HTreeItem): TTreeNTNode;

var Item : TTVItem;

begin
  with Item do
  begin
    hItem:=ItemId;
    Mask:=TVIF_PARAM or TVIF_HANDLE;
  end;
  if TreeView_GetItem(Handle,Item) then Result:=TTreeNTNode(Item.lParam)
  else Result:=nil;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.SetItem(Index: Integer; Value: TTreeNTNode);

begin
  GetNodeFromIndex(Index).Assign(Value);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.BeginUpdate;

begin
  if FUpdateCount = 0 then SetUpdateState(True);
  Inc(FUpdateCount);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.SetUpdateState(Updating: Boolean);

begin
  SendMessage(Handle,WM_SETREDRAW,Ord(not Updating),0);
  if not Updating then
  begin
    InvalidateRect(Handle,nil,True);
    if assigned(Owner) and (Owner.SortType in [stText,stBoth])
      then Owner.AlphaSort
      else Owner.Refresh;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.EndUpdate;

begin
  if FUpdateCount > 0 then Dec(FUpdateCount);
  if FUpdateCount = 0 then SetUpdateState(False);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.Assign(Source: TPersistent);

var TreeNTNodes : TTreeNTNodes;
    MemStream  : TMemoryStream;

begin
  if Source is TTreeNTNodes then
  begin
    TreeNTNodes:=TTreeNTNodes(Source);
    Clear;
    MemStream:=TMemoryStream.Create;
    try
      TreeNTNodes.WriteData(MemStream);
      MemStream.Position:=0;
      ReadData(MemStream);
    finally
      MemStream.Free;
    end;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.DefineProperties(Filer: TFiler);

  function WriteNodes: Boolean;

  var I     : Integer;
      Nodes : TTreeNTNodes;

  begin
    Nodes:=TTreeNTNodes(Filer.Ancestor);
    if (Nodes <> nil) and (Nodes.Count = Count) then
      for I:=0 to Count - 1 do
      begin
        Result:=not Item[I].IsEqual(Nodes[I]);
        if Result then Break;
      end
    else Result:=Count > 0;
  end;

begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('Data',ReadData,WriteData,WriteNodes);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.ReadData(Stream: TStream);

var I,
    Count    : Integer;
    NodeInfo : TNodeInfo;

begin
  Stream.ReadBuffer(Count, SizeOf(Count));
  for I:=0 to Count-1 do Add(nil,'').ReadData(Stream,@NodeInfo);
end;

//------------------------------------------------------------------------------

procedure TTreeNTNodes.WriteData(Stream: TStream);

var I        : Integer;
    Node     : TTreeNTNode;
    NodeInfo : TNodeInfo;

begin
  I:=0;
  Node:=GetFirstNode;
  while Node <> nil do
  begin
    Inc(I);
    Node:=Node.GetNextSibling;
  end;
  Stream.WriteBuffer(I,SizeOf(I));
  Node:=GetFirstNode;
  while Node <> nil do
  begin
    Node.WriteData(Stream,@NodeInfo);
    Node:=Node.GetNextSibling;
  end;
end;

//----------------- TTreeNTStrings ---------------------------------------------

type
  TTreeNTStrings = class(TStrings)
  private
    FOwner: TTreeNTNodes;
  protected
    function Get(Index: Integer): String; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create(AOwner: TTreeNTNodes);
    function Add(const S: String): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: String); override;
    property Owner: TTreeNTNodes read FOwner;
  end;

//------------------------------------------------------------------------------

constructor TTreeNTStrings.Create(AOwner: TTreeNTNodes);

begin
  inherited Create;
  FOwner:=AOwner;
end;

//------------------------------------------------------------------------------

function TTreeNTStrings.Get(Index: Integer): String;

const TAB = Chr(9);

var Level,
    I      : Integer;
    Node   : TTreeNTNode;

begin
  Result:='';
  Node:=Owner.GetNodeFromIndex(Index);
  Level:=Node.Level;
  for I:=0 to Level - 1 do Result:=Result+TAB;
  Result:=Result + Node.Text;
end;

//------------------------------------------------------------------------------

function TTreeNTStrings.GetObject(Index: Integer): TObject;

begin
  Result:=Owner.GetNodeFromIndex(Index).Data;
end;

//------------------------------------------------------------------------------

procedure TTreeNTStrings.PutObject(Index: Integer; AObject: TObject);

begin
  Owner.GetNodeFromIndex(Index).Data:=AObject;
end;

//------------------------------------------------------------------------------

function TTreeNTStrings.GetCount: Integer;

begin
  Result:=Owner.Count;
end;

//------------------------------------------------------------------------------

procedure TTreeNTStrings.Clear;

begin
  Owner.Clear;
end;

//------------------------------------------------------------------------------

procedure TTreeNTStrings.Delete(Index: Integer);

begin
  Owner.GetNodeFromIndex(Index).Delete;
end;

//------------------------------------------------------------------------------

procedure TTreeNTStrings.SetUpdateState(Updating: Boolean);

begin
  SendMessage(Owner.Handle,WM_SETREDRAW,Ord(not Updating),0);
  if not Updating then Owner.Owner.Refresh;
end;

//------------------------------------------------------------------------------

function TTreeNTStrings.Add(const S: String): Integer;

var Level,
    OldLevel,
    I         : Integer;
    NewStr    : String;
    Node      : TTreeNTNode;

  function GetBufStart(Buffer: PChar; var Level: Integer): PChar;

  begin
    Level:=0;
    while Buffer^ in [' ', #9] do
    begin
      Inc(Buffer);
      Inc(Level);
    end;
    Result:=Buffer;
  end;

begin
  Result:=GetCount;
  if (Length(S) = 1) and (S[1] = Chr($1A)) then Exit;
  Node:=nil;
  OldLevel:=0;
  NewStr:=GetBufStart(PChar(S), Level);
  if Result > 0 then
  begin
    Node:=Owner.GetNodeFromIndex(Result - 1);
    OldLevel:=Node.Level;
  end;
  if (Level > OldLevel) or (Node = nil) then
  begin
    if Level - OldLevel > 1 then TreeNTError(sInvalidLevel);
  end
  else
  begin
    for I:=OldLevel downto Level do
    begin
      Node:=Node.Parent;
      if (Node = nil) and (I - Level > 0) then
        TreeNTError(sInvalidLevel);
    end;
  end;
  Owner.AddChild(Node, NewStr);
end;

//------------------------------------------------------------------------------

procedure TTreeNTStrings.Insert(Index: Integer; const S: String);

begin
  with Owner do Insert(GetNodeFromIndex(Index),S);
end;
                           
//----------------- TCustomTreeNT ----------------------------------------------

constructor TCustomTreeNT.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  ControlStyle:=ControlStyle+[csCaptureMouse]+[csDisplayDragImage];
  Width:=121;
  Height:=97;
  TabStop:=True;
  ParentColor:=False;
  //Font.OnChange:=FontChanged; // track font color changes
  FAutoScroll:=True;
  FTreeNTNodes:=TTreeNTNodes.Create(Self);
  FScrollTimer:=TTimer.Create(Self);
  FScrollTimer.OnTimer:=DoScroll;
  FBorderStyle:=bsSingle;
  FDragImage:=TImageList.CreateSize(32,32);
  FSaveIndent:=-1;
  FSaveItemHeight:=-1;
  FOptions:=[toShowLines,toShowRoot];
  FTreeCanvas:=TCanvas.Create;
  FEditInstance:=MakeObjectInstance(EditWndProc);
  FImageChangeLink:=TChangeLink.Create;
  FImageChangeLink.OnChange:=ImageListChange;
  FStateChangeLink:=TChangeLink.Create;
  FStateChangeLink.OnChange:=ImageListChange;
  if (csDesigning in ComponentState) and
     (AOwner is TControl)            then FDesigner:=GetParentForm(TControl(AOwner)).Designer as TFormDesigner;
end;

//------------------------------------------------------------------------------

destructor TCustomTreeNT.Destroy;

begin
  FTreeCanvas.Free;
  FScrollTimer.Free;
  Items.Free;
  FSaveItems.Free;
  FDragImage.Free;
  FMemStream.Free;
  FreeObjectInstance(FEditInstance);
  FImageChangeLink.Free;
  FStateChangeLink.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CreateParams(var Params: TCreateParams);

begin
  InitCommonControls;
  inherited CreateParams(Params);
  CreateSubClass(Params,WC_TreeVIEW);
  with Params do
  begin
    if FBorderStyle = bsSingle then Style:=Style+WS_BORDER;
    if toShowLines in FOptions then Style:=Style+TVS_HASLINES;
    if toShowRoot in FOptions then Style:=Style+TVS_LINESATROOT;
    if toShowButtons in FOptions then Style:=Style+TVS_HASBUTTONS;
    if not (toReadOnly in FOptions) then Style:=Style+TVS_EDITLABELS;
    if not (toHideSelection in FOptions) then Style:=Style+TVS_SHOWSELALWAYS;
    if DragMode = dmManual then Style:=Style+TVS_DISABLEDRAGDROP;
    if toHotTrack in FOptions then Style:=Style+TVS_TRACKSELECT;
    if toCheckBoxes in FOptions then Style:=Style+TVS_CHECKBOXES;
    if not (toToolTips in FOptions) then Style:=Style+TVS_NOTOOLTIPS;
    if toSingleExpand in FOptions then Style:=Style+TVS_SINGLEEXPAND;
    if toInfoTip in FOptions then Style:=Style+TVS_INFOTIP;
    if toNoScroll in FOptions then Style:=Style+TVS_NOSCROLL;
    if toFullRowSelect in FOptions then Style:=Style+TVS_FULLROWSELECT;

    if Ctl3D and (FBorderStyle = bsSingle) then ExStyle:=Params.ExStyle or WS_EX_CLIENTEDGE;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CreateWnd;

var TempImages : TImageList;

begin
  inherited CreateWnd;
  if FMemStream <> nil then
  begin
    Items.ReadData(FMemStream);
    FMemStream.Destroy;
    FMemStream:=nil;
    SetTopItem(Items.GetNodeFromIndex(FSaveTopIndex));
    FSaveTopIndex:=0;
    SetSelection(Items.GetNodeFromIndex(FSaveIndex));
    FSaveIndex:=0;
  end;
  if FSaveIndent <> -1 then Indent:=FSaveIndent;
  if FSaveItemHeight <> -1 then ItemHeight:=FSaveItemHeight;
  if (Images <> nil) and Images.HandleAllocated then SetImageList(Images.Handle, TVSIL_NORMAL);
  if (StateImages <> nil) and StateImages.HandleAllocated then SetImageList(StateImages.Handle, TVSIL_STATE);
  if not assigned(FImages) and (toCheckBoxes in FOptions) then
  begin
    TempImages:=TImageList.Create(nil);
    try
      Images:=TempImages;
      Images:=nil;
    finally
      TempImages.Free;
    end;
  end;
  FontChanged(nil);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DestroyWnd;

var Node : TTreeNTNode;

begin
  if Items.Count > 0 then
  begin
    FMemStream:=TMemoryStream.Create;
    Items.WriteData(FMemStream);
    FMemStream.Position:=0;
    Node:=GetTopItem;
    if Node <> nil then FSaveTopIndex:=Node.AbsoluteIndex;
    Node:=Selected;
    if Node <> nil then FSaveIndex:=Node.AbsoluteIndex;
  end;
  FSaveIndent:=Indent;
  FSaveItemHeight:=ItemHeight;
  inherited DestroyWnd;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DoScroll(Sender: TObject);

var ScrollInfo : TScrollInfo;

begin
  if Dragging and assigned(FDragObject) then FDragObject.HideDragImage;
  Inc(FScrollCount,8);
  if (FScrollCount) < 100 then FScrollTimer.Interval:=100-FScrollCount;
  with ScrollInfo do
  begin
    cbSize:=SizeOf(TScrollInfo);
    fMask:=SIF_ALL;
    GetScrollInfo(Handle,SB_VERT,ScrollInfo);
    if sdUp in FScrollDirection then
    begin
      Perform(WM_VSCROLL,SB_LINEUP,0);
      if ScrollInfo.nPos = nMin then Exclude(FScrollDirection,sdUp);
    end;
    if sdDown in FScrollDirection then
    begin
      Perform(WM_VSCROLL,SB_LINEDOWN,0);
      if ScrollInfo.nPos = nMax then Exclude(FScrollDirection,sdDown);
    end;
    GetScrollInfo(Handle,SB_HORZ,ScrollInfo);
    if sdLeft in FScrollDirection then
    begin
      Perform(WM_HSCROLL,SB_LINELEFT,0);
      if ScrollInfo.nPos = nMin then Exclude(FScrollDirection,sdleft);
    end;
    if sdRight in FScrollDirection then
    begin
      Perform(WM_HSCROLL,SB_LINERIGHT,0);
      if ScrollInfo.nPos = nMax then Exclude(FScrollDirection,sdRight);
    end;
  end;
  if Dragging and assigned(FDragObject) then FDragObject.ShowDragImage;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.EditWndProc(var Message: TMessage);

begin
  try
    with Message do
    begin
      case Msg of
        WM_KEYDOWN,
        WM_SYSKEYDOWN  : if DoKeyDown(TWMKey(Message)) then Exit;
        WM_CHAR        : if DoKeyPress(TWMKey(Message)) then Exit;
        WM_KEYUP,
        WM_SYSKEYUP    : if DoKeyUp(TWMKey(Message)) then Exit;
        CN_KEYDOWN,
        CN_CHAR,
        CN_SYSKEYDOWN,
        CN_SYSCHAR     : begin
                           WndProc(Message);
                           Exit;
                         end;
      end;
      Result:=CallWindowProc(FDefEditProc, FEditHandle, Msg, WParam, LParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.FontChanged(Sender: TObject);

begin
  Perform(WM_SETFONT,Font.Handle,MakeLParam(1,0));
  TreeView_SetTextColor(Handle,ColorToRGB(Font.Color));
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.ActivateScrollTimer;

begin
  FScrollCount:=0;
  FScrollTimer.Interval:=100;
  FScrollTimer.Enabled:=True;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CMColorChanged(var Message: TMessage);

begin
  inherited;
  RecreateWnd;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CMCtl3DChanged(var Message: TMessage);

begin
  inherited;
  if FBorderStyle = bsSingle then RecreateWnd;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.AlphaSort: Boolean;

begin
  if HandleAllocated then Result:=CustomSort(nil,0)
                     else Result:=False;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;

var SortCB : TTVSortCB;
    I      : Integer;
    Node   : TTreeNTNode;

begin
  Result:=False;
  if HandleAllocated then
  begin
    with SortCB do
    begin
      if not assigned(SortProc) then lpfnCompare:=@DefaultTreeNTSort
                                else lpfnCompare:=SortProc;
      hParent:=TVI_ROOT;
      lParam:=Data;
      Result:=TreeView_SortChildrenCB(Handle,SortCB,0);
    end;

    {------------------------------------------
    GW 5-7-96 Should be able to replace with the following
      Note: for some reason above algorithm sorts every
      node individually. Why not use TreeView_SortChildren
      with recurse = True?
      Anyhow, version below at least avoids slow
      Items[] (TTreeNTNodes.GetNodeFromIndex)
    -------------------------------------------}
    If Items.Count > 0 then
    begin
      Node:=Items[0];
      While Node <> nil do
      begin
        if Node.Count > 0 then Node.CustomSort(SortProc,Data);
        Node:=Node.GetNext;
      end;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetSortType(Value: TSortType);

begin
  if SortType <> Value then
  begin
    FSortType:=Value;
    if ((SortType in [stData, stBoth]) and assigned(OnCompare)) or
       (SortType in [stText, stBoth]) then
      AlphaSort;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetStyle(Value: Integer; UseStyle: Boolean);

var Style : Integer;

begin
  if HandleAllocated then
  begin
    Style:=GetWindowLong(Handle,GWL_STYLE);
    if not UseStyle then Style:=Style and not Value
                    else Style:=Style or Value;
    SetWindowLong(Handle,GWL_STYLE, Style);
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetBorderStyle(Value: TBorderStyle);

begin
  if BorderStyle <> Value then
  begin
    FBorderStyle:=Value;
    RecreateWnd;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetDragMode(Value: TDragMode);

begin
  if Value <> DragMode then SetStyle(TVS_DISABLEDRAGDROP,Value = dmManual);
  inherited;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetItemHeight(Value: ShortInt);

begin
  if (Value < 1) and (Value <> -1) then Value:=-1;
  TreeView_SetItemHeight(Handle,Value);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetOptions(Values: TTreeOptions);

var ToBeSet,
    ToBeCleared : TTreeOptions;

begin
  ToBeSet:=Values-FOptions;
  ToBeCleared:=FOptions-Values;

  if toHideSelection in ToBeSet then SetStyle(TVS_SHOWSELALWAYS,False);
  if toInfoTip in ToBeSet then SetStyle(TVS_INFOTIP,True);
  if toShowLines in ToBeSet then SetStyle(TVS_HASLINES,True);
  if toShowRoot in ToBeSet then SetStyle(TVS_LINESATROOT,True);
  if toShowButtons in ToBeSet then SetStyle(TVS_HASBUTTONS,True);
  if toSingleExpand in ToBeSet then SetStyle(TVS_SINGLEEXPAND,True);
  ToBeSet:=ToBeSet-[toShowLines,toShowRoot,toShowButtons,toHideSelection,
                    toInfoTip,toWantReturn];

  if toHideSelection in ToBeCleared then SetStyle(TVS_SHOWSELALWAYS,True);
  if toInfoTip in ToBeSet then SetStyle(TVS_INFOTIP,False);
  if toShowLines in ToBeCleared then SetStyle(TVS_HASLINES,False);
  if toShowRoot in ToBeCleared then SetStyle(TVS_LINESATROOT,False);
  if toShowButtons in ToBeCleared then SetStyle(TVS_HASBUTTONS,False);
  if toSingleExpand in ToBeCleared then SetStyle(TVS_SINGLEEXPAND,False);
  ToBeCleared:=ToBeCleared-[toShowLines,toShowRoot,toShowButtons,toHideSelection,
                            toInfoTip,toWantReturn];

  FOptions:=Values;
  if (ToBeSet+ToBeCleared) <> [] then RecreateWnd;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetNodeAt(X, Y: Integer): TTreeNTNode;

var HitTest : TTVHitTestInfo;

begin
  with HitTest do
  begin
    pt.X:=X;
    pt.Y:=Y;
    if TreeView_HitTest(Handle,HitTest) <> nil then Result:=Items.GetNode(HitTest.hItem)
                                               else Result:=nil;
  end;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetHitTestInfoAt(X, Y: Integer): THitTests;

var HitTest : TTVHitTestInfo;

begin
  Result:=[];
  with HitTest do
  begin
    pt.X:=X;
    pt.Y:=Y;
    TreeView_HitTest(Handle,HitTest);
    if (flags and TVHT_ABOVE) <> 0 then Include(Result,htAbove);
    if (flags and TVHT_BELOW) <> 0 then Include(Result,htBelow);
    if (flags and TVHT_NOWHERE) <> 0 then Include(Result,htNowhere);
    if (flags and TVHT_ONITEM) <> 0 then Include(Result,htOnItem);
    if (flags and TVHT_ONITEMBUTTON) <> 0 then Include(Result,htOnButton);
    if (flags and TVHT_ONITEMICON) <> 0 then Include(Result,htOnIcon);
    if (flags and TVHT_ONITEMINDENT) <> 0 then Include(Result,htOnIndent);
    if (flags and TVHT_ONITEMLABEL) <> 0 then Include(Result,htOnLabel);
    if (flags and TVHT_ONITEMRIGHT) <> 0 then Include(Result,htOnRight);
    if (flags and TVHT_ONITEMSTATEICON) <> 0 then Include(Result,htOnStateIcon);
    if (flags and TVHT_TOLEFT) <> 0 then Include(Result,htToLeft);
    if (flags and TVHT_TORIGHT) <> 0 then Include(Result,htToRight);
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetTreeNTNodes(Value: TTreeNTNodes);

begin
  Items.Assign(Value);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetIndent(Value: Integer);

begin
  if Value <> Indent then TreeView_SetIndent(Handle,Value);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetIndent: Integer;

begin
  Result:=TreeView_GetIndent(Handle)
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.FullExpand;

var Node : TTreeNTNode;

begin
  Node:=Items.GetFirstNode;
  while Node <> nil do
  begin
    Node.Expand(True);
    Node:=Node.GetNextSibling;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.FullCollapse;

var Node : TTreeNTNode;

begin
  Node:=Items.GetFirstNode;
  while Node <> nil do
  begin
    Node.Collapse(True);
    Node:=Node.GetNextSibling;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Loaded;

begin
  inherited Loaded;
  if (csDesigning in ComponentState) and
     (FTreeNTNodes.Count > 0)        then FTreeNTNodes.GetFirstNode.Expand(False);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetTopItem: TTreeNTNode;

begin
  if HandleAllocated then Result:=Items.GetNode(TreeView_GetFirstVisible(Handle))
                     else Result:=nil;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetTopItem(Value: TTreeNTNode);

begin
  if HandleAllocated and (Value <> nil) then TreeView_SelectSetFirstVisible(Handle,Value.ItemId);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetScrollTime: Integer;

begin
  FScrollTime:=Treeview_GetScrollTime(Handle);
  Result:=FScrollTime;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetSearchString: String;

var AString : array[0..1024] of Char;
    Count   : Integer;

begin
  Count:=Integer(TreeView_GetISearchString(Handle,AString));
  if Count > 0 then Result:=AString
               else Result:='';
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetSelection: TTreeNTNode;

begin
  if HandleAllocated then Result:=Items.GetNode(TreeView_GetSelection(Handle))
                     else Result:=nil;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetScrollTime(Value: Integer);

begin
  if Value <> FScrollTime then
  begin
    FScrollTime:=Value;
    if FScrollTime < 100 then FScrollTime:=100;
    Treeview_SetScrollTime(Handle,FScrollTime);
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetSelection(Value: TTreeNTNode);

begin
  if assigned(Value) then Value.Selected:=True
                     else TreeView_SelectItem(Handle,nil);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetDropTarget: TTreeNTNode;

begin
  if HandleAllocated then
  begin
    Result:=Items.GetNode(TreeView_GetDropHilite(Handle));
    if Result = nil then Result:=FLastDropTarget;
  end
  else Result:=nil;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetDropTarget(Value: TTreeNTNode);

begin
  if HandleAllocated then if Value <> nil then Value.DropTarget:=True
                     else TreeView_SelectDropTarget(Handle, nil);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetItemHeight: ShortInt;

begin
  Result:=TreeView_GetItemHeight(Handle);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetNodeFromItem(const Item: TTVItem): TTreeNTNode;

begin
  with Item do
    if (State and TVIF_PARAM) <> 0 then Result:=Pointer(lParam)
                                   else Result:=Items.GetNode(hItem);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.IsEditing: Boolean;

begin
  Result:=TreeView_GetEditControl(Handle) <> 0;
end;

//------------------------------------------------------------------------------

function MakeNodeState(ItemStates: UINT): TNodeStates;

begin
  Result:=[];
  if (ItemStates and CDIS_SELECTED) <> 0 then Include(Result,nsSelected);
  if (ItemStates and CDIS_CHECKED) <> 0 then Include(Result,nsChecked);
  if (ItemStates and CDIS_FOCUS) <> 0 then Include(Result,nsFocused);
  if (ItemStates and CDIS_DISABLED) <> 0 then Include(Result,nsDisabled);
  if (ItemStates and CDIS_GRAYED) <> 0 then Include(Result,nsGrayed);
  if (ItemStates and CDIS_HOT) <> 0 then Include(Result,nsHot);
  if (ItemStates and CDIS_INDETERMINATE) <> 0 then Include(Result,nsIndeterminate);
  if (ItemStates and CDIS_MARKED) <> 0 then Include(Result,nsMarked);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CNNotify(var Message: TWMNotify);

var Node       : TTreeNTNode;
    NodeStates : TNodeStates;
    NewText    : String;
    Allowed    : Boolean;

begin
  with Message.NMHdr^ do
  case code of
    TVN_BEGINDRAG:
      begin
        FDragged:=True;
        with PNMTreeView(Pointer(Message.NMHdr))^ do
          FDragNode:=GetNodeFromItem(ItemNew);
      end;
    TVN_BEGINLABELEDIT:
      begin
        with PTVDispInfo(Pointer(Message.NMHdr))^ do
          if Dragging or not CanEdit(GetNodeFromItem(Item)) then Message.Result:=1;
        if Message.Result = 0 then
        begin
          FEditHandle:=TreeView_GetEditControl(Handle);
          FDefEditProc:=Pointer(GetWindowLong(FEditHandle,GWL_WNDPROC));
          SetWindowLong(FEditHandle,GWL_WNDPROC,LongInt(FEditInstance));
        end;
      end;
    TVN_ENDLABELEDIT:
      with PTVDispInfo(Pointer(Message.NMHdr))^ do Edit(Item);
    TVN_ITEMEXPANDING:
      with PNMTreeView(Pointer(Message.NMHdr))^ do
      begin
        Node:=GetNodeFromItem(ItemNew);
        if (action = TVE_EXPAND) {and (Node.FProgramExpand = 0)} then
          if not CanExpand(Node) then Message.Result:=1;

        if (action = TVE_COLLAPSE) {and (Node.FProgramCollapse = 0)} then
          if not CanCollapse(Node) then Message.Result:=1;
      end;
    TVN_ITEMEXPANDED:
      with PNMTreeView(Pointer(Message.NMHdr))^ do
      begin
        Node:=GetNodeFromItem(ItemNew);
        if (action = TVE_EXPAND) {and (Node.FProgramExpand = 0)}
          then Expand(Node); // else do it later in ExpandItem

        if (action = TVE_COLLAPSE) {and (Node.FProgramCollapse = 0)}
          then Collapse(Node); // else do it later in ExpandItem 
      end;
    TVN_SELCHANGING:
      with PNMTreeView(Pointer(Message.NMHdr))^ do
        if not CanChange(GetNodeFromItem(ItemNew)) then Message.Result:=1;
    TVN_SELCHANGED:
      with PNMTreeView(Pointer(Message.NMHdr))^ do Change(GetNodeFromItem(ItemNew));
    TVN_DELETEITEM:
      if not (csDestroying in ComponentState) then
      begin
        with PNMTreeView(Pointer(Message.NMHdr))^ do Node:=GetNodeFromItem(ItemOld);
        if Node <> nil then
        begin
          Node.FItemId:=nil;
          Items.Delete(Node);
        end;
      end;
    TVN_SETDISPINFO:
      with PTVDispInfo(Pointer(Message.NMHdr))^ do
      begin
        Node:=GetNodeFromItem(Item);
        if (Node <> nil) and ((Item.Mask and TVIF_TEXT) <> 0) then Node.Text:=Item.pszText;
      end;
    TVN_GETDISPINFO:
      with PTVDispInfo(Pointer(Message.NMHdr))^ do
      begin
        Node:=GetNodeFromItem(Item);
        if Node <> nil then
        begin
          if (Item.Mask and TVIF_TEXT) <> 0 then
            StrLCopy(Item.pszText, PChar(Node.Text),Item.cchTextMax);
          if (Item.Mask and TVIF_IMAGE) <> 0 then
          begin
            GetImageIndex(Node);
            Item.iImage:=Node.ImageIndex;
          end;
          if (Item.Mask and TVIF_SELECTEDIMAGE) <> 0 then
          begin
            GetSelectedIndex(Node);
            Item.iSelectedImage:=Node.SelectedIndex;
          end;
        end;
      end;
    TVN_SINGLEEXPAND :
      with PNMTreeView(Pointer(Message.NMHdr))^ do
      begin
        Node:=GetNodeFromItem(ItemNew);
        Allowed:=True;
        if assigned(FOnSingleExpanded) then FOnSingleExpanded(Self,Node,Allowed);
        if not Allowed then Message.Result:=1;
      end;
    TVN_GETINFOTIP   :
      if not (csDesigning in ComponentState) and assigned(FOnHint) then
        with PNMTVGetInfoTip(Pointer(Message.NMHdr))^ do
        begin
          Node:=TTreeNTNode(lParam);
          NewText:=FOnHint(Self,Node);
          StrPLCopy(pszText,NewText,cchTextMax-1);
        end;
    NM_CUSTOMDRAW:
      with PNMTVCustomDraw(Pointer(Message.NMHdr))^ do
      case nmcd.dwDrawStage of
        CDDS_PREPAINT     : begin
                              FTreeCanvas.Handle:=nmcd.hdc;
                              SetTextColor(nmcd.hdc,ColorToRGB(Font.Color));
                              if assigned(BeforePaint) then FBeforePaint(Self,FTreeCanvas);
                              Message.Result:=CDRF_NOTIFYPOSTPAINT or
                                              CDRF_NOTIFYITEMDRAW or
                                              CDRF_NOTIFYITEMERASE or
                                              CDRF_NOTIFYPOSTERASE;
                            end;
        CDDS_ITEMPREPAINT : begin
                              Message.Result:=CDRF_NOTIFYPOSTPAINT or CDRF_NOTIFYITEMERASE;
                              Node:=TTreeNTNode(nmcd.lItemlParam);
                              NodeStates:=MakeNodeState(nmcd.uItemState);
                              if assigned(FBeforeItemPaint) then
                                FBeforeItemPaint(Self,Node,FTreeCanvas,nmcd.rc,NodeStates);
                              if not (nsSelected in NodeStates) then
                                if Node.ParentColor then SetBkColor(FTreeCanvas.Handle,ColorToRGB(Color))
                                                    else SetBkColor(FTreeCanvas.Handle,ColorToRGB(Node.Color));
                              if not Node.ParentFont and assigned(Node.FFont) THEN
                              begin
                                SelectObject(nmcd.hdc,Node.Font.Handle);
                                SetTextColor(nmcd.hdc,ColorToRGB(Node.Font.Color));
                                Message.Result:=Message.Result or CDRF_NEWFONT;
                              end;
                            end;
        CDDS_ITEMPREERASE  : Message.Result:=CDRF_NOTIFYPOSTERASE; // not yet supported
        CDDS_ITEMPOSTERASE : Message.Result:=CDRF_DODEFAULT; // not yet supported
        CDDS_ITEMPOSTPAINT : begin
                               Node:=TTreeNTNode(nmcd.lItemlParam);
                               if assigned(FAfterItemPaint) then
                               begin
                                 NodeStates:=MakeNodeState(nmcd.uItemState);
                                 FAfterItemPaint(Self,Node,FTreeCanvas,nmcd.rc,NodeStates);
                               end;
                               Message.Result:=CDRF_DODEFAULT;
                             end;
        CDDS_POSTPAINT : begin
                           if assigned(FAfterPaint) then FAfterPaint(Self,FTreeCanvas);
                           Message.Result:=CDRF_DODEFAULT;
                         end;
        CDDS_PREERASE  : Message.Result:=CDRF_DODEFAULT; // not yet supported
        CDDS_POSTERASE : Message.Result:=CDRF_DODEFAULT; // not yet supported
      else Message.Result:=CDRF_DODEFAULT;
      end;
    NM_RCLICK: FRClicked:=True;
    NM_RETURN: if toWantReturn in FOptions then
               begin
                 if assigned(Selected) then
                 begin
                   Node:=Selected.GetNext;
                   if Node = nil then Node:=Items.GetFirstNode;
                 end
                 else Node:=Items.GetFirstNode;     
                 if assigned(Node) then Node.Selected:=True;
                 Message.Result:=1;
               end;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CMDesignHitTest(var Message: TCMDesignHitTest);

var HitInfo : THitTests;

begin
  HitInfo:=GetHitTestInfoAt(Message.XPos,Message.YPos);
  if (htOnItem in HitInfo) or (htOnButton in HitInfo)
    then Message.Result:=1
    else Message.Result:=0;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CMMouseLeave(var Message: TMessage);

begin
  FScrollTimer.Enabled:=False;
  inherited;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.GetDragImages: TCustomImageList;

begin
  if FDragImage.Count > 0 then Result:=FDragImage
                          else Result:=nil;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.WndProc(var Message: TMessage);

begin
  if Message.Msg = CM_DRAG then Dispatch(Message)
                           else
    if not (csDesigning in ComponentState) and ((Message.Msg = WM_LBUTTONDOWN) or
      (Message.Msg = WM_LBUTTONDBLCLK)) and not Dragging and (DragMode = dmAutomatic) then
    begin
      if not IsControlMouseMsg(TWMMouse(Message)) then
      begin
        ControlState:=ControlState+[csLButtonDown];
        Dispatch(Message);
      end;
    end
    else inherited WndProc(Message);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DoStartDrag(var DragObject: TDragObject);

var ImageHandle : HImageList;
    DragNode    : TTreeNTNode;
    P           : TPoint;

begin
  inherited DoStartDrag(DragObject);
  DragNode:=FDragNode;
  FLastDropTarget:=nil;
  FDragNode:=nil;
  if DragNode = nil then
  begin
    GetCursorPos(P);
    with ScreenToClient(P) do DragNode:=GetNodeAt(X, Y);
  end;
  if DragNode <> nil then
  begin
    ImageHandle:=TreeView_CreateDragImage(Handle,DragNode.ItemId);
    if ImageHandle <> 0 then
      with FDragImage do
      begin
        Handle:=ImageHandle;
        SetDragImage(0,2,2);
      end;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DoAutoScroll(X,Y: Integer);

var DefOffset  : Integer;
    VSInfo,
    HSInfo     : TScrollInfo;

begin
  // prepare structures
  FillChar(HSInfo,SizeOf(HSInfo),0);
  HSInfo.cbSize:=SizeOf(TScrollInfo);
  HSInfo.fMask:=SIF_ALL;
  VSInfo:=HSInfo;

  if ItemHeight > 0 then DefOffset:=ItemHeight div 2 // v 4.71+
                    else DefOffset:=Abs(Font.Size); // v 4.70
  FScrollDirection:=[];

  GetScrollInfo(Handle,SB_VERT,VSInfo);
  GetScrollInfo(Handle,SB_HORZ,HSInfo);

  with HSInfo do
  begin
    if (X < DefOffset) and (nPos > nMin) then Include(FScrollDirection,sdLeft);
    if (X > ClientWidth-DefOffset) and (nPos < nMax) then Include(FScrollDirection,sdRight);
  end;

  with VSInfo do
  begin
    if (Y < DefOffset) and (nPos > nMin) then Include(FScrollDirection,sdUp);
    if (Y > ClientHeight-DefOffset) and (nPos < nMax) then Include(FScrollDirection,sdDown);
  end;

  if FScrollDirection <> [] then ActivateScrollTimer
                            else FScrollTimer.Enabled:=False;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DoEndDrag(Target: TObject; X, Y: Integer);

begin
  inherited DoEndDrag(Target,X,Y);
  FLastDropTarget:=nil;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.CMDrag(var Message: TCMDrag);

begin
  inherited;
  if Message.Result <> 0 then
    with Message,DragRec^ do
      case DragMessage of
        dmDragMove  : with ScreenToClient(Pos) do DoDragOver(Source,X,Y);
        dmDragLeave : begin
                        TDragObject(Source).HideDragImage;
                        FLastDropTarget:=DropTarget;
                        DropTarget:=nil;
                        TDragObject(Source).ShowDragImage;
                      end;
        dmDragDrop : FLastDropTarget:=nil;
      end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.DoDragOver(Source: TDragObject; X, Y: Integer);

var Node : TTreeNTNode;

begin
  FDragObject:=Source;
  if FAutoScroll then DoAutoScroll(X,Y);
  Node:=GetNodeAt(X,Y);
  if (Node <> nil) and
    ((Node <> DropTarget) or (Node = FLastDropTarget)) then
  begin
    FLastDropTarget:=nil;
    TDragObject(Source).HideDragImage;
    Node.DropTarget:=True;
    TDragObject(Source).ShowDragImage;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.GetImageIndex(Node: TTreeNTNode);

begin
  if assigned(FOnGetImageIndex) then FOnGetImageIndex(Self,Node);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.GetSelectedIndex(Node: TTreeNTNode);

begin
  if assigned(FOnGetSelectedIndex) then FOnGetSelectedIndex(Self,Node);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CanChange(Node: TTreeNTNode): Boolean;

begin
  Result:=True;
  if assigned(FOnChanging) then FOnChanging(Self,Node,Result);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Change(Node: TTreeNTNode);

begin
  if assigned(FOnChange) then FOnChange(Self,Node);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Expand(Node: TTreeNTNode);

begin
  if assigned(FOnExpanded) then FOnExpanded(Self,Node);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CanExpand(Node: TTreeNTNode): Boolean;

begin
  Result:=True;
  if assigned(FOnExpanding) then FOnExpanding(Self,Node,Result);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Collapse(Node: TTreeNTNode);

begin
  if assigned(FOnCollapsed) then FOnCollapsed(Self,Node);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CanCollapse(Node: TTreeNTNode): Boolean;

begin
  Result:=True;
  if assigned(FOnCollapsing) then FOnCollapsing(Self,Node,Result);
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CanEdit(Node: TTreeNTNode): Boolean;

begin
  Result:=True;
  if assigned(FOnEditing) then FOnEditing(Self,Node,Result);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Edit(const Item: TTVItem);

var S    : String;
    Node : TTreeNTNode;

begin
  with Item do
    if pszText <> nil then
    begin
      S:=pszText;
      Node:=GetNodeFromItem(Item);
      if assigned(FOnEdited) then FOnEdited(Self,Node,S);
      if Node <> nil then Node.Text:=S;
      if assigned(Designer) then Designer.Modified;
    end;
end;

//------------------------------------------------------------------------------

function TCustomTreeNT.CreateNode: TTreeNTNode;

begin
  Result:=TTreeNTNode.Create(Items);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetImageList(Value: HImageList; Flags: Integer);

begin
  if HandleAllocated then TreeView_SetImageList(Handle,Value,Flags);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.ImageListChange(Sender: TObject);

var ImageHandle : HImageList;

begin
  if HandleAllocated then
  begin
    ImageHandle:=TImageList(Sender).Handle;
    if Sender = Images then SetImageList(ImageHandle,TVSIL_NORMAL)
                       else
      if Sender = StateImages then SetImageList(ImageHandle,TVSIL_STATE);
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.MouseMove(Shift: TShiftState; X, Y: Integer);

begin
  if FAutoScroll then DoAutoScroll(X,Y);
  inherited MouseMove(Shift,X,Y);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.Notification(AComponent: TComponent; Operation: TOperation);

begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if AComponent = Images then Images:=nil;
    if AComponent = StateImages then StateImages:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetImages(Value: TImageList);

begin
  if Images <> nil then Images.UnRegisterChanges(FImageChangeLink);
  FImages:=Value;
  if Images <> nil then
  begin
    Images.RegisterChanges(FImageChangeLink);
    SetImageList(Images.Handle,TVSIL_NORMAL)
  end
  else SetImageList(0,TVSIL_NORMAL);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SetStateImages(Value: TImageList);

begin
  if StateImages <> nil then StateImages.UnRegisterChanges(FStateChangeLink);
  FStateImages:=Value;
  if StateImages <> nil then
  begin
    StateImages.RegisterChanges(FStateChangeLink);
    SetImageList(StateImages.Handle,TVSIL_STATE)
  end
  else
  begin
    SetImageList(0,TVSIL_STATE);
    // workaround for checkboxes
    if toCheckBoxes in FOptions then RecreateWnd;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.LoadFromFile(const FileName: String);

var Stream : TStream;

begin
  Stream:=TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.LoadFromStream(Stream: TStream);

begin
  with TTreeNTStrings.Create(Items) do
    try
      LoadFromStream(Stream);
    finally
      Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SaveToFile(const FileName: String);

var Stream : TStream;

begin
  Stream:=TFileStream.Create(FileName,fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.SaveToStream(Stream: TStream);

begin
  with TTreeNTStrings.Create(Items) do
    try
      SaveToStream(Stream);
    finally
      Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.ShowInsertMark(Node: TTreeNTNode; After: Boolean);

begin
  if assigned(Node) then TreeView_SetInsertmark(Handle,Node.ItemID,After)
                    else TreeView_SetInsertmark(Handle,HTREEITEM(0),After);
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.WMRButtonDown(var Message: TWMRButtonDown);

var MousePos : TPoint;

begin
  FRClicked:=False;
  inherited;
  if FRClicked then
  begin
    GetCursorPos(MousePos);
    with PointToSmallPoint(ScreenToClient(MousePos)) do
      Perform(WM_RBUTTONUP, 0, MakeLong(X, Y));
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.WMLButtonDown(var Message: TWMLButtonDown);

var Node     : TTreeNTNode;
    MousePos : TPoint;
    Allowed  : Boolean;

begin
  FDragged:=False;
  FDragNode:=nil;
  try
    inherited;
    if DragMode = dmAutomatic then
    begin
      SetFocus;
      if not FDragged then
      begin
        GetCursorPos(MousePos);
        with PointToSmallPoint(ScreenToClient(MousePos)) do
          Perform(WM_LBUTTONUP,0,MakeLong(X,Y));
      end
      else
      begin
        Node:=GetNodeAt(Message.XPos,Message.YPos);
        if FDesignerMode and
           assigned(FDesigner) and
           assigned(Node) and
           assigned(Node.Data) and
           (TObject(Node.Data) is TComponent) then
        begin
          Allowed:=True;
          if assigned(FOnDesignClick) then FOnDesignClick(Self,Node,Allowed);
          if allowed then FDesigner.SelectComponent(TComponent(Node.Data));
        end;
        if Node <> nil then BeginDrag(False);
      end;
    end
    else
    begin
      Node:=GetNodeAt(Message.XPos,Message.YPos);
      if Node <> nil then Node.Selected:=True;
    end;
  finally
    FDragNode:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure TCustomTreeNT.WMNCHitTest(var Message: TWMNCHitTest);

begin
  DefaultHandler(Message);
end;

//----------------- TTreeNTNodesProperty ---------------------------------------

type
  TTreeNTNodesProperty = class(TClassProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

//------------------------------------------------------------------------------

procedure TTreeNTNodesProperty.Edit;

begin
  if EditTreeViewItems(TTreeNTNodes(GetOrdValue)) then Modified;
end;

//------------------------------------------------------------------------------

function TTreeNTNodesProperty.GetAttributes: TPropertyAttributes;

begin
  Result:=[paDialog,paReadOnly];
end;

//------------------------------------------------------------------------------

procedure Register;

begin
  RegisterComponents('Freeware',[TTreeNT]);
  RegisterPropertyEditor(TypeInfo(TTreeNTNodes),nil,'',TTreeNTNodesProperty);
end;

//------------------------------------------------------------------------------

end.
