{Uncomment this $DEFINE to produce the shareware version.}
{You must also uncomment the line in PgPrnAbt.}
//{$DEFINE PAGEPRINT_SHAREWARE}
(*******************************************************************************

TPagePrinter Version 2.0
8/8/96 - 10/26/97
Copyright  1996-1997 Bill Menees
bmenees@usit.net
http://www.public.usit.net/bmenees

This is a VCL component that encapsulates the Printer object and does
Print Preview.  I make no claim to it's correct functioning, so use
it at your own risk.

It **REQUIRES** long strings, enhanced metafiles, the Win32 common
controls, and it makes use of several Win32 specific API calls.
This means it can't be used with Delphi 1.0, so please don't ask, beg,
threaten, etc.  It has been tested with and seems to work fine with
Delphi 2.0, Delphi 3.0, and C++Builder 1.0.

Before you e-mail me with a question, MAKE SURE YOU CHECK THE SOURCE CODE FIRST!
I don't mind helping people with problems if they have honestly tried to
solve the problem first.  However, I won't even reply to questions whose
answers are obvious when you look at the source (e.g. Can I use this with
Delphi 1.0?).

Historical Note: This component has its origins in TLinePrinter.
I started off calling this component TLinePrinter Version 2.0, but
I decided a new class name was more appropriate for several reasons.  The
main reason was that TLinePrinter is a non-visual component, and the new
component is a visual component.  I didn't want the new visual component to
start showing up on forms where the V.1.0 component hadn't shown!  A new
class name also gave me the chance to redefine the interface entirely.
I added, edited, renamed, and deleted many properties, methods, events,
and units.  I think you'll agree the changes are for the better.

*******************************************************************************)

{$LONGSTRINGS ON}
unit PagePrnt;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Printers, ExtCtrls,
{$IFDEF PAGEPRINT_SHAREWARE}
  PgPrnPrg, PgPrnAbt;
{$ELSE}
  PgPrnPrg;
{$ENDIF}

const
  {In Pixels}
  DefaultBorderWidth = 2;
  DefaultDPI = 300;
  {In Inches}
  DefaultAvailablePageHeightIn = 10.5;
  DefaultAvailablePageWidthIn = 8.0;
  DefaultGutterLeftIn = 0.25;
  DefaultGutterTopIn = 0.25;
  DefaultPhysicalPageHeightIn = 11.0;
  DefaultPhysicalPageWidthIn = 8.5;
  {In Millimeters}
  DefaultAvailablePageHeightMm = 284.0;
  DefaultAvailablePageWidthMm = 198.0;
  DefaultGutterLeftMm = 6.0;
  DefaultGutterTopMm = 6.0;
  DefaultPhysicalPageHeightMm = 297.0;
  DefaultPhysicalPageWidthMm = 210.0;
  {These are expanded only in Headers, Footers, and Tables.}
  DateField = '{$DATE}';
  LineField = '{$LINE}';
  PageField = '{$PAGE}';
  TimeField = '{$TIME}';
  TitleField = '{$TITLE}';
  {Progress Dialog Messages}
  ProgressFinishMsg = '<FINISH>';
  SendingPagesMsg = 'Sending Pages To Printer';

type
  EPagePrinter = class(EPrinter);
  TGradientOrientation = (goHorizontal, goVertical);
  TLineSpacing = (lsHalfSpace, lsSingleSpace, lsSingleAndAHalf, lsDoubleSpace);
  TMeasurement = Double;
  TMeasureUnit = (muInches, muMillimeters);
  TPageBorder = (pbTop, pbBottom, pbLeft, pbRight);
  TPageBorders = set of TPageBorder;
  TPixels = Cardinal;
  TPrintCanvas = TMetafileCanvas;
  TPrintPage = TMetafile;
  TZoomLocation = (zlTopLeft, zlTopCenter, zlCenter);

  TPageList = class(TList)
  public
        destructor Destroy; override;
        function GetPage(const Index: Integer): TPrintPage;
  end;

  TPagePrinter = class(TScrollBox)
  private
    { Private declarations }
    fAbortOnCancel: Boolean;
    fAlignment: TAlignment;
    fAutoFooterFont: Boolean;
    fAutoHeaderFont: Boolean;
    fCancelPrinting: Boolean;
    fCanvas: TPrintCanvas;
    fCollate: Boolean;
    fCopies: Cardinal;
    {These X,Y are relative to the printable space.
     They should normally be bounded by the Margins.
     So 0,0 is the left,top corner of the printable space.
     fCurrentY is negative only when printing the header.}
    fCurrentX, fCurrentY: Integer;
    fDefaultColWidth: TMeasurement;
    fFileName: String;
    fFileVar: TextFile;
    fFooterFont: TFont;
    fFriendlyFooter: String;
    fFriendlyHeader: String;
    fGradientBackground: Boolean;
    fHeader, fFooter: String;
    fHeaderFont: TFont;
    fHeaderFormat, fFooterFormat: String;
    fLineNumber: Cardinal;
    fLines: TStrings;
    fLineSpace: TPixels;
    fLineSpacing: TLineSpacing;
    fMarginBottom: TMeasurement;
    fMarginLeft: TMeasurement;
    fMarginRight: TMeasurement;
    fMarginTop: TMeasurement;
    fMeasureUnit: TMeasureUnit;
    fOnNewLine: TNotifyEvent;
    fOnNewPage: TNotifyEvent;
    fPage: TPrintPage;
    fPageBorderOffset: TMeasurement;
    fPageBorders: TPageBorders;
    fPageNumber: Cardinal;
    fPages: TPageList;
    fPaintBox: TPaintBox;
    fPPPrnPrgDlg: TPPPrnPrgDlg;
    fPrinter: TPrinter;
    fPrintFromPage: Cardinal;
    fPrinting: Boolean;
    fPrintingHeaderOrFooter: Boolean;
    fPrintingToFile: Boolean;
    fPrintToFile: Boolean;
    fPrintToPage: Cardinal;
    fProgressMessage: String;
    fShadowColor: TColor;
    fShadowOffset: TPixels;
    fShowCancel: Boolean;
    fShowMargins: Boolean;
    fShowProgress: Boolean;
    fStillCreating: Boolean;
    fTableFormat: String;
    fTableGrid: Boolean;
    fTabSize: Cardinal;
    fTextMetrics: TTextMetric;
    fTokenSeparator: Char;
    fUpdateRefCount: Cardinal;
    fUsingTempPage: Boolean;
    fWordWrap: Boolean;
    fZoomLocation: TZoomLocation;
    fZoomPercent: Cardinal;

    function GetAutoFooterFont: Boolean;
    function GetAutoHeaderFont: Boolean;
    function GetAvailablePageHeight: TMeasurement;
    function GetAvailablePageWidth: TMeasurement;
    function GetCanvas: TPrintCanvas;
    function GetCanvasPosition: TPoint;
    function GetCollate: Boolean;
    function GetCopies: Cardinal;
    function GetDefaultColWidth: TMeasurement;
    function GetFileName: String;
    function GetFooterFont: TFont;
    function GetFooterFormat: String;
    function GetFriendlyFooter: String;
    function GetFriendlyHeader: String;
    function GetGradientBackground: Boolean;
    function GetGutterBottom: TMeasurement;
    function GetGutterLeft: TMeasurement;
    function GetGutterRight: TMeasurement;
    function GetGutterTop: TMeasurement;
    function GetHeaderFont: TFont;
    function GetHeaderFormat: String;
    function GetLineNumber: Cardinal;
    function GetLines: TStrings;
    function GetLineSpacing: TLineSpacing;
    function GetMarginBottom: TMeasurement;
    function GetMarginLeft: TMeasurement;
    function GetMarginRight: TMeasurement;
    function GetMarginTop: TMeasurement;
    function GetMeasureUnit: TMeasureUnit;
    function GetOrientation: TPrinterOrientation;
    function GetPageBorderOffset: TMeasurement;
    function GetPageBorders: TPageBorders;
    function GetPageCount: Cardinal;
    function GetPageNumber: Cardinal;
    function GetPages(Indx: Cardinal): TPrintPage;
    function GetPhysicalPageHeight: TMeasurement;
    function GetPhysicalPageWidth: TMeasurement;
    function GetPreviewPagePixelsH: TPixels;
    function GetPreviewPagePixelsV: TPixels;
    function GetPrintableHeight: TMeasurement;
    function GetPrintableWidth: TMeasurement;
    function GetPrintFromPage: Cardinal;
    function GetPrinting: Boolean;
    function GetPrintToFile: Boolean;
    function GetPrintToPage: Cardinal;
    function GetProgressMessage: String;
    function GetShadowColor: TColor;
    function GetShadowOffset: TPixels;
    function GetShowCancel: Boolean;
    function GetShowMargins: Boolean;
    function GetShowProgress: Boolean;
    function GetTableFormat: String;
    function GetTitle: String;
    function GetZoomPercent: Cardinal;
    function PixelPrintHeight: TPixels;
    function PixelPrintWidth: TPixels;
    function StartingBottom: TPixels;
    function StartingLeft: TPixels;
    function StartingRight: TPixels;
    function StartingTop: TPixels;
    function StoreFooterAndFormat: Boolean;
    function StoreFooterFont: Boolean;
    function StoreHeaderAndFormat: Boolean;
    function StoreHeaderFont: Boolean;

    procedure CMFontChanged(var Msg: TMessage); message CM_FONTCHANGED;
    procedure CreateTempPage;
    procedure DoNewPageProcessing;
    procedure FinishPrintPage;
    procedure NewPrintPage;
    procedure OnCancelPrinting(Sender: TObject);
    procedure ResetPageList(CreateForReal: Boolean);
    procedure SetAutoFooterFont(Value: Boolean);
    procedure SetAutoHeaderFont(Value: Boolean);
    procedure SetCollate(Value: Boolean);
    procedure SetCopies(Value: Cardinal);
    procedure SetDefaultColWidth(Value: TMeasurement);
    procedure SetFileName(Value: String);
    procedure SetFooterFont(Value: TFont);
    procedure SetFooterFormat(Value: String);
    procedure SetFriendlyFooter(Value: String);
    procedure SetFriendlyHeader(Value: String);
    procedure SetGradientBackground(Value: Boolean);
    procedure SetHeaderFont(Value: TFont);
    procedure SetHeaderFormat(Value: String);
    procedure SetLines(Value: TStrings);
    procedure SetLineSpacing(Value: TLineSpacing);
    procedure SetMarginBottom(Value: TMeasurement);
    procedure SetMarginLeft(Value: TMeasurement);
    procedure SetMarginRight(Value: TMeasurement);
    procedure SetMarginTop(Value: TMeasurement);
    procedure SetMeasureUnit(Value: TMeasureUnit);
    procedure SetOrientation(Value: TPrinterOrientation);
    procedure SetPageBorderOffset(Value: TMeasurement);
    procedure SetPageBorders(Value: TPageBorders);
    procedure SetPageNumber(Value: Cardinal);
    procedure SetPrintFromPage(Value: Cardinal);
    procedure SetPrintToFile(Value: Boolean);
    procedure SetPrintToPage(Value: Cardinal);
    procedure SetProgressMessage(Value: String);
    procedure SetShadowColor(Value: TColor);
    procedure SetShadowOffset(Value: TPixels);
    procedure SetShowCancel(Value: Boolean);
    procedure SetShowMargins(Value: Boolean);
    procedure SetShowProgress(Value: Boolean);
    procedure SetTableFormat(Value: String);
    procedure SetTitle(Value: String);
    procedure SetZoomPercent(Value: Cardinal);

  protected
    { Protected declarations }
    function ExpandLogicalFields(S: String): String;
    function GetClippedLine(const Line: String; const Width: TPixels): String;
    function GetPreviewPagePixels(Horz: Boolean): TPixels;
    function GetPrinterHandle: HDC;
    function GetScaleFactor(Horz: Boolean): Double;
    function MeasureUnitsToScreenPixels(const Value: TMeasurement; Horz: Boolean): TPixels;
    function ScaleValue(Value: TMeasurement; Horz: Boolean): TPixels;
    function ValidateFormatString(const Fmt: String; const ConvertUnits: Boolean): String;
    procedure ExpandFriendlyFormat(const UserFmt: String; AsHeader: Boolean);
    procedure Invalidate; override;
    procedure Loaded; override;
    procedure PaintPreview(Sender: TObject); virtual; //OnPaint handler for TPaintBox
    procedure ParseFormatToken(var CurToken: String; var CurAlignment: TAlignment; var CurWidth: TMeasurement);
    procedure SetPixelsPerInch;
    procedure SplitLine(var CurLine: String; var Buffer: String; const ClipWidth: TPixels; const TrimLastWhiteSpace: Boolean);
    procedure SplitLineAndPrint(const Line: String; UseWrite: Boolean);
    procedure UpdateDesigner;
    procedure UpdatePagePreviewSize;
    procedure UpdateProgressDlg(const Status: String; const CurrentPage, FromPage, ToPage: Cardinal);
    procedure WriteTableGrid(const CurWidth: TPixels; const TopGrid, BottomGrid: Boolean);

  public
    { Public declarations }
    {Largest printable space on the page.}
    property AvailablePageHeight: TMeasurement read GetAvailablePageHeight;
    property AvailablePageWidth: TMeasurement read GetAvailablePageWidth;
    property Canvas: TPrintCanvas read GetCanvas;
    property CanvasPosition: TPoint read GetCanvasPosition;
    property GutterBottom: TMeasurement read GetGutterBottom;
    property GutterLeft: TMeasurement read GetGutterLeft;
    property GutterRight: TMeasurement read GetGutterRight;
    property GutterTop: TMeasurement read GetGutterTop;
    property LineNumber: Cardinal read GetLineNumber;
    property PageCount: Cardinal read GetPageCount;
    property PageNumber: Cardinal read GetPageNumber write SetPageNumber;
    property Pages[Indx: Cardinal]: TPrintPage read GetPages;
    property PhysicalPageHeight: TMeasurement read GetPhysicalPageHeight;
    property PhysicalPageWidth: TMeasurement read GetPhysicalPageWidth;
    {Printable space bounded by the margins.}
    property PrintableHeight: TMeasurement read GetPrintableHeight;
    property PrintableWidth: TMeasurement read GetPrintableWidth;
    property PrintFromPage: Cardinal read GetPrintFromPage write SetPrintFromPage default 0;
    property Printing: Boolean read GetPrinting;
    property PrintToPage: Cardinal read GetPrintToPage write SetPrintToPage default 0;

    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
    function MeasureUnitsToPixelsH(const M: TMeasurement): TPixels;
    function MeasureUnitsToPixelsV(const M: TMeasurement): TPixels;
    function NewLine: Cardinal;
    function NewPage: Cardinal;
    function PixelsToMeasureUnitsH(const P: TPixels): TMeasurement;
    function PixelsToMeasureUnitsV(const P: TPixels): TMeasurement;
    function PrevLine: Boolean;
    function Print: Boolean;
    procedure BeginDoc;
    procedure BeginUpdate;
    procedure Clear;
    procedure EndDoc;
    procedure EndUpdate;
    procedure RefreshProperties;
    procedure Write(const Line: String);
    procedure WriteLine(const Line: String);
    procedure WriteLineAligned(const AAlignment: TAlignment; const Line: String);
    procedure WriteLines(const LinesAsTable: Boolean);
    procedure WriteTableLine(const Line: String);
    procedure ZoomToFit;
    procedure ZoomToHeight;
    procedure ZoomToWidth;

  published
    { Published declarations }
    {Because everything in TPagePrinter depends on it, this property MUST be the first
    TPagePrinter-specific property loaded.  If you edit the form as text and move it
    around in the streaming order, things may not work correctly.  This dependency kind
    of stinks, but I can't think of any way around it.  Sorry.}
    property MeasureUnit: TMeasureUnit read GetMeasureUnit write SetMeasureUnit default muInches;

    property AbortOnCancel: Boolean read fAbortOnCancel write fAbortOnCancel default False;
    property Align;
    property Alignment: TAlignment read fAlignment write fAlignment default taLeftJustify;
    property AutoFooterFont: Boolean read GetAutoFooterFont write SetAutoFooterFont default True;
    property AutoHeaderFont: Boolean read GetAutoHeaderFont write SetAutoHeaderFont default True;
    property Collate: Boolean read GetCollate write SetCollate default True;
    property Color;
    property Copies: Cardinal read GetCopies write SetCopies default 1;
    property DefaultColWidth: TMeasurement read GetDefaultColWidth write SetDefaultColWidth stored True;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FileName: String read GetFileName write SetFileName;
    property Footer: String read fFooter write fFooter stored StoreFooterAndFormat;
    property FooterFont: TFont read GetFooterFont write SetFooterFont stored StoreFooterFont;
    property FooterFormat: String read GetFooterFormat write SetFooterFormat stored StoreFooterAndFormat;
    property FriendlyFooter: String read GetFriendlyFooter write SetFriendlyFooter;
    property FriendlyHeader: String read GetFriendlyHeader write SetFriendlyHeader;
    property GradientBackground: Boolean read GetGradientBackground write SetGradientBackground default True;
    property Header: String read fHeader write fHeader stored StoreHeaderAndFormat;
    property HeaderFont: TFont read GetHeaderFont write SetHeaderFont stored StoreHeaderFont;
    property HeaderFormat: String read GetHeaderFormat write SetHeaderFormat stored StoreHeaderAndFormat;
    property Lines: TStrings read GetLines write SetLines;
    property LineSpacing: TLineSpacing read GetLineSpacing write SetLineSpacing default lsSingleSpace;
    property MarginBottom: TMeasurement read GetMarginBottom write SetMarginBottom;
    property MarginLeft: TMeasurement read GetMarginLeft write SetMarginLeft;
    property MarginRight: TMeasurement read GetMarginRight write SetMarginRight;
    property MarginTop: TMeasurement read GetMarginTop write SetMarginTop;
    property OnNewLine: TNotifyEvent read fOnNewLine write fOnNewLine;
    property OnNewPage: TNotifyEvent read fOnNewPage write fOnNewPage;
    property OnStartDrag;
    property Orientation: TPrinterOrientation read GetOrientation write SetOrientation default poPortrait;
    property PageBorderOffset: TMeasurement read GetPageBorderOffset write SetPageBorderOffset;
    property PageBorders: TPageBorders read GetPageBorders write SetPageBorders default [];
    property ParentColor;
    property ParentFont default False;
    property PrintToFile: Boolean read GetPrintToFile write SetPrintToFile default False;
    property ProgressMessage: String read GetProgressMessage write SetProgressMessage;
    property ShadowColor: TColor read GetShadowColor write SetShadowColor default clBtnShadow;
    property ShadowOffset: TPixels read GetShadowOffset write SetShadowOffset default 5;
    property ShowHint;
    property ShowCancel: Boolean read GetShowCancel write SetShowCancel default True;
    property ShowMargins: Boolean read GetShowMargins write SetShowMargins default True;
    property ShowProgress: Boolean read GetShowProgress write SetShowProgress default True;
    property TableFormat: String read GetTableFormat write SetTableFormat;
    property TableGrid: Boolean read fTableGrid write fTableGrid default False;
    property TabSize: Cardinal read fTabSize write fTabSize default 8;
    property Title: String read GetTitle write SetTitle nodefault;
    property TokenSeparator: Char read fTokenSeparator write fTokenSeparator default '|';
    property Visible;
    property WordWrap: Boolean read fWordWrap write fWordWrap default True;
    property ZoomLocation: TZoomLocation read fZoomLocation write fZoomLocation default zlTopLeft;
    property ZoomPercent: Cardinal read GetZoomPercent write SetZoomPercent default 25;
  end;

function ExpandTabsAsSpaces(const S: String; const TabSize: Cardinal): String;
function GenSpace(const Size: Cardinal): String;
function ReplaceSubString(const OldSubStr, NewSubStr: String; S: String): String;
function StripBackToWhiteSpace(const S: String): String;
procedure FillGradient(Canvas: TCanvas; Rc: TRect; LeftTopColor, RightBottomColor: TColor; Orientation: TGradientOrientation);
procedure TokenizeString(const S: String; TokenSeparator: Char; Tokens: TStringList);

implementation

{Typically gutters are symmetrical on printers, but GetDeviceCaps
doesn't report back bottom or right gutters.  If I calculate these
gutters based on other information returned by GetDeviceCaps (instead
of just assuming things are symmetrical), I get a smaller and
typically incorrect result.  I think symmetric gutters are what we
want in most cases, but you can comment this $DEFINE out if you want
the gutters to be calculated based on the exact values returned by
GetDeviceCaps.}
{$DEFINE USE_SYMMETRIC_GUTTERS}

{$R PagePrnt.dcr}

{=============================================================================}
{ Non-methods that may prove useful elsewhere.                                }
{=============================================================================}

function ReplaceSubString(const OldSubStr, NewSubStr: String; S: String): String;
var
   P: Cardinal;
begin
     {Currently, this routine is terribly inefficient since Pos
     always starts back at the beginning of the string.  However,
     our header, footer, and table strings are usually very short,
     so this doesn't matter much in practice.}
     Result := '';
     P := Pos(OldSubStr, S);
     while (P <> 0) do
     begin
          Result := Result + Copy(S, 1, P-1) + NewSubStr;
          Delete(S, 1, P-1+Length(OldSubStr));
          P := Pos(OldSubStr, S);
     end;
     Result := Result+S;
end;

procedure TokenizeString(const S: String; TokenSeparator: Char;
          Tokens: TStringList);
var
   i, Len: Cardinal;
   CurToken: String;
begin
     Tokens.Clear;
     CurToken:='';
     Len:=Length(S);
     for i:=1 to Len do
     begin
          if S[i] = TokenSeparator then
          begin
               Tokens.Add(CurToken);
               CurToken:='';
          end
          else
              CurToken:=CurToken+S[i];
     end;
     Tokens.Add(CurToken);
end;

function StripBackToWhiteSpace(const S: String): String;
var
   i, Len, Mark: Cardinal;
begin
     Mark:=0;
     Len:=Length(S);
     for i:=Len downto 1 do
     begin
          if S[i] in [#0..#32] then
          begin
               Mark:=i;
               Break;
          end;
     end;

     if Mark > 0 then Result:=Copy(S, 1, Mark)
     {If there is nowhere to break, just return the whole line.}
     else Result:=S;
end;

function ExpandTabsAsSpaces(const S: String; const TabSize: Cardinal): String;
var
   i, Len, Size: Cardinal;
   Buffer: String;
begin
     {TabStr:='';
     for i:=1 to TabSize do TabStr:=TabStr+' ';}

     Buffer:='';
     Len:=Length(S);
     for i:=1 to Len do
     begin
          if S[i]=#9 then
          begin
               Size:=TabSize-(Length(Buffer) mod TabSize);
               Buffer:=Buffer+GenSpace(Size);
          end
          else Buffer:=Buffer+S[i];
     end;
     Result:=Buffer;
end;

function GenSpace(const Size: Cardinal): String;
var
   Str: String;
begin
     Str:='';
     while Length(Str) < Size do Str:=Str+' ';
     GenSpace:=Str;
end;

procedure FillGradient(Canvas: TCanvas; Rc: TRect;
          LeftTopColor, RightBottomColor: TColor; Orientation: TGradientOrientation);
var
   LR, LG, LB, RR, RG, RB, MR, MG, MB: Integer; //Left, Right, and Mix RGBs
   i, LeftRGB, RightRGB, LeftWeight, RightWeight, Times: Longint;
   MixColor: TColor;
begin
     LeftRGB:=ColorToRGB(LeftTopColor);
     LR:=GetRValue(LeftRGB);
     LG:=GetGValue(LeftRGB);
     LB:=GetBValue(LeftRGB);
     RightRGB:=ColorToRGB(RightBottomColor);
     RR:=GetRValue(RightRGB);
     RG:=GetGValue(RightRGB);
     RB:=GetBValue(RightRGB);
     if Orientation = goHorizontal then
        Times:=Rc.Right-Rc.Left-1
     else
         Times:=Rc.Bottom-Rc.Top-1;
     if Times > 0 then
     begin
          for i:=0 to Times do
          begin
               LeftWeight:=Times-i;
               RightWeight:=i;
               MR:=(LR*LeftWeight+RR*RightWeight) div Times;
               MG:=(LG*LeftWeight+RG*RightWeight) div Times;
               MB:=(LB*LeftWeight+RB*RightWeight) div Times;
               MixColor:=RGB(MR, MG, MB);
               with Canvas do
               begin
                    {Pen.Color:=MixColor;}
                    Brush.Color:=MixColor;
                    if Orientation = goHorizontal then
                       {PolyLine([ Point(Rc.Left+i,Rc.Top), Point(Rc.Left+i,Rc.Bottom) ])}
                       FillRect(Rect(Rc.Left+i, Rc.Top, Rc.Left+i+1, Rc.Bottom))
                    else
                        {PolyLine([ Point(Rc.Left, Rc.Top+i), Point(Rc.Right, Rc.Top+i) ]);}
                        FillRect(Rect(Rc.Left, Rc.Top+i, Rc.Right, Rc.Top+i+1));
               end;
          end;
     end;
end;

function Minimum(Value1, Value2: Cardinal): Cardinal;
begin
     Result:=Value1;
     if Value1 > Value2 then Result:=Value2;
end;

{=============================================================================}
{ Public stuff for TPageList.                                                 }
{=============================================================================}

destructor TPageList.Destroy;
var
   i: Integer;
   Page: TPrintPage;
begin
     for i:=0 to Count-1 do
     begin
          Page:=GetPage(i);
          if Page <> nil then Page.Free;
     end;

     inherited Destroy;
end;

function TPageList.GetPage(const Index: Integer): TPrintPage;
begin
     Result:=TPrintPage(Items[Index]);
end;

{=============================================================================}
{ Public stuff for TPagePrinter.                                              }
{=============================================================================}

constructor TPagePrinter.Create(Owner: TComponent);
{$IFDEF PAGEPRINT_SHAREWARE}
var
   AboutBox: TPgPrnAboutBox;
{$ENDIF}
begin
     fStillCreating:=True;
     inherited Create(Owner);
{$IFDEF PAGEPRINT_SHAREWARE}
     AboutBox:=TPgPrnAboutBox.Create(Application);
     try
        AboutBox.ShowModal;
     finally
            AboutBox.Free;
     end;
{$ENDIF}
     {We don't want a TPagePrinter to be a
     container like normal TScrollBoxes can.}
     ControlStyle := ControlStyle - [csAcceptsControls];
     fUpdateRefCount:=0;

     fHeaderFont:=TFont.Create;
     fFooterFont:=TFont.Create;
     fLines := TStringList.Create;
     {Make this explicitly nil so UpdateProgressDlg
     can tell if it needs to Create or Free itself.}
     fPPPrnPrgDlg := nil;
     fCancelPrinting:=False;
     {Setup the scrollbars.}
     HorzScrollBar.Tracking:=True;
     HorzScrollBar.Increment:=16;
     VertScrollBar.Tracking:=True;
     VertScrollBar.Increment:=16;

     fCurrentX:=0;
     fCurrentY:=0;
     fLineNumber:=0;
     fPageNumber:=0;
     fPrintingToFile:=False;
     {It's hard to zero a non-typed variable...}
     FillChar(fFileVar, sizeof(fFileVar), #0);

     {Keep our own pointer to the Printer object.}
     fPrinter:=Printers.Printer;

     fTokenSeparator := '|';
     fZoomPercent := 25;
     fZoomLocation := zlTopLeft;
     fPrinting := False;
     fShadowOffset:=5;
     fUsingTempPage:=True;
     fDefaultColWidth:=1;

     {Setup the drawing surface.}
     fPaintBox := TPaintBox.Create(Self);
     fPaintBox.Parent := Self;
     fPaintBox.Align := alClient;
     fPaintBox.OnPaint:=PaintPreview;
     {Setup the page list.}
     ResetPageList(False);

     {Now setup the remaining properties which depend on the canvas.}
     Font.Name := 'Courier New';
     Font.Size := 10;
     Font.Style := [];
     HeaderFont:=Font;
     FooterFont:=Font;
     AutoHeaderFont:=True;
     AutoFooterFont:=True;

     Width := 89;
     Height := 115;
     Orientation:=poPortrait;
     Title:='';

     LineSpacing:=lsSingleSpace;
     TabSize:=8;
     WordWrap:=True;
     Alignment:=taLeftJustify;
     PageBorders:=[];
     ShowProgress:=True;
     ShowCancel:=True;
     Header:='';
     HeaderFormat:='';
     Footer:='';
     FooterFormat:='';
     TableFormat:='';
     PageBorderOffset:=0;
     DefaultColWidth:=0;
     MeasureUnit:=muInches;
     TableGrid:=False;
     PrintToFile:=False;
     ShadowColor:=clBtnShadow;
     ShowMargins:=True;
     GradientBackground:=True;
     Collate:=True;
     Copies:=1;
     PrintToPage:=0;
     PrintFromPage:=0;
     AbortOnCancel:=False;

     MarginTop:=GutterTop;
     MarginBottom:=GutterBottom;
     MarginLeft:=GutterLeft;
     MarginRight:=GutterRight;

     fStillCreating:=False;
end;

destructor TPagePrinter.Destroy;
begin
     FinishPrintPage;
     fPaintBox.Free;
     fLines.Free;
     fPages.Free;
     fHeaderFont.Free;
     fFooterFont.Free;
     inherited Destroy;
end;

procedure TPagePrinter.Clear;
begin
     if not Printing then
     begin
          ResetPageList(False);
          UpdatePagePreviewSize;
          Invalidate;
     end
     else
         raise EPagePrinter.Create('Can''t clear contents while printing');
end;

procedure TPagePrinter.BeginDoc;
begin
     {Do this before we set printing to True}
     {so it refreshes the margins too.}
     RefreshProperties;
     {Now we set the printing flag.}
     fPrinting := True;
     fPrintingHeaderOrFooter := False;
     if PrintToFile then
     begin
          SetPixelsPerInch;
          AssignFile(fFileVar, FileName);
          Rewrite(fFileVar);
          fPrintingToFile:=True;
     end
     else
         ResetPageList(True);
     {Make extra sure we get the Font.PixelsPerInch}
     {property set correctly.}
     SetPixelsPerInch;
     fPageNumber:=1;
     DoNewPageProcessing;
end;

procedure TPagePrinter.EndDoc;
begin
     fPrinting := False;
     if PrintToFile then
     begin
          CloseFile(fFileVar);
          FillChar(fFileVar, sizeof(fFileVar), #0);
          fPrintingToFile:=False;
     end
     else
     begin
          FinishPrintPage;
          CreateTempPage;
          PageNumber := 1;
          UpdatePagePreviewSize;
          Invalidate;
     end;
end;

function TPagePrinter.NewPage: Cardinal;
begin
     if fPrintingToFile then
        Writeln(fFileVar, #12)
     else
         NewPrintPage;
     Inc(fPageNumber);
     DoNewPageProcessing;
     Result:=PageNumber;
end;

function TPagePrinter.NewLine: Cardinal;
begin
     fCurrentX:=0;
     fCurrentY:=fCurrentY+fLineSpace;

     {See if the entire next line will fit.}
     if (not fPrintingToFile) and (not fPrintingHeaderOrFooter) and
        ((fCurrentY+fLineSpace) >= PixelPrintHeight) then
         NewPage
     else
     begin
          if fPrintingToFile then Writeln(fFileVar);
          Inc(fLineNumber);
     end;
     {Fire the event handler if it exists.}
     if Assigned(fOnNewLine) then fOnNewLine(Self);
     Result:=LineNumber;
end;

{This function returns whether it was successful.}
function TPagePrinter.PrevLine: Boolean;
begin
     Result:=False;
     if (fCurrentY >= fLineSpace) and not fPrintingToFile then
     begin
          fCurrentX:=0;
          fCurrentY:=fCurrentY-fLineSpace;
          Dec(fLineNumber);
          Result:=True;
     end;
end;

procedure TPagePrinter.Write(const Line: String);
var
   LineWidth: TPixels;
   Buffer: String;
begin
     if Pos(#9, Line)>0 then Buffer:=ExpandTabsAsSpaces(Line, TabSize)
     else Buffer:=Line;

     LineWidth:=Canvas.TextWidth(Buffer);
     if (LineWidth > (PixelPrintWidth-fCurrentX)) and (Length(Buffer) > 1) then
     begin
          if WordWrap then SplitLineAndPrint(Buffer, True)
          else Write(GetClippedLine(Buffer, PixelPrintWidth-fCurrentX));
     end
     else
     begin
          {Make sure we don't write off the end of the page.}
          if (fCurrentY+fLineSpace) >= PixelPrintHeight then
             if not fPrintingToFile then NewPage;
          {Now print the line.}
          if fPrintingToFile then
             System.Write(fFileVar, Buffer)
          else
              Canvas.TextOut(StartingLeft+fCurrentX, StartingTop+fCurrentY, Buffer);
          fCurrentX:=fCurrentX+LineWidth;
     end;
end;

procedure TPagePrinter.WriteLine(const Line: String);
var
   LineWidth: TPixels;
   Buffer: String;
begin
     if Pos(#9, Line)>0 then Buffer:=ExpandTabsAsSpaces(Line, TabSize)
     else Buffer:=Line;

     LineWidth:=Canvas.TextWidth(Buffer);
     if (LineWidth > PixelPrintWidth) and (Length(Buffer) > 1) then
     begin
          fCurrentX:=0;
          if WordWrap then SplitLineAndPrint(Buffer, False)
          else WriteLine(GetClippedLine(Buffer, PixelPrintWidth));
     end
     else
     begin
          case Alignment of
               taRightJustify: fCurrentX := PixelPrintWidth-LineWidth;
               taCenter: fCurrentX := (PixelPrintWidth-LineWidth) shr 1;
          else
              fCurrentX:=0;
          end;
          {Make sure we don't write off the end of the page.}
          if (fCurrentY+fLineSpace) >= PixelPrintHeight then
             if not fPrintingToFile then NewPage;
          {Now print the line.}
          if fPrintingToFile then //Put spaces in for alignment purposes.
             Writeln(fFileVar, GenSpace(fCurrentX div Canvas.TextWidth(' '))+Buffer)
          else
              Canvas.TextOut(StartingLeft+fCurrentX, StartingTop+fCurrentY, Buffer);
          NewLine;
     end;
end;

procedure TPagePrinter.WriteLineAligned(const AAlignment: TAlignment; const Line: String);
var
   OldAlign: TAlignment;
begin
     OldAlign:=Alignment;
     try
        Alignment:=AAlignment;
        WriteLine(Line);
     finally
            Alignment:=OldAlign;
     end;
end;

procedure TPagePrinter.WriteTableLine(const Line: String);
var
   FormatTokens, LineTokens: TStringList;
   i, CurWidth, LeftPos, Offset: Integer;
   FloatCurWidth: TMeasurement;
   CurAlignment: TAlignment;
   CurToken, Buffer: String;
   RepeatAgain, TopGrid, WriteGrid: Boolean;
   {This local procedure sets up a lot of local variables.}
   procedure WriteTableGridHelper(const i: Integer);
   begin
        {Get the Width and Alignment from the current column format.}
        CurToken:=FormatTokens[i];
        ParseFormatToken(CurToken, CurAlignment, FloatCurWidth);
        WriteGrid:=TableGrid;
        {Temporarily force the drawing of the Grid OFF}
        if FloatCurWidth < 0 then WriteGrid:=False;
        {Get the Absolute value of FloatCurWidth}
        CurWidth:=MeasureUnitsToPixelsH(Abs(FloatCurWidth));
   end;
begin
     FormatTokens:=TStringList.Create;
     LineTokens:=TStringList.Create;
     TopGrid:=TableGrid;

     try
        TokenizeString(TableFormat, TokenSeparator, FormatTokens);
        TokenizeString(Line, TokenSeparator, LineTokens);

        repeat
              RepeatAgain:=False;
              fCurrentX:=StartingLeft;
              for i:=0 to FormatTokens.Count-1 do
              begin
                   {Set up a bunch of local variables within the local procedure.}
                   WriteTableGridHelper(i);

                   {Now get a line token even if it's blank.}
                   if i < LineTokens.Count then CurToken:=LineTokens[i]
                   else CurToken:='';

                   (*Expand logical field names (e.g. {$LINE}).*)
                   {The '{$' check is just to see if we can skip it.}
                   if Pos('{$', CurToken) > 0 then
                      CurToken:=ExpandLogicalFields(CurToken);

                   if WordWrap then
                   begin
                        {Determine just what will fit in the current column}
                        Buffer:=CurToken;
                        SplitLine(CurToken, Buffer, CurWidth, False);
                        {Check if a forced line break is requested}
                        LeftPos:=Pos(#10, CurToken);
                        if LeftPos > 0 then
                        begin
                             if Length(Buffer) > 0 then Buffer:=' '+Buffer;
                             Buffer:=Copy(CurToken, LeftPos+1, Length(CurToken))+Buffer;
                             Delete(CurToken, LeftPos, Length(CurToken));
                        end;
                        if i < LineTokens.Count then LineTokens[i]:=Buffer;
                        {Need to repeat loop if there is any unprinted text}
                        if Length(Buffer) > 0 then RepeatAgain:=True;
                   end
                   else
                       {Get just what will fit in the current column.}
                       CurToken:=GetClippedLine(CurToken, CurWidth);

                   {Figure out where the X position will be in the current column.}
                   case CurAlignment of
                        taCenter: LeftPos:=(CurWidth-Canvas.TextWidth(CurToken)) shr 1;
                        taRightJustify: LeftPos:=CurWidth-Canvas.TextWidth(CurToken);
                   else
                       LeftPos:=0;
                   end;
                   {We try to offset the text so it's not overlapping a grid border.}
                   Offset:=3*DefaultBorderWidth*(GetDeviceCaps(fPrinter.Handle, LOGPIXELSX) div DefaultDPI);
                   if Canvas.TextWidth(CurToken) < (CurWidth-Offset) then
                      if CurAlignment = taRightJustify then LeftPos:=LeftPos - Offset
                      else if CurAlignment = taLeftJustify then LeftPos:=LeftPos + Offset;

                   {Print out the current token.}
                   if fPrintingToFile then
                      System.Write(fFileVar, GenSpace(LeftPos div Canvas.TextWidth(' '))+CurToken+GenSpace((CurWidth-(LeftPos+Canvas.TextWidth(CurToken))) div Canvas.TextWidth(' ') ) )
                   else
                       Canvas.TextOut(fCurrentX+LeftPos, fCurrentY+StartingTop, CurToken);

                   {Draw the cell lines if necessary.}
                   if WriteGrid then WriteTableGrid(CurWidth, TopGrid, False);
                   {Increase fCurrentX by the COLUMN width.}
                   fCurrentX:=fCurrentX+CurWidth;
              end;
              {Do not draw the top of the grid again}
              TopGrid:=False;
              if RepeatAgain then NewLine;
        until RepeatAgain = False;

        {Draw the Bottom of the grid if necessary}
        if TableGrid then
        begin
             fCurrentX:=StartingLeft;
             for i:=0 to FormatTokens.Count-1 do
             begin
                  {Set up a bunch of local variables within the local procedure.}
                  WriteTableGridHelper(i);
                  {Draw the cell lines if necessary.}                  
                  if WriteGrid then WriteTableGrid(CurWidth, False, True);
                  {Increase fCurrentX by the COLUMN width.}
                  fCurrentX:=fCurrentX+CurWidth;
             end;
        end;
     finally
            FormatTokens.Free;
            LineTokens.Free;
     end;

     {If we're not printing the Header or Footer, go to a new line.}
     {if ((fCurrentY >= 0) and (fCurrentY < PixelPrintHeight)) or}
     if (not fPrintingHeaderOrFooter) or fPrintingToFile then NewLine;
end;

procedure TPagePrinter.WriteTableGrid(const CurWidth: TPixels; const TopGrid, BottomGrid: Boolean);
var
   X, Y: Integer;
begin
     {Draw the cell lines.}
     X:=fCurrentX;
     Y:=fCurrentY+StartingTop;
     case LineSpacing of
          lsHalfSpace: Y:=Y-(fTextMetrics.tmExternalLeading shr 1);
          lsSingleSpace: Y:=Y-fTextMetrics.tmExternalLeading;
          lsSingleAndAHalf: Y:=Y-Round(fTextMetrics.tmExternalLeading*1.5);
          lsDoubleSpace: Y:=Y-(fTextMetrics.tmExternalLeading shl 1);
     end;
     {Draw the horizontal lines.}
     Canvas.Pen.Width:=DefaultBorderWidth*(GetDeviceCaps(fPrinter.Handle, LOGPIXELSY) div DefaultDPI);
     if TopGrid then
        Canvas.PolyLine([Point(X, Y), Point(X+CurWidth, Y)]);
     if BottomGrid then
        Canvas.PolyLine([Point(X, Y+fLineSpace), Point(X+CurWidth, Y+fLineSpace)]);
     {Draw the vertical lines.}
     Canvas.Pen.Width:=DefaultBorderWidth*(GetDeviceCaps(fPrinter.Handle, LOGPIXELSX) div DefaultDPI);
     Canvas.PolyLine([Point(X, Y), Point(X, Y+fLineSpace)]);
     Canvas.PolyLine([Point(X+CurWidth, Y), Point(X+CurWidth, Y+fLineSpace)]);
end;

procedure TPagePrinter.WriteLines(const LinesAsTable: Boolean);
var
   i: Integer; {This must allow negatives for Lines.Count-1}
begin
     for i:=0 to Lines.Count-1 do
     begin
          if LinesAsTable and (TableFormat<>'') then
             WriteTableLine(Lines[i])
          else
              WriteLine(Lines[i]);
     end;
end;

procedure TPagePrinter.RefreshProperties;
begin
     if not Printing then
     begin
          {Make sure the margins are correct for the current printer.}
          SetMarginTop(MarginTop);
          SetMarginBottom(MarginBottom);
          SetMarginLeft(MarginLeft);
          SetMarginRight(MarginRight);
     end;
     {Make sure the font gets sized correctly for the page.}
     GetTextMetrics(GetPrinterHandle, fTextMetrics);
     SetPixelsPerInch;
     {Make sure fLineSpace gets updated.}
     SetLineSpacing(LineSpacing);
     {Any column widths in friendly headers need to be recalculated}
     {now that we have a valid printer DC that we can get a DPI from.}
     {Also, any date and time fields need to be updated.}
     if FriendlyHeader <> '' then SetFriendlyHeader(FriendlyHeader);
     if FriendlyFooter <> '' then SetFriendlyFooter(FriendlyFooter);
end;

{=============================================================================}
{ Private and Protected stuff for TPagePrinter.                               }
{=============================================================================}

procedure TPagePrinter.DoNewPageProcessing;
var
   PixelPageBorderOffset: TPixels;
   OldTableFormat: String;
   OldTableGrid, OldAutoHeaderFont, OldAutoFooterFont: Boolean;
   OldFont: TFont;
   CurX, CurY, CurLine: Integer;
begin
     {Reset the fields and fire new page event.}
     fCurrentX:=0;
     fCurrentY:=0;
     fLineNumber:=0;
     if Assigned(fOnNewPage) then fOnNewPage(Self);

     {Save these values in case OnNewPage modifies them.}
     CurX:=fCurrentX;
     CurY:=fCurrentY;
     CurLine:=fLineNumber;
     {Keep TableFormat because we temporarily
     change it for the Header and Footer.
     Similar for TableGrid.}
     OldTableFormat:=TableFormat;
     OldTableGrid:=TableGrid;
     TableGrid:=False;
     OldFont:=TFont.Create;
     OldFont.Assign(Font);
     OldAutoHeaderFont:=AutoHeaderFont;
     AutoHeaderFont:=False;
     OldAutoFooterFont:=AutoFooterFont;
     AutoFooterFont:=False;
     fPrintingHeaderOrFooter:=True;
     {I'm trying to center the header and footer between
      the top of the page and the PageBorders.}
     try
        {Print the header.}
        if Header <> '' then
        begin
             Font.Assign(HeaderFont);
             {fCurrentY should be negative here since we're drawing above the top margin.}
             //TLinePrinter way: fCurrentY:=((StartingTop-fLineSpace) shr 1)-StartingTop;
             if pbTop in PageBorders then
                fCurrentY:=((MeasureUnitsToPixelsV(MarginTop-PageBorderOffset)-fLineSpace) shr 1)-MeasureUnitsToPixelsV(MarginTop)
             else
                 fCurrentY:=((MeasureUnitsToPixelsV(MarginTop)-fLineSpace) shr 1)-MeasureUnitsToPixelsV(MarginTop);
             if (fCurrentY + StartingTop) < 0 then fCurrentY := -1*StartingTop;
             TableFormat:=HeaderFormat;
             WriteTableLine(Header);
        end;

        {Print the footer.}
        if Footer <> '' then
        begin
             Font.Assign(FooterFont);
             //TLinePrinter Way: fCurrentY:=PixelPrintHeight+((StartingBottom-fLineSpace) shr 1);
             if pbBottom in PageBorders then
                fCurrentY:=PixelPrintHeight+MeasureUnitsToPixelsV(PageBorderOffset)+((MeasureUnitsToPixelsV(MarginBottom-PageBorderOffset)-fLineSpace) shr 1)
             else
                 fCurrentY:=PixelPrintHeight+((MeasureUnitsToPixelsV(MarginBottom)-fLineSpace) shr 1);
             if (StartingTop+fCurrentY+fLineSpace) > MeasureUnitsToPixelsV(AvailablePageHeight) then
                fCurrentY:=MeasureUnitsToPixelsV(AvailablePageHeight)-StartingTop-fLineSpace;
             TableFormat:=FooterFormat;
             WriteTableLine(Footer);
        end;
     finally
            fPrintingHeaderOrFooter:=False;
            {Restore the original values.}
            TableFormat:=OldTableFormat;
            TableGrid:=OldTableGrid;
            Font.Assign(OldFont);
            OldFont.Free;
            {I'm intentionally setting the private fields here.}
            fAutoHeaderFont:=OldAutoHeaderFont;
            fAutoFooterFont:=OldAutoFooterFont;
     end;

     {We must reset these here because printing the header and footer modifies them.}
     fCurrentX:=CurX;
     fCurrentY:=CurY;
     fLineNumber:=CurLine;
     {Fire the OnNewLine event for the first line on the page.}
     if Assigned(fOnNewLine) then fOnNewLine(Self);

     {Print the PageBorders.}
     with Canvas do
     begin
          Pen.Width:=DefaultBorderWidth*(GetDeviceCaps(fPrinter.Handle, LOGPIXELSY) div DefaultDPI);
          PixelPageBorderOffset:=MeasureUnitsToPixelsV(PageBorderOffset);
          if pbTop in PageBorders then
          begin
               MoveTo(StartingLeft-PixelPageBorderOffset,StartingTop-PixelPageBorderOffset);
               LineTo(StartingLeft+PixelPrintWidth+PixelPageBorderOffset, StartingTop-PixelPageBorderOffset);
          end;
          if pbBottom in PageBorders then
          begin
               MoveTo(StartingLeft-PixelPageBorderOffset, StartingTop+PixelPrintHeight+PixelPageBorderOffset);
               LineTo(StartingLeft+PixelPrintWidth+PixelPageBorderOffset, StartingTop+PixelPrintHeight+PixelPageBorderOffset);
          end;

          Pen.Width:=DefaultBorderWidth*(GetDeviceCaps(fPrinter.Handle, LOGPIXELSX) div DefaultDPI);
          PixelPageBorderOffset:=MeasureUnitsToPixelsH(PageBorderOffset);
          if pbLeft in PageBorders then
          begin
               MoveTo(StartingLeft-PixelPageBorderOffset, StartingTop-PixelPageBorderOffset);
               LineTo(StartingLeft-PixelPageBorderOffset, StartingTop+PixelPrintHeight+PixelPageBorderOffset);
          end;
          if pbRight in PageBorders then
          begin
               MoveTo(StartingLeft+PixelPrintWidth+PixelPageBorderOffset, StartingTop-PixelPageBorderOffset);
               LineTo(StartingLeft+PixelPrintWidth+PixelPageBorderOffset, StartingTop+PixelPrintHeight+PixelPageBorderOffset);
          end;
     end;
end;

procedure TPagePrinter.SplitLine(var CurLine: String; var Buffer: String; const ClipWidth: TPixels; const TrimLastWhiteSpace: Boolean);
var
   Len: Cardinal;
begin
     CurLine:=GetClippedLine(Buffer, ClipWidth);
     Len:=Length(CurLine);
     {If the next character isn't whitespace, slide back to the nearest.
     Also, like most word processors do, I'm going to delete the
     first leading whitespace character left in the next-line buffer after
     the delete/newline (if one exists).}
     if Len<Length(Buffer) then
     begin
          if not (Buffer[Len+1] in [#0..#32]) then
          begin
               CurLine:=StripBackToWhiteSpace(CurLine);
               Len:=Length(CurLine);
          end
          else
              if TrimLastWhiteSpace then Inc(Len);
     end;
     Delete(Buffer, 1, Len);
end;

procedure TPagePrinter.SplitLineAndPrint(const Line: String; UseWrite: Boolean);
var
   Buffer, CurLine: String;
begin
     Buffer:=Line;
     repeat
           SplitLine(CurLine, Buffer, PixelPrintWidth-fCurrentX, not UseWrite);
           if UseWrite then
           begin
                Write(CurLine);
                if Length(Buffer) > 0 then NewLine;
           end
           else
               WriteLine(CurLine);
     until Buffer='';
end;

function TPagePrinter.GetClippedLine(const Line: String; const Width: TPixels): String;
var
   PixelLen: TPixels;
   StartPos, EndPos, Mid, PreviousMid: Cardinal;
begin
     PixelLen:=Canvas.TextWidth(Line);
     if PixelLen > Width then
     begin
          EndPos:=Length(Line);
          StartPos:=1;
          Mid:=0;
          repeat
                PreviousMid:=Mid;
                Mid:=(StartPos+EndPos) shr 1;
                PixelLen:=Canvas.TextWidth(Copy(Line,1,Mid));

                if PixelLen > Width then EndPos:=Mid
                else if PixelLen < Width then StartPos:=Mid
                else StartPos:=EndPos;
          until (Mid=PreviousMid) or (StartPos>=EndPos);
          Result:=Copy(Line, 1, Mid);
     end
     else
         Result:=Line;
end;

function TPagePrinter.PixelPrintWidth: TPixels;
begin
     try
        Result:=MeasureUnitsToPixelsH(AvailablePageWidth)-StartingLeft-StartingRight;
     except
           on ERangeError do Result:=0;
     end;
end;

function TPagePrinter.PixelPrintHeight: TPixels;
begin
     try
        Result:=MeasureUnitsToPixelsV(AvailablePageHeight)-StartingTop-StartingBottom;
     except
           on ERangeError do Result:=0;
     end;
end;

function TPagePrinter.GetTitle: String;
begin
     try
        Result:=fPrinter.Title;
     except
           on EPrinter do Result:='<Unknown>';
     end;
end;

procedure TPagePrinter.SetTitle(Value: String);
begin
     try
        fPrinter.Title:=Value
     except
           on EPrinter do ;
     end;
end;

function TPagePrinter.GetOrientation: TPrinterOrientation;
begin
     try
        Result:=fPrinter.Orientation;
     except
           on EPrinter do Result:=poPortrait;
     end;
end;

procedure TPagePrinter.SetOrientation(Value: TPrinterOrientation);
begin
     if not Printing then
     begin
          try
             fPrinter.Orientation:=Value;
          except
                on EPrinter do ;
          end;
          UpdatePagePreviewSize;
          Invalidate;
     end
     else
         raise EPagePrinter.Create('Unable to change orientation while printing');
end;

function TPagePrinter.GetAvailablePageHeight: TMeasurement;
begin
     try
{$IFDEF USE_SYMMETRIC_GUTTERS}
        Result:=PhysicalPageHeight-GutterTop-GutterBottom;
{$ELSE}
        Result:=PixelsToMeasureUnitsV(GetDeviceCaps(fPrinter.Handle, VERTRES));
{$ENDIF}
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultAvailablePageHeightIn
           else Result:=DefaultAvailablePageHeightMm;
     end;
end;

function TPagePrinter.GetAvailablePageWidth: TMeasurement;
begin
     try
{$IFDEF USE_SYMMETRIC_GUTTERS}
        Result:=PhysicalPageWidth-GutterLeft-GutterRight;
{$ELSE}
        Result:=PixelsToMeasureUnitsH(GetDeviceCaps(fPrinter.Handle, HORZRES));
{$ENDIF}
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultAvailablePageWidthIn
           else Result:=DefaultAvailablePageWidthMm;
     end;
end;

function TPagePrinter.GetPageNumber: Cardinal;
begin
     Result:=fPageNumber;
end;

function TPagePrinter.GetPrinting: Boolean;
begin
     if PrintToFile then
        Result:=fPrintingToFile
     else
         Result:=fPrinting;
end;

procedure TPagePrinter.SetPixelsPerInch;
var
   FontSize: Integer;
begin
     {This routine gets us around the Delphi tiny font bug.}
     FontSize := Canvas.Font.Size;
     try
        Canvas.Font.PixelsPerInch := GetDeviceCaps(fPrinter.Handle, LOGPIXELSY);
     except
           on EPrinter do Canvas.Font.PixelsPerInch:=DefaultDPI;
     end;
     if FontSize < 144 then
        Canvas.Font.Size := FontSize + 1
     else
         Canvas.Font.Size := FontSize - 1;
     Canvas.Font.Size := FontSize;
end;

procedure TPagePrinter.SetMarginTop(Value: TMeasurement);
begin
     if not Printing then
     begin
          if Value >= GutterTop then
          begin
               if Value <= (PhysicalPageHeight-GutterBottom) then fMarginTop:=Value
               else
               begin
                    fMarginTop:=PhysicalPageHeight-GutterBottom;
               end;
          end
          else
              fMarginTop:=GutterTop;
          Invalidate;
     end
     else raise EPagePrinter.Create('Unable to change top margin while printing');
end;

procedure TPagePrinter.SetMarginBottom(Value: TMeasurement);
begin
     if not Printing then
     begin
          if Value >= GutterBottom then
          begin
               if Value <= (PhysicalPageHeight-GutterTop) then fMarginBottom:=Value
               else
               begin
                    fMarginBottom:=PhysicalPageHeight-GutterTop;
               end;
          end
          else
              fMarginBottom:=GutterBottom;
          Invalidate;
     end
     else raise EPagePrinter.Create('Unable to change bottom margin while printing');
end;

procedure TPagePrinter.SetMarginLeft(Value: TMeasurement);
begin
     if not Printing then
     begin
          if Value >= GutterLeft then
          begin
               if Value <= (PhysicalPageWidth-GutterRight) then fMarginLeft:=Value
               else
               begin
                    fMarginLeft:=PhysicalPageWidth-GutterRight;
               end;
          end
          else
              fMarginLeft:=GutterLeft;
          Invalidate;
     end
     else raise EPagePrinter.Create('Unable to change left margin while printing');
end;

procedure TPagePrinter.SetMarginRight(Value: TMeasurement);
begin
     if not Printing then
     begin
          if Value >= GutterRight then
          begin
               if Value < (PhysicalPageWidth-GutterLeft) then fMarginRight:=Value
               else
               begin
                    fMarginRight:=PhysicalPageWidth-GutterLeft;
               end;
          end
          else
              fMarginRight:=GutterRight;
          Invalidate;
     end
     else raise EPagePrinter.Create('Unable to change right margin while printing');
end;

procedure TPagePrinter.SetMeasureUnit(Value: TMeasureUnit);
begin
     if Value <> fMeasureUnit then
     begin
          fMeasureUnit:=Value;
          if not fStillCreating then
          begin
               { Update the measurements if the units have changed.}
               if MeasureUnit = muInches then
               begin
                    MarginTop:=MarginTop/25.4;
                    MarginBottom:=MarginBottom/25.4;
                    MarginLeft:=MarginLeft/25.4;
                    MarginRight:=MarginRight/25.4;
                    PageBorderOffset:=PageBorderOffset/25.4;
                    DefaultColWidth:=DefaultColWidth/25.4;
               end
               else
               begin
                    MarginTop:=MarginTop*25.4;
                    MarginBottom:=MarginBottom*25.4;
                    MarginLeft:=MarginLeft*25.4;
                    MarginRight:=MarginRight*25.4;
                    PageBorderOffset:=PageBorderOffset*25.4;
                    DefaultColWidth:=DefaultColWidth*25.4;
               end;
               RefreshProperties;
               if FriendlyFooter = '' then fFooterFormat:=ValidateFormatString(FooterFormat, True);
               if FriendlyHeader = '' then fHeaderFormat:=ValidateFormatString(HeaderFormat, True);
               fTableFormat:=ValidateFormatString(TableFormat, True);
          end;
     end;
end;

procedure TPagePrinter.SetLineSpacing(Value: TLineSpacing);
var
   H: TPixels;
begin
     fLineSpacing:=Value;
     H:=Abs(Canvas.Font.Height);
     case Value of
          lsHalfSpace: fLineSpace:=H shr 1;
          lsSingleSpace: fLineSpace:=H;
          lsSingleAndAHalf: fLineSpace:=H+(H shr 1);
          lsDoubleSpace: fLineSpace:=H+H;
     end;
end;

function TPagePrinter.GetLines: TStrings;
begin
     {I have to do Check/Create here because C++Builder}
     {1.0 kept giving access violations if I didn't...}
     if fLines = nil then
        fLines := TStringList.Create;
     GetLines := fLines;
end;

procedure TPagePrinter.SetLines(Value: TStrings);
begin
     if Value = nil then
     begin
          fLines.Free;
          fLines := nil;
          Exit;
     end;
     Lines.Assign(Value);
end;

procedure TPagePrinter.SetDefaultColWidth(Value: TMeasurement);
begin
     fDefaultColWidth:=Value;
end;

procedure TPagePrinter.UpdateProgressDlg(const Status: String; const CurrentPage, FromPage, ToPage: Cardinal);
const
     DefaultProgClientHeight = 98;
begin
     if ShowProgress and (Status <> ProgressFinishMsg) then
     begin
          {Create it if is doesn't already exist.}
          if fPPPrnPrgDlg = nil then
          begin
               fPPPrnPrgDlg:=TPPPrnPrgDlg.Create(Application);
               {Set the cancel event handler.}
               fPPPrnPrgDlg.btnCancel.OnClick:=OnCancelPrinting;
               {Hide the cancel button if necessary.}
               if not ShowCancel then
               begin
                    fPPPrnPrgDlg.btnCancel.Enabled:=False;
                    fPPPrnPrgDlg.btnCancel.Visible:=False;
                    fPPPrnPrgDlg.ClientHeight:=(DefaultProgClientHeight*fPPPrnPrgDlg.PixelsPerInch) div 96;
               end;
          end;
          {Show it and bring it to the front.}
          fPPPrnPrgDlg.Show;
          {Update it as necessary.}
          with fPPPrnPrgDlg do
          begin
               Caption:=Title;
               if Status = '' then
                  lblStatus.Caption:=SendingPagesMsg
               else
                   lblStatus.Caption:=Status;
               ProgBar.Max:=Copies*(ToPage-FromPage+1);
               ProgBar.StepIt;
               lblPageNumber.Caption:='Page '+IntToStr(CurrentPage);
               if FromPage = 1 then
               begin
                    lblPageNumber.Caption:=lblPageNumber.Caption+' of '+IntToStr(ToPage-FromPage+1);
                    if Copies > 1 then
                       lblPageNumber.Caption:=lblPageNumber.Caption+' x '+IntToStr(Copies)+' Copies';
               end;
               if Showing then Update;
          end;
     end
     else
     begin
          {If it exists, get rid of it.}
          if fPPPrnPrgDlg <> nil then
          begin
               {If it is visible, close it.}
               if fPPPrnPrgDlg.Visible then fPPPrnPrgDlg.Close;
               fPPPrnPrgDlg.Free;
               fPPPrnPrgDlg := nil;
          end;
     end;
end;

function TPagePrinter.GetGutterTop: TMeasurement;
begin
     try
        Result:=PixelsToMeasureUnitsV(GetDeviceCaps(fPrinter.Handle, PHYSICALOFFSETY));
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultGutterTopIn
           else Result:=DefaultGutterTopMm;
     end;
end;

function TPagePrinter.GetGutterBottom: TMeasurement;
begin
{$IFDEF USE_SYMMETRIC_GUTTERS}
     Result:=GutterTop;
{$ELSE}
     Result:=PhysicalPageHeight-AvailablePageHeight-GutterTop;
{$ENDIF}
end;

function TPagePrinter.GetGutterLeft: TMeasurement;
begin
     try
        Result:=PixelsToMeasureUnitsH(GetDeviceCaps(fPrinter.Handle, PHYSICALOFFSETX));
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultGutterLeftIn
           else Result:=DefaultGutterLeftMm;
     end;
end;

function TPagePrinter.GetGutterRight: TMeasurement;
begin
{$IFDEF USE_SYMMETRIC_GUTTERS}
     Result:=GutterLeft;
{$ELSE}
     Result:=PhysicalPageWidth-AvailablePageWidth-GutterLeft;
{$ENDIF}
end;

function TPagePrinter.StartingLeft: TPixels;
begin
     Result:=MeasureUnitsToPixelsH(MarginLeft-GutterLeft);
end;

function TPagePrinter.StartingRight: TPixels;
begin
     Result:=MeasureUnitsToPixelsH(MarginRight-GutterRight);
end;

function TPagePrinter.StartingTop: TPixels;
begin
     Result:=MeasureUnitsToPixelsV(MarginTop-GutterTop);
end;

function TPagePrinter.StartingBottom: TPixels;
begin
     Result:=MeasureUnitsToPixelsV(MarginBottom-GutterBottom);
end;

function TPagePrinter.MeasureUnitsToPixelsH(const M: TMeasurement): TPixels;
var
  Temp: TMeasurement;
begin
     Temp := M;
     try
        if MeasureUnit = muMillimeters then Temp := M / 25.4;
        Result:=Round((Temp*GetDeviceCaps(fPrinter.Handle, LOGPIXELSX)));
     except
           on EPrinter do Result:=Round(Temp*DefaultDPI);
     end;
end;

function TPagePrinter.MeasureUnitsToPixelsV(const M: TMeasurement): TPixels;
var
  Temp: TMeasurement;
begin
     Temp := M;
     try
        if MeasureUnit = muMillimeters then Temp := M / 25.4;
        Result:=Round((Temp*GetDeviceCaps(fPrinter.Handle, LOGPIXELSY)));
     except
           on EPrinter do Result:=Round(Temp*DefaultDPI);
     end;
end;

function TPagePrinter.PixelsToMeasureUnitsH(const P: TPixels): TMeasurement;
begin
     try
        Result:=(P / GetDeviceCaps(fPrinter.Handle, LOGPIXELSX));
     except
           on EZeroDivide do Result:=P/DefaultDPI;
           on EPrinter do Result:=P/DefaultDPI;
     end;
     if MeasureUnit = muMillimeters then Result:=Result*25.4;
end;

function TPagePrinter.PixelsToMeasureUnitsV(const P: TPixels): TMeasurement;
begin
     try
        Result:=(P / GetDeviceCaps(fPrinter.Handle, LOGPIXELSY));
     except
           on EZeroDivide do Result:=P/DefaultDPI;
           on EPrinter do Result:=P/DefaultDPI;
     end;
     if MeasureUnit = muMillimeters then Result:=Result*25.4;
end;

procedure TPagePrinter.ParseFormatToken(var CurToken: String; var CurAlignment: TAlignment;
          var CurWidth: TMeasurement);
begin
     if CurToken = '' then CurToken:='<'+FloatToStr(DefaultColWidth);
     if Length(CurToken) = 1 then
        if (CurToken[1] in ['<', '^', '>']) then CurToken:=CurToken+FloatToStr(DefaultColWidth);

     {Alignment}
     case CurToken[1] of
          '<': begin
                    CurAlignment:=taLeftJustify;
                    Delete(CurToken, 1, 1);
               end;
          '^': begin
                    CurAlignment:=taCenter;
                    Delete(CurToken, 1, 1);
               end;
          '>': begin
                    CurAlignment:=taRightJustify;
                    Delete(CurToken, 1, 1);
               end;
     else
         CurAlignment:=taLeftJustify;
     end;

     {Width}
     try
        CurWidth:=StrToFloat(CurToken);
     except
           on EConvertError do CurWidth:=DefaultColWidth;
     end;
end;

function TPagePrinter.ExpandLogicalFields(S: String): String;
begin
     S:=ReplaceSubString(LineField, IntToStr(LineNumber), S);
     S:=ReplaceSubString(PageField, IntToStr(PageNumber), S);
     S:=ReplaceSubString(DateField, FormatDateTime('ddddd',Date), S);
     S:=ReplaceSubString(TimeField, FormatDateTime('tt',Time), S);
     S:=ReplaceSubString(TitleField, Title, S);
     Result:=S;
end;

procedure TPagePrinter.SetPageBorderOffset(Value: TMeasurement);
begin
     fPageBorderOffset:=Value;
end;

function TPagePrinter.GetPhysicalPageHeight: TMeasurement;
begin
     try
        Result:=PixelsToMeasureUnitsV(GetDeviceCaps(fPrinter.Handle, PHYSICALHEIGHT));
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultPhysicalPageHeightIn
           else Result:=DefaultPhysicalPageHeightMm;
     end;
end;

function TPagePrinter.GetPhysicalPageWidth: TMeasurement;
begin
     try
        Result:=PixelsToMeasureUnitsH(GetDeviceCaps(fPrinter.Handle, PHYSICALWIDTH));
     except
        on EPrinter do
           if MeasureUnit = muInches then Result:=DefaultPhysicalPageWidthIn
           else Result:=DefaultPhysicalPageWidthMm;
     end;
end;

function TPagePrinter.GetPrintableWidth: TMeasurement;
begin
     Result:=PhysicalPageWidth-MarginLeft-MarginRight;
end;

function TPagePrinter.GetPrintableHeight: TMeasurement;
begin
     Result:=PhysicalPageHeight-MarginTop-MarginBottom;
end;

procedure TPagePrinter.SetPrintToFile(Value: Boolean);
begin
     if Value <> fPrintToFile then
     begin
          if not Printing then
          begin
               fPrintToFile:=Value;
          end
          else
              raise EPagePrinter.Create('Can''t change PrintToFile while printing');
     end;
end;

function TPagePrinter.GetFileName: String;
begin
     Result:=fFileName;
end;

procedure TPagePrinter.SetFileName(Value: String);
begin
     if Value <> fFileName then
     begin
          if not Printing then
          begin
               fFileName:=Trim(Value);
          end
          else
              raise EPagePrinter.Create('Can''t change FileName while printing');
     end;
end;

{ New stuff for TPagePrinter}

function TPagePrinter.MeasureUnitsToScreenPixels(const Value: TMeasurement; Horz: Boolean): TPixels;
var
   Temp: TMeasurement;
   Index: Integer;
begin
     Temp := Value;
     if MeasureUnit = muMillimeters then Temp := Value / 25.4;
     if Horz then Index:=LOGPIXELSX else Index:=LOGPIXELSY;
     if not fStillCreating then
        Result:=Round(Temp*GetDeviceCaps(fPaintBox.Canvas.Handle, Index))
     else
         Result:=Round(Temp*Screen.PixelsPerInch);
end;

function TPagePrinter.ScaleValue(Value: TMeasurement; Horz: Boolean): TPixels;
begin
     Result:=MeasureUnitsToScreenPixels(Value, Horz);
     Result:=Round((Result*ZoomPercent)/100);
end;

procedure TPagePrinter.PaintPreview(Sender: TObject);
var
   PagePixelsWidth, PagePixelsHeight: TPixels;
   XOffset, YOffset: Integer;
   R, MarginRect: TRect;
begin
     if GradientBackground then
     begin
          {At some point in the future I'd like to use the new Win32 Gradient APIs...}
          FillGradient( fPaintBox.Canvas, fPaintBox.ClientRect,
                        clBlack, Color, goVertical);
     end;

     with fPaintBox.Canvas do
     begin
          PagePixelsHeight:=GetPreviewPagePixelsV;
          PagePixelsWidth:=GetPreviewPagePixelsH;
          {Calculate the page area.}
          XOffset:=(fPaintBox.Width - PagePixelsWidth) div 2;
          YOffset:=(fPaintBox.Height - PagePixelsHeight) div 2;
          R.Left := XOffset;
          R.Top := YOffset;
          R.Right := PagePixelsWidth+XOffset;
          R.Bottom := PagePixelsHeight+YOffset;
          MarginRect:=R; //Save this for later.
          Pen.Style := psSolid;
          if ShadowOffset > 0 then
          begin
               {Draw the shadow.}
               Brush.Color := ShadowColor;
               Pen.Color := ShadowColor;
               {Right Portion}
               Rectangle(R.Left+PagePixelsWidth, R.Top+ShadowOffset,
                     R.Right+ShadowOffset, R.Bottom+ShadowOffset);
               {Bottom Portion}
               Rectangle(R.Left+ShadowOffset, R.Top+PagePixelsHeight,
                     R.Right, R.Bottom+ShadowOffset);
          end;
          {Draw the page.}
          Brush.Color := clWhite;
          Pen.Color := clBlack;
          Rectangle(R.Left, R.Top, R.Right, R.Bottom);

          {Draw the current page on the canvas.}
          if (PageNumber >= 1) and (PageNumber <= PageCount) then
          begin
               {Shrink the Rect for the picture to go in}
               Inc(R.Left, ScaleValue(GutterLeft, True)+1);
               Inc(R.Top, ScaleValue(GutterTop, False)+1);
               Inc(R.Right, 1);
               Inc(R.Bottom, 1);
               {Get the picture and draw it.}
               StretchDraw(R, Pages[PageNumber]);
          end;

          {Draw the margins if necessary.}
          if ShowMargins then
          begin
               Pen.Style:=psDot;
               Pen.Color:=clSilver;
               Brush.Style:=bsClear;

               Inc(MarginRect.Left, ScaleValue(MarginLeft, True));
               Inc(MarginRect.Top, ScaleValue(MarginTop, False));
               Dec(MarginRect.Right, ScaleValue(MarginRight, True));
               Dec(MarginRect.Bottom, ScaleValue(MarginBottom, False));
               Rectangle( MarginRect.Left, MarginRect.Top,
                          MarginRect.Right, MarginRect.Bottom );
          end;
          {Leave the pen so the TPaintBox}
          {design time border doesn't show.}
          Pen.Color:=Color;
     end;
end;

function TPagePrinter.GetPageCount: Cardinal;
begin
     if (fPages <> nil) then
        Result := fPages.Count
     else
         Result := 0;
end;

function TPagePrinter.GetZoomPercent: Cardinal;
begin
     {C++Builder 1.0 doesn't like Cardinal properties
     (or String properties) with a SetXXX procedure and
     no GetXXX function.  It will always read 0 in that
     situation.  However, Cardinal properties with no
     Get and no Set as well as properties with both
     Get and Set functions work. Thus the existence
     of this little function.} 
     Result:=fZoomPercent;
end;

procedure TPagePrinter.SetZoomPercent(Value: Cardinal);
begin
     fZoomPercent:=Value;
     UpdatePagePreviewSize;
     Invalidate;
end;

procedure TPagePrinter.ResetPageList(CreateForReal: Boolean);
begin
     FinishPrintPage;
     {Destroy the old list and all pages it contains.}
     fPages.Free;
     fPage := nil;
     {Create a new, empty list.}
     fPages:=TPageList.Create;
     fPageNumber:=0;
     {Create a new page so all the Canvas functions work.}
     if CreateForReal then
        NewPrintPage
     else
         CreateTempPage;
end;

procedure TPagePrinter.NewPrintPage;
begin
     CreateTempPage;
     fUsingTempPage:=False;
     fPages.Add(fPage);
end;

procedure TPagePrinter.FinishPrintPage;
begin
     fCanvas.Free;
     fCanvas:=nil;
     if fUsingTempPage then fPage.Free;
end;

procedure TPagePrinter.CreateTempPage;
begin
     FinishPrintPage;
     fUsingTempPage:=True;
     fPage:=TPrintPage.Create;
     {We MUST set the page height and width before we create the canvas.}
     fPage.Width:=MeasureUnitsToPixelsH(PhysicalPageWidth-GutterRight);
     fPage.Height:=MeasureUnitsToPixelsV(PhysicalPageHeight-GutterBottom);

     fCanvas:=TPrintCanvas.Create(fPage, GetPrinterHandle);
     fCanvas.Brush.Style:=bsClear;
     fCanvas.Font.Assign(Font);
     SetPixelsPerInch;
end;

procedure TPagePrinter.ZoomToFit;
var
   PagePixelsWidth, PagePixelsHeight: TPixels;
begin
     PagePixelsHeight:=GetPreviewPagePixelsV;
     PagePixelsWidth:=GetPreviewPagePixelsH;
     if (PagePixelsWidth/PagePixelsHeight) > (ClientWidth/ClientHeight) then
        ZoomToWidth
     else
         ZoomToHeight;
end;

procedure TPagePrinter.ZoomToWidth;
var
   ScrollWidth: Integer;
begin
     ScrollWidth:=GetSystemMetrics(SM_CXVSCROLL);
     ZoomPercent:=Trunc(((100*(ClientWidth-ScrollWidth-2*(ShadowOffset+1)))*GetScaleFactor(True))/MeasureUnitsToPixelsH(PhysicalPageWidth));
end;

procedure TPagePrinter.ZoomToHeight;
var
   ScrollHeight: Integer;
begin
     ScrollHeight:=GetSystemMetrics(SM_CYHSCROLL);
     ZoomPercent:=Trunc(((100*(ClientHeight-ScrollHeight-2*(ShadowOffset+1)))*GetScaleFactor(False))/MeasureUnitsToPixelsV(PhysicalPageHeight));
end;

function TPagePrinter.GetScaleFactor(Horz: Boolean): Double;
var
   Index: Integer;
begin
     Result:=1;
     if not fStillCreating then
     begin
          if Horz then Index:=LOGPIXELSX
          else Index:=LOGPIXELSY;

          try
             Result:=GetDeviceCaps(fPrinter.Handle, Index) / GetDeviceCaps(fPaintBox.Canvas.Handle, Index);
          except
                on EPrinter do Result:=1;
          end;
     end;
end;

function TPagePrinter.GetPreviewPagePixels(Horz: Boolean): TPixels;
var
   Index: Integer;
   Measure: TMeasurement;
begin
     if Horz then Index:=PHYSICALWIDTH
     else Index:=PHYSICALHEIGHT;

     try
        Result:=GetDeviceCaps(fPrinter.Handle, Index);
     except
           on EPrinter do
           begin
                if MeasureUnit = muInches then
                begin
                     if Horz then Measure:=DefaultPhysicalPageWidthIn
                     else Measure:=DefaultPhysicalPageHeightIn;
                end
                else
                begin
                     if Horz then Measure:=DefaultPhysicalPageWidthMm
                     else Measure:=DefaultPhysicalPageHeightMm;
                end;
                Result:=MeasureUnitsToScreenPixels(Measure, Horz);
           end;
     end;
     Result:=Round((Result*ZoomPercent/GetScaleFactor(Horz))/100);
end;

function TPagePrinter.GetPreviewPagePixelsH: TPixels;
begin
     Result:=GetPreviewPagePixels(True);
end;

function TPagePrinter.GetPreviewPagePixelsV: TPixels;
begin
     Result:=GetPreviewPagePixels(False);
end;

procedure TPagePrinter.SetPageNumber(Value: Cardinal);
begin
     if (Value >= 1) and (Value <= PageCount) then
     begin
          fPageNumber:=Value;
          UpdatePagePreviewSize;
          Invalidate;
     end
     else
     begin
          if (csLoading in ComponentState) or (PageCount = 0) then
             fPageNumber:=0
          else
              raise EPagePrinter.Create('PageNumber must be between 1 and '+IntToStr(PageCount));
     end;
end;

procedure TPagePrinter.SetShadowOffset(Value: TPixels);
begin
     if Value <> fShadowOffset then
     begin
          fShadowOffset:=Value;
          UpdatePagePreviewSize;
          Invalidate;
     end;
end;

procedure TPagePrinter.UpdatePagePreviewSize;
begin
     {Setup the scrolling region.}
     HorzScrollBar.Range := GetPreviewPagePixelsH+2*(ShadowOffset+1);
     VertScrollBar.Range := GetPreviewPagePixelsV+2*(ShadowOffset+1);
     if not fStillCreating then
     begin
          case ZoomLocation of
               zlTopLeft:
               begin
                    {Sometimes nothing happens if I go straight to 0;}
                    HorzScrollBar.Position := 1;
                    HorzScrollBar.Position := 0;
                    VertScrollBar.Position := 1;
                    VertScrollBar.Position := 0;
               end;
               zlTopCenter: //Center the page horizontally and go to top vertically.
               begin
                    HorzScrollBar.Position := (HorzScrollBar.Range - ClientWidth) div 2;
                    VertScrollBar.Position := 1;
                    VertScrollBar.Position := 0;
               end;
               zlCenter:
               begin
                    HorzScrollBar.Position := (HorzScrollBar.Range - ClientWidth) div 2;
                    VertScrollBar.Position := (VertScrollBar.Range - ClientHeight) div 2;
               end;
          end;
     end;
end;

procedure TPagePrinter.Loaded;
begin
     inherited Loaded;
     RefreshProperties;
     ResetPageList(False);
     UpdatePagePreviewSize;
     {Make it so the TPaintBox design time border is invisible.}
     fPaintBox.Canvas.Pen.Color:=Color;
     Invalidate;
end;

procedure TPagePrinter.CMFontChanged(var Msg: TMessage);
begin
     inherited;
     Canvas.Font.Assign(Font);
     SetPixelsPerInch;
     GetTextMetrics(GetPrinterHandle, fTextMetrics);
     {Force fLineSpace to be updated.}
     SetLineSpacing(LineSpacing);
     if AutoHeaderFont then HeaderFont:=Font;
     if AutoFooterFont then FooterFont:=Font;
end;

procedure TPagePrinter.SetShadowColor(Value: TColor);
begin
     if Value <> fShadowColor then
     begin
          fShadowColor:=Value;
          Invalidate;
     end;
end;

procedure TPagePrinter.SetShowMargins(Value: Boolean);
begin
     if Value <> fShowMargins then
     begin
          fShowMargins:=Value;
          Invalidate;
     end;
end;

procedure TPagePrinter.SetGradientBackground(Value: Boolean);
begin
     if Value <> fGradientBackground then
     begin
          fGradientBackground:=Value;
          Invalidate;
     end;
end;

procedure TPagePrinter.UpdateDesigner;
begin
     if csDesigning in ComponentState then
        if (GetParentForm(Self) <> nil) then
           if (GetParentForm(Self).Designer <> nil) then
              GetParentForm(Self).Designer.Modified;
end;

procedure TPagePrinter.ExpandFriendlyFormat(const UserFmt: String; AsHeader: Boolean);
var
   LayoutTokens: TStringList;
   NextCharIsSpecifier, TextAddOn: Boolean;
   i: Integer;
   FmtText, FmtLayout, Value: String;
begin
    LayoutTokens:=TStringList.Create;
    try
       NextCharIsSpecifier := False;
       FmtText:='';
       for i:=1 to Length(UserFmt) do
       begin
            {If it's an &, the next char is important.}
            if (UserFmt[i] = '&') then
               NextCharIsSpecifier := True
            {The current char is a specifier of some sort.}
            else if NextCharIsSpecifier then
            begin
                 Value := '';
                 TextAddOn := True;
                 case UserFmt[i] of
                     'f': Value := ExtractFileName(FileName);
                     'F': Value := FileName;
                     'd': Value := FormatDateTime('ddddd', Date);
                     't': Value := FormatDateTime('t', Time);
                     'D': Value := FormatDateTime('dddddd', Date);
                     'T': Value := FormatDateTime('tt', Time);
                     'p': Value := PageField;
                     'i': Value := TitleField;
                     '&': Value := '&';
                     'l': begin Value := '<'; TextAddOn := False; end;
                     'c': begin Value := '^'; TextAddOn := False; end;
                     'r': begin Value := '>'; TextAddOn := False; end;
                 end;
                 {An empty filename will throw things off so we put in a space.}
                 if Value = '' then Value:=' ';

                 if TextAddOn then
                    FmtText:=FmtText+Value
                 else //We encountered an alignment specifier.
                 begin
                      if FmtText <> '' then
                      begin
                           if LayoutTokens.Count = 0 then LayoutTokens.Add('<');
                           FmtText:=FmtText+TokenSeparator;
                      end;
                      LayoutTokens.Add(Value);
                 end;
                 NextCharIsSpecifier := False;
            end
            {Otherwise we just add it on the Text string.}
            else
                FmtText:=FmtText+UserFmt[i];
       end;

       {Now build the Layout string;}
       FmtLayout:='';
       if LayoutTokens.Count = 0 then LayoutTokens.Add('<');
       Value := Trim(FloatToStrF((PrintableWidth / LayoutTokens.Count), ffFixed, 8, 2));
       for i:=0 to LayoutTokens.Count-1 do
       begin
            if FmtLayout <> '' then FmtLayout:=FmtLayout+TokenSeparator;
            FmtLayout:=FmtLayout+LayoutTokens[i]+Value;
       end;
    finally
           LayoutTokens.Free;
    end;

    {Now set the header or footer properties.}
    if AsHeader then
    begin
         Header:=FmtText;
         HeaderFormat:=FmtLayout;
    end
    else
    begin
         Footer:=FmtText;
         FooterFormat:=FmtLayout;
    end;
end;

function TPagePrinter.GetFriendlyFooter: String;
begin
     Result:=fFriendlyFooter;
end;

procedure TPagePrinter.SetFriendlyFooter(Value: String);
begin
     fFriendlyFooter:=Value;
     if not (csLoading in ComponentState) then
        ExpandFriendlyFormat(Value, False);
end;

function TPagePrinter.GetFriendlyHeader: String;
begin
     Result:=fFriendlyHeader;
end;

procedure TPagePrinter.SetFriendlyHeader(Value: String);
begin
     fFriendlyHeader:=Value;
     if not (csLoading in ComponentState) then
        ExpandFriendlyFormat(Value, True);
end;

function TPagePrinter.GetPrinterHandle: HDC;
begin
     try
        Result:=fPrinter.Handle;
     except
           on EPrinter do Result:=0;
     end;
end;

procedure TPagePrinter.SetCopies(Value: Cardinal);
begin
     if Value > 0 then
        fCopies:=Value
     else
         fCopies:=1;
end;

function TPagePrinter.GetCopies: Cardinal;
begin
     GetCopies:=fCopies;
end;

procedure TPagePrinter.SetCollate(Value: Boolean);
begin
     fCollate:=Value
end;

function TPagePrinter.GetPrintToPage: Cardinal;
begin
     Result:=fPrintToPage;
end;

procedure TPagePrinter.SetPrintToPage(Value: Cardinal);
begin
     if Value <> fPrintToPage then
     begin
          fPrintToPage:=Value;
          if fPrintToPage < fPrintFromPage then
             PrintFromPage:=fPrintToPage;
     end;
end;

function TPagePrinter.GetPrintFromPage: Cardinal;
begin
     Result:=fPrintFromPage;
end;

procedure TPagePrinter.SetPrintFromPage(Value: Cardinal);
begin
     if Value <> fPrintFromPage then
     begin
          fPrintFromPage:=Value;
          if fPrintFromPage > fPrintToPage then
             PrintToPage:=fPrintFromPage;
     end;
end;

type //This type is only used in this function.
  ECancelPrinting = class(EPagePrinter);

function TPagePrinter.Print: Boolean;
    procedure DoPrintPage(Pg, FromPage, ToPage: Integer; LastPage: Boolean);
    begin
         {Cancel printing if necessary.}
         if ShowProgress and ShowCancel then
            Application.ProcessMessages;
         if fCancelPrinting then
            raise ECancelPrinting.Create('Printing Cancelled');
            
         {Now print the page.}
         UpdateProgressDlg(ProgressMessage, Pg, FromPage, ToPage);
         fPrinter.Canvas.Draw(0, 0, Pages[Pg]);
         if not LastPage then fPrinter.NewPage;
    end;    
var
   Cp, Pg, PrevPrnCopies: Integer;
   ToPage, FromPage: Cardinal;
begin
     {Return True unless printing was cancelled.}
     Result:=True;
     {Call EndDoc if necessary.}
     if Printing then EndDoc;

     if PageCount > 0 then
     begin
          {Determine the page range to print.}
          if PrintFromPage = 0 then
             FromPage:=1
          else
              FromPage:=Minimum(PrintFromPage, PageCount);
          if PrintToPage = 0 then
             ToPage:=PageCount
          else
              ToPage:=Minimum(PrintToPage, PageCount);

          {We're handling the copies not the printer.}
          PrevPrnCopies:=fPrinter.Copies;
          fPrinter.Copies:=1;
          
          {Set up the print cancelling code.}
          fCancelPrinting:=False;
          try
             {Print the pages on the printer using Collate and Copies.}
             try
                fPrinter.BeginDoc;
                if Collate then //Print 1,2,3; 1,2,3
                begin
                     for Cp:=1 to Copies do
                         for Pg:=FromPage to ToPage do
                             DoPrintPage(Pg, FromPage, ToPage, ((Pg=ToPage) and (Cp=Copies)));
                end
                else //Print 1,1; 2,2; 3,3
                begin
                     for Pg:=FromPage to ToPage do
                         for Cp:=1 to Copies do
                             DoPrintPage(Pg, FromPage, ToPage, ((Pg=ToPage) and (Cp=Copies)));
                end;
                fPrinter.EndDoc;
             finally
                    UpdateProgressDlg(ProgressFinishMsg, 0, 0, 0);
                    if not fPrinter.Printing then
                       fPrinter.Copies:=PrevPrnCopies;
             end;
          except
                on ECancelPrinting do
                begin
                     Result:=False; //Return False since is was cancelled.
                     if fPrinter.Printing then
                     begin
                          {I'd like to always Abort printing,
                          but I've found that calling Abort
                          multiple times will crash programs.
                          Thus you can optionally use EndDoc.}
                          if AbortOnCancel then fPrinter.Abort
                          else fPrinter.EndDoc;
                     end;
                end;
          end;
     end;
end;

function TPagePrinter.GetHeaderFont: TFont;
begin
     Result:=fHeaderFont;
end;

procedure TPagePrinter.SetHeaderFont(Value: TFont);
begin
     fHeaderFont.Assign(Value);
end;

function TPagePrinter.GetFooterFont: TFont;
begin
     Result:=fFooterFont;
end;

procedure TPagePrinter.SetFooterFont(Value: TFont);
begin
     fFooterFont.Assign(Value);
end;

procedure TPagePrinter.SetAutoHeaderFont(Value: Boolean);
begin
     if fAutoHeaderFont <> Value then
     begin
          fAutoHeaderFont:=Value;
          if not (csLoading in ComponentState) then
             if fAutoHeaderFont then
                HeaderFont:=Font;
     end;
end;

procedure TPagePrinter.SetAutoFooterFont(Value: Boolean);
begin
     if fAutoFooterFont <> Value then
     begin
          fAutoFooterFont:=Value;
          if not (csLoading in ComponentState) then
             if fAutoFooterFont then
                FooterFont:=Font;
     end;
end;

procedure TPagePrinter.Invalidate;
begin
     if fUpdateRefCount = 0 then
        inherited Invalidate;
end;

procedure TPagePrinter.BeginUpdate;
begin
     Inc(fUpdateRefCount);
end;

procedure TPagePrinter.EndUpdate;
begin
     if fUpdateRefCount > 0 then Dec(fUpdateRefCount);
     if fUpdateRefCount = 0 then Invalidate;
end;

function TPagePrinter.GetCanvasPosition: TPoint;
begin
     Result:=Point(fCurrentX+StartingLeft, fCurrentY+StartingTop);
end;

function TPagePrinter.GetPages(Indx: Cardinal): TPrintPage;
begin
     Result:=fPages.GetPage(Indx-1);
end;

function TPagePrinter.GetProgressMessage: String;
begin
     Result:=fProgressMessage;
end;

procedure TPagePrinter.SetProgressMessage(Value: String);
begin
     fProgressMessage:=Value;
end;

function TPagePrinter.GetCanvas: TPrintCanvas;
begin
     Result:=fCanvas;
end;

function TPagePrinter.GetLineNumber: Cardinal;
begin
     Result:=fLineNumber;
end;

function TPagePrinter.GetAutoFooterFont: Boolean;
begin
     Result:=fAutoFooterFont;
end;

function TPagePrinter.GetAutoHeaderFont: Boolean;
begin
     Result:=fAutoHeaderFont;
end;

function TPagePrinter.GetCollate: Boolean;
begin
     Result:=fCollate;
end;

function TPagePrinter.GetDefaultColWidth: TMeasurement;
begin
     Result:=fDefaultColWidth;
end;

function TPagePrinter.GetGradientBackground: Boolean;
begin
     Result:=fGradientBackground;
end;

function TPagePrinter.GetLineSpacing: TLineSpacing;
begin
     Result:=fLineSpacing;
end;

function TPagePrinter.GetMarginBottom: TMeasurement;
begin
     Result:=fMarginBottom;
end;

function TPagePrinter.GetMarginLeft: TMeasurement;
begin
     Result:=fMarginLeft;
end;

function TPagePrinter.GetMarginRight: TMeasurement;
begin
     Result:=fMarginRight;
end;

function TPagePrinter.GetMarginTop: TMeasurement;
begin
     Result:=fMarginTop;
end;

function TPagePrinter.GetMeasureUnit: TMeasureUnit;
begin
     Result:=fMeasureUnit;
end;

function TPagePrinter.GetPageBorderOffset: TMeasurement;
begin
     Result:=fPageBorderOffset;
end;

function TPagePrinter.GetPrintToFile: Boolean;
begin
     Result:=fPrintToFile;
end;

function TPagePrinter.GetShadowColor: TColor;
begin
     Result:=fShadowColor;
end;

function TPagePrinter.GetShadowOffset: TPixels;
begin
     Result:=fShadowOffset;
end;

function TPagePrinter.GetShowMargins: Boolean;
begin
     Result:=fShowMargins;
end;

function TPagePrinter.GetShowProgress: Boolean;
begin
     Result:=fShowProgress;
end;

function TPagePrinter.GetPageBorders: TPageBorders;
begin
     Result:=fPageBorders;
end;

procedure TPagePrinter.SetPageBorders(Value: TPageBorders);
begin
     fPageBorders:=Value;
end;

procedure TPagePrinter.OnCancelPrinting(Sender: TObject);
begin
     fCancelPrinting:=True;
end;

procedure TPagePrinter.SetShowProgress(Value: Boolean);
begin
     fShowProgress:=Value;
end;

function TPagePrinter.GetShowCancel: Boolean;
begin
     Result:=fShowCancel;
end;

procedure TPagePrinter.SetShowCancel(Value: Boolean);
begin
     fShowCancel:=Value;
end;

function TPagePrinter.StoreFooterAndFormat: Boolean;
begin
     Result:=FriendlyFooter = '';
end;

function TPagePrinter.StoreFooterFont: Boolean;
begin
     Result:=not AutoFooterFont;
end;

function TPagePrinter.StoreHeaderAndFormat: Boolean;
begin
     Result:=FriendlyHeader = '';
end;

function TPagePrinter.StoreHeaderFont: Boolean;
begin
     Result:=not AutoHeaderFont;
end;

function TPagePrinter.ValidateFormatString(const Fmt: String; const ConvertUnits: Boolean): String;
var
   FormatTokens: TStringList;
   i: Integer;
   CurAlignment: TAlignment;
   CurWidth: TMeasurement;
   AlignmentChar: Char;
   Buffer: String;
begin
     Result:='';
     if Fmt <> '' then
     begin
          FormatTokens:=TStringList.Create;
          try
             TokenizeString(Fmt, TokenSeparator, FormatTokens);
             for i:=0 to FormatTokens.Count-1 do
             begin
                  Buffer:=FormatTokens[i];
                  ParseFormatToken(Buffer, CurAlignment, CurWidth);
                  if Result <> '' then Result:=Result+TokenSeparator;
                  case CurAlignment of
                       taRightJustify: AlignmentChar:='>';
                       taCenter: AlignmentChar:='^';
                  else
                      AlignmentChar:='<';
                  end;
                  if ConvertUnits then
                  begin
                       if MeasureUnit = muInches then CurWidth:=CurWidth/25.4
                       else CurWidth:=CurWidth*25.4;
                  end;
                  Result:=Result+AlignmentChar+Trim(FloatToStr(CurWidth));
             end;
          finally
                 FormatTokens.Free;
          end;
     end;
end;

function TPagePrinter.GetFooterFormat: String;
begin
     Result:=fFooterFormat;
end;

procedure TPagePrinter.SetFooterFormat(Value: String);
begin
     fFooterFormat:=ValidateFormatString(Value, false);
end;

function TPagePrinter.GetHeaderFormat: String;
begin
     Result:=fHeaderFormat;
end;

procedure TPagePrinter.SetHeaderFormat(Value: String);
begin
     fHeaderFormat:=ValidateFormatString(Value, false);
end;

function TPagePrinter.GetTableFormat: String;
begin
     Result:=fTableFormat;
end;

procedure TPagePrinter.SetTableFormat(Value: String);
begin
     fTableFormat:=ValidateFormatString(Value, false);
end;

end.

