{-----------------------------------------------------------------------------}
{ A form to provide gradient filled caption bars ala Microsoft Office.        }
{ Copyright 1996, Brad Stowers.  All Rights Reserved.                         }
{ This component can be freely used and distributed in commercial and private }
{ environments, provided this notice is not modified in any way and there is  }
{ no charge for it other than nomial handling fees.  Contact me directly for  }
{ modifications to this agreement.                                            }
{-----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at bstowers@pobox.com (preferred) or 72733,3374 on CompuServe.              }
{ The lateset version will always be available on the web at:                 }
{   http://www.pobox.com/~bstowers/delphi/                                    }
{-----------------------------------------------------------------------------}
{ Date last modified:  1/6/96                                                 }
{-----------------------------------------------------------------------------}

{ ----------------------------------------------------------------------------}
{ TGradientForm v1.12                                                         }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A form that paints it's caption bar in a gradient pattern, like the new   }
{   Microsoft Office applications.  It starts with black and moves gradually  }
{   to the system defined color.                                              }
{ Notes:                                                                      }
{   * Be aware that this form has had some problems in MDI applications.  I   }
{     think I have worked them out, but I suggest you test everything very    }
{     thoroughly.  I used a small hack to do it (see GradClientWndProc).  It  }
{     flickers a lot now when changing active MDI child windows when          }
{     maximized, but it's better than before.                                 }
{     that, but under the right circumstances, it does draw incorrectly and   }
{     that looks very unprofessional.                                         }
{   * The best way to use this form is to add it to your Object Repository.   }
{     Simply open this unit in Delphi, right click on the form and select Add }
{     To Repository.  Then, when you want a TGradientForm, you just select it }
{     from the repository (File | New) and use the "Inherit" option so you    }
{     don't have to see all this code in your form.                           }
{     If you have existing forms that you want converted to gradient forms,   }
{     simply add "GradForm" to your "Uses" clause, and change your form's     }
{     ancestor to TGradientForm.  An example:                                 }
{        Change:                                                              }
{           TMyForm = class(TForm)                                            }
{        To:                                                                  }
{           TMyForm = class(TGradientForm)                                    }
{   * Special thanks go to Michiel Ouwehand of Epic MegaGames for the         }
{     clipping region tips (See the WMNCPaint method) and for pointing out    }
{     the DrawFrameControl API function (see the PaintCaptionButtons method). }
{   * I've used strictly GDI calls for the painting in this component.  No    }
{     TCanvas, TBitmap, TBrush, etc.  This is because that although they are  }
{     extremely nice to use, they are not nearly as efficient.  That's not a  }
{     slam on them.  They have to be able to know how to do a lot of things,  }
{     and that requires overhead.  I have a very specific set of things to do }
{     here, and I am *very* interested in getting it to do it as fast as      }
{     possible, so I'm willing to sacrifice the convenience of the classes    }
{     for the speed of the API.  One day I'll sit down and do a speed         }
{     comparison to see if I really gained that much this way.  If you don't  }
{     understand the GDI calls, leave them alone, or use them to experiment   }
{     with in learning how to use them.                                       }
{   * This form will only work on Delphi 2.  I have used very few calls that  }
{     would prevent it from working on Delphi 1, so it will be fairly simple  }
{     to convert to Delphi 1, but there is a lot of stuff that would have to  }
{     be changed to make it look right.  For instance, I paint an icon in the }
{     left corner for the system menu.  This is a Win95 style only.  Also, I  }
{     allow for BorderStyles like bsToolWindow that don't exist in Delphi 1.  }
{     Converting to Delphi 1 could be done, and shouldn't be all that         }
{     difficult since all the painting routines should work fine except for   }
{     the icon painting, and you don't need that for Win 3.1x anyway.         }
{     I just don't do any Delphi 1 development any more, so I leave it to one }
{     of you to implement.  If you do, I would appreciate it if you could do  }
{     it using $IFDEF WIN32, and send me the changes so others can use it as  }
{     well.                                                                   }
{                                                                             }
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 1.00:  + Initial release                                                    }
{ 1.01:  + Fixed the inactive flashing when GradientOnBackground is set.  It  }
{          helps if you read the docs instead of assuming you know what it    }
{          says.... grr....                                                   }
{        + Fixed problem with incorrect button drawn on maximized windows.    }
{          Now correctly draws restore button if window is maximized.         }
{        + Redid button size code.  Should be right now, but still not 100%   }
{          sure.  The GetSystemMetrics return values don't seem to be correct.}
{ 1.02:  + Added some new painting routines.  They are controlled below with  }
{          the conditional defines PAINT-OLD-WAY, STRETCH-WITH-BRUSH and      }
{          STRETCH-WITH-PEN. Please experiment with them and tell me which    }
{          you find to be the fastest.                                        }
{ 1.03:  + Fixed problem with redrawing inactive window when                  }
{          GradientOnInactive set to FALSE.                                   }
{        + Fixed problem of gradient disappearing if Caption set at run-time. }
{        + Fixed all the MDI bugs that I know about.                          }
{        + Gradient would goof up in some cases where window was sized very   }
{          small horizontally.  Fixed.                                        }
{        + Fiddled with the button drawing some more and I think the sizes    }
{          are finally right under all circumstances.                         }
{        + Didn't recreate the caption font if the user changed it.           }
{          (WM_SETTINGCHANGE)  I knew about this one for a while, just kept   }
{          fogetting to fix it...                                             }
{        + Resizing window so that it was too small for the caption text to   }
{          fit in the available space caused it to write under the buttons.   }
{          It now properly substitutes ellipsis ("...") if the string won't   }
{          fit.  I was dreading this one until I found that the Delphi Win32  }
{          docs don't document a DrawText flag (DT_END_ELLIPSIS) that will    }
{          do this for you automagically.  Happy day. :)                      }
{ 1.04:  + Found a way to get a nice 16x16 version of the icon.  Icon is now  }
{          drawn much better than before.                                     }
{ 1.10:  + Fixed problem that could cause the caption to be painted all black }
{          when first created.                                                }
{        + Added OnPaintCaption property.  Makes event available to paint on  }
{          the caption after the gradient, icon, and buttons have been drawn, }
{          but before the text is.  See demo program for example of use.      }
{ 1.11:  + Fixed problem for people who want gradient MDI child windows.      }
{        + MDI child caption text changing at runtime when child window was   }
{          maximized would not update caption text.  Fixed.                   }
{        + Got rid of some of the old painting code that wasn't used, as it   }
{          didn't work very well anyway.  There are now two painting routines }
{          (FillRectGradientHigh, FillRectGradientLow) for high color mode    }
{          and 256 or less color mode.  The former is faster, but doesn't     }
{          work on less than high color mode (16-bit depth) because it is     }
{          "palette stupid".                                                  }
{ 1.12:  + Another MDI bug.  If the MDI child form is not a TGradientForm     }
{          descendant, the caption text will not update if the child is       }
{          maximized and the child's caption text is changed.  There is a     }
{          work-around for this, but it causes an annoying flicker which I    }
{          can not stand.  If you can stand it, look at the GradClientWndProc }
{          method and uncomment the code there.  The better solution, I       }
{          think, is to make your MDI child form's a descendant of            }
{          TGradientForm.  For those of you who want your app to be "just     }
{          like Word", you can use the new FPaintGradient property (below)    }
{          so that the child windows are never painted with a gradient. This  }
{          solution doesn't suffer from the flicker problem.                  }
{        + Changed FGradientOnInactive boolean property to FPaintGradient set }
{          property.  Three possible values: gfpAlways, gfpActive and         }
{          gfpNever.                                                          }
{        + Cleaned up some code that was now longer necessary.  May have sped }
{          up the paint time by a nanosecond or two.  :)                      }
{ ----------------------------------------------------------------------------}

unit GradForm;

// Only define one of the following.

{$IFNDEF WIN32}
  ERROR!  This unit only available for Win32!
{$ENDIF}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

const
  MIN_GRADIENT_COLORS = 8;   // Any less than 8 colors doesn't look much like a gradient.
  MAX_GRADIENT_COLORS = 256; // Any more than 256 colors is not noticeable, and just
                             // slows the painting down.
  DEF_GRADIENT_COLORS = 64;  // This is a good compromise between speed and appearance.

type
  TGFPaintWhen = (gfpAlways, // Always draw the gradient caption
                  gfpActive, // Only draw the gradient when the form is active
                  gfpNever); // Never draw the gradient
  TGFOnCaptionPaint = procedure(Sender: TObject; const Canvas: TCanvas;
                                var R: TRect) of object;
  TGradientForm = class(TForm)
  private
    // Internal variables
    Colors: array[0..1, 0..MAX_GRADIENT_COLORS-1] of TColorRef;
    CaptionFont: HFONT;
    FGradDefClientProc: TFarProc;
    FGradClientInstance: TFarProc;
    // Property variables
    FGradientColors: integer;
    FPaintGradient: TGFPaintWhen;
    FCaptionText: string;
    FOnCaptionPaint: TGFOnCaptionPaint;

    // Internal methods
    function IsActiveWindow: boolean;
    procedure CalculateColors;
    procedure CreateCaptionFont;
    function GetCaptionRect: TRect;
    function DrawCaption(FormDC: HDC; Active: boolean): TRect;
    procedure PaintMenuIcon(DC: HDC; var R: TRect; Active: boolean);
    procedure FillRectSolid(DC: HDC; const R: TRect; Active: boolean);
    procedure FillRectGradientHigh(DC: HDC; const R: TRect; Active: boolean);
    procedure FillRectGradientLow(DC: HDC; const R: TRect; Active: boolean);
    procedure PaintCaptionText(DC: HDC; R: TRect);
    procedure PaintCaptionButtons(DC: HDC; var Rect: TRect);
    // Window message handlers
    procedure WMNCActivate(var Msg: TWMNCActivate); message WM_NCACTIVATE;
    procedure WMNCPaint(var Msg: TMessage); message WM_NCPAINT;
    procedure WMSysColorChange(var Msg: TWMSysColorChange); message WM_SYSCOLORCHANGE;
    procedure WMSize(var Msg: TWMSize); message WM_SIZE;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSetText(var Msg: TWMSetText); message WM_SETTEXT;
    procedure WMGetText(var Msg: TWMGetText); message WM_GETTEXT;
    procedure WMGetTextLength(var Msg: TWMGetTextLength); message WM_GETTEXTLENGTH;
    procedure WMSettingChange(var Msg: TMessage); message WM_SETTINGCHANGE;
    // MDI Client Window Procedure
    procedure GradClientWndProc(var Message: TMessage);
  protected
    // Overriden methods
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    // Property methods
    procedure SetGradientColors(Val: integer);
    procedure SetPaintGradient(Val: TGFPaintWhen);
    procedure SetCaptionText(const Val: string);
  public
    procedure Draw(Active: boolean);
    // Overridden methods
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    // Properties
    property Caption: string read FCaptionText write SetCaptionText stored TRUE;
    // Determines the number of colors used to paint the gradient pattern.
    property GradientColors: integer
             read FGradientColors write SetGradientColors default DEF_GRADIENT_COLORS;
    //Determines if/when we should paint the gradient.
    property PaintGradient: TGFPaintWhen
             read FPaintGradient write SetPaintGradient;
    property OnCaptionPaint: TGFOnCaptionPaint
             read FOnCaptionPaint write FOnCaptionPaint;
  end;


implementation

{$R *.DFM}

// A variant record (known as a union in C) to allow easy access to the individual red,
// green, and blue values of a TColorRef (RGB) value.
type
  TRGBMap = packed record
    case boolean of
      TRUE:  ( RGBVal: DWORD );
      FALSE: ( Red,
               Green,
               Blue,
               Unused: byte );
  end;


constructor TGradientForm.Create(AOwner: TComponent);
begin
  // We set our new property values first so that they will be valid in the OnCreate
  // event handler.  The inherited Create is what calls that event, so we set up first.

  // Set the number of colors to use to create the gradient fill.
  FGradientColors := DEF_GRADIENT_COLORS;
  // Should we paint the gradient when window is inactive.
  FPaintGradient := gfpAlways;
  FOnCaptionPaint := NIL;

  // Calculate the colors we need to paint the gradient.
  CalculateColors;
  // Create a font for the caption bar.
  CaptionFont := 0;
  CreateCaptionFont;

  inherited Create(AOwner);
end;

destructor TGradientForm.Destroy;
begin
  // Clean up the font we created.
  if CaptionFont <> 0 then
    DeleteObject(CaptionFont);
  inherited Destroy;
end;

procedure TGradientForm.CreateWnd;
begin
  inherited CreateWnd;
  if FormStyle = fsMDIForm then begin
    FGradClientInstance := MakeObjectInstance(GradClientWndProc);
    FGradDefClientProc := Pointer(GetWindowLong(ClientHandle, GWL_WNDPROC));
    SetWindowLong(ClientHandle, GWL_WNDPROC, Longint(FGradClientInstance));
  end;
end;

procedure TGradientForm.DestroyWnd;
begin
  if FormStyle = fsMDIForm then begin
    SetWindowLong(ClientHandle, GWL_WNDPROC, Longint(FGradDefClientProc));
    FreeObjectInstance(FGradClientInstance);
  end;
  inherited;
end;

procedure TGradientForm.SetGradientColors(Val: integer);
begin
  if (Val = FGradientColors) or
     (Val < MIN_GRADIENT_COLORS) or (Val > MAX_GRADIENT_COLORS) then exit;
  FGradientColors := Val;
  // The number of colors have changes, we need to recalculate the colors we use to paint.
  CalculateColors;
  // Make the non client area repaint.
  SetWindowPos(Handle, 0, 0, 0, 0, 0, SWP_DRAWFRAME or SWP_NOACTIVATE or SWP_NOMOVE or
                                      SWP_NOSIZE or SWP_NOZORDER);
end;

procedure TGradientForm.SetPaintGradient(Val: TGFPaintWhen);
begin
  if FPaintGradient = Val then exit;
  FPaintGradient := Val;
  // Make the non client area repaint.
  SetWindowPos(Handle, 0, 0, 0, 0, 0, SWP_DRAWFRAME or SWP_NOACTIVATE or SWP_NOMOVE or
                                      SWP_NOSIZE or SWP_NOZORDER);
end;

procedure TGradientForm.SetCaptionText(const Val: string);
begin
  if Val = FCaptionText then exit;
  FCaptionText := Val;
  if ((FormStyle = fsMDIChild) and (WindowState = wsMaximized)) then
    // Need to cause main form's caption to be redrawn, not the MDI child.
    SetWindowPos(Application.MainForm.Handle, 0, 0, 0, 0, 0,
             SWP_DRAWFRAME or SWP_NOACTIVATE or SWP_NOMOVE or SWP_NOSIZE or SWP_NOZORDER)
  else
    Draw(IsActiveWindow);
end;

function TGradientForm.IsActiveWindow: boolean;
begin
  if FormStyle = fsMDIChild then
    if assigned(Application.MainForm) then
      Result := (GetActiveWindow = Application.MainForm.Handle) and
                (TForm(Application.MainForm).ActiveMDIChild = Self)
    else
      Result := FALSE
  else
    Result := GetActiveWindow=Handle;
end;

procedure TGradientForm.CalculateColors;
var
  SysColor: TRGBMap;
  RedPct,
  GreenPct,
  BluePct: real;
  x,
  Band: byte;
begin
  // Get colors for both active and inactive captions.
  for x := 0 to 1 do begin
    if x = 0 then
      SysColor.RGBVal := GetSysColor(COLOR_INACTIVECAPTION)
    else
      SysColor.RGBVal := GetSysColor(COLOR_ACTIVECAPTION);
    // Figure out the percentage of each RGB value needed for banding
    with SysColor do begin
      RedPct   := Red / (FGradientColors-1);
      GreenPct := Green / (FGradientColors-1);
      BluePct  := Blue / (FGradientColors-1);
    end;
    // Use the percentage of each color to create each band color.
    for Band := 0 to (FGradientColors-1) do
      Colors[x][Band] := RGB(round(RedPct * (Band)),
                             round(GreenPct * (Band)), round(BluePct * (Band)));
  end;
end;

procedure TGradientForm.CreateCaptionFont;
var
  NCM: TNonClientMetrics;
begin
  if CaptionFont <> 0 then
    DeleteObject(CaptionFont);
  NCM.cbSize := SizeOf(NCM);
  if SystemParametersInfo(SPI_GETNONCLIENTMETRICS, 0, @NCM, 0) then begin
    if BorderStyle in [bsToolWindow, bsSizeToolWin] then
      CaptionFont := CreateFontIndirect(NCM.lfSmCaptionFont)
    else
      CaptionFont := CreateFontIndirect(NCM.lfCaptionFont);
  end else
    CaptionFont := 0;
end;

// The caption rect is the rectangle we are interested in painting.  This will be the
// area that contains the caption icon, text and buttons.
function TGradientForm.GetCaptionRect: TRect;
begin
  // if we have no border style, then just set the rectange empty.
  if BorderStyle = bsNone then
    SetRectEmpty(Result)
  else begin
    GetWindowRect(Handle, Result);
    // Convert rect from screen (absolute) to client (0 based) coordinates.
    OffsetRect(Result, -Result.Left, -Result.Top);
    // Shrink rectangle to allow for window border.  We let Windows paint it.
    case BorderStyle of
      bsToolWindow, bsSingle, bsDialog:
          InflateRect(Result, -GetSystemMetrics(SM_CXFIXEDFRAME),
                              -GetSystemMetrics(SM_CYFIXEDFRAME));
      bsSizeable, bsSizeToolWin:
          InflateRect(Result, -GetSystemMetrics(SM_CXSIZEFRAME),
                              -GetSystemMetrics(SM_CYSIZEFRAME));
    end;

    // Set the appropriate height of caption bar.
    if BorderStyle in [bsToolWindow, bsSizeToolWin] then
      Result.Bottom := Result.Top + GetSystemMetrics(SM_CYSMCAPTION) - 1
    else
      Result.Bottom := Result.Top + GetSystemMetrics(SM_CYCAPTION) - 1;
  end;
end;

// Paint the icon for the system menu.
procedure TGradientForm.PaintMenuIcon(DC: HDC; var R: TRect; Active: boolean);
const
  LR_COPYFROMRESOURCE = $4000; // Missing from WINDOWS.PAS!
var
  IconHandle: HIcon;
  Size: integer;
begin
  // Does the form have an icon assigned to it?
  if Icon.Handle <> 0 then
    IconHandle := Icon.Handle
  // If not, does the application have an icon?
  else if Application.Icon.Handle <> 0 then
    IconHandle := Application.Icon.Handle
  // If not, then just use the system defined application icon.
  else
    IconHandle := LoadIcon(0, IDI_APPLICATION);

  Size := GetSystemMetrics(SM_CXSMICON);
  with R do
    // Let CopyImage() make get us a nice 16x16 version of the icon and we'll paint it.
    DrawIconEx(DC, Left+1, Top+1,
               CopyImage(IconHandle, IMAGE_ICON, Size, Size, LR_COPYFROMRESOURCE),
               0, 0, 0, 0, DI_NORMAL);
  Inc(R.Left, Size+1);
end;

// Paint the given rectangle with the system solid color.
procedure TGradientForm.FillRectSolid(DC: HDC; const R: TRect; Active: boolean);
var
  OldBrush,
  Brush: HBrush;
begin
  // Create a brush with the appropriate color\
  if Active then
    Brush := CreateSolidBrush(GetSysColor(COLOR_ACTIVECAPTION))
  else
    Brush := CreateSolidBrush(GetSysColor(COLOR_INACTIVECAPTION));
  // Select that brush into the temporary DC.
  OldBrush := SelectObject(DC, Brush);
  try
    // Fill the rectangle using the selected brush -- PatBlt is faster than FillRect
    with R do
    PatBlt(DC, Left, Top, Right-Left, Bottom-Top, PATCOPY);
  finally
    // Clean up the brush
    SelectObject(DC, OldBrush);
    DeleteObject(Brush);
  end;
end;

// Paint the given rectangle with the gradient pattern.
procedure TGradientForm.FillRectGradientHigh(DC: HDC; const R: TRect; Active: boolean);
var
  Band: integer;
  H: integer;
  OldBmp,
  TmpBmp: HBitmap;
  TmpDC: HDC;
begin
  H := R.Bottom - R.Top;

  TmpDC := CreateCompatibleDC(DC);
  TmpBmp := CreateCompatibleBitmap(DC, FGradientColors, 1);
  OldBmp := SelectObject(TmpDC, TmpBmp);
  try
    // Start filling bands
    for Band := 0 to (FGradientColors-1) do
      SetPixel(TmpDC, Band, 0, Colors[ord(Active)][Band]);
    StretchBlt(DC, 0, 0, R.Right-R.Left, H, TmpDC, 0, 0, FGradientColors-1, 1, SRCCOPY);
  finally
    SelectObject(TmpDC, OldBmp);
    DeleteObject(TmpBmp);
    DeleteDC(TmpDC);
  end;
end;

procedure TGradientForm.FillRectGradientLow(DC: HDC; const R: TRect; Active: boolean);
var
  OldBrush,
  Brush: HBrush;
  Step: real;
  Band: integer;
  H: integer;
begin
  // Determine how large each band should be in order to cover the
  // rectangle (one band for every color intensity level).
  Step := (R.Right - R.Left) / FGradientColors;
  H := R.Bottom - R.Top;

  // Start filling bands
  for Band := 0 to (FGradientColors-1) do begin
    // Create a brush with the appropriate color for this band
    Brush := CreateSolidBrush(Colors[ord(Active)][Band]);
    // Select that brush into the temporary DC.
    OldBrush := SelectObject(DC, Brush);
    try
      // Fill the rectangle using the selected brush -- PatBlt is faster than FillRect
      PatBlt(DC, round(Band*Step), 0, round((Band+1)*Step), H, PATCOPY);
    finally
      // Clean up the brush
      SelectObject(DC, OldBrush);
      DeleteObject(Brush);
    end;
  end; // for
end;

procedure TGradientForm.PaintCaptionText(DC: HDC; R: TRect);
var
  OldColor: TColorRef;
  OldMode: integer;
  OldFont: HFont;
  CaptionText: string;
begin
  CaptionText := Caption;
  // Have to turn off complete boolean eval for this "if" statement.  I never have it on
  // anyway, but some do.
  {$IFOPT B+} {$DEFINE RESET_BOOL_EVAL} {$B-} {$ENDIF}
  if ((FormStyle = fsMDIForm) and (ActiveMDIChild <> NIL) and
      (ActiveMDIChild.WindowState = wsMaximized)) then
    CaptionText := CaptionText + ' - [' + ActiveMDIChild.Caption + ']';
  {$IFDEF RESET_BOOL_EVAL} {$B+} {$ENDIF}

  Inc(R.Left, 2);
  // Set the text color to white, not the system color.  We always color from black to
  // the system color, so we will be almost guaranteed of having a nearly black background
  // under the text.  White is the safest color to use in this case (Microsoft thinks so,
  // too, so I can't be too wrong).
  OldColor := SetTextColor(DC, RGB(255,255,255));
  // Set the background text painting mode to transparent so that drawing text doesn't
  // distrub the gradient we just painted.  If you didn't do this, then drawing text would
  // also fill the text rectangle with a solid background color, screwing up our gradient.
  OldMode := SetBkMode(DC, TRANSPARENT);
  // Select in the system defined caption font (see Create constructor).
  if CaptionFont <> 0 then
    OldFont := SelectObject(DC, CaptionFont)
  else
    OldFont := 0;
  try
    // Draw the text making it left aligned, centered vertically, allowing no line breaks.
    DrawText(DC, PChar(CaptionText), -1, R, DT_LEFT or DT_VCENTER or
             DT_SINGLELINE or DT_END_ELLIPSIS);
  finally
    // Clean up all the drawing objects.
    if OldFont <> 0 then
      SelectObject(DC, OldFont);
    SetBkMode(DC, OldMode);
    SetTextColor(DC, OldColor);
  end;
end;

// Paint the min/max/help/close buttons.
procedure TGradientForm.PaintCaptionButtons(DC: HDC; var Rect: TRect);
var
  BtnWidth: integer;
  Flag: UINT;
  SrcRect: TRect;
begin
  SrcRect := Rect;
  InflateRect(SrcRect, -2, -2);
  if BorderStyle in [bsToolWindow, bsSizeToolWin] then begin
    // Tool windows only have the close button, nothing else.
    with SrcRect do
      Left := Right - (GetSystemMetrics(SM_CXSMSIZE)) + 2;
    DrawFrameControl(DC, SrcRect, DFC_CAPTION, DFCS_CAPTIONCLOSE);
    Rect.Right := SrcRect.Left-5;
  end else begin
//  BtnWidth := R.Bottom - R.Top;
    BtnWidth := GetSystemMetrics(SM_CXSMICON)-2;
    SrcRect.Left := SrcRect.Right - BtnWidth - 2;
    // if it has system menu, it has a close button.
    if biSystemMenu in BorderIcons then begin
      DrawFrameControl(DC, SrcRect, DFC_CAPTION, DFCS_CAPTIONCLOSE);
      OffsetRect(SrcRect, -BtnWidth-4, 0);
      Dec(Rect.Right,BtnWidth+4);
    end;
    // Minimize and Maximized don't show up at all if BorderStyle is bsDialog
    if BorderStyle <> bsDialog then begin
      if WindowState = wsMaximized then
        Flag := DFCS_CAPTIONRESTORE
      else
        Flag := DFCS_CAPTIONMAX;
      // if it doesn't have max in style, then it shows up disabled
      if not (biMaximize in BorderIcons) then
        Flag := Flag or DFCS_INACTIVE;
      DrawFrameControl(DC, SrcRect, DFC_CAPTION, Flag);
      OffsetRect(SrcRect, -BtnWidth-2, 0);

      Flag := DFCS_CAPTIONMIN;
      // if it doesn't have min in style, then it shows up disabled
      if not (biMinimize in BorderIcons) then
        Flag := Flag or DFCS_INACTIVE;
      DrawFrameControl(DC, SrcRect, DFC_CAPTION, Flag);
      OffsetRect(SrcRect, -BtnWidth-2, 0);
      Dec(Rect.Right,(BtnWidth+2) * 2);
    end else // Help only shows up in bsDialog style
      if (biHelp in BorderIcons) then begin
        DrawFrameControl(DC, SrcRect, DFC_CAPTION, DFCS_CAPTIONHELP);
        Dec(Rect.Right,BtnWidth+2);
      end;
    Dec(Rect.Right, 3);
  end;
end;

// This draws everything we need, using all the other methods to put it all together.
function TGradientForm.DrawCaption(FormDC: HDC; Active: boolean): TRect;
var
  R: TRect;
  OldBmp,
  Bmp: HBitmap;
  BmpDC: HDC;
  BmpCanvas: TCanvas;
  w,h:integer;
begin
  // Get only the portion we need to draw.
  R := GetCaptionRect;
  Result := R;

  // Convert to logical (0-based) coordinates
  OffsetRect(R, -R.Left, -R.Top);

  W := R.Right - R.Left;
  H := R.Bottom - R.Top;

  // Create a temporary device context to draw on.  Drawing on a temporary DC and copying
  // it to the real DC accomplishes two things:
  // 1) It is faster because Windows doesn't have to draw anything in the temporary DC on
  // the screen, it only draws when you paint something on a real DC.  Then it just draws
  // everything at once when we copy it, instead of drawing a little, do some
  // calculations, draw a little, etc.
  // 2) It looks much better because it is drawn faster.  It reduces the "flicker" that
  // you would see from each individual part being drawn, especially the gradient bands.
  BmpDC := CreateCompatibleDC(FormDC);
  Bmp := CreateCompatibleBitmap(FormDC, W, H);
  OldBmp := SelectObject(BmpDC, Bmp);

  try
    if (FPaintGradient = gfpAlways) or (Active and (FPaintGradient = gfpActive)) then begin
    // Draw the gradient background in the temporary DC
      // This may look backwards, but it's not.  If the device capabilities indicate that
      // there are palette entries (more than 0), then we are in a low color mode.  This
      // is because when in high color mode or above, Windows doesn't use palettes.
      if GetDeviceCaps(BmpDC, SIZEPALETTE) > 0 then
        FillRectGradientLow(BmpDC, R, Active)
      else
        FillRectGradientHigh(BmpDC, R, Active);
    end else
      FillRectSolid(BmpDC, R, Active);

    Inc(R.Left, 1);
    // Do we need to paint an icon for the system menu?
    if (biSystemMenu in BorderIcons) and (BorderStyle in [bsSingle, bsSizeable]) then
      PaintMenuIcon(BmpDC, R, Active); // PaintMenuIcon will adjust the rect so that future
                               // drawing operations happen in the right spot.
    PaintCaptionButtons(BmpDC, R); // Paint the min/max/help/close buttons.
    if assigned(FOnCaptionPaint) then begin
      BmpCanvas := TCanvas.Create;
      try
        BmpCanvas.Handle := BmpDC;
        BmpCanvas.Font.Handle := CaptionFont;
        FOnCaptionPaint(Self, BmpCanvas, R);
      finally
        BmpCanvas.Free;
      end;
    end;
    PaintCaptionText(BmpDC, R); // Paint the caption text.

    // Copy the gradient caption bar to the real DC.
    BitBlt(FormDC, Result.Left, Result.Top, W, H, BmpDC, 0, 0, SRCCOPY);
  finally
    // Clean up all the temporary drawing objects.
    SelectObject(BmpDC, OldBmp);
    DeleteObject(Bmp);
    DeleteDC(BmpDC);
  end;
end;

// Windows sends this message when the window has been activated or deactivated.
procedure TGradientForm.WMNCActivate(var Msg: TWMNCActivate);
begin
  Draw(Msg.Active);
  Msg.Result := 1;

(* we always need to paint it now since we are managing the caption text.

  // Do we need to paint a gradient?
  if Msg.Active or FGradientOnInactive then begin
    Draw(Msg.Active);
    // Tell Windows we have handled the message and it doesn't need to do anything else.
    Msg.Result := 1;
  end else
    // Let Windows paint the caption.
    inherited;
*)
end;

// Windows sends this message whenever any part of the non-client area (caption, window
// border) needs repainting.
procedure TGradientForm.WMNCPaint(var Msg: TMessage);
var
  WR, R: TRect;
  DC: HDC;
  MyRgn: HRGN;
  DeleteRgn: boolean;
begin
  DeleteRgn := FALSE;
  // The region that needs painting is passed in WParam.  A region is a Windows object
  // used to describe the non-rectangular area used by a combination of rectangles.
  MyRgn := Msg.wParam;
  DC := GetWindowDC(Handle);
  try
    GetWindowRect(Handle, WR);
    // Select the update region as the clipping region.  Clipping regions guarantee that
    // any painting done outside of the selected region is not shown (thrown away).
    if SelectClipRgn(DC, MyRgn) = ERROR then begin
      // We got passed an invalid region.  Generally, this happens when the window is
      // first created.  We'll create our own region (the rectangle that makes up the
      // entire window) and use that instead.
      with WR do
        MyRgn := CreateRectRgn(Left, Top, Right, Bottom);
      SelectClipRgn(DC, MyRgn);
      DeleteRgn := TRUE;
    end;
    // Convert the clipping region coordinates from screen to client.
    OffsetClipRgn(DC, -WR.Left, -WR.Top);
    // Draw our gradient caption.
    R := DrawCaption(DC, IsActiveWindow);
    // Here's the trick.  DrawCaption returns the rectangle that we painted.  We now
    // exclude that rectangle from the clipping region.  This guarantees that any further
    // painting that occurs will not happen in this rectangle.  That means that when we
    // let the default painting for WM_NCPAINT occur, it will not paint over our gradient.
    // It only paints the stuff that we didn't, like the window borders.
    ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
    // Convert coordinates back into screen-based.
    OffsetClipRgn(DC, WR.Left, WR.Top);
    // Get the region that is now described by the clipping region.
    GetClipRgn(DC, MyRgn);
    // Pass that region on to the default WM_NCPAINT handler.  Remember, we excluded the
    // rectangle that we painted, so Windows will not be able to paint over what we did.
    // Most gradient captions components just let windows draw its stuff first, and then
    // paint the gradient.  This results in an irritating "flicker", caused by the area
    // being painted normally, and then painted over a second time by the gradient.
    Msg.Result := DefWindowProc(Handle, Msg.Msg, MyRgn, Msg.lParam);
  finally
    // If we had to create our own region, we have to clean it up.
    if DeleteRgn then
      DeleteObject(MyRgn);
    ReleaseDC(Handle, DC); // NEVER leave this hanging.
  end;
end;

// Windows sends this message if the user changes any of the system colors.
procedure TGradientForm.WMSysColorChange(var Msg: TWMSysColorChange);
begin
  CalculateColors;
  inherited;
end;

// The window has been resized.
procedure TGradientForm.WMSize(var Msg: TWMSize);
begin
  inherited;
  // If the window was maximized or restored, we need to redraw so the right caption
  // button is painted.
  if (Msg.SizeType = SIZE_MAXIMIZED) or (Msg.SizeType = SIZE_RESTORED) then
    Draw(IsActiveWindow);
end;

// Windows would like to have a cursor displayed.  I know, you're wondering why the
// hell I care about this, aren't you?  Well, in the inherited handling (default Windows
// processing) of this message, if the mouse is over a resizeable border section, Windows
// repaints the caption buttons.  Why?  I have absolutely no idea.  However, that's not
// the important part.  When it repaints those buttons, it also repaints the background
// around them in the last color it painted the caption in.  Now, usually this would just
// result in losing a few bands of the caption gradient, which 99.44% of all users would
// never notice.  However, because we don't always allow default processing of
// WM_NCACTIVATE, sometimes Windows doesn't have the right idea about which color is
// currently the background.  This cause the background to get painted in the wrong color
// sometimes, which 99.44% of all users *will* notice.  We fix it by setting the
// appropriate cursor and not allowing the default processing to occur.
procedure TGradientForm.WMSetCursor(var Msg: TWMSetCursor);
begin
  // Tell Windows we handled the message
  Msg.Result := 1;
  // Load and display the correct cursor for the border area being hit
  case Msg.HitTest of
    HTTOP,
    HTBOTTOM:      SetCursor(LoadCursor(0, MakeIntResource(IDC_SIZENS)));
    HTLEFT,
    HTRIGHT:       SetCursor(LoadCursor(0, MakeIntResource(IDC_SIZEWE)));
    HTTOPRIGHT,
    HTBOTTOMLEFT:  SetCursor(LoadCursor(0, MakeIntResource(IDC_SIZENESW)));
    HTTOPLEFT,
    HTBOTTOMRIGHT: SetCursor(LoadCursor(0, MakeIntResource(IDC_SIZENWSE)));
  else
    // Wasn't anything we cared about, so tell Windows we didn't handle it.
    Msg.Result := 0;
    inherited;
  end;
end;

procedure TGradientForm.WMSetText(var Msg: TWMSetText);
begin
  // Need to repaint the caption because by default (i.e., inherited), Windows will
  // repaint the caption bar in it's normal state.
  Caption := Msg.Text;
  Msg.Result := 1;
end;

procedure TGradientForm.WMGetText(var Msg: TWMGetText);
begin
  StrLCopy(Msg.Text, PChar(FCaptionText), Msg.TextMax-1);
  Msg.Result := StrLen(Msg.Text)+1;
end;

procedure TGradientForm.WMGetTextLength(var Msg: TWMGetTextLength);
begin
  Msg.Result := Length(FCaptionText);
end;

procedure TGradientForm.WMSettingChange(var Msg: TMessage);
begin
  if Msg.wParam = SPI_SETNONCLIENTMETRICS then  // User might have changed NC font.
    CreateCaptionFont;
  inherited;
end;

procedure TGradientForm.Draw(Active: boolean);
var
  DC: HDC;
begin
  // Get the DC we need to paint in.  GetDC would only get the DC for the client area, we
  // need it for non-client area, too, so we use GetWindowDC.
  DC := GetWindowDC(Handle);
  try
    DrawCaption(DC, Active);
  finally
    ReleaseDC(Handle, DC); // NEVER leave this hanging.
  end;
end;

procedure TGradientForm.GradClientWndProc(var Message: TMessage);
begin
  with Message do begin
    Result := CallWindowProc(FGradDefClientProc, ClientHandle, Msg, wParam, lParam);
    // if you don't want your MDI child forms to be TGradientForm descendants, you will
    // need to use uncomment the code in the following line.
    // It will work, but it causes an annoying flicker.

    if {(Msg = WM_MDIREFRESHMENU) or} (Msg = $003F) then // Magic number.  I hate this.  :(
      Draw(IsActiveWindow);
  end;
end;

end.

