unit DropFile;

{
Author     : Craig Manley
Email      : craig@skybound.nl
WebSite    : http://www.skybound.nl
Created    : 4 July 1999
Last modi. : 5 July 1999
Version    : 1.1

This source code is freeware.
All I ask is to leave this header intact and report any modifications or bugs
to me.

Components:
TDropFileEdit
TCustomEdit descendent that implements the ability
to accept single files/folders that are dropped onto it. When a file is dropped
onto it then an OnDropFile event is triggered containing the filename as one of
the parameters. Use the new published boolean property AcceptFiles to enable or
disable this new functionality.

TDropFileListBox
TCustomListBox descendent that implements the ability
to accept multiple files/folders that are dropped onto it. When files are dropped
onto it then an OnDropFiles event is triggered containing the filenames as one of
the parameters. Use the new published boolean property AcceptFiles to enable or
disable this new functionality.
}

{$R-}

{$R *.RES}

interface

uses
  Windows, Messages, Classes, StdCtrls, ShellAPI;

type
  { TDropFileEdit }
  TDropFileEvent = procedure(Sender: TObject; FileName: string; DropPoint: TPoint) of object;
  TDropFileEdit = class(TCustomEdit)
  private
    FAcceptFile: Boolean;
    FOnDropFile: TDropFileEvent;
    procedure SetAcceptFile(Value: Boolean);
    procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
  protected
    procedure CreateWnd; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property AcceptFile: Boolean read FAcceptFile write SetAcceptFile;
    //property Anchors;
    property AutoSelect;
    property AutoSize;
    //property BiDiMode;
    property BorderStyle;
    property CharCase;
    property Color;
    //property Constraints;
    property Ctl3D;
    property DragCursor;
    //property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property MaxLength;
    property OEMConvert;
    //property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnDropFile: TDropFileEvent read FOnDropFile write FOnDropFile;    
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    //property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    //property OnStartDock;
    property OnStartDrag;
  end;
  
  { TDropFileListBox }  
  TDropFilesEvent = procedure(Sender: TObject; FileList: TStringList; DropPoint: TPoint) of object;
  TDropFileListBox = class(TCustomListBox)
  private
    FAcceptFiles: Boolean;
    FOnDropFiles: TDropFilesEvent;
    procedure SetAcceptFiles(Value: Boolean);
    procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
  published
    property AcceptFiles: Boolean read FAcceptFiles write SetAcceptFiles;
    property Align;
    //property Anchors;
    //property BiDiMode;
    property BorderStyle;
    property Color;
    property Columns;
    //property Constraints;
    property Ctl3D;
    property DragCursor;
    //property DragKind;
    property DragMode;
    property Enabled;
    property ExtendedSelect;
    property Font;
    property ImeMode;
    property ImeName;
    property IntegralHeight;
    property ItemHeight;
    property Items;
    property MultiSelect;
    //property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Sorted;
    property Style;
    property TabOrder;
    property TabStop;
    property TabWidth;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnDropFiles: TDropFilesEvent read FOnDropFiles write FOnDropFiles;
    //property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMeasureItem;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    //property OnStartDock;
    property OnStartDrag;
  public
    constructor Create(AOwner: Tcomponent); override;
  protected
    procedure CreateWnd; override;
  end;
  

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Freeware', [TDropFileEdit,TDropFileListBox]);  
end;

{ TDropFileEdit }

constructor TDropFileEdit.Create(AOwner: Tcomponent);
begin
  FAcceptFile := True;
  inherited Create(AOwner);
end;

procedure TDropFileEdit.CreateWnd;
begin
  inherited CreateWnd;
  SetAcceptFile(FAcceptFile);
end;

procedure TDropFileEdit.SetAcceptFile(Value: Boolean);
begin
  FAcceptFile := Value;
  if not (csDesigning in ComponentState) then
    DragAcceptFiles(self.Handle,Value);
end;

procedure TDropFileEdit.WMDropFiles(var Msg: TWMDropFiles);
var
  rc: UINT;
  iBuffSize: integer;
  lpszFile: PChar;
  FileName: string;
  DropPoint: TPoint;
begin
  // Get count dropped files
  rc := DragQueryFile(Msg.Drop,$FFFFFFFF,nil,0);
  try
    if rc > 1 then
    begin
      DragFinish(Msg.Drop);
      Exit;
    end;
    // Get buffer size
    iBuffSize := DragQueryFile(Msg.Drop,0,nil,0) + 1;
    GetMem(lpszFile,iBuffSize);
    try
      // Get file name
      DragQueryFile(Msg.Drop,0,lpszFile,iBuffSize);
      FileName := string(lpszFile);
    finally
      FreeMem(lpszFile);
    end;
    // Get drop point
    DragQueryPoint(Msg.Drop,DropPoint);
    if Assigned(FOnDropFile) then FOnDropFile(Self,FileName,DropPoint);
  finally
    DragFinish(Msg.Drop);
  end;
  Msg.Result := 0;
end;


{ TDropFileListBox }

constructor TDropFileListBox.Create(AOwner: Tcomponent);
begin
  FAcceptFiles := True;
  inherited Create(AOwner);
end;

procedure TDropFileListBox.CreateWnd;
begin
  inherited CreateWnd;
  SetAcceptFiles(FAcceptFiles);
end;

procedure TDropFileListBox.SetAcceptFiles(Value: Boolean);
begin
  FAcceptFiles := Value;
  if not (csDesigning in ComponentState) then
    DragAcceptFiles(self.Handle,Value);
end;

procedure TDropFileListBox.WMDropFiles(var Msg: TWMDropFiles);
var
  rc: UINT;
  i, iBuffSize: integer;
  lpszFile: PChar;
  FileList: TStringList;
  DropPoint: TPoint;
begin
  FileList := TStringList.Create;
  try
    // Get count dropped files
    rc := DragQueryFile(Msg.Drop,$FFFFFFFF,nil,0);
    for i := 0 to rc-1 do
    begin
      // Get buffer size
      iBuffSize := DragQueryFile(Msg.Drop,i,nil,0) + 1;
      GetMem(lpszFile,iBuffSize);
      try
        // Get file name
        DragQueryFile(Msg.Drop,i,lpszFile,iBuffSize);
        FileList.Append(string(lpszFile));
      finally
        FreeMem(lpszFile);
      end;
    end;
    // Get drop point
    DragQueryPoint(Msg.Drop,DropPoint);
    if Assigned(FOnDropFiles) then FOnDropFiles(Self,FileList,DropPoint);
  finally
    DragFinish(Msg.Drop);
    FileList.Free;
  end;
  Msg.Result := 0;
end;

end.
