// ..................................................................
//
//                          mdMsgComponent
//
//              Copyright  1997 by Martin Djerns
//
// ..................................................................
// Initial Date : 20 October 1997 - MD
// 12 November 1997 - Version 1.0 released
// ..................................................................
// Description :
// - Creates a component which can hook into a windowed component
//   and use it's handle and get the messages send to it.
//   The component is createt for creating a Drag'n'Drop component
//   which have to hook into the event of the windowed component
//   which is going to accept drop's.
// ..................................................................
// Known issues :
// - Meight requere more resources, since the right component has to
//   be found each time.
// ..................................................................

Unit mdMsgCmp;

Interface

Uses
  Windows, Classes, Forms, Messages;

Type
  TmdMsgComponent = class(TComponent)
  Private
    FForm : TForm;
    FHandle : THandle;
    Procedure SetHandle(Value : THandle);
  Protected
    Function FindForm(AComponent : TComponent) : TForm;
    Procedure Loaded; Override;
    Procedure SetHook(Handle : THandle); Virtual;
    Procedure RemoveHook(Handle : THandle); Virtual;
  Public
    Constructor Create(AOwner : TComponent); Override;
    Destructor Destroy; Override;
    Property ParentForm : TForm Read FForm;
    Property Handle : THandle Read FHandle Write SetHandle;
  end;


Implementation

Type
  // Window proc type
  TWindowProc = Function(Window: HWnd; Message : Integer;
                         WParam, LParam: Longint): Longint; StdCall;
  // TmdMsgComponent data which is of interrest
  PMsgData = ^TMsgData;
  TMsgData = Record
     MsgCmp : TmdMsgComponent;
     Window : HWnd;
     WndProc : TWindowProc;
  end;

  // List of TmdMsgComponents which want events
  TMsgCmpList = class(TList)
  Public
    Function Add(MsgCmp : TmdMsgComponent; Handle : THandle) : Integer;
    Function Remove(MsgCmp : TmdMsgComponent; Handle : THandle) : Integer;
    Function FindWindow(Start : Integer; Window : HWnd) : Integer;
    Function FindWndProc(WndProc : TWindowProc) : Integer;
  end;


Const
  mdMsgList : TMsgCmpList = NIL;
  OldWindowProc : TWindowProc = NIL;
  Start = -1;


// ..................................................................

// Substitute Window procedure
Function mdMsgWindowProc(Window: HWnd; Message:Integer; WParam,
  LParam: Longint): Longint; StdCall;
Var
  Counter : Integer;
  Msg : TMessage;
  MsgData : PMsgData;
Begin
  MsgData := NIL;
  Result := 0;
  Counter := Start;
  Repeat
    // Find a component which want a copy of the event to this "window"
    Counter := mdMsgList.FindWindow(Counter, Window);
    If Counter <> -1 Then
    Begin
      MsgData := PMsgData(mdMsgList[Counter]); // Get Data
      If (MsgData.MsgCmp <> NIL) AND
         (NOT (csDestroying IN MsgData.MsgCmp.ComponentState)) Then
      Begin
        Msg.Msg := Message;   // Create Msg structure
        Msg.WParam := WParam;
        Msg.LParam := LParam;
        Msg.Result := Result;
        If MsgData.MsgCmp <> NIL Then   // If the components not is deleted
          MsgData.MsgCmp.Dispatch(Msg); // Send it the event/message
        Result := Msg.Result;
      end;
    end;
  Until Counter = -1;

  // Make sure that the original window procedure is called as well
  If MsgData <> NIL Then
    Result := CallWindowProc(@MsgData.WndProc,
                             Window, Message, WParam, LParam);
end;

// ..................................................................

Function TMsgCmpList.Add(MsgCmp : TmdMsgComponent; Handle : THandle) : Integer;
Var
  MsgData : PMsgData;
  OldWndProc : Integer;
  WndProc : TWindowProc;
Begin
  // Get the old window procedure for this window handle
  WndProc := TWindowProc(GetWindowLong(Handle,GWL_WNDPROC));
  // Check if the window procedure allready is in our list
  OldWndProc := FindWindow(Start,Handle);
  // if it is in our list and not is our version of window proc
  // then make sure that it is added again
  If (OldWndProc >= 0) AND (@mdMsgWindowProc <> @WndProc) Then
    OldWndProc := -1;

  New(MsgData);
  If OldWndProc = -1 Then
    // Make sure that the ole window proc is going to be called
    MsgData^.WndProc := WndProc
  else
    // Call the old window proc directly (the last added proc)
    MsgData^.WndProc := PMsgData(Items[OldWndProc])^.WndProc;
  MsgData^.MsgCmp := MsgCmp;
  MsgData^.Window := Handle;
  Result := Inherited Add(MsgData); // Add the pointer to the list

  // Set our version of the Window proc if it is not allready set
  If OldWndProc = -1 Then
    SetWindowLong(Handle,GWL_WNDPROC,LongInt(@mdMsgWindowProc));
end;

Function TMsgCmpList.Remove(MsgCmp : TmdMsgComponent; Handle : THandle) : Integer;
Var
  AMsgCmp : TmdMsgComponent;
  MsgData : PMsgData;
Begin
  For Result := 0 to Count - 1 do
  Begin
    MsgData := PMsgData(Items[Result]);
    AMsgCmp := MsgData.MsgCmp;
    If (AMsgCmp = MsgCmp) AND (MsgData.Window = Handle) Then // If the component is found
    Begin
      MsgData.MsgCmp := NIL; // Remove the reference to it
      Exit;
    end;
  end;
  Result := -1;  // The component was not found !
end;

// Find a window from a start position
// For searching from the start, use start = -1 start start pos
Function TMsgCmpList.FindWindow(Start : Integer; Window : HWnd) : Integer;
Var
  MsgData : PMsgData;
Begin
  For Result := Start+1 to Count - 1 do
  Begin
    MsgData := PMsgData(Items[Result]);
    If (MsgData^.Window = Window) Then
      Exit; // If found the return whith the index number

  end;
  Result := -1;  // Nothing found
end;

// Search for a window proc in the list
Function TMsgCmpList.FindWndProc(WndProc : TWindowProc) : Integer;
Var
  FMsgData : PMsgData;
Begin
  For Result := 0 To Count - 1 do
  Begin
    FMsgData := PMsgData(Items[Result]);
    If @FMsgData^.WndProc = @WndProc Then
      Exit;  // If found the return whith the index number
  end;
  Result := -1;  // Nothing found
end;

// ..................................................................

Constructor TmdMsgComponent.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  // Search for the form which the component is placed on
  FForm := FindForm(AOwner);
end;

Destructor TmdMsgComponent.Destroy;
Begin
  // Remove all event acknowledgesments to this component
  mdMsgList.Remove(Self, FHandle);
  Inherited Destroy;
end;

Procedure TmdMsgComponent.Loaded;
Begin
  Inherited Loaded;
  SetHandle(FHandle); // Make sure that a handle is set after being loaded
end;

// If a extra hook is needed...
Procedure TmdMsgComponent.SetHook(Handle : THandle);
Begin
  mdMsgList.Add(Self,Handle);
end;

// ... it can also be removed again
Procedure TmdMsgComponent.RemoveHook(Handle : THandle);
Begin
  mdMsgList.Remove(Self,Handle);
end;

// Search for a form in the direct list of owners....
Function TmdMsgComponent.FindForm(AComponent : TComponent) : TForm;
Begin
  If AComponent = NIL Then
    Result := NIL
  else
    If AComponent.InheritsFrom(TForm) Then
      Result := (AComponent AS TForm)
    else
      Result := FindForm(AComponent.Owner)
end;

Procedure TmdMsgComponent.SetHandle(Value : THandle);
Begin
  // Make sure that the new handle also is new
  If Value = Handle Then
    Exit;
  // Make sure that all components//windows exists before
  // registering the event....
  If csLoading IN ComponentState Then
  Begin
    FHandle := Value;       // Set value
    Exit;                   // Do it in loaded
  end;

  // If we have a handle allready, then we also get events
  If FHandle <> 0 Then
    mdMsgList.Remove(Self, FHandle); // stop that

  FHandle := Value;                  // Set value
  If FHandle <> 0 Then
    mdMsgList.Add(Self, FHandle);    // Require events (if we have a handle)
end;

Initialization
  mdMsgList := TMsgCmpList.Create; // List of TmdMsgComponents
Finalization
  mdMsgList.Free;
end.