// ..................................................................
//
//                             mdDrop
//
//              Copyright  1997 by Martin Djerns
//
// ..................................................................
// Initial Date : 20 October 1997 - MD
// 12 November 1997 - Version 1.0 released
// ..................................................................
// Description :
// - A Drag'n'Drop component created for not having to make forms and
//   edit-boxes accept drops each time.
// - Is tried in Delphi 2.0 and 3.0.
// ..................................................................
// Known issues :
// - Based on mdMsgComponent (= meight requere a bit of resources)
// - Meight not work without a form, but who will make drag-drop
//   without a form?
// ..................................................................

unit mdDrop;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics,
  Controls, Forms, Dialogs, mdMsgCmp;

Const
  defAccept = True;

type
  TmdDropFiles = class(TmdMsgComponent)
  private
    { Private declarations }
    FAccept : Boolean;
    FDropWindow : TWinControl;
    FFileList : TStrings;
    FPoint : TPoint;
    FFilesDroped : TNotifyEvent;
    Procedure SetDropWindow(Value : TWinControl);
    Procedure SetAccept(Value : Boolean);
  protected
    { Protected declarations }
    Procedure DoFilesDroped; Virtual;
    Procedure RegisterDrop; Virtual;
    Procedure UnRegisterDrop; Virtual;
  public
    { Public declarations }
    Constructor Create(AOwner : TComponent); Override;
    Destructor Destroy; Override;
    Procedure Loaded; Override;
    Procedure Notification(AComponent : TComponent;
                           Operation : TOperation); Override;

    // Message received from the TmdMsgComponent
    Procedure WMDropFiles(Var Msg : TWMDropFiles); Message WM_DROPFILES;

    Property Files : TStrings Read FFileList;
    Property Point : TPoint Read FPoint;
  published
    { Published declarations }
    Property DropWindow : TWinControl Read FDropWindow Write SetDropWindow;
    Property OnFilesDroped : TNotifyEvent Read FFilesDroped Write FFilesDroped;
    Property Accept : Boolean Read FAccept Write SetAccept default defAccept;
  end;

procedure Register;

implementation

Uses
  ShellAPI;

// ----- Standard functions ------

Constructor TmdDropFiles.Create(AOwner : TComponent);
Begin
  Inherited Create(AOwner);
  FFileList := TStringList.Create;
  FAccept := defAccept;
end;

Destructor TmdDropFiles.Destroy;
Begin
  UnregisterDrop;
  FFileList.Free;
  Inherited Destroy;
end;

Procedure TmdDropFiles.Notification(AComponent : TComponent;
                                    Operation : TOperation);
Begin
  Inherited Notification(AComponent, Operation);
  // If the component is stored in our property it
  // must be a notification of that it is removed,
  // so remove our reference to the component
  If AComponent = FDropWindow Then
  Begin
    UnregisterDrop;
    If (ParentForm <> NIL) AND
       (NOT (csDestroying IN ParentForm.ComponentState)) Then
      Handle := ParentForm.Handle // Get the parent form handle
                                  // as default handle
    else
      Handle := 0;                // Set no handle
    FDropWindow := NIL;
  end;
end;

Procedure TmdDropFiles.Loaded;
Begin
  Accept := FAccept; // Must be before calling the inherited proc
  Inherited Loaded;
end;

// ------ Message function(s) ------

Procedure TmdDropFiles.WMDropFiles(Var Msg : TWMDropFiles);
Var
  FileCount : Integer;
  Buffer : PChar;
  BufferSize : Integer;
Begin
  If NOT Accept Then   // If we do not accept drop then quit
    Exit;
  With Msg do
  Try
    Try
      BufferSize := MAX_PATH;
      DragQueryPoint(Drop, FPoint);  // Get drop point
      // Get drop count
      FileCount := DragQueryFile(Drop,$FFFFFFFF,NIL,BufferSize);
      If FileCount > 0 Then
        FFileList.Clear;
      Dec(FileCount);
      Buffer := StrAlloc(BufferSize+1);
      Try
        While FileCount >= 0 do
        Begin
          Buffer[0] := #0;
          // Get the size of the next filename
          DragQueryFile(Drop,FileCount,NIL,BufferSize);
          // Get the next filename
          DragQueryFile(Drop,FileCount,Pointer(Buffer),BufferSize);
          FFileList.Add(StrPas(Buffer));
          Dec(FileCount);
        end;
      Finally
        StrDispose(Buffer);
      end;
      Result := 0;
      DoFilesDroped;  // Signal Delphi event
    Finally
      DragFinish(Drop);  // Close handle
    end;
  Except
    On Exception do ;
  end;
end;

// ------ Property functions ------

Procedure TmdDropFiles.SetDropWindow(Value : TWinControl);
Begin
  If FDropWindow = Value Then
    Exit;
  If Accept Then
    UnRegisterDrop;
  FDropWindow := Value;
  If Accept Then
    RegisterDrop;
end;

Procedure TmdDropFiles.SetAccept(Value : Boolean);
Begin
  FAccept := Value;
  If csReading IN ComponentState Then
    Exit;
  If Accept Then
    RegisterDrop
  else
    UnRegisterDrop;
end;

// ------ Drop functions ------

Procedure TmdDropFiles.RegisterDrop;
Begin
  // Make sure that we have a handle to use
  If FDropWindow = NIL Then
  Begin
    If ParentForm <> NIL Then
      Handle := ParentForm.Handle
    else
      Exit;
  end
  else
    Handle := FDropWindow.Handle;

  DragAcceptFiles(Handle,defAccept);
end;

Procedure TmdDropFiles.UnRegisterDrop;
Begin
  DragAcceptFiles(Handle,False);
end;

// ------ Event functions ------

Procedure TmdDropFiles.DoFilesDroped;
Begin
  If Assigned(FFilesDroped) Then
    FFilesDroped(Self);
end;

// ------ VCL funstions ------

procedure Register;
begin
  RegisterComponents('mdVCL', [TmdDropFiles]);
end;

end.
