unit DragBar;
(*
	VCL to create a panel that can be dragged by the user.  By attaching
  a window, that window will be stretch to fit.

	Author: William R. Florac
  Company: FITCO, Verona, WI (wee little company from my house)
	Copyright 1997, FITCO.  All rights reserved.

 1)  Users of DragBar (and it's components) must accept this disclaimer of
     warranty: "DragBar is supplied as is.  The author disclaims all
     warranties, expressed or implied, including, without limitation,
     the warranties of merchantability and of fitness for any purpose.
     The author assumes no liability for damages, direct or conse-
     quential, which may result from the use of DragBar."

  2) This Software is donated to the public as public domain except as
     noted below.

  3) If you distribute this software, you must include all parts and pages without
  	 modification.

  4) Software may be used, modified and distributed freely if compiled in with
     commercial or private applications (not another VCL).

  5) Fitco retains the copyright to this Software.  You may not distribute
     the source code (PAS) or its compiled unit (DCU) for profit.

  6) If you do find this component handy and you feel guilty
    for using such a great product without paying someone,
    please send a few bucks ($25) to support further development.
    I have spent a lot of time making this VCL the best it can be and
    have included a help file to make it complete.

	7) This file was formatted with tabs set to 2.

	8) Thanks to all those who suggested (and got) improvements.

  9) Latest version can always be found at http://sumac.etcconnect.com/fitco/

	Please forward any comments or suggestions to Bill Florac at:
	 	email: flash@etcconnect.com
		mail: FITCO
					209 Jenna Dr
					Verona, WI  53593

===============================================================================
version 1.0
notes:
	- Attached window must be aligned as alLeft, alRight, alTop or alBottom
  - Setting the Align will find a window to attach to if possible
  - Changing the BarType will remove any alignment and AttachedWindow
  - If you put controls on the Bar, you will have to click on the
    Bar to move it (not the components)
  - You can use this anywhere in a stack of aligned controls.  I spent a lot
    of time making sure the controls stay in the same order!
  - The movement of the window is limited so the attached window will not get
  	smaller than one (width or height).  It seems the alignment will get lost if
    a value of 0 is set.
  - A DragBar without an AttachedWindow will not move
  - You can not attach onw DragBar to another
  - Only appropriate Align will be allow depending on the BarType.
version 1.1
	- fixed loading default widths by changing csReading test to csLoading
version 1.2
	- fixed order of sizing so attached window does not overwrite other windows
    during dynamic drag

wishlist:
	- never could find a way to cancel the drag with the escape key

===============================================================================
*)

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

const
	DragBarVersion = '1.2';

type
	TBarStyle = (bsDynamic, bsSolid, bsFrame);
  TBarType = (btHorz, btVert);

  TDragBar = class(TCustomPanel)
  private
    { Private declarations }
    FBarStyle: TBarStyle;
    FBarType: TBarType;
    FAttachedWindow: TControl;
    Falign: TAlign;
    MouseOffsetX: integer;
    MouseOffsetY: integer;
    LastRect: TRect;
		function PanelToParent(APoint: TPoint): TPoint;
		function GetBounds: TRect;
		procedure SetAttachedWindow(Value: TControl);
    procedure SetBarType(value: TBarType);
    procedure SetAlign(value:TAlign);
		function GetAlign:TAlign;
		function GetAlignedControl: TControl;

  protected
    { Protected declarations }
    procedure MouseDown(Button: TMouseButton;
                        Shift: TShiftState; X, Y: Integer);    override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer);    override;
    procedure MouseUp(Button: TMouseButton;
                      Shift: TShiftState; X, Y: Integer);      override;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation);             override;
  public
    { Public declarations }
    BarMoving: boolean;
    constructor Create(AOwner: TComponent); override;
		procedure CancelDrag;

  published
    property Align: TAlign read GetAlign write SetAlign;
//    property Align;
    property Alignment;
    property BevelInner;
    property BevelOuter default bvNone;
//    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
//    property DragCursor;
//    property DragMode;
    property Enabled;
    property Caption;
    property Color;
    property Ctl3D;
    property Font;
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
//    property OnDblClick;
//    property OnDragDrop;
//    property OnDragOver;
//    property OnEndDrag;
//    property OnEnter;
//    property OnExit;
//    property OnMouseDown;
//    property OnMouseMove;
//    property OnMouseUp;
    property OnResize;
//    property OnStartDrag;

    {overwritten inherited properties}
    procedure SetParent(Value: TWinControl);                   override;

    {inherited properties}
    property Height default 5;
    property Width default 5;
    property Cursor default crVSplit;
    property Left default 0;

    { New published declarations }
		property BarStyle: TBarStyle read FBarStyle write FBarStyle default bsDynamic;
    property AttachedWindow: TControl read FAttachedWindow write SetAttachedWindow default nil;
    property BarType: TBarType read FBarType write SetBarType default btHorz;
  end;


procedure Register;

implementation


//==============================================================================
constructor TDragBar.Create(AOwner: TComponent);
begin
	inherited create(AOwner);
  BevelOuter := bvNone;
  Height := 5;
  Width := 5;
  Cursor := crVSplit;
  FBarStyle := bsDynamic;
  FBarType := btHorz;
  FAttachedWindow := nil;
  FAlign := alNone;
 	inherited align := FAlign;
end; {of Create}


//==============================================================================
procedure TDragBar.SetParent(Value: TWinControl);
begin
  { Set Parent to the new value }
  inherited SetParent(Value);
  { Set Parent to the new value }
  {only in design mode and when not reading from stream}
	if (csDesigning in ComponentState) and (not (csLoading in ComponentState)) and (Parent <> nil) then begin
    SetBounds(GetBounds.Left,Top, GetBounds.Right-GetBounds.Left, Height);
    Caption := '';
  end;
end;

//==============================================================================
procedure TDragBar.SetBarType(value: TBarType);
var
   ARect : TRect;
begin
	if (FBarType = value) then exit;
	{only in design mode and when not reading from stream}
  if (csDesigning in ComponentState) and not (csLoading in ComponentState) then begin
	 	if FAlign <> alNone then Align := alNone; {this will also remove attached window}
	  ARect := GetBounds;
	 	if FBarType = btHorz
   	{Horz to vert} {l t w h}
   	then SetBounds(ARect.Left + ((ARect.right - ARect.left) div 2), ARect.Top, height, ARect.Bottom - ARect.Top)
	 	{Vert to Horz}
   	else SetBounds(ARect.Left, ARect.Top + ((ARect.Bottom - ARect.Top) div 2), ARect.right - ARect.left, width);
  end;

	FBarType := value;
  if FBarType = btHorz
  then Cursor := crVSplit
  else Cursor := crHSplit;
end;

//==============================================================================
procedure TDragBar.Notification(AComponent: TComponent; Operation: TOperation);
begin
	inherited Notification(AComponent,Operation);
  if Operation = opRemove then begin
    if AComponent = FAttachedWindow then FAttachedWindow := nil;
  end;
end;

//==============================================================================
function TDragBar.GetAlign:TAlign;
begin
	result := FAlign;
end;

//==============================================================================
procedure TDragBar.SetAlign(value:TAlign);
begin
	if (Value = Falign) then exit;

	{if we reading stream, just set value the exit}
	if (csLoading in ComponentState) then begin
	  FAlign := value;
  	inherited align := value;
  	exit;
  end;

  {only allow matchin alignments}
	if (FBarType = btHorz) and ((value <> alLeft) and (value <> alRight)) then begin
	  FAlign := value;
  	inherited align := value;
    {only in design mode}
    if (csDesigning in ComponentState) then FAttachedWindow := GetAlignedControl;
  end;

 	if (FBarType = btVert) and ((value <> alTop) and (value <> alBottom)) then begin
	  FAlign := value;
  	inherited align := value;
    {only in design mode}
    if (csDesigning in ComponentState) then FAttachedWindow := GetAlignedControl;
  end;
end;

//==============================================================================
procedure TDragBar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
	Arect: Trect;
  ADC: hDC;
begin
  inherited MouseDown(Button, Shift, X, Y);
	{only start on left buttons}
	if Button <> mbLeft then exit;
  if Parent = nil then exit;
  if FAttachedWindow = nil then exit;

	{save grap point}
	MouseOffsetY := Y;
	MouseOffsetX := X;
	{create rectangle of for bar}
	ARect := Rect(Left, Top, Left + Width, Top + Height);

  case FBarStyle of
	  bsDynamic: begin
	  end;
  	bsSolid: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE );
			{highlight our panel}
	  	InvertRect(aDC, ARect);
		  {give DC back}
	  	ReleaseDC(Parent.Handle, aDC);
	  end;
  	bsFrame: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE );
			{highlight our panel}
			Windows.DrawFocusRect(aDC,ARect);
		  {give DC back}
  		ReleaseDC(Parent.Handle, aDC);
	  end;
  end;
  {save this as last one}
 	LastRect := ARect;
  {flag to indictate BarMovinging mode}
  BarMoving := true;
end;

//==============================================================================
procedure TDragBar.MouseMove(Shift: TShiftState; X, Y: Integer);
var
	Arect: Trect;
  ADC: hDC;
  PointOnParent: TPoint;
  Delta: integer;
begin
  inherited MouseMove(Shift, X, Y);
	{Only do if BarMovinging}
	if not BarMoving then exit;
  if Parent = nil then exit;
  if FAttachedWindow = nil then exit;
  {what! cancel it}
	if not (ssLeft in Shift) then begin
  	CancelDrag;
    exit;
  end;
  {adjust for exactly where grabed with mouse}
	Y := Y - MouseOffsetY;
	X := X - MouseOffsetX;
  {where is this on our form?}
  PointOnParent := PanelToParent(Point(X,Y));

  {range check}
  case Falign of
  alLeft: begin
    if PointOnParent.X < (FAttachedWindow.Left + 1)
  	then PointOnParent.X := FAttachedWindow.Left + 1;
    if PointOnParent.X > GetBounds.Right - width - 1
    then PointOnParent.X := GetBounds.Right - width -1;
  end;
  alRight: begin
  	if PointOnParent.X > ( (FAttachedWindow.Left + FAttachedWindow.width) - width - 1)
  	then PointOnParent.X := ((FAttachedWindow.Left + FAttachedWindow.width) - width - 1);
  	if PointOnParent.X < (GetBounds.Left + 1)
  	then PointOnParent.X := (GetBounds.Left + 1);
 	end;
  alTop: begin
  	if PointOnParent.Y < (FAttachedWindow.Top + 1)
   	then PointOnParent.Y := FAttachedWindow.Top + 1;
  	if PointOnParent.Y > (GetBounds.Bottom - Height - 1)
   	then PointOnParent.Y := (GetBounds.Bottom - Height - 1);
  end;
  alBottom: begin
  	if PointOnParent.Y > ((FAttachedWindow.Top + FAttachedWindow.height) - height - 1)
  	then PointOnParent.Y := ((FAttachedWindow.Top + FAttachedWindow.Height) - height - 1);
    if PointOnParent.Y < (GetBounds.Top + 1)
    then PointOnParent.Y := (GetBounds.Top + 1);
  end;
  end;

  {create a rectangle}
	if FBarType = btHorz
  then ARect := Rect(left, PointOnParent.Y, Left + Width, PointOnParent.Y+ Height)
  else ARect := Rect(PointOnParent.X, Top, PointOnParent.X + Width, Top + Height);

  {dont BarMoving if we dont need to}
  if EqualRect(ARect, LastRect) then exit;

  {disablealign}
  case FBarStyle of
		bsDynamic: begin
    	if FAttachedWindow <> nil then begin
      	Parent.DisableAlign;
				if FBarType = btHorz then begin
	       	if Falign = alTop then begin
            Delta := PointOnParent.Y - (FAttachedWindow.Top + FAttachedWindow.Height);
	          FAttachedWindow.Height := FAttachedWindow.Height + Delta;
          end
	        else begin
          	{this little trick prevents overwritting other windows}
		        Delta := FAttachedWindow.Top - (PointOnParent.Y + Height);
            if delta < 0 then begin
  		        FAttachedWindow.Height := FAttachedWindow.Height + Delta;
		          FAttachedWindow.Top := FAttachedWindow.Top - Delta;
            end else begin
		          FAttachedWindow.Top := FAttachedWindow.Top - Delta;
  		        FAttachedWindow.Height := FAttachedWindow.Height + Delta;
            end;
          end;
        end
        else begin
	       	if Falign = alLeft then begin
            Delta := PointOnParent.X - (FAttachedWindow.Left + FAttachedWindow.Width);
	          FAttachedWindow.Width := FAttachedWindow.Width + Delta;
          end
	        else begin
		        Delta := FAttachedWindow.Left - (PointOnParent.X + Width);
          	{this little trick prevents overwritting other windows}
            if delta < 0 then begin
	  	        FAttachedWindow.Left := FAttachedWindow.Left - Delta;
		          FAttachedWindow.Width := FAttachedWindow.Width + Delta;
            end else begin
		          FAttachedWindow.Width := FAttachedWindow.Width + Delta;
	  	        FAttachedWindow.Left := FAttachedWindow.Left - Delta;
            end;
          end;
        end;
				Parent.EnableAlign;
		  end;
    end;
  	bsSolid: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE);
	  	{remove old}
	  	InvertRect(aDC, LastRect);
		  {add new}
  		InvertRect(aDC, ARect);
		  {don't forget to release DC}
  		ReleaseDC(Parent.Handle, aDC);
	  end;
  	bsFrame: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE);
	  	{remove old}
	  	Windows.DrawFocusRect(aDC, LastRect);
		  {add new}
  		Windows.DrawFocusRect(aDC, ARect);
		  {don't forget to release DC}
  		ReleaseDC(Parent.Handle, aDC);
	  end;
  end;
  {same latest}
  LastRect := ARect;
end;

//==============================================================================
procedure TDragBar.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  ADC: hDC;
  PointOnParent: TPoint;
  Delta: integer;
begin
  inherited MouseUp(Button, Shift, X, Y);
	{only start on left buttons}
	if Button <> mbLeft then exit;
	{Only do if BarMovinging}
	if not BarMoving then exit;
  if Parent = nil then exit;
  if FAttachedWindow = nil then exit;

  {adjust for exactly where grabed with mouse}
	Y := Y - MouseOffsetY;
	X := X - MouseOffsetX;
  {where is this on our form?}
  PointOnParent := PanelToParent(Point(X,Y));

  {range check}
  case Falign of
  alLeft: begin
    if PointOnParent.X < (FAttachedWindow.Left + 1)
  	then PointOnParent.X := FAttachedWindow.Left + 1;
    if PointOnParent.X > GetBounds.Right - width - 1
    then PointOnParent.X := GetBounds.Right - width -1;
    end;
  alRight: begin
  	if PointOnParent.X > ( (FAttachedWindow.Left + FAttachedWindow.width) - width - 1)
  	then PointOnParent.X := ((FAttachedWindow.Left + FAttachedWindow.width) - width - 1);
  	if PointOnParent.X < (GetBounds.Left + 1)
  	then PointOnParent.X := (GetBounds.Left + 1);
 	end;
  alTop: begin
  	if PointOnParent.Y < (FAttachedWindow.Top + 1)
   	then PointOnParent.Y := FAttachedWindow.Top + 1;
  	if PointOnParent.Y > (GetBounds.Bottom - Height - 1)
   	then PointOnParent.Y := (GetBounds.Bottom - Height - 1);
  end;
  alBottom: begin
  	if PointOnParent.Y > ((FAttachedWindow.Top + FAttachedWindow.height) - height - 1)
  	then PointOnParent.Y := ((FAttachedWindow.Top + FAttachedWindow.Height) - height - 1);
    if PointOnParent.Y < (GetBounds.Top + 1)
    then PointOnParent.Y := (GetBounds.Top + 1);
  end;
  end;


  case FBarStyle of
		bsDynamic: begin
	  end;
  	bsSolid: begin
		  {Get DC}
		  aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE or DCX_CLIPSIBLINGS);
  		{move last box}
		  InvertRect(aDC, LastRect);
		  {give DC back!}
	  	ReleaseDC(Parent.Handle, aDC);
	  end;
  	bsFrame: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE);
	  	{remove old}
	  	Windows.DrawFocusRect(aDC, LastRect);
		  {don't forget to release DC}
  		ReleaseDC(Parent.Handle, aDC);
	  end;
  end;
  {not in BarMoving mode any longer}
 	BarMoving := false;
	{adjust attached window}
 	if FAttachedWindow <> nil then begin
      	Parent.DisableAlign;
				if FBarType = btHorz then begin
	       	if Falign = alTop then begin
            Delta := PointOnParent.Y - (FAttachedWindow.Top + FAttachedWindow.Height);
	          FAttachedWindow.Height := FAttachedWindow.Height + Delta;
          end
	        else begin
		        Delta := FAttachedWindow.Top - (PointOnParent.Y + Height);
	          FAttachedWindow.Top := FAttachedWindow.Top - Delta;
  	        FAttachedWindow.Height := FAttachedWindow.Height + Delta;
          end;
        end
        else begin
	       	if Falign = alLeft then begin
            Delta := PointOnParent.X - (FAttachedWindow.Left + FAttachedWindow.Width);
	          FAttachedWindow.Width := FAttachedWindow.Width + Delta;
          end
	        else begin
		        Delta := FAttachedWindow.Left - (PointOnParent.X + Width);
  	        FAttachedWindow.Left := FAttachedWindow.Left - Delta;
	          FAttachedWindow.Width := FAttachedWindow.Width + Delta;
          end;
        end;
				Parent.EnableAlign;
	end;
end;

//==============================================================================
procedure TDragBar.CancelDrag;
var
  ADC: hDC;
begin
	{only if safe!}
	if not BarMoving then exit;
  if Parent = nil then exit;
  if FAttachedWindow = nil then exit;

  case FBarStyle of
		bsDynamic: begin
	  end;
  	bsSolid: begin
		  {Get DC}
		  aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE or DCX_CLIPSIBLINGS);
  		{move last box}
		  InvertRect(aDC, LastRect);
		  {give DC back!}
	  	ReleaseDC(Parent.Handle, aDC);
	  end;
  	bsFrame: begin
		  {get DC}
  		aDC := GetDCEx(Parent.Handle, 0, DCX_PARENTCLIP or DCX_CACHE);
	  	{remove old}
	  	Windows.DrawFocusRect(aDC, LastRect);
		  {don't forget to release DC}
  		ReleaseDC(Parent.Handle, aDC);
	  end;
  end;
  {not in BarMoving mode any longer}
 	BarMoving := false;
end;

//==============================================================================
procedure TDragBar.SetAttachedWindow(Value: TControl);
begin
  if (Value = FAttachedWindow)
  or (Value = Self)
  or (value is TDragBar)
  then exit;
  FAttachedWindow := Value;
end;

//==============================================================================
function TDragBar.PanelToParent(APoint: TPoint): TPoint;
begin
	result.x := 0;
  result.y := 0;
  if Parent = nil then exit;
  Result := ClientToScreen(APoint);
  Result := Parent.ScreenToClient(Result);
end;

//==============================================================================
function TDragBar.GetBounds: TRect;
var
	x: integer;
begin
	Result.Left := 0;
  Result.Top := 0;
  Result.Top := 0;
	Result.Bottom := 0;
	{be safe!}
  if Parent = nil then exit;
	with parent do begin
		{Get left}
    Result.Right := ClientWidth;
    Result.Bottom := ClientHeight;

	  for x := 0 to ControlCount - 1 do begin
    	if Controls[x] = self then continue;
     	case Controls[x].Align of
				alLeft: 	if Controls[x].Left + Controls[x].Width > result.left
	         				then result.left := Controls[x].Left + Controls[x].Width;
        alTop:		if Controls[x].Top + Controls[x].Height > result.top
	        				then result.Top := Controls[x].Top + Controls[x].Height;
        alRight:  if Controls[x].left < result.right
         					then result.right := Controls[x].left;
        alBottom: if Controls[x].top < result.Bottom
         					then result.bottom := Controls[x].top;
			end;
    end;
  end;
end;

//==============================================================================
// Find near window, wont assign self
// Will only assign if itself is assigned
function TDragBar.GetAlignedControl: TControl;
var
	x: integer;
  l, t, r, b: integer;
begin
	result := nil;
	{be safe!}
  if Parent = nil then exit;
	with parent do begin
	  l := 0;
	  t := 0;
  	r := ClientWidth;
	  b := ClientHeight;
	  for x := 0 to ControlCount - 1 do begin
    	if Controls[x] = self then continue;
     	case Controls[x].Align of
				alLeft: 	if (FAlign = alLeft) and (Controls[x].Left + Controls[x].Width > l) then begin
	         					l := Controls[x].Left + Controls[x].Width;
                    result := Controls[x];
                  end;
        alTop:		if (FAlign = alTop) and (Controls[x].Top + Controls[x].Height > t)  then begin
	        					t := Controls[x].Top + Controls[x].Height;
                    result := Controls[x];
                  end;
        alRight:  if (FAlign = alRight) and (Controls[x].left < r)  then begin
        						r := Controls[x].left;
                    result := Controls[x];
                  end;
        alBottom: if (FAlign = alBottom) and (Controls[x].top < b) then begin
         						b := Controls[x].top;
                    result := Controls[x];
                  end;
			end;
    end;
  end;
end;


//==============================================================================
procedure Register;
begin
  RegisterComponents('Fitco', [TDragBar]);
end;

end.
