// The Digital Keys Library  Copyright (C) 2000 RITLABS S.R.L.
// LibTest  Copyright (C) 2000 RITLABS S.R.L.
// Sample application using the Digital Keys Library

program LibTest;

uses
  Controls,
  Messages,
  Graphics,
  StdCtrls,
  Forms,
  Windows,
  dklibapi,
  SysUtils;


// Import the Digital Keys Library functions

const
  DLL = 'dklib32.dll';

function DK_LoadKeyrings(const Params: TdkLoadKeyringsParam): BOOL; stdcall; external DLL;
function DK_Encode(const Params: TdkEncodeParam): BOOL; stdcall; external DLL;
function DK_Decode(const Params: TdkDecodeParam): BOOL; stdcall; external DLL;
function DK_Free: BOOL; stdcall; external DLL;
function DK_GetLastError: DWORD; stdcall; external DLL;

function GetAveCharSize(Canvas: TCanvas): TPoint;
// Returns average char size on specified canvas
var
  I: Integer;
  Buffer: array[0..51] of Char;
begin
  for I := 0 to 25 do Buffer[I] := Chr(I + Ord('A'));
  for I := 0 to 25 do Buffer[I + 26] := Chr(I + Ord('a'));
  GetTextExtentPoint(Canvas.Handle, Buffer, 52, TSize(Result));
  Result.X := Result.X div 52;
end;

function InputQueryEx(const ACaption, APrompt: string; var Value: string; APassword: Boolean): Boolean;
// Displays an input dialog that enables the user to enter a string,
// optionally masking the input.
// This is an extension of standard InputQuery function of Dialogs unit.
var
  Form: TForm;
  Prompt: TLabel;
  Edit: TEdit;
  DialogUnits: TPoint;
  ButtonTop, ButtonWidth, ButtonHeight: Integer;
begin
  Result := False;
  Form := TForm.Create(Application);
  with Form do
    try
      Canvas.Font := Font;
      DialogUnits := GetAveCharSize(Canvas);
      BorderStyle := bsDialog;
      Caption := ACaption;
      ClientWidth := MulDiv(280, DialogUnits.X, 4);
      ClientHeight := MulDiv(63, DialogUnits.Y, 8);
      Position := poScreenCenter;
      Prompt := TLabel.Create(Form);
      with Prompt do
      begin
        Parent := Form;
        AutoSize := True;
        Left := MulDiv(8, DialogUnits.X, 4);
        Top := MulDiv(8, DialogUnits.Y, 8);
        Caption := APrompt;
      end;
      Edit := TEdit.Create(Form);
      with Edit do
      begin
        if APassword then PasswordChar := '*';
        Parent := Form;
        Left := Prompt.Left;
        Top := MulDiv(19, DialogUnits.Y, 8);
        Width := MulDiv(264, DialogUnits.X, 4);
        MaxLength := 255;
        Text := Value;
        SelectAll;
      end;
      ButtonTop := MulDiv(41, DialogUnits.Y, 8);
      ButtonWidth := MulDiv(50, DialogUnits.X, 4);
      ButtonHeight := MulDiv(14, DialogUnits.Y, 8);
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := 'OK';
        ModalResult := mrOk;
        Default := True;
        SetBounds(MulDiv(138, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
      end;
      with TButton.Create(Form) do
      begin
        Parent := Form;
        Caption := 'Cancel';
        ModalResult := mrCancel;
        Cancel := True;
        SetBounds(MulDiv(192, DialogUnits.X, 4), ButtonTop, ButtonWidth,
          ButtonHeight);
      end;
      if ShowModal = mrOk then
      begin
        Value := Edit.Text;
        Result := True;
      end;
    finally
      Form.Free;
    end;
end;

procedure ShowTextBuf(ABuf: PChar; const ACaption: string);
// Displays a text pointed by ABuf in a window
var
  Form: TForm;
  Memo: TMemo;
  DialogUnits: TPoint;
begin
  Form := TForm.Create(Application);
  with Form do
  begin
    DialogUnits := GetAveCharSize(Canvas);
    Caption := ACaption;
    ClientWidth := MulDiv(380, DialogUnits.X, 4);
    ClientHeight := MulDiv(180, DialogUnits.Y, 8);
    Position := poScreenCenter;
  end;
  Memo := TMemo.Create(Form);
  Memo.Font.Name := 'Courier New';
  Form.InsertControl(Memo);
  Memo.Align := alClient;
  Memo.SetTextBuf(ABuf);
  Form.ShowModal;
  Form.Free;
end;


// Custom class that is used to pass data from call-back procedures to the application

type
  TDkLibTest = class
    FDataSize: Integer;
    FDataPtr: Pointer;
    function LoadKeyrings: Boolean;
    function Encode: Boolean;
    function Decode: Boolean;
    procedure InputPincode(var Params: TdkInputPINCode);
    procedure InputPassphrase(var Params: TdkInputPassphrase);
    procedure SetBuf(ASrcPtr: Pointer; ASrcSize: Integer);
    procedure JustShow(APtr: Pointer; ASize: Integer);
    procedure UnableDecode(APtr: Pointer; ACount: Integer);
  end;


procedure TdkLibTest.InputPincode(var Params: TdkInputPINCode);
// This procedure is called from LoadKeyringsCallback when
// DK_LoadKeyrings requires a PIN-code to be entered from the computer keyboard.
// It may only occur only when the bUseSmartCard member of the
// TdkLoadKeyringsParam structure is TRUE.
var
  s: string;
  i: Integer;
begin
  if Params.cbSize <> SizeOf(Params) then Exit;
  if not InputQueryEx('Please enter PIN code', 'PIN Code', s, True) then Exit;
  i := Length(s);
  if (i < 1) or (i > dkMaxPincodeLength) then Exit;
  Move(s[1], Params.PincodeBuf, i);
  Params.bOK := True;
end;

procedure LoadKeyringsCallback(What: Integer; ParamA, ParamB, AppData: Pointer); stdcall;
// The TLoadKeyringsCallback procedure is an application-defined call-back procedure.
// The Digital Keys Library calls this call-back procedure during DK_LoadKeyrings execution.
var
  c: TDkLibTest;
begin
  c := AppData;
  case What of
    dkScSInitReaderBefore : ;
    dkScSInitCardBefore   : ;
    dkScSVerifyPINBefore  : ;
    dkScSReadKeyBefore    : ;

    dkScSInitReaderAfter  : ;
    dkScSInitCardAfter    : ;
    dkScSVerifyPINAfter   : ;
    dkScSReadKeyAfter     : ;

    dkScEKeyUnkOwner      : ;
    dkScEKeyDiffOwner     : ;

    dkInputPinCode        : c.InputPincode(PdkInputPINCode(ParamA)^);
  end;
end;

function TDkLibTest.LoadKeyrings: Boolean;
var
  Params: TdkLoadKeyringsParam; // structure for the keyrings parameter
begin
  FillChar(Params, SizeOf(Params), 0);

// Specifies the size of the structure in bytes.
  Params.cbSize := SizeOf(Params);

// Points to a null-terminated string that specifies
// the name of a file containing the public keyring as defined in RFC-1991.
  Params.lpszPubring  := 'd:\pgp\pubring.pgp';

// Points to a null-terminated string that specifies the name of a file
// containing the secret keyring as defined in RFC-1991.
// If bUseSmartCard is TRUE, then lpszSecring pointer must be nil.
  Params.lpszSecring  := 'd:\pgp\secring.pgp';

// Points to a null-terminated string that specifies the name of a file
// containing pseudo-random data.
  Params.lpszRandseed := 'd:\pgp\randseed.bin';

// Contains the low-order 32 bits of the 64-bit secret key id that will be
// used as the default key for decryption and digital signature generation
// by the DK_Encode and DK_Decode functions.
  Params.dwDefaultKeyIdA := $60B81209;

// Contains the high-order 32 bits of the 64-bit secret key id that will be
// used as the default key for decryption and digital signature generation by
// the DK_Encode and DK_Decode functions.
  Params.dwDefaultKeyIdB := $223A8C7F;

// A pointer to the application-defined call-back procedure for use during
// DK_LoadKeyrings execution.
  Params.lpfnLoadKeyringsCallback := LoadKeyringsCallback;

// Specifies a 32-bit application-defined value. The DK_LoadKeyrings function
// passes this value to the call-back procedure along with other information
// (AppData pointer of LoadKeyringsCallback).
  Params.lpAppData := Self;

  Result := DK_LoadKeyrings(Params);
end;

procedure TDkLibTest.InputPassphrase(var Params: TdkInputPassphrase);
// The DK_Decode or DK_Encode function needs a secret key passphrase to decrypt the data.
// This may only occur when the bUseSmartCard member of the
// TdkLoadKeyringsParam parameter to the DK_LoadKeyrings function is FALSE
var
  s: string;
  i: Integer;
begin
  if Params.cbSize <> SizeOf(Params) then Exit;
  if not InputQueryEx('Input pass phrase for '+Params.lpszKeyName, 'Pass phrase', s, True) then Exit;
  i := Length(s);
  if (i < 1) or (i > dkMaxPassphraseLength) then Exit;
  Move(s[1], Params.PassphraseBuf, i);
  Params.bOK := True;
end;

procedure EncodeCallback(What: Integer; ParamA, ParamB, AppData: Pointer); stdcall;
// The EncodeCallback procedure is an application-defined call-back procedure.
// The Digital Keys Library calls this call-back procedure during DK_Encode execution.
var
  c: TdkLibTest;
begin
  c := AppData;
  case What of
    dkEcEPubKeysNotFound  :;
    dkEcOutBuf            : c.SetBuf(ParamA, Integer(ParamB));
    dkInputPassphrase     : c.InputPassphrase(PdkInputPassphrase(ParamA)^);
  end;
end;

procedure TDkLibTest.SetBuf;
// DK_Encode has successfully completed signing / encrypting and is returning
// the output data. We allocate the memory and copy this data.
begin
  FDataSize := ASrcSize;
  GetMem(FDataPtr, FDataSize+1);
  Move(ASrcPtr^, FDataPtr^, FDataSize);
  PChar(Integer(FDataPtr)+FDataSize)^ := #0;
end;

procedure TDkLibTest.UnableDecode(APtr: Pointer; ACount: Integer);
// The DK_Decode function cannot find the secret key that is required to decrypt the data.
var
  s, z: string;
  i: Integer;
  P: PChar;
begin
  P := APtr;
  for i := 1 to ACount do
  begin
    z := P;
    Inc(P, Length(z)+1);
    s := s + z+#13#10;
  end;
  ShowTextBuf(PChar(s), 'Unable to decrypt');
end;


procedure TDkLibTest.JustShow(APtr: Pointer; ASize: Integer);
// The DK_Decode function has completed decryption / digital signature
// validation and is returning the output data. Display this data.
var
  P: PChar;
begin
  GetMem(P, ASize+1);
  Move(APtr^, P^, ASize);
  PChar(Integer(P)+ASize)^ := #0;
  ShowTextBuf(P, 'Original message');
end;


function TDkLibTest.Encode: Boolean;
const
  CSource = 'Hello Vasya!';
var
  Params: TdkEncodeParam;
begin
  FillChar(Params, SizeOf(Params), 0);

// Specifies the size of the structure in bytes.
  Params.cbSize := SizeOf(Params);

// Pointer to a sequence of recipient names. A recipient is a person able to
// decrypt data using their private key. Each recipient name is a
// null-terminated string. The last string is terminated by two consecutive
// nulls. This member is used during encryption. If bEncrypt is FALSE, the
// value of Rcpts must be nil.
  Params.lpRcpts  := 'Sean Rima'#0'Colin Plumb'#0#0;

// The number of sequential recipients pointed to by Rcpts.
// This member is used for encryption. If bEncrypt is FALSE, the value of
// nRcpts must be 0.
  Params.nRcpts := 2;

// If bEncrypt is TRUE, the DK_Encode function will encrypt data.
// If bSign is also TRUE, the function will sign and then encrypt the data.
  Params.bEncrypt := True;

// If bSign is TRUE, the DK_Encode function will sign data.
// If bEncrypt is also TRUE, the function will sign and then encrypt the data.
  Params.bSign := True;

// If bCompress is TRUE, the DK_Encode function will compress the data before
// encryption. Compression increases the cryptographic strength of the data
// and produces smaller output. If bEncrypt is FALSE, the value of bCompress
// must also be FALSE.
  Params.bCompress := True;

// If bEncryptDefKey is TRUE, the owner of the default key will also be able
// to decrypt the data. The default key id is set by dwDefaultKeyIdA /
// dwDefaultKeyIdB in the TdkLoadKeyringsParam parameter to the DK_LoadKeyrings
// function. If bEncrypt is FALSE, the value of bEncryptDefKey must also be FALSE.
  Params.bEncryptDefKey := True;

// Pointer to the source text data that will be signed and/or encrypted.
  Params.lpSource := CSource;

// The size of the source text data in bytes.
  Params.dwSrcSize := Length(CSource);

// Pointer to the application-defined call-back procedure that will be called
// during DK_Encode execution.
  Params.lpfnEncodeCallback := EncodeCallback;

// Specifies a 32-bit application-defined value. This value is passed to the
// call-back procedure along with other information.
  Params.lpAppData := Self;

  Params.bBinary := True;

  Result := DK_Encode(Params);
end;

procedure DecodeCallback(What: Integer; ParamA, ParamB, AppData: Pointer); stdcall;
// The DecodeCallback procedure is an application-defined call-back procedure.
// The Digital Keys Library calls this call-back procedure during DK_Decode execution.
var
  c: TdkLibTest;
begin
  c := AppData;
  case What of
    dkDcSeckeyNotFound    : c.UnableDecode(ParamA, Integer(ParamB));
    dkDcOutBuf            : c.JustShow(ParamA, Integer(ParamB));
    dkInputPassphrase     : c.InputPassphrase(PdkInputPassphrase(ParamA)^);
  end;
end;

function TDkLibTest.Decode: Boolean;
var
  Params: TdkDecodeParam;
begin
  FillChar(Params, SizeOf(Params), 0);

// Specifies the size of the structure in bytes.
  Params.cbSize := SizeOf(Params);

// Points to the source data that will be decrypted and / or checked for
// digital signature validity.
  Params.lpSource := FDataPtr;

// Size of the source data in bytes.
  Params.dwSrcSize := FDataSize;

// If bReqSig is TRUE, the DK_Decode function will return FALSE (indicating
// an error) if the source data has no valid digital signature.
// In this case, if no other errors had occurred, the error code will be set
// to ERROR_DK_SIG_REQUIRED. If the source has both unsigned (or signed with
// an invalid signature) and signed data, the unsigned data will be discarded.
// If bReqSig is FALSE, the digital signature validity doesn't affect the
// DK_Decode calls success and unsigned data will be returned together with
// signed data.
  Params.bReqSig := True;

// Points to the application-defined call-back procedure that will be called
// during DK_Decode execution.
  Params.lpfnDecodeCallback := DecodeCallback;

// Specifies a 32-bit application-defined value. The DK_Decode function
// passes this value to the call-back procedure along with other information.
  Params.lpAppData := Self;
  Params.bBinary := True;
  Result := DK_Decode(Params);
end;

// String representation of error codes.

const
  MaxDkLibErorr = 25;
  CDkLibErorr: array[0..MaxDkLibErorr] of string = (
  'NO_ERROR',
  'ACCESS_SMARTCARD',
  'INVALID_PARAMETER',
  'ALREADY_INITIALIZED',
  'DLL_NOT_FOUND',
  'LOAD_PUBRING',
  'LOAD_SECRING',
  'NO_DEFAULT_KEY',
  'SC_UNKNOWN_OWNER',
  'SC_DIFF_OWNER',
  'NOT_INITIALIZED',
  'SIGNATURE_FORMAT',
  'UNKNOWN_SIGNER',
  'SECKEY_NOT_FOUND',
  'OPEN_SECKEY',
  'RECURSION',
  'INVALID_ACCESS',
  'SC_DRIVER_FAILURE',
  'SIG_REQUIRED',
  'BAD_DATA_FORMAT',
  'BAD_ARMOR',
  'CANT_DECRYPT',
  'CANT_SIGN',
  'CANT_ENCRYPT',
  'PUBKEY_NOT_FOUND',
  'INPUT_CANCELLED');

function DK_Error2Str(ACode: DWORD): string;
begin
  Str(ACode, Result);
  if Acode <= MaxDkLibErorr then Result := Result+' ('+CDkLibErorr[ACode]+')';
end;


procedure ShowErrorAndExit;
var
  e: DWORD;
  s: string;
begin
  e := DK_GetLastError;
  s := DK_Error2Str(e);
  ShowTextBuf(PChar(s), 'Error');
  Halt;
end;

var
  Test: TdkLibTest;
begin
// Create a custom class that is used to pass data from call-back procedures to the application
  Test := TdkLibTest.Create;
// Load keyrings
  if not Test.LoadKeyrings then ShowErrorAndExit;
// Encrypt & sign the data
  if not Test.Encode then ShowErrorAndExit;
// Show produced otput
//  ShowTextBuf(Test.FDataPtr, 'Text Output'); do not show it as it's binary
// Decrypt the data and check a digital signature
  if not Test.Decode then ShowErrorAndExit;
//
  DK_Free;
  Test.Free;
end.