//******************************************************************************
//** A binary compatible implementation of RC6 *********************************
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit RC6;

interface
uses
  Classes, Sysutils, DCPcrypt;

const
  NUMROUNDS= 20; { number of rounds must be between 16-24 }

type
  TDCP_rc6= class(TDCP_blockcipher)
  protected
    IV, LB: array[0..15] of byte;
    KeyData: array[0..((NUMROUNDS*2)+3)] of DWord;
    procedure Encrypt(const InBlock; var OutBlock);
    procedure Decrypt(const InBlock; var OutBlock);
  public
    procedure Init(var Key; Size: integer; IVector: pointer); override;
    procedure Burn; override;
    procedure Reset; override;
    procedure EncryptECB(const InBlock; var OutBlock); override;
    procedure DecryptECB(const InBlock; var OutBlock); override;
    procedure EncryptCBC(const InData; var OutData; Size: integer); override;
    procedure DecryptCBC(const InData; var OutData; Size: integer); override;
    procedure EncryptCFB(const InData; var OutData; Size: integer); override;
    procedure DecryptCFB(const InData; var OutData; Size: integer); override;
    constructor Create(AOwner: TComponent); override;
  end;

//******************************************************************************
//******************************************************************************
implementation

{$R-}{$Q-}

const
  sBox: array[0..51] of DWord= (
    $B7E15163,$5618CB1C,$F45044D5,$9287BE8E,$30BF3847,$CEF6B200,
    $6D2E2BB9,$0B65A572,$A99D1F2B,$47D498E4,$E60C129D,$84438C56,
    $227B060F,$C0B27FC8,$5EE9F981,$FD21733A,$9B58ECF3,$399066AC,
    $D7C7E065,$75FF5A1E,$1436D3D7,$B26E4D90,$50A5C749,$EEDD4102,
    $8D14BABB,$2B4C3474,$C983AE2D,$67BB27E6,$05F2A19F,$A42A1B58,
    $42619511,$E0990ECA,$7ED08883,$1D08023C,$BB3F7BF5,$5976F5AE,
    $F7AE6F67,$95E5E920,$341D62D9,$D254DC92,$708C564B,$0EC3D004,
    $ACFB49BD,$4B32C376,$E96A3D2F,$87A1B6E8,$25D930A1,$C410AA5A,
    $62482413,$007F9DCC,$9EB71785,$3CEE913E);

constructor TDCP_rc6.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fAlgorithm:= 'RC6';
  fBlockSize:= 128;
  fMaxKeySize:= 2048;
  fID:= 4;
  Burn;
end;

procedure TDCP_rc6.Encrypt(const InBlock; var OutBlock);
var
  x: array[0..3] of DWord;
  u, t: DWord;
  i: integer;
begin
  Move(InBlock,x,Sizeof(x));
  x[1]:= x[1] + KeyData[0];
  x[3]:= x[3] + KeyData[1];
  for i:= 1 to NUMROUNDS do
  begin
    t:= Lrot32(x[1] * (2*x[1] + 1),5);
    u:= Lrot32(x[3] * (2*x[3] + 1),5);
    x[0]:= Lrot32(x[0] xor t,u) + KeyData[2*i];
    x[2]:= Lrot32(x[2] xor u,t) + KeyData[2*i+1];
    t:= x[0]; x[0]:= x[1]; x[1]:= x[2]; x[2]:= x[3]; x[3]:= t;
  end;
  x[0]:= x[0] + KeyData[(2*NUMROUNDS)+2];
  x[2]:= x[2] + KeyData[(2*NUMROUNDS)+3];
  Move(x,OutBlock,Sizeof(x));
end;

procedure TDCP_rc6.Decrypt(const InBlock; var OutBlock);
var
  x: array[0..3] of DWord;
  u, t: DWord;
  i: integer;
begin
  Move(InBlock,x,Sizeof(x));
  x[2]:= x[2] - KeyData[(2*NUMROUNDS)+3];
  x[0]:= x[0] - KeyData[(2*NUMROUNDS)+2];
  for i:= NUMROUNDS downto 1 do
  begin
    t:= x[0]; x[0]:= x[3]; x[3]:= x[2]; x[2]:= x[1]; x[1]:= t;
    u:= Lrot32(x[3] * (2*x[3] + 1),5);
    t:= Lrot32(x[1] * (2*x[1] + 1),5);
    x[2]:= Rrot32(x[2] - KeyData[2*i+1],t) xor u;
    x[0]:= Rrot32(x[0] - KeyData[2*i],u) xor t;
  end;
  x[3]:= x[3] - KeyData[1];
  x[1]:= x[1] - KeyData[0];
  Move(x,OutBlock,Sizeof(x));
end;

procedure TDCP_rc6.Init(var Key; Size: integer; IVector: pointer);
var
  xKeyD: array[0..63] of DWord;
  i, j, k, xKeyLen: integer;
  A, B: DWord;
begin
  if fInitialized then
    Burn;
  if (Size> fMaxKeySize) or (Size<= 0) or ((Size mod 8)<> 0) then
    Exception.Create(Format('RC6: Invalid key size - %d',[Size]));
  Size:= Size div 8;
  FillChar(xKeyD,Sizeof(xKeyD),0);
  Move(Key,xKeyD,Size);
  xKeyLen:= Size div 4;
  if (Size mod 4)<> 0 then
    Inc(xKeyLen);
  Move(sBox,KeyData,((NUMROUNDS*2)+4)*4);
  i:= 0; j:= 0;
  A:= 0; B:= 0;
  if xKeyLen> ((NUMROUNDS*2)+4) then
    k:= xKeyLen*3
  else
    k:= ((NUMROUNDS*2)+4)*3;
  for k:= 1 to k do
  begin
    A:= LRot32(KeyData[i]+A+B,3);
    KeyData[i]:= A;
    B:= LRot32(xKeyD[j]+A+B,A+B);
    xKeyD[j]:= B;
    i:= (i+1) mod ((NUMROUNDS*2)+4);
    j:= (j+1) mod xKeyLen;
  end;
  FillChar(xKeyD,Sizeof(xKeyD),0);

  if IVector= nil then
  begin
    FillChar(IV,Sizeof(IV),$FF);
    Encrypt(IV,IV);
    Move(IV,LB,Sizeof(LB));
    fInitialized:= true;
  end
  else
  begin
    Move(IVector^,IV,Sizeof(IV));
    Move(IV,LB,Sizeof(IV));
  end;
end;

procedure TDCP_rc6.Burn;
begin
  FillChar(KeyData,Sizeof(KeyData),$FF);
  FillChar(IV,Sizeof(IV),$FF);
  FillChar(LB,Sizeof(LB),$FF);
  fInitialized:= false;
end;

procedure TDCP_rc6.Reset;
begin
  Move(IV,LB,Sizeof(LB));
end;

procedure TDCP_rc6.EncryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  Encrypt(InBlock,OutBlock);
end;

procedure TDCP_rc6.DecryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  Decrypt(InBlock,OutBlock);
end;

procedure TDCP_rc6.EncryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..15] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  for i:= 1 to (Size div 16) do
  begin
    XorBlock(pointer(integer(@InData)+((i-1)*16)),@LB,@TB,Sizeof(TB));
    Encrypt(TB,TB);
    Move(TB,pointer(integer(@OutData)+((i-1)*16))^,Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 16)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 16))^,@pointer(integer(@OutData)+Size-(Size mod 16))^,Size mod 16);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_rc6.DecryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..15] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  for i:= 1 to (Size div 16) do
  begin
    Move(pointer(integer(@InData)+((i-1)*16))^,TB,Sizeof(TB));
    Decrypt(pointer(integer(@InData)+((i-1)*16))^,pointer(integer(@OutData)+((i-1)*16))^);
    XorBlock(@LB,pointer(integer(@OutData)+((i-1)*16)),pointer(integer(@OutData)+((i-1)*16)),Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 16)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 16))^,@pointer(integer(@OutData)+Size-(Size mod 16))^,Size mod 16);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_rc6.EncryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..15] of byte;
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  for i:= 0 to Size-1 do
  begin
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],15);
    LB[15]:= PByteArray(@OutData)[i];
  end;
end;

procedure TDCP_rc6.DecryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..15] of byte;
  b: byte;
begin
  if not fInitialized then
    raise Exception.Create('RC6: Not initialized');
  for i:= 0 to Size-1 do
  begin
    b:= PByteArray(@InData)[i];
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],15);
    LB[15]:= b;
  end;
end;


end.
