//******************************************************************************
//** A binary compatible implementation of RC2 *********************************
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit RC2;

interface
uses
  Classes, Sysutils, DCPcrypt;

type
  TDCP_rc2= class(TDCP_blockcipher)
  protected
    IV, LB: array[0..7] of byte;
    KeyData: array[0..63] of word;
    procedure Encrypt(const InBlock; var OutBlock);
    procedure Decrypt(const InBlock; var OutBlock);
  public
    procedure Init(var Key; Size: integer; IVector: pointer); override;
    procedure Burn; override;
    procedure Reset; override;
    procedure EncryptECB(const InBlock; var OutBlock); override;
    procedure DecryptECB(const InBlock; var OutBlock); override;
    procedure EncryptCBC(const InData; var OutData; Size: integer); override;
    procedure DecryptCBC(const InData; var OutData; Size: integer); override;
    procedure EncryptCFB(const InData; var OutData; Size: integer); override;
    procedure DecryptCFB(const InData; var OutData; Size: integer); override;
    constructor Create(AOwner: TComponent); override;
  end;

//******************************************************************************
//******************************************************************************
implementation

{$R-}{$Q-}
{$I RC2.Inc}

constructor TDCP_rc2.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fID:= 1;
  fAlgorithm:= 'RC2';
  fBlockSize:= 64;
  fMaxKeySize:= 1024;
end;

procedure TDCP_rc2.Encrypt(const InBlock; var OutBlock);
var
  i, j: integer;
  w: array[0..3] of word;
begin
  Move(InBlock,w,Sizeof(w));
  for i:= 0 to 15 do
  begin
    j:= i*4;
    w[0]:= LRot16((w[0]+(w[1] and (not w[3]))+(w[2] and w[3])+KeyData[j+0]),1);
    w[1]:= LRot16((w[1]+(w[2] and (not w[0]))+(w[3] and w[0])+KeyData[j+1]),2);
    w[2]:= LRot16((w[2]+(w[3] and (not w[1]))+(w[0] and w[1])+KeyData[j+2]),3);
    w[3]:= LRot16((w[3]+(w[0] and (not w[2]))+(w[1] and w[2])+KeyData[j+3]),5);
    if (i= 4) or (i= 10) then
    begin
      w[0]:= w[0]+KeyData[w[3] and 63];
      w[1]:= w[1]+KeyData[w[0] and 63];
      w[2]:= w[2]+KeyData[w[1] and 63];
      w[3]:= w[3]+KeyData[w[2] and 63];
    end;
  end;
  Move(w,OutBlock,Sizeof(w));
end;

procedure TDCP_rc2.Decrypt(const InBlock; var OutBlock);
var
  i, j: integer;
  w: array[0..3] of word;
begin
  Move(InBlock,w,Sizeof(w));
  for i:= 15 downto 0 do
  begin
    j:= i*4;
    w[3]:= RRot16(w[3],5)-(w[0] and (not w[2]))-(w[1] and w[2])-KeyData[j+3];
    w[2]:= RRot16(w[2],3)-(w[3] and (not w[1]))-(w[0] and w[1])-KeyData[j+2];
    w[1]:= RRot16(w[1],2)-(w[2] and (not w[0]))-(w[3] and w[0])-KeyData[j+1];
    w[0]:= RRot16(w[0],1)-(w[1] and (not w[3]))-(w[2] and w[3])-KeyData[j+0];
    if (i= 5) or (i= 11) then
    begin
      w[3]:= w[3]-KeyData[w[2] and 63];
      w[2]:= w[2]-KeyData[w[1] and 63];
      w[1]:= w[1]-KeyData[w[0] and 63];
      w[0]:= w[0]-KeyData[w[3] and 63];
    end;
  end;
  Move(w,OutBlock,Sizeof(w));
end;

procedure TDCP_rc2.Init(var Key; Size: integer; IVector: pointer);
var
  i: integer;
  KeyB: array[0..127] of byte;
begin
  if fInitialized then
    Burn;
  if (Size> fMaxKeySize) or (Size<= 0) or ((Size mod 8)<> 0) then
    Exception.Create(Format('RC2: Invalid key size - %d',[Size]));
  Move(Key,KeyB,Size div 8);
  for i:= (Size div 8) to 127 do
    KeyB[i]:= sBox[(KeyB[i-(Size div 8)]+KeyB[i-1]) and $FF];
  KeyB[0]:= sBox[KeyB[0]];
  Move(KeyB,KeyData,Sizeof(KeyData));

  if IVector= nil then
  begin
    FillChar(IV,Sizeof(IV),$FF);
    Encrypt(IV,IV);
    Move(IV,LB,Sizeof(LB));
    fInitialized:= true;
  end
  else
  begin
    Move(IVector^,IV,Sizeof(IV));
    Move(IV,LB,Sizeof(IV));
  end;
end;

procedure TDCP_rc2.Burn;
begin
  FillChar(KeyData,Sizeof(KeyData),$FF);
  FillChar(IV,Sizeof(IV),$FF);
  FillChar(LB,Sizeof(LB),$FF);
  fInitialized:= false;
end;

procedure TDCP_rc2.Reset;
begin
  Move(IV,LB,Sizeof(LB));
end;

procedure TDCP_rc2.EncryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  Encrypt(InBlock,OutBlock);
end;

procedure TDCP_rc2.DecryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  Decrypt(InBlock,OutBlock);
end;

procedure TDCP_rc2.EncryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    XorBlock(pointer(integer(@InData)+((i-1)*8)),@LB,@TB,Sizeof(TB));
    Encrypt(TB,TB);
    Move(TB,pointer(integer(@OutData)+((i-1)*8))^,Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_rc2.DecryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    Move(pointer(integer(@InData)+((i-1)*8))^,TB,Sizeof(TB));
    Decrypt(pointer(integer(@InData)+((i-1)*8))^,pointer(integer(@OutData)+((i-1)*8))^);
    XorBlock(@LB,pointer(integer(@OutData)+((i-1)*8)),pointer(integer(@OutData)+((i-1)*8)),Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_rc2.EncryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  for i:= 0 to Size-1 do
  begin
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= PByteArray(@OutData)[i];
  end;
end;

procedure TDCP_rc2.DecryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
  b: byte;
begin
  if not fInitialized then
    raise Exception.Create('RC2: Not initialized');
  for i:= 0 to Size-1 do
  begin
    b:= PByteArray(@InData)[i];
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= b;
  end;
end;


end.
