//******************************************************************************
//** A binary compatible implementation of Misty1 ******************************
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit Misty1;

interface
uses
  Classes, Sysutils, DCPcrypt;

const
  NUMROUNDS= 8;

type
  TDCP_misty1= class(TDCP_blockcipher)
  protected
    IV, LB: array[0..7] of byte;
    KeyData: array[0..31] of DWord;
    function FI(const FI_IN, FI_KEY: DWord): DWord;
    function FO(const FO_IN: DWord; const k: integer): DWord;
    function FL(const FL_IN: DWord; const k: integer): DWord;
    function FLINV(const FL_IN: DWord; const k: integer): DWord;
    procedure Encrypt(const InBlock; var OutBlock);
    procedure Decrypt(const InBlock; var OutBlock);
  public
    procedure Init(var Key; Size: integer; IVector: pointer); override;
    procedure Burn; override;
    procedure Reset; override;
    procedure EncryptECB(const InBlock; var OutBlock); override;
    procedure DecryptECB(const InBlock; var OutBlock); override;
    procedure EncryptCBC(const InData; var OutData; Size: integer); override;
    procedure DecryptCBC(const InData; var OutData; Size: integer); override;
    procedure EncryptCFB(const InData; var OutData; Size: integer); override;
    procedure DecryptCFB(const InData; var OutData; Size: integer); override;
    constructor Create(AOwner: TComponent); override;
  end;

//******************************************************************************
//******************************************************************************
implementation

{$R-}{$Q-}
{$I Misty1.Inc}

constructor TDCP_misty1.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fAlgorithm:= 'Misty1';
  fBlockSize:= 64;
  fMaxKeySize:= 128;
  fID:= 11;
  Burn;
end;

function TDCP_misty1.FI(const FI_IN, FI_KEY: DWord): DWord;
var
  d7, d9: DWord;
begin
  d9:= (FI_IN shr 7) and $1ff;
  d7:= FI_IN and $7f;
  d9:= S9Table[d9] xor d7;
  d7:= (S7Table[d7] xor d9) and $7f;
  d7:= d7 xor ((FI_KEY shr 9) and $7f);
  d9:= d9 xor (FI_KEY and $1ff);
  d9:= S9Table[d9] xor d7;
  Result:= (d7 shl 9) or d9;
end;

function TDCP_misty1.FO(const FO_IN: DWord; const k: integer): DWord;
var
  t0, t1: DWord;
begin
  t0:= FO_IN shr 16;
  t1:= FO_IN and $FFFF;
  t0:= t0 xor KeyData[k];
  t0:= FI(t0,KeyData[((k+5) mod 8) + 8]);
  t0:= t0 xor t1;
  t1:= t1 xor KeyData[(k+2) mod 8];
  t1:= FI(t1,KeyData[((k+1) mod 8) + 8]);
  t1:= t1 xor t0;
  t0:= t0 xor KeyData[(k+7) mod 8];
  t0:= FI(t0,KeyData[((k+3) mod 8) + 8]);
  t0:= t0 xor t1;
  t1:= t1 xor KeyData[(k+4) mod 8];
  Result:= (t1 shl 16) or t0;
end;

function TDCP_misty1.FL(const FL_IN: DWord; const k: integer): DWord;
var
  d0, d1: DWord;
  t: byte;
begin
  d0:= FL_IN shr 16;
  d1:= FL_IN and $FFFF;
  if (k mod 2)<> 0 then
  begin
    t:= (k-1) div 2;
    d1:= d1 xor (d0 and KeyData[((t + 2) mod 8) + 8]);
    d0:= d0 xor (d1 or KeyData[(t + 4) mod 8]);
  end
  else
  begin
    t:= k div 2;
    d1:= d1 xor (d0 and KeyData[t]);
    d0:= d0 xor (d1 or KeyData[((t+6) mod 8) + 8]);
  end;
  Result:= (d0 shl 16) or d1;
end;

function TDCP_misty1.FLINV(const FL_IN: DWord; const k: integer): DWord;
var
  d0, d1: DWord;
  t: byte;
begin
  d0:= FL_IN shr 16;
  d1:= FL_IN and $FFFF;
  if (k mod 2)<> 0 then
  begin
    t:= (k-1) div 2;
    d0:= d0 xor (d1 or KeyData[(t+4) mod 8]);
    d1:= d1 xor (d0 and KeyData[((t+2) mod 8) + 8]);
  end
  else
  begin
    t:= k div 2;
    d0:= d0 xor (d1 or KeyData[((t+6) mod 8) + 8]);
    d1:= d1 xor (d0 and KeyData[t]);
  end;
  Result:= (d0 shl 16) or d1;
end;

procedure TDCP_misty1.Encrypt(const InBlock; var OutBlock);
var
  d0, d1: DWord;
  i: integer;
begin
  Move(InBlock,d0,4);
  Move(pointer(integer(@InBlock)+4)^,d1,4);
  for i:= 0 to NUMROUNDS-1 do
  begin
    if (i mod 2)= 0 then
    begin
      d0:= FL(D0,i);
      d1:= FL(D1,i+1);
      d1:= d1 xor FO(d0,i);
    end
    else
      d0:= d0 xor FO(d1,i);
  end;
  d0:= FL(d0,NUMROUNDS);
  d1:= FL(d1,NUMROUNDS+1);
  Move(d1,OutBlock,4);
  Move(d0,pointer(integer(@OutBlock)+4)^,4);
end;

procedure TDCP_misty1.Decrypt(const InBlock; var OutBlock);
var
  d0, d1: DWord;
  i: integer;
begin
  Move(InBlock,d1,4);
  Move(pointer(integer(@InBlock)+4)^,d0,4);
  d1:= FLINV(d1,NUMROUNDS+1);
  d0:= FLINV(d0,NUMROUNDS);
  for i:= NUMROUNDS-1 downto 0 do
  begin
    if (i mod 2)= 0 then
    begin
      d1:= d1 xor FO(d0,i);
      d0:= FLINV(D0,i);
      d1:= FLINV(D1,i+1);
    end
    else
      d0:= d0 xor FO(d1,i);
  end;
  Move(d0,OutBlock,4);
  Move(d1,pointer(integer(@OutBlock)+4)^,4);
end;

procedure TDCP_misty1.Init(var Key; Size: integer; IVector: pointer);
var
  KeyB: array[0..15] of byte;
  i: integer;
begin
  if fInitialized then
    Burn;
  if (Size> fMaxKeySize) or (Size<= 0) or ((Size mod 8)<> 0) then
    Exception.Create(Format('Misty1: Invalid key size - %d',[Size]));
  FillChar(KeyB,Sizeof(KeyB),0);
  Move(Key,KeyB,Size div 8);
  for i:= 0 to 7 do
    KeyData[i]:= (KeyB[i*2] * 256) + KeyB[i*2+1];
  for i:= 0 to 7 do
  begin
    KeyData[i+8]:= FI(KeyData[i],KeyData[(i+1) mod 8]);
    KeyData[i+16]:= KeyData[i+8] and $1FF;
    KeyData[i+24]:= KeyData[i+8] shr 9;
  end;

  if IVector= nil then
  begin
    FillChar(IV,Sizeof(IV),$FF);
    Encrypt(IV,IV);
    Move(IV,LB,Sizeof(LB));
    fInitialized:= true;
  end
  else
  begin
    Move(IVector^,IV,Sizeof(IV));
    Move(IV,LB,Sizeof(IV));
  end;
end;

procedure TDCP_misty1.Burn;
begin
  FillChar(KeyData,Sizeof(KeyData),$FF);
  FillChar(IV,Sizeof(IV),$FF);
  FillChar(LB,Sizeof(LB),$FF);
  fInitialized:= false;
end;

procedure TDCP_misty1.Reset;
begin
  Move(IV,LB,Sizeof(LB));
end;

procedure TDCP_misty1.EncryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  Encrypt(InBlock,OutBlock);
end;

procedure TDCP_misty1.DecryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  Decrypt(InBlock,OutBlock);
end;

procedure TDCP_misty1.EncryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    XorBlock(pointer(integer(@InData)+((i-1)*8)),@LB,@TB,Sizeof(TB));
    Encrypt(TB,TB);
    Move(TB,pointer(integer(@OutData)+((i-1)*8))^,Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_misty1.DecryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    Move(pointer(integer(@InData)+((i-1)*8))^,TB,Sizeof(TB));
    Decrypt(pointer(integer(@InData)+((i-1)*8))^,pointer(integer(@OutData)+((i-1)*8))^);
    XorBlock(@LB,pointer(integer(@OutData)+((i-1)*8)),pointer(integer(@OutData)+((i-1)*8)),Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_misty1.EncryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  for i:= 0 to Size-1 do
  begin
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= PByteArray(@OutData)[i];
  end;
end;

procedure TDCP_misty1.DecryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
  b: byte;
begin
  if not fInitialized then
    raise Exception.Create('Misty1: Not initialized');
  for i:= 0 to Size-1 do
  begin
    b:= PByteArray(@InData)[i];
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= b;
  end;
end;


end. 
