//******************************************************************************
//** A binary compatible implementation of IDEA ********************************
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit IDEA;

interface
uses
  Classes, Sysutils, DCPcrypt;


type
  TDCP_idea= class(TDCP_blockcipher)
  protected
    IV, LB: array[0..7] of byte;
    EK: array[0..51] of word;
    DK: array[0..51] of word;
    procedure Encrypt(const InBlock; var OutBlock);
    procedure Decrypt(const InBlock; var OutBlock);
  public
    procedure Init(var Key; Size: integer; IVector: pointer); override;
    procedure Burn; override;
    procedure Reset; override;
    procedure EncryptECB(const InBlock; var OutBlock); override;
    procedure DecryptECB(const InBlock; var OutBlock); override;
    procedure EncryptCBC(const InData; var OutData; Size: integer); override;
    procedure DecryptCBC(const InData; var OutData; Size: integer); override;
    procedure EncryptCFB(const InData; var OutData; Size: integer); override;
    procedure DecryptCFB(const InData; var OutData; Size: integer); override;
    constructor Create(AOwner: TComponent); override;
  end;

//******************************************************************************
//******************************************************************************
implementation

{$R-}{$Q-}

constructor TDCP_idea.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fAlgorithm:= 'IDEA';
  fBlockSize:= 64;
  fMaxKeySize:= 128;
  fID:= 12;
  Burn;
end;

procedure Mul(var x: word; const y: word);
var
  p: DWord;
  t16: word;
begin
  p:= DWord(x)*y;
  if p= 0 then
    x:= 1 - x - y
  else
  begin
    x:= p shr 16;
    t16:= p and $FFFF;
    x:= t16 - x;
    if (t16 < x) then
      Inc(x);
  end;
end;

function MulInv(x: word): word;
var
  t0, t1, q, y: word;
begin
  if x<= 1 then
  begin
    Result:= x;
    Exit;
  end;
  t1:= DWord($10001) div x;
  y:= DWord($10001) mod x;
  if y= 1 then
  begin
    Result:= (1 - t1) and $FFFF;
    Exit;
  end;
  t0:= 1;
  repeat
    q:= x div y;
    x:= x mod y;
    t0:= t0 + (q*t1);
    if x= 1 then
    begin
      Result:= t0;
      Exit;
    end;
    q:= y div x;
    y:= y mod x;
    t1:= t1 + (q*t0);
  until y= 1;
  Result:= (1-t1) and $FFFF;
end;

procedure TDCP_idea.Encrypt(const InBlock; var OutBlock);
var
  x: array[1..4] of word;
  s3, s2: word;
  i: integer;
begin
  Move(InBlock,x,Sizeof(x));
  for i:= 1 to 4 do
    x[i]:= (x[i] shl 8) or (x[i] shr 8);
  for i:= 0 to 7 do
  begin
    Mul(x[1],EK[(i*6)+0]);
    Inc(x[2],EK[(i*6)+1]);
    Inc(x[3],EK[(i*6)+2]);
    Mul(x[4],EK[(i*6)+3]);
    s3:= x[3];
    x[3]:= x[3] xor x[1];
    Mul(x[3],EK[(i*6)+4]);
    s2:= x[2];
    x[2]:= x[2] xor x[4];
    Inc(x[2],x[3]);
    Mul(x[2],EK[(i*6)+5]);
    Inc(x[3],x[2]);
    x[1]:= x[1] xor x[2];
    x[4]:= x[4] xor x[3];
    x[2]:= x[2] xor s3;
    x[3]:= x[3] xor s2;
  end;
  Mul(x[1],EK[48]);
  Inc(x[3],EK[49]);
  Inc(x[2],EK[50]);
  Mul(x[4],EK[51]);
  x[1]:= (x[1] shl 8) or (x[1] shr 8);
  s2:= (x[3] shl 8) or (x[3] shr 8);
  x[3]:= (x[2] shl 8) or (x[2] shr 8);
  x[4]:= (x[4] shl 8) or (x[4] shr 8);
  x[2]:= s2;
  Move(x,OutBlock,Sizeof(x));
end;

procedure TDCP_idea.Decrypt(const InBlock; var OutBlock);
var
  x: array[1..4] of word;
  s3, s2: word;
  i: integer;
begin
  Move(InBlock,x,Sizeof(x));
  for i:= 1 to 4 do
    x[i]:= (x[i] shl 8) or (x[i] shr 8);
  for i:= 0 to 7 do
  begin
    Mul(x[1],DK[(i*6)+0]);
    Inc(x[2],DK[(i*6)+1]);
    Inc(x[3],DK[(i*6)+2]);
    Mul(x[4],DK[(i*6)+3]);
    s3:= x[3];
    x[3]:= x[3] xor x[1];
    Mul(x[3],DK[(i*6)+4]);
    s2:= x[2];
    x[2]:= x[2] xor x[4];
    Inc(x[2],x[3]);
    Mul(x[2],DK[(i*6)+5]);
    Inc(x[3],x[2]);
    x[1]:= x[1] xor x[2];
    x[4]:= x[4] xor x[3];
    x[2]:= x[2] xor s3;
    x[3]:= x[3] xor s2;
  end;
  Mul(x[1],DK[48]);
  Inc(x[3],DK[49]);
  Inc(x[2],DK[50]);
  Mul(x[4],DK[51]);
  x[1]:= (x[1] shl 8) or (x[1] shr 8);
  s2:= (x[3] shl 8) or (x[3] shr 8);
  x[3]:= (x[2] shl 8) or (x[2] shr 8);
  x[4]:= (x[4] shl 8) or (x[4] shr 8);
  x[2]:= s2;
  Move(x,OutBlock,Sizeof(x));
end;

procedure InvertKey(EK, DK: PWord);
var
  i: integer;
  t1, t2, t3: word;
  temp: array[0..51] of word;
  p: PWord;
begin
  p:= pointer(integer(@temp)+Sizeof(Temp));
  Dec(p);
  t1:= MulInv(EK^);
  Inc(EK);
  t2:= -EK^;
  Inc(EK);
  t3:= -EK^;
  Inc(EK);
  p^:= MulInv(EK^);
  Inc(EK);
  Dec(p);
  p^:= t3;
  Dec(p);
  p^:= t2;
  Dec(p);
  p^:= t1;
  Dec(p);
  for i:= 0 to 6 do
  begin
    t1:= EK^;
    Inc(EK);
    p^:= EK^;
    Inc(EK);
    Dec(p);
    p^:= t1;
    Dec(p);
    t1:= MulInv(EK^);
    Inc(EK);
    t2:= -EK^;
    Inc(EK);
    t3:= -EK^;
    Inc(EK);
    p^:= MulInv(EK^);
    Inc(EK);
    Dec(p);
    p^:= t2;
    Dec(p);
    p^:= t3;
    Dec(p);
    p^:= t1;
    Dec(p);
  end;
  t1:= EK^;
  Inc(EK);
  p^:= EK^;
  Dec(p);
  Inc(EK);
  p^:= t1;
  Dec(p);

  t1:= MulInv(EK^);
  Inc(EK);
  t2:= -EK^;
  Inc(EK);
  t3:= -EK^;
  Inc(EK);
  p^:= MulInv(EK^);
  Dec(p);
  p^:= t3;
  Dec(p);
  p^:= t2;
  Dec(p);
  p^:= t1;
  Move(Temp,DK^,Sizeof(Temp));
  FillChar(Temp,Sizeof(Temp),0);
end;

procedure TDCP_idea.Init(var Key; Size: integer; IVector: pointer);
var
  i: integer;
begin
  if fInitialized then
    Burn;
  if (Size> fMaxKeySize) or (Size<= 0) or ((Size mod 8)<> 0) then
    Exception.Create(Format('IDEA: Invalid key size - %d',[Size]));
  Size:= Size div 8;
  FillChar(EK,Sizeof(EK),0);
  Move(Key,EK,Size);
  for i:= 0 to 7 do
    EK[i]:= (EK[i] shl 8) or (EK[i] shr 8);
  for i:= 1 to 5 do
  begin
    EK[(i*8)+0]:= (EK[((i-1)*8)+1] shl 9) or (EK[((i-1)*8)+2] shr 7);
    EK[(i*8)+1]:= (EK[((i-1)*8)+2] shl 9) or (EK[((i-1)*8)+3] shr 7);
    EK[(i*8)+2]:= (EK[((i-1)*8)+3] shl 9) or (EK[((i-1)*8)+4] shr 7);
    EK[(i*8)+3]:= (EK[((i-1)*8)+4] shl 9) or (EK[((i-1)*8)+5] shr 7);
    EK[(i*8)+4]:= (EK[((i-1)*8)+5] shl 9) or (EK[((i-1)*8)+6] shr 7);
    EK[(i*8)+5]:= (EK[((i-1)*8)+6] shl 9) or (EK[((i-1)*8)+7] shr 7);
    EK[(i*8)+6]:= (EK[((i-1)*8)+7] shl 9) or (EK[((i-1)*8)+0] shr 7);
    EK[(i*8)+7]:= (EK[((i-1)*8)+0] shl 9) or (EK[((i-1)*8)+1] shr 7);
  end;
  EK[48]:= (EK[41] shl 9) or (EK[42] shr 7);
  EK[49]:= (EK[42] shl 9) or (EK[43] shr 7);
  EK[50]:= (EK[43] shl 9) or (EK[44] shr 7);
  EK[51]:= (EK[44] shl 9) or (EK[45] shr 7);
  InvertKey(@EK,@DK);

  if IVector= nil then
  begin
    FillChar(IV,Sizeof(IV),$FF);
    Encrypt(IV,IV);
    Move(IV,LB,Sizeof(LB));
    fInitialized:= true;
  end
  else
  begin
    Move(IVector^,IV,Sizeof(IV));
    Move(IV,LB,Sizeof(IV));
  end;
end;

procedure TDCP_idea.Burn;
begin
  FillChar(EK,Sizeof(EK),$FF);
  FillChar(DK,Sizeof(DK),$FF);
  FillChar(IV,Sizeof(IV),$FF);
  FillChar(LB,Sizeof(LB),$FF);
  fInitialized:= false;
end;

procedure TDCP_idea.Reset;
begin
  Move(IV,LB,Sizeof(LB));
end;

procedure TDCP_idea.EncryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  Encrypt(InBlock,OutBlock);
end;

procedure TDCP_idea.DecryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  Decrypt(InBlock,OutBlock);
end;

procedure TDCP_idea.EncryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    XorBlock(pointer(integer(@InData)+((i-1)*8)),@LB,@TB,Sizeof(TB));
    Encrypt(TB,TB);
    Move(TB,pointer(integer(@OutData)+((i-1)*8))^,Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_idea.DecryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    Move(pointer(integer(@InData)+((i-1)*8))^,TB,Sizeof(TB));
    Decrypt(pointer(integer(@InData)+((i-1)*8))^,pointer(integer(@OutData)+((i-1)*8))^);
    XorBlock(@LB,pointer(integer(@OutData)+((i-1)*8)),pointer(integer(@OutData)+((i-1)*8)),Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_idea.EncryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  for i:= 0 to Size-1 do
  begin
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= PByteArray(@OutData)[i];
  end;
end;

procedure TDCP_idea.DecryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
  b: byte;
begin
  if not fInitialized then
    raise Exception.Create('IDEA: Not initialized');
  for i:= 0 to Size-1 do
  begin
    b:= PByteArray(@InData)[i];
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= b;
  end;
end;


end.
