//******************************************************************************
//** A binary compatible implementation of Gost ********************************
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit Gost;

interface
uses
  Classes, Sysutils, DCPcrypt;

type
  TDCP_gost= class(TDCP_blockcipher)
  protected
    IV, LB: array[0..7] of byte;
    KeyData: array[0..7] of DWord;
    procedure Encrypt(const InBlock; var OutBlock);
    procedure Decrypt(const InBlock; var OutBlock);
  public
    procedure Init(var Key; Size: integer; IVector: pointer); override;
    procedure Burn; override;
    procedure Reset; override;
    procedure EncryptECB(const InBlock; var OutBlock); override;
    procedure DecryptECB(const InBlock; var OutBlock); override;
    procedure EncryptCBC(const InData; var OutData; Size: integer); override;
    procedure DecryptCBC(const InData; var OutData; Size: integer); override;
    procedure EncryptCFB(const InData; var OutData; Size: integer); override;
    procedure DecryptCFB(const InData; var OutData; Size: integer); override;
    constructor Create(AOwner: TComponent); override;
  end;

//******************************************************************************
//******************************************************************************
implementation

{$R-}{$Q-}
{$I Gost.Inc}

constructor TDCP_gost.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fAlgorithm:= 'Gost';
  fBlockSize:= 64;
  fMaxKeySize:= 256;
  fID:= 8;
  Burn;
end;

procedure TDCP_gost.Encrypt(const InBlock; var OutBlock);
var
  n1, n2: DWord;
  i: integer;
begin
  Move(InBlock,n1,4);
  Move(pointer(integer(@InBlock)+4)^,n2,4);
  for i:= 0 to 2 do
  begin
    n2:= n2 xor (sTable[3,(n1+KeyData[0]) shr 24] xor sTable[2,((n1+KeyData[0]) shr 16) and $FF] xor sTable[1,((n1+KeyData[0]) shr 8) and $FF] xor sTable[0,(n1+KeyData[0]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[1]) shr 24] xor sTable[2,((n2+KeyData[1]) shr 16) and $FF] xor sTable[1,((n2+KeyData[1]) shr 8) and $FF] xor sTable[0,(n2+KeyData[1]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[2]) shr 24] xor sTable[2,((n1+KeyData[2]) shr 16) and $FF] xor sTable[1,((n1+KeyData[2]) shr 8) and $FF] xor sTable[0,(n1+KeyData[2]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[3]) shr 24] xor sTable[2,((n2+KeyData[3]) shr 16) and $FF] xor sTable[1,((n2+KeyData[3]) shr 8) and $FF] xor sTable[0,(n2+KeyData[3]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[4]) shr 24] xor sTable[2,((n1+KeyData[4]) shr 16) and $FF] xor sTable[1,((n1+KeyData[4]) shr 8) and $FF] xor sTable[0,(n1+KeyData[4]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[5]) shr 24] xor sTable[2,((n2+KeyData[5]) shr 16) and $FF] xor sTable[1,((n2+KeyData[5]) shr 8) and $FF] xor sTable[0,(n2+KeyData[5]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[6]) shr 24] xor sTable[2,((n1+KeyData[6]) shr 16) and $FF] xor sTable[1,((n1+KeyData[6]) shr 8) and $FF] xor sTable[0,(n1+KeyData[6]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[7]) shr 24] xor sTable[2,((n2+KeyData[7]) shr 16) and $FF] xor sTable[1,((n2+KeyData[7]) shr 8) and $FF] xor sTable[0,(n2+KeyData[7]) and $FF]);
  end;
  n2:= n2 xor (sTable[3,(n1+KeyData[7]) shr 24] xor sTable[2,((n1+KeyData[7]) shr 16) and $FF] xor sTable[1,((n1+KeyData[7]) shr 8) and $FF] xor sTable[0,(n1+KeyData[7]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[6]) shr 24] xor sTable[2,((n2+KeyData[6]) shr 16) and $FF] xor sTable[1,((n2+KeyData[6]) shr 8) and $FF] xor sTable[0,(n2+KeyData[6]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[5]) shr 24] xor sTable[2,((n1+KeyData[5]) shr 16) and $FF] xor sTable[1,((n1+KeyData[5]) shr 8) and $FF] xor sTable[0,(n1+KeyData[5]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[4]) shr 24] xor sTable[2,((n2+KeyData[4]) shr 16) and $FF] xor sTable[1,((n2+KeyData[4]) shr 8) and $FF] xor sTable[0,(n2+KeyData[4]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[3]) shr 24] xor sTable[2,((n1+KeyData[3]) shr 16) and $FF] xor sTable[1,((n1+KeyData[3]) shr 8) and $FF] xor sTable[0,(n1+KeyData[3]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[2]) shr 24] xor sTable[2,((n2+KeyData[2]) shr 16) and $FF] xor sTable[1,((n2+KeyData[2]) shr 8) and $FF] xor sTable[0,(n2+KeyData[2]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[1]) shr 24] xor sTable[2,((n1+KeyData[1]) shr 16) and $FF] xor sTable[1,((n1+KeyData[1]) shr 8) and $FF] xor sTable[0,(n1+KeyData[1]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[0]) shr 24] xor sTable[2,((n2+KeyData[0]) shr 16) and $FF] xor sTable[1,((n2+KeyData[0]) shr 8) and $FF] xor sTable[0,(n2+KeyData[0]) and $FF]);
  Move(n2,OutBlock,4);
  Move(n1,pointer(integer(@OutBlock)+4)^,4);
end;

procedure TDCP_gost.Decrypt(const InBlock; var OutBlock);
var
  n1, n2: DWord;
  i: integer;
begin
  Move(InBlock,n1,4);
  Move(pointer(integer(@InBlock)+4)^,n2,4);
  n2:= n2 xor (sTable[3,(n1+KeyData[0]) shr 24] xor sTable[2,((n1+KeyData[0]) shr 16) and $FF] xor sTable[1,((n1+KeyData[0]) shr 8) and $FF] xor sTable[0,(n1+KeyData[0]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[1]) shr 24] xor sTable[2,((n2+KeyData[1]) shr 16) and $FF] xor sTable[1,((n2+KeyData[1]) shr 8) and $FF] xor sTable[0,(n2+KeyData[1]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[2]) shr 24] xor sTable[2,((n1+KeyData[2]) shr 16) and $FF] xor sTable[1,((n1+KeyData[2]) shr 8) and $FF] xor sTable[0,(n1+KeyData[2]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[3]) shr 24] xor sTable[2,((n2+KeyData[3]) shr 16) and $FF] xor sTable[1,((n2+KeyData[3]) shr 8) and $FF] xor sTable[0,(n2+KeyData[3]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[4]) shr 24] xor sTable[2,((n1+KeyData[4]) shr 16) and $FF] xor sTable[1,((n1+KeyData[4]) shr 8) and $FF] xor sTable[0,(n1+KeyData[4]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[5]) shr 24] xor sTable[2,((n2+KeyData[5]) shr 16) and $FF] xor sTable[1,((n2+KeyData[5]) shr 8) and $FF] xor sTable[0,(n2+KeyData[5]) and $FF]);
  n2:= n2 xor (sTable[3,(n1+KeyData[6]) shr 24] xor sTable[2,((n1+KeyData[6]) shr 16) and $FF] xor sTable[1,((n1+KeyData[6]) shr 8) and $FF] xor sTable[0,(n1+KeyData[6]) and $FF]);
  n1:= n1 xor (sTable[3,(n2+KeyData[7]) shr 24] xor sTable[2,((n2+KeyData[7]) shr 16) and $FF] xor sTable[1,((n2+KeyData[7]) shr 8) and $FF] xor sTable[0,(n2+KeyData[7]) and $FF]);
  for i:= 0 to 2 do
  begin
    n2:= n2 xor (sTable[3,(n1+KeyData[7]) shr 24] xor sTable[2,((n1+KeyData[7]) shr 16) and $FF] xor sTable[1,((n1+KeyData[7]) shr 8) and $FF] xor sTable[0,(n1+KeyData[7]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[6]) shr 24] xor sTable[2,((n2+KeyData[6]) shr 16) and $FF] xor sTable[1,((n2+KeyData[6]) shr 8) and $FF] xor sTable[0,(n2+KeyData[6]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[5]) shr 24] xor sTable[2,((n1+KeyData[5]) shr 16) and $FF] xor sTable[1,((n1+KeyData[5]) shr 8) and $FF] xor sTable[0,(n1+KeyData[5]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[4]) shr 24] xor sTable[2,((n2+KeyData[4]) shr 16) and $FF] xor sTable[1,((n2+KeyData[4]) shr 8) and $FF] xor sTable[0,(n2+KeyData[4]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[3]) shr 24] xor sTable[2,((n1+KeyData[3]) shr 16) and $FF] xor sTable[1,((n1+KeyData[3]) shr 8) and $FF] xor sTable[0,(n1+KeyData[3]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[2]) shr 24] xor sTable[2,((n2+KeyData[2]) shr 16) and $FF] xor sTable[1,((n2+KeyData[2]) shr 8) and $FF] xor sTable[0,(n2+KeyData[2]) and $FF]);
    n2:= n2 xor (sTable[3,(n1+KeyData[1]) shr 24] xor sTable[2,((n1+KeyData[1]) shr 16) and $FF] xor sTable[1,((n1+KeyData[1]) shr 8) and $FF] xor sTable[0,(n1+KeyData[1]) and $FF]);
    n1:= n1 xor (sTable[3,(n2+KeyData[0]) shr 24] xor sTable[2,((n2+KeyData[0]) shr 16) and $FF] xor sTable[1,((n2+KeyData[0]) shr 8) and $FF] xor sTable[0,(n2+KeyData[0]) and $FF]);
  end;
  Move(n2,OutBlock,4);
  Move(n1,pointer(integer(@OutBlock)+4)^,4);
end;

procedure TDCP_gost.Init(var Key; Size: integer; IVector: pointer);
var
  i: integer;
  userkey: array[0..31] of byte;
begin
  if fInitialized then
    Burn;
  if (Size> fMaxKeySize) or (Size<= 0) or ((Size mod 8)<> 0) then
    Exception.Create(Format('Gost: Invalid key size - %d',[Size]));
  Size:= Size div 8;
  FillChar(userkey,Sizeof(userkey),0);
  Move(Key,userkey,Size);
  for i:= 0 to 7 do
    KeyData[i]:= (UserKey[4*i+3] shl 24) or (UserKey[4*i+2] shl 16) or
      (UserKey[4*i+1] shl 8) or (UserKey[4*i+0]);

  if IVector= nil then
  begin
    FillChar(IV,Sizeof(IV),$FF);
    Encrypt(IV,IV);
    Move(IV,LB,Sizeof(LB));
    fInitialized:= true;
  end
  else
  begin
    Move(IVector^,IV,Sizeof(IV));
    Move(IV,LB,Sizeof(IV));
  end;
end;

procedure TDCP_gost.Burn;
begin
  FillChar(KeyData,Sizeof(KeyData),$FF);
  FillChar(IV,Sizeof(IV),$FF);
  FillChar(LB,Sizeof(LB),$FF);
  fInitialized:= false;
end;

procedure TDCP_gost.Reset;
begin
  Move(IV,LB,Sizeof(LB));
end;

procedure TDCP_gost.EncryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  Encrypt(InBlock,OutBlock);
end;

procedure TDCP_gost.DecryptECB(const InBlock; var OutBlock);
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  Decrypt(InBlock,OutBlock);
end;

procedure TDCP_gost.EncryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    XorBlock(pointer(integer(@InData)+((i-1)*8)),@LB,@TB,Sizeof(TB));
    Encrypt(TB,TB);
    Move(TB,pointer(integer(@OutData)+((i-1)*8))^,Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_gost.DecryptCBC(const InData; var OutData; Size: integer);
var
  TB: array[0..7] of byte;
  i: integer;
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  for i:= 1 to (Size div 8) do
  begin
    Move(pointer(integer(@InData)+((i-1)*8))^,TB,Sizeof(TB));
    Decrypt(pointer(integer(@InData)+((i-1)*8))^,pointer(integer(@OutData)+((i-1)*8))^);
    XorBlock(@LB,pointer(integer(@OutData)+((i-1)*8)),pointer(integer(@OutData)+((i-1)*8)),Sizeof(TB));
    Move(TB,LB,Sizeof(TB));
  end;
  if (Size mod 8)<> 0 then
  begin
    Encrypt(LB,TB);
    XorBlock(@TB,@pointer(integer(@InData)+Size-(Size mod 8))^,@pointer(integer(@OutData)+Size-(Size mod 8))^,Size mod 8);
  end;
  FillChar(TB,Sizeof(TB),$FF);
end;

procedure TDCP_gost.EncryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  for i:= 0 to Size-1 do
  begin
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= PByteArray(@OutData)[i];
  end;
end;

procedure TDCP_gost.DecryptCFB(const InData; var OutData; Size: integer);
var
  i: integer;
  TB: array[0..7] of byte;
  b: byte;
begin
  if not fInitialized then
    raise Exception.Create('Gost: Not initialized');
  for i:= 0 to Size-1 do
  begin
    b:= PByteArray(@InData)[i];
    Encrypt(LB,TB);
    PByteArray(@OutData)[i]:= PByteArray(@InData)[i] xor TB[0];
    Move(LB[1],LB[0],7);
    LB[7]:= b;
  end;
end;


end.
