//******************************************************************************
//** Base classes for block cipher and hash algorithm implementations **********
//******************************************************************************
//** Written by David Barton (davebarton@bigfoot.com) **************************
//** http://www.hertreg.ac.uk/ss/ **********************************************
//******************************************************************************
unit DCPcrypt;

interface
uses
  Classes, Sysutils;

{$M+}

const
  DCPpage= 'DCPcrypt';

type
  pword= ^word;
  pdword= ^dword;
{$IFDEF VER120}
  dword= longword;
{$ELSE}
  dword= longint;
{$ENDIF}
  Pdwordarray= ^Tdwordarray;
  Tdwordarray= array[0..1023] of dword;

//******************************************************************************
//******************************************************************************
type
  TDCP_blockcipher= class(TComponent)
  protected
    fID: integer;
    fInitialized: boolean;
    fAlgorithm: string;
    fBlockSize: integer;
    fMaxKeySize: integer;
    fNullStr: string;
    fNullInt: integer;
  public
    property ID: integer
      read fID;
    property Initialized: boolean
      read fInitialized;
    procedure Init(var Key; Size: integer; IV: pointer); virtual; abstract;
    procedure InitStr(Key: string);
    procedure Burn; virtual; abstract;
    procedure Reset; virtual; abstract;
    procedure EncryptECB(const InBlock; var OutBlock); virtual; abstract;
    procedure DecryptECB(const InBlock; var OutBlock); virtual; abstract;
    procedure EncryptCBC(const InData; var OutData; Size: integer); virtual; abstract;
    procedure DecryptCBC(const InData; var OutData; Size: integer); virtual; abstract;
    procedure EncryptCFB(const InData; var OutData; Size: integer); virtual; abstract;
    procedure DecryptCFB(const InData; var OutData; Size: integer); virtual; abstract;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Algorithm: string
      read fAlgorithm write fNullStr;
    property BlockSize: integer
      read fBlockSize write fNullInt;
    property MaxKeySize: integer
      read fMaxKeySize write fNullInt;
  end;

//******************************************************************************
//******************************************************************************
type
  TDCP_hash= class(TComponent)
  protected
    fID: integer;
    fInitialized: boolean;
    fAlgorithm: string;
    fHashSize: integer;
    fNullStr: string;
    fNullInt: integer;
  public
    property ID: integer
      read fID;
    property Initialized: boolean
      read fInitialized;
    procedure Init; virtual; abstract;
    procedure Burn; virtual; abstract;
    procedure Update(const Buffer; Size: integer); virtual; abstract;
    procedure UpdateStr(const Buffer: string);
    procedure Final(var Digest); virtual; abstract;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Algorithm: string
      read fAlgorithm write fNullStr;
    property HashSize: integer
      read fHashSize write fNullInt;
  end;

//******************************************************************************
//******************************************************************************
function LRot16(X: word; c: integer): word; assembler;
function RRot16(X: word; c: integer): word; assembler;
function LRot32(X: dword; c: integer): dword; assembler;
function RRot32(X: dword; c: integer): dword; assembler;
procedure XorBlock(I1, I2, O1: PByteArray; Len: integer);

//******************************************************************************
//******************************************************************************
implementation
uses
  SHA1;

type
  TDCP_defaulthash= TDCP_sha1;

//******************************************************************************
//******************************************************************************
constructor TDCP_blockcipher.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Burn;
end;

destructor TDCP_blockcipher.Destroy;
begin
  if fInitialized then
    Burn;
  inherited Destroy;
end;

procedure TDCP_blockcipher.InitStr;
var
  Hash: TDCP_defaulthash;
  KeyHash: pointer;
begin
  Hash:= TDCP_defaulthash.Create(Self);
  Hash.Init;
  Hash.Update(Key[1],Length(Key));
  GetMem(KeyHash,Hash.HashSize div 8);
  Hash.Final(KeyHash^);
  if Hash.HashSize> fMaxKeySize then
    Init(KeyHash^,fMaxKeySize,nil)
  else
    Init(KeyHash^,Hash.HashSize,nil);
  FillChar(KeyHash^,Hash.HashSize div 8,$FF);
  FreeMem(KeyHash);
  Hash.Free;
end;

//******************************************************************************
//******************************************************************************
constructor TDCP_hash.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Burn;
end;

destructor TDCP_hash.Destroy;
begin
  if fInitialized then
    Burn;
  inherited Destroy;
end;

procedure TDCP_hash.UpdateStr(const Buffer: string);
begin
  Update(Buffer[1],Length(Buffer));
end;

//******************************************************************************
//******************************************************************************
function LRot16(X: word; c: integer): word; assembler;
asm
  mov ecx,&c
  mov ax,&X
  rol ax,cl
  mov &Result,ax
end;

function RRot16(X: word; c: integer): word; assembler;
asm
  mov ecx,&c
  mov ax,&X
  ror ax,cl
  mov &Result,ax
end;

function LRot32(X: dword; c: integer): dword; register; assembler;
asm
  mov ecx, edx
  rol eax, cl
end;

function RRot32(X: dword; c: integer): dword; register; assembler;
asm
  mov ecx, edx
  ror eax, cl
end;

procedure XorBlock(I1, I2, O1: PByteArray; Len: integer);
var
  i: integer;
begin
  for i:= 0 to Len-1 do
    O1[i]:= I1[i] xor I2[i];
end;

//******************************************************************************
//******************************************************************************
end.
