unit DiskChange;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;


type
  TNotifyFilterType = (FILE_NAME,
                       DIR_NAME,
                       ATTRIBUTES,
                       SIZE,
                       LAST_WRITE,
                       SECURITY);
  TNotifyFilters = set of TNotifyFilterType;

  TDiskChangeNotify = class;

  TDiskWatchThread = class( TThread )
  private
     DCN : TDiskChangeNotify;
     procedure NotifyDCN;
  public
     procedure Execute; override;
  end;

  TDiskChangeNotify = class(TComponent)
  private
    WatchThread : TDiskWatchThread;
    FWatchSubTree : boolean;
    FEnabled : boolean;
    FWatchDir : string;
    FOnDiskChange : TNotifyEvent;
    FNotifyFilter : TNotifyFilters;
    NotifyFilterW : word;
    procedure CreateWatchThread;
    procedure DestroyWatchThread;
  protected
    procedure SetWatchSubTree (Value : boolean);
    procedure SetEnabled (Value : boolean);
    procedure SetWatchDir (Value : string);
    procedure SetNotifyFilter (Filters : TNotifyFilters);
  public
    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property WatchDir : string read FWatchDir write SetWatchDir;
    property WatchSubTree : boolean read FWatchSubTree write SetWatchSubTree default true;
    property Enabled : boolean read FEnabled write SetEnabled default false;
    property OnDiskChange : TNotifyEvent read FOnDiskChange write FOnDiskChange;
    property NotifyFilter : TNotifyFilters read FNotifyFilter write SetNotifyFilter;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Samples', [TDiskChangeNotify]);
end;

procedure TDiskWatchThread.NotifyDCN;
begin
  DCN.OnDiskChange (DCN);
end;

procedure TDiskWatchThread.Execute;
var
  TempArr : array [0..Max_Path-1] of char;
  AHandle : THandle;
begin
  AHandle := FindFirstChangeNotification (StrPCopy (TempArr, DCN.WatchDir),
                                          DCN.WatchSubTree,
                                          DCN.NotifyFilterW);

  while (not Terminated) and (DCN <> nil) do begin
     WaitForSingleObject (AHandle, INFINITE);
     if Assigned(DCN.OnDiskChange) then
       // Synchronize (NotifyDCN);
       NotifyDCN;
     FindNextChangeNotification (AHandle);
  end;

  FindCloseChangeNotification (AHandle);
end;

constructor TDiskChangeNotify.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
  WatchThread := nil;
  FWatchDir := '';
  FWatchSubTree := true;
  FEnabled := false;
  FNotifyFilter := [FILE_NAME, DIR_NAME, ATTRIBUTES, SIZE, LAST_WRITE];
end;

destructor TDiskChangeNotify.Destroy;
begin
  Enabled := false;
  inherited Destroy;
end;

procedure TDiskChangeNotify.SetWatchSubTree (Value : boolean);
begin
  if (FWatchSubTree <> Value) then begin
    FWatchSubTree := Value;
    if (Enabled) then begin
      Enabled := false;
      Enabled := true;
    end;
  end;
end;

procedure TDiskChangeNotify.SetEnabled (Value : boolean);
begin
  if (Value) and (csDesigning in ComponentState) then begin
    ShowMessage ('Set to true at runtime only');
    Exit;
  end;

  if (not (csDesigning in ComponentState)) and (FEnabled <> Value) then begin
    FEnabled := Value;
    if (Value) then
      CreateWatchThread
    else
      DestroyWatchThread;
  end;
end;

procedure TDiskChangeNotify.SetWatchDir (Value : string);
begin
  if (CompareText (FWatchDir, Value) <> 0) then begin
    FWatchDir := Value;
    if (Enabled) then begin
      Enabled := false;
      Enabled := true;
    end;
  end;
end;

procedure TDiskChangeNotify.SetNotifyFilter (Filters : TNotifyFilters);
begin
  FNotifyFilter := Filters;
  NotifyFilterW := 0;
  if (FILE_NAME in Filters)  then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_FILE_NAME;
  if (DIR_NAME in Filters)   then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_DIR_NAME;
  if (ATTRIBUTES in Filters) then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_ATTRIBUTES;
  if (SIZE in Filters)       then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_SIZE;
  if (LAST_WRITE in Filters) then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_LAST_WRITE;
  if (SECURITY in Filters)   then NotifyFilterW := NotifyFilterW or FILE_NOTIFY_CHANGE_SECURITY;

  if (Enabled) then begin
    Enabled := false;
    Enabled := true;
  end;
end;

procedure TDiskChangeNotify.CreateWatchThread;
begin
  WatchThread := TDiskWatchThread.Create (true);
  WatchThread.DCN := self;
  WatchThread.FreeOnTerminate := true;
  WatchThread.Resume;
end;

procedure TDiskChangeNotify.DestroyWatchThread;
begin
  if (Assigned (WatchThread)) then begin
    TerminateThread (WatchThread.Handle, 0);
    WatchThread := nil;
  end;
end;

end.

