(*******************************************************************************
*  Extended DBGrid                                                             *
*  Copyright (c) 1997, Javier Tari Agullo                                      *
*  Author: Javier Tari Agullo                   E-MAIL:  cyber@ctv.es          *
********************************************************************************
*       Class: TDBGridPlus         Data Bound: YES                             *
*    Ancestor: TDBGrid                                                         *
********************************************************************************
*  Enhancements:                                                               *
*    - Cand display memos and graphics (whithout deforming them)               *
*    - Event for detecting the clicking of a column                            *
*    - Capable of row height adjustment with the mouse (better memo reading)   *
*    - Event for selecting any brush and font for any cell, selected or not.   *
*      (in short: select the colors *without* having to draw the cell)         *
********************************************************************************
*   Revision History                                                           *
*   Version 1.0                                                                *
*    - Implemented One-Line Memo drawing, and adjusted Graphic drawing         *
*    - Implemented OnColumnClick Event                                         *
*   Version 1.1                                                                *
*    - Corrected bug with empty graphic fields                                 *
*    - Implemented Multi-Line Memo drawing                                     *
*    - Surfaced DefaultRowHeight property, and implemented accordingly         *
*   Version 1.2                                                                *
*    - Event for changing the brush and font of any cell                       *
*    - Implemented Multi-Line Memo drawing                                     *
*    - Surfaced DefaultRowHeight property, and implemented accordingly         *
*    Delayed enhancements/fixes: Edition of Memo and Graphic, fix to RowSizing *
*   Version 2.0  (not freeware)                                                *
*    - Any control in any column, including populated panels                   *
********************************************************************************
*  Based upon works and ideas from everywhere on the 'net, books and magazines *
********************************************************************************
*  It's freeware, and you know all the legal things about it...                *
********************************************************************************
*  Please, if you enhance the component, send me a copy,and I'll try to include*
*  it in future versions (if any). By the way, it's better NOT to modify       *
*  the component, but inherit it and modify the inherited, so you don't need to*
*  modify anything with future enhanced versions (if any).                     *
********************************************************************************
*  I'm developed it for contributing to the Delphi Freeware Community, as I've *
*    obtained good components and ideas from others, and it's time to return   *
*******************************************************************************)


unit DBGridPlus;

interface

uses
  Windows, Messages, SysUtils, Classes, TypInfo, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, Grids, DBGrids, DB, DBTables;

type
  TdbeNativeClick = Set Of (ncssShift, ncssCtrl, ncssAlt);
  { new event type for column header click with mouse state & column index }
  TColumnClickEvent = procedure(Sender: TObject; Index: Integer;
                               Button: TMouseButton; ShiftState: TShiftState;X,Y:Integer) of object;
  TColumnClickSimpleEvent = procedure(Sender: TObject; Index: Integer;
                               ShiftState: TShiftState;X,Y:Integer) of object;
  TChangeDrawingEvent = procedure(Sender: TObject;DataCol,DataRow:Integer;
                           AState: TGridDrawState;var Highlighted:Boolean)
                           of object;
  TgedmMemo = (gedmNo,gedmYes);
  TgedmGraphic = (gedgNo,gedgStretch,gedgAdjust);

  { new DBGrid class }
  TDBGridPlus = class(TDBGrid)
  private
    FColumnClick: integer;
    FColumnClickControl: Boolean;
    FButton: TMouseButton;
    FShiftState: TShiftState;
    FOnColumnClick: TColumnClickEvent;
    FOnColumnLeftClick: TColumnClickSimpleEvent;
    FOnColumnRightClick: TColumnClickSimpleEvent;
    FOnColumnMiddleClick: TColumnClickSimpleEvent;
    FOnChangeDrawing : TChangeDrawingEvent;
    FNativeClick : TdbeNativeClick;
    FDrawMemo : TgedmMemo;
    FDrawGraphic : TgedmGraphic;
    FNewDefaultRowHeight : Integer;
    FRowSizingAllowed : Boolean;
    procedure SetRowSizingAllowed(Value:Boolean);
  protected
    procedure MouseDown(Button: TMouseButton; ShiftState: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(ShiftState: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; ShiftState: TShiftState;
      X, Y: Integer); override;
    function  HighlightCell(DataCol, DataRow: Integer; const Value: string;
      AState: TGridDrawState): Boolean; override;
    procedure DrawColumnCell(const Rect: TRect; DataCol: Integer;
      Column: TColumn; State: TGridDrawState); override;
    function GetDefaultRowHeight: Integer;
    procedure SetDefaultRowHeight(Value: Integer);
    procedure LayoutChanged; override;
    procedure RowHeightsChanged; override;
    procedure DoOnColumnClick(Sender: TObject;Index:Integer;Button: TMouseButton;
                   ShiftState:TShiftState;X,Y:Integer);         dynamic;
    procedure DoOnColumnLeftClick(Sender: TObject;Index:Integer;
                   ShiftState:TShiftState;X,Y:Integer);         dynamic;
    procedure DoOnColumnRightClick(Sender: TObject;Index:Integer;
                   ShiftState:TShiftState;X,Y:Integer);         dynamic;
    procedure DoOnColumnMiddleClick(Sender: TObject;Index:Integer;
                   ShiftState:TShiftState;X,Y:Integer);         dynamic;
    function GetCurrentColumn:Integer;
  public
    constructor Create(AOwner: TComponent); override;
    property Canvas;
    property TopRow;
    property Row;
    property SelectedRows;
    procedure MouseToCell(X,Y:Integer; var ACol,ARow:Integer); dynamic;
  published
    property GridLineWidth;
    property OnMouseDown;
    property OnMouseUp;
    property OnMouseMove;
    property NativeClick:TdbeNativeClick Read FNativeClick Write FNativeClick Default [];
    property DrawMemo:TgedmMemo read FDrawMemo write FDrawMemo default gedmYes;
    property DrawGraphic:TgedmGraphic read FDrawGraphic write FDrawGraphic default gedgAdjust;
    property DefaultRowHeight:Integer read GetDefaultRowHeight write SetDefaultRowHeight;
    property OnChangeDrawing:TChangeDrawingEvent read FOnChangeDrawing write FOnChangeDrawing;
    property OnColumnClick:TColumnClickEvent read FOnColumnClick write FOnColumnClick;
    property OnColumnLeftClick:TColumnClickSimpleEvent read FOnColumnLeftClick write FOnColumnLeftClick;
    property OnColumnRightClick:TColumnClickSimpleEvent read FOnColumnRightClick write FOnColumnRightClick;
    property OnColumnMiddleClick:TColumnClickSimpleEvent read FOnColumnMiddleClick write FOnColumnRightClick;
    property RowSizingAllowed:Boolean read FRowSizingAllowed write SetRowSizingAllowed default False;
//    property OnEditCellEvent:TOnEditCellEvent read FOnEditCellEvent write FOnEditCellEvent;
    property CurrentColumn:Integer read GetCurrentColumn;
  end;



procedure Register;


implementation



constructor TDBGridPlus.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csAcceptsControls,csDisplayDragImage]; //////////////////
  FColumnClickControl:=False;
  FDrawMemo:=gedmYes;
  FDrawGraphic:=gedgAdjust;
  FNativeClick:=[];
  FRowSizingAllowed:=False;
end;

Type
  TCustomGridHack = class(TCustomGrid)
                    Public
                      Property Options;
                    end;

procedure TDBGridPlus.SetRowSizingAllowed(Value:Boolean);
begin
  If Value<>FRowSizingAllowed Then Begin
    FRowSizingAllowed:=Value;
    If FRowSizingAllowed Then
      TCustomGridHack(Self).Options:=TCustomGridHack(Self).Options+[goRowSizing]
    Else
      TCustomGridHack(Self).Options:=TCustomGridHack(Self).Options-[goRowSizing];
  End
end;


procedure TDBGridPlus.MouseToCell(X,Y:Integer; var ACol,ARow:Integer);
var
  Coord:TGridCoord;
begin
  Coord:=MouseCoord(X, Y);
  ACol:=Coord.X;
  ARow:=Coord.Y;
end;


procedure TDBGridPlus.MouseDown(Button: TMouseButton;ShiftState:TShiftState; X,Y:Integer);
var
   ACol,
   ARow: Integer;
   TmpShift : TdbeNativeClick;
begin
   MouseToCell(X, Y, ACol, ARow);
   if (ARow=0) and (dgTitles in Options) then begin
     TmpShift:=[];
     If ssShift In ShiftState Then
       TmpShift:=TmpShift+[ncssShift];
     If ssCtrl In ShiftState Then
       TmpShift:=TmpShift+[ncssCtrl];
     If ssAlt In ShiftState Then
       TmpShift:=TmpShift+[ncssAlt];
     If FNativeClick=TmpShift Then
       inherited MouseDown(Button,ShiftState,X,Y)
     Else Begin
       FColumnClickControl:=True;
       FColumnClick:=ACol;
       FButton:=Button;
       FShiftState:=ShiftState;
     end
    End
   Else begin
     FColumnClickControl:=False;
     Inherited;
   end
end;

procedure TDBGridPlus.MouseMove(ShiftState: TShiftState; X, Y: Integer);
var
  ACol,
  ARow: Integer;
begin
  inherited;
  if FColumnClickControl And (FColumnClick>=0) then begin
    MouseToCell(X, Y, ACol, ARow);
    if (ACol<>FColumnClick) or (ARow<>0) then
      FColumnClickControl:=False;
  end;
end;

procedure TDBGridPlus.MouseUp(Button: TMouseButton; ShiftState:TShiftState; X,Y:Integer);
var
  ACol,
  ARow: Integer;
begin
  inherited;
  MouseToCell(X, Y, ACol, ARow);
  if FColumnClickControl And (FColumnClick=ACol) then begin
    if dgIndicator in Options then
      Dec(FColumnClick);
    if (FColumnClick>=-1) then begin
      DoOnColumnClick(self, FColumnClick, Button, FShiftState, X,Y);
      If (FButton=mbLeft) then
        DoOnColumnLeftClick(self, FColumnClick, FShiftState, X,Y);
      If (FButton=mbRight) then
        DoOnColumnRightClick(self, FColumnClick, FShiftState, X,Y);
      If (FButton=mbMiddle) then
        DoOnColumnMiddleClick(self, FColumnClick, FShiftState, X,Y);
    end;
    FColumnClickControl:=False;
  end;
end;

procedure TDBGridPlus.DoOnColumnClick(Sender: TObject;Index:Integer;Button: TMouseButton;ShiftState:TShiftState;X,Y:Integer);
begin
  If Assigned(FOnColumnClick) then
    FOnColumnClick(self, FColumnClick, Button, FShiftState, X,Y);
end;

procedure TDBGridPlus.DoOnColumnLeftClick(Sender: TObject;Index:Integer;ShiftState:TShiftState;X,Y:Integer);
begin
  If Assigned(FOnColumnLeftClick) then
    FOnColumnLeftClick(self, FColumnClick, FShiftState, X,Y);
end;

procedure TDBGridPlus.DoOnColumnRightClick(Sender: TObject;Index:Integer;ShiftState:TShiftState;X,Y:Integer);
begin
  If Assigned(FOnColumnRightClick) then
    FOnColumnRightClick(self, FColumnClick, FShiftState, X,Y);
end;

procedure TDBGridPlus.DoOnColumnMiddleClick(Sender: TObject;Index:Integer;ShiftState:TShiftState;X,Y:Integer);
begin
  If Assigned(FOnColumnMiddleClick) then
    FOnColumnMiddleClick(self, FColumnClick, FShiftState, X,Y);
end;


function TDBGridPlus.GetDefaultRowHeight:Integer;
begin
  Result:=Inherited DefaultRowHeight;
end;

procedure TDBGridPlus.SetDefaultRowHeight(Value: Integer);
begin
  If Value=0 Then
    Value:=inherited DefaultRowHeight;
  inherited DefaultRowHeight:=Value;
  FNewDefaultRowHeight:=Value;
  if dgTitles in Options then begin
    Canvas.Font:=TitleFont;
    RowHeights[0]:=Canvas.TextHeight('Wg')+4;
  end;
end;

procedure TDBGridPlus.LayoutChanged;
begin
  Inherited;
  SetDefaultRowHeight(FNewDefaultRowHeight);
end;

procedure TDBGridPlus.RowHeightsChanged;
var
  i,ThisHasChanged,Def : Integer;
begin
  ThisHasChanged:=-1;
  Def:=DefaultRowHeight;
  For i:=Ord(dgTitles In Options) to RowCount Do
    If RowHeights[i]<>Def Then Begin
      ThisHasChanged:=i;
      Break;
    End;
  If ThisHasChanged<>-1 Then Begin
    SetDefaultRowHeight(RowHeights[ThisHasChanged]);
    RecreateWnd;
  End;
  inherited;
end;


function TDBGridPlus.HighlightCell(DataCol, DataRow: Integer; const Value: string;
      AState: TGridDrawState): Boolean;
var
  Highlighted: Boolean;
begin
  Highlighted:=inherited HighlightCell(DataCol, DataRow, Value, AState);
  If Assigned(FOnChangeDrawing) Then
    FOnChangeDrawing(Self,DataCol,DataRow,AState,Highlighted);
  Result:=Highlighted;
end;



procedure TDBGridPlus.DrawColumnCell(const Rect: TRect; DataCol: Integer;
  Column: TColumn; State: TGridDrawState);
var
  Bmp : TBitmap;
  R : TRect;
  FactorX,FactorY : Double;
  TmpWidth,TmpHeight : Integer;
  CWidth, CHeight : Integer;
  BWidth, BHeight : Integer;
begin
  If Assigned(DataSource) And Assigned(DataSource.DataSet) And DataSource.DataSet.Active Then Begin
    If Assigned(Column.Field) Then Begin
      If (Column.Field.DataType=ftMemo) Then Begin
        If FDrawMemo<>gedmNo Then Begin
          R:=Rect;
          Canvas.FillRect(R);
          Inc(R.Top,2);
          Inc(R.Left,2);
          DrawText(Canvas.Handle, PChar(Column.Field.AsString),
                     Length(Column.Field.AsString), R,
                     DT_WORDBREAK OR DT_NOPREFIX);
        End
       End
      Else If (Column.Field.DataType=ftGraphic) Then Begin
        Bmp:=TBitmap.Create;
        try
          Bmp.Assign(Column.Field);
          If FDrawGraphic=gedgAdjust Then Begin
            CWidth:=Rect.Right-Rect.Left+1-1;
            CHeight:=Rect.Bottom-Rect.Top+1-1;
            BWidth:=Bmp.Width;
            BHeight:=Bmp.Height;
            Try
              If BWidth=0 Then
                FactorX:=0
              Else
                FactorX:=CWidth/BWidth;
              If BHeight=0 Then
                FactorY:=0
              Else
                FactorY:=CHeight/BHeight;
            Except
              FactorX:=1;
              FactorY:=1;
            End;
            If FactorY<FactorX Then Begin
              TmpWidth:=Round(BWidth*FactorY);
              TmpHeight:=CHeight;
             End
            Else Begin
              TmpHeight:=Round(BHeight*FactorX);
              TmpWidth:=CWidth;
            End;
            SetRect(R, 0, 0, TmpWidth, TmpHeight);
            OffsetRect(R, Rect.Left+((CWidth - TmpWidth) div 2), Rect.Top+((CHeight - TmpHeight) div 2));
            Canvas.FillRect(Rect);
            Canvas.StretchDraw(R, Bmp);
           End
          Else Begin
            Canvas.StretchDraw(Rect,Bmp)
          End
        finally
          Bmp.Free;
        end
      End
    End;
  end;
  inherited;
end;


function TDBGridPlus.GetCurrentColumn:Integer;
begin
  GetCurrentColumn:=RawToDataColumn(Col)
end;

procedure Register;
begin
  RegisterComponents('DB Plus', [TDBGridPlus]);
end;

end.
