unit DBcvt;
(*
	Unit to import an ascii delmited file in to a (paradox) table
	On a 486/33 this parses about 150 lines per second (with 10 fields per line.)
	It's not fast but it works real good.

	Author: William R. Florac
  Company: FITCO, Verona, WI (wee little company from my house)
	Copyright 1996, FITCO.  All rights reserved.

 1)  Users of DBCVT (and it's components) must accept this disclaimer of
     warranty: "DBCVT is supplied as is.  The author disclaims all
     warranties, expressed or implied, including, without limitation,
     the warranties of merchantability and of fitness for any purpose.
     The author assumes no liability for damages, direct or conse-
     quential, which may result from the use of DBCVT."

  2) This Software is donated to the public as public domain except as
     noted below.

  3) If you distribute this software, you must include all parts and pages without
  	 modification.

  4) Software may be used, modified and distributed freely if compiled in with
     commercial or private applications (not another VCL).

  5) Fitco retains the copyright to this Software.  You may not distribute
     the source code (PAS) or its compiled unit (DCU) for profit.

  6) If you do find this component handy and you feel guilty
    for using such a great product without paying someone,
    please feel free to send a few bucks ($25) to support further
    development. .  I have spent a lot of time making this VCL
		the best it can be and have included a help file to make
		it complete.

	7) This file was formatted with tabs set to 2.

	8) Thanks to all those who suggested (and got) improvements.

  9) Latest version can always be found at http://sumac.etcconnect.com/~fitco/

	Please forward any comments or suggestions to Bill Florac at:
	 	email: flash@etcconnect.com
		mail: FITCO
					209 Jenna Dr
					Verona, WI  53593


Suppored:
		paradox				Tfield				example							notes
		----------    -----------   ------------        ------------
		Alpha					ftString      "<text>"						255 char max
		Number				ftFloat				"100.00"						() and , allowed
		Money					ftCurrency    "12.75"							(), $ and , are allowed
		Short					ftSmallInt		"100"       				-32,768 to 32,767,
		LongInt				ftInteger     "100000: 						-2,147,483,648 to 2,147,483,647
		BCD						ftBCD         "100.00"						18 digits
		Data					ftDate        "11/15/95"
		Time					ftTime        "2:30pm" or "14:00"
		TimeStamp			ftDateTime    "1/1/96 6:00am"
		Memo					ftMemo        "<text>"
		Formated Memo ftBlob   			No support
		Graphic				ftGraphic     No support
		OLE						ftBlob      	No support
		Logical				ftBoolean			"F" or "False"
		AutoIncrement ftInteger   	No support
		Binary				ftBlob      	No support
		Bytes					ftBytes     	No support

	For all above decimals can be represented as 10E10 and -10E-10
  Integers can be represented in hex $XXXX.
  Note: Delphi 2 does NOT support negive hex numbers (-$XXXX) as 1.0 did
*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DBtables, DB, StdCtrls, DbiErrs;

const
	DBcvtVersion = '2.0b';

type
	{escape enumberations}
	TDBcvtEscape =  (esNL, esCR, esHT, esBslash, esSquote, esDquote, esQues, esNull);
	TDBcvtEscapes = set of TDBcvtEscape;
  TDBcvtMode = (mCopy, mAppend, mUpdate, mAppendUpdate);

 	TDBcvtDateFormat = (dfWindows, dfMDY, dfDMY, dfYMD, dfBCD);

	TDBcvtError = (eOK,
		eNoExclusiveAccess,
		eAsciiFileNotFound,
		eNoFieldsInList,
		eDataTypeMismatch,
		eFieldNotInTable,
		eFieldTooLong,
		eNoAsciiFileAccess,
		eTooManyFieldsInList,
		eNoIndexFieldFound,
		eIndexFieldNotInList,
		eMissingFieldWidth,
		eAppendOrEditError,
		ePostError,
		eUserAborted,
		eTableMissing,
		eBadDatabaseName,
		eAsciiNameMissing,
		eCantOpenErrorFile,
		eErrorFileWriteError,
		eNoRecordsFound,
		eFieldCount,
		eConversion,
		eKeyViolation,
		eMemoTooLong,
		eMultiError,
    eOtherDatabaseError,
		eCannotOpenTable,
    eExportFileWriteError
		);

	{A memory stream to contain the field in any length}
	TFieldStream = class(TMemoryStream)
  public
		FieldLength: LongInt;	{length of field, for fixed fields}
		FieldString: string; 	{string version of parsed text}
    Field: TField;				{field in table for this item in list}
		constructor Create;
	end;


{my parse event}
	TParseEvent = procedure(Sender: TObject; var ParsedList: TList;
								aRecord: string; var NumberOfFields: LongInt) of object;

	TPrePostEvent = procedure(Sender: TObject; var ATable: TTable; var OkToPost: boolean) of object;

  TAscii2DB = class(TComponent)
  private
    { Private declarations }
		FAnnotateErrorFile: boolean;
		FAllowComments: boolean;
		FDelimiter: char;
		FSeparator: char;
		FEndOfRecord: char;
		FEscapes: TDBcvtEscapes;
    FShowDlg: boolean;
		FAbortCaption: string;
		FShowCaption: string;
		FNotifyCount: LongInt;
		FRecordLimit: longInt;
    FDestination: TTable;
    FAsciiFile: TFileName;
    FErrorFile: TFileName;
		FFieldList: TStrings;
		FMaxRecordErrors: longInt;
		FOnNotifyCount: TNotifyEvent;
		FOnParse: TParseEvent;
		FOnPrePost: TPrePostEvent;
		FWriteEnable: boolean;
		FNamesFirst: boolean;
		FMode: TDBcvtMode;
		FFixedWidth: boolean;
    FDateFormat: TDBcvtDateFormat;
    FDateSeparat: char;
    FTimeSeparat: char;
    FDecimalSeparat: char;
    FMaxFieldSize: LongInt;
    sRecord: string;
    sField: string;
    sFieldCount: LongInt;
		sQuoteEnd: LongInt;
    OldActive: boolean;

		procedure SetFieldList(value: TStrings);
		procedure SetRecordLimit(value: LongInt);
		procedure SetMaxRecordErrors(value: longInt);
		procedure SetNamesFirst(value: boolean);
		procedure SetFixedWidth(value: boolean);
		procedure SetEndOfRecord(value: char);
		procedure SetAsciiFile(value: TFileName);
		procedure SetErrorFile(value: TFileName);
    procedure SetMaxFieldSize(value: LongInt);
		procedure AbortTransfer(Sender: TObject);

  protected
    { Protected declarations }
		function GetFixedRecord(var List: TList; var f: File): LongInt;
		function GetRecord(var List: TList; var f: File): LongInt;
  public
    { Public declarations }
		AbortedCount: LongInt; 			{record aborted by user}
		AppendedCount: longInt;			{record appended}
		Busy: boolean;							{busy}
		ConversionErrors: longInt;  {records with conversion errors}
		EmptyCount:  longInt;				{empty strings}
		ErrorCode: TDBcvtError;			{cause of failure }
		ErrorLine: longInt;					{last line read from file}
		ErrorField: longInt;						{last field read from file}
		FieldCountErrors: longInt;	{records with miscount of fields}
		KeyViolationErrors: LongInt;{records with key violations}
		LineCount: longInt;					{lines in ascii file}
		MemoErrors: longInt;		    {records with memo errors}
		ParsedCount: longInt;				{lined read from file, except empty lines}
		RecordErrorCount: longInt;	{count of records with errors}
		UpdatedCount: longInt;			{records updated}

		constructor Create(Aowner: Tcomponent); override;
		destructor Destroy; override;
		function GetRecordCount:LongInt;
		procedure Execute;
		procedure StopExecute;
  published
    { Published declarations }
		property AbortCaption: string read FAbortCaption write FAbortCaption;
		property AllowComments: boolean read FAllowComments write FAllowComments default true;
		property AnnotateErrorFile: boolean  read FAnnotateErrorFile write FAnnotateErrorFile default false;
		property AsciiFile: TFileName read FAsciiFile write SetAsciiFile;
    property DateFormat: TDBcvtDateFormat read FDateFormat write FDateFormat default dfWindows;
 		property DateSeparat: char read FDateSeparat write FDateSeparat default '/';
 		property TimeSeparat: char read FTimeSeparat write FTimeSeparat default ':';
 		property DecimalSeparat: char read FDecimalSeparat write FDecimalSeparat default '.';
		property Delimiter: char read Fdelimiter write Fdelimiter default '"';
    property Destination: TTable read FDestination write FDestination;
		property EndOfRecord: char read FEndOfRecord write SetEndOfRecord default #10;
    property ErrorFile: TFileName read FErrorFile write SetErrorFile;
    property Escapes: TDBcvtEscapes read FEscapes write FEscapes default [];
		property FieldList: TStrings read FFieldList write SetFieldList;
		property FixedWidth: boolean read FFixedWidth write SetFixedWidth default false;
		property MaxRecordErrors: LongInt read FMaxRecordErrors write SetMaxRecordErrors default 1;
		property MaxFieldSize: LongInt read FMaxFieldSize write SetMaxFieldSize default 255;
    property Mode: TDBcvtMode read FMode write FMode default mCopy;
		property NamesFirst: boolean  read FNamesFirst write SetNamesFirst default false;
		property NotifyCount: LongInt read FNotifyCount write FNotifyCount default 100;
		property RecordLimit: LongInt read FRecordLimit write SetRecordLimit default 0;
		property Separator: char read Fseparator write Fseparator default ',';
		property ShowCaption: string read FShowCaption write FShowCaption;
		property ShowDlg: boolean read FshowDlg write FShowDlg default false;
		property WriteEnable: boolean read FWriteEnable write FWriteEnable default true;

		property OnNotifyCount: TNotifyEvent read FOnNotifyCount write FOnNotifyCount;
		property OnParse: TParseEvent read FOnParse write FOnParse;
		property OnPrePost: TPrePostEvent read FOnPrePost write FOnPrePost;
	end;

{******************************************************************************}
TDB2Ascii = class(TComponent)
  private
    { Private declarations }
		FAbortCaption: string;
    FAsciiFile: TFileName;
		FDelimiter: char;
		FEndOfRecord: char;
		FNotifyCount: LongInt;
		FOnNotifyCount: TNotifyEvent;
		FSeparator: char;
		FShowCaption: string;
    FShowDlg: boolean;
    FSource: TTable;
    FAlwaysQuote: boolean;
		procedure AbortTransfer(Sender: TObject);
		procedure SetAsciiFile(value: TFileName);
		procedure SetEndOfRecord(value: char);
  protected
    { Protected declarations }
  public
    { Public declarations }
		Busy: boolean;							{busy}
		ErrorCode: TDBcvtError;			{cause of failure }
		WrittenCount: LongInt; 			{records written to ascii file}
		constructor Create(Aowner: Tcomponent); override;
		procedure Execute;
		procedure StopExecute;
  published
    { Published declarations }
		property AbortCaption: string read FAbortCaption write FAbortCaption;
    property AlwaysQuote: boolean read FAlwaysQuote write FAlwaysQuote default true;
		property AsciiFile: TFileName read FAsciiFile write SetAsciiFile;
		property Delimiter: char read Fdelimiter write Fdelimiter default '"';
		property EndOfRecord: char read FEndOfRecord write SetEndOfRecord default #10;
		property NotifyCount: LongInt read FNotifyCount write FNotifyCount default 100;
		property Separator: char read Fseparator write Fseparator default ',';
		property ShowDlg: boolean read FshowDlg write FShowDlg default false;
		property ShowCaption: string read FShowCaption write FShowCaption;
    property Source: TTable read FSource write FSource;
  	property OnNotifyCount: TNotifyEvent read FOnNotifyCount write FOnNotifyCount;

	end;

{global calls}
function GetErrorString(Ecode: TDBcvtError): string;
function GetDatabasePath(aDataSet: TDBDataSet): string;


procedure Register;

implementation
{$B-}

const
  DefFieldLength = 255;
	DefBlockLength = 255;

{******************************************************************************}
constructor TAscii2DB.Create(Aowner: Tcomponent);
begin
	inherited create(Aowner);
	{set default public properties}
	Busy := false;

	ErrorCode := eOk;
	ErrorLine := 0;
	ErrorField := 0;

	LineCount := 0;
		AbortedCount := 0;
 		EmptyCount := 0;
		ParsedCount := 0;
			AppendedCount := 0;
			UpdatedCount := 0;

			FieldCountErrors := 0;
			KeyViolationErrors := 0;
			ConversionErrors := 0;
			MemoErrors := 0;

	{set default published properties}
	FAsciiFile := '';
	Fdelimiter := '"';
	Fseparator := ',';
	FEndOfRecord := #10;
	FEscapes := [];
  FShowDlg := False;
	FShowCaption := 'Transfer Progress';
	FAbortCaption := '&Cancel';
	FNotifyCount := 100;
	FRecordLimit := 0;
	FNamesFirst := false;
	FMode := mCopy;
	FMaxRecordErrors := 1;
  FFixedWidth := false;
	FWriteEnable := true;
  FAnnotateErrorFile := false;
  FDateFormat := dfWindows;
  FDateSeparat := '/';
  FTimeSeparat := ':';
  FDecimalSeparat := '.';
	FAllowComments := true;
  FMaxFieldSize := 255;

	{create some things needed}
	FFieldList := TStringList.create;
  FFieldList.Clear;

end; {of create}


{******************************************************************************}
destructor TAscii2DB.Destroy;
begin
	FFieldList.Free;
	inherited Destroy;
end; {of destroy}

{******************************************************************************}
procedure Tascii2DB.Execute;
type
	TEditMode = (emNone,emEdit,emAppend);
var
	{General temp data}
	AField: TField;
	AFieldStream: TFieldStream;

	InFile: File;								{file we are reading}
	Efile: File;								{file to put error}
	DBIndex: LongInt;  					{an index}
	NumberOfFields: LongInt;		{fields parsed}

	ParsedList: TList;					{a place to put parsed strings/streams}
	ProgressForm: TForm;				{progress dialog}
	ProgressCancel: TButton;		{cancel button in progress dialog}
	MyTable: TTable;						{copy of table}

	ProgressTick: LongInt;			{a bytes between dlg updates}
	NextTick: LongInt;					{a counter of bytes parsed}

	FieldName: string;					{temp field name holder}
  FieldWidth: word;						{width of field name (before comma}
	EndFieldName: byte;					{Location of , in field name}
	OkToPost: Boolean;					{flag to indicate user sait ok to Post}
	EditMode: TEditMode;				{Append/Edit/None}

	SaveToErrorFile: boolean;		{flag to indicate we should same error to a file}
//	aResult: word;							{temp result from a function}
	CharWritten: LongInt;				{number of bytes written to a file}
	LastLineCount: longInt;			{previous line during an error}
	X: LongInt;									{a counter}
	s: string;

  OldDateFormat: string;  {for date conversion}
  OldDateSeparator: char;
  OldTimeSeparator: char;
  OldDecimalSeparator: char;


{local procedure to execute}
procedure UpdateDlg;
var
	GPercent: word;					{graphical progress in percent}
	TPercent: word;					{text progress in percent}
	PercentText: String;    {percent text}
  NewRect: TRect;					{a rectangle}
begin
	if FileSize(Infile) > 0 then begin
		GPercent := ProgressForm.ClientWidth * (FilePos(Infile) +1) div FileSize(Infile);
		TPercent := 100 * (FilePos(Infile) +1) div FileSize(Infile);
	end
  else begin
		GPercent := 0;
		TPercent := 0;
	end;

	{draw left background}
	if FNotifyCount > 0
	then NewRect := Rect(0, 0, GPercent, ProgressForm.ClientHeight div 2)
	else NewRect := Rect(0, 0, GPercent, ProgressForm.ClientHeight);
	ProgressForm.Canvas.Brush.Style := bsSolid;
	ProgressForm.Canvas.Brush.Color := clRed;
	ProgressForm.Canvas.FillRect(NewRect);

	{draw right background}
	if FNotifyCount > 0
	then NewRect := Rect(GPercent, 0, ProgressForm.ClientWidth,
              ProgressForm.ClientHeight div 2)
  else NewRect := Rect(GPercent, 0, ProgressForm.ClientWidth,
              ProgressForm.ClientHeight);

	ProgressForm.Canvas.Brush.Color := clBtnFace;
	ProgressForm.Canvas.FillRect(NewRect);

	{draw text}
	PercentText := IntToStr(TPercent) + '%';
	ProgressForm.Canvas.Brush.Style := bsClear;
  ProgressForm.Canvas.TextOut(90,2,PercentText);
end; {of UpdateDlg}


{******************************************************************************}
{Should we abort on Error  enter with error code set}
function AbortOnError(FieldIndex: LongInt): boolean;
var
	s : string;
begin
	{assume true}
	Result := true;

	{save first error}
	if ErrorLine = 0 then begin
		ErrorLine := LineCount;     	{this is the line we are on so it starts at 1}
		ErrorField := FieldIndex + 1; {this is zero base so add one}
	end;

	{save error to file (if enabled) but only one per record}
	if LineCount <> LastLineCount then begin
		case errorCode of
    eFieldCount: inc(FieldCountErrors);
		eMemoTooLong: inc(MemoErrors);
		eConversion: inc(ConversionErrors);
		eKeyViolation: inc(KeyViolationErrors);
		end;

		if SaveToErrorFile then begin
			if FAnnotateErrorFile then begin
				s := ';' + 'line:' + IntToStr(LineCount) + ' field:'+ IntToStr(FieldIndex)
								 + ' message: ' + GetErrorString(errorCode) + #13#10;
				BlockWrite(efile, s[1],Length(s),CharWritten);
				if CharWritten <> Length(s) then begin
					errorCode := eErrorFileWriteError;
					exit;
				end;
			end;

			BlockWrite(eFile, sRecord[1], length(sRecord), CharWritten);
			if CharWritten <> Length(sRecord) then begin
				errorCode := eErrorFileWriteError;
				exit;
			end;
  	end;
	end;
	LastLineCount := LineCount;

	{test to see if we should abort}
  if not ((FMaxRecordErrors > 0) and (RecordErrorCount >= FMaxRecordErrors)) then begin
		inc(RecordErrorCount);
		errorCode := eOK;
		result := false;
	end;
end;

{******************************************************************************}
{remove all but numbers and - from string.  () is converted to negitive}
{E is allow for expo numbers too}
function PrepNumber(cs: string): string;
var
	x: LongInt;
  neg: boolean;
begin
{	result := cs;
  exit;}
	neg := false;
  result := '';
	for x := 1 to length(cs) do begin
  	if cs[x] in ['-','0'..'9', 'e', 'E', DecimalSeparator] then result := result + cs[x];
    if cs[x] = '(' then neg := true;
  end;
  {make it negitive it is not already}
  if neg and (cs[1] <> '-') then result := '-' + result;
end;

{clean up LongInt strings}
{allow for hex notation as $xxxx}
function PrepInt(cs: string): string;
var
	x: LongInt;
  neg: boolean;
begin
	neg := false;
  result := '';
	for x := 1 to length(cs) do begin
  	if cs[x] in ['-','0'..'9','A'..'F','a'..'f','$'] then result := result + cs[x];
    if cs[x] = '(' then neg := true;
  end;
  {make it negitive it is not already}
  if neg and (cs[1] <> '-') then result := '-' + result;
end;

{******************************************************************************}
{convert CCYYMMDD or 19961101 to date}
function BCDToDate(ds: string): TDateTime;
var
	Day,Month,Year: word;
begin
	Year := StrToIntDef(Copy(ds,1,4),0);
	Month := StrToIntDef(Copy(ds,5,2),0);
	Day := StrToIntDef(Copy(ds,7,2),0);
  if (Year < 1) or (Month < 1) or (Day < 1)
  then raise EConvertError.Create('Invalid BCDdate format.')
  else Result := EncodeDate(Year,Month, Day);
end;

{******************************************************************************}
{find if key exist}
function KeyExist: boolean;
var
	MyIndex: LongInt;
begin
	MyTable.SetKey;
	for MyIndex := 0 to ParsedList.Count - 1 do begin
	 	AFieldStream := TFieldStream(ParsedList.Items[MyIndex]);
		{skip fields will be nil}
    if (AFieldStream.Field <> nil) and (AFieldStream.Field.IsIndexField) then begin
			case AFieldStream.Field.DataType of
	      {indexes can not be on memo fields}
				ftString: AFieldStream.Field.AsString := AFieldStream.FieldString;
				ftCurrency,ftFloat, ftBCD:
        	AFieldStream.Field.AsString := PrepNumber(AFieldStream.FieldString);
        ftSmallint, ftInteger, ftWord:
        	AFieldStream.Field.AsString := PrepInt(AFieldStream.FieldString);
        ftDate: begin
 	    	  if DateFormat = dfBCD
   	    	then TDateField(AFieldStream.Field).value := BCDToDate(Trim(AFieldStream.FieldString))
    	    else AFieldStream.Field.AsString := Trim(AFieldStream.FieldString)
  	    end;
	      else AFieldStream.Field.AsString := Trim(AFieldStream.FieldString);
 	 		end;
    end; {of if Field <> nil}
  end;  {of for}
  if MyTable.GotoKey
  then result := true
  else result := false;
end; {of KeyExist}

{******************************************************************************}
begin {of Execute}

 	{abort if we are busy}
	if busy then exit;
  Busy := true;

	{clear globals}
	ErrorCode := eOk;
	ErrorLine := 0;
	ErrorField := 0;

	LineCount := 0;
		AbortedCount := 0;
 		EmptyCount := 0;
		ParsedCount := 0;
			AppendedCount := 0;
			UpdatedCount := 0;

			FieldCountErrors := 0;
			KeyViolationErrors := 0;
			ConversionErrors := 0;
			MemoErrors := 0;
	LastLineCount := 0;

	{init locals}
  ParsedList := nil;
	ProgressForm := nil;
	ProgressCancel := nil;
	MyTable := nil;
  ProgressTick := 0;
	NextTick := 0;
	SaveToErrorFile := false;

  {set up DateFormat}
  OldDateFormat := ShortDateFormat;
  OldDateSeparator := DateSeparator;
  OldDecimalSeparator := DecimalSeparator;
  OldTimeSeparator := TimeSeparator;

  DateSeparator := FDateSeparat;
  TimeSeparator := FTimeSeparat;

  DecimalSeparator := FDecimalSeparat;

  case FDateFormat of
	dfDMY: ShortDateFormat:= 'dd'+DateSeparator+'mm'+DateSeparator+'yyyy';
	dfMDY: ShortDateFormat:= 'mm'+DateSeparator+'dd'+DateSeparator+'yyyy';
	dfYMD: ShortDateFormat:= 'yyyy'+DateSeparator+'mm'+DateSeparator+'dd';
  end;


	{If enabled, make sure table is there}
	if FWriteEnable then begin
		if (FDestination = nil)
		or (FDestination.TableName = '')
		then begin
			ErrorCode := eTableMissing;
			busy := false;
			exit;
		end;

		if FDestination.DataBaseName = '' then begin
			ErrorCode := eBadDatabaseName;
			busy := false;
			exit;
		end;
	end; {of if WriteEnable}

	{does source file exist?}
	if FAsciiFile = '' then begin
		errorCode := eAsciiNameMissing;
		busy := false;
		exit;
	end;

	s := FAsciiFile + #0;
	AnsiToOEM(@s[1],@s[1]);
	SetLength(s, Length(s) -1);
	if not FileExists(s) then begin
		errorCode := eAsciiFileNotFound;
		busy := false;
		exit;
	end;

	{open ascii file}
	try
		AssignFile(InFile, s);
		Reset(InFile,1);
	except
		on EInOutError do begin
			errorCode := eNoAsciiFileAccess;
			busy := false;
			exit;
		end;
	end;

	{does error file exist?}
	if FErrorFile <> '' then begin
		s := FErrorFile + #0;
		AnsiToOEM(@s[1],@s[1]);
		SetLength(s, length(s) -1);
		{try to open it}
		try
			AssignFile(eFile, s);
			Rewrite(eFile,1);
		except
			on EInOutError do begin
				CloseFile(InFile);
				ErrorCode := eCantOpenErrorFile;
				busy := false;
				exit;
			end;
		end;
		SaveToErrorFile := True;
	end;

(* CODE IF WE WANT TO VERIFY EXISTANCE OF ERROR FILE
	if FErrorFile <> '' then begin
		aResult := mrYes;
		if FileExists(FErrorFile) then
			Aresult := messagedlg(FErrorFile + #13#10 +
														'This file allready exist.' + #13#10 +
														'Do you want to replace it?',
														mtConfirmation, [mbYes,mbNo],0);
		if aResult = mrYes then begin
     	{open a file here!}
			try
				AssignFile(eFile, FErrorFile);
				Rewrite(eFile,1);
				except
					on EInOutError do begin
					errorCode := eCantOpenErrorFile;
					exit;
				end;
			end;
			SaveToErrorFile := True;
		end
		else begin
			ErrorCode := eUserAborted;
			busy := false;
			exit; {from from procedure}
    end
	end;
*)

	{try/finally to force us to dispose of string pRecord and pField and close files}
	try
  	{make objects we need}
		ParsedList := TList.create; 		{create the string list}
		ParsedList.Clear;

		{remove any blanks from list}
		if FFieldList.Count > 0 then begin
			for DBIndex := FFieldList.Count - 1 downto 0 do begin
				if Length(Trim(FFieldList[DBIndex])) = 0
				then FFieldList.Delete(DBindex);
		  end;
		end;

		{read first line and create fields if asked to}
		if FNamesFirst and not FFixedWidth then begin

			{get number of fields}
			{since ParseList.Count = 0, only NumberOfField will be returned}
			NumberOfFields := 0;
			while not eof(InFile) do begin
				NumberOfFields := GetRecord(ParsedList, Infile);
			 	{skip blanks }
				if NumberOfFields = 0 then continue; {blank or comment}
				{-1 means line too long}
				if NumberOfFields < 0 then begin
					ErrorCode := eFieldTooLong;
					exit;
	  	  end;
				{otherwise}
				break;
			end;
			if NumberOfFields = 0 then begin
				ErrorCode := eNoRecordsFound;
				exit;
			end;

			{now get the names, first create some objects}
			for DBIndex := 0 to NumberOfFields - 1 do begin
				AFieldStream := TFieldStream.Create;
	      ParsedList.Add(AFieldStream);
			end;

			{get fields again but this time we have some items}
			Seek(InFile,0);
			while not eof(InFile) do begin
				NumberOfFields := GetRecord(ParsedList, Infile);
			 	{skip blanks }
				if NumberOfFields = 0 then continue; {blank or comment}
				{-1 means line too long}
				if NumberOfFields < 0 then begin
					ErrorCode := eFieldTooLong;
					exit; {this function}
	  	  end;
				{otherwise}
				break;
			end;
			if NumberOfFields = 0 then begin
				ErrorCode := eNoRecordsFound;
				exit;
			end;

			{go fill in names}
			FFieldList.clear;
			for DBIndex := 0 to NumberOfFields - 1 do begin
				FFieldList.add(Trim(TFieldStream(ParsedList.items[DBindex]).FieldString));
			end;

			{now delete our objects}
			for X := 0 to ParsedList.Count - 1 do begin
				if ParsedList.Items[x] <> nil then TFieldStream(ParsedList.Items[x]).Free;
    	end;
			ParsedList.Clear;
			{don't seek to start because we want to start at next record}
    end;

		{make sure we have at least one field to get}
		if FFieldList.Count < 1 then begin
			ErrorCode := eNoFieldsInList;
			exit;
		end;

		{make sure table is working}
		if FWriteEnable then begin
    	{save the state of other table}
    	OldActive := FDestination.Active;

			{table must be inactive, so we can get exclusive}
			if FDestination.Active then FDestination.Active := False;

			{create my own TTable. This give me default Fields as 1:1 of FieldDefs
			 and includes ALL the fields.}
			MyTable := TTable.Create(Self);
			MyTable.DatabaseName := FDestination.DatabaseName;
			MyTable.TableName := FDestination.TableName;
			MyTable.TableType := FDestination.TableType;
      MyTable.IndexName := FDestination.IndexName;

      {only need exclusive if we are empting table}
			if Fmode = mCopy then MyTable.Exclusive := True;

			{open database}
			try
				MyTable.Open;
			except
				on e: EDatabaseError do begin
					errorCode := 	eNoExclusiveAccess;
					exit;
        end;
      end;

			{if user list is longer then tables}
			X := 0;
			for DBIndex := 0 to FFieldList.Count - 1 do begin
				if UpperCase(FFieldList[DBIndex]) <> 'SKIP' then inc(X);
			end;

			if X > MyTable.FieldCount then begin
				errorCode := eTooManyFieldsInList;
				exit;
  	  end;

    end; {of if FWriteEnable}

		{can we find all the field names?, create OUR fields for them}
		for DBIndex := 0 to FFieldList.Count - 1 do begin
			FieldName := UpperCase(Trim(FFieldList[DBIndex]));

			{mark the end}
			EndFieldName := Pos(',',FieldName);
			FieldWidth := 0;
			if FFixedWidth then begin
				{must have comma}
				if EndFieldName = 0 then begin
					ErrorCode := eMissingFieldWidth;
					exit;
				end;
				FieldWidth := StrToIntDef(copy(FieldName,EndFieldName + 1, Length(FieldName)),0);
				if FieldWidth < 0 then begin
					ErrorCode := eMissingFieldWidth;
					exit;
        end;
			end;

      {remove length values}
			if EndFieldName > 0 then FieldName := Copy(FieldName,0,EndFieldName - 1);

			{Assign Field stream.  This is faster than using FindField on each call}
			if (FieldName <> 'SKIP') and FWriteEnable then begin {only if enabled}
				{Get the field}
				Afield := MyTable.FindField(FieldName);
				{Make a pointer to it}
				if AField = nil then begin
					{exit if we did not find it}
					errorCode := eFieldNotInTable;
					exit;
        end;
				{create a field stream and assign it}
				AFieldStream := TFieldStream.Create;
				AFieldStream.Field := AField;
	      AFieldStream.FieldLength := FieldWidth;
				ParsedList.Add(AFieldStream);
			end
			else begin {skip or write diable}
				{create a field stream.  Field will be nil}
				AFieldStream := TFieldStream.Create;
       	AFieldStream.FieldLength := FieldWidth;
				ParsedList.Add(AFieldStream);
			end;
		end; {of for DBIndex...}

		{if we are in update mode, one of the ascii fields must be the }
		{index key, find them.}

		if FWriteEnable and ((Fmode = mAppendUpdate) or (Fmode = mUpdate)) then begin
    	{make sure file is indexed}
    	if MyTable.IndexFieldCount = 0 then begin
				ErrorCode := eNoIndexFieldFound;
				exit;
      end
    end;

		{create dialog if needed}
		if FShowDlg then begin
			{setup ticker}
			ProgressTick := FileSize(InFile) div 100;
			if ProgressTick = 0 then begin
				ProgressTick := FileSize(InFile) div 10;
				if ProgressTick = 0 then ProgressTick := 1;
			end;

			{ok create controls}
			ProgressForm := TForm.Create(Self);
			with ProgressForm do begin
				Parent := Parent;
				ClientWidth := 200;
	    	if FNotifyCount > 0
				then ClientHeight:= 40
				else ClientHeight:= 20;
				BorderStyle := bsSingle;
				BorderIcons := [];
				Caption := FShowCaption;
				FormStyle := fsStayOnTop;
      end;

			{There only a cancel button if there is a notify count}
			if FNotifyCount > 0 then begin
				ProgressCancel := Tbutton.Create(Self);
				with ProgressCancel do begin
					Parent := ProgressForm;
					left := 0;
					width := ProgressForm.ClientWidth;
					top := ProgressForm.ClientHeight div 2 ;
					height := top;
					caption := FAbortCaption;
					OnClick := AbortTransfer;
				end;
			end;

	    ProgressForm.Left := TForm(Owner).Left + (TForm(Owner).Width div 2) - ProgressForm.Width div 2;
			ProgressForm.Top := TForm(Owner).Top + (TForm(Owner).Height div 2) - ProgressForm.Height div 2;
			ProgressForm.Show;
			ProgressForm.Update;
		end;

		{we are ready so empty old data}
		if FWriteEnable then begin
			if Fmode =	mCopy then MyTable.EmptyTable;
		end;

		{rip through file}
		while (not eof(InFile)) and (errorCode = eOK) do begin

			if FFixedWidth
			then NumberOfFields := GetFixedRecord(ParsedList, Infile)
			else NumberOfFields := GetRecord(ParsedList, Infile);

			{inc line counter}
			inc(LineCount);

			{-1 means line too long, can not proceed!}
			if NumberOfFields < 0 then begin
				ErrorCode := eFieldTooLong;
				break;
			end;

     	{skip blanks}
			if NumberOfFields = 0 then begin
				inc(EmptyCount);
				continue;
			end;

			{give user a chance to fix it}
			if assigned(FOnParse) then FOnParse(Self, ParsedList, sRecord, NumberOfFields);

			{did user abort?}
			if NumberOfFields < 0 then begin
				ErrorCode := eUserAborted;
				break;
			end;

     	{did user want to skip}
			if NumberOfFields = 0 then begin;
				inc(AbortedCount);
				continue;
			end;

			{we have a valid parse so increment counter}
		  inc(ParsedCount);

			{did we get the right number of fields}
			if NumberOfFields <> FFieldList.Count then begin
				errorCode := eFieldCount;
				if AbortOnError(NumberOfFields) then break;
				continue;
			end;

    	{This is part that is slow!}
			EditMode := emNone;
			if FWriteEnable then begin
				{see if record exist, if so, update it!}
				try
					if ((Fmode = mAppendUpdate) or (Fmode = mUpdate)) and KeyExist then begin
						MyTable.Edit;
						EditMode  := emEdit;
					end
    			else begin {we did not find it!}
	          if Fmode <> mUpdate then begin {no appends in mUpdate}
							MyTable.Append; {puts in edit mode}
							EditMode := emAppend;
          	end;
					end;
 				except
					on EDBEngineError do begin
						{other error}
	       		errorCode := eAppendOrEditError;
						break; {from record processing}
					end;
    	  end; {of try}

				{default true}
				OkToPost := true;
				{assign each field its string if we need to}
				if EditMode <> emNone then begin
					for DBIndex := 0 to ParsedList.Count - 1 do begin
          	AFieldStream := TFieldStream(ParsedList.Items[DBIndex]);
						{skip fields will be nil}
						if AFieldStream.Field <> nil then begin
							try
								case AFieldStream.Field.DataType of
								ftMemo: begin
									if AFieldStream.Size <= 32758
									then TmemoField(AFieldStream.Field).LoadFromStream(AFieldStream)
									else begin
										OkToPost := false;
										errorCode := eMemoTooLong;
										if AbortOnError(DBIndex) then break; {from field processing}
									end;
 	      	      end;
	              ftString: AFieldStream.Field.AsString := AFieldStream.FieldString;
								ftCurrency,ftFloat, ftBCD:
        					AFieldStream.Field.AsString := PrepNumber(AFieldStream.FieldString);
				        ftSmallint, ftInteger, ftWord:
        					AFieldStream.Field.AsString := PrepInt(AFieldStream.FieldString);
                ftDate: begin
                	if DateFormat = dfBCD
                  then TDateField(AFieldStream.Field).value := BCDToDate(Trim(AFieldStream.FieldString))
                  else AFieldStream.Field.AsString := Trim(AFieldStream.FieldString)
                end;
                else AFieldStream.Field.AsString := Trim(AFieldStream.FieldString);
              	end;
		  	  	  except
								{not all convert errors are ConvertErrors!}
								on EConvertError do begin
									OkToPost := false;
          		   	errorCode := eConversion;
									if AbortOnError(DBIndex) then break; {from field processing}
 	  	          end; {of EConvertError}

								on EDataBaseError do begin
									OkToPost := false;
									errorCode := eConversion;
									if AbortOnError(DBIndex) then break; {from field processing}
								end; {of EDataBaseError}
	  		      end; {of try...}
						end; {of if Field <> nil}
	    	  end;  {of for}

					{are we bailing out!}
					if OkToPost then begin
						{call OnPrePost for user}
		   			if assigned(FOnPrePost) then FOnPrePost(Self,MyTable,OKtoPost);
					end;

					{post the data}
					try
						if OkToPost then begin
  	  				MyTable.Post;
							if EditMode = emAppend
							then inc(AppendedCount)
							else inc(UpdatedCount);
						end
						else begin
							MyTable.Cancel;
						end;
    	  	except
						on E: EDBEngineError do begin
							MyTable.Cancel;
							{key violation? One would think there would be and easier way to do this}
							if E.Errors[0].ErrorCode = DBIERR_KEYVIOL {9729} then begin
								errorCode := eKeyViolation;
								if AbortOnError(0) then break;
	      	    end
							else begin
								{other error}
								errorCode := ePostError;
								break; {from processing records}
							end;
						end; {of on EDBEnginError}
					end; {of try...}
				end; {of if EditMode ...}
	    end; {of if WriteEnabled...}


			{call user update and give some time to other apps}
			if FNotifyCount > 0 then begin
				if ParsedCount mod FNotifyCount = 0 then begin
					Application.ProcessMessages;
					if assigned(FOnNotifyCount) then FOnNotifyCount(Self);
	 			end; {of ParsedCount...}
			end; {of FNotifyCount...}

			{update dialog}
			if FShowDlg and (FilePos(InFile) > NextTick) then begin
				NextTick := FilePos(InFile) + ProgressTick;
				UpdateDlg;
			end;
			{abort?}
			if (FRecordLimit > 0) and (ParsedCount >= FRecordLimit) then break;

 		end; {of while not eof...}

		{last update of dialog}
		if errorCode = eOk then begin
			if FShowDlg then UpdateDlg;
		end;

	{ok, shut it all down and clean up what we created}
	finally
  	{return date formats}
    ShortDateFormat := OldDateFormat;
	 	DateSeparator := OldDateSeparator;
    TimeSeparator := OldTimeSeparator;
  	DecimalSeparator := OldDecimalSeparator;

		{if we have multi errors then use general error}
		if FieldCountErrors + KeyViolationErrors + ConversionErrors + MemoErrors > 0
    then ErrorCode := eMultiError;

		{see if we have any records}
		if (errorCode = eOk) and (ParsedCount = 0) then begin
			ErrorCode := eNoRecordsFound;
		end;

		{set error line, MultiErrors are format errors are handled above}
		if (errorCode <> eOk) and (errorCode <> eMultiError) then begin
			ErrorLine := LineCount;
			ErrorField := 0;
		end;

		{close ascii file}
		CloseFile(InFile);

		{close error file}
		if SaveToErrorFile then begin
			CloseFile(efile);
			if FieldCountErrors + KeyViolationErrors + ConversionErrors + MemoErrors =  0
		 	then SysUtils.DeleteFile(FErrorFile);
		end;

		{delete MemoryStreams we added to the parsed list}
		{note: we don't delete the field object as that is deleted with the table}
		if (ParsedList <> nil) and (ParsedList.Count > 0) then begin
			for X := 0 to ParsedList.Count - 1 do begin
				if ParsedList.Items[x] <> nil then TFieldStream(ParsedList.Items[x]).Free;
    	end;
		end;

		{delete List itself}
		if ParsedList <> nil then ParsedList.Free;

		{Close table}
		if MyTable <> nil then begin
			MyTable.Close; {close the database}
			if Fmode = mCopy then MyTable.Exclusive := False;  {release control}
			MyTable.Free;
		end;

		{nuke progress window (and memory for it)}
		if ProgressCancel <> nil	then ProgressCancel.Free;
		if ProgressForm <> nil then begin
			ProgressForm.Close;
			ProgressForm.Free;
		end;

    {return table to it's previous state}
		if FWriteEnable then begin
	    if OldActive <> FDestination.Active
  	  then FDestination.Active := OldActive;
    end;

		{we're done!}
		busy := false;
	end;
end; {of execute}

{******************************************************************************}
procedure Tascii2DB.AbortTransfer(Sender: TObject);
begin
	ErrorCode := eUserAborted;
end; {of aborttranfer}


{******************************************************************************}
function Tascii2DB.GetRecordCount: LongInt;
type
	TEditMode = (emNone,emEdit,emAppend);
var
	InFile: File;								{file we are reading}
	NumberOfFields: LongInt;		{fields parsed}
	ParsedList: TList;					{a place to put parsed strings}
  s: string;									{a string}
	X: LongInt;									{a counter}
  AFieldStream: TFieldStream;
begin {of Execute}

	result := 0;
 	{abort if we are busy}
	if busy then exit;
  Busy := true;

	{clear globals}
  ErrorCode := eOk;
	ErrorLine := 0;
	ErrorField := 0;

	LineCount := 0;
	AbortedCount := 0;
	EmptyCount := 0;
	ParsedCount := 0;
	AppendedCount := 0;
	UpdatedCount := 0;

	FieldCountErrors := 0;
	KeyViolationErrors := 0;
	ConversionErrors := 0;
	MemoErrors := 0;

	{init locals}
  ParsedList := nil;

	{does source file exist?}
	if FAsciiFile = '' then begin
		errorCode := eAsciiNameMissing;
		busy := false;
		exit;
	end;

 	s := FAsciiFile + #0;
	AnsiToOEM(@s[1],@s[1]);
	SetLength(s, length(s) -1);
	if not FileExists(s) then begin
		errorCode := eAsciiFileNotFound;
		busy := false;
		exit;
	end;

	{open ascii file}
	try
		AssignFile(InFile, s);
		Reset(InFile,1);
	except
		on EInOutError do begin
			errorCode := eNoAsciiFileAccess;
			busy := false;
			exit;
		end;
	end;


	{try/finally to force us to dispose of string pRecord and pField}
	try
  	{make objects we need}
		ParsedList := TList.create; 		{create the string list}
		ParsedList.Clear;

    {we need at least one record}
 		AFieldStream := TFieldStream.Create;
    AFieldStream.FieldLength := 1;
    ParsedList.Add(AFieldStream);

		{rip through file}
		while (not eof(InFile)) and (errorCode = eOK) do begin

			if FFixedWidth
			then NumberOfFields := GetFixedRecord(ParsedList, Infile)
			else NumberOfFields := GetRecord(ParsedList, Infile);

			{inc line counter}
			inc(LineCount);

			{-1 means line too long}
			if NumberOfFields < 0 then begin
				ErrorCode := eFieldTooLong;
				break;
			end;

  		{skip blanks}
			if NumberOfFields = 0 then begin
 				inc(EmptyCount);
      	continue;
      end;

			{we have a valid parse so increment counter}
		  inc(ParsedCount);

			{call user update and give some time to other apps}
			if FNotifyCount > 0 then begin
				if ParsedCount mod FNotifyCount = 0 then begin
					Application.ProcessMessages;
 				end; {of ParsedCount...}
			end; {of FNotifyCount...}

  	end; {of while not eof...}

		{see if we have any records}
		if (errorCode = eOk) and (ParsedCount = 0) then begin
			ErrorCode := eNoRecordsFound;
		end;

	{ok, shut it all down and clean up what we created}
	finally
		{close ascii file}
		CloseFile(InFile);

		{delete MemoryStreams we added to the parsed list}
		{note: we don't delete the field object as that is deleted with the table}
		if (ParsedList <> nil) and (ParsedList.Count > 0) then begin
			for X := 0 to ParsedList.Count - 1 do begin
				if ParsedList.Items[x] <> nil then TFieldStream(ParsedList.Items[x]).Free;
    	end;
		end;

		{delete List itself}
		if ParsedList <> nil then ParsedList.Free;

		{we're done!}
		busy := false;
		result := ParsedCount;
	end;
end; {of GetRecordCount}


{******************************************************************************}
procedure Tascii2DB.StopExecute;
begin
	{setting error code aborts}
	ErrorCode := eUserAborted;
end; {of StopExecute}


{******************************************************************************}
procedure Tascii2DB.SetFieldList(Value: TStrings);
begin
  FFieldList.Assign(Value);
end; {of setfieldlist}

{******************************************************************************}
procedure Tascii2DB.SetRecordLimit(value: LongInt);
begin
	if value < 0
	then FRecordLimit := 0
	else FRecordLimit := value;
end; {of setRecordLimit}

{******************************************************************************}
procedure Tascii2DB.SetNamesFirst(value: boolean);
begin
	{First names can not be set on fixed width files}
	if (value = true) and (FFixedWidth = false)
	then FNamesFirst := true
	else FNamesFirst := value;
end;

{******************************************************************************}
procedure Tascii2DB.SetFixedWidth(value: boolean);
begin
	{Fixed width can not be set on files with first names}
	if (value = true) and (FFixedWidth = false)
	then FFixedWidth := true
	else FFixedWidth := value;
end;

{******************************************************************************}
procedure Tascii2DB.SetEndOfRecord(value: char);
begin
	{#0 not allowed!}
	if value < #1 then value := #13;
  FEndOfRecord := value;
end; {of setEndOfRecord}

{******************************************************************************}
procedure Tascii2DB.SetAsciiFile(value: TFileName);
begin
	FAsciiFile := Trim(value);
end; {of setAsciiFile}

{******************************************************************************}
procedure Tascii2DB.SetErrorFile(value: TFileName);
begin
	FErrorFile := Trim(value);
end; {of setErrorFile}

{******************************************************************************}
procedure Tascii2DB.SetMaxRecordErrors(value: longInt);
begin
	if value < 0
	then FMaxRecordErrors := 0
	else FMaxRecordErrors := value;
end;

{******************************************************************************}
procedure Tascii2DB.SetMaxFieldSize(value: LongInt);
begin
	if value < 0
	then FMaxFieldSize := 0
	else FMaxFieldSize := value;
end;


(******************************************************************************
 	Gets n fields from the specified file and returns then in a TList
  the entire record is also returned in pRecord (global)
  - File must be opened before calling.
  - File pointer left pointing to next character.
	-	Escaped characters (if enabled) are processed inside and out of delimiters.
	-	All characters inside delimiter are literal except:
    	a) <delimiter><delimiter> = <delimiter>
			b) FEscapes (if enabled)
	-	Outside of delimiters, <delimiter>, <seperater>, and <#26> are not allowed
		unless escaped.
	-	Delimiter are only allowed inside a delimiter pair if the are pair too.
	-	Comment lines are ignored (begin with ;)
	-	Comment line must terminate it a <EndOfRecord>.
	- Blank and comment lines return a 0 count.
  - <#26> (eof) is treated as end of file except inside quotes
	- If List is nil, nothing will be saved but the field count will be valid
  - If an error occurs, -1 will be reported, field count will be 0
	- If there are more fields found, extra fields are lost and the
		result will indicate number found.
	- If there are less fields found, extra fields not cleared
		result will indicate number found.

If List is empty (not nil) then a count of records will be return and pRecord will be filled.
*)
{******************************************************************************}
function TAscii2DB.GetRecord(var List: TList; var f: File): LongInt;
type
  TParseMode = (Normal, InQuote, InComment, Error, InEscape);

var
	sBlock: string;
	sBlockIndex: LongInt;
	sBlockEnd: LongInt;

  FoundEOR: boolean;		  {flag to indicate EOR was located}
	OnlySpaces: boolean;		{flag to indicate if only spaces have been parsed}
	CharsFound: LongInt;		{number of char read into buffer}

  ParsedFields: LongInt;  {Count of fields parsed}
	Escape: char;						{replacement escape char}
	QuoteCount: LongInt;	 	{counter of nested quotes}
	ParseMode: TParseMode;	{parse mode}
	FirstPass: boolean;			{marker to know this is the first read block}
  PreEscapeMode: TparseMode;
  IndexedChar: char;			{char being evaluated}

{copy the field data to the List}
procedure SaveField;
var
	OldFieldLength: LongInt;
begin
	{retain sField length}
	OldFieldLength := Length(sField);

	{test for count too high or write disabled}
	if ParsedFields < List.count then begin
	  {make local pointer}
  	with TFieldStream(List.Items[ParsedFields]) do begin
			{copy string, we have to set length}
  		SetLength(sField,sFieldCount);
	  	FieldString := sField;
		  {save data}
  		if Field <> nil then begin {skip fields are nil}
				if Field.DataType = ftMemo then begin
					{and write as stream too}
					Clear;
			 		Write(sField[1],sFieldCount);
				end;
			end;
    end;
  end;
  {point to next field and keep count}
	inc(ParsedFields);

	{reset the length to the default}
  SetLength(sField,OldFieldLength);
	sFieldCount := 0;
	OnlySpaces := true; {start looking for a non-space again}
end;


{******************************************************}
{******************************************************}
{******************************************************}
begin {of GetRecord}
	{default local variables}
	FoundEOR := false;
	ParsedFields := 0;
	OnlySpaces := true;
	ParseMode := Normal;
	QuoteCount := 0;
	Result := 0; 		{default to error!}
	FirstPass := true;
  sBlockIndex := 1;
  PreEscapeMode := Normal;

  {clear record and field}
  setLength(sRecord,0);
  SetLength(sField,0);

	{point field to start}
  sFieldCount := 0;

	{loop until a)EOR, b)EOF}
  while (not eof(f)) and (not FoundEOR) and (result = 0) do begin
	  {insure sBlock is long enough and unique}
		SetLength(sBlock, DefBlockLength);

  	{read some data}
		BlockRead(F, sblock[1], DefBlockLength, CharsFound);

		{start at begin}
    SetLength(sBlock, CharsFound);
    sBlockEnd := CharsFound;

		{see if this is a comment line}
		if FirstPass and (sBlock[1] = ';') and FAllowComments then ParseMode := InComment;
		FirstPass := false;

    {insure sField in long enough}
    if Length(sField) < sFieldCount + DefBlockLength
    then SetLength(sField, sFieldCount + DefBlockLength);

		{start of block processing from beginning of block}
    {block index point to char processing}
		sBlockIndex := 1;
		repeat
    	{make this local}
      IndexedChar := sBlock[sBlockIndex];
      if sFieldCount > FMaxFieldSize then begin
	      result := -1;
        break;
      end;

			{Comment, ignore all till EOR or EOF}
      if ParseMode = InComment then begin
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26)
				then begin
					{break from block processing}
					FoundEOR := true;
					break;
				end;
        {continue with block processing}
	    	inc(sBlockIndex);
				continue;
	    end;

			{Error Mode, try to resync and exit}
  	 	if ParseMode = Error then begin
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26)
				then begin
					{break from block processing}
					FoundEOR := true;
					break; {from block processing}
    	  end;
        {continue with block processing}
	    	inc(sBlockIndex);
				continue;
      end;

			if ParseMode = InEscape then begin
				escape := ' ';
				case IndexedChar of
					't': {Horz tab}
						if (esHT in FEscapes) then escape := #9;
          'n': {newline}
	     			if (esNL in FEscapes) then escape := #10;
					'r': {CR}
						if (esCR in FEscapes) then escape := #13;
					'\': {back slash}
						if (esBSlash in FEscapes) then escape := '\';
					#34: {double quote}
						if (esDQuote in FEscapes) then escape := '"';
					#39: {single quote}
						if (esSQuote in FEscapes) then escape := #39;
					'?': {double quote}
						if (esQues in FEscapes) then escape := '?';
					'0': {null}
						if (esNull in FEscapes) then escape := #0;
				end;
        {set mode back}
        ParseMode := PreEscapeMode;

        {see if we found an escaped character}
				if escape <> ' ' then begin
        	{save char}
          inc(sFieldCount);
          sField[sFieldCount] := escape;

					{continue with block processing}
		    	inc(sBlockIndex);
					continue;
        end;
       	{else add the \ char back}
        inc(sFieldCount);
        sField[sFieldCount] := '\';
      end;

// Note we should only be here if ParseMode = Normal or InQuote

      {Test for escape char}
			if (IndexedChar = '\') and (not (FEscapes = [])) then begin
      	{change mode}
        PreEscapeMode := ParseMode;
				ParseMode := InEscape;

        {continue with block processing}
	    	inc(sBlockIndex);
       	continue;
      end;

			{handle inside delimiter processing}
			if ParseMode = InQuote then begin
      	{is this the end?}
				if IndexedChar = FDelimiter then begin
        	{keep count}
        	inc(QuoteCount);

					{convert delimiter-delimiter to delimiter}
          if (odd(QuoteCount)) and (sField[sFieldCount] = FDelimiter) then begin
						{continue with block processing}
						inc(sBlockIndex);
            continue;
					end;

          {save potential end}
					sQuoteEnd := sFieldCount;

          {save as character}
					inc(sFieldCount);
          sField[sFieldCount] := IndexedChar;

					{continue with block processing}
					inc(sBlockIndex);
					continue;
				end;

				{seperater allowed on even quote joints only}
        {so, we stay in quote untill we find a seperator}
        {Note: EOF nor EOR will abort quote}

        {only exit InQuote if we have matched quotes}
				if (not odd(QuoteCount)) and
         ((IndexedChar = FSeparator)
					 or (IndexedChar = FEndOfRecord)
           or (IndexedChar = #26)) then begin
          {trim to end of quote}
        	sFieldCount := sQuoteEnd;

          {change mode}
					ParseMode := Normal;
        end
				{save and loop for more chars}
				else begin
          {save as character}
          inc(sFieldCount);
        	sField[sFieldCount] := IndexedChar;

					{continue with block processing}
					inc(sBlockIndex);
					continue;
				end;
			end;

			{handle normal processing}
			if ParseMode = Normal then begin
				{end of field?}
				if IndexedChar = FSeparator then begin
					{assign block to list}
					SaveField;

		    	inc(sBlockIndex);
					continue; {with block processing}
				end;

				{end of record?}
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26) then begin
					{remove trailing CR if processing LF as EOR}
          if sFieldCount > 0 then begin
						if (FEndOfRecord = #10) and (sField[sFieldCount] = #13) then begin
							dec(sFieldCount);
            end;
          end;

					{skip if this is an empty line}
					if not ((ParsedFields = 0) and (sFieldCount = 0)) then begin
						{assign block to list}
						SaveField;
					end;

					{break from block processing}
					FoundEOR := true;
					break;
				end;

				{else...is it the start of a delimiter?}
        if (IndexedChar = FDelimiter) and (OnlySpaces) then begin
        	{set up start of field}
					sFieldCount := 0;

					sQuoteEnd := 1;
					QuoteCount := 1;

          {change mode}
					ParseMode := InQuote;

          {continue with block processing}
          inc(sBlockIndex);
          continue;
				end;

        {record if not space}
				if IndexedChar <> ' ' then OnlySpaces := false;

        {save char}
        inc(sFieldCount);
        sField[sFieldCount] := IndexedChar;

        {continue with block processing}
        inc(sBlockIndex);
        {continue}
			end; {of normal parse mode}

		until sBlockIndex > sBlockEnd; {of block processing}

    {save block}
    setLength(sBlock, sBlockIndex -1);
    sRecord := sRecord + sBlock;
	end; {getting a record}

	{clean up any remaining text. Caused my missing EOR on last' record!}
	{Are there character in the field buffer}
	if sFieldCount > 0 then begin
		SaveField;
	end;
	{put pointer back}
	Seek(F,FilePos(F) - (CharsFound - sBlockIndex));

  {save results (unless there was an error)}
  if result = 0 then result := ParsedFields;

end; {of GetRecord}


{******************************************************************************}
function TAscii2DB.GetFixedRecord(var List: TList; var f: File): LongInt;
type
  TParseMode = (Normal, InComment, Error);

var
	sBlock: string;
	sBlockIndex: LongInt;
	sBlockEnd: LongInt;

  FoundEOR: boolean;		  {flag to indicate EOR was located}
	CharsFound: LongInt;		{number of char read into buffer}

  ParsedFields: LongInt;  {Count of fields parsed}
	ParseMode: TParseMode;	{parse mode}
	FirstPass: boolean;			{marker to know this is the first read block}
  IndexedChar: char;			{char being evaluated}


{copy the field data to the List}
procedure SaveField;
var
	OldFieldLength: LongInt;
begin
	{retain sField length}
	OldFieldLength := Length(sField);

	{test for count too high or write disabled}
	if ParsedFields < List.count then begin
	  {make local pointer}
  	with TFieldStream(List.Items[ParsedFields]) do begin
		  {save data}
  		if Field <> nil then begin {skip fields are nil}
				{copy string, we have to set length}
	  		SetLength(sField,sFieldCount);
		  	FieldString := sField;

				if Field.DataType = ftMemo then begin
					{and write as stream too}
					Clear;
			 		Write(sField[1],sFieldCount);
				end;
			end;
    end;
    {point to next field}
		inc(ParsedFields);
  end;

	{reset the length to the default}
  SetLength(sField,OldFieldLength);
	sFieldCount := 0;
end;

begin
	{default local variables}
	FoundEOR := false;
	ParsedFields := 0;
	ParseMode := Normal;
	Result := 0; 		{default to error!}
	FirstPass := true;
  sBlockIndex := 1;

  {clear record and field}
  SetLength(sRecord,0);
  SetLength(sField,0);

	{point field to start}
  sFieldCount := 0;

	{loop until a)EOR, b)EOF}
  while (not eof(f)) and (not FoundEOR) and (result = 0) do begin
	  {insure sBlock is long enough and unique}
		SetLength(sBlock, DefBlockLength);

  	{read some data}
		BlockRead(F, sblock[1], DefBlockLength, CharsFound);

		{start at begin}
    SetLength(sBlock, CharsFound);
    sBlockEnd := CharsFound;

		{see if this is a comment line}
		if FirstPass and (sBlock[1] = ';') and FAllowComments then ParseMode := InComment;
		FirstPass := false;

    {insure sField in long enough}
    if Length(sField) < sFieldCount + DefBlockLength
    then SetLength(sField, sFieldCount + DefBlockLength);

		{start of block processing from beginning of block}
    {block index point to char processing}
		sBlockIndex := 1;
		repeat
    	{make this local}
      IndexedChar := sBlock[sBlockIndex];
      if sFieldCount > FMaxFieldSize then begin
	      result := -1;
        break;
      end;

			{Comment, ignore all till EOR or EOF}
      if ParseMode = InComment then begin
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26)
				then begin
					{break from block processing}
					FoundEOR := true;
					break;
				end;
        {continue with block processing}
	    	inc(sBlockIndex);
				continue;
	    end;

			{Error Mode, try to resync and exit}
  	 	if ParseMode = Error then begin
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26)
				then begin
					{break from block processing}
					FoundEOR := true;
					break; {from block processing}
    	  end;
        {continue with block processing}
	    	inc(sBlockIndex);
				continue;
      end;

// Note we should only be here if ParseMode = Normal

			{handle normal processing}
			if ParseMode = Normal then begin
				{end of field?}

        if ParsedFields < List.Count then begin
					if sFieldCount >= TFieldStream(List.Items[ParsedFields]).FieldLength then begin
						{assign block to list}
						SaveField;
					end;
         end;

				{end of record?}
				if (IndexedChar = FEndOfRecord) or (IndexedChar = #26) then begin
					{remove trailing CR if processing LF as EOR}
          if sFieldCount > 0 then begin
						if (FEndOfRecord = #10) and (sField[sFieldCount] = #13) then begin
							dec(sFieldCount);
            end;
          end;

					{skip if this is an empty line}
					if not ((ParsedFields = 0) and (sFieldCount = 0)) then begin
						{assign block to list}
						SaveField;
					end;

					{break from block processing}
					FoundEOR := true;
					break;
				end;

        {save char}
        inc(sFieldCount);
        sField[sFieldCount] := IndexedChar;

        {continue with block processing}
        inc(sBlockIndex);
        {continue}
			end; {of normal parse mode}

		until sBlockIndex > sBlockEnd; {of block processing}

    {save block}
    setLength(sBlock, sBlockIndex -1);
    sRecord := sRecord + sBlock;
	end; {getting a record}

	{clean up any remaining text. Caused my missing EOR on last' record!}
	{Are there character in the field buffer}
	if sFieldCount > 0 then begin
		SaveField;
	end;
	{put pointer back}
	Seek(F,FilePos(F) - (CharsFound - sBlockIndex));

  {save results (unless there was an error)}
  if result = 0 then result := ParsedFields;

end; {of GetFixedRecord}


{******************************************************************************}
{******************************************************************************}
constructor TDB2Ascii.Create(Aowner: Tcomponent);
begin
	inherited create(Aowner);
	{set default public properties}
	Busy := false;
	FAsciiFile := '';
	Fdelimiter := '"';
	Fseparator := ',';
	FEndOfRecord := #13;
	FShowDlg := false;
	FNotifyCount := 100;
  FAlwaysQuote := true;

 	FShowCaption := 'Transfer Progress';
	FAbortCaption := '&Cancel';

  ErrorCode := eOk;
end; {of create}

{******************************************************************************}
procedure TDB2Ascii.Execute;
var
	x: LongInt;
	wfile: file;								{file to put error}
  s: string;
  WasClosed: boolean;
	CharWritten: LongInt;					{number of bytes written to a file}
  FirstPass: boolean;
  SkipIt: Boolean;

 	ProgressForm: TForm;				{progress dialog}
	ProgressCancel: TButton;		{cancel button in progress dialog}
	ProgressTick: LongInt;			{a bytes between dlg updates}
	NextTick: LongInt;					{a counter of bytes parsed}

{local procedure to execute}
procedure UpdateDlg;
var
	GPercent: word;					{graphical progress in percent}
	TPercent: word;					{text progress in percent}
	PercentText: String;    {percent text}
  NewRect: TRect;					{a rectangle}
begin
	if Source.RecordCount > 0 then begin
		GPercent := ProgressForm.ClientWidth * (Source.RecordCount +1) div Source.RecordCount;
		TPercent := 100 * WrittenCount div Source.RecordCount;
	end
  else begin
		GPercent := 0;
		TPercent := 0;
	end;

	{draw left background}
	if FNotifyCount > 0
	then NewRect := Rect(0, 0, GPercent, ProgressForm.ClientHeight div 2)
	else NewRect := Rect(0, 0, GPercent, ProgressForm.ClientHeight);
	ProgressForm.Canvas.Brush.Style := bsSolid;
	ProgressForm.Canvas.Brush.Color := clRed;
	ProgressForm.Canvas.FillRect(NewRect);

	{draw right background}
	if FNotifyCount > 0
	then NewRect := Rect(GPercent, 0, ProgressForm.ClientWidth,
              ProgressForm.ClientHeight div 2)
  else NewRect := Rect(GPercent, 0, ProgressForm.ClientWidth,
              ProgressForm.ClientHeight);

	ProgressForm.Canvas.Brush.Color := clBtnFace;
	ProgressForm.Canvas.FillRect(NewRect);

	{draw text}
	PercentText := IntToStr(TPercent) + '%';
	ProgressForm.Canvas.Brush.Style := bsClear;
  ProgressForm.Canvas.TextOut(90,2,PercentText);
end; {of UpdateDlg}

{******************************************************************************}
begin {of execute}
 	{abort if we are busy}
	if busy then exit;
  Busy := true;
 	ErrorCode := eOk;
	WrittenCount := 0;

	{init locals}
	ProgressForm := nil;
	ProgressCancel := nil;
  ProgressTick := 0;
	NextTick := 0;

	{does the table exist?}
 		if (FSource = nil)
		or (FSource.TableName = '')
		then begin
			ErrorCode := eTableMissing;
			busy := false;
			exit;
		end;

		if FSource.DataBaseName = '' then begin
			ErrorCode := eBadDatabaseName;
			busy := false;
			exit;
		end;

  {oem it}
	s := FAsciiFile + #0;
	AnsiToOEM(@s[1],@s[1]);
	SetLength(s, length(s) -1);

	{was destination file entered?}
	if s = '' then begin
		errorCode := eAsciiNameMissing;
		busy := false;
		exit;
	end;

  {does the file exist? If so, overwrite}
(*	aResult := mrYes;
  	if FileExists(s) then
		Aresult := messagedlg(s + #13#10 +
													'This file allready exist.' + #13#10 +
													'Do you want to replace it?',
													mtConfirmation, [mbYes,mbNo],0);
	if aResult = mrNo then begin
		ErrorCode := eUserAborted;
    busy := false;
		exit; {from from procedure}
  end;
*)

	{open a file here!}
	try
		AssignFile(wFile, s);
		Rewrite(wFile,1);
	except
		on EInOutError do begin
			errorCode := eNoAsciiFileAccess;
			exit;
		end;
	end;

	{open database}
	try
  	WasClosed := true;
  	if Source.Active
    then WasClosed := false
		else Source.Open;
	except
		on e: EDatabaseError do begin
			errorCode := 	eCannotOpenTable;
			exit;
    end;
  end;

	{create dialog if needed}
	if FShowDlg then begin
		{setup ticker}
		ProgressTick := Source.RecordCount div 100;
		if ProgressTick = 0 then begin
			ProgressTick := Source.RecordCount div 10;
			if ProgressTick = 0 then ProgressTick := 1;
		end;

		{ok create controls}
		ProgressForm := TForm.Create(Self);
		with ProgressForm do begin
			Parent := Parent;
			ClientWidth := 200;
	   	if FNotifyCount > 0
			then ClientHeight:= 40
			else ClientHeight:= 20;
			BorderStyle := bsSingle;
			BorderIcons := [];
			Caption := FShowCaption;
			FormStyle := fsStayOnTop;
  	end;

		{There only a cancel button if there is a notify count}
		if FNotifyCount > 0 then begin
			ProgressCancel := Tbutton.Create(Self);
			with ProgressCancel do begin
				Parent := ProgressForm;
				left := 0;
				width := ProgressForm.ClientWidth;
				top := ProgressForm.ClientHeight div 2 ;
				height := top;
				caption := FAbortCaption;
				OnClick := AbortTransfer;
			end;
		end;

	  ProgressForm.Left := TForm(Owner).Left + (TForm(Owner).Width div 2) - ProgressForm.Width div 2;
		ProgressForm.Top := TForm(Owner).Top + (TForm(Owner).Height div 2) - ProgressForm.Height div 2;
		ProgressForm.Show;
		ProgressForm.Update;
  end;


  {disable controls}
  Source.DisableControls;

  {write file}
  with Source do begin
  	First;
    while (not eof) and (ErrorCode = eOK) do begin      {wtext.txt}
    	FirstPass := true;
	    for x := 0 to FieldCount - 1 do begin
      	SkipIt := false;
        if Fields[x].visible = false
        then SkipIt := true
        else begin
					case Fields[x].DataType of
		        ftBCD, ftCurrency, ftFloat, ftInteger, ftSmallint, ftWord: begin
            	if AlwaysQuote then s := FDelimiter + Fields[X].AsString + FDelimiter
              else s := Fields[X].AsString;
            end;
    	      ftBoolean, ftDate, ftTime, ftDateTime, ftString: s := FDelimiter + Fields[X].AsString + FDelimiter;
	    	    else  SkipIt := True; {ftMemo, ftGraphic, ftBlob, ftBytes, ftVarBytes, ftUnknown,}

(*	DON'T SUPPRT THESE IN WRITE!
					ftMemo: begin
						if AFieldStream.Size <= 32758
						then TmemoField(AFieldStream.Field).LoadFromStream(AFieldStream)
						else begin
							errorCode := eMemoTooLong;
							if AbortOnError(DBIndex) then break; {from field processing}
						end;
*)

(* COULD IMPROVE THESE WITH FORMAT CHANGING
      	  ftDate,ftdateTime, ftTime: begin
          	if DateFormat = dfBCD
            then TDateField(AFieldStream.Field).value := BCDToDate(Trim(AFieldStream.FieldString))
	          else AFieldStream.Field.AsString := Trim(AFieldStream.FieldString)
*)

					end;
				end;
        if not SkipIt then begin
	        if not FirstPass then s := FSeparator + s;
  	      Firstpass := False;
					BlockWrite(wfile, s[1],Length(s),CharWritten);
					if CharWritten <> Length(s) then begin
						errorCode := eExportFileWriteError;
						exit;
					end;
        end;
  	  end;

			if FEndOfRecord = #13
      then s := #13#10
      else s := FEndOfRecord;
			BlockWrite(wfile, s[1],Length(s),CharWritten);
      inc(WrittenCount);

			{call user update and give some time to other apps}
			if FNotifyCount > 0 then begin
				if WrittenCount mod FNotifyCount = 0 then begin
					Application.ProcessMessages;
					if assigned(FOnNotifyCount) then FOnNotifyCount(Self);
	 			end; {of ParsedCount...}
			end; {of FNotifyCount...}


			{update dialog}
			if FShowDlg and (WrittenCount > NextTick) then begin
				NextTick := WrittenCount + ProgressTick;
				UpdateDlg;
			end;

      next;
    end;
  end;

	{close ascii file}
	CloseFile(wFile);

  {enable controls}
  Source.EnableControls;

  {close table if we opened!}
	if WasClosed then Source.Close;

	{nuke progress window (and memory for it)}
	if ProgressCancel <> nil	then ProgressCancel.Free;
	if ProgressForm <> nil then begin
		ProgressForm.Close;
		ProgressForm.Free;
  end;

	{we're done!}
	busy := false;

end;

{******************************************************************************}
procedure TDB2Ascii.SetEndOfRecord(value: char);
begin
	{#0 not allowed!}
	if value < #1 then value := #10;
  FEndOfRecord := value;
end; {of setEndOfRecord}

{******************************************************************************}
procedure TDB2Ascii.SetAsciiFile(value: TFileName);
begin
	FAsciiFile := Trim(value);
end; {of setAsciiFile}

{******************************************************************************}
procedure TDB2Ascii.StopExecute;
begin
	{setting error code aborts}
	ErrorCode := eUserAborted;
end; {of StopExecute}

{******************************************************************************}
procedure TDB2Ascii.AbortTransfer(Sender: TObject);
begin
	ErrorCode := eUserAborted;
end; {of aborttranfer}

{******************************************************************************}
{Global functions}
{******************************************************************************}
{Get alias of a table or query.  Returns empty if not found}
function GetDatabasePath(aDataSet: TDBDataSet): string;
var
	aList: TStringList;
	x: LongInt;
begin
	{this is a lot of work to get a database path}
	Result := aDataSet.DataBaseName;
	{if its not a directory the see if it is an alias}
	if not ((Result = '') or (Pos(':', Result) <> 0) or (Pos('\', Result) <> 0)) then begin
		{get a list of session aliases}
		AList := TStringlist.Create;
		Session.GetAliasNames(aList);
		{see if it is a good alias}
		Result := '';
		for x := 0 to alist.count - 1 do begin
			if alist[x] = aDataSet.DataBaseName then begin {it is an alias so get path}
     		Session.GetAliasParams(aDataSet.DataBaseName,alist);
				Result := Trim(alist[0]);
				delete(Result,1,5); {remove PATH=}
				break;
     	end;
		end;
		AList.free;
	end;
end;

{******************************************************************************}
function GetErrorString(Ecode: TDBcvtError): string;
begin
	case eCode of
		eOk: result 								  := 'No errors detected.';
		eNoExclusiveAccess: result 	  := 'Can not get exclusive access to table.';
	 	eAsciiFileNotFound: result 	  := 'Ascii file does not exist.'{: [' + FAsciiFile + ']'};
		eNoFieldsInList: result 			:= 'No fields in list';
		eDataTypeMismatch: result 		:= 'Data type mismatch during conversion.';
		eFieldNotInTable: result 			:= 'Listed field not found in table.';
		eFieldTooLong: result 				:= 'Field contains too many characters.'{, line: [' + IntToStr(Errorline) + ']'};
		eNoAsciiFileAccess: result 		:= 'Could not open ascii file.'	{: [' + FAsciiFile + ']'};
		eTooManyFieldsInList: result  := 'More fields in list than table.';
		eNoIndexFieldFound: result 		:= 'Table is not indexed.';
		eIndexFieldNotInList: result 	:= 'Table index field is not in list.';
		eMissingFieldWidth: result 		:= 'Zero, invalid or missing field width.';
		eAppendOrEditError: result		:= 'Error during database edit or append process.';
		ePostError: result						:= 'Error during database post process';
		eUserAborted: result 					:= 'Aborted by user.';
		eTableMissing: result     		:= 'Table not specified or missing table name';
		eBadDatabaseName: result			:= 'Missing database name in table.';
		eAsciiNameMissing: result 		:= 'Ascii file not specified';
		eCantOpenErrorFile: result		:= 'Error opening error file.'{: [' + FErrorFile + ']'};
		eErrorFileWriteError: result 	:= 'Error writting to error file.';
		eNoRecordsFound: result				:= 'No records found in ascii file';

		eFieldCount: result 					:= 'Wrong number of fields found.';
		eConversion: result 					:= 'Error converting data.';
		eKeyViolation: result 				:= 'Table index key violation error.';
		eMemoTooLong: result 				  := 'Record to long for a memofield.';
		eMultiError: result						:= 'General record error.';
    eOtherDatabaseError: result 	:= 'Other database error.';

 		eCannotOpenTable: result 	    := 'Can open table.';
		eExportFileWriteError: result := 'Error writting to export file.';

	  else result := 'Unknown error.';
	end;
end; {of get errorstring}


{******************************************************************************}
constructor TFieldStream.create;
begin
	inherited create;
	Clear;
	FieldLength := 0;
  FieldString := '';
	Field := nil;
end;


{******************************************************************************}
procedure Register;
begin
  RegisterComponents('Fitco', [Tascii2DB,TDB2ascii]);
end; {of register}


end. {of unit}


