{+--------------------------------------------------------------------------+
 | Component:   TmwComboTree
 | Created:     05.98
 | Author:      Martin Waldenburg
 | Copyright    1998, all rights reserved.
 | Description: Study on how to implement a ComboBox like component
 | Version:     1.0
 | DISCLAIMER:  This is provided as is, expressly without a warranty of any kind.
 |              You use it at your own risc.
 +--------------------------------------------------------------------------+}
unit mwCombTree;


interface

uses
  Windows, SysUtils, Messages, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, StdCtrls, ExtCtrls, Buttons, ComCtrls;

type
  TmwComboTree = class;

  TmwScreenTreeView = class(TTreeView)
  private
    FMouseInControl: Boolean;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
  end;

  TmwComboTree = class(TCustomPanel)
    Box: TSpeedButton;
    Edit: TEdit;
    Tree: TTreeView;
    procedure BoxClick(Sender: TObject);
  private
    function GetTree_Height: Integer;
    procedure SetTree_Height(newValue: Integer);
    procedure SetBox_Glyph(newValue: TBitmap);
    function GetBox_Glyph: TBitmap;
    procedure SetBox_Width(newValue: Integer);
    function GetBox_Width: Integer;
    function GetText: TCaption;
    procedure SetText(newValue: TCaption);
    procedure SetItems(newValue: TTreeNodes);
    function GetItems: TTreeNodes;
    function GetFont: TFont;
    procedure SetFont(newValue: TFont);
    procedure SetEdit_ReadOnly(newValue: Boolean);
    function GetEdit_ReadOnly: Boolean;
    procedure SetTree_Cursor(newValue: TCursor);
    function GetTree_Cursor: TCursor;
    procedure SetTree_Enabled(newValue: Boolean);
    function GetTree_Enabled: Boolean;
    procedure SetImages(newValue: TImageList);
    function GetImages: TImageList;
    procedure SetIndent(newValue: Integer);
    function GetIndent: Integer;
    procedure SetTree_ReadOnly(newValue: Boolean);
    function GetTree_ReadOnly: Boolean;
    procedure SetShowButtons(newValue: Boolean);
    function GetShowButtons: Boolean;
    procedure SetShowLines(newValue: Boolean);
    function GetShowLines: Boolean;
    procedure SetShowRoot(newValue: Boolean);
    function GetShowRoot: Boolean;
    procedure SetSortType(newValue: TSortType);
    function GetSortType: TSortType;
    procedure SetStateImages(newValue: TImageList);
    function GetStateImages: TImageList;
    procedure SetTree_Visible(newValue: Boolean);
    function GetTree_Visible: Boolean;
    function GetOnEdit_Change: TNotifyEvent;
    procedure SetOnEdit_Change(newValue: TNotifyEvent);
    function GetOnEdit_Click: TNotifyEvent;
    procedure SetOnEdit_Click(newValue: TNotifyEvent);
    function GetOnEdit_DblClick: TNotifyEvent;
    procedure SetOnEdit_DblClick(newValue: TNotifyEvent);
    function GetOnEdit_Enter: TNotifyEvent;
    procedure SetOnEdit_Enter(newValue: TNotifyEvent);
    function GetOnEdit_Exit: TNotifyEvent;
    procedure SetOnEdit_Exit(newValue: TNotifyEvent);
    function GetOnChange: TTVChangedEvent;
    procedure SetOnChange(newValue: TTVChangedEvent);
    function GetOnChanging: TTVChangingEvent;
    procedure SetOnChanging(newValue: TTVChangingEvent);
    function GetOnTree_Click: TNotifyEvent;
    procedure SetOnTree_Click(newValue: TNotifyEvent);
    function GetOnCollapsed: TTVExpandedEvent;
    procedure SetOnCollapsed(newValue: TTVExpandedEvent);
    function GetOnCollapsing: TTVCollapsingEvent;
    procedure SetOnCollapsing(newValue: TTVCollapsingEvent);
    function GetOnCompare: TTVCompareEvent;
    procedure SetOnCompare(newValue: TTVCompareEvent);
    function GetOnTree_DblClick: TNotifyEvent;
    procedure SetOnTree_DblClick(newValue: TNotifyEvent);
    function GetOnDeletion: TTVExpandedEvent;
    procedure SetOnDeletion(newValue: TTVExpandedEvent);
    function GetOnEdited: TTVEditedEvent;
    procedure SetOnEdited(newValue: TTVEditedEvent);
    function GetOnEditing: TTVEditingEvent;
    procedure SetOnEditing(newValue: TTVEditingEvent);
    function GetOnTree_Enter: TNotifyEvent;
    procedure SetOnTree_Enter(newValue: TNotifyEvent);
    function GetOnTree_Exit: TNotifyEvent;
    procedure SetOnTree_Exit(newValue: TNotifyEvent);
    function GetOnExpanded: TTVExpandedEvent;
    procedure SetOnExpanded(newValue: TTVExpandedEvent);
    function GetOnExpanding: TTVExpandingEvent;
    procedure SetOnExpanding(newValue: TTVExpandingEvent);
    function GetOnGetImageIndex: TTVExpandedEvent;
    procedure SetOnGetImageIndex(newValue: TTVExpandedEvent);
    function GetOnGetSelectedIndex: TTVExpandedEvent;
    procedure SetOnGetSelectedIndex(newValue: TTVExpandedEvent);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure SetBounds(Left, Top, Width, Height: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Add(Node: TTreeNode; const S: string): TTreeNode;
    function AddChild(Node: TTreeNode; const S: string): TTreeNode;
  published
    property Font: TFont read GetFont write SetFont;
    property Tree_Height: Integer read GetTree_Height write SetTree_Height;
    property Box_Glyph: TBitmap read GetBox_Glyph write SetBox_Glyph;
    property Box_Width: Integer read GetBox_Width write SetBox_Width;
    property Color;
    property Enabled;
    property Text: TCaption read GetText write SetText;
    property Visible;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property Items: TTreeNodes read GetItems write SetItems;
    property Edit_ReadOnly: Boolean read GetEdit_ReadOnly write SetEdit_ReadOnly;
    property Tree_Cursor: TCursor read GetTree_Cursor write SetTree_Cursor;
    property Tree_Enabled: Boolean read GetTree_Enabled write SetTree_Enabled;
    property Images: TImageList read GetImages write SetImages;
    property Indent: Integer read GetIndent write SetIndent;
    property Tree_ReadOnly: Boolean read GetTree_ReadOnly write SetTree_ReadOnly;
    property ShowButtons: Boolean read GetShowButtons write SetShowButtons;
    property ShowLines: Boolean read GetShowLines write SetShowLines;
    property ShowRoot: Boolean read GetShowRoot write SetShowRoot;
    property SortType: TSortType read GetSortType write SetSortType;
    property StateImages: TImageList read GetStateImages write SetStateImages;
    property Tree_Visible: Boolean read GetTree_Visible write SetTree_Visible;
    property OnEdit_Change: TNotifyEvent read GetOnEdit_Change write SetOnEdit_Change;
    property OnEdit_Click: TNotifyEvent read GetOnEdit_Click write SetOnEdit_Click;
    property OnEdit_DblClick: TNotifyEvent read GetOnEdit_DblClick write SetOnEdit_DblClick;
    property OnEdit_Enter: TNotifyEvent read GetOnEdit_Enter write SetOnEdit_Enter;
    property OnEdit_Exit: TNotifyEvent read GetOnEdit_Exit write SetOnEdit_Exit;
    property OnChange: TTVChangedEvent read GetOnChange write SetOnChange;
    property OnChanging: TTVChangingEvent read GetOnChanging write SetOnChanging;
    property OnTree_Click: TNotifyEvent read GetOnTree_Click write SetOnTree_Click;
    property OnCollapsed: TTVExpandedEvent read GetOnCollapsed write SetOnCollapsed;
    property OnCollapsing: TTVCollapsingEvent read GetOnCollapsing write SetOnCollapsing;
    property OnCompare: TTVCompareEvent read GetOnCompare write SetOnCompare;
    property OnTree_DblClick: TNotifyEvent read GetOnTree_DblClick write SetOnTree_DblClick;
    property OnDeletion: TTVExpandedEvent read GetOnDeletion write SetOnDeletion;
    property OnEdited: TTVEditedEvent read GetOnEdited write SetOnEdited;
    property OnEditing: TTVEditingEvent read GetOnEditing write SetOnEditing;
    property OnTree_Enter: TNotifyEvent read GetOnTree_Enter write SetOnTree_Enter;
    property OnTree_Exit: TNotifyEvent read GetOnTree_Exit write SetOnTree_Exit;
    property OnExpanded: TTVExpandedEvent read GetOnExpanded write SetOnExpanded;
    property OnExpanding: TTVExpandingEvent read GetOnExpanding write SetOnExpanding;
    property OnGetImageIndex: TTVExpandedEvent read GetOnGetImageIndex write SetOnGetImageIndex;
    property OnGetSelectedIndex: TTVExpandedEvent read GetOnGetSelectedIndex write SetOnGetSelectedIndex;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('mw', [TmwComboTree]);
end; { Register }

procedure TmwScreenTreeView.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin Style := Style or WS_BORDER; ExStyle := WS_EX_TOOLWINDOW or
    WS_EX_TOPMOST; WindowClass.Style := CS_SAVEBITS;
  end;
end; { CreateParams }

procedure TmwScreenTreeView.CreateWnd;
begin
  inherited CreateWnd;
  Windows.SetParent(Handle, 0); CallWindowProc(DefWndProc, Handle, wm_SetFocus, 0, 0);
end; { CreateWnd }

procedure TmwScreenTreeView.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  FMouseInControl := True;
  ReleaseCapture;
end; { CMMouseEnter }

procedure TmwScreenTreeView.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  FMouseInControl := False;
  SetCaptureControl(Self);
end; { CMMouseLeave }

procedure TmwScreenTreeView.WMMouseMove(var Message: TWMMouseMove);
var
  TreeHitTest: THitTests;
begin
  inherited;
  if FMouseInControl and Enabled then
  begin
    TreeHitTest := GetHitTestInfoAt(Message.XPos, Message.YPos);
    if htOnLabel in TreeHitTest then
      Selected := GetNodeAt(Message.XPos, Message.YPos);
  end;
end; { WMMouseMove }

procedure TmwScreenTreeView.WMLButtonDown(var Message: TWMLButtonDown);
var
  HitTest: THitTests;
begin
  inherited;
  Case FMouseInControl of
    False:
      begin
        ReleaseCapture;
        Enabled := False;
        Visible := False;
      end;
    True:
      begin
        SendMessage((Owner as TmwComboTree).Handle, WM_LBUTTONDOWN, 0, 0);
        HitTest := GetHitTestInfoAt(Message.XPos, Message.YPos);
        if htOnLabel in HitTest then
        begin
          (Owner as TmwComboTree).Edit.Text := Selected.Text;
          Selected.EndEdit(True);
          Enabled := False;
          Visible := False;
        end;
      end;
  end;
end; { WMLButtonDown }

destructor TmwComboTree.Destroy;
begin
  inherited Destroy;
end; { Destroy }

constructor TmwComboTree.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 141;
  Height := 21;
  TabOrder := 0;
  Caption := '';
  Text := '';
  BevelOuter := bvNone;
  BorderStyle := bsSingle;

  Box := TSpeedButton.Create(Self);
  Box.Parent := Self;
  Box.Width := 17;

  Edit := TEdit.Create(Self);
  Edit.Parent := Self;

  Tree := TmwScreenTreeView.Create(Self);
  Tree.Parent := Self;
  Tree.Height := 97;
end; { Create }

procedure TmwComboTree.CreateWindowHandle(const Params: TCreateParams);
begin
  inherited CreateWindowHandle(Params);

  with Box do
  begin
    Top := 0;
    Left := Self.Width - Box.Width - 4;
    Height := Self.Height - 4;
    OnClick := BoxClick;
  end; { Box }

  with Edit do
  begin
    Left := -2;
    Top := -2;
    Height := Self.Height;
    Width := Self.Width - Box.Width - 2;
   // AutoSize := False;
  end; { Edit }

  with Tree do
  begin
    Left := 0;
    Top := 0;
    Width := 0;
    Enabled := False;
    Visible := False;
  end; { Tree }
end; { CreateWindowHandle }

procedure TmwComboTree.BoxClick(Sender: TObject);
var
  CP, SP: TPoint;
begin
  CP.X := Left;
  CP.Y := Top + Height;
  SP := (Owner as TForm).ClientToScreen(CP);

  with Tree do
  begin
    Left := SP.X;
    Top := SP.Y;
    Width := Self.Width;
    Enabled := True;
    Visible := True;
  end; { Tree }
  SetCaptureControl(Tree);
end; { BoxClick }

procedure TmwComboTree.SetBounds(Left, Top, Width, Height: Integer);
begin
  Case Parent <> nil of
    True:
      begin
        inherited SetBounds(Left, Top, Width, Height);
        with Box do
        begin
          Left := Self.Width - Box.Width - 4;
          Height := Self.Height - 4;
        end;
        with Edit do
        begin
          Height := Self.Height;
          Width := Self.Width - Box.Width - 2;
        end;
      end;
    False: inherited SetBounds(Left, Top, Width, Height);
  end;
end; { SetBounds }

function TmwComboTree.GetTree_Height: Integer;
begin
  GetTree_Height := Tree.Height
end; { GetTree_Height }

procedure TmwComboTree.SetTree_Height(newValue: Integer);
begin
  Tree.Height := newValue;
end; { SetTree_Height }

procedure TmwComboTree.SetBox_Glyph(newValue: TBitmap);
begin
  Box.Glyph := newValue;
end; { SetBox_Glyph }

function TmwComboTree.GetBox_Glyph: TBitmap;
begin
  GetBox_Glyph := Box.Glyph;
end; { GetBox_Glyph }

procedure TmwComboTree.SetBox_Width(newValue: Integer);
begin
  Box.Width := newValue;
  Edit.Width := Width - newValue - 2;
  Box.Left := Width - newValue - 4;
end; { SetBox_Width }

function TmwComboTree.GetBox_Width: Integer;
begin
  GetBox_Width := Box.Width;
end; { GetBox_Width }

function TmwComboTree.GetText: TCaption;
begin
  GetText := Edit.Text;
end; { GetText }

procedure TmwComboTree.SetText(newValue: TCaption);
begin
  Edit.Text := newValue;
end; { SetText }

procedure TmwComboTree.SetItems(newValue: TTreeNodes);
begin
  Tree.Items := newValue;
end; { SetItems }

function TmwComboTree.GetItems: TTreeNodes;
begin
  GetItems := Tree.Items;
end; { GetItems }

function TmwComboTree.GetFont: TFont;
begin
  GetFont := Inherited Font;
end; { GetFont }

procedure TmwComboTree.SetFont(newValue: TFont);
begin
  Inherited Font := newValue;
  Height := Edit.Height;
  Box.Height := Edit.Height - 4;
end; { SetFont }

procedure TmwComboTree.SetEdit_ReadOnly(newValue: Boolean);
begin
  Edit.ReadOnly := newValue;
end; { SetEdit_ReadOnly }

function TmwComboTree.GetEdit_ReadOnly: Boolean;
begin
  GetEdit_ReadOnly := Edit.ReadOnly;
end; { GetEdit_ReadOnly }

procedure TmwComboTree.SetTree_Cursor(newValue: TCursor);
begin
  Tree.Cursor := newValue;
end; { SetTree_Cursor }

function TmwComboTree.GetTree_Cursor: TCursor;
begin
  GetTree_Cursor := Tree.Cursor;
end; { GetTree_Cursor }

procedure TmwComboTree.SetTree_Enabled(newValue: Boolean);
begin
  Tree.Enabled := newValue;
end; { SetTree_Enabled }

function TmwComboTree.GetTree_Enabled: Boolean;
begin
  GetTree_Enabled := Tree.Enabled;
end; { GetTree_Enabled }

procedure TmwComboTree.SetImages(newValue: TImageList);
begin
  Tree.Images := newValue;
end; { SetImages }

function TmwComboTree.GetImages: TImageList;
begin
  GetImages := Tree.Images;
end; { GetImages }

procedure TmwComboTree.SetIndent(newValue: Integer);
begin
  Tree.Indent := newValue;
end; { SetIndent }

function TmwComboTree.GetIndent: Integer;
begin
  GetIndent := Tree.Indent;
end; { GetIndent }

procedure TmwComboTree.SetTree_ReadOnly(newValue: Boolean);
begin
  Tree.ReadOnly := newValue;
end; { SetTree_ReadOnly }

function TmwComboTree.GetTree_ReadOnly: Boolean;
begin
  GetTree_ReadOnly := Tree.ReadOnly;
end; { GetTree_ReadOnly }

procedure TmwComboTree.SetShowButtons(newValue: Boolean);
begin
  Tree.ShowButtons := newValue;
end; { SetShowButtons }

function TmwComboTree.GetShowButtons: Boolean;
begin
  GetShowButtons := Tree.ShowButtons;
end; { GetShowButtons }

procedure TmwComboTree.SetShowLines(newValue: Boolean);
begin
  Tree.ShowLines := newValue;
end; { SetShowLines }

function TmwComboTree.GetShowLines: Boolean;
begin
  GetShowLines := Tree.ShowLines;
end; { GetShowLines }

procedure TmwComboTree.SetShowRoot(newValue: Boolean);
begin
  Tree.ShowRoot := newValue;
end; { SetShowRoot }

function TmwComboTree.GetShowRoot: Boolean;
begin
  GetShowRoot := Tree.ShowRoot;
end; { GetShowRoot }

procedure TmwComboTree.SetSortType(newValue: TSortType);
begin
  Tree.SortType := newValue;
end; { SetSortType }

function TmwComboTree.GetSortType: TSortType;
begin
  GetSortType := Tree.SortType;
end; { GetSortType }

procedure TmwComboTree.SetStateImages(newValue: TImageList);
begin
  Tree.StateImages := newValue;
end; { SetStateImages }

function TmwComboTree.GetStateImages: TImageList;
begin
  GetStateImages := Tree.StateImages;
end; { GetStateImages }

procedure TmwComboTree.SetTree_Visible(newValue: Boolean);
begin
  Tree.Visible := newValue;
end; { SetTree_Visible }

function TmwComboTree.GetTree_Visible: Boolean;
begin
  GetTree_Visible := Tree.Visible;
end; { GetTree_Visible }

function TmwComboTree.GetOnEdit_Change: TNotifyEvent;
begin
  GetOnEdit_Change := Edit.OnChange
end; { GetOnEdit_Change }

procedure TmwComboTree.SetOnEdit_Change(newValue: TNotifyEvent);
begin
  Edit.OnChange := newValue;
end; { SetOnEdit_Change }

function TmwComboTree.GetOnEdit_Click: TNotifyEvent;
begin
  GetOnEdit_Click := Edit.OnClick;
end; { GetOnEdit_Click }

procedure TmwComboTree.SetOnEdit_Click(newValue: TNotifyEvent);
begin
  Edit.OnClick := newValue;
end; { SetOnEdit_Click }

function TmwComboTree.GetOnEdit_DblClick: TNotifyEvent;
begin
  GetOnEdit_DblClick := Edit.OnDblClick;
end; { GetOnEdit_DblClick }

procedure TmwComboTree.SetOnEdit_DblClick(newValue: TNotifyEvent);
begin
  Edit.OnDblClick := newValue;
end; { SetOnEdit_DblClick }

function TmwComboTree.GetOnEdit_Enter: TNotifyEvent;
begin
  GetOnEdit_Enter := Edit.OnEnter;
end; { GetOnEdit_Enter }

procedure TmwComboTree.SetOnEdit_Enter(newValue: TNotifyEvent);
begin
  Edit.OnEnter := newValue;
end; { SetOnEdit_Enter }

function TmwComboTree.GetOnEdit_Exit: TNotifyEvent;
begin
  GetOnEdit_Exit := Edit.OnExit;
end; { GetOnEdit_Exit }

procedure TmwComboTree.SetOnEdit_Exit(newValue: TNotifyEvent);
begin
  Edit.OnExit := newValue;
end; { SetOnEdit_Exit }

function TmwComboTree.GetOnChange: TTVChangedEvent;
begin
  GetOnChange := Tree.OnChange;
end; { GetOnChange }

procedure TmwComboTree.SetOnChange(newValue: TTVChangedEvent);
begin
  Tree.OnChange := newValue;
end; { SetOnChange }

function TmwComboTree.GetOnChanging: TTVChangingEvent;
begin
  GetOnChanging := Tree.OnChanging;
end; { GetOnChanging }

procedure TmwComboTree.SetOnChanging(newValue: TTVChangingEvent);
begin
  Tree.OnChanging := newValue;
end; { SetOnChanging }

function TmwComboTree.GetOnTree_Click: TNotifyEvent;
begin
  GetOnTree_Click := Tree.OnClick;
end; { GetOnTree_Click }

procedure TmwComboTree.SetOnTree_Click(newValue: TNotifyEvent);
begin
  Tree.OnClick := newValue;
end; { SetOnTree_Click }

function TmwComboTree.GetOnCollapsed: TTVExpandedEvent;
begin
  GetOnCollapsed := Tree.OnCollapsed;
end; { GetOnCollapsed }

procedure TmwComboTree.SetOnCollapsed(newValue: TTVExpandedEvent);
begin
  Tree.OnCollapsed := newValue;
end; { SetOnCollapsed }

function TmwComboTree.GetOnCollapsing: TTVCollapsingEvent;
begin
  GetOnCollapsing := Tree.OnCollapsing;
end; { GetOnCollapsing }

procedure TmwComboTree.SetOnCollapsing(newValue: TTVCollapsingEvent);
begin
  Tree.OnCollapsing := newValue;
end; { SetOnCollapsing }

function TmwComboTree.GetOnCompare: TTVCompareEvent;
begin
  GetOnCompare := Tree.OnCompare;
end; { GetOnCompare }

procedure TmwComboTree.SetOnCompare(newValue: TTVCompareEvent);
begin
  Tree.OnCompare := newValue;
end; { SetOnCompare }

function TmwComboTree.GetOnTree_DblClick: TNotifyEvent;
begin
  GetOnTree_DblClick := Tree.OnDblClick;
end; { GetOnTree_DblClick }

procedure TmwComboTree.SetOnTree_DblClick(newValue: TNotifyEvent);
begin
  Tree.OnDblClick := newValue;
end; { SetOnTree_DblClick }

function TmwComboTree.GetOnDeletion: TTVExpandedEvent;
begin
  GetOnDeletion := Tree.OnDeletion;
end; { GetOnDeletion }

procedure TmwComboTree.SetOnDeletion(newValue: TTVExpandedEvent);
begin
  Tree.OnDeletion := newValue;
end; { SetOnDeletion }

function TmwComboTree.GetOnEdited: TTVEditedEvent;
begin
  GetOnEdited := Tree.OnEdited;
end; { GetOnEdited }

procedure TmwComboTree.SetOnEdited(newValue: TTVEditedEvent);
begin
  Tree.OnEdited := newValue;
end; { SetOnEdited }

function TmwComboTree.GetOnEditing: TTVEditingEvent;
begin
  GetOnEditing := Tree.OnEditing;
end; { GetOnEditing }

procedure TmwComboTree.SetOnEditing(newValue: TTVEditingEvent);
begin
  Tree.OnEditing := newValue;
end; { SetOnEditing }

function TmwComboTree.GetOnTree_Enter: TNotifyEvent;
begin
  GetOnTree_Enter := Tree.OnEnter;
end; { GetOnTree_Enter }

procedure TmwComboTree.SetOnTree_Enter(newValue: TNotifyEvent);
begin
  Tree.OnEnter := newValue;
end; { SetOnTree_Enter }

function TmwComboTree.GetOnTree_Exit: TNotifyEvent;
begin
  GetOnTree_Exit := Tree.OnExit;
end; { GetOnTree_Exit }

procedure TmwComboTree.SetOnTree_Exit(newValue: TNotifyEvent);
begin
  Tree.OnExit := newValue;
end; { SetOnTree_Exit }

function TmwComboTree.GetOnExpanded: TTVExpandedEvent;
begin
  GetOnExpanded := Tree.OnExpanded;
end; { GetOnExpanded }

procedure TmwComboTree.SetOnExpanded(newValue: TTVExpandedEvent);
begin
  Tree.OnExpanded := newValue;
end; { SetOnExpanded }

function TmwComboTree.GetOnExpanding: TTVExpandingEvent;
begin
  GetOnExpanding := Tree.OnExpanding;
end; { GetOnExpanding }

procedure TmwComboTree.SetOnExpanding(newValue: TTVExpandingEvent);
begin
  Tree.OnExpanding := newValue;
end; { SetOnExpanding }

function TmwComboTree.GetOnGetImageIndex: TTVExpandedEvent;
begin
  GetOnGetImageIndex := Tree.OnGetImageIndex;
end; { GetOnGetImageIndex }

procedure TmwComboTree.SetOnGetImageIndex(newValue: TTVExpandedEvent);
begin
  Tree.OnGetImageIndex := newValue;
end; { SetOnGetImageIndex }

function TmwComboTree.GetOnGetSelectedIndex: TTVExpandedEvent;
begin
  GetOnGetSelectedIndex := Tree.OnGetSelectedIndex;
end; { GetOnGetSelectedIndex }

procedure TmwComboTree.SetOnGetSelectedIndex(newValue: TTVExpandedEvent);
begin
  Tree.OnGetSelectedIndex := newValue;
end; { SetOnGetSelectedIndex }

function TmwComboTree.Add(Node: TTreeNode; const S: string): TTreeNode;
begin
  Result := Tree.Items.Add(Node, S);
end; { Add }

function TmwComboTree.AddChild(Node: TTreeNode; const S: string): TTreeNode;
begin
  Result := Tree.Items.AddChild(Node, S);
end; { AddChild }

end.

