{ -------------------------------------------------------------------------------------}
{ A "matcher" component for Delphi32.                                                  }
{ Copyright 1996, Patrick Brisacier and Jean-Fabien Connault.  All Rights Reserved.    }
{ This component can be freely used and distributed in commercial and private          }
{ environments, provided this notice is not modified in any way.                       }
{ -------------------------------------------------------------------------------------}
{ Feel free to contact us if you have any questions, comments or suggestions at        }
{   PBrisacier@mail.dotcom.fr (Patrick Brisacier)                                      }
{   JFConnault@mail.dotcom.fr (Jean-Fabien Connault)                                   }
{ You can always find the latest version of this component at:                         }
{   http://www.worldnet.net/~cycocrew/delphi/                                          }
{ -------------------------------------------------------------------------------------}
{ Date last modified:  08/08/96                                                        }
{ -------------------------------------------------------------------------------------}

{ -------------------------------------------------------------------------------------}
{ TMatch v1.01                                                                         }
{ -------------------------------------------------------------------------------------}
{ Description:                                                                         }
{   A component that allows you to match files.                                        }
{ Properties:                                                                          }
{   property CaseSensitive: Boolean;                                                   }
{   property Pattern: String;                                                          }
{   property Source: String;                                                           }
{   property SourceIsFile: Boolean;                                                    }
{ Procedures and functions:                                                            }
{   function Match: Boolean;                                                           }
{                                                                                      }
{ See example contained in example.zip file for more details.                          }
{ -------------------------------------------------------------------------------------}
{ Revision History:                                                                    }
{ 1.00:  + Initial release                                                             }
{ 1.01:  + Added CaseSensitive property                                                }
{ -------------------------------------------------------------------------------------}

unit Match;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, BrkApart;

type
  TMatch = class(TComponent)
  private
    { Private-dclarations }
    FCaseSensitive: Boolean;
    FSourceIsFile: Boolean;
    FSource: TStringList;
    FStringSource: String;
    FPattern: String;
    FConvert: TBrkApart;
    function GetSource: String;
    procedure SetSource(ASource: String);
    function GetPattern: String;
    procedure SetPattern(APattern: String);
  protected
    { Protected-dclarations }
  public
    { Public-dclarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Match: Boolean;
  published
    { Published-dclarations }
    property CaseSensitive: Boolean read FCaseSensitive write FCaseSensitive;
    property Pattern: String read GetPattern write SetPattern;
    property Source: String read GetSource write SetSource;
    property SourceIsFile: Boolean read FSourceIsFile write FSourceIsFile;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Systme', [TMatch]);
end;

constructor TMatch.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSource := TStringList.Create;
  FConvert := TBrkApart.Create(Self);
  FConvert.BreakString := ';';
end;

destructor TMatch.Destroy;
begin
  FSource.Free;
  FConvert.Free;
  inherited Destroy;
end;

function TMatch.GetSource: String;
begin
  if FSourceIsFile then begin { Source = nom(s) de fichier }
    FConvert.StringList.Assign(FSource);
    FConvert.ReverseBreakApart;
    Result := FConvert.BaseString;
  end
  else begin { Source = une chaine }
    Result := FStringSource;
  end;
end;

procedure TMatch.SetSource(ASource: String);
begin
  if FSourceIsFile then begin { Source = nom(s) de fichier }
    FConvert.BaseString := ASource;
    FConvert.BreakApart;
    FSource.Assign(FConvert.StringList);
  end
  else begin { Source = une chaine }
    FStringSource := ASource;
  end;
end;

function TMatch.GetPattern: String;
begin
  Result := Copy(FPattern, 2, Length(FPattern) - 2);
end;

procedure TMatch.SetPattern(APattern: String);
begin
  FPattern := '*' + APattern + '*';
end;

function TMatch.Match: Boolean;
{ Returns true if the string Source matches the Pattern, which may contain
wildcards * and ?. }

function RMatch(s: PChar; i: Integer; p: PChar; j: Integer): Boolean;
Var
  matched        : Boolean;
  k              : Integer;
begin
  if p[0]=#0 then
    RMatch :=  TRUE
  else while TRUE do
    if (s[i]=#0) and (p[j]=#0) then begin
      RMatch :=  TRUE;
      exit
    end
    else
      if p[j] = #0 then begin
        RMatch :=  FALSE;
        exit
      end
      else if (p[j] = '*') then begin
        k := i;
        if (p[j + 1] = #0) then begin
          RMatch :=  TRUE;
          exit
        end
        else while TRUE do begin
          matched := RMatch(s, k, p, j + 1);
          if matched OR (s[k] = #0) then begin
            RMatch :=  matched;
            exit;
          end; {if}
          inc(k);

        end; {while}
      end {if}
       else
       begin
        if not FCaseSensitive then
         begin
          p[j] := UpCase(p[j]);
          s[i] := UpCase(s[i]);
         end;

        if ((p[j] = '?') and (s[i] <> #0)) OR (p[j] = s[i]) then
         begin
          inc(i);
          inc(j);
         end
        else
         begin
          RMatch :=  FALSE;
          exit
         end;

       end;

end;

var
  F: TextFile;
  PSource, PPattern: PChar;
  PLine: array[0..255] of Char;
  BMatch: Boolean;
  iFile: Integer;
begin {Match}
  GetMem(PPattern, Length(FPattern) + 1);
  StrPCopy(PPattern, FPattern);
  if FSourceIsFile then begin { Source = nom(s) de fichier }
    BMatch := False;
    for iFile := 0 to FSource.Count - 1 do begin { pour chaque fichier }
      AssignFile(F, FSource[iFile]); { Fichier slectionn }
      Reset(F);
      while not Eof(F) do begin
        Readln(F, PLine);
        BMatch := BMatch or RMatch(PLine, 0, PPattern, 0);
      end;
      CloseFile(F);
    end; { for iFile }
    Match := BMatch;
  end
  else begin { Source = une chaine }
    GetMem(PSource, Length(FStringSource) + 1);
    StrPCopy(PSource, FStringSource);
    BMatch :=  RMatch(PSource, 0, PPattern, 0);
    Match := BMatch;
    FreeMem(PSource, Length(FStringSource) + 1);
  end;
  FreeMem(PPattern, Length(FPattern) + 1);


end;

end.
