{ -------------------------------------------------------------------------------------}
{ A "file finder" component for Delphi32.                                              }
{ Copyright 1996, Patrick Brisacier and Jean-Fabien Connault.  All Rights Reserved.    }
{ This component can be freely used and distributed in commercial and private          }
{ environments, provided this notice is not modified in any way.                       }
{ -------------------------------------------------------------------------------------}
{ Feel free to contact us if you have any questions, comments or suggestions at        }
{ PBrisacier@mail.dotcom.fr (Patrick Brisacier)                                        }
{ JFConnault@mail.dotcom.fr (Jean-Fabien Connault)                                     }
{ -------------------------------------------------------------------------------------}
{ Date last modified:  08/20/96                                                        }
{ -------------------------------------------------------------------------------------}

{ -------------------------------------------------------------------------------------}
{ TFileFind v1.04                                                                      }
{ -------------------------------------------------------------------------------------}
{ Description:                                                                         }
{   A component that allows you to find files through drives.                          }
{ Properties:                                                                          }
{   property FilePattern: String;                                                      }
{   property FilesFound: TStringList;                                                  }
{   property MatchCaseSensitive: Boolean;                                              }
{   property MatchEnabled: Boolean;                                                    }
{   property MatchString: String;                                                      }
{   property Priority: TThreadPriority;                                                }
{   property Recursive: Boolean;                                                       }
{   property StartDir: String;                                                         }
{   property OnTerminated: TNotifyEvent;                                               }
{ Procedures and functions:                                                            }
{   procedure Execute;                                                                 }
{   procedure ThreadExecute;                                                           }
{ Needs:                                                                               }
{   TBrkApart component from Patrick Brisacier and Jean-Fabien Connault                }
{   TMatch component from Patrick Brisacier and Jean-Fabien Connault                   }
{                                                                                      }
{ See example contained in example.zip file for more details.                          }
{ -------------------------------------------------------------------------------------}
{ Revision History:                                                                    }
{ 1.00:  + Initial release                                                             }
{ 1.01:  + Added Recursive property                                                    }
{ 1.02:  + Added support for french and english languages                              }
{ 1.03:  + Added MatchEnable, MatchString and MatchCaseSensitive properties            }
{        + Renamed Priority property in ThreadPriority                                 }
{ 1.04   + Renamed MatchEnable property in MatchEnabled                                }
{ -------------------------------------------------------------------------------------}

unit FileFind;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, BrkApart, Match;

type
  {*********************************************************************}
  { Exceptions                                                          }
  {*********************************************************************}
  EFileFind = class(Exception);
  EFileFindChDir = class(EFileFind);
  EFileFindOutOfResources = class(EFileFind);

  {*********************************************************************}
  { TFileFind                                                           }
  {*********************************************************************}
  TFileFind = class(TComponent)
  private
    { Dclarations private }
    FFilePattern: TStringList;
    FFilesFound: TStringList;
    FMatchCaseSensitive: Boolean;
    FMatchEnabled: Boolean;
    FMatchString: String;
    FThreadPriority: TThreadPriority;
    FRecursive: Boolean;
    FStartDir: TStringList;
    FOnTerminated: TNotifyEvent;
    FConvert: TBrkApart;
    FMatch: TMatch;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetFilePattern: String;
    procedure SetFilePattern(AFilePattern: String);
    procedure SetFilesFound(AFilesFound: TStringList);
    function GetStartDir: String;
    procedure SetStartDir(AStartDir: String);
    function CheckDir(ADir: String): String;
    procedure FindThreadDone(Sender: TObject);
  protected
    { Dclarations protected }
  public
    { Dclarations public }
    procedure Execute;
    procedure ThreadExecute;
  published
    { Dclarations published }
    property FilePattern: String read GetFilePattern write SetFilePattern;
    property FilesFound: TStringList read FFilesFound write SetFilesFound;
    property MatchCaseSensitive: Boolean read FMatchCaseSensitive write FMatchCaseSensitive;
    property MatchEnabled: Boolean read FMatchEnabled write FMatchEnabled;
    property MatchString: String read FMatchString write FMatchString;
    property Recursive: Boolean read FRecursive write FRecursive default True;
    property StartDir: String read GetStartDir write SetStartDir;
    property ThreadPriority: TThreadPriority read FThreadPriority write FThreadPriority;
    property OnTerminated: TNotifyEvent read FOnTerminated write FOnTerminated;
  end;

  {*********************************************************************}
  { TFindThread                                                         }
  {*********************************************************************}
  TFindThread = class(TThread)
  private
    FFilePattern: TStringList;
    FStartDir: TStringList;
    FFilesFound: TStringList;
    FRecursive: Boolean;
    FMatch: TMatch;
    FMatchCaseSensitive: Boolean;
    FMatchEnabled: Boolean;
    FMatchString: String;
   protected
    procedure Execute; override;
    procedure SearchTree(AFilePattern: String);
  public
    constructor Create(AStartDir: TStringList; AFilePattern: TStringList;
                       AFilesFound: TStringList; ARecursive: Boolean; AMatch: TMatch;
                       AMatchCaseSensitive: Boolean;AMatchEnabled: Boolean; AMatchString: String);
  end;

const
  { French Messages }
  MSG_TOO_MUCH_FILES = 'Trop de fichiers.';

  { English Messages }
  {MSG_TOO_MUCH_FILES = 'Too much files.';}

procedure Register;

implementation

{*********************************************************************}
{ procedure Register                                                  }
{*********************************************************************}
procedure Register;
begin
  RegisterComponents('Exemples', [TFileFind]);
end; {Register}

{*********************************************************************}
{ Composant TFileFind                                                 }
{*********************************************************************}

{*********************************************************************}
{ constructor TFileFind.Create                                        }
{*********************************************************************}
constructor TFileFind.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFilePattern := TStringList.Create;
  FFilesFound := TStringList.Create;
  FStartDir := TStringList.Create;
  FConvert := TBrkApart.Create(Self);
  FConvert.BreakString := ';';
  FMatch := TMatch.Create(Self);
  FMatch.SourceIsFile := true;
  FRecursive := true;
end; {TFileFind.Create}

{*********************************************************************}
{ destructor TFileFind.Destroy                                        }
{*********************************************************************}
destructor TFileFind.Destroy;
begin
  FFilePattern.Free;
  FFilesFound.Free;
  FStartDir.Free;
  FConvert.Free;
  FMatch.Free;
  inherited Destroy;
end; {TFileFind.Destroy}

{*********************************************************************}
{ function TFileFind.GetFilePattern                                   }
{*********************************************************************}
function TFileFind.GetFilePattern: String;
begin
  FConvert.StringList.Assign(FFilePattern);
  FConvert.ReverseBreakApart;
  Result := FConvert.BaseString;
end; {TFileFind.GetFilePattern}

{*********************************************************************}
{ procedure TFileFind.SetFilePattern                                  }
{*********************************************************************}
procedure TFileFind.SetFilePattern(AFilePattern: String);
begin
  FConvert.BaseString := AFilePattern;
  FConvert.BreakApart;
  FFilePattern.Assign(FConvert.StringList);
end; {TFileFind.SetFilePattern}

{*********************************************************************}
{ procedure TFileFind.SetFilesFound                                   }
{*********************************************************************}
procedure TFileFind.SetFilesFound(AFilesFound: TStringList);
begin
  FFilesFound.Assign(AFilesFound);
end; {TFileFind.SetFilesFound}

{*********************************************************************}
{ function TFileFind.GetStartDir                                      }
{*********************************************************************}
function TFileFind.GetStartDir: String;
begin
  FConvert.StringList.Assign(FStartDir);
  FConvert.ReverseBreakApart;
  Result := FConvert.BaseString;
end; {TFileFind.GetStartDir}

{*********************************************************************}
{ procedure TFileFind.SetStartDir                                     }
{*********************************************************************}
procedure TFileFind.SetStartDir(AStartDir: String);
var
  i: Integer;
begin
  FConvert.BaseString := AStartDir;
  FConvert.BreakApart;
  FStartDir.Assign(FConvert.StringList);
  for i := 0 to FStartDir.Count - 1 do
    FStartDir[i] := CheckDir(FStartDir[i]);
end; {TFileFind.SetStartDir}

{*********************************************************************}
{ function TFileFind.CheckDir                                         }
{*********************************************************************}
function TFileFind.CheckDir(ADir: String): String;
var
  TempDir: String; { est-ce bien necessaire ??? }
begin
    TempDir := ADir;
    if ((TempDir <> '\') and
        (TempDir[Length(TempDir)] = '\') and
        (not ((Length(TempDir)=3) and (TempDir[2]=':') and (TempDir[3]='\')))
        ) then
      TempDir := Copy(TempDir, 1, Length(TempDir) - 1);
    Result := TempDir;
end; {TFileFind.CheckDir}

{*********************************************************************}
{ procedure TFileFind.FindThreadDone                                  }
{*********************************************************************}
procedure TFileFind.FindThreadDone(Sender: TObject);
begin
  if Assigned(FOnTerminated) then FOnTerminated(Self);
end; {TFileFind.FindThreadDone}

{*********************************************************************}
{ procedure TFileFind.Execute                                         }
{*********************************************************************}
procedure TFileFind.Execute;

  procedure SearchTree(AFilePattern: String);
  var
    SearchRec: TSearchRec;
    DosError: integer;
    dir: string;
  begin
    GetDir(0, dir);
    if dir[length(dir)] <> '\' then dir := dir + '\';
    DosError := FindFirst(AFilePattern, 0, SearchRec);
    while DosError = 0 do begin
      try
       if (FMatchEnabled = true) and (FMatchString <> '') then
        begin
         FMatch.source := dir + SearchRec.name;
         if FMatch.match then FFilesFound.add(dir + SearchRec.name);
        end
       else
        FFilesFound.add(dir + SearchRec.name);
      except
        on EOutOfResources do begin
          raise EFileFindOutOfResources.Create(MSG_TOO_MUCH_FILES);
          abort;
        end;
      end;
      DosError := FindNext(SearchRec);
    end;
    SysUtils.FindClose(SearchRec);

    {Now that we have all the files we need, lets go to a subdirectory.}
    if FRecursive = true then
     begin
      DosError := FindFirst('*.*', faDirectory, SearchRec);
      while DosError = 0 do begin
       {If there is one, go there and search.}
       if ((SearchRec.attr and faDirectory = faDirectory) and
        (SearchRec.name <> '.') and (SearchRec.name <> '..')) then begin

        ChDir(SearchRec.name);
        SearchTree(AFilePattern); {Time for the recursion!}
        ChDir('..'); {Down one level.}
      end;
      DosError := FindNext(SearchRec); {Look for another subdirectory}
     end;
     SysUtils.FindClose(SearchRec);

    end;
  end; {SearchTree}

var
  iDir, iFP: Integer;
begin
  FFilesFound.clear;
  FMatch.CaseSensitive := FMatchCaseSensitive;
  FMatch.Pattern := FMatchString;
  for iDir := 0 to FStartDir.Count - 1 do begin
    try
      ChDir(FStartDir[iDir]);
    except
      on E:Exception do raise EFileFindChDir.Create(E.Message);
    end;
    for iFP := 0 to FFilePattern.Count - 1 do begin
      SearchTree(FFilePattern[iFP]);
    end;
  end;
end; {TFileFind.Execute}

{*********************************************************************}
{ procedure TFileFind.ThreadExecute                                   }
{*********************************************************************}
procedure TFileFind.ThreadExecute;
begin
  with TFindThread.Create(FStartDir, FFilePattern, FFilesFound, FRecursive, FMatch, FMatchCaseSensitive, FMatchEnabled, FMatchString) do begin
    OnTerminate := FindThreadDone;
    Priority := FThreadPriority;
  end;
end; {TFileFind.ThreadExecute}


{*********************************************************************}
{ Objet TFindThread                                                   }
{*********************************************************************}

{*********************************************************************}
{ constructor TFindThread.Create                                      }
{*********************************************************************}
constructor TFindThread.Create(AStartDir: TStringList; AFilePattern: TStringList;
                               AFilesFound: TStringList; ARecursive: Boolean;AMatch: TMatch;
                               AMatchCaseSensitive: Boolean;AMatchEnabled: Boolean; AMatchString: String);
begin
  inherited Create(False);
  FStartDir := AStartDir;
  FFilePattern := AFilePattern;
  FFilesFound := AFilesFound;
  FRecursive := ARecursive;
  FMatchCaseSensitive := AMatchCaseSensitive;
  FMatchEnabled := AMatchEnabled;
  FMatchString := AMatchString;
  FMatch := AMatch;
  FreeOnTerminate := True;
end; {TFindThread.Create}

{*********************************************************************}
{ procedure TFindThread.Execute                                       }
{*********************************************************************}
procedure TFindThread.Execute;
var
  iDir, iFP: Integer;
begin
  FFilesFound.clear;
  FMatch.CaseSensitive := FMatchCaseSensitive;
  FMatch.Pattern := FMatchString;
  for iDir := 0 to FStartDir.Count - 1 do begin
    try
      ChDir(FStartDir[iDir]);
    except
      on E:Exception do raise EFileFindChDir.Create(E.Message);
    end;
    for iFP := 0 to FFilePattern.Count - 1 do begin
      SearchTree(FFilePattern[iFP]);
    end;
  end;
end; {TFindThread.Execute}

{*********************************************************************}
{ procedure TFindThread.SearchTree                                    }
{*********************************************************************}
procedure TFindThread.SearchTree(AFilePattern: String);
var
  SearchRec: TSearchRec;
  DosError: integer;
  dir: string;
begin
  GetDir(0, dir);
  if dir[length(dir)] <> '\' then dir := dir + '\';
  DosError := FindFirst(AFilePattern, 0, SearchRec);
  while DosError = 0 do begin
    try
     if (FMatchEnabled = true) and (FMatchString <> '') then
      begin
       FMatch.source := dir + SearchRec.name;
       if FMatch.match then FFilesFound.add(dir + SearchRec.name);
      end
     else
      FFilesFound.add(dir + SearchRec.name);

    except
      on EOutOfResources do begin
        raise EFileFindOutOfResources.Create('Trop de fichiers.');
        abort;
      end;
    end;
    if Terminated then Exit;
    DosError := FindNext(SearchRec);
  end;
  {Now that we have all the files we need, lets go to a subdirectory.}
  if FRecursive = true then
   begin
    DosError := FindFirst('*.*', faDirectory, SearchRec);
    while DosError = 0 do begin
    {If there is one, go there and search.}
     if ((SearchRec.attr and faDirectory = faDirectory) and
      (SearchRec.name <> '.') and (SearchRec.name <> '..')) then begin

      ChDir(SearchRec.name);
      SearchTree(AFilePattern); {Time for the recursion!}
      ChDir('..'); {Down one level.}
    end;
    DosError := FindNext(SearchRec); {Look for another subdirectory}
   end;
  end;
end; {TFindThread.SearchTree}

end.
