unit GraphView;

interface

uses
  Windows, Classes, Forms, Dialogs, Graphics,
  StdCtrls, Buttons, CheckLst, Chart, TeEngine,
  Series, Controls, ExtCtrls;

type
  TGraphStyle = (gsLine, gsArea, gsBar, gsPoint);
  TGraphViewItem = class(TObject)
    Name: String;
    Style: TGraphStyle;
    Checked: Boolean;
    Color: TColor;
    constructor Create(aName: String; aStyle: TGraphStyle; aChecked: Boolean; aColor: TColor);
   end;

var
  CurrentGraphViewItem: TGraphViewItem;

type
  TLanguage = (lEnglish, lRussian, lUkrainian);
  
  TGraphViewDialog = class(TForm)
    CloseBtn: TBitBtn;
    ListBox: TCheckListBox;
    LineImage: TImage;
    AreaImage: TImage;
    BarImage: TImage;
    ColorDialog: TColorDialog;
    PointImage: TImage;
    Bevel1: TBevel;
    Bevel2: TBevel;
    Bevel3: TBevel;
    Bevel4: TBevel;
    Label1: TLabel;
    StyleBtn: TBitBtn;
    ColorBtn: TBitBtn;
    UpBtn: TSpeedButton;
    DownBtn: TSpeedButton;
    TitleBtn: TBitBtn;
    procedure ListBoxDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure ListBoxDblClick(Sender: TObject);
    procedure ListBoxMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure StyleBtnClick(Sender: TObject);
    procedure ColorBtnClick(Sender: TObject);
    procedure TitleBtnClick(Sender: TObject);
    procedure ListBoxClickCheck(Sender: TObject);
    procedure UpBtnClick(Sender: TObject);
    procedure DownBtnClick(Sender: TObject);
  private
    List: TList;

    MouseX, MouseY: Integer;
  public
    MasterChart: TChart;
    Language: TLanguage;

    procedure Clear;
    procedure Add(aName: String; aStyle: TGraphStyle; aChecked: Boolean; aColor: TColor);
  end;

var
  GraphViewDialog: TGraphViewDialog;

implementation

uses GraphStyle;

{$R *.DFM}

constructor TGraphViewItem.Create(aName: String; aStyle: TGraphStyle; aChecked: Boolean; aColor: TColor);
begin
  Name := aName;
  Style := aStyle;
  Checked := aChecked;
  Color := aColor;
end;

procedure TGraphViewDialog.ListBoxDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  Image: TImage;
  SwapColor: TColor;
begin
  with ListBox.Canvas do
   begin
    FillRect(Rect);
    try
     CurrentGraphViewItem := List[Index];
    except
     CurrentGraphViewItem := nil;
    end;
    if CurrentGraphViewItem <> nil then
     with CurrentGraphViewItem do
      begin
       case Style of
         gsPoint: Image := PointImage;
         gsArea: Image := AreaImage;
         gsBar: Image := BarImage;
         else Image := LineImage;
        end;
       Draw(Rect.Left + 2, Rect.Top + 2, Image.Picture.Graphic);
       SwapColor := Brush.Color;
       Brush.Color := Color;
       Rectangle(Rect.Left + 23, Rect.Top + 2, Rect.Left + 40, Rect.Bottom - 3);
       Brush.Color := SwapColor;
       TextOut(Rect.Left + 43, Rect.Top + 4, Name);
      end;
   end;
end;

procedure TGraphViewDialog.FormCreate(Sender: TObject);
begin
  List := TList.Create;
end;

procedure TGraphViewDialog.FormDestroy(Sender: TObject);
begin
  List.Free;
end;

procedure TGraphViewDialog.Clear;
begin
  List.Clear;
  ListBox.Clear;
end;

procedure TGraphViewDialog.Add(aName: String; aStyle: TGraphStyle; aChecked: Boolean; aColor: TColor);
begin
  CurrentGraphViewItem := TGraphViewItem.Create(aName, aStyle, aChecked, aColor);
  List.Add(CurrentGraphViewItem);
  ListBox.Items.Add(aName);
  ListBox.Checked[ListBox.Items.Count - 1] := aChecked;
end;

procedure TGraphViewDialog.FormHide(Sender: TObject);
begin
  Clear;
end;

procedure TGraphViewDialog.ListBoxDblClick(Sender: TObject);
begin
  if (MouseX >= 18) and (MouseX <= 36) then StyleBtnClick(nil)
  else
   if (MouseX >= 39) and (MouseX <= 55) then ColorBtnClick(nil)
   else
    if MouseX > 58 then TitleBtnClick(nil);
end;

procedure TGraphViewDialog.ListBoxMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
begin
  MouseX := X;
  MouseY := Y;
end;

procedure TGraphViewDialog.StyleBtnClick(Sender: TObject);
var
  NewSerie, OldSerie: TChartSeries;
begin
  if ListBox.ItemIndex <> - 1 then
   begin
    try
     GraphStyleDialog := TGraphStyleDialog.Create(Application);
     CurrentGraphViewItem := List[ListBox.ItemIndex];
     GraphStyleDialog.CurrentBtn := CurrentGraphViewItem.Style;

     with GraphStyleDialog do
      case Language of
        lEnglish: begin
                   Caption := 'Series style';
                   CancelBtn.Caption := 'Cancel';
                   LineBtn.Caption := 'Line';
                   BarBtn.Caption := 'Bar';
                   AreaBtn.Caption := 'Area';
                   PointBtn.Caption := 'Points';
                  end;
        lRussian: begin
                   Caption := '  ';
                   CancelBtn.Caption := '';
                   LineBtn.Caption := '';
                   BarBtn.Caption := '';
                   AreaBtn.Caption := '';
                   PointBtn.Caption := '';
                  end;
        lUkrainian: begin
                     Caption := '  i';
                     CancelBtn.Caption := 'ii';
                     LineBtn.Caption := 'ii';
                     BarBtn.Caption := '';
                     AreaBtn.Caption := 'i';
                     PointBtn.Caption := '';
                    end;
       end;

     if GraphStyleDialog.ShowModal = id_Ok then
      begin
       CurrentGraphViewItem.Style := GraphStyleDialog.CurrentBtn;
       ListBox.Refresh;
       OldSerie := MasterChart.Series[ListBox.ItemIndex];
       case CurrentGraphViewItem.Style of
         gsArea: NewSerie := TAreaSeries.Create(MasterChart);
         gsBar: NewSerie := TBarSeries.Create(MasterChart);
         gsPoint: NewSerie := TPointSeries.Create(MasterChart);
         else NewSerie := TLineSeries.Create(MasterChart);
        end;
       MasterChart.AddSeries(NewSerie);
       MasterChart.ExchangeSeries(ListBox.ItemIndex, MasterChart.SeriesCount - 1);
       NewSerie.Assign(OldSerie);
       MasterChart.RemoveSeries(OldSerie);
      end;
    except
    end;
   GraphStyleDialog.Free;
  end;
end;

procedure TGraphViewDialog.ColorBtnClick(Sender: TObject);
begin
  if ListBox.ItemIndex <> -1 then
   try
    CurrentGraphViewItem := List[ListBox.ItemIndex];
    ColorDialog.Color := CurrentGraphViewItem.Color;
    if ColorDialog.Execute then
     begin
      CurrentGraphViewItem.Color := ColorDialog.Color;
      ListBox.Refresh;
      MasterChart.Series[ListBox.ItemIndex].SeriesColor := ColorDialog.Color;
     end;
   except
   end;
end;

procedure TGraphViewDialog.TitleBtnClick(Sender: TObject);
var
  St: String;
  b: Boolean;
begin
  if ListBox.ItemIndex <> -1 then
   begin
    try
     CurrentGraphViewItem := List[ListBox.ItemIndex];
     St := CurrentGraphViewItem.Name;
     case Language of
       lRussian: b := InputQuery(' ', '  :', St);
       lUkrainian: b := InputQuery(' ', 'i  :', St)
       else b := InputQuery('Series title', 'Enter the series title:', St);
      end;
     if b then
      begin
       CurrentGraphViewItem.Name := St;
       ListBox.Refresh;
       MasterChart.Series[ListBox.ItemIndex].Title := St;
      end;
    except
    end;
  end;  
end;

procedure TGraphViewDialog.ListBoxClickCheck(Sender: TObject);
var
  i: Integer;
begin
  i := ListBox.ItemIndex;
  if i > -1 then
   MasterChart.Series[i].Active := not MasterChart.Series[i].Active;
end;

procedure TGraphViewDialog.UpBtnClick(Sender: TObject);
var
  CurrentSeries: TChartSeries;
begin
  if (List.Count > 1) and
     (ListBox.ItemIndex > 0) then
   try
    CurrentSeries := MasterChart.Series[ListBox.ItemIndex];
    MasterChart.SeriesUp(CurrentSeries);
    List.Exchange(ListBox.ItemIndex, ListBox.ItemIndex - 1);
    ListBox.ItemIndex := ListBox.ItemIndex - 1;
    ListBox.Refresh;
   except
   end;
end;

procedure TGraphViewDialog.DownBtnClick(Sender: TObject);
var
  CurrentSeries: TChartSeries;
begin
  if (List.Count > 1) and
     (List.Count > ListBox.ItemIndex + 1) and
     (ListBox.ItemIndex <> -1) then
   try
    CurrentSeries := MasterChart.Series[ListBox.ItemIndex];
    MasterChart.SeriesDown(CurrentSeries);
    List.Exchange(ListBox.ItemIndex, ListBox.ItemIndex + 1);
    ListBox.ItemIndex := ListBox.ItemIndex + 1;
    ListBox.Refresh;
   except
   end;
end;


end.
