unit ticker;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls;

type
  TScrollType = (stLeftToRight, stRightToLeft, stTopToBottom, stBottomToTop);
  TCF_Ticker = class(TGraphicControl)
  private
    { Private declarations }
    Timer: TTimer;                                // Used when the ticker is moving
    Queue: TList;                                 // Holds the queue of desired bitmaps
    Surface,                                      // Holds the ticker-canvas to be used in the PAINT-method
    ClearBuffer,                                  // A cleared bitmap for use when clearing the ticker
    WorkBuffer: TBitMap;                          // A bitmap that is used when modifying the tickers bitmap
    NotMoving: Boolean;                           // If true the ticker is moving, and vice-versa
    Position,                                     // A coordinate to the current scrolling bitmap
    Wall: Integer;                                // The coordinate where the current scrolling bitmap is stopped
    SType: TScrollType;                           // Decides the direction of the scrolling ticker
    TStep,                                        // The distance for every scroll
    StepSave: Integer;
    BackGroundColor: TColor;
    function  GetSpeed: Integer;
    procedure SetSpeed(NewSpeed: Integer);
    procedure SetStep(NewStep: Integer);
    procedure SetBgColor(NewColor: TColor);
    procedure SetScrollType(NewSType: TScrollType);
    procedure Display(Sender: TObject);           // This procedure handles the scrolling bitmap
  protected
    { Protected declarations }
    procedure Paint; override;                    // I didn't want to use this, but I had to ...
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property  Canvas;
    procedure Initialize;                         // This will initialize the ticker, resetting the variables
    procedure ShowTickerData(Data: TBitMap);     // Call this procedure when you want something to move on the ticker
  published
    { Published declarations }
    property Speed: Integer read GetSpeed write SetSpeed default 100;      // This property decides the scrolling frequency
    property Step: Integer read TStep write SetStep default 1;
    property BgColor: TColor read BackGroundColor write SetBgColor default clBlack;
    property ScrollType: TScrollType read SType write SetScrollType default stLeftToRight;
    property Left default 10;
    property Top default 10;
    property Width default 400;
    property Height default 40;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('CF', [TCF_Ticker]);
end;

constructor TCF_Ticker.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Left := 10;
  Top := 10;
  Width := 400;
  Height := 40;
  Timer := TTimer.Create(Self);
  Timer.Enabled := False;
  Timer.OnTimer := Display;
  SetSpeed(10);
  Queue := TList.Create;
  Surface := TBitMap.Create;
  Surface.Width := Width;
  Surface.Height := Height;
  WorkBuffer := TBitMap.Create;
  ClearBuffer := TBitMap.Create;
  BackGroundColor := clBlack;
  NotMoving := True;
  Position := 0;
  Step := 1;
  Wall := 0;
end;

destructor TCF_Ticker.Destroy;
begin
  Timer.Enabled := False;
  Timer.Free;
  Queue.Free;
  Surface.Free;
  WorkBuffer.Free;
  ClearBuffer.Free;
  inherited Destroy;
  MessageDlg('TCF_Ticker - Component for Borland Delphi' + #13#10#13#10
     + '(c) 1997  Christian Fagrell  All rights reserved', mtInformation, [mbOK], 0);
end;

procedure TCF_Ticker.Display(Sender: TObject);
var Temp: TBitMap;
begin
  If Queue.Count > 0 then begin
    If NotMoving then begin
      // Save the desired bitmap in the temp-variable
      Temp := Queue[0];
      If SType in [stLeftToRight,stRightToLeft] then Wall := Temp.Width
      Else Wall := Temp.Height;
      Position := 0;
      // Build the new moving bitmap
      Case SType of
        stLeftToRight:Begin
                        WorkBuffer.Canvas.CopyRect(Rect(0,0,Wall,Height),Temp.Canvas,Rect(0,0,Wall,Height));
                        WorkBuffer.Canvas.CopyRect(Rect(Wall,0,Wall+Width,Height),Canvas,Rect(0,0,Width,Height));
                      End;
        stRightToLeft:Begin
                        WorkBuffer.Canvas.CopyRect(Rect(Width,0,Width+Wall,Height),Temp.Canvas,Rect(0,0,Wall,Height));
                        WorkBuffer.Canvas.CopyRect(Rect(0,0,Width,Height),Canvas,Rect(0,0,Width,Height));
                      End;
        stBottomToTop:Begin
                        WorkBuffer.Canvas.CopyRect(Rect(0,0,Width,Wall),Temp.Canvas,Rect(0,0,Width,Wall));
                        WorkBuffer.Canvas.CopyRect(Rect(0,Wall,Width,Wall+Height),Canvas,Rect(0,0,Width,Height));
                      End;
        stTopToBottom:Begin
                        WorkBuffer.Canvas.CopyRect(Rect(0,Height,Width,Height+Wall),Temp.Canvas,Rect(0,0,Width,Wall));
                        WorkBuffer.Canvas.CopyRect(Rect(0,0,Width,Height),Canvas,Rect(0,0,Width,Height));
                      End;
      End;
      NotMoving := False;
    End;
    // Clear the ticker
    Surface.Canvas.CopyRect(Rect(0,0,Width,Height),ClearBuffer.Canvas,Rect(0,0,ClearBuffer.Width,ClearBuffer.Height));
    Inc(Position, TStep);
    // Move the ticker-bitmap
    Case SType of
      stLeftToRight:Surface.Canvas.Copyrect(Rect(0,0,Width,Height),WorkBuffer.Canvas,Rect(Wall-Position,0,Width+(Wall-Position),Height));
      stRightToLeft:Surface.Canvas.Copyrect(Rect(0,0,Width,Height),WorkBuffer.Canvas,Rect(Position,0,Width+Position,Height));
      stBottomToTop:Surface.Canvas.Copyrect(Rect(0,0,Width,Height),WorkBuffer.Canvas,Rect(0,Wall-Position,Width,Height+(Wall-Position)));
      stTopToBottom:Surface.Canvas.Copyrect(Rect(0,0,Width,Height),WorkBuffer.Canvas,Rect(0,Position,Width,Height+Position));
    End;
    Paint;
    If Position >= (Wall - TStep) then begin
      StepSave := TStep;
      TStep := Wall - Position;
    End;
    // If the bitmap has finished moving, remove the bitmap from the queue
    If Position = Wall then begin
      NotMoving := True;
      Queue.Delete(0);
      TStep := StepSave;
    End;
  end;
end;

function TCF_Ticker.GetSpeed: Integer;
begin
  GetSpeed := Timer.Interval;
end;

procedure TCF_Ticker.SetSpeed(NewSpeed: Integer);
begin
  Timer.Interval := NewSpeed;
end;

procedure TCF_Ticker.SetStep(NewStep: Integer);
begin
  If NotMoving then TStep := NewStep;
end;

procedure TCF_Ticker.SetBgColor(NewColor: TColor);
begin
  BackGroundColor := NewColor;
  Surface.Width := Width;
  Surface.Height := Height;
  Surface.Canvas.Brush.Color := BackGroundColor;
  Surface.Canvas.FillRect(Rect(0,0,Width,Height));
  Paint;
end;

procedure TCF_Ticker.SetScrollType(NewSType: TScrollType);
begin
  If NotMoving then SType := NewSType;
end;

procedure TCF_Ticker.Initialize;    // Call this procedure before the first scrolling or when you want to reset the ticker
begin
  Position := 0;
  Wall := 0;
  NotMoving := True;
  Surface.Width := Width;
  Surface.Height := Height;
  Surface.Canvas.Brush.Color := BackGroundColor;
  Surface.Canvas.FillRect(Rect(0,0,Width,Height));
  ClearBuffer.Width := Width;
  ClearBuffer.Height := Height;
  ClearBuffer.Canvas.Brush.Color := BackGroundColor;
  ClearBuffer.Canvas.FillRect(Rect(0,0,ClearBuffer.Width,ClearBuffer.Height));
  WorkBuffer.Width := Width + Width;
  WorkBuffer.Height := Height + Height;
  Queue.Clear;
  Paint;
end;

procedure TCF_Ticker.ShowTickerData(Data: TBitMap);
begin
  Queue.Add(Data);
  Timer.Enabled := True;
end;

procedure TCF_Ticker.Paint;
begin
  Canvas.Draw(0,0,Surface);
end;

end.
